// SPDX-FileCopyrightText: 2020 - 2023 Tobias Fella <tobias.fella@kde.org>
// SPDX-FileCopyrightText: 2022 James Graham <james.h.graham@protonmail.com>
// SPDX-License-Identifier: GPL-2.0-or-later

import QtQuick
import QtQuick.Controls as QQC2
import QtQuick.Layouts

import org.kde.kirigami as Kirigami
import org.kde.kirigamiaddons.formcard as FormCard

import org.kde.neochat

FormCard.AbstractFormDelegate {
    id: root

    required property string id
    required property string timestamp
    required property string displayName
    required property int type

    required property DevicesModel devicesModel

    property bool editDeviceName: false
    property bool showVerifyButton

    onClicked: root.editDeviceName = true

    contentItem: RowLayout {
        spacing: Kirigami.Units.largeSpacing

        Kirigami.Icon {
            source: "network-connect"
            implicitWidth: Kirigami.Units.iconSizes.medium
            implicitHeight: Kirigami.Units.iconSizes.medium
        }
        ColumnLayout {
            id: deviceLabel
            Layout.fillWidth: true
            spacing: Kirigami.Units.smallSpacing
            visible: !root.editDeviceName

            QQC2.Label {
                Layout.fillWidth: true
                text: root.displayName
                elide: Text.ElideRight
                wrapMode: Text.Wrap
                maximumLineCount: 2
            }

            QQC2.Label {
                Layout.fillWidth: true
                text: i18nc("@label", "%1, Last activity: %2", root.id, root.timestamp)
                color: Kirigami.Theme.disabledTextColor
                font: Kirigami.Theme.smallFont
                elide: Text.ElideRight
                visible: text.length > 0
            }
        }
        Kirigami.ActionTextField {
            id: nameField
            Accessible.description: i18n("New device name")
            Layout.fillWidth: true
            Layout.preferredHeight: deviceLabel.implicitHeight
            visible: root.editDeviceName

            text: root.displayName

            rightActions: [
                Kirigami.Action {
                    text: i18n("Cancel editing display name")
                    icon.name: "edit-delete-remove"
                    onTriggered: {
                        root.editDeviceName = false;
                    }
                },
                Kirigami.Action {
                    text: i18n("Confirm new display name")
                    icon.name: "checkmark"
                    visible: nameField.text !== root.displayName
                    onTriggered: {
                        root.devicesModel.setName(root.id, nameField.text);
                    }
                }
            ]

            onAccepted: root.devicesModel.setName(root.id, nameField.text)
        }
        QQC2.ToolButton {
            display: QQC2.AbstractButton.IconOnly
            text: i18nc("@action:button", "Edit device name")
            icon.name: "document-edit"
            onClicked: root.editDeviceName = true
            QQC2.ToolTip.text: text
            QQC2.ToolTip.delay: Kirigami.Units.toolTipDelay
            QQC2.ToolTip.visible: hovered
        }
        QQC2.ToolButton {
            display: QQC2.AbstractButton.IconOnly
            visible: root.showVerifyButton && (root.type !== DevicesModel.Verified || NeoChatConfig.alwaysVerifyDevice)
            text: i18nc("@action:button", "Verify device")
            icon.name: "security-low-symbolic"
            onClicked: root.devicesModel.connection.startKeyVerificationSession(root.devicesModel.connection.localUserId, root.id)

            QQC2.ToolTip.text: text
            QQC2.ToolTip.delay: Kirigami.Units.toolTipDelay
            QQC2.ToolTip.visible: hovered
        }
        Kirigami.Icon {
            visible: root.showVerifyButton && root.type === DevicesModel.Verified
            implicitWidth: Kirigami.Units.iconSizes.smallMedium
            implicitHeight: Kirigami.Units.iconSizes.smallMedium
            source: "security-high"

            HoverHandler {
                id: verifyHover
            }
            QQC2.ToolTip {
                text: i18nc("This device is verified", "Verified")
                delay: Kirigami.Units.toolTipDelay
                visible: verifyHover.hovered
            }
        }
        QQC2.ToolButton {
            display: QQC2.AbstractButton.IconOnly
            text: i18nc("@action:button", "Logout device")
            icon.name: "edit-delete-remove"
            onClicked: {
                passwordSheet.deviceId = root.id;
                passwordSheet.open();
            }

            QQC2.ToolTip.text: text
            QQC2.ToolTip.delay: Kirigami.Units.toolTipDelay
            QQC2.ToolTip.visible: hovered
        }
    }

    Kirigami.Dialog {
        id: passwordSheet

        property string deviceId

        preferredWidth: Kirigami.Units.gridUnit * 24

        title: i18nc("@title:dialog", "Remove device")

        standardButtons: QQC2.Dialog.Cancel

        Component.onCompleted: passwordField.forceActiveFocus()

        contentItem: FormCard.FormTextFieldDelegate {
            id: passwordField
            label: i18n("Password:")
            echoMode: TextInput.Password
        }
        customFooterActions: [
            Kirigami.Action {
                text: i18nc("@action:button As in 'Remove this device'", "Remove")
                icon.name: "delete"
                onTriggered: {
                    root.devicesModel.logout(passwordSheet.deviceId, passwordField.text);
                    passwordField.text = "";
                    passwordSheet.close();
                }
            }
        ]
    }
}
