/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent.proxy;

import java.net.URI;

import junit.framework.TestCase;

import org.apache.commons.lang.StringUtils;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.auth.AuthScope;
import org.apache.http.auth.Credentials;
import org.apache.http.auth.UsernamePasswordCredentials;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.params.ClientPNames;
import org.apache.http.conn.ClientConnectionManager;
import org.apache.http.conn.scheme.PlainSocketFactory;
import org.apache.http.conn.scheme.Scheme;
import org.apache.http.conn.scheme.SchemeRegistry;
import org.apache.http.conn.ssl.SSLSocketFactory;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.http.impl.conn.tsccm.ThreadSafeClientConnManager;
import org.apache.http.params.BasicHttpParams;

/**
 * Example test case to show how to implement multi-threaded http client invocations with authentications.
 * 
 * @version $Id: TestHttpClientAuth.java 818597 2009-09-24 19:19:21Z woonsan $
 */
public class TestHttpClientAuth extends TestCase
{
    private String authHost;
    private int authPort = 80;
    private String authRealm;
    private String username;
    private String password;
    private String badusername = "bad_user_name_not_existing";
    private String badpassword = "bad_user_password";
    
    private String targetURI;
    private boolean targetURIAvailable;
    
    private AuthScope authScope;
    private Credentials credentials;
    private Credentials badcredentials;
    
    private ClientConnectionManager connectionManager;    
    private int workersCount = 20;
    
    @Override
    public void setUp()
    {
        targetURI = System.getProperty("TestHttpClientAuth.targetURI");
        
        if (targetURI == null)
        {
            System.out.println("Test skipped: You need to set a system property to run this test case as follows:\n\n" +
                               "\t-DTestHttpClientAuth.targetURI=http://manager:manager@localhost:8080/manager/list\n");
            return;
        }
        
        URI theTargetURI = URI.create(targetURI);
        targetURIAvailable = true;
        
        authHost = theTargetURI.getHost();
        
        if (theTargetURI.getPort() > 0)
        {
            authPort = theTargetURI.getPort();
        }
        
        authRealm = "Tomcat Manager Application";
        
        String userInfo = theTargetURI.getUserInfo();
        String [] userInfoArray = StringUtils.split(userInfo, ":");
        username = userInfoArray[0];
        password = userInfoArray[1];
        
        if (authRealm == null)
        {
            authScope = new AuthScope(authHost, authPort);
        }
        else
        {
            authScope = new AuthScope(authHost, authPort, authRealm);
        }
        
        credentials = new UsernamePasswordCredentials(username, password);
        badcredentials = new UsernamePasswordCredentials(badusername, badpassword);

        SchemeRegistry schemeRegistry = new SchemeRegistry();
        schemeRegistry.register(new Scheme("http", PlainSocketFactory.getSocketFactory(), 80));
        schemeRegistry.register(new Scheme("https", SSLSocketFactory.getSocketFactory(), 443));

        connectionManager = new ThreadSafeClientConnManager(new BasicHttpParams(), schemeRegistry);
    }

    @Override
    public void tearDown()
    {
        // When HttpClient instance is no longer needed,
        // shut down the connection manager to ensure
        // immediate deallocation of all system resources
        if (connectionManager != null)
        {
            connectionManager.shutdown();
        }
    }

    public void testClientAuthentication() throws Exception
    {
        if (!targetURIAvailable)
        {
            return;
        }
        
        Thread [] workers = new Thread[workersCount];
        AuthRequestJob [] jobs = new AuthRequestJob[workersCount];
        
        for (int i = 0; i < workersCount; i++)
        {
            boolean isBadCredentials = (i % 2 != 0);
            jobs[i] = new AuthRequestJob(targetURI, authScope, (isBadCredentials ? badcredentials : credentials), isBadCredentials);
            workers[i] = new Thread(jobs[i], "WORKER-" + i);
        }
        
        for (int i = 0; i < workersCount; i++)
        {
            workers[i].start();
        }
        
        for (int i = 0; i < workersCount; i++)
        {
            workers[i].join();
        }
        
        for (int i = 0; i < workersCount; i++)
        {
            assertNull("auth request job has exception.", jobs[i].getException());
        }
    }
    
    private void executeAuthRequest(String target, AuthScope targetAuthScope, Credentials targetCredentials, boolean isBadCredentials) throws Exception
    {
        HttpGet httpget = null;
        HttpEntity entity = null;
        
        try
        {
            DefaultHttpClient httpclient = new DefaultHttpClient(connectionManager, new BasicHttpParams());
            httpclient.getParams().setBooleanParameter(ClientPNames.HANDLE_REDIRECTS, false);

            httpclient.getCredentialsProvider().setCredentials(targetAuthScope, targetCredentials);
            
            httpget = new HttpGet(target);
            System.out.println(Thread.currentThread().getName() + ": executing request" + httpget.getRequestLine());
            
            HttpResponse response = httpclient.execute(httpget);
            entity = response.getEntity();
            
            System.out.println(Thread.currentThread().getName() + ": Status Line: " + response.getStatusLine());
            int statusCode = response.getStatusLine().getStatusCode();
            
            if (!isBadCredentials && (statusCode >= 400))
            {
                throw new RuntimeException("the status code shows it's not authorized automatically: " + statusCode);
            }
            else if (isBadCredentials && (statusCode < 400))
            {
                throw new RuntimeException("the status code shows it's authorized automatically for bad credentials: " + statusCode);
            }
            
            //System.out.println(Thread.currentThread().getName() + ": " + EntityUtils.toString(entity));
            
            entity.consumeContent();
        }
        catch (Exception e)
        {
            httpget.abort();
            entity = null;
            throw e;
        }
        finally
        {
            if (entity != null)
            {
                entity.consumeContent();
            }
        }
    }
    
    private class AuthRequestJob implements Runnable
    {
        private String target;
        private AuthScope authScope;
        private Credentials credentials;
        private boolean isBadCredentials;
        private Exception exception;
        
        private AuthRequestJob(String target, AuthScope authScope, Credentials credentials, boolean isBadCredentials)
        {
            this.target = target;
            this.authScope = authScope;
            this.credentials = credentials;
            this.isBadCredentials = isBadCredentials;
        }
        
        public Exception getException()
        {
            return exception;
        }
        
        public void run()
        {
            try
            {
                executeAuthRequest(target, authScope, credentials, isBadCredentials);
            }
            catch (Exception e)
            {
                System.out.println(Thread.currentThread().getName() + ": " + e + ", " + credentials);
                e.printStackTrace();
                exception = e;
            }
        }
    }
    
}
