/* 
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent2.portlet.proxy;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.portlet.PortletConfig;
import javax.portlet.PortletException;

import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.BooleanUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.http.conn.socket.LayeredConnectionSocketFactory;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.conn.ssl.SSLContexts;
import org.apache.http.conn.ssl.X509HostnameVerifier;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.client.HttpClients;
import org.apache.portals.applications.webcontent2.proxy.HttpClientContextBuilder;
import org.apache.portals.applications.webcontent2.proxy.ProxyMapping;
import org.apache.portals.applications.webcontent2.proxy.ProxyMappingRegistry;
import org.apache.portals.applications.webcontent2.proxy.builder.ProxyServices;
import org.apache.portals.applications.webcontent2.proxy.impl.DefaultProxyMappingRegistry;
import org.apache.portals.applications.webcontent2.proxy.impl.DefaultReverseProxyService;
import org.apache.portals.applications.webcontent2.proxy.impl.ProxyProcessingChain;
import org.apache.portals.applications.webcontent2.proxy.util.ProxyCommandUtils;
import org.apache.portals.applications.webcontent2.rewriter.ContentRewriter;
import org.apache.portals.applications.webcontent2.rewriter.htmlcleaner.DefaultSerializerFactory;
import org.apache.portals.applications.webcontent2.rewriter.htmlcleaner.HtmlCleanerContentRewriter;
import org.htmlcleaner.SimpleHtmlSerializer;
import org.htmlcleaner.TagNodeVisitor;

/**
 * Simple reverse proxy portlet implementation with using {@link HtmlCleanerContentRewriter}
 * as the default {@link ContentRewriter}.
 * And this is given as an example and a base class to be overriden.
 * <p>
 * This portlet reads {@link #XPATH_EXPRESSION_PARAM_NAME} parameter value (<code>//body</code> by default)
 * to select only the specified DOM element and its descendants.
 * Also, this portlet can transform the content by using <code>HtmlCleaner</code> transformation rules.
 * Finally, this portlet can use {@link TagNodeVisitor}s to give custom {@link TagNodeVisitor}s a chance
 * to transform the selected content.
 * </p>
 */
public class SimpleReverseProxyPortlet extends GenericReverseProxyPortlet
{

    /**
     * Parameter name for the XPath expression to select only the specified DOM element
     * after retrieving the remote content.
     */
    private static final String XPATH_EXPRESSION_PARAM_NAME = "xpath.expression";

    /**
     * Parameter name for the flag whether or not the inner elements should be selected by the XPath expression.
     */
    private static final String INNER_ELEMENTS_ONLY_PARAM_NAME = "inner.elements.only";

    /**
     * Default value of the flag whether or not the inner elements should be selected by the XPath expression.
     */
    private static final boolean DEFAULT_INNER_ELEMENTS_ONLY_VALUE = true;

    /**
     * Parameter name for the <code>HtmlCleander</code> transformation rule.
     */
    private static final String HTML_CLEANER_TRANSFORMATION_PARAM_NAME = "html.cleaner.transformation";

    /**
     * Internal reverse proxy mapping registry.
     */
    private ProxyMappingRegistry proxyMappingRegistry;

    /**
     * Internal {@link HttpClientBuilder} instance.
     */
    private HttpClientBuilder httpClientBuilder;

    /**
     * Internal {@link HttpClientContextBuilder} instance.
     */
    private HttpClientContextBuilder httpClientContextBuilder;

    /**
     * SSL Socket Connection Factory.
     */
    private LayeredConnectionSocketFactory sslSocketFactory;

    /**
     * SSL Hostname Verifier.
     */
    private X509HostnameVerifier hostnameVerifier = SSLConnectionSocketFactory.BROWSER_COMPATIBLE_HOSTNAME_VERIFIER;

    /**
     * Internal root reverse proxy processing chain instance.
     */
    private ProxyProcessingChain proxyServiceCommand;

    /**
     * Zero-argument default constructor.
     */
    public SimpleReverseProxyPortlet()
    {
        super();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void init(PortletConfig portletConfig) throws PortletException
    {
        super.init(portletConfig);
        initHostnameVerifier(portletConfig);

        proxyServiceCommand = createProxyServiceCommand();
        ProxyCommandUtils.initializeAllCommands(proxyServiceCommand);

        DefaultReverseProxyService proxyService = new DefaultReverseProxyService(proxyServiceCommand);
        setProxyService(proxyService);
    }

    /**
     * Initializes SSL hostnameVerifier (type of {@link X509HostnameVerifier})
     * from the init parameter, "ssl-hostname-verifier".
     * <p>
     * The init parameter can be any of "ALLOW_ALL_HOSTNAME_VERIFIER", "BROWSER_COMPATIBLE_HOSTNAME_VERIFIER"
     * or "STRICT_HOSTNAME_VERIFIER", case-insensitively. The default value is "BROWSER_COMPATIBLE_HOSTNAME_VERIFIER".
     * </p>
     * @param servletConfig
     */
    protected void initHostnameVerifier(PortletConfig portletConfig) {
        String param = StringUtils.trim(portletConfig.getInitParameter("ssl-hostname-verifier"));

        if (StringUtils.isNotEmpty(param))
        {
            if (StringUtils.equalsIgnoreCase("ALLOW_ALL_HOSTNAME_VERIFIER", param))
            {
                setHostnameVerifier(SSLConnectionSocketFactory.ALLOW_ALL_HOSTNAME_VERIFIER);
            }
            else if (StringUtils.equalsIgnoreCase("BROWSER_COMPATIBLE_HOSTNAME_VERIFIER", param))
            {
                setHostnameVerifier(SSLConnectionSocketFactory.BROWSER_COMPATIBLE_HOSTNAME_VERIFIER);
            }
            else if (StringUtils.equalsIgnoreCase("STRICT_HOSTNAME_VERIFIER", param))
            {
                setHostnameVerifier(SSLConnectionSocketFactory.STRICT_HOSTNAME_VERIFIER);
            }
        }
    }

    /**
     * Returns the internal {@link ProxyMappingRegistry} instance.
     * @return
     */
    public ProxyMappingRegistry getProxyMappingRegistry()
    {
        if (proxyMappingRegistry == null)
        {
            proxyMappingRegistry = new DefaultProxyMappingRegistry();
            proxyMappingRegistry.addProxyMapping(getProxyMapping());
        }

        return proxyMappingRegistry;
    }

    /**
     * Sets the internal {@link ProxyMappingRegistry} instance.
     * @param proxyMappingRegistry
     */
    public void setProxyMappingRegistry(ProxyMappingRegistry proxyMappingRegistry)
    {
        this.proxyMappingRegistry = proxyMappingRegistry;
    }

    /**
     * Returns the internal {@link HttpClientBuilder} instance.
     * @return
     */
    public HttpClientBuilder getHttpClientBuilder()
    {
        if (httpClientBuilder == null)
        {
            httpClientBuilder = createHttpClientBuilder();
        }

        return httpClientBuilder;
    }

    /**
     * Sets the internal {@link HttpClientBuilder} instance.
     * @param httpClientBuilder
     */
    public void setHttpClientBuilder(HttpClientBuilder httpClientBuilder)
    {
        this.httpClientBuilder = httpClientBuilder;
    }

    /**
     * Returns the internal {@link HttpClientContextBuilder} instance.
     * @return
     */
    public HttpClientContextBuilder getHttpClientContextBuilder()
    {
        return httpClientContextBuilder;
    }

    /**
     * Sets the internal {@link HttpClientContextBuilder} instance.
     * @param httpClientContextBuilder
     */
    public void setHttpClientContextBuilder(HttpClientContextBuilder httpClientContextBuilder)
    {
        this.httpClientContextBuilder = httpClientContextBuilder;
    }

    /**
     * Returns the internal SSL Socket Connection Factory ({@link LayeredConnectionSocketFactory}).
     * If not set, it invokes {@link #createSslSocketFactory()} to create a default one.
     * @return
     */
    public LayeredConnectionSocketFactory getSslSocketFactory()
    {
        if (sslSocketFactory == null)
        {
            sslSocketFactory = createSslSocketFactory();
        }

        return sslSocketFactory;
    }

    /**
     * Sets the internal SSL Socket Connection Factory ({@link LayeredConnectionSocketFactory}).
     * @param sslSocketFactory
     */
    public void setSslSocketFactory(LayeredConnectionSocketFactory sslSocketFactory)
    {
        this.sslSocketFactory = sslSocketFactory;
    }

    /**
     * Returns the internal SSL Hostname Verifier.
     * @return
     */
    public X509HostnameVerifier getHostnameVerifier()
    {
        return hostnameVerifier;
    }

    /**
     * Sets the internal SSL Hostname Verifier.
     * @param hostnameVerifier
     */
    public void setHostnameVerifier(X509HostnameVerifier hostnameVerifier)
    {
        this.hostnameVerifier = hostnameVerifier;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void destroy()
    {
        ProxyCommandUtils.destroyAllCommands(proxyServiceCommand);
        super.destroy();
    }

    /**
     * {@link #init(PortletConfig) method invokes this method to
     * create the root reverse proxy processing chain ({@link ProxyProcessingChain}).
     * @return
     */
    protected ProxyProcessingChain createProxyServiceCommand() {
        return ProxyServices.createDefault().build(getProxyMappingRegistry(), getHttpClientBuilder(), getHttpClientContextBuilder());
    }

    /**
     * Returns a {@link ProxyMapping} instance ({@link PortletAnyProxyMapping} by default).
     * @return
     */
    protected ProxyMapping getProxyMapping()
    {
        PortletAnyProxyMapping portletAnyProxyMapping = new PortletAnyProxyMapping(getPortletConfig());
        portletAnyProxyMapping.setContentRewriters(getContentRewriters());
        return portletAnyProxyMapping;
    }

    /**
     * Returns a map containing only a {@link ContentRewriter} for <code>text/html</code> content
     * by default. The {@link ContentRewriter} for HTML content is {@link HtmlCleanerContentRewriter}, by default,
     * which is implemented with <code>HtmlCleander</code> library.
     * @return
     */
    protected Map<String, ContentRewriter> getContentRewriters()
    {
        Map<String, ContentRewriter> contentRewriters = new HashMap<String, ContentRewriter>();

        DefaultSerializerFactory serializerFactory = new DefaultSerializerFactory();
        serializerFactory.setSerializerClass(SimpleHtmlSerializer.class);

        HtmlCleanerContentRewriter contentRewriter = new HtmlCleanerContentRewriter();
        contentRewriter.setSerializerFactory(serializerFactory);
        contentRewriter.setXpathExpression(getXpathExpression());
        contentRewriter.setInnerHtmlOnly(isInnerElementsOnly());

        String [] cleanerTransformations = getCleanerTransformationStringArray();

        if (!ArrayUtils.isEmpty(cleanerTransformations))
        {
            contentRewriter.setCleanerTransformationStringArray(cleanerTransformations);
        }

        List<TagNodeVisitor> tagNodeVisitors = getTagNodeVisitors();

        if (!tagNodeVisitors.isEmpty())
        {
            for (TagNodeVisitor visitor : tagNodeVisitors)
            {
                contentRewriter.addTagNodeVisitor(visitor);
            }
        }

        contentRewriters.put("text/html", contentRewriter);

        return contentRewriters;
    }

    /**
     * Returns the XPath expression to select only the specific DOM element and its descendants from the remote content document.
     * If not set, it returns <code>//body</code> by default.
     * @return
     */
    protected String getXpathExpression()
    {
        String expression = getPortletConfig().getInitParameter(XPATH_EXPRESSION_PARAM_NAME);

        if (expression == null || StringUtils.isBlank(expression))
        {
            return "//body";
        }

        return expression;
    }

    /**
     * Returns the flag whether or not only the inner elements should be selected by the XPath expression.
     * @return
     */
    protected boolean isInnerElementsOnly() {
        String param = getPortletConfig().getInitParameter(INNER_ELEMENTS_ONLY_PARAM_NAME);

        if (StringUtils.isBlank(param)) {
            return DEFAULT_INNER_ELEMENTS_ONLY_VALUE;
        } else {
            return BooleanUtils.toBoolean(param);
        }
    }

    /**
     * Returns <code>HtmlCleaner</code> transformation string array.
     * @return
     */
    protected String [] getCleanerTransformationStringArray()
    {
        String param = getPortletConfig().getInitParameter(HTML_CLEANER_TRANSFORMATION_PARAM_NAME);

        if (param == null || StringUtils.isBlank(param))
        {
            return ArrayUtils.EMPTY_STRING_ARRAY;
        }

        String [] tokens = StringUtils.split(param, "|");

        for (int i = 0; i < tokens.length; i++)
        {
            tokens[i] = StringUtils.trim(tokens[i]);
        }

        return tokens;
    }

    /**
     * Returns a list of {@link TagNodeVisitor}s which are given a chance to visit all the tag nodes.
     * Custom {@TagNodeVisitor}s can be used to translate the selected content from the remote content.
     * @return
     */
    protected List<TagNodeVisitor> getTagNodeVisitors()
    {
        List<TagNodeVisitor> visitors = new ArrayList<TagNodeVisitor>();
        visitors.add(new DefaultPortletProxyTagNodeVisitor());
        return visitors;
    }

    /**
     * Creates {@link HttpClientBuilder} instance.
     * This method is invoked by {@link #getHttpClientBuilder()}
     * if an {@link HttpClientBuilder} instance was not set before.
     * @return
     */
    protected HttpClientBuilder createHttpClientBuilder() {
        HttpClientBuilder builder = HttpClients.custom().useSystemProperties();

        LayeredConnectionSocketFactory sslSocketFactory = getSslSocketFactory();

        if (sslSocketFactory != null)
        {
            builder.setSSLSocketFactory(sslSocketFactory);
        }

        return builder;
    }

    /**
     * Creates the default internal SSL Socket Connection Factory ({@link LayeredConnectionSocketFactory}).
     * @return
     */
    protected LayeredConnectionSocketFactory createSslSocketFactory()
    {
        return new SSLConnectionSocketFactory(SSLContexts.createDefault(), getHostnameVerifier());
    }
}
