/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent2.proxy.util;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

import java.io.IOException;
import java.net.URI;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.util.List;

import org.apache.portals.applications.webcontent2.proxy.ProxyMapping;
import org.apache.portals.applications.webcontent2.proxy.impl.RegexProxyMapping;
import org.apache.portals.applications.webcontent2.proxy.impl.SimpleProxyMapping;
import org.apache.portals.applications.webcontent2.rewriter.ContentRewritingContext;
import org.apache.portals.applications.webcontent2.rewriter.ContentRewritingException;
import org.apache.portals.applications.webcontent2.rewriter.impl.AbstractTextLineContentRewriter;
import org.junit.Test;

public class YamlConfigUtilsTest
{

    private static final String SIMPLE_MAPPINGS = "\n"
                    + "--- !simple\n"
                    + "local: /portals/applications/\n"
                    + "remote: http://portals.apache.org/applications/\n"
                    + "\n"
                    + "--- !simple\n"
                    + "local: /portals/bridges/\n"
                    + "remote: http://portals.apache.org/bridges/\n"
                    + "\n"
                    + "--- !simple\n"
                    + "local: /portals/\n"
                    + "remote: http://portals.apache.org/\n" 
                    + "\n"
                    + "--- !simple\n"
                    + "local: /\n"
                    + "remote: http://www.apache.org/\n"
                    + "contentRewriters:\n"
                    + "    text/html: !!" + NOOPTextLineContentRewriter.class.getName() + " []\n";

    private static final String REGEX_MAPPINGS = "\n"
                    + "--- !regex\n"
                    + "localPattern: ^/(\\w+)/(\\w+)/(.*)$\n"
                    + "remoteReplace: http://$1.apache.org/$2/$3\n"
                    + "remotePattern: ^http://(\\w+)\\.apache\\.org/(\\w+)/(.*)$\n"
                    + "localReplace: /$1/$2/$3\n"
                    + "\n"
                    + "--- !regex\n" 
                    + "localPattern: ^/(.*)$\n"
                    + "remoteReplace: http://www.apache.org/$1\n"
                    + "remotePattern: ^http://www.apache.org/(.*)$\n"
                    + "localReplace: /$1\n"
                    + "contentRewriters:\n"
                    + "    text/html: !!" + LineNumberingTextLineContentRewriter.class.getName() + " [ 00000 ]\n";

    @Test
    public void testSimpleMappings() throws Exception
    {
        List<ProxyMapping> mappings = YamlConfigUtils.loadProxyMappings(SIMPLE_MAPPINGS, null);
        assertEquals(4, mappings.size());

        for (ProxyMapping mapping : mappings)
        {
            assertTrue(mapping instanceof SimpleProxyMapping);
        }

        int index = 0;
        assertEquals("/portals/applications/", ((SimpleProxyMapping) mappings.get(index)).getLocal());
        assertEquals(URI.create("http://portals.apache.org/applications/"), ((SimpleProxyMapping) mappings.get(index)).getRemote());
        assertEquals("/portals/applications/a1/b1/c1",
                     mappings.get(index).resolveLocalFromRemote(URI.create("http://portals.apache.org/applications/a1/b1/c1")));
        assertEquals("http://portals.apache.org/applications/a2/b2/c2",
                     mappings.get(index).resolveRemoteFromLocal("/portals/applications/a2/b2/c2"));
        assertNull(mappings.get(index).getContentRewriter("text/html"));

        ++index;
        assertEquals("/portals/bridges/", ((SimpleProxyMapping) mappings.get(index)).getLocal());
        assertEquals(URI.create("http://portals.apache.org/bridges/"), ((SimpleProxyMapping) mappings.get(index)).getRemote());
        assertEquals("/portals/bridges/a1/b1/c1",
                     mappings.get(index).resolveLocalFromRemote(URI.create("http://portals.apache.org/bridges/a1/b1/c1")));
        assertEquals("http://portals.apache.org/bridges/a2/b2/c2",
                     mappings.get(index).resolveRemoteFromLocal("/portals/bridges/a2/b2/c2"));
        assertNull(mappings.get(index).getContentRewriter("text/html"));

        ++index;
        assertEquals("/portals/", ((SimpleProxyMapping) mappings.get(index)).getLocal());
        assertEquals(URI.create("http://portals.apache.org/"), ((SimpleProxyMapping) mappings.get(index)).getRemote());
        assertEquals("/portals/a1/b1/c1",
                     mappings.get(index).resolveLocalFromRemote(URI.create("http://portals.apache.org/a1/b1/c1")));
        assertEquals("http://portals.apache.org/a2/b2/c2",
                     mappings.get(index).resolveRemoteFromLocal("/portals/a2/b2/c2"));
        assertNull(mappings.get(index).getContentRewriter("text/html"));

        ++index;
        assertEquals("/", ((SimpleProxyMapping) mappings.get(index)).getLocal());
        assertEquals(URI.create("http://www.apache.org/"), ((SimpleProxyMapping) mappings.get(index)).getRemote());
        assertEquals("/a1/b1/c1",
                     mappings.get(index).resolveLocalFromRemote(URI.create("http://www.apache.org/a1/b1/c1")));
        assertEquals("http://www.apache.org/a2/b2/c2",
                     mappings.get(index).resolveRemoteFromLocal("/a2/b2/c2"));
        assertNotNull(mappings.get(index).getContentRewriter("text/html"));
        assertNotNull(mappings.get(index).getContentRewriter("text/html") instanceof NOOPTextLineContentRewriter);
    }

    @Test
    public void testRegexMappings() throws Exception
    {
        List<ProxyMapping> mappings = YamlConfigUtils.loadProxyMappings(REGEX_MAPPINGS, null);
        assertEquals(2, mappings.size());

        for (ProxyMapping mapping : mappings)
        {
            assertTrue(mapping instanceof RegexProxyMapping);
        }

        int index = 0;
        assertEquals("/portals/applications/a1/b1/c1",
                     mappings.get(index).resolveLocalFromRemote(URI.create("http://portals.apache.org/applications/a1/b1/c1")));
        assertEquals("http://portals.apache.org/applications/a2/b2/c2",
                     mappings.get(index).resolveRemoteFromLocal("/portals/applications/a2/b2/c2"));

        assertEquals("/portals/bridges/a1/b1/c1",
                     mappings.get(index).resolveLocalFromRemote(URI.create("http://portals.apache.org/bridges/a1/b1/c1")));
        assertEquals("http://portals.apache.org/bridges/a2/b2/c2",
                     mappings.get(index).resolveRemoteFromLocal("/portals/bridges/a2/b2/c2"));

        assertEquals("/portals/a1/b1/c1",
                     mappings.get(index).resolveLocalFromRemote(URI.create("http://portals.apache.org/a1/b1/c1")));
        assertEquals("http://portals.apache.org/a2/b2/c2",
                     mappings.get(index).resolveRemoteFromLocal("/portals/a2/b2/c2"));

        assertNull(mappings.get(index).getContentRewriter("text/html"));

        ++index;
        assertEquals("/a1/b1/c1",
                     mappings.get(index).resolveLocalFromRemote(URI.create("http://www.apache.org/a1/b1/c1")));
        assertEquals("http://www.apache.org/a2/b2/c2",
                     mappings.get(index).resolveRemoteFromLocal("/a2/b2/c2"));
        assertNotNull(mappings.get(index).getContentRewriter("text/html"));
        assertNotNull(mappings.get(index).getContentRewriter("text/html") instanceof LineNumberingTextLineContentRewriter);
    }

    public static class NOOPTextLineContentRewriter extends AbstractTextLineContentRewriter
    {
        @Override
        protected String rewriteLine(String line, ContentRewritingContext context) throws ContentRewritingException, IOException
        {
            return line;
        }
    }

    public static class LineNumberingTextLineContentRewriter extends AbstractTextLineContentRewriter
    {
        private int lineNum = 0;
        private NumberFormat numberFormat;

        public LineNumberingTextLineContentRewriter(String numberingFormat)
        {
            numberFormat = new DecimalFormat(numberingFormat);
        }

        @Override
        protected String rewriteLine(String line, ContentRewritingContext context) throws ContentRewritingException, IOException
        {
            return "" + numberFormat.format(++lineNum) + ": " + line;
        }
    }
}
