/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.portals.applications.webcontent2.rewriter.impl;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;

import org.apache.portals.applications.webcontent2.rewriter.Source;

/**
 * Acts as an holder for a content rewriting Source in the form of a byte or character stream.
 * <p>
 * <em>Note:</em> Due to their internal use of either a Reader or InputStream instance,
 * <code>StreamSource</code> instances may only be used once.
 * </p>
 */
public class StreamSource implements Source
{

    /**
     * The underlying {@link InputStream} which can be created once by calling {@link #getInputStream()}.
     */
    private InputStream input;

    /**
     * Character encoding to be used when {@link #getReader()} is invoked to create a
     * reader from the underlying gzipped byte stream.
     */
    private String characterEncoding;

    /**
     * The underlying {@link Reader} which can be created once by calling {@link #getReader()}.
     */
    private Reader reader;

    /**
     * Construct a StreamSource from a byte stream.
     * @param input
     */
    public StreamSource(InputStream input)
    {
        this(input, null);
    }

    /**
     * Construct a StreamSource from a byte stream with the specified <code>encoding</code>.
     * The <code>encoding</code> is used when {@link #getReader()} is invoked to create a
     * reader from the underlying byte stream.
     * @param input
     * @param encoding
     */
    public StreamSource(InputStream input, String characterEncoding)
    {
        if (input == null)
        {
            throw new IllegalArgumentException("input must not be null.");
        }

        this.input = input;
        this.characterEncoding = characterEncoding;
    }

    /**
     * Construct a StreamSource from a character reader.
     * @param reader
     */
    public StreamSource(Reader reader)
    {
        if (reader == null)
        {
            throw new IllegalArgumentException("reader must not be null.");
        }

        this.reader = reader;
    }

    /**
     * {@inheritDoc}
     * <P>
     * Either this method or {@link #getReader} may be called to read the body, not both.
     * </P>
     */
    public InputStream getInputStream() throws IOException
    {
        if (reader != null)
        {
            throw new IllegalStateException("getReader() has already been called on this source.");
        }

        return input;
    }

    /**
     * {@inheritDoc}
     * <P>
     * Either this method or {@link #getInputStream} may be called to read the body, not both.
     * </P>
     */
    public Reader getReader() throws IOException
    {
        if (input != null)
        {
            throw new IllegalStateException("getInputStream() has already been called on this source.");
        }

        if (reader == null)
        {
            if (characterEncoding == null)
            {
                reader = new InputStreamReader(input);
            }
            else
            {
                reader = new InputStreamReader(input, characterEncoding);
            }
        }

        return reader;
    }

}
