// Copyright ©2015 The Gonum Authors. All rights reserved.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

package testlapack

import (
	"math"
	"testing"

	"golang.org/x/exp/rand"

	"gonum.org/v1/gonum/floats"
)

type Dlasq2er interface {
	Dgetrfer
	Dlasq2(n int, z []float64) (info int)
}

func Dlasq2Test(t *testing.T, impl Dlasq2er) {
	dTol := 1e-6
	// Answers from calling the netlib Dlasq2 function directly.
	for c, test := range []struct {
		n    int
		z    []float64
		info int
		zOut []float64
	}{
		{
			z:    []float64{0.9975595900926172E+00, 0.5668247076112733E+00, 0.9659153754961249E+00, 0.7479276854714322E+00, 0.3673908973747557E+00, 0.4806368987547315E+00, 0.7375426363398452E-01, 0.5355229277727247E-02, 0.3470812885180155E+00, 0.3422438160728350E+00, 0.2179517263384726E+00, 0.1331604100136593E+00, 0.9005245144218924E+00, 0.3867660104574036E+00, 0.4454822893878481E+00, 0.6619321808958428E+00, 0.1610830043055933E-01, 0.6508548361039168E+00, 0.6464088254838254E+00, 0.3229872909405558E+00, 0.8556924028853313E+00, 0.4012869193638139E+00, 0.2068743292187569E+00, 0.9685394642165999E+00, 0.5983995343181346E+00, 0.6729807327762632E+00, 0.4568823106729620E+00, 0.3300151235733747E+00, 0.1003829265021773E+00, 0.7554533047597268E+00, 0.6056932669802318E+00, 0.7190479134084530E+00, 0.8973346038865299E+00, 0.6582291204811077E+00, 0.1507168371301046E+00, 0.6123149049967000E+00, 0.9786602381321465E+00, 0.9991422694268032E+00, 0.2567979861831603E+00, 0.5508654031552206E+00, 0.6590475178936379E+00, 0.5540051353968156E+00, 0.9777600986094505E+00, 0.9019233044604550E+00, 0.6579246844202109E+00, 0.7288585083995637E+00, 0.4024552650632751E+00, 0.9286276631540373E+00, 0.1478351900438915E+00, 0.6745292990637881E+00, 0.7696143092758962E+00, 0.3393225521457851E+00, 0.1158188549489931E+00, 0.6143691843615144E+00, 0.8206171394658319E+00, 0.9470946559240657E+00, 0.7311286518535550E+00, 0.4976039064630389E+00, 0.3748017407872005E+00, 0.4215058560045335E+00, 0.5529030382557849E+00, 0.9979192791781042E+00, 0.9903947480415133E+00, 0.7463096553073095E+00, 0.9537590617479789E+00, 0.9327469033343772E-01, 0.7340236871158429E+00, 0.7517616159256260E+00, 0.9468485056759987E+00, 0.7061763632511533E+00, 0.8138096664824992E+00, 0.5585945195965548E+00, 0.6170557687507783E-01, 0.4803807801853603E+00, 0.5976897721963292E+00, 0.1375319188363359E+00, 0.5873952004066820E+00, 0.5199682629163997E+00, 0.8858783448129579E+00, 0.3038101719904888E+00, 0.6696573039722583E+00, 0.6649400915297697E+00, 0.5036768993979404E+00, 0.2615751204119269E+00},
			n:    21,
			zOut: []float64{0.2550031738262204E+01, 0.2480977513501848E+01, 0.2366602269912148E+01, 0.2169993432366266E+01, 0.1810817848712033E+01, 0.1681677309759878E+01, 0.1303743375476047E+01, 0.1202161769544433E+01, 0.1183377192742860E+01, 0.1094206688544886E+01, 0.9154376259418607E+00, 0.8241395430971566E+00, 0.7318275341991307E+00, 0.5186198053161721E+00, 0.4143051093784424E+00, 0.2112065329503869E+00, 0.1093987140067686E+00, 0.9751785856405315E-01, 0.2818174710670554E-01, 0.1697846193036144E-01, 0.2117542506861687E-04, 0.5241368559131172E-22, 0.4369342452764876E-18, 0.4084491703488284E-18, 0.1303743375476047E+01, 0.0000000000000000E+00, 0.2325140442247288E-18, 0.2353545363971710E-18, 0.1202161769544433E+01, 0.0000000000000000E+00, 0.2163565864913247E-18, 0.2222159192042978E-18, 0.1183377192742860E+01, 0.2033579915108999E-22, 0.2652482201353177E-18, 0.2438256017573345E-18, 0.1094206688544886E+01, 0.3167883469916549E-16, 0.9806070694382835E-18, 0.1224486171222500E-17, 0.9154376259418607E+00, 0.2764930279233778E-14, 0.2171122324673871E+02, 0.2171122324673871E+02, 0.1020000000000000E+03, 0.3081632653061225E+01, 0.2941176470588236E+01, 0.2548071704976161E-16, 0.7318275341991307E+00, 0.4959852501050381E-23, 0.8676862906242004E-16, 0.1414000300319855E-15, 0.5186198053161721E+00, 0.1503295986001297E-14, 0.3452895805257589E-14, 0.7981257539768321E-14, 0.4143051093784424E+00, 0.1303656847202082E-15, 0.9855809897129084E-12, 0.2614986238618434E-11, 0.2112065329503869E+00, 0.8411884198867843E-17, 0.2293573303077261E-09, 0.9833566024906726E-10, 0.1093987140067686E+00, 0.2696165428113804E-17, 0.5825676764620370E-08, 0.2638661645752538E-08, 0.9751785856405315E-01, 0.6617412588681544E-21, 0.2809066689524106E-07, 0.6165545233233256E-07, 0.2818174710670554E-01, 0.1396210563637443E-18, 0.2854399814229969E-05, 0.1344352644992036E-05, 0.1697846193036144E-01, 0.6162975822039155E-31, 0.3724766654883956E-05, 0.4695180568393632E-05, 0.2117542506861687E-04, 0.0000000000000000E+00, 0.9895328911616120E-03, 0.1620493249248586E-02},
			info: 0,
		},
		{
			z:    []float64{0.7655950151081736E-01, 0.1012496627944287E+00, 0.5492657397218519E+00, 0.3755849474908193E+00, 0.1514950257902281E-01, 0.7929154460744389E+00, 0.6208775449015512E+00, 0.7736035774285512E+00, 0.9535807609862423E+00, 0.1142443721338974E+00, 0.3184626428755636E+00, 0.5968198462235605E+00, 0.4815290260558158E-01, 0.1142057780273510E+00, 0.2159649191761220E+00, 0.1005733924928321E+00, 0.7334180249993905E-01, 0.2468617397276878E+00, 0.4433842667962828E+00, 0.2083675732544600E+00, 0.5669983427348256E+00, 0.2431239969651688E-01, 0.4202905770712168E+00, 0.3978530241169832E+00, 0.9765854254920880E+00, 0.6926050329928320E+00, 0.4943367486777372E-02, 0.1299210324451839E+00, 0.4677725963527957E-01, 0.8397777412847708E+00, 0.6784888222113696E+00, 0.5819508167748705E+00, 0.7335259194567719E+00, 0.1160427446694747E+00, 0.8402996546275487E+00, 0.8349959735427709E+00, 0.7465363962886192E+00, 0.8432008930683078E+00, 0.5288390014350658E+00, 0.6654846664813405E+00, 0.7307365692392301E+00, 0.4106042636492306E+00, 0.3557215961646439E+00, 0.7353770423280176E+00, 0.4713176648251016E+00, 0.4626254343486430E+00, 0.7596917071958065E+00, 0.7024594192654096E+00, 0.2579658084846748E+00, 0.9377050325544740E+00, 0.4561035840049426E+00, 0.8084892970067921E+00, 0.9088480529888660E+00, 0.6948766633235142E+00, 0.2194885062923195E+00, 0.8549545559228027E+00, 0.7443966896835119E+00, 0.3011130612705175E+00, 0.6719685452983997E+00, 0.6187140363783860E+00, 0.9675736114028010E+00, 0.9902393027740470E+00, 0.3380065169449795E+00, 0.9207672475882130E+00, 0.3390733399571427E+00, 0.9309435300662920E+00, 0.5465285509796513E+00, 0.4655125893217942E+00, 0.1769140958718970E+00, 0.3779940975005719E+00, 0.1752206228227682E+00, 0.3568582675012224E+00, 0.6656764250906758E+00, 0.6185492680730227E+00, 0.4689472437795331E+00, 0.3162518610221317E+00, 0.2688799086902824E+00, 0.1999212438248075E+00, 0.4154279763213168E+00, 0.9793029133347251E+00, 0.5432115888768534E+00, 0.1295182752745038E+00, 0.8047416637896615E+00, 0.8458210244967665E+00},
			n:    21,
			zOut: []float64{0.2649710614371106E+01, 0.2321564474027070E+01, 0.2090779203479937E+01, 0.2001510674733695E+01, 0.1702794694134603E+01, 0.1077066053646038E+01, 0.1060140274732043E+01, 0.9894235909971354E+00, 0.9539342071687115E+00, 0.8046649468928653E+00, 0.7009142227469247E+00, 0.5502651835254770E+00, 0.4423863025187732E+00, 0.3697086796938907E+00, 0.1446051340026323E+00, 0.1110032523123295E+00, 0.7513603923341917E-01, 0.6860214953971246E-01, 0.2434478048112329E-01, 0.6989818532012803E-03, 0.7811996215926567E-04, 0.7156177943897596E-01, 0.2292199980814605E-03, 0.1899018778701386E-03, 0.1060140274732043E+01, 0.8660746506696473E-01, 0.1357005210961402E-39, 0.1331360138522907E-23, 0.9894235909971354E+00, 0.8804208964992894E-17, 0.8660746506696473E-01, 0.8679736700028205E-01, 0.9539342071687115E+00, 0.2761013168273541E-29, 0.1040577915698430E-22, 0.8107757314001177E-24, 0.8046649468928653E+00, 0.5098326619997980E-22, 0.1416795225784663E-10, 0.1216165317638265E-10, 0.7009142227469247E+00, 0.1183291357831518E-29, 0.1813933158005285E+02, 0.1813933158005285E+02, 0.1090000000000000E+03, 0.3231292517006803E+01, 0.5504587155963303E+01, 0.4068037876491279E-10, 0.4423863025187732E+00, 0.2477754901417239E-20, 0.1737857614720001E-07, 0.5295826057530262E-07, 0.3697086796938907E+00, 0.5143373102040997E-26, 0.1312431380925897E-05, 0.1614374370413396E-05, 0.1446051340026323E+00, 0.9928287808749566E-25, 0.3777515963415321E-05, 0.2773141909621761E-05, 0.1110032523123295E+00, 0.0000000000000000E+00, 0.2532463507333992E-05, 0.2934028940292093E-05, 0.7513603923341917E-01, 0.5394210206791908E-19, 0.6032617175984252E-07, 0.2575740214720034E-06, 0.6860214953971246E-01, 0.7542232825258426E-19, 0.3784397549471832E-09, 0.1131370986389306E-09, 0.2434478048112329E-01, 0.1013559914197709E-18, 0.1483665133446019E-05, 0.1887408451311279E-05, 0.6989818532012803E-03, 0.9466330862652142E-28, 0.2539111990097616E-03, 0.9240506410060376E-03, 0.7811996215926567E-04, 0.3786532345060857E-28, 0.1283811140869274E-03, 0.1684766835109566E-03},
			info: 0,
		},
		{
			z:    []float64{0.7090328374865308E+00, 0.5289732604915972E+00, 0.6638408676820399E+00, 0.9007392577685345E+00, 0.9411642674278379E+00, 0.1144202631330921E+00, 0.8666675899972089E+00, 0.6323680737151134E+00, 0.2427877245566508E+00, 0.1603957810123527E+00, 0.2123565414665629E-01, 0.1902539934369850E+00, 0.3123401294447815E+00, 0.8096475663099788E+00, 0.5737184899962203E+00, 0.8262606712006769E+00, 0.2864548736211023E+00, 0.1605147759074373E+00, 0.3309241142010387E+00, 0.2658533720189140E+00, 0.5447811280109880E+00, 0.5535203163154195E+00, 0.2161483615344606E-01, 0.1487480590324270E+00, 0.1049033762317957E+00, 0.6142137665037214E+00, 0.4062909815669834E+00, 0.9015282863354186E+00, 0.9428381000165521E+00, 0.1566198551488233E+00, 0.9369593797444957E+00, 0.1354729715149707E+00, 0.3712821832808397E+00, 0.3215112250898227E+00, 0.7149642455474942E+00, 0.7297077774959745E+00, 0.3785927874981971E+00, 0.8289344072747380E+00, 0.9536643756705742E+00, 0.3084192540817629E+00, 0.5797406195506623E+00, 0.2334488720392539E+00, 0.8444034680728870E+00, 0.2581324672603945E+00, 0.8397436353763532E+00, 0.5376562757207516E+00, 0.6215703771401371E+00, 0.4908247191212207E+00, 0.2848423854658637E+00, 0.2355591822832676E+00, 0.6513799258897619E+00, 0.6071703403704332E+00, 0.3981458512505542E+00, 0.8487946917355450E+00, 0.6376169839751418E+00, 0.2588233361358196E+00, 0.3588309885453386E+00, 0.5618301121344036E+00, 0.1214685095104872E+00, 0.9784693864155333E+00, 0.3832063578469370E+00, 0.7206987579838013E+00, 0.3582640246022534E+00, 0.9098815720988161E+00, 0.5436872318254895E+00, 0.9596753969896468E+00, 0.9294728975416107E+00, 0.6333208483475120E+00, 0.2956089747729942E+00, 0.1747395639573214E+00, 0.8425785962238860E+00, 0.6185896542974498E+00, 0.7058928872472775E+00, 0.2424399357290958E+00, 0.7979610473510427E+00, 0.3910140128664733E+00, 0.1099592891010870E+00, 0.7536879342329168E+00, 0.2979717756246736E-01, 0.7006607890734408E+00, 0.5252033671714352E+00, 0.7525598235593517E+00, 0.6476778890126167E+00, 0.8710847630132640E+00},
			n:    21,
			zOut: []float64{0.2486733497028020E+01, 0.2359071020584524E+01, 0.2276435093308226E+01, 0.2061969496713965E+01, 0.1674052496256098E+01, 0.1634406821090099E+01, 0.1324515540477209E+01, 0.1213036064691536E+01, 0.1142092570196908E+01, 0.8479728747156923E+00, 0.7723287409164926E+00, 0.6845846156476076E+00, 0.4690003902541462E+00, 0.3969394481970619E+00, 0.3209839311891622E+00, 0.1762493640751034E+00, 0.1552318325824889E+00, 0.1299582823757977E+00, 0.4568511390642456E-01, 0.2058742849236605E-01, 0.6687292092604351E-04, 0.0000000000000000E+00, 0.9075510178548721E-16, 0.1809089799756099E-15, 0.1324515540477209E+01, 0.1972152263052530E-28, 0.7207148989396829E-15, 0.1135837106477965E-14, 0.1213036064691536E+01, 0.6865185493883853E-20, 0.3862483738583121E-14, 0.5795548067006594E-14, 0.1142092570196908E+01, 0.1237761291638161E-15, 0.1761387669893985E-13, 0.1204467761940304E-13, 0.8479728747156923E+00, 0.5980780785048348E-20, 0.5961915529495600E-12, 0.8546407799162654E-12, 0.7723287409164926E+00, 0.9983823616477125E-26, 0.2019190149561986E+02, 0.2019190149561985E+02, 0.1120000000000000E+03, 0.3267573696145125E+01, 0.7142857142857143E+01, 0.2981196239393922E-10, 0.4690003902541462E+00, 0.1411215361455214E-23, 0.1291839253366726E-09, 0.2114581039065388E-09, 0.3969394481970619E+00, 0.3215018396446338E-24, 0.8992249144513522E-09, 0.1460279739604385E-08, 0.3209839311891622E+00, 0.5301710738582062E-20, 0.5956459688113839E-08, 0.9509260151006140E-08, 0.1762493640751034E+00, 0.3305918938014538E-18, 0.2145126200057182E-06, 0.1391189925457742E-06, 0.1552318325824889E+00, 0.0000000000000000E+00, 0.8222867095540534E-06, 0.8520250895881005E-06, 0.1299582823757977E+00, 0.1946507846527860E-19, 0.4299487364203470E-05, 0.2670451838837521E-05, 0.4568511390642456E-01, 0.1783600337601646E-17, 0.1688330183620908E-03, 0.2840434656464124E-03, 0.2058742849236605E-01, 0.2127174768075152E-17, 0.5266290662212281E-04, 0.3082204001196310E-04, 0.6687292092604351E-04, 0.4725269249209171E-22, 0.3479992598793783E-02, 0.7157534417412210E-03},
			info: 0,
		},
		{
			z:    []float64{0.8307212987388154E+00, 0.9571804405037874E+00, 0.2222674360399970E+00, 0.5794927951917870E+00, 0.9907605078898842E+00, 0.8399673535531696E+00, 0.1748054629244838E+00, 0.7095417514667766E+00, 0.3015507353664195E+00, 0.2504155371033889E-01, 0.9167396887269817E+00, 0.6888065712047682E+00, 0.7567577995550097E+00, 0.9327280452266753E+00, 0.5500991579404824E-01, 0.4989499454693689E+00, 0.5949898652751251E-01, 0.1718660951302574E+00, 0.6883620036166904E+00, 0.3066205390931208E+00, 0.7351901912310475E+00, 0.7521684164853142E+00, 0.3957210266430871E+00, 0.8778502662051324E-01, 0.4896297811280748E-01, 0.7793618951247419E+00, 0.6807766665219679E+00, 0.3115699624095645E+00, 0.5227646665950632E+00, 0.5057964692468060E+00, 0.4385260593432981E+00, 0.5876392780963172E+00, 0.2973252261444674E+00, 0.2948995727695043E+00, 0.7378444260331429E-01, 0.9749457382350615E+00, 0.7281626398300706E+00, 0.7580903878987629E+00, 0.3091762186458183E+00, 0.2464983240671905E+00, 0.2568872936019900E+00, 0.6180627613788815E+00, 0.9377882352337240E+00, 0.9921824152100049E+00, 0.2248192100106184E+00, 0.9472840045361628E+00, 0.8477853668436538E+00, 0.7001263233958666E+00, 0.5544440083201581E+00, 0.4415839019184687E+00, 0.2491818172148126E+00, 0.8976627554942691E+00, 0.1457298176556392E+00, 0.4345936198183369E+00, 0.4573762249453559E+00, 0.3093805347522776E+00, 0.4809246108862507E+00, 0.6897426752489948E+00, 0.8107988692352123E+00, 0.3921644162683642E+00, 0.2931487419364392E+00, 0.5759780604710760E+00, 0.7253418085569345E+00, 0.1427733376597652E+00, 0.3335555091947459E+00, 0.1603897948564823E+00, 0.5345943097231743E+00, 0.7233253065718669E+00, 0.2281802915301432E+00, 0.1990064435881517E+00, 0.9726196083044836E+00, 0.9935713768557323E+00, 0.8195201715007894E-01, 0.2508806553004002E+00, 0.2158489797899930E+00, 0.1957771185505375E+00, 0.8502651710165565E+00, 0.4468235108377239E-01, 0.8639211642125266E+00, 0.6713373034638015E-01, 0.4273173462824215E+00, 0.1373188375801981E+00, 0.1190874985301729E+00, 0.3059495456429208E+00},
			n:    21,
			zOut: []float64{0.2399198387954965E+01, 0.2388283868878418E+01, 0.2234728187505421E+01, 0.1947641030296732E+01, 0.1845273105794422E+01, 0.1837097911259683E+01, 0.1471118007292610E+01, 0.1105651020532553E+01, 0.9883044021137968E+00, 0.9478841363642468E+00, 0.8550293065516678E+00, 0.7566674361156733E+00, 0.5310790726859680E+00, 0.3571217714437427E+00, 0.2893653184719416E+00, 0.2534414197270934E+00, 0.2173582726294167E+00, 0.4908458037260943E-01, 0.1246498683649276E-01, 0.5862142085722914E-02, 0.4704104342516643E-04, 0.1939986690620616E-19, 0.1034652479755287E-16, 0.1536368559490732E-16, 0.1471118007292610E+01, 0.5364686198585281E-13, 0.6088846084221820E-07, 0.1529604999486430E-06, 0.1105651020532553E+01, 0.2903008131213323E-27, 0.1160127138141562E-06, 0.1335479603764263E-06, 0.9883044021137968E+00, 0.2125606020077514E-18, 0.5906967539735199E-07, 0.6675815740532400E-07, 0.9478841363642468E+00, 0.1331202777560457E-29, 0.4438231138534822E-07, 0.3948328125271971E-07, 0.8550293065516678E+00, 0.5860341995390921E-20, 0.2049270140595660E+02, 0.2049270140595660E+02, 0.9800000000000000E+02, 0.2893424036281179E+01, 0.4081632653061225E+01, 0.1377712740565197E-07, 0.5310790726859680E+00, 0.1537961403283359E-19, 0.8011211497963711E-08, 0.8400248580143560E-08, 0.3571217714437427E+00, 0.5389497704469953E-25, 0.6965765014245226E-08, 0.7336584331089061E-08, 0.2893653184719416E+00, 0.6081961202185573E-17, 0.9973730165755502E-08, 0.8562226724273748E-08, 0.2534414197270934E+00, 0.3644137891043716E-17, 0.1552441635408352E-06, 0.1605842922922618E-07, 0.2173582726294167E+00, 0.1710250442519154E-26, 0.2805940935843235E-04, 0.1153299285109565E-04, 0.4908458037260943E-01, 0.1140499340513312E-15, 0.3458275194991493E-03, 0.2689727098393103E-03, 0.1246498683649276E-01, 0.4437342591868191E-30, 0.1429141842628293E-03, 0.1683368406483595E-03, 0.5862142085722914E-02, 0.2515327803596613E-22, 0.2394538487567536E-03, 0.1603201011413965E-03, 0.4704104342516643E-04, 0.0000000000000000E+00, 0.2149753858259932E-02, 0.4092589741085703E-02},
			info: 0,
		},
		{
			z:    []float64{0.6132370003914160E+00, 0.6636632940312478E+00, 0.8959100063178993E+00, 0.7536793777540497E+00, 0.6298705159481864E+00, 0.2209880035966978E+00, 0.4838373503885766E+00, 0.6150203746616348E+00, 0.9876587724998732E+00, 0.5591876171910803E+00, 0.3708840600033242E+00, 0.3375947236522073E+00, 0.7454250822233153E+00, 0.5697349010621601E+00, 0.4561367007431339E+00, 0.6481896738286699E+00, 0.6835960923418801E+00, 0.1703905177923075E+00, 0.2730661115049011E+00, 0.4407624886449348E+00, 0.4577220273293221E+00, 0.3952566732937318E+00, 0.6883342106097436E+00, 0.2988887099704991E+00, 0.2052274263883085E+00, 0.1922842703083140E+00, 0.4189999528960880E+00, 0.6906452535299822E+00, 0.1405554539793709E+00, 0.1437412601859918E+00, 0.6624635742176765E+00, 0.5756752017566100E+00, 0.7749505183416766E+00, 0.7752965769666669E+00, 0.7653541054136069E+00, 0.8765107194498384E+00, 0.9921378985956986E-01, 0.3345156485504346E+00, 0.6450936562634260E+00, 0.1316181443467312E+00, 0.8075436662342752E+00, 0.5418723210737144E+00, 0.2657525890099199E+00, 0.4588749915888913E+00, 0.5109250246010762E+00, 0.5048342930643721E+00, 0.9617739307304302E+00, 0.5502351700731637E+00, 0.1896288106400514E+00, 0.2011189112848212E+00, 0.2487376305081674E+00, 0.9700914054633822E+00, 0.7763920500140777E+00, 0.2727004477756484E+00, 0.4969507403482072E+00, 0.3987515903118115E+00, 0.6132963384905279E+00, 0.8526865910258985E-01, 0.7370021348269777E+00, 0.8628692345603857E+00, 0.3410351221174988E+00, 0.6944408188735030E+00, 0.6418832405583447E+00, 0.8549561267667016E-01, 0.7079842121947655E+00, 0.5924596383322266E+00, 0.2327865765542259E+00, 0.7519783901092147E-01, 0.3243359400926964E+00, 0.9130914950306788E+00, 0.2042866226175709E+00, 0.4527677275849958E+00, 0.7271733772634871E+00, 0.2835797318864963E-01, 0.5170298378180002E+00, 0.7676612860726464E+00, 0.8943646165240225E+00, 0.9444900692816881E+00, 0.2898061819682256E+00, 0.5314658042534958E+00, 0.2328255902743269E+00, 0.3127766212762262E+00, 0.1415213232426512E+00, 0.3425808715425687E+00},
			n:    21,
			zOut: []float64{0.2568497247033635E+01, 0.2474290441512907E+01, 0.2254355189310800E+01, 0.1987099258783412E+01, 0.1595738223681455E+01, 0.1520583769371047E+01, 0.1349481184037943E+01, 0.1286943522977408E+01, 0.1233633359467258E+01, 0.1105388350935419E+01, 0.8741663391031235E+00, 0.7042784684853729E+00, 0.6809050564171195E+00, 0.5312535309083547E+00, 0.3765999563195902E+00, 0.2982885955048700E+00, 0.1761408438030470E+00, 0.1021950592815966E+00, 0.5149411253987173E-01, 0.2409819364774552E-01, 0.3292801347387554E-02, 0.7649046082865931E-16, 0.2073367755475779E-07, 0.1625592945762753E-07, 0.1349481184037943E+01, 0.1183291357831518E-29, 0.4660703060512034E-13, 0.1807862490656452E-12, 0.1286943522977408E+01, 0.1649665924998180E-25, 0.1294990598064480E-10, 0.5231829186377068E-10, 0.1233633359467258E+01, 0.8526291632353489E-16, 0.3101844497115867E-08, 0.1140775990076438E-07, 0.1105388350935419E+01, 0.9624103043696344E-28, 0.5663389137423196E-06, 0.4346359020206590E-06, 0.8741663391031235E+00, 0.3067428736041242E-19, 0.2119872350446936E+02, 0.2119872350446936E+02, 0.1030000000000000E+03, 0.3056689342403628E+01, 0.2912621359223301E+01, 0.2039882393707806E-06, 0.6809050564171195E+00, 0.0000000000000000E+00, 0.3469744030301619E-06, 0.2644131172050035E-06, 0.5312535309083547E+00, 0.3043881270837009E-21, 0.5541811479797519E-05, 0.3827690844980446E-05, 0.3765999563195902E+00, 0.1909043390634849E-27, 0.3412067971678679E-04, 0.3418356678006945E-04, 0.2982885955048700E+00, 0.1167175485134229E-15, 0.1390598737761422E-03, 0.1685020271483295E-03, 0.1761408438030470E+00, 0.3311298373357662E-16, 0.2687183242739271E-03, 0.2970561145148615E-03, 0.1021950592815966E+00, 0.3861046454850253E-20, 0.3877059717439437E-03, 0.4274183271611622E-03, 0.5149411253987173E-01, 0.3171094621243632E-24, 0.5614281929126997E-03, 0.6201503766753644E-03, 0.2409819364774552E-01, 0.5854827030937197E-31, 0.8580439238194122E-03, 0.9937175751003719E-03, 0.3292801347387554E-02, 0.1687712113428427E-19, 0.6404816195044538E-02, 0.3574263865391779E-02},
			info: 0,
		},
		{
			z:    []float64{0.1535280608392028E+00, 0.8492164242958209E+00, 0.9720508422713467E+00, 0.4409240313091006E+00, 0.7744413453832578E+00, 0.4562143200311385E+00, 0.8645004927526174E+00, 0.7279750979364787E+00, 0.6373358699411871E+00, 0.7240388495630282E+00, 0.6042124182518795E+00, 0.2629417492305242E-01, 0.2927658696806527E+00, 0.1971917075100318E+00, 0.9486620805007744E+00, 0.5318461733579811E+00, 0.1035922758398233E-01, 0.4521355876782263E+00, 0.9635828765019094E+00, 0.9861179636072996E+00, 0.3409131837267612E+00, 0.9592913933135179E+00, 0.3267081303471464E+00, 0.7042329006712417E+00, 0.7904477014671945E+00, 0.7235636505410947E+00, 0.7289557043239185E+00, 0.4453819137782510E+00, 0.3817792285714174E+00, 0.9222246883547156E+00, 0.7484233302016036E+00, 0.4267300781414476E+00, 0.2174272124496083E+00, 0.7294135193889231E+00, 0.7969150359898325E+00, 0.5584233158827989E+00, 0.8854167010119880E+00, 0.3831257236275667E+00, 0.2407966333760054E+00, 0.3150832526072347E+00, 0.4329046703236793E+00, 0.4798261053116015E-01, 0.4240797212710790E+00, 0.7373271503520912E+00, 0.8727856342792233E+00, 0.3079871491200393E+00, 0.8433176604823556E+00, 0.8160920845908722E+00, 0.4068006968345205E+00, 0.8649881745683563E+00, 0.8522504948442566E+00, 0.5120820890311298E+00, 0.7764141319706099E+00, 0.4928600452500731E+00, 0.2048536231221674E+00, 0.4451308747598662E+00, 0.4335546033385644E+00, 0.5471256397417612E-01, 0.7004960984988220E+00, 0.9192372420579460E+00, 0.3890133989146303E+00, 0.2488782917619309E+00, 0.3663470187625480E+00, 0.9986636866778884E+00, 0.6620792031393874E+00, 0.1797345205931197E+00, 0.8723761011602340E+00, 0.2439575243714007E+00, 0.1163919437436586E+00, 0.8182195270597429E+00, 0.5356403226658167E+00, 0.9880537276300997E+00, 0.4163057033602857E+00, 0.8081980088045720E+00, 0.2399383760782623E+00, 0.9702581513812658E+00, 0.3763181880939181E+00, 0.6334004896983517E+00, 0.1368995930243956E+00, 0.7684724492197631E+00, 0.1011989548860721E+00, 0.5519462180485674E+00, 0.8316516728108886E+00, 0.3772586994144635E+00},
			n:    21,
			zOut: []float64{0.2579297838383781E+01, 0.2566242251743589E+01, 0.2402206059499122E+01, 0.2314572748831938E+01, 0.2210246243389976E+01, 0.2073991263591532E+01, 0.1647852770173070E+01, 0.1458062583915517E+01, 0.1424747680105817E+01, 0.1229594627898745E+01, 0.8682818248784862E+00, 0.7800126440184885E+00, 0.7769870810337002E+00, 0.4018986220147585E+00, 0.3451236345722463E+00, 0.2059421010707241E+00, 0.1665395041671736E+00, 0.1384541468623413E+00, 0.4569625338134105E-01, 0.3572364569397507E-01, 0.7785678858960618E-04, 0.1325624140040768E-19, 0.1464254707155794E-03, 0.9424803187834284E-04, 0.1647852770173070E+01, 0.1564186274351218E-15, 0.3473328524848360E-05, 0.4592867365821283E-05, 0.1458062583915517E+01, 0.3231174267785264E-26, 0.1403657959312935E-04, 0.9634257574341414E-05, 0.1424747680105817E+01, 0.1020140793606197E-15, 0.1723203566561169E-05, 0.6067379842882497E-05, 0.1229594627898745E+01, 0.2663149152121406E-17, 0.1061013981937667E-04, 0.1674993264096194E-04, 0.8682818248784862E+00, 0.8099755562101574E-24, 0.2367155138201492E+02, 0.2367155138201491E+02, 0.1110000000000000E+03, 0.3174603174603174E+01, 0.5405405405405405E+01, 0.8398559432747079E-06, 0.7769870810337002E+00, 0.0000000000000000E+00, 0.6532651675838807E-05, 0.1393736055608104E-04, 0.4018986220147585E+00, 0.1488702756444268E-16, 0.4786924897797409E-03, 0.2484818164296786E-03, 0.3451236345722463E+00, 0.1393994753571567E-18, 0.7967956438667165E-06, 0.1566848631115136E-05, 0.2059421010707241E+00, 0.1872521839860791E-17, 0.1674828806776016E-04, 0.3829209207735075E-04, 0.1665395041671736E+00, 0.5127595883936577E-29, 0.1145979873394449E-04, 0.6763541499702777E-05, 0.1384541468623413E+00, 0.6162975822039155E-31, 0.1509781169748205E-03, 0.4934238518442497E-04, 0.4569625338134105E-01, 0.3815207438965640E-25, 0.3298388086972868E-02, 0.1367005987525658E-02, 0.3572364569397507E-01, 0.2773930797150972E-21, 0.3955985102398275E-03, 0.3433403686203228E-03, 0.7785678858960618E-04, 0.7790054198074833E-21, 0.1914938157793930E-01, 0.1330976133014081E-01},
			info: 0,
		},
		{
			z:    []float64{0.8677677383100214E+00, 0.6246622741727026E+00, 0.5523136647126087E+00, 0.2116103484147348E+00, 0.8227197076777661E+00, 0.3636019443697256E+00, 0.2305222034118074E+00, 0.3291143648196290E+00, 0.6800926583943011E+00, 0.4290450477605455E+00, 0.4523048516285796E+00, 0.9276084588261152E+00, 0.4462311938877914E+00, 0.6672491542978151E+00, 0.7808537049355564E+00, 0.7908271145868994E+00, 0.1977402045683964E+00, 0.2563794060302639E+00, 0.6768044668199432E+00, 0.8985309013607360E+00, 0.3750140409512910E+00, 0.2003663066929584E-01, 0.4020022629723622E+00, 0.9909625073113403E+00, 0.9747495116121061E+00, 0.9702779584867087E+00, 0.9688371740876045E+00, 0.3747269639045615E+00, 0.9728403214040160E-01, 0.7507234588709335E+00, 0.8245099981459856E+00, 0.7244448753186800E+00, 0.3670523289486082E+00, 0.6220520718362722E+00, 0.3063226611493952E+00, 0.3012769332894331E+00, 0.6763821017483275E+00, 0.8300640433662996E+00, 0.5563237542291373E+00, 0.5594449054752113E+00, 0.3278038518373678E+00, 0.7307041435680680E+00, 0.1582529760475658E+00, 0.5328751876026443E+00, 0.7841944053171677E+00, 0.2157818394558657E+00, 0.3253134136288985E+00, 0.5014502641306768E+00, 0.2229178930043155E+00, 0.4664985859426845E+00, 0.1867987929192785E+00, 0.1951091673564507E+00, 0.2069185156156168E+00, 0.9058058542218750E+00, 0.7999188612304056E+00, 0.7349484002829904E+00, 0.4153480749961962E+00, 0.8109879258422723E+00, 0.9883741464101338E+00, 0.4911202582082937E+00, 0.2193166361224579E+00, 0.7618820222564749E+00, 0.9073087210515056E+00, 0.5896383620860655E+00, 0.8706198942119541E+00, 0.3860710050052940E-01, 0.9380336544251486E+00, 0.4690283060543987E-01, 0.8970820572284144E+00, 0.1974579201509297E-01, 0.9992442335759629E+00, 0.9801967887432729E+00, 0.6116617864086051E+00, 0.5875675950916066E+00, 0.9622537584002233E+00, 0.8934200351761762E+00, 0.2022182729782195E+00, 0.1273316669652236E-01, 0.2549904925850485E+00, 0.8904280763656908E+00, 0.2695748922419082E+00, 0.8890196843116763E+00, 0.5191439214393291E+00, 0.3672061831861048E+00},
			n:    21,
			zOut: []float64{0.3071910136434907E+01, 0.2366684893081943E+01, 0.2247985022656176E+01, 0.2134160618983571E+01, 0.1929380862422316E+01, 0.1895668448583188E+01, 0.1735809456784568E+01, 0.1390623822795548E+01, 0.1372853012616850E+01, 0.1197489438156041E+01, 0.1106600205023489E+01, 0.8564547950868329E+00, 0.4949238197552136E+00, 0.3889452841496929E+00, 0.3405947975938995E+00, 0.2516597920662951E+00, 0.2113890461921598E+00, 0.1169778386549868E+00, 0.1072268106836703E+00, 0.7646784687889400E-02, 0.1286588928027629E-02, 0.1194767075191949E-22, 0.1162094363453686E-09, 0.8691558890693288E-10, 0.1735809456784568E+01, 0.8134816760001801E-14, 0.1019665656815186E-08, 0.1906916241197346E-08, 0.1390623822795548E+01, 0.0000000000000000E+00, 0.1594323424098234E-05, 0.3312168640703357E-05, 0.1372853012616850E+01, 0.0000000000000000E+00, 0.3830524268846776E-04, 0.1577534719508939E-04, 0.1197489438156041E+01, 0.1592850300841939E-16, 0.1256296423282912E-03, 0.1149042579564614E-03, 0.1106600205023489E+01, 0.6530365529382261E-18, 0.2322627147533726E+02, 0.2322627147533726E+02, 0.1020000000000000E+03, 0.2963718820861678E+01, 0.3921568627450980E+01, 0.6483516393063832E-03, 0.4949238197552136E+00, 0.1972152263052530E-28, 0.5430985229870038E-04, 0.4447663749968261E-04, 0.3889452841496929E+00, 0.1329049288162406E-17, 0.2738942590248176E-04, 0.1582812372822289E-04, 0.3405947975938995E+00, 0.1187508759187252E-18, 0.7465410105999781E-04, 0.1255487400488340E-03, 0.2516597920662951E+00, 0.4141519752410312E-29, 0.2152266118694742E-04, 0.6710408995051273E-05, 0.2113890461921598E+00, 0.3333294884283764E-18, 0.6132111109262542E-03, 0.4277764486586744E-03, 0.1169778386549868E+00, 0.2711709361697228E-30, 0.3832079371139971E-03, 0.3904887652242632E-03, 0.1072268106836703E+00, 0.2176254805972020E-18, 0.4306247975784355E-03, 0.4948147025217374E-03, 0.7646784687889400E-02, 0.1110488679882733E-19, 0.1569670832436511E-02, 0.2092370053382882E-02, 0.1286588928027629E-02, 0.5442848430598922E-18, 0.4460307636637516E-02, 0.7046212350403941E-02},
			info: 0,
		},
		{
			z:    []float64{0.9838241499564321E+00, 0.8769514477589457E+00, 0.6814467524843054E+00, 0.3624618995089947E+00, 0.1981908649829056E+00, 0.1872278407669582E+00, 0.9876554444003832E+00, 0.1826583579385833E+00, 0.1183269767614670E-01, 0.9437175277915953E+00, 0.4919320768587329E+00, 0.9805291749478650E+00, 0.6132842150855770E+00, 0.3626870797541525E+00, 0.5323780753704027E+00, 0.1457937981484730E+00, 0.1328300737138376E+00, 0.3811835111034733E+00, 0.7003413504474413E+00, 0.5497308136521228E+00, 0.8255713255718768E+00, 0.5373393799690840E+00, 0.6623327475759663E+00, 0.5316913920782962E-01, 0.1714649183335913E+00, 0.6186592957613974E+00, 0.7587327126921758E+00, 0.9100586255702885E+00, 0.7140000863381626E+00, 0.7400119974078728E+00, 0.3567025022071233E+00, 0.8218612151637966E+00, 0.5520556331010515E+00, 0.6508027169553372E+00, 0.1875664464519897E+00, 0.8183092398415237E+00, 0.2239662369231676E-01, 0.6571790154721793E+00, 0.1196415656369646E+00, 0.3853512775546503E+00, 0.2426450009067070E+00, 0.9653447575661293E+00, 0.4852906340727067E+00, 0.9923496603563310E+00, 0.2812811891225394E+00, 0.9784300144088514E+00, 0.9528689986126391E+00, 0.2431080020200448E+00, 0.4166208849723857E+00, 0.7357806826601304E+00, 0.1952831632199464E+00, 0.6393407699957093E+00, 0.6296804002658308E+00, 0.1028694374663701E+00, 0.1387452591062114E+00, 0.4549213107961220E+00, 0.9843671628976580E+00, 0.4249152858158138E+00, 0.7048043599695387E+00, 0.3653998904622074E+00, 0.2003765309007774E+00, 0.4095912484595889E+00, 0.2392233396621621E+00, 0.7050044605645847E+00, 0.7908086424110450E-01, 0.2038177493661538E+00, 0.1735168081909322E-01, 0.3343100300667304E+00, 0.2848572906345944E+00, 0.8454138442968477E+00, 0.4944340369822147E+00, 0.7101083458004501E+00, 0.2546477176633619E+00, 0.8227235336957786E+00, 0.2816146807741101E+00, 0.6247995550265835E+00, 0.7458069042729408E+00, 0.8739345864578790E+00, 0.1278937298671606E+00, 0.2791833532060612E+00, 0.9234762152279486E+00, 0.9863012584972185E+00, 0.7892913846591531E+00, 0.3479266835120939E+00},
			n:    21,
			zOut: []float64{0.2623495482601064E+01, 0.2337512212773574E+01, 0.2310508039367694E+01, 0.2188663096012427E+01, 0.1954141717332891E+01, 0.1371476701077329E+01, 0.1267869950064541E+01, 0.1203016320141301E+01, 0.1132845050819559E+01, 0.9979737678355206E+00, 0.8633053234635751E+00, 0.6603430282082061E+00, 0.6064350433288983E+00, 0.4855760962077864E+00, 0.4280373135604714E+00, 0.3659370670152315E+00, 0.1453682093766173E+00, 0.9617861239701422E-01, 0.7192949634365248E-01, 0.1892143303127712E-02, 0.3946528731286962E-05, 0.0000000000000000E+00, 0.9258243476772570E-07, 0.9000418438492766E-07, 0.1267869950064541E+01, 0.0000000000000000E+00, 0.2500234135491857E-07, 0.1856395032302831E-07, 0.1203016320141301E+01, 0.2658426139847422E-18, 0.4168972835699797E-07, 0.5531467635213317E-07, 0.1132845050819559E+01, 0.2518044009465470E-24, 0.1194059466661935E-08, 0.2159625808467239E-08, 0.9979737678355206E+00, 0.1047506307937635E-23, 0.4478743649191460E-07, 0.9991281120631680E-07, 0.8633053234635751E+00, 0.5895762127714383E-22, 0.2111250861775921E+02, 0.2111250861775921E+02, 0.9900000000000000E+02, 0.2927437641723356E+01, 0.4040404040404041E+01, 0.9244672049175682E-07, 0.6064350433288983E+00, 0.1817535525629211E-26, 0.9398974933478891E-07, 0.9547946669183887E-07, 0.4855760962077864E+00, 0.3185418565406740E-17, 0.9937648506460343E-07, 0.9792626502078532E-07, 0.4280373135604714E+00, 0.2958228394578794E-30, 0.9921525775570031E-07, 0.1006425128472401E-06, 0.3659370670152315E+00, 0.4461863922899316E-21, 0.1047488573169732E-06, 0.1032270990738791E-06, 0.1453682093766173E+00, 0.6504493884088473E-15, 0.1102576940374778E-06, 0.1180293023862048E-06, 0.9617861239701422E-01, 0.6467797587852522E-18, 0.1844536574504796E-06, 0.1488485108626942E-06, 0.7192949634365248E-01, 0.2114103686638959E-13, 0.8531410278849863E-06, 0.1408572284551695E-05, 0.1892143303127712E-02, 0.0000000000000000E+00, 0.6242841403373791E-04, 0.3522115697192072E-04, 0.3946528731286962E-05, 0.6203854594147708E-24, 0.1209929086462260E-02, 0.4906802871097585E-03},
			info: 0,
		},
		{
			z:    []float64{0.7643247442799757E+00, 0.1930914554442843E+00, 0.6441117687067544E-01, 0.4522061669974708E-01, 0.1465182320932497E+00, 0.8069033698425149E+00, 0.6560188128523874E+00, 0.2446749124420051E+00, 0.3243649544497533E+00, 0.9836733438970116E+00, 0.6137527621157332E+00, 0.2925225554635034E+00, 0.4955619768673408E+00, 0.7361289066349539E+00, 0.3953262269487445E-01, 0.1565854250658760E-01, 0.4785236355428808E+00, 0.7766165270899886E+00, 0.9069259340739989E+00, 0.9151936268801151E+00, 0.3338292872636017E+00, 0.2993135005490550E+00, 0.8287022194584887E+00, 0.4593628480546942E+00, 0.1484428934315618E-01, 0.3390977660295040E+00, 0.5819746692101802E-01, 0.9277352020779983E-01, 0.6517310862845033E+00, 0.3951866136249272E+00, 0.6199300068296295E+00, 0.8252199884043853E+00, 0.3390764160478065E+00, 0.3084679704183562E+00, 0.3609211283483548E+00, 0.4116857210304998E+00, 0.4609476939442378E+00, 0.3824129844545763E+00, 0.6431169626236315E+00, 0.2416519334621993E+00, 0.7156780768158061E+00, 0.8568018406565006E+00, 0.1639408452444666E+00, 0.7313040585903831E+00, 0.5099183572592548E+00, 0.2801641590611897E+00, 0.9930562568266591E+00, 0.7612468327164370E+00, 0.3271886985411110E-01, 0.7066605591101006E+00, 0.6720867845388043E+00, 0.9004209067318458E-01, 0.4611918168927077E+00, 0.2638384667560661E+00, 0.3554909044606102E+00, 0.7231384727137510E+00, 0.2729268177075039E-01, 0.5848397641916390E+00, 0.6942450598380807E+00, 0.1106927451590289E+00, 0.5262572286481777E+00, 0.9936768911750095E+00, 0.7262536107933111E+00, 0.7604873714570974E+00, 0.2036182130165832E+00, 0.8632527217998969E+00, 0.1924957421132846E+00, 0.8815968660524154E-01, 0.2412813036310424E+00, 0.9105232147744077E+00, 0.8298587923387578E+00, 0.7754575363227978E-01, 0.8714764594177653E+00, 0.5571901679066804E+00, 0.3971870017409277E+00, 0.9993097753535422E+00, 0.7700188328643205E+00, 0.7856615172810383E+00, 0.7533953600915489E+00, 0.5602208064288483E+00, 0.6944234372397261E+00, 0.8756090822068926E+00, 0.1433700749181099E-01, 0.2857413018245216E+00},
			n:    21,
			zOut: []float64{0.2546753248931182E+01, 0.2051980365170268E+01, 0.1958366389726797E+01, 0.1631238683423845E+01, 0.1586034790683645E+01, 0.1538534425510682E+01, 0.1255412586747036E+01, 0.1121690972560896E+01, 0.9718147643766369E+00, 0.9049110982817965E+00, 0.7946671649858114E+00, 0.5884376759528481E+00, 0.4050217422474118E+00, 0.3189813042957475E+00, 0.2502079076659038E+00, 0.1378021633219403E+00, 0.1230588999356998E+00, 0.5024634530670342E-01, 0.4118371112977081E-01, 0.5279820157992808E-02, 0.1421284452028254E-03, 0.2680065184671810E-21, 0.2301930285321580E-14, 0.5132436462785431E-14, 0.1255412586747036E+01, 0.2424007516335156E-18, 0.1913029885324846E-10, 0.9021761757831104E-11, 0.1121690972560896E+01, 0.1977581519346623E-22, 0.4602280167506019E-08, 0.2460881124989248E-08, 0.9718147643766369E+00, 0.1104484640438811E-18, 0.4278245612761057E-07, 0.2387897413722468E-07, 0.9049110982817965E+00, 0.2768434971275651E-16, 0.3015076539683485E-07, 0.2961312447326530E-07, 0.7946671649858114E+00, 0.1262042385269225E-19, 0.1828176618885781E+02, 0.1828176618885782E+02, 0.1040000000000000E+03, 0.3122448979591837E+01, 0.3846153846153846E+01, 0.1851926648424158E-06, 0.4050217422474118E+00, 0.9293221980374701E-17, 0.1284245841610550E-05, 0.2164676294343609E-05, 0.3189813042957475E+00, 0.1378649105642637E-16, 0.4610281885819925E-06, 0.1003327305702036E-05, 0.2502079076659038E+00, 0.2177821011994986E-13, 0.3677246803752887E-04, 0.3095541926811936E-04, 0.1378021633219403E+00, 0.9210656120777790E-20, 0.9401458780737957E-05, 0.1039122642078444E-04, 0.1230588999356998E+00, 0.4930380657631324E-30, 0.6528493871226699E-06, 0.1524653341983116E-05, 0.5024634530670342E-01, 0.6838275466788171E-23, 0.2104276631320477E-03, 0.3160258883556432E-03, 0.4118371112977081E-01, 0.6162975822039155E-32, 0.9318133168194381E-04, 0.1261364640367337E-03, 0.5279820157992808E-02, 0.7005084838362585E-27, 0.1903158606362315E-02, 0.7288242235414376E-03, 0.1421284452028254E-03, 0.7703719777548943E-33, 0.7890744155827079E-03, 0.6129022301104231E-03},
			info: 0,
		},
		{
			z:    []float64{0.1813251202119042E+00, 0.8077156210855166E+00, 0.5525396662071069E+00, 0.4858391136124270E+00, 0.5367389425969793E+00, 0.4534294101162717E-01, 0.1438386197304258E+00, 0.7801305810912117E+00, 0.7706244548712898E+00, 0.1595286891781422E+00, 0.4689597797472277E+00, 0.1630160981693277E+00, 0.5488525733069047E+00, 0.3050379405729475E+00, 0.1544160351990825E+00, 0.5750210099163514E+00, 0.1440876969429373E+00, 0.1932969959898051E+00, 0.6910796187912660E+00, 0.9501882645686455E+00, 0.1314109921701710E+00, 0.7099941843070685E+00, 0.3797313713209320E+00, 0.3979808845174081E+00, 0.1006018923100482E-02, 0.4131896358268045E+00, 0.8038132966802857E+00, 0.8778643630407806E+00, 0.1407889867849830E+00, 0.4305134083308382E-01, 0.5656496703997997E+00, 0.5820188952601091E-01, 0.8713537916178680E+00, 0.3443162753417257E+00, 0.4898335360866699E+00, 0.2822165738696382E+00, 0.2385629061714932E+00, 0.5496478420286478E+00, 0.7138724344268773E+00, 0.9028268569137332E+00, 0.7573748459251011E+00, 0.5392259606733280E+00, 0.6459213128573325E+00, 0.7769497211063184E+00, 0.8790154346127051E+00, 0.6027389165195913E+00, 0.6151696559086139E-01, 0.4391208291942791E+00, 0.8705350585502258E-02, 0.2995750933531418E+00, 0.9735013653226658E+00, 0.9551710085419519E+00, 0.2132875677034638E+00, 0.6739668529657764E+00, 0.5361543284664445E+00, 0.3596977928604983E+00, 0.1373006735837845E-02, 0.8494363479416891E+00, 0.4214720423022895E+00, 0.7478449906096684E+00, 0.1164429527473354E-01, 0.6143683285709396E+00, 0.4444360064738268E+00, 0.6626608299302873E+00, 0.6887459689591686E+00, 0.9391262494647765E+00, 0.7167310461528731E+00, 0.4516657019045726E+00, 0.6345059624902227E+00, 0.2241941493026407E+00, 0.3981621310014529E+00, 0.8300976429090324E+00, 0.1390103241004258E+00, 0.7559319384422415E+00, 0.7929892786129927E+00, 0.2279061631835571E+00, 0.5280867615120838E+00, 0.1374499172030500E+00, 0.1739503966093568E+00, 0.1475298919309330E+00, 0.8100098524713260E+00, 0.2514850916075489E+00, 0.4100580488172028E+00, 0.9565966467338131E+00},
			n:    21,
			zOut: []float64{0.2319179815575576E+01, 0.1965907124623829E+01, 0.1887317422176422E+01, 0.1806195743608332E+01, 0.1738584542532610E+01, 0.1518839817892385E+01, 0.1233018482483247E+01, 0.1147944957333420E+01, 0.8847632334488245E+00, 0.7964611162721277E+00, 0.7478364459856081E+00, 0.6404526251941426E+00, 0.4533452491188792E+00, 0.4218888103472739E+00, 0.2727838365998446E+00, 0.2511907405802137E+00, 0.9874721307137806E-01, 0.7639943902083833E-01, 0.4105860224136690E-01, 0.2835039177122555E-01, 0.1849635761425839E-05, 0.5687744349931984E-17, 0.4733652108368680E-16, 0.8474188088642906E-16, 0.1233018482483247E+01, 0.2772075531849652E-16, 0.2588283290162890E-14, 0.3835910828290052E-14, 0.1147944957333420E+01, 0.1246926305809672E-21, 0.2331804915191085E-13, 0.1628473983494628E-13, 0.8847632334488245E+00, 0.1029638495848151E-18, 0.6869501877456999E-13, 0.5177641249986829E-13, 0.7964611162721277E+00, 0.2366582715663035E-28, 0.1931319748910825E-12, 0.1483226506162133E-12, 0.7478364459856081E+00, 0.0000000000000000E+00, 0.1833026745951331E+02, 0.1833026745951330E+02, 0.1060000000000000E+03, 0.3281179138321995E+01, 0.4716981132075472E+01, 0.1032982741787823E-11, 0.4533452491188792E+00, 0.1306947070833229E-13, 0.7648022419924529E-11, 0.9563864191611805E-11, 0.4218888103472739E+00, 0.2958228394578794E-30, 0.1901963755802408E-10, 0.2450915092704452E-10, 0.2727838365998446E+00, 0.2459678163240430E-25, 0.5533379126041796E-09, 0.3623624255644135E-09, 0.2511907405802137E+00, 0.1890110728909544E-26, 0.3358805149198491E-08, 0.5375184613165106E-08, 0.9874721307137806E-01, 0.5174927538249837E-27, 0.5300164302540724E-07, 0.8261446474891677E-07, 0.7639943902083833E-01, 0.5686866711287438E-23, 0.3075058441398368E-06, 0.1980286028193141E-06, 0.4105860224136690E-01, 0.3391723239215845E-21, 0.3233943307567427E-08, 0.7908141105983331E-09, 0.2835039177122555E-01, 0.0000000000000000E+00, 0.6263230510933575E-04, 0.3843833760247782E-04, 0.1849635761425839E-05, 0.1263270231490984E-18, 0.3749684628997549E-02, 0.9345197933294263E-02},
			info: 0,
		},
		{
			z:    []float64{0.7197398584131909E+00, 0.8353840049537183E+00, 0.7768334206137515E+00, 0.5378835466874095E+00, 0.6112887154160760E+00, 0.6941229566444685E+00, 0.6006831303610110E+00, 0.9634869426512738E-01, 0.5491965547787625E+00, 0.2928965321959978E+00, 0.2058564636772363E+00, 0.8430652032339261E+00, 0.6514967955084195E+00, 0.9630832188959448E+00, 0.6099558369768557E+00, 0.4038812997556026E+00, 0.1286146012481256E+00, 0.1377932216741019E+00, 0.2151249629422850E+00, 0.5230096632076910E+00, 0.7097266420339532E+00, 0.5223889086485353E+00, 0.1844557902493219E+00, 0.8630886970819189E+00, 0.3143018526841452E+00, 0.3064747008129198E-02, 0.9995777893655772E+00, 0.8804003743833377E+00, 0.2437734590599224E+00, 0.5439224193002447E+00, 0.7609946860078259E+00, 0.6816222186177212E+00, 0.4213100187092527E+00, 0.5947299218072758E+00, 0.4464510683518146E+00, 0.9444991162538406E+00, 0.1663546170185037E+00, 0.9745722490509305E+00, 0.8691156687582791E-01, 0.1511122923287322E+00, 0.9099013034006386E+00, 0.2055280668016559E+00, 0.2215435147609097E+00, 0.1341458340087506E+00, 0.6974464577249216E+00, 0.9770585050570383E+00, 0.2836484389736937E-02, 0.4942205711890229E+00, 0.2476589071081227E+00, 0.9025674475802441E+00, 0.1043173067377164E+00, 0.7752297952628227E+00, 0.1045812556997563E+00, 0.6207135868564088E+00, 0.3343332589884191E+00, 0.9777410164371825E+00, 0.6919265640807399E+00, 0.8276791372331573E+00, 0.9681277631171376E+00, 0.2774385945770376E+00, 0.3660516291700039E+00, 0.1542243233614283E+00, 0.9809065840498956E+00, 0.9317468521553756E+00, 0.2342755587307467E+00, 0.1233924367977165E+00, 0.3662228319745224E+00, 0.8394462754756703E+00, 0.1257927855780998E+00, 0.9899742295380913E+00, 0.4800112259347978E+00, 0.9614338190585195E+00, 0.9605383026678687E+00, 0.7532278237974727E+00, 0.6455994058946259E+00, 0.6648840697262024E+00, 0.8792931963163917E+00, 0.7296338860815629E+00, 0.6962759488463338E+00, 0.5343477930729980E+00, 0.7488075021642351E+00, 0.8267598422017227E+00, 0.4128864062622362E-02, 0.2248794313377519E+00},
			n:    21,
			zOut: []float64{0.2440396187223568E+01, 0.2429593845031884E+01, 0.2248977224883395E+01, 0.1983141106497864E+01, 0.1775047097539426E+01, 0.1685850120713433E+01, 0.1685238074648001E+01, 0.1228212776246503E+01, 0.1167582065653490E+01, 0.1163468828895948E+01, 0.9840288601088477E+00, 0.8446765744020336E+00, 0.5858419229944851E+00, 0.5828733061329966E+00, 0.4321746091262585E+00, 0.2506332278777725E+00, 0.1601952029039821E+00, 0.9519704710728188E-01, 0.5380488026634178E-01, 0.2070966044404248E-02, 0.4144953892415127E-03, 0.1577721810442024E-28, 0.2622007349295965E-13, 0.1959395597612919E-13, 0.1685238074648001E+01, 0.5298116057209151E-24, 0.4690429322435832E-13, 0.6275308499802102E-13, 0.1228212776246503E+01, 0.3786532345060857E-26, 0.6968636880972768E-12, 0.1483661956038748E-11, 0.1167582065653490E+01, 0.4135903062764930E-23, 0.1852086238235612E-10, 0.7750996975985970E-11, 0.1163468828895948E+01, 0.1171277614412583E-17, 0.1085363071939462E-09, 0.2643024434742409E-09, 0.9840288601088477E+00, 0.6878867093527223E-27, 0.2179941841968715E+02, 0.2179941841968716E+02, 0.1010000000000000E+03, 0.3140589569160998E+01, 0.4950495049504950E+01, 0.1077573597463830E-06, 0.5858419229944851E+00, 0.3627601229823987E-15, 0.8463992810044351E-06, 0.7647582718993818E-06, 0.5828733061329966E+00, 0.5798443197736525E-25, 0.5805124866712180E-06, 0.5336279095847515E-06, 0.4321746091262585E+00, 0.1164890830118439E-21, 0.3270430863981586E-06, 0.3120015969970153E-06, 0.2506332278777725E+00, 0.1344740577327136E-22, 0.6612009870063018E-06, 0.4716796838663449E-06, 0.1601952029039821E+00, 0.0000000000000000E+00, 0.1909143391170965E-04, 0.1029106625513428E-04, 0.9519704710728188E-01, 0.5127595883936577E-28, 0.3722911957200607E-04, 0.5334790194682025E-04, 0.5380488026634178E-01, 0.4916199468524512E-16, 0.8145434425645190E-04, 0.4879722252791062E-03, 0.2070966044404248E-02, 0.2051115390772406E-31, 0.1886522673573371E-02, 0.1146190938097804E-02, 0.4144953892415127E-03, 0.1134788402846185E-20, 0.5837570815253673E-02, 0.1435831510150291E-01},
			info: 0,
		},
		{
			z:    []float64{0.4141124863190200E+00, 0.6286592513493726E+00, 0.4875442526190354E-01, 0.6631999174821102E+00, 0.1778328741285358E+00, 0.5820997881750413E+00, 0.9610329112936564E+00, 0.2860180457593013E+00, 0.2800413372286658E+00, 0.8782990052007664E+00, 0.6769848820880126E+00, 0.2361981145928655E+00, 0.1864129841856804E+00, 0.6207802268756868E-01, 0.1668832170172669E+00, 0.2136536864861034E+00, 0.4266698341219798E+00, 0.9672331991204097E-01, 0.2350195655513042E+00, 0.1297162551474343E+00, 0.9428942697229301E+00, 0.3355426683568981E+00, 0.6977568272344947E-01, 0.4926170514192555E+00, 0.6966571762613959E+00, 0.7422949877167778E+00, 0.1216493993328296E+00, 0.5414775352707472E+00, 0.4507250901570525E+00, 0.6851601412475350E+00, 0.8339584749808495E+00, 0.9321374665508209E+00, 0.7664819871766777E+00, 0.5935455015911359E+00, 0.4047285985374618E+00, 0.9996987445838635E+00, 0.1347158425141065E+00, 0.3601395592420920E+00, 0.1128763847009464E+00, 0.6316809196633902E+00, 0.1559195263655234E+00, 0.4041853050912981E+00, 0.1854167847278504E+00, 0.6852913644854558E+00, 0.6126092157077416E+00, 0.2444516268314137E+00, 0.1447238269517732E+00, 0.7179956614587130E+00, 0.8198652148251639E+00, 0.1186997025611126E+00, 0.1267596520196094E+00, 0.2863431547875360E+00, 0.9867539057590858E+00, 0.2454746550565152E+00, 0.4891318294681372E+00, 0.2969454310391295E-01, 0.2349636207332649E+00, 0.4537731552146703E+00, 0.2883801124165087E+00, 0.7150664117598651E+00, 0.4359050919721921E+00, 0.6116081525473241E+00, 0.5619840590438748E+00, 0.3431726287099235E+00, 0.7715898362788249E+00, 0.4981359837313185E+00, 0.4537925078699018E+00, 0.6956466175473529E+00, 0.1814680828887681E+00, 0.4413903687143481E+00, 0.4367593306147978E+00, 0.5875066776157845E-01, 0.3688522862246468E+00, 0.4308468783440871E+00, 0.1986553761412286E+00, 0.3949957841484223E+00, 0.2468986938324821E+00, 0.6216444107315144E-01, 0.9076510210089642E+00, 0.5197394332656491E+00, 0.8460559187475114E-01, 0.3959477165848089E+00, 0.1014864219123556E+00, 0.4789226780612066E-01},
			n:    21,
			zOut: []float64{0.2746085040517442E+01, 0.2006646947029628E+01, 0.1843368381126182E+01, 0.1738309804669461E+01, 0.1678478075610122E+01, 0.1408432532104903E+01, 0.1219097982435216E+01, 0.1121569511965291E+01, 0.9315294884679637E+00, 0.7914377647612258E+00, 0.7334412101899261E+00, 0.6113376828240070E+00, 0.4134386595159317E+00, 0.3303980566325673E+00, 0.2866331058299493E+00, 0.2287206720574312E+00, 0.1408981800627584E+00, 0.6649497607659796E-01, 0.5733628497886722E-01, 0.1387598234001070E-02, 0.2497701489633820E-04, 0.3487740233155725E-22, 0.3361270559917294E-05, 0.1214091727263821E-05, 0.1219097982435216E+01, 0.2843433355651033E-23, 0.3877657623657789E-06, 0.2861458561047538E-06, 0.1121569511965291E+01, 0.1501849799666580E-22, 0.7980776610337536E-07, 0.1296132789777516E-06, 0.9315294884679637E+00, 0.1397279453391396E-14, 0.3797866386428551E-06, 0.5462183171930727E-06, 0.7914377647612258E+00, 0.1231982124821006E-21, 0.6463973892160826E-06, 0.4555968092429713E-06, 0.7334412101899261E+00, 0.9423416829408119E-25, 0.1835506693210437E+02, 0.1835506693210436E+02, 0.9800000000000000E+02, 0.2884353741496599E+01, 0.2040816326530612E+01, 0.6355562713716383E-07, 0.4134386595159317E+00, 0.4284860196510039E-21, 0.4931868922851139E-05, 0.2760082849901006E-05, 0.3303980566325673E+00, 0.1448288531514775E-16, 0.4631299886341806E-05, 0.9149811670123540E-05, 0.2866331058299493E+00, 0.1558000287811498E-28, 0.1337090019087612E-05, 0.8008074996380075E-06, 0.2287206720574312E+00, 0.2366582715663035E-29, 0.4303808308110317E-06, 0.3181248709867330E-06, 0.1408981800627584E+00, 0.2443697032165075E-19, 0.2973006759321362E-05, 0.5017304713868616E-05, 0.6649497607659796E-01, 0.1131941655598205E-16, 0.8328041136745409E-04, 0.8552508571764750E-04, 0.5733628497886722E-01, 0.9398173280441046E-25, 0.5097763200650848E-04, 0.5644224660768206E-04, 0.1387598234001070E-02, 0.8888394955180400E-22, 0.1868721561136477E-03, 0.1091351925463373E-03, 0.2497701489633820E-04, 0.0000000000000000E+00, 0.1020791029288490E-01, 0.2791888375404797E-02},
			info: 0,
		},
		{
			z:    []float64{0.2396228092919251E+00, 0.6434601049058037E+00, 0.5488667726527494E+00, 0.1843504645657519E+00, 0.4269001369847240E+00, 0.6276093013166625E+00, 0.8724336393628729E-01, 0.5371039952903035E+00, 0.8447460457667564E+00, 0.4094766929502263E+00, 0.2031514124499976E+00, 0.8271428312997948E+00, 0.9658284302973903E+00, 0.3777353969048113E+00, 0.6471334022587341E+00, 0.6777435175679861E+00, 0.4456890557292859E+00, 0.1720826572683752E+00, 0.3520207432901141E+00, 0.3291396508341311E+00, 0.7468262262373744E+00, 0.2715104496960554E+00, 0.5612917863396235E+00, 0.1324429469286776E+00, 0.7983766064014711E+00, 0.2059599586019641E+00, 0.2655510256425775E-01, 0.1962390264484870E+00, 0.7845246896371560E-02, 0.8217737559711210E+00, 0.9444498218704492E+00, 0.4679957695831033E+00, 0.1135918053039552E+00, 0.1604948252571132E+00, 0.8452381101613837E+00, 0.8537070814716049E+00, 0.3013932596816266E+00, 0.2511715406702125E+00, 0.2008647638036438E+00, 0.2607897573444329E+00, 0.2571050910078490E+00, 0.6286174024463315E+00, 0.4739701849723611E+00, 0.8698947534844018E+00, 0.6486079589595961E+00, 0.8894321904301987E+00, 0.7952475840185558E+00, 0.8801661721215410E+00, 0.2147272194381030E+00, 0.9518356646476019E+00, 0.7191152289414877E+00, 0.1106524971173308E+00, 0.9574587230734148E+00, 0.6469474230679395E+00, 0.5818260757443584E+00, 0.1292202837075540E-01, 0.2761722913438078E+00, 0.8004535925585585E+00, 0.3278973855590038E+00, 0.5794037438664289E+00, 0.2396155443104175E-01, 0.6699048475753794E+00, 0.1814492937128086E+00, 0.2866184936222242E+00, 0.6584557212506269E+00, 0.5889651227775431E+00, 0.6369766519060738E-01, 0.1328193529598596E+00, 0.7416631417262871E+00, 0.5003318278706693E+00, 0.3144077296395376E-01, 0.9907393156403193E+00, 0.3423696594914594E-02, 0.6992922072850826E+00, 0.7590519617064543E+00, 0.2542182514116359E+00, 0.3314657184555089E+00, 0.5728659684019645E+00, 0.9915854796414423E+00, 0.4896749730105238E+00, 0.5611369720796141E+00, 0.6064142215043516E+00, 0.8567590164766914E-01, 0.2581731932678185E+00},
			n:    21,
			zOut: []float64{0.2319327880258489E+01, 0.2079713994229665E+01, 0.1953873370330741E+01, 0.1755147049127663E+01, 0.1569664109844580E+01, 0.1497177879850755E+01, 0.1442939549198070E+01, 0.1142654640257259E+01, 0.1037259380018118E+01, 0.7336163063097807E+00, 0.6300572204242147E+00, 0.3966542874116493E+00, 0.3925821156107781E+00, 0.2721798695257625E+00, 0.2365431978248397E+00, 0.2044065204974935E+00, 0.1570073942488483E+00, 0.8918832674569614E-01, 0.5938710196407761E-01, 0.3241742658191688E-02, 0.4778146591607052E-04, 0.1325286320771300E-27, 0.1485620696270319E-14, 0.4653485276256090E-14, 0.1442939549198070E+01, 0.1846131417899111E-22, 0.1162592678154001E-11, 0.3263011919854044E-11, 0.1142654640257259E+01, 0.4412642697382792E-11, 0.1735169830505648E-08, 0.3333560963091213E-08, 0.1037259380018118E+01, 0.1055565179678919E-18, 0.1940309609380032E-07, 0.3465529731850831E-07, 0.7336163063097807E+00, 0.4440642298320086E-11, 0.3810071749579748E-06, 0.5916980508399715E-06, 0.6300572204242147E+00, 0.4658681208910640E-20, 0.1797266971780259E+02, 0.1797266971780259E+02, 0.1110000000000000E+03, 0.3253968253968254E+01, 0.3603603603603604E+01, 0.2492194730770069E-13, 0.3925821156107781E+00, 0.0000000000000000E+00, 0.7667233942003248E-12, 0.2416678934089916E-12, 0.2721798695257625E+00, 0.2051915459060795E-16, 0.1487877135020801E-08, 0.6143545671036994E-08, 0.2365431978248397E+00, 0.1972152263052530E-29, 0.1960096418178738E-05, 0.2274829676605902E-05, 0.2044065204974935E+00, 0.2021903259098794E-21, 0.8745843979356665E-06, 0.9251106153042376E-06, 0.1570073942488483E+00, 0.5482898835648120E-25, 0.7118492322907369E-04, 0.3316595178521346E-03, 0.8918832674569614E-01, 0.1022725463125710E-19, 0.2167108979032774E-02, 0.1228336641518804E-02, 0.5938710196407761E-01, 0.1994240368398718E-26, 0.2711464061500906E-02, 0.2561834534417173E-02, 0.3241742658191688E-02, 0.1814380082008327E-27, 0.4237092909216320E-02, 0.4420407214578293E-02, 0.4778146591607052E-04, 0.2596519942803954E-20, 0.1820843269063370E-02, 0.8929397931439835E-02},
			info: 0,
		},
		{
			z:    []float64{0.4704013808869884E+00, 0.8133248937317098E+00, 0.7246209022332254E+00, 0.9930301346620946E+00, 0.6565279241713462E-01, 0.7480498761899035E-01, 0.8311943018468732E+00, 0.4127487814312358E+00, 0.6620288516904083E+00, 0.2540390589975015E+00, 0.4830377447593585E+00, 0.6124023244542695E+00, 0.6830062885714996E+00, 0.4006497306883100E+00, 0.2108017907398356E+00, 0.9099126499549155E+00, 0.4756780627224959E+00, 0.8563186613679700E+00, 0.9168447467108382E+00, 0.5153963449488133E+00, 0.7188966629304918E+00, 0.2875525393671987E+00, 0.6200986280097742E-01, 0.3765186242141257E+00, 0.2589923271177115E+00, 0.5404093081826918E+00, 0.5343437371907505E+00, 0.3104227497255196E-01, 0.4221838478889377E+00, 0.4233113354732956E+00, 0.6486058235731677E+00, 0.1590032665029245E+00, 0.3877113621801469E+00, 0.6038849620024899E+00, 0.5400341407425849E+00, 0.7060013907006117E+00, 0.1173345165878200E+00, 0.8548311704210352E+00, 0.5804532700087333E+00, 0.5231399784962363E+00, 0.4754841524148748E+00, 0.3395282932287647E-01, 0.7668594432348432E+00, 0.8184682560752043E+00, 0.5633264054575460E-01, 0.2127176392631261E+00, 0.7432372685121402E+00, 0.3098453278874418E+00, 0.1150335122169536E+00, 0.8681635561145716E+00, 0.3593935949259885E+00, 0.9085822737263088E+00, 0.2122810259380176E+00, 0.3589456483065394E+00, 0.9437026007300053E-01, 0.6518875210051764E+00, 0.9046367728621729E+00, 0.7638617707854546E+00, 0.8924521343092110E+00, 0.8353112965912926E+00, 0.5076189975461921E+00, 0.4955137299922637E+00, 0.5041148053921253E+00, 0.7686114107944846E+00, 0.3833696141187980E+00, 0.9181102072361824E-01, 0.9260679284685853E+00, 0.7081687340916837E+00, 0.8312629527805896E+00, 0.8010881636433624E+00, 0.5891143654107377E+00, 0.8149927954305255E+00, 0.7192672792238343E+00, 0.9981622901025261E+00, 0.9368513325443091E+00, 0.6909941993236977E+00, 0.7208744872845105E+00, 0.6334330703247726E+00, 0.8856108044732016E+00, 0.7612967011342276E+00, 0.6840498257884192E+00, 0.8627400929877541E+00, 0.2970500386847904E+00, 0.9828991929119790E+00},
			n:    21,
			zOut: []float64{0.2548749189884266E+01, 0.2318283655447516E+01, 0.1949991272828373E+01, 0.1903113092077109E+01, 0.1681173723989745E+01, 0.1621144658162279E+01, 0.1435953736019784E+01, 0.1373215273744876E+01, 0.1276544145324249E+01, 0.8508650942792525E+00, 0.7433578075251207E+00, 0.7333566617981635E+00, 0.6327824956484905E+00, 0.5146506379994326E+00, 0.4497218333883862E+00, 0.2736876649003249E+00, 0.2262999758546705E+00, 0.6320618886076086E-01, 0.1413547856395308E-01, 0.6182735080657475E-02, 0.1223662826415605E-02, 0.9844984097158227E-26, 0.2664250906052687E-13, 0.8143635875649387E-14, 0.1435953736019784E+01, 0.7237830359838992E-24, 0.1147364015889613E-09, 0.4819137169184198E-10, 0.1373215273744876E+01, 0.5490471900338242E-27, 0.1654936529088367E-07, 0.7460880307437913E-08, 0.1276544145324249E+01, 0.2958228394578794E-30, 0.1455672577071588E-05, 0.7169454013791148E-06, 0.8508650942792525E+00, 0.4041314959719700E-19, 0.1917383253612269E-05, 0.1657209462004936E-05, 0.7433578075251207E+00, 0.2357464745776125E-22, 0.2061763898420382E+02, 0.2061763898420383E+02, 0.9600000000000000E+02, 0.2888888888888889E+01, 0.3125000000000000E+01, 0.6718966036888534E-06, 0.6327824956484905E+00, 0.1006441314168352E-20, 0.4252254757420258E-06, 0.3750418335727261E-06, 0.5146506379994326E+00, 0.1125208173090593E-20, 0.1634173128014842E-06, 0.1777130114194832E-06, 0.4497218333883862E+00, 0.1397861524051633E-26, 0.1373153840978587E-06, 0.1424351942740530E-06, 0.2736876649003249E+00, 0.3055476705078142E-15, 0.2530372171390620E-06, 0.3073674645773779E-06, 0.2262999758546705E+00, 0.3987473575125276E-15, 0.5814400581048744E-06, 0.7370753629310487E-06, 0.6320618886076086E-01, 0.9597492304117215E-21, 0.2453687283764395E-05, 0.3193610655869065E-05, 0.1413547856395308E-01, 0.8528167491936360E-20, 0.7231103044220720E-05, 0.9769896858142881E-05, 0.6182735080657475E-02, 0.0000000000000000E+00, 0.3039337033448658E-04, 0.5367078032707280E-04, 0.1223662826415605E-02, 0.6617444900424221E-23, 0.7353916438463039E-03, 0.2144171525794732E-02},
			info: 0,
		},
		{
			z:    []float64{0.9714680453643365E-01, 0.6357804703219754E+00, 0.6664990519488578E-01, 0.1729636879767130E+00, 0.4638342338705743E+00, 0.4390808892462035E+00, 0.8041041328438946E+00, 0.4288731182982508E+00, 0.7624625505229141E+00, 0.7083386603826051E+00, 0.2464792096944004E+00, 0.9011775040224685E+00, 0.6339539510596026E-01, 0.1795036824346197E-01, 0.9473751738931280E+00, 0.4886196718733559E+00, 0.3845982510172906E+00, 0.8175566437299375E-01, 0.4796454941504319E+00, 0.1995303497008669E+00, 0.3542217290845850E+00, 0.4313075601907782E+00, 0.9062634832132850E+00, 0.9362490171108290E+00, 0.7454458344912314E+00, 0.9678028365928690E+00, 0.8785588729275070E+00, 0.2582842558253936E+00, 0.7937433110799214E+00, 0.7912406502001545E-01, 0.1813501620770241E-01, 0.3340784700607168E+00, 0.3348119644971841E+00, 0.5042866134297835E+00, 0.3808798817625770E+00, 0.8932448815066745E+00, 0.2612469692343123E+00, 0.4943911382967600E+00, 0.7396437014435053E+00, 0.6180267747680381E+00, 0.6768529193184074E+00, 0.7346572400466492E+00, 0.8658232798365260E+00, 0.6590888814314796E+00, 0.9020629495832085E+00, 0.2719526227756762E+00, 0.9337960905099231E+00, 0.1402012445044727E+00, 0.3363383773836534E+00, 0.4741712787560632E+00, 0.1270352327896518E-02, 0.2317652069389444E+00, 0.9583986396877475E+00, 0.8857657675309066E+00, 0.5564654172742324E+00, 0.3373121694331064E+00, 0.1866413892025005E+00, 0.3829716065530064E+00, 0.6660176769447556E+00, 0.9140619394089379E+00, 0.4126532961354976E+00, 0.9883322088463480E+00, 0.1678362296131076E+00, 0.1246866166129172E+00, 0.5083959276848560E+00, 0.4743484764552197E+00, 0.1561141239638767E+00, 0.4867628897662705E+00, 0.8068909470080686E+00, 0.9878597697456749E+00, 0.8535449450676436E-01, 0.9609706407601909E+00, 0.7015171470525342E+00, 0.4808513401395343E+00, 0.4421870922261545E+00, 0.2476919441028158E-01, 0.2417065613871903E+00, 0.1174043678151601E+00, 0.4259626380136439E-02, 0.2614829171642307E+00, 0.5973781466265196E+00, 0.3195524674053074E-01, 0.4778949570977242E+00, 0.2741401812748340E+00},
			n:    21,
			zOut: []float64{0.2956562879292357E+01, 0.2160920112738185E+01, 0.2057048821491083E+01, 0.1826665062315388E+01, 0.1626476426276149E+01, 0.1617659828407750E+01, 0.1458559932420392E+01, 0.1037845007643388E+01, 0.9436387491729998E+00, 0.8889042931978449E+00, 0.8147110102488896E+00, 0.7938854029224469E+00, 0.5240642869032099E+00, 0.4281570616174346E+00, 0.4032231719469492E+00, 0.2134769254146812E+00, 0.1570954054223686E+00, 0.5771888278451791E-01, 0.2771650645038780E-01, 0.1168397530048623E-02, 0.8626671344136174E-03, 0.6837796509826355E-18, 0.5416843762303492E-12, 0.3045298824248355E-12, 0.1458559932420392E+01, 0.8473754517266856E-24, 0.4783112378014045E-11, 0.8163747631331082E-11, 0.1037845007643388E+01, 0.8077935669463161E-27, 0.2442833758035109E-10, 0.3576901446935589E-10, 0.9436387491729998E+00, 0.1292469707114106E-25, 0.1060154010453575E-09, 0.7148292157772074E-10, 0.8889042931978449E+00, 0.5995342879679690E-28, 0.2265830666026058E-09, 0.3413067902462323E-09, 0.8147110102488896E+00, 0.6874131484520357E-19, 0.1999636083133089E+02, 0.1999636083133089E+02, 0.1000000000000000E+03, 0.3002267573696145E+01, 0.2000000000000000E+01, 0.2441378744031031E-06, 0.5240642869032099E+00, 0.2169367489357782E-29, 0.9564347597274963E-06, 0.8629398155870994E-06, 0.4281570616174346E+00, 0.1533177238774536E-16, 0.2198723423439569E-05, 0.1852641299824983E-05, 0.4032231719469492E+00, 0.1742814620686677E-24, 0.6898774546312495E-05, 0.1242988026413374E-04, 0.2134769254146812E+00, 0.1303198215425112E-25, 0.3054946518379762E-04, 0.2458010895625246E-04, 0.1570954054223686E+00, 0.2480178686014861E-25, 0.5573684427056079E-04, 0.2879079609909993E-04, 0.5771888278451791E-01, 0.0000000000000000E+00, 0.4895501114079361E-04, 0.6615663653270374E-04, 0.2771650645038780E-01, 0.6559433690090112E-18, 0.2090848897194263E-03, 0.3843856621792604E-03, 0.1168397530048623E-02, 0.0000000000000000E+00, 0.2396113765895983E-02, 0.4353699886144585E-02, 0.8626671344136174E-03, 0.1420984816841217E-20, 0.3572874909786609E-04, 0.2591037812233495E-05},
			info: 0,
		},
		{
			z:    []float64{0.7761482897976135E-01, 0.7690133527529974E-01, 0.8589697242168557E+00, 0.5718901484414876E+00, 0.8450349347421171E-01, 0.5688250545090190E+00, 0.6202247495663940E+00, 0.7388100008042665E+00, 0.9890328265375476E+00, 0.2969659976080158E+00, 0.5442392738534707E-01, 0.9562602329262000E+00, 0.8530135563879812E+00, 0.5471012261519459E-01, 0.9710076793297340E+00, 0.4869447961696205E+00, 0.4659851368281548E+00, 0.7647605444900396E-01, 0.7872766307526106E+00, 0.3237029299700311E+00, 0.7194522935712468E+00, 0.7270865543961295E+00, 0.5183429612108851E+00, 0.3145688411100903E+00, 0.9096514649777033E+00, 0.7816253592835525E+00, 0.4809049584349555E+00, 0.2052154618933422E-01, 0.1481963756328628E+00, 0.8742358520807045E+00, 0.4565878430880338E+00, 0.1977913919695908E+00, 0.6526813957349178E+00, 0.8032154703264597E+00, 0.2356108621359847E+00, 0.4857415898450358E+00, 0.2259727396620309E+00, 0.1176413330041139E+00, 0.8335507369731385E+00, 0.4539477133958770E+00, 0.8910876835374724E+00, 0.1559757847548392E+00, 0.1885199219441583E+00, 0.1520953639100613E+00, 0.6035948595957944E+00, 0.9286538294716984E+00, 0.9676298024186124E+00, 0.1209629819848071E+00, 0.1660874781174386E+00, 0.8622065054293394E+00, 0.7485077140876945E+00, 0.9420716698171637E+00, 0.2904751962009996E+00, 0.2180437045655230E+00, 0.6692025851955981E+00, 0.9258369042460884E+00, 0.6243742234586596E+00, 0.6760203577391158E+00, 0.2199572443863818E-02, 0.5981650670220221E+00, 0.8252506688545779E+00, 0.9433813762695346E+00, 0.5320093960532474E+00, 0.3846966829839934E+00, 0.6727780061271659E+00, 0.4937802596181958E+00, 0.1177802539787526E+00, 0.2963448754612064E-01, 0.2479547713122239E+00, 0.2694003173993550E+00, 0.8186090816580105E+00, 0.2886041049393229E-01, 0.7384561600560133E+00, 0.8345282295575730E+00, 0.1740871089016751E+00, 0.7946006936544379E+00, 0.5640836203667244E+00, 0.4951346622832815E+00, 0.4981407974176422E+00, 0.2513094341381288E+00, 0.4316077257561814E+00, 0.1685190107506288E+00, 0.3264570829830372E+00, 0.1676643753771256E-01},
			n:    21,
			zOut: []float64{0.2309990065237698E+01, 0.2261037149894265E+01, 0.1946075797700671E+01, 0.1868880050926239E+01, 0.1838164096767709E+01, 0.1792848619787804E+01, 0.1601503570675395E+01, 0.1526383850720226E+01, 0.1395433670210245E+01, 0.1010109387319946E+01, 0.8644978203099237E+00, 0.7632668975571170E+00, 0.4552388959212355E+00, 0.4364144823342648E+00, 0.2696980874891977E+00, 0.1302983706884531E+00, 0.1278582769210706E+00, 0.7956979307866878E-01, 0.7389266586143965E-01, 0.8102308371415692E-02, 0.2690335013870282E-02, 0.5752609692135112E-13, 0.2842400234743069E-06, 0.1045980487121886E-05, 0.1601503570675395E+01, 0.1668093198756260E-18, 0.5190756964721459E-05, 0.2367180586416983E-05, 0.1526383850720226E+01, 0.1262177448353619E-27, 0.1974011051185169E-03, 0.4728862593194215E-03, 0.1395433670210245E+01, 0.2337388813815525E-19, 0.1632969150852556E-06, 0.4026698297028720E-06, 0.1010109387319946E+01, 0.3802651360818291E-21, 0.1380334146666910E-03, 0.1568641113059460E-03, 0.8644978203099237E+00, 0.2788260408793746E-19, 0.2076195419278686E+02, 0.2076195419278686E+02, 0.1070000000000000E+03, 0.3183673469387755E+01, 0.2803738317757009E+01, 0.1273578837393323E-04, 0.4552388959212355E+00, 0.1016579123433529E-20, 0.9426551662020599E-04, 0.5179870206653002E-04, 0.4364144823342648E+00, 0.7336406418555410E-28, 0.1112142557098828E-05, 0.1460057881679474E-05, 0.2696980874891977E+00, 0.8909623382378835E-18, 0.3232451044433717E-03, 0.3229988347563871E-03, 0.1302983706884531E+00, 0.0000000000000000E+00, 0.2867336104451064E-03, 0.2890982620847502E-03, 0.1278582769210706E+00, 0.6875938841518755E-23, 0.2127348925826359E-06, 0.9655734242129450E-07, 0.7956979307866878E-01, 0.3291208348307372E-17, 0.5717743824937758E-07, 0.2183606505996598E-06, 0.7389266586143965E-01, 0.7801771243763389E-24, 0.1012732311499954E-04, 0.2863554462185051E-05, 0.8102308371415692E-02, 0.5719241562852336E-29, 0.6292939995324734E-04, 0.3382281194073554E-04, 0.2690335013870282E-02, 0.4951522129891247E-25, 0.3008856418189283E-03, 0.2691027365699742E-03},
			info: 0,
		},
		{
			z:    []float64{0.1834451937168287E+00, 0.4501020603239488E+00, 0.9673873583575473E+00, 0.2207688184907047E+00, 0.4205773270263516E+00, 0.6767264871143511E+00, 0.2493506785870784E+00, 0.7809638015955807E+00, 0.8297195492174936E+00, 0.1617611869044215E+00, 0.9206032300762146E+00, 0.1914715918877316E+00, 0.7976268296797403E+00, 0.9839522457816396E+00, 0.8841932579084628E+00, 0.6979540605797908E+00, 0.8049859576106227E+00, 0.8117504351099887E+00, 0.3842107296635322E+00, 0.6368881208030270E+00, 0.2369530469547876E+00, 0.8005533377467571E+00, 0.7105152962140180E+00, 0.5356376472112734E-02, 0.2192323464228777E+00, 0.8992715082939398E+00, 0.5988055508193698E+00, 0.4081645962203215E+00, 0.1002944447483383E+00, 0.1052431248059748E+00, 0.5730054733733171E+00, 0.2386396328544268E+00, 0.6503058109411604E+00, 0.9019164849722401E+00, 0.4702244241412763E+00, 0.5759947448793571E+00, 0.7588959717342861E+00, 0.4126392985506339E+00, 0.9987799941641187E+00, 0.5598172962702971E+00, 0.2151524348024290E+00, 0.7511708041336773E-01, 0.1539602741757972E+00, 0.4584191954304165E+00, 0.8937659116858414E+00, 0.6186897030318780E+00, 0.6386630452844105E+00, 0.9151803046984296E+00, 0.3614526997644965E+00, 0.9143967867261915E+00, 0.1523575506386343E+00, 0.9745493424336691E+00, 0.4149640117372425E+00, 0.7418302113477876E+00, 0.3041602774831942E-01, 0.4852660298208696E+00, 0.9115870107661288E+00, 0.9024516441581734E+00, 0.9201550825657324E+00, 0.5200815862126855E+00, 0.7353784634707106E+00, 0.8264932398040087E+00, 0.7035838297463828E+00, 0.6738634069153341E+00, 0.7657166064092835E-01, 0.4678396966074577E+00, 0.9927685693816569E+00, 0.5845157970225999E+00, 0.5638346985777254E+00, 0.5850492500516753E+00, 0.3856054034589584E+00, 0.8134648529141429E+00, 0.4174766661784889E+00, 0.3336151222017159E+00, 0.5355530597844248E+00, 0.9274106499154580E+00, 0.7442750408730826E+00, 0.2948331405701974E+00, 0.5696937454611060E+00, 0.3104181547964658E+00, 0.4694844160987418E+00, 0.1538578615150018E+00, 0.5969883537795497E+00, 0.3873507919637014E+00},
			n:    21,
			zOut: []float64{0.2914952592069403E+01, 0.2353566179964282E+01, 0.2062119351173516E+01, 0.1965819938313762E+01, 0.1942714243081098E+01, 0.1812271492844174E+01, 0.1712144141310849E+01, 0.1666892005264074E+01, 0.1208030687686805E+01, 0.1159871012571561E+01, 0.9581193432281622E+00, 0.7738358894690665E+00, 0.7454032361827709E+00, 0.3901865930581417E+00, 0.3240765847275939E+00, 0.2173673669709887E+00, 0.1778074780703053E+00, 0.5660863840842725E-01, 0.3038731391621440E-01, 0.1466323002394464E-01, 0.7362797481955845E-02, 0.1496427221292385E-15, 0.2798385737764086E-17, 0.6348906733176861E-18, 0.1712144141310849E+01, 0.2625127144183791E-23, 0.5842516095868784E-15, 0.1699577922475928E-15, 0.1666892005264074E+01, 0.2934562567422164E-27, 0.7233314809955462E-13, 0.2286859678494404E-12, 0.1208030687686805E+01, 0.8467561517631538E-12, 0.2143114506868008E-08, 0.4305471816838367E-08, 0.1159871012571561E+01, 0.4930380657631324E-31, 0.3270528389239616E-07, 0.6410715811146478E-07, 0.9581193432281622E+00, 0.8911174733768286E-23, 0.2249420011581710E+02, 0.2249420011581710E+02, 0.1050000000000000E+03, 0.3197278911564626E+01, 0.2857142857142857E+01, 0.6896683202589802E-07, 0.7454032361827709E+00, 0.3255610023185390E-23, 0.5315428740895598E-04, 0.2875762635390457E-03, 0.3901865930581417E+00, 0.4610239445276015E-22, 0.9146669702187219E-04, 0.4183987855699404E-04, 0.3240765847275939E+00, 0.4688433862556419E-23, 0.8204008598914952E-05, 0.1523341313649621E-04, 0.2173673669709887E+00, 0.4570683409889788E-20, 0.9316307389873694E-05, 0.2741587710667867E-05, 0.1778074780703053E+00, 0.3706159570762342E-17, 0.9605798472890332E-04, 0.3099752510959849E-03, 0.5660863840842725E-01, 0.5638682600030866E-16, 0.1627758109301002E-02, 0.5977814903797443E-03, 0.3038731391621440E-01, 0.7161878634670464E-18, 0.4935537640644100E-05, 0.4281293465082162E-05, 0.1466323002394464E-01, 0.9343665470806783E-19, 0.1384059308707374E-05, 0.7139695062761456E-05, 0.7362797481955845E-02, 0.1252221392641006E-22, 0.1398979151771859E-02, 0.1830558814226920E-02},
			info: 0,
		},
		{
			z:    []float64{0.1901253435444810E+00, 0.1005117877448146E+00, 0.7431497892893435E-02, 0.4180244733822649E+00, 0.5723993917742813E+00, 0.5092276894610764E+00, 0.6914570054828403E+00, 0.6343440208263138E+00, 0.9714240100252226E+00, 0.9708780190932921E+00, 0.7346948492781401E+00, 0.9261658900466149E+00, 0.6500321763704298E-01, 0.2101988980308164E+00, 0.2569765082490436E+00, 0.6872855071573251E+00, 0.7393915007490531E+00, 0.5293076476804583E+00, 0.5736721160165409E+00, 0.6249080544922647E+00, 0.1805969142240157E+00, 0.6235867578362418E+00, 0.3156395994838945E+00, 0.6768110107418109E+00, 0.5036038244498190E+00, 0.2096885591435443E+00, 0.2471056170621776E+00, 0.4420201221816504E+00, 0.5954916898507195E+00, 0.6849385183400085E+00, 0.6865376402638398E+00, 0.5015659601935997E+00, 0.9646916309223035E+00, 0.1976274617660123E+00, 0.6173831761654865E+00, 0.9161785290588952E+00, 0.8303766381148641E+00, 0.4163407208236638E+00, 0.2406183750735502E-01, 0.9033217043220854E+00, 0.3435327703530300E-01, 0.4851293431546745E+00, 0.1308952665690980E+00, 0.1639949900839996E+00, 0.6530528840204755E+00, 0.4007776795656349E+00, 0.6520732821596675E-01, 0.2482448577499247E+00, 0.6417548063561368E+00, 0.6287927677318972E+00, 0.2491821746122613E+00, 0.1309766085785375E+00, 0.1799023200797401E+00, 0.6166520678731939E+00, 0.4068347418426869E+00, 0.8022499378251485E+00, 0.2090643045880468E+00, 0.9552039287477454E-01, 0.3405285833145572E+00, 0.9010017729234689E+00, 0.8087026571264999E-01, 0.6784547103017031E+00, 0.5001579210546183E+00, 0.9044563563261202E+00, 0.6248337185292430E-01, 0.7231216956646513E+00, 0.3330924448715603E+00, 0.5906941805835393E+00, 0.6144619726434308E-01, 0.4494963607638514E-01, 0.5817524116985039E+00, 0.4323943825883398E+00, 0.9871518362913079E-01, 0.9577530955877981E+00, 0.7720928198042669E+00, 0.1473259710759589E+00, 0.7227644962870149E+00, 0.9964485210410867E+00, 0.3235582385765219E+00, 0.5984954533580267E-01, 0.1531719633524309E-01, 0.8969070444410814E+00, 0.8320043732823273E+00, 0.7541623800831910E+00},
			n:    21,
			zOut: []float64{0.2943190889471759E+01, 0.2404733889291933E+01, 0.2164120218254414E+01, 0.2101040443717650E+01, 0.1828309865967689E+01, 0.1647179310188885E+01, 0.1393092850251358E+01, 0.1217421086377993E+01, 0.1157068329170861E+01, 0.9693609911235493E+00, 0.7556225164779018E+00, 0.6051010122910800E+00, 0.5244111548905432E+00, 0.4204021043595824E+00, 0.3294230503008973E+00, 0.2905705444026201E+00, 0.1318042053167392E+00, 0.6748792703931331E-01, 0.3115549452672914E-01, 0.3850968595106850E-02, 0.1766035472385010E-05, 0.2996366283941816E-12, 0.8018633237482923E-17, 0.2249911398990393E-16, 0.1393092850251358E+01, 0.8588781736260544E-19, 0.1768526861021460E-14, 0.8068659675231548E-15, 0.1217421086377993E+01, 0.2465190328815662E-30, 0.6841595655394777E-14, 0.1344694857995563E-13, 0.1157068329170861E+01, 0.2714422906896024E-19, 0.9771202940807984E-13, 0.1895423136689650E-12, 0.9693609911235493E+00, 0.1093011646009194E-15, 0.5148224565067754E-11, 0.9904113235317579E-11, 0.7556225164779018E+00, 0.6752649348691861E-27, 0.2098534861805207E+02, 0.2098534861805208E+02, 0.9600000000000000E+02, 0.2832199546485261E+01, 0.2083333333333333E+01, 0.3437245789676428E-09, 0.5244111548905432E+00, 0.2665011951551765E-23, 0.6566411319039389E-08, 0.3635282930812000E-08, 0.4204021043595824E+00, 0.1249353725478346E-22, 0.6690013285699780E-07, 0.1185995569029548E-06, 0.3294230503008973E+00, 0.3250106929510569E-26, 0.6264577969246473E-06, 0.1086745069147475E-05, 0.2905705444026201E+00, 0.1121066009627684E-24, 0.5553814969713904E-05, 0.9554866377781883E-05, 0.1318042053167392E+00, 0.1875595688253478E-25, 0.3215548179944659E-03, 0.4036498498040293E-03, 0.6748792703931331E-01, 0.4043956767801041E-20, 0.1191830515693705E-02, 0.1851908589351717E-02, 0.3115549452672914E-01, 0.1124126789939942E-28, 0.7898349055483677E-03, 0.3617649115749287E-02, 0.3850968595106850E-02, 0.4455486392688275E-26, 0.3971779475237804E-01, 0.1814591379891354E-01, 0.1766035472385010E-05, 0.2780981219403279E-20, 0.1014029004767683E-01, 0.8718057102094371E-02},
			info: 0,
		},
		{
			z:    []float64{0.3437994630115073E+00, 0.2969584004763312E+00, 0.5428789456218400E+00, 0.3643219910857877E+00, 0.8842011736515154E-01, 0.4426793535323088E+00, 0.9851327949196382E+00, 0.9928836193844874E+00, 0.9294888334528270E+00, 0.5131906939013953E+00, 0.8372743749579292E-01, 0.3799113781856175E+00, 0.9980979619320958E+00, 0.1940068951405882E+00, 0.1265916571930514E+00, 0.7533120158726621E+00, 0.7604310151358937E+00, 0.9521608240404056E-01, 0.6048411980078113E+00, 0.8887307551694945E-01, 0.9109406880842844E+00, 0.9457532858494480E+00, 0.7192031611135805E+00, 0.5051759890546635E+00, 0.1927673350758418E+00, 0.6784576615085791E+00, 0.4582307655806892E+00, 0.4110913566635671E+00, 0.5843861752552199E+00, 0.1849014446101248E+00, 0.5286425989283670E+00, 0.9425316584065953E+00, 0.9757360470900053E+00, 0.4697986765545260E+00, 0.5423082602059015E-01, 0.4516134263086808E+00, 0.2022962345269688E+00, 0.4899666677522230E-01, 0.9657583521001409E+00, 0.7147044884479304E+00, 0.8698989147298346E+00, 0.1620630683580001E+00, 0.4917083507892124E+00, 0.6587971895356357E+00, 0.2830046270176211E+00, 0.8907428232631098E-01, 0.3385480253038764E-01, 0.3808208973559474E+00, 0.6301790073720519E+00, 0.7283330529106518E+00, 0.8754045015579843E+00, 0.5691786300850922E+00, 0.2221459247333947E+00, 0.3807253530058278E+00, 0.5099681617674874E+00, 0.8815981352292609E+00, 0.2670679959789202E-02, 0.3061419069919761E+00, 0.9088634553510815E+00, 0.3137439308670030E+00, 0.3256630871489109E+00, 0.7397175549546020E-01, 0.8600700672393357E+00, 0.2944165267259768E-01, 0.1962740610392520E+00, 0.7223088666910011E+00, 0.1555262415199788E+00, 0.7683934132272936E+00, 0.9837306979871404E+00, 0.4307236017454147E+00, 0.2763099157748476E+00, 0.5623054633156477E-01, 0.1414438705425911E+00, 0.3910643421231946E+00, 0.3414855836828583E+00, 0.4833767934540261E+00, 0.1874948602631823E+00, 0.6456680649630069E+00, 0.5377739930676261E+00, 0.4911199998049517E+00, 0.3925147326457343E+00, 0.1205990368593943E+00, 0.6208170182205421E+00, 0.4308681300066786E+00},
			n:    21,
			zOut: []float64{0.2838978926644850E+01, 0.2478331275466363E+01, 0.2456585262660691E+01, 0.2180185738821122E+01, 0.1712066680562540E+01, 0.1602701074250422E+01, 0.1503969746560551E+01, 0.1233768461187729E+01, 0.9787455830966086E+00, 0.9668289373592873E+00, 0.7729605529095109E+00, 0.7035556930615693E+00, 0.5295938697567051E+00, 0.4165425267136870E+00, 0.3781386605204551E+00, 0.3701077812928417E+00, 0.1575477292510020E+00, 0.7328369731218511E-01, 0.3974472017556452E-01, 0.4256064253578745E-02, 0.1985700463370783E-02, 0.0000000000000000E+00, 0.2182718854229073E-04, 0.3091215918753839E-04, 0.1503969746560551E+01, 0.7889046064910921E-15, 0.8525655623819580E-10, 0.5987167574554257E-09, 0.1233768461187729E+01, 0.1850899338648654E-19, 0.6965867087478826E-06, 0.6532556049979120E-06, 0.9787455830966086E+00, 0.3101927297073854E-24, 0.3608646891594891E-06, 0.2907657476508735E-06, 0.9668289373592873E+00, 0.5936688180990121E-20, 0.1421522625242826E-07, 0.7063390122609291E-08, 0.7729605529095109E+00, 0.1231338578304599E-17, 0.2139987868232064E+02, 0.2139987868232063E+02, 0.9600000000000000E+02, 0.2854875283446712E+01, 0.2083333333333333E+01, 0.2575545978160743E-06, 0.5295938697567051E+00, 0.8537368260663878E-24, 0.6647760383095535E-05, 0.6726768703723720E-05, 0.4165425267136870E+00, 0.5898327163094257E-18, 0.6007815560144262E-05, 0.6070984870789135E-05, 0.3781386605204551E+00, 0.1932242212135588E-23, 0.5780348596252855E-05, 0.5727260717649887E-05, 0.3701077812928417E+00, 0.2291245128765917E-16, 0.5589406660782675E-05, 0.5617175435749363E-05, 0.1575477292510020E+00, 0.3855952104720306E-26, 0.8404567502440466E-05, 0.1060626986994499E-04, 0.7328369731218511E-01, 0.1176980470589750E-26, 0.3402042835087641E-05, 0.5518338211116594E-05, 0.3974472017556452E-01, 0.3697785493223493E-30, 0.6423930978115149E-05, 0.5690659965930955E-05, 0.4256064253578745E-02, 0.1813010266067178E-20, 0.3330330670135602E-05, 0.3509207052336586E-05, 0.1985700463370783E-02, 0.9911627066431288E-23, 0.2656536456367949E-04, 0.9644853918428974E-05},
			info: 0,
		},
		{
			z:    []float64{0.6087347824331316E+00, 0.2344592930815549E+00, 0.1839180152431182E+00, 0.8946895357068063E-01, 0.3789275576065679E+00, 0.1028144130339237E-01, 0.8432962228323871E+00, 0.5720035990048252E+00, 0.9674750777949459E+00, 0.1387850858951470E+00, 0.1049983295079311E+00, 0.1869597635484084E+00, 0.2049847474830355E+00, 0.9353902309354832E+00, 0.7711597045807418E+00, 0.6677931698783810E+00, 0.5710020050743191E+00, 0.7559352115290241E+00, 0.8799700817290574E+00, 0.2989086447618220E+00, 0.7223221852829088E+00, 0.9172044263776469E-01, 0.8222766733565161E+00, 0.4357755436106698E+00, 0.3174399370719186E+00, 0.3716323007311311E+00, 0.5710261889832198E+00, 0.5875728000816982E-01, 0.1970825867540514E+00, 0.7514821290356389E+00, 0.4768968814440278E+00, 0.1414806690779697E+00, 0.5126469713172739E+00, 0.2618595999903317E-01, 0.3414455658478163E+00, 0.7845736413672471E+00, 0.3193997824316201E+00, 0.5244207174429636E+00, 0.7114985438553419E+00, 0.6380953409710879E+00, 0.5110544105167975E+00, 0.6987353947091863E+00, 0.2359552404464268E+00, 0.1389779414569779E+00, 0.4283565484470430E+00, 0.5978343235372734E+00, 0.8941068622870674E+00, 0.9993918591572127E+00, 0.9276104816334276E+00, 0.6840640077048410E+00, 0.1906252952212643E+00, 0.3896055694019390E+00, 0.8889540838698040E-01, 0.6972017776398595E+00, 0.7403854978121883E+00, 0.9624319129844994E-01, 0.7107468555423346E-01, 0.2167351328914763E+00, 0.5250363507339506E-01, 0.3012298012035964E+00, 0.7467053513110199E+00, 0.6311877906561809E+00, 0.6465457722939830E+00, 0.9462483307457270E-01, 0.1615629644808589E+00, 0.3714392567485900E+00, 0.5909086992570731E+00, 0.2147078824711308E+00, 0.8083150503324107E+00, 0.1136496291628873E+00, 0.3556622878263437E+00, 0.1687673308056986E+00, 0.2663369573375586E+00, 0.9992292880074174E+00, 0.4743706799728067E+00, 0.3784791276003248E+00, 0.9641428139214289E+00, 0.4526980047510465E-01, 0.5291384859598749E+00, 0.9114927597067145E+00, 0.2015740390806801E+00, 0.9111109510473957E+00, 0.4797378470164849E+00, 0.4344842342843944E+00},
			n:    21,
			zOut: []float64{0.2535248843080872E+01, 0.2027297538476471E+01, 0.1999341423958793E+01, 0.1905819580237666E+01, 0.1558994176508815E+01, 0.1443542425236611E+01, 0.1321312714967014E+01, 0.1044341128979687E+01, 0.9175998827905956E+00, 0.8543653487373744E+00, 0.6128068085852322E+00, 0.5723484802692980E+00, 0.4865842293721045E+00, 0.4743009418296827E+00, 0.4424121716041568E+00, 0.2157744105271115E+00, 0.1401155020588904E+00, 0.9545054691199920E-01, 0.4846378139764250E-01, 0.2597131641839770E-01, 0.9574417588705024E-02, 0.7983827718913915E-23, 0.5897957913656464E-08, 0.4905965358471980E-08, 0.1321312714967014E+01, 0.1437620113674772E-24, 0.2330664696165742E-08, 0.2036082149427908E-08, 0.1044341128979687E+01, 0.7099748146989106E-29, 0.1191493821660667E-08, 0.1294122655697948E-08, 0.9175998827905956E+00, 0.7069460558567906E-16, 0.2320495180901029E-08, 0.3242686959161780E-08, 0.8543653487373744E+00, 0.3414430154917050E-13, 0.2143028595220802E-07, 0.3181347787023606E-07, 0.6128068085852322E+00, 0.0000000000000000E+00, 0.1873166566953712E+02, 0.1873166566953712E+02, 0.1040000000000000E+03, 0.3224489795918367E+01, 0.2884615384615385E+01, 0.3533200143891392E-06, 0.4865842293721045E+00, 0.4552522969733916E-17, 0.1776353409716398E-09, 0.6419018055060889E-09, 0.4743009418296827E+00, 0.4679164786798698E-18, 0.4782181507340309E-06, 0.1264406900548246E-06, 0.4424121716041568E+00, 0.1728289684559415E-20, 0.3046857378867568E-04, 0.2182730305393890E-04, 0.2157744105271115E+00, 0.1120182485413837E-27, 0.4003877877583194E-03, 0.2588513428463001E-03, 0.1401155020588904E+00, 0.0000000000000000E+00, 0.6916990427934659E-04, 0.5501406364660823E-04, 0.9545054691199920E-01, 0.6162975822039155E-32, 0.1059944134272042E-03, 0.1421052287262650E-03, 0.4846378139764250E-01, 0.1755514705909767E-17, 0.4499204819889415E-04, 0.2567010249779503E-03, 0.2597131641839770E-01, 0.1143848312570467E-28, 0.5289175880660015E-04, 0.5437725505075631E-04, 0.9574417588705024E-02, 0.3697785493223493E-31, 0.2606391166909807E-05, 0.6294847321319143E-06},
			info: 0,
		},
		{
			z:    []float64{0.3257261020965838E+00, 0.9499222527145429E+00, 0.3597416524070001E+00, 0.7608369220069154E+00, 0.3348964244425234E-01, 0.9918240230913730E+00, 0.2435073049094116E-01, 0.2927090414647138E-01, 0.2724578324033788E+00, 0.7486167980650003E+00, 0.2854897136111678E+00, 0.2267361867398041E+00, 0.4294380831801550E+00, 0.7722184192395405E+00, 0.5889718788171796E+00, 0.5201154937785833E+00, 0.1341644467607439E-01, 0.3251983707424297E+00, 0.3712514383296180E+00, 0.8346822920283823E+00, 0.2613611110008908E+00, 0.7377020224374462E+00, 0.4290238209924468E-01, 0.8422463567888154E+00, 0.1390406385354674E+00, 0.8899155813346852E+00, 0.5366293959824918E+00, 0.1311808378334547E+00, 0.5127829646320820E+00, 0.3806825827063995E+00, 0.9008244723505014E+00, 0.8397173164045483E+00, 0.8005434668537068E+00, 0.4188784622408616E+00, 0.4815341947551696E+00, 0.2004130034689415E+00, 0.9620560381224039E+00, 0.9936712626034268E+00, 0.7996240982628953E+00, 0.9066259300527590E+00, 0.7119745510778175E+00, 0.4009593231138309E+00, 0.7503038869147766E+00, 0.2307657656692345E+00, 0.5796381959344894E+00, 0.4564585857276309E+00, 0.3087632483431468E-01, 0.4925813444884515E+00, 0.2548189626174928E+00, 0.7548007003619551E+00, 0.2595357873879023E+00, 0.2576161595637174E+00, 0.6530834734960895E+00, 0.9659056802808288E+00, 0.4073417244577716E+00, 0.7973054858347279E+00, 0.1098542135402614E-03, 0.5270152208809449E+00, 0.9503221531609122E+00, 0.3389760481404548E+00, 0.2936463197963648E+00, 0.1532837593240226E+00, 0.7039965742744211E+00, 0.1538522429841973E+00, 0.4565500206514618E+00, 0.4890306322657354E+00, 0.7406911389536464E+00, 0.4609110724077861E+00, 0.3010957214922741E+00, 0.4706875322819619E+00, 0.9245411658881026E+00, 0.1225486958523536E-02, 0.7513538867599376E+00, 0.7480092012350148E+00, 0.1668544624863488E+00, 0.6027497133760584E+00, 0.4060848284780354E+00, 0.7777661676033926E+00, 0.5476587806685121E+00, 0.3318235867866506E+00, 0.4865355710680780E+00, 0.9789804028805801E+00, 0.6449091593541079E-01, 0.3386899175207851E+00},
			n:    21,
			zOut: []float64{0.2916446272919928E+01, 0.2518523111789073E+01, 0.1881620740030630E+01, 0.1802535552376618E+01, 0.1705712498790935E+01, 0.1630934300031556E+01, 0.1493206464268041E+01, 0.1289863129801601E+01, 0.1141879970494692E+01, 0.1064939765404915E+01, 0.9148995873510519E+00, 0.6667419164223021E+00, 0.5793746811807822E+00, 0.5763823737203533E+00, 0.5577209455074719E+00, 0.2258972104173268E+00, 0.1775772659782911E+00, 0.1224839196229167E+00, 0.7643325286481288E-01, 0.1088888144792382E-01, 0.1013218159024319E-07, 0.1529626957613473E-16, 0.4760904670371191E-26, 0.1685698503770275E-25, 0.1493206464268041E+01, 0.4740164255731978E-15, 0.6203207773025231E-22, 0.1697026618649852E-21, 0.1289863129801601E+01, 0.1555744028694447E-17, 0.2220966892754199E-18, 0.8352563633635413E-19, 0.1141879970494692E+01, 0.8283039504820624E-29, 0.9137114046044313E-17, 0.2186983724354431E-16, 0.1064939765404915E+01, 0.7560947786617519E-24, 0.1476623491156538E-14, 0.6521525514986541E-15, 0.9148995873510519E+00, 0.3956572890903056E-23, 0.2135406185055340E+02, 0.2135406185055340E+02, 0.1110000000000000E+03, 0.3462585034013606E+01, 0.7207207207207207E+01, 0.3107640874581820E-11, 0.5793746811807822E+00, 0.0000000000000000E+00, 0.5611258671513064E-10, 0.1000932867794445E-09, 0.5763823737203533E+00, 0.2935229715008717E-19, 0.5869279282596528E-09, 0.3232392517855140E-09, 0.5577209455074719E+00, 0.7161080495560552E-17, 0.6771347567635994E-08, 0.1273942735381484E-07, 0.2258972104173268E+00, 0.2531843062293473E-19, 0.3233128964696781E-05, 0.1839431244951091E-05, 0.1775772659782911E+00, 0.1025519176787315E-28, 0.1632093348006395E-04, 0.1394944946706296E-04, 0.1224839196229167E+00, 0.3147671031182890E-20, 0.4090539240175316E-04, 0.3499438731982919E-04, 0.7643325286481288E-01, 0.4418024966021139E-23, 0.7751221398283570E-04, 0.6522231732461535E-04, 0.1088888144792382E-01, 0.0000000000000000E+00, 0.1168516922729128E-02, 0.6798960389683572E-03, 0.1013218159024319E-07, 0.8311253061997563E-15, 0.2266806408535672E-02, 0.8228900118596701E-02},
			info: 0,
		},
		{
			z:    []float64{0.9013837094783728E-01, 0.4093071371456288E+00, 0.7563443593246222E-01, 0.6047712831067179E+00, 0.3616734158068488E+00, 0.3179352368154293E+00, 0.4877726790168605E+00, 0.7283754332956183E+00, 0.6564126870433257E+00, 0.7426267273836333E+00, 0.9575970808689874E+00, 0.2328163542816707E+00, 0.9804488839127946E+00, 0.5880329556580101E+00, 0.6720101508999355E+00, 0.3029641470022006E+00, 0.9176198736416905E+00, 0.9273523415722473E-01, 0.7356501689695268E+00, 0.3516482504433937E+00, 0.5613309946663773E+00, 0.8064085547776983E+00, 0.2421909924481950E+00, 0.4321291285180304E+00, 0.7195726435630079E+00, 0.4201959590716675E-01, 0.6971840239025084E-03, 0.6931587576362976E+00, 0.4512374813335629E+00, 0.9105433528826555E+00, 0.8233348499205092E+00, 0.5270028711667427E+00, 0.5092902528068642E+00, 0.6527825111058926E+00, 0.8077059946216197E+00, 0.5037249441803524E+00, 0.7474524841642615E+00, 0.8361353459662745E+00, 0.6675800963161582E+00, 0.5004244151247563E+00, 0.7976263900002633E+00, 0.1983499023889042E+00, 0.7522366994446783E+00, 0.3654861843844912E+00, 0.8247542543249066E+00, 0.9124235995852386E+00, 0.1818459043196227E+00, 0.8196813052173683E+00, 0.5538880208895494E+00, 0.9080443570480472E+00, 0.2981632101337024E+00, 0.6044133722200155E+00, 0.1883025355842233E+00, 0.2434431972046255E+00, 0.3420561860884004E+00, 0.3312086041097358E+00, 0.8261796050670067E-01, 0.2597483126975232E+00, 0.9037350022806145E+00, 0.5691251256015616E+00, 0.5433137600205407E-01, 0.2948423088781353E+00, 0.7125818959241866E+00, 0.1168360183447441E+00, 0.2271931321361120E-01, 0.5832011197836875E+00, 0.4321285977614958E+00, 0.9214755199664483E+00, 0.2929460789941122E+00, 0.9911839954873096E+00, 0.9352897353900299E+00, 0.6549195608155846E+00, 0.7733376794047198E-01, 0.7345459966842849E+00, 0.2997108680399072E+00, 0.6923968300089707E+00, 0.9895794156524929E+00, 0.2694202749089550E+00, 0.2419924375765692E-01, 0.2728256718882756E+00, 0.7688978378093437E+00, 0.6794051971520565E-01, 0.8211384369868754E+00, 0.7581877817293103E-01},
			n:    21,
			zOut: []float64{0.2587082763537058E+01, 0.2518438946693926E+01, 0.2381638511714923E+01, 0.2064758819607097E+01, 0.1861546331198453E+01, 0.1823771413694203E+01, 0.1462413605342042E+01, 0.1295483717305927E+01, 0.1204792654692795E+01, 0.1153126691944095E+01, 0.9646786889507554E+00, 0.8876075724637406E+00, 0.7328920239265601E+00, 0.5128961040525114E+00, 0.4187721082034697E+00, 0.3198505150609477E+00, 0.1661143074789019E+00, 0.1249876230738883E+00, 0.5554047196429147E-01, 0.2079344839078052E-02, 0.4713171572265143E-04, 0.1874281408120013E-19, 0.2104034717604745E-09, 0.2619972849686678E-09, 0.1462413605342042E+01, 0.9259545558536950E-18, 0.2890370685025388E-10, 0.3242632456342165E-10, 0.1295483717305927E+01, 0.1323488980084844E-22, 0.2279224462592833E-10, 0.2555931367214743E-10, 0.1204792654692795E+01, 0.2978738778114541E-26, 0.7485181957332586E-10, 0.4566030880095840E-10, 0.1153126691944095E+01, 0.6733803272339514E-21, 0.2633277318950464E-08, 0.1440235383519861E-08, 0.9646786889507554E+00, 0.7499216124683276E-17, 0.2253851934746039E+02, 0.2253851934746038E+02, 0.1040000000000000E+03, 0.3111111111111111E+01, 0.5769230769230769E+01, 0.2633869415622400E-06, 0.7328920239265601E+00, 0.6980417424693642E-13, 0.1603044445805627E-06, 0.1255532606981617E-06, 0.5128961040525114E+00, 0.6981404369947554E-21, 0.8484909168369356E-07, 0.7367232980661418E-07, 0.4187721082034697E+00, 0.2465190328815662E-30, 0.1470967366725541E-06, 0.2274394647752198E-06, 0.3198505150609477E+00, 0.9424343740971753E-25, 0.8932721831570202E-05, 0.5259727349591482E-05, 0.1661143074789019E+00, 0.1697763391846713E-18, 0.4712327641645974E-05, 0.7305995006514358E-05, 0.1249876230738883E+00, 0.9266645908481002E-19, 0.3596318120009657E-04, 0.2001742175196116E-04, 0.5554047196429147E-01, 0.5985976337495067E-16, 0.4811066675748340E-03, 0.3416912105856511E-03, 0.2079344839078052E-02, 0.8697917232094458E-24, 0.5710266423841766E-02, 0.2981491759065521E-02, 0.4713171572265143E-04, 0.0000000000000000E+00, 0.3288287603570621E-02, 0.1038987759212308E-02},
			info: 0,
		},
		{
			z:    []float64{0.3453921918700114E+00, 0.8256584009865165E+00, 0.5027306146870630E+00, 0.5014619412470180E+00, 0.9910264683396075E+00, 0.7385152238869899E+00, 0.9936695869989243E+00, 0.6206355260490602E+00, 0.6835539321104728E+00, 0.3987727521414011E+00, 0.2155721742355131E+00, 0.8212295891891395E+00, 0.7021626430853991E+00, 0.3753314442932942E+00, 0.2575710525141791E+00, 0.6551328468445583E+00, 0.3738398426943970E+00, 0.8805730558289802E+00, 0.6848909798416613E+00, 0.5520069040756137E-01, 0.8817067703079440E+00, 0.3238117712584516E+00, 0.1001825089826291E+00, 0.7369585910327309E+00, 0.7601318359400422E+00, 0.4312449588193277E+00, 0.4932891154015395E+00, 0.1099211239093930E+00, 0.5764729343614433E+00, 0.1286539516629047E+00, 0.5170000684627285E+00, 0.4006756226519864E+00, 0.5897018197291505E+00, 0.3774659871747915E+00, 0.4211183018281148E+00, 0.6260997273722376E+00, 0.7810234892670644E+00, 0.9934154489086249E+00, 0.5116204079460294E+00, 0.7948452919719275E+00, 0.5998807734230309E+00, 0.4868691071952969E-01, 0.7977221292955683E+00, 0.4907992372099745E+00, 0.4337635562764541E-01, 0.4865658084478622E+00, 0.7887982210525304E+00, 0.6218441262679077E+00, 0.1807768964120138E-01, 0.7341707407508724E+00, 0.4738048273586776E+00, 0.9324751870816922E+00, 0.4643019354540665E+00, 0.2401905271601299E+00, 0.6072757133952530E+00, 0.6257046236339610E+00, 0.1662572295885112E-01, 0.3360107439465247E+00, 0.5925152832499899E+00, 0.4206234399601897E+00, 0.1828405120649758E+00, 0.8923354266778754E+00, 0.6211472795399554E+00, 0.3215240169291051E+00, 0.3967102459115966E+00, 0.4376502563326659E+00, 0.4571232280132043E-01, 0.2973405440159681E+00, 0.1737124961451384E+00, 0.1123665207380958E+00, 0.8069742345287393E+00, 0.5302931605046994E+00, 0.2541675285808478E+00, 0.9099997182043015E+00, 0.7562530283478575E-01, 0.2521110146593245E+00, 0.2525672524245413E+00, 0.8930826245328107E+00, 0.8545394186204489E+00, 0.1856025965008533E+00, 0.8541401621096930E+00, 0.7406868214039234E+00, 0.5851270108744718E+00, 0.7554853143122133E+00},
			n:    21,
			zOut: []float64{0.2818806710434786E+01, 0.2596766072486540E+01, 0.2041400548479265E+01, 0.1919012388179902E+01, 0.1900863080415201E+01, 0.1778136512482050E+01, 0.1649297429111022E+01, 0.1375943901620400E+01, 0.1287933944237744E+01, 0.1246950647018126E+01, 0.9346701076388835E+00, 0.8619836761278706E+00, 0.6928534822715838E+00, 0.5816592049742084E+00, 0.4759893592204403E+00, 0.2537953742738920E+00, 0.2205935192530503E+00, 0.8350022078832350E-01, 0.3307087507037688E-01, 0.2105652896299196E-01, 0.3857874617179747E-02, 0.9177981866196648E-17, 0.6043035417783778E-14, 0.7682115486038763E-14, 0.1649297429111022E+01, 0.9047287949798740E-23, 0.2120842457393041E-14, 0.2617862539054627E-14, 0.1375943901620400E+01, 0.1499552906513639E-17, 0.3376573256769321E-15, 0.2886563968474245E-15, 0.1287933944237744E+01, 0.1966621906344823E-17, 0.2937881411718043E-15, 0.4577311861870994E-15, 0.1246950647018126E+01, 0.2742371736879744E-17, 0.2913637797754287E-14, 0.1483948533056680E-14, 0.9346701076388835E+00, 0.5800058106323822E-18, 0.2277814145766384E+02, 0.2277814145766384E+02, 0.9800000000000000E+02, 0.2997732426303855E+01, 0.1020408163265306E+01, 0.9030119244814722E-12, 0.6928534822715838E+00, 0.1471303281095442E-16, 0.2634125396797654E-10, 0.5071585384120369E-10, 0.5816592049742084E+00, 0.1365675999118616E-25, 0.3358753759221612E-09, 0.6258446588335648E-09, 0.4759893592204403E+00, 0.1479114197289397E-30, 0.3734630956704811E-08, 0.6716634735981551E-08, 0.2537953742738920E+00, 0.7467396771867359E-25, 0.3072344915986320E-06, 0.1829961043616119E-06, 0.2205935192530503E+00, 0.4437342591868191E-30, 0.2385309167517414E-05, 0.3978543929472447E-05, 0.8350022078832350E-01, 0.2436129622029919E-19, 0.4589207629691226E-04, 0.2836221395611648E-04, 0.3307087507037688E-01, 0.3081487911019577E-32, 0.1362072635971665E-04, 0.2978962617014088E-04, 0.2105652896299196E-01, 0.2711709361697228E-30, 0.2444625596237189E-05, 0.7827983423195974E-05, 0.3857874617179747E-02, 0.8925769947935872E-20, 0.9826965253502863E-02, 0.1259209175531780E-01},
			info: 0,
		},
		{
			z:    []float64{0.9560856702670685E+00, 0.5885863857778361E+00, 0.8071258175651005E+00, 0.5175476341605639E+00, 0.1941426739297742E+00, 0.8451490614471896E+00, 0.9052305950784733E+00, 0.6791526261185319E+00, 0.6480230038264980E+00, 0.8438040949137990E+00, 0.3034844043967853E+00, 0.3090582723248356E+00, 0.3282769832503399E+00, 0.7489751201458807E+00, 0.6672021557248126E+00, 0.1697989591956236E+00, 0.5723860504781838E+00, 0.3660621222567975E+00, 0.1505291330148173E-01, 0.7436839329892437E+00, 0.3679212455376497E+00, 0.5305714072096270E+00, 0.5792221354999488E+00, 0.3031533452885045E+00, 0.9514596930492207E+00, 0.3278419583960311E+00, 0.5742176036064726E+00, 0.7216680447710266E+00, 0.6608941225668634E+00, 0.3887216422692293E+00, 0.6952172825541724E+00, 0.8072929471269218E+00, 0.8776672848787970E-01, 0.9363773133978057E+00, 0.3470717327786855E+00, 0.9438495003134180E+00, 0.5511880105145929E+00, 0.5826637033513504E+00, 0.3416802173197317E+00, 0.5650234571673005E+00, 0.2927187699119284E+00, 0.4936451925034285E+00, 0.7496439056333554E+00, 0.8462697295115110E+00, 0.4661250827466905E+00, 0.3912590593193351E+00, 0.9140228064649625E+00, 0.4016314378444454E+00, 0.7105479239000361E+00, 0.3751330850058705E+00, 0.7309804928539319E+00, 0.1378037023247644E+00, 0.7527056268407295E+00, 0.1640478774378548E+00, 0.6030857470066908E+00, 0.7427119073248016E+00, 0.9996405952923426E-01, 0.7406467144237725E+00, 0.3762498316380406E+00, 0.8295145255291534E+00, 0.8322845800948830E-01, 0.6101670185797291E+00, 0.9451596878306588E+00, 0.1176379504958335E+00, 0.7940514845464750E+00, 0.3068100097277620E+00, 0.2371282369406802E+00, 0.2163781309135768E+00, 0.4705542393165623E+00, 0.4003463805482307E+00, 0.2731285781388088E+00, 0.8862626881898287E+00, 0.5854390777280626E+00, 0.4661955367614972E+00, 0.1993415035442425E+00, 0.9213995968359628E+00, 0.1542739762296447E+00, 0.9419076789558214E+00, 0.9611493561784724E+00, 0.7596360150269755E+00, 0.6590873675145426E+00, 0.3894853423829039E+00, 0.8407621320868611E+00, 0.4517123559923821E+00},
			n:    21,
			zOut: []float64{0.2583735261373388E+01, 0.2239759728480232E+01, 0.2157858012928955E+01, 0.2146305201359927E+01, 0.1853050031878152E+01, 0.1788497392649839E+01, 0.1558387970436693E+01, 0.1494017846927041E+01, 0.1323723239351723E+01, 0.1188003233397794E+01, 0.9827022795744828E+00, 0.8335582164080391E+00, 0.7854672559750093E+00, 0.6236176061997519E+00, 0.4677974993044671E+00, 0.2968385185647548E+00, 0.2304456569541352E+00, 0.1793008998018142E+00, 0.2471364396669566E-01, 0.7537422761275628E-02, 0.3241997301555496E-04, 0.1184063810429910E-21, 0.1281976731077423E-10, 0.2849227344309828E-10, 0.1558387970436693E+01, 0.3001369155282220E-19, 0.2940080271049715E-08, 0.1685793775386140E-08, 0.1494017846927041E+01, 0.1696366490587264E-24, 0.4557030850938984E-08, 0.5166797185591864E-08, 0.1323723239351723E+01, 0.1862197338326049E-12, 0.1908610209474282E-08, 0.1816527091730617E-08, 0.1188003233397794E+01, 0.1613415991578665E-17, 0.2801618631477443E-08, 0.2177314380835442E-08, 0.9827022795744828E+00, 0.1073695634537585E-17, 0.2276534933826718E+02, 0.2276534933826718E+02, 0.1060000000000000E+03, 0.3113378684807256E+01, 0.3773584905660377E+01, 0.2107018980320373E-07, 0.7854672559750093E+00, 0.7395570986446986E-31, 0.5134101120680572E-07, 0.6949862017183880E-07, 0.6236176061997519E+00, 0.1700003115737185E-12, 0.4917011909790346E-06, 0.6444201346122313E-06, 0.4677974993044671E+00, 0.1676329423594650E-29, 0.2178752696080751E-05, 0.1697685142168825E-05, 0.2968385185647548E+00, 0.2903323170704433E-20, 0.9062193610690628E-05, 0.7062135859641892E-05, 0.2304456569541352E+00, 0.9860761315262648E-30, 0.1460945060339096E-04, 0.1904220434906198E-04, 0.1793008998018142E+00, 0.4606947686490709E-27, 0.4451684144215141E-07, 0.1728984935219646E-06, 0.2471364396669566E-01, 0.1407327854914285E-26, 0.1942342839475662E-02, 0.3052712133602924E-02, 0.7537422761275628E-02, 0.9865964111678957E-20, 0.3214451590395407E-02, 0.5215635567340069E-02, 0.3241997301555496E-04, 0.0000000000000000E+00, 0.1184307710713820E-01, 0.9427403294935964E-02},
			info: 0,
		},
		{
			z:    []float64{0.4101594051788160E+00, 0.1002974419660251E+00, 0.8434067081152193E+00, 0.7258916354493068E+00, 0.9256853341957688E+00, 0.8379265467529036E+00, 0.1564408324797836E+00, 0.8292039738515364E+00, 0.3771190822561328E+00, 0.8236097732630416E+00, 0.3320638300046872E+00, 0.6850750205938005E-01, 0.5616647045816037E+00, 0.8305938719521835E+00, 0.6006351465721590E+00, 0.8469146468608777E+00, 0.1179160781320751E+00, 0.6975716721259332E+00, 0.3237957597653003E+00, 0.1297318227605904E-01, 0.8416519209661123E+00, 0.2623987624579764E+00, 0.3323289069826861E+00, 0.9848995259497869E+00, 0.1295798367899781E-01, 0.4196318718239949E+00, 0.8791346827157692E+00, 0.9015205418757779E+00, 0.2557984197164971E+00, 0.7295221077613672E+00, 0.8796719157178668E+00, 0.6983240767110626E+00, 0.6461186804049626E+00, 0.5003600760145155E-01, 0.3411701822058120E+00, 0.6026846298507913E+00, 0.4649210011845201E+00, 0.4169632595817018E+00, 0.9745024564197420E+00, 0.9159749674174896E+00, 0.3798779762430529E+00, 0.6757581567027395E+00, 0.1501600637708167E+00, 0.8841295845033917E+00, 0.1700642059156726E+00, 0.5745602516925833E+00, 0.7146078194346678E+00, 0.7876321314233041E+00, 0.3623958693011297E+00, 0.1316504845867333E+00, 0.6150819295249010E+00, 0.9948396700221377E+00, 0.2186788175334711E+00, 0.3433450548510013E+00, 0.4434886569716980E+00, 0.6596978167473024E+00, 0.7067569387667516E+00, 0.2710992196058543E+00, 0.2531644393448058E+00, 0.6179743389634106E+00, 0.6395694224823617E+00, 0.6233992467844496E-01, 0.9270082862838847E-01, 0.6813409948026068E+00, 0.8489230752840353E+00, 0.4248777519526148E+00, 0.9258241367507150E+00, 0.7591306919901064E-01, 0.1337231221457533E+00, 0.6673452115706187E+00, 0.3161361894802919E+00, 0.9287930266484576E+00, 0.9211411715651807E+00, 0.7110889130885073E+00, 0.6783920645346327E+00, 0.5798290672446252E+00, 0.6003988035906362E+00, 0.8338909998640337E+00, 0.1150264810560550E+00, 0.8268177524919460E+00, 0.2868686997384918E+00, 0.7734675037869032E+00, 0.8495624932870276E+00, 0.7382167415731139E+00},
			n:    21,
			zOut: []float64{0.2565771617602568E+01, 0.2464815559540473E+01, 0.2377249642737961E+01, 0.2182027799722737E+01, 0.1994511663794419E+01, 0.1777930797653110E+01, 0.1532836111328556E+01, 0.1298104160091711E+01, 0.1162548328490141E+01, 0.1031426069242260E+01, 0.9067767207978197E+00, 0.7702116437984168E+00, 0.7023507576840374E+00, 0.5846573732521254E+00, 0.3836750731694669E+00, 0.3732780616075560E+00, 0.2146097506115993E+00, 0.5462411591703480E-01, 0.3208967690429450E-01, 0.2788258306628742E-02, 0.1838228532938987E-03, 0.1703684557466520E-17, 0.8886730161731292E-14, 0.1725592004801531E-13, 0.1532836111328556E+01, 0.2310659810285515E-15, 0.8778995686431983E-12, 0.1396745951735637E-11, 0.1298104160091711E+01, 0.0000000000000000E+00, 0.8497749401208094E-11, 0.5925398902948847E-11, 0.1162548328490141E+01, 0.1292469707114106E-25, 0.1515360700076446E-10, 0.2085752134421418E-10, 0.1031426069242260E+01, 0.1195331315767287E-17, 0.8898419915788489E-10, 0.1186621585503233E-09, 0.9067767207978197E+00, 0.1888339631531495E-12, 0.2241246700510621E+02, 0.2241246700510621E+02, 0.1190000000000000E+03, 0.3539682539682540E+01, 0.7563025210084033E+01, 0.2596357213906324E-08, 0.7023507576840374E+00, 0.1416275483756143E-17, 0.5106480833143134E-08, 0.6438584684288567E-08, 0.5846573732521254E+00, 0.1317619996636992E-15, 0.1896058302301025E-07, 0.2367797858508706E-07, 0.3836750731694669E+00, 0.9199951564453634E-11, 0.8967796171107713E-07, 0.1296821709728121E-06, 0.3732780616075560E+00, 0.1615587133892578E-26, 0.3088999981101691E-06, 0.2343043411729366E-06, 0.2146097506115993E+00, 0.2055968734232262E-28, 0.9134179982896841E-05, 0.6280501592969192E-05, 0.5462411591703480E-01, 0.1901142159808155E-23, 0.3768301572228235E-05, 0.8777513601148953E-05, 0.3208967690429450E-01, 0.1479114197289397E-29, 0.1428515450408294E-03, 0.5208612044311805E-04, 0.2788258306628742E-02, 0.2057124874537592E-17, 0.2063479623178836E-02, 0.3566698154744510E-02, 0.1838228532938987E-03, 0.3273772756667199E-28, 0.8538542526496483E-03, 0.1571849612061994E-03},
			info: 0,
		},
		{
			z:    []float64{0.1589037341524596E+00, 0.1455226073481288E+00, 0.1938395273812957E+00, 0.1962691615573132E+00, 0.1687822326883440E+00, 0.6650981615477494E+00, 0.8270615916362424E+00, 0.1710502783654970E+00, 0.2206713720159043E+00, 0.4369721197399372E+00, 0.9301314325299896E+00, 0.5722466234411457E+00, 0.1135089451836189E+00, 0.5846068541971138E+00, 0.9216025315561343E+00, 0.9066267701445474E+00, 0.5584279552920539E+00, 0.4343925455049338E+00, 0.9858452060735702E+00, 0.6563046209578444E+00, 0.7077011175365495E+00, 0.4947310926689127E+00, 0.7385820400647596E+00, 0.5937728681009750E+00, 0.9762402194896037E+00, 0.3450302019452143E+00, 0.3805099852593911E+00, 0.7529397554746390E+00, 0.7330092480109731E+00, 0.3330086450808912E+00, 0.9038518043983117E+00, 0.6574079221280790E+00, 0.2509077957372493E+00, 0.6249580721461656E+00, 0.2757992349681960E+00, 0.5001238277403597E+00, 0.9621194298806066E+00, 0.3061711826567661E+00, 0.1119838022409911E+00, 0.2976298035182879E+00, 0.6363137835930621E-01, 0.4930031667447814E+00, 0.1865676338649569E-01, 0.4250211458984191E+00, 0.1095501188334868E+00, 0.2763357459722963E+00, 0.7075737347301593E+00, 0.5363987747039417E+00, 0.3986581101260644E+00, 0.2692140175173962E+00, 0.5588052030209328E+00, 0.5539805437492590E+00, 0.6882419411200217E+00, 0.6525011142322502E+00, 0.2351213214810641E+00, 0.4862287407564582E+00, 0.9494457572755843E-01, 0.2775605674126462E+00, 0.6088686305117397E-01, 0.1602712446759722E+00, 0.6726078044316741E-01, 0.1771280849255007E+00, 0.9403546682132294E+00, 0.1182787468569617E+00, 0.6084837514797026E+00, 0.4725411123129805E+00, 0.2891346904124859E-01, 0.3056197468639377E-01, 0.3597281640784106E+00, 0.3047108802830497E+00, 0.9462624965987684E+00, 0.6795048656132132E+00, 0.3142878274718968E+00, 0.2255775176392846E+00, 0.5741746909618360E+00, 0.6125119194512041E+00, 0.8636232492271716E+00, 0.5595270150362563E-01, 0.4894380312911951E+00, 0.2945948126181174E+00, 0.4782956027971728E+00, 0.1100404727801326E+00, 0.2692801555328298E+00, 0.4567347029351195E+00},
			n:    21,
			zOut: []float64{0.2662663085210597E+01, 0.2404979690430721E+01, 0.2214109781172517E+01, 0.2010332580463078E+01, 0.1879931283999763E+01, 0.1831239586461748E+01, 0.1664682288858752E+01, 0.1512516241203824E+01, 0.1081258277181201E+01, 0.8471473481460485E+00, 0.6221178288483303E+00, 0.5493156782308360E+00, 0.4496825320738197E+00, 0.3768653101376996E+00, 0.2833465507506064E+00, 0.2013980912694376E+00, 0.1418989984144841E+00, 0.7096946004385345E-01, 0.4542843847166723E-01, 0.6747056508451696E-02, 0.1343590842619022E-02, 0.4141519752410312E-28, 0.4845145585707669E-05, 0.3343357997526164E-05, 0.1664682288858752E+01, 0.8527270841077049E-25, 0.1655046287299363E-04, 0.2231546414257453E-04, 0.1512516241203824E+01, 0.5692210911982269E-19, 0.1383329297784762E-03, 0.1072264803352635E-03, 0.1081258277181201E+01, 0.1459392674658872E-28, 0.4158771817821724E-03, 0.3910579975830214E-03, 0.8471473481460485E+00, 0.2949199315750023E-16, 0.2216793811640345E-03, 0.2203201078239036E-03, 0.6221178288483303E+00, 0.2055113688275276E-18, 0.2085797369872005E+02, 0.2085797369872006E+02, 0.9000000000000000E+02, 0.2732426303854875E+01, 0.0000000000000000E+00, 0.1772659828830871E-05, 0.4496825320738197E+00, 0.4437342591868191E-30, 0.2138076899966292E-04, 0.1340116970023588E-04, 0.3768653101376996E+00, 0.2904017873172006E-24, 0.4668906275608195E-04, 0.7182837805013731E-04, 0.2833465507506064E+00, 0.4888968615550842E-22, 0.1290326696826671E-05, 0.6479808231987577E-06, 0.2013980912694376E+00, 0.4436553730962970E-26, 0.1673384055036320E-03, 0.7106703019344439E-04, 0.1418989984144841E+00, 0.3443985257364436E-16, 0.1415746442632383E-02, 0.1286112519803737E-02, 0.7096946004385345E-01, 0.1380506584136771E-29, 0.1898294559619081E-02, 0.3536485589649533E-02, 0.4542843847166723E-01, 0.9860761315262648E-31, 0.1085724134073318E-03, 0.2838967874935677E-03, 0.6747056508451696E-02, 0.2742900553697359E-16, 0.1307657950440773E-02, 0.4192938789348323E-02, 0.1343590842619022E-02, 0.2423380700838948E-25, 0.4340570868086748E-02, 0.9691995934364128E-02},
			info: 0,
		},
		{
			z:    []float64{0.6603307676446488E+00, 0.5215490552441340E+00, 0.9279560414420993E+00, 0.3871591216009305E+00, 0.4489001158072897E+00, 0.2115798060938074E+00, 0.6013990647859857E-01, 0.6977506014884622E+00, 0.8981757899514635E+00, 0.6366966965554082E+00, 0.5447619366248987E+00, 0.2926906196279220E+00, 0.9538633412864880E+00, 0.4052441866103130E+00, 0.9970193775088028E+00, 0.1550480233350559E+00, 0.9697068099139655E+00, 0.3748402815554551E+00, 0.3906465669660910E+00, 0.4103562008634933E+00, 0.9514404866677418E+00, 0.5608812699221558E+00, 0.6226680723027547E+00, 0.8653188982275728E+00, 0.3999642050403411E+00, 0.8968004947108832E+00, 0.1007886665847859E+00, 0.2055140520989622E+00, 0.8479005785073866E+00, 0.4772281284198989E+00, 0.3664785333935638E+00, 0.9311163393120341E+00, 0.2051534177095988E+00, 0.5588930379235971E+00, 0.9385948972790567E+00, 0.1380523811906209E+00, 0.9076293986285472E+00, 0.9937542903014829E+00, 0.5330397746663563E+00, 0.5373590340921116E+00, 0.2667502750074380E-01, 0.9996497580467543E+00, 0.7460509377370035E+00, 0.5868152625248035E+00, 0.2352631583529476E+00, 0.2062979890255305E+00, 0.1864397664760209E-02, 0.1562262993494207E+00, 0.1880894213999632E+00, 0.5369024951393064E+00, 0.6560943776892021E+00, 0.9558063214302399E+00, 0.1463497248954505E+00, 0.6977928179134164E+00, 0.8204108917709737E+00, 0.9530026100161404E+00, 0.7191901603465490E+00, 0.4974721533134605E+00, 0.8445129869563219E+00, 0.9007488855711614E+00, 0.4164322161987429E+00, 0.5750362072910759E+00, 0.1935672755254650E+00, 0.2663393354536251E+00, 0.3913931132271733E+00, 0.7724103066167269E+00, 0.7871420607985310E+00, 0.3766724710952808E+00, 0.9224124235283799E+00, 0.1080194858843619E+00, 0.3512364719709385E+00, 0.9868075881536598E-02, 0.9893144531298202E+00, 0.5983612801716819E+00, 0.2882889058269555E-01, 0.2759662054778022E+00, 0.4865673169207868E+00, 0.4317793431152178E+00, 0.4039098350536063E-01, 0.7594353556100967E+00, 0.6702102151313558E+00, 0.5757953377080282E+00, 0.7036908158147406E+00, 0.3078985083785768E+00},
			n:    21,
			zOut: []float64{0.2422149931211175E+01, 0.2394118175176597E+01, 0.2305882752563478E+01, 0.2031709011458775E+01, 0.2015634640599460E+01, 0.1933740155888154E+01, 0.1715509223328114E+01, 0.1559424733880939E+01, 0.1425500044444135E+01, 0.1021162532677627E+01, 0.8999999270989295E+00, 0.8371987758218183E+00, 0.7469639623683311E+00, 0.6494523294995699E+00, 0.4251816116188064E+00, 0.2881024603279346E+00, 0.2247792687845145E+00, 0.8318936570759092E-01, 0.1886599847837073E-01, 0.1029977709531541E-01, 0.8015530498900421E-03, 0.8380373580927862E-22, 0.1621276583815588E-04, 0.5889624784943332E-05, 0.1715509223328114E+01, 0.2616454581976124E-17, 0.2563506401168800E-03, 0.3048038004625151E-03, 0.1559424733880939E+01, 0.5629222562364776E-20, 0.1824080863602310E-03, 0.8021857862878124E-04, 0.1425500044444135E+01, 0.1078233452249475E-15, 0.2986624164553499E-03, 0.4306384056300427E-03, 0.1021162532677627E+01, 0.2425965640253176E-21, 0.3415477187080772E-04, 0.2730971166672021E-04, 0.8999999270989295E+00, 0.1110716154551185E-26, 0.2300966623107952E+02, 0.2300966623107953E+02, 0.9800000000000000E+02, 0.2877551020408163E+01, 0.3061224489795918E+01, 0.5341669529353412E-07, 0.7469639623683311E+00, 0.1147369787651374E-23, 0.3199917396615717E-04, 0.1114773640824220E-03, 0.6494523294995699E+00, 0.1324417262847731E-16, 0.4143040953885707E-03, 0.3815591968277185E-03, 0.4251816116188064E+00, 0.4294702353329713E-20, 0.5586157832836348E-03, 0.4985099391803953E-03, 0.2881024603279346E+00, 0.3449343154346125E-19, 0.3018840928486242E-04, 0.4804099248496092E-04, 0.2247792687845145E+00, 0.8223054085459871E-16, 0.6535536077614123E-03, 0.5704051254743707E-03, 0.8318936570759092E-01, 0.1269822990867506E-16, 0.2750660948852983E-03, 0.2439371709019565E-03, 0.1886599847837073E-01, 0.2261821987449685E-25, 0.1693307421164339E-03, 0.1879460762504110E-03, 0.1029977709531541E-01, 0.2207699818464282E-23, 0.8362752373159153E-06, 0.5564443135649789E-05, 0.8015530498900421E-03, 0.1142002837193949E-22, 0.2005722186927794E-03, 0.1790331282576109E-02},
			info: 0,
		},
		{
			z:    []float64{0.7180522604364612E+00, 0.1926580568606098E+00, 0.9998274874952506E-01, 0.5945840846677781E+00, 0.6814004826706310E-01, 0.6532586662527488E+00, 0.6162535781196632E+00, 0.6323752897874746E+00, 0.6913197528064650E+00, 0.8218961980054600E+00, 0.2391583485552943E+00, 0.6893465701234240E+00, 0.8646254741258573E+00, 0.4399082441653793E+00, 0.7657207697437258E+00, 0.6442902814401660E+00, 0.2997828758931008E+00, 0.3311754429773528E+00, 0.4602753981904284E-01, 0.2937498413335546E+00, 0.9438033090912972E+00, 0.6689413829611792E+00, 0.3690605709328096E+00, 0.1297746871929878E+00, 0.5761771518385747E+00, 0.9470476076636891E+00, 0.6065387228873672E+00, 0.3279182193602236E+00, 0.7521029317260848E+00, 0.3448791591170053E-01, 0.8260379882894462E+00, 0.3381289908082724E+00, 0.6350192899014672E+00, 0.6181098759446546E+00, 0.7961594478307139E+00, 0.9929813242449080E+00, 0.4678192142649594E+00, 0.8317252674327552E+00, 0.1301328932823079E+00, 0.9710090231649569E+00, 0.1522128889234372E+00, 0.8594558260240532E+00, 0.6013715693159983E+00, 0.8742383713574672E+00, 0.7309721379377087E+00, 0.5021385621750218E+00, 0.9190365975533138E+00, 0.8027800227596282E-01, 0.1149439815205374E+00, 0.3491372322884074E+00, 0.3638810075690350E+00, 0.7397316797734275E+00, 0.6765505976038609E+00, 0.6412509132149137E+00, 0.6636796276324973E+00, 0.2769301898862327E+00, 0.6293983439129569E+00, 0.7661949976862148E+00, 0.1787796367414251E+00, 0.2134901744318570E+00, 0.8213730837735125E+00, 0.5732135541056227E+00, 0.8956530012694059E+00, 0.5095442527120628E+00, 0.8428538487299666E+00, 0.6837140150023939E+00, 0.1256023460764972E+00, 0.1828667371827231E-01, 0.3922785557614588E+00, 0.9906794824083128E+00, 0.6528984219237121E+00, 0.2175815548938115E+00, 0.5215238469901794E+00, 0.6874084104558049E+00, 0.4144717036123762E+00, 0.5901719497674505E+00, 0.3178975712304457E+00, 0.7061784208820305E+00, 0.6028763180221938E+00, 0.6012903376000152E+00, 0.6812913355997376E+00, 0.7728807542206401E+00, 0.4888911469874758E+00, 0.1012713084879602E+00},
			n:    21,
			zOut: []float64{0.2588311953706596E+01, 0.2370344164465735E+01, 0.2152772432177807E+01, 0.2121020324458281E+01, 0.1935798000017230E+01, 0.1639966751245229E+01, 0.1447991497477954E+01, 0.1303796318075928E+01, 0.1122178474206467E+01, 0.1028684266176079E+01, 0.9663785372526962E+00, 0.8401839996503420E+00, 0.6468021461987292E+00, 0.5148128374433424E+00, 0.4081393344645305E+00, 0.3744044838513741E+00, 0.1951863401557736E+00, 0.1222936930600034E+00, 0.3610852087288927E-01, 0.2215175692702855E-02, 0.1055251342481684E-03, 0.5816113682013476E-24, 0.3490790538048844E-06, 0.5279190866861155E-06, 0.1447991497477954E+01, 0.6749793798367842E-21, 0.1187240552619153E-05, 0.8669686727541259E-06, 0.1303796318075928E+01, 0.2818270319791895E-16, 0.1425495821091504E-08, 0.4238958678865879E-08, 0.1122178474206467E+01, 0.1974523135472784E-17, 0.8140039386639262E-06, 0.3278912117172865E-06, 0.1028684266176079E+01, 0.3004196424688596E-17, 0.2459469608149132E-06, 0.1027360875624831E-06, 0.9663785372526962E+00, 0.8319175140295936E-21, 0.2181749477578394E+02, 0.2181749477578394E+02, 0.1040000000000000E+03, 0.3040816326530612E+01, 0.3846153846153846E+01, 0.3006532860022149E-05, 0.6468021461987292E+00, 0.2129924444096732E-28, 0.3097317061779137E-05, 0.3301766384461962E-05, 0.5148128374433424E+00, 0.5841522899747606E-17, 0.6589564866042633E-05, 0.8825655271690012E-05, 0.4081393344645305E+00, 0.5522026336547083E-29, 0.6929345622607317E-04, 0.1121666398022498E-03, 0.3744044838513741E+00, 0.4071868297361024E-19, 0.4964849153130707E-03, 0.6240579194345771E-03, 0.1951863401557736E+00, 0.3096999756315481E-22, 0.1538823843362176E-02, 0.5557701536471033E-03, 0.1222936930600034E+00, 0.9508642973366516E-17, 0.9344625728185554E-03, 0.1303932473154565E-02, 0.3610852087288927E-01, 0.2671050208464689E-19, 0.5349955893117279E-02, 0.9991286391872665E-02, 0.2215175692702855E-02, 0.2465190328815662E-30, 0.2737788875312504E-02, 0.2378589653250575E-02, 0.1055251342481684E-03, 0.0000000000000000E+00, 0.1487915832966980E-01, 0.9945584373260818E-02},
			info: 0,
		},
		{
			z:    []float64{0.2443960928887767E+00, 0.5306690011327519E+00, 0.8844980893594125E+00, 0.8505050348613927E+00, 0.1179365025679910E+00, 0.6787232231004032E+00, 0.4781425309482933E+00, 0.8596658025119301E+00, 0.7267566040500681E+00, 0.7988986860541384E+00, 0.5066697395752245E+00, 0.9256193759419001E+00, 0.1614174453766865E+00, 0.7938782145253169E+00, 0.3207377639351938E+00, 0.5936236877473402E+00, 0.5294512496859689E+00, 0.3341300959064973E+00, 0.8739309914591177E+00, 0.1643518115289022E-01, 0.9695643985202173E+00, 0.9508789318813814E+00, 0.6599781146531454E+00, 0.8013437877388673E+00, 0.1929493765752847E+00, 0.7655052376601326E+00, 0.6319580998438735E+00, 0.6831519307583876E+00, 0.3834927697748418E+00, 0.7094548193343077E+00, 0.7020510393185050E+00, 0.6520146205437408E+00, 0.6562814689912274E+00, 0.5240471964184201E+00, 0.2362557689024753E+00, 0.4347621644220048E+00, 0.1660979681267404E+00, 0.2422730270980831E+00, 0.7891782502389337E+00, 0.5007812255281535E+00, 0.4444633531981611E+00, 0.3026778693869716E+00, 0.9990331636150707E-02, 0.4097233632314936E+00, 0.9940240633376396E+00, 0.1626467187830966E+00, 0.3461071508544706E+00, 0.6476433090511620E+00, 0.7204399230581238E-01, 0.6885344672911693E+00, 0.1941295874559149E+00, 0.5883115695765473E+00, 0.7359955011333836E+00, 0.4277388472789430E+00, 0.4017294206583021E+00, 0.7838762945853397E+00, 0.9128155969033611E+00, 0.4856080402782125E+00, 0.4763260467232836E+00, 0.5586410048770161E+00, 0.8031586091432461E+00, 0.6068401118620381E-01, 0.9247455618544214E+00, 0.5857147883918616E+00, 0.7467203590494284E+00, 0.1453563221196804E+00, 0.2297646010524917E+00, 0.8950965613541395E+00, 0.3335127494359967E+00, 0.9671078623732762E+00, 0.8023166958816280E+00, 0.1059134943829219E+00, 0.7122485680883537E+00, 0.3211310961264330E+00, 0.1626983074059779E+00, 0.4042697895935750E+00, 0.4799615975845870E+00, 0.3426607739026810E-02, 0.5147088911567483E+00, 0.7533023355653040E+00, 0.9693400349159813E+00, 0.7251923493536844E+00, 0.9132309180417886E-01, 0.4001770994110867E+00},
			n:    21,
			zOut: []float64{0.2574587707602998E+01, 0.2548516143888565E+01, 0.2328646361106399E+01, 0.2152335028165169E+01, 0.1900054508136636E+01, 0.1821843561986577E+01, 0.1641395708303600E+01, 0.1593513640807486E+01, 0.1254245828754644E+01, 0.1184500984874290E+01, 0.9407518766665146E+00, 0.8321813659237318E+00, 0.6961683120499664E+00, 0.5315544763312857E+00, 0.4082285577131941E+00, 0.4022023658419822E+00, 0.2892256147411429E+00, 0.1412172326671640E+00, 0.7919841586115514E-01, 0.2055312514153672E-02, 0.1458583715266580E-03, 0.5995342879679690E-28, 0.2082901809770958E-07, 0.9383395476751434E-08, 0.1641395708303600E+01, 0.1321835131711384E-15, 0.1191741688360734E-04, 0.3695487724817346E-05, 0.1593513640807486E+01, 0.5607617048340495E-15, 0.5232768419257040E-05, 0.4082864434107817E-05, 0.1254245828754644E+01, 0.1850697850686327E-14, 0.2071860298001858E-05, 0.2753629328588578E-05, 0.1184500984874290E+01, 0.3549874073494553E-29, 0.9441520198112501E-05, 0.5871348343515868E-05, 0.9407518766665146E+00, 0.3656881477565973E-23, 0.2332256886230818E+02, 0.2332256886230818E+02, 0.1000000000000000E+03, 0.2984126984126984E+01, 0.3000000000000000E+01, 0.1218265160373847E-03, 0.6961683120499664E+00, 0.3847797347882626E-19, 0.2944801840696127E-03, 0.4565874591840146E-03, 0.5315544763312857E+00, 0.2366582715663035E-29, 0.5951942600827748E-04, 0.7082415940475930E-04, 0.4082285577131941E+00, 0.1744834104604043E-24, 0.4087064324555941E-04, 0.4051735857075661E-04, 0.4022023658419822E+00, 0.5055621275172978E-20, 0.1443306547898801E-05, 0.5013891517830428E-06, 0.2892256147411429E+00, 0.1446770900175336E-26, 0.7439355829951562E-04, 0.1741194257531676E-03, 0.1412172326671640E+00, 0.3448013628188340E-14, 0.2774107535050150E-02, 0.3338675080262731E-02, 0.7919841586115514E-01, 0.2413026984508734E-18, 0.7157138061395222E-03, 0.6803125903293225E-03, 0.2055312514153672E-02, 0.3994185778863994E-23, 0.5722078415717462E-03, 0.5910838594703896E-03, 0.1458583715266580E-03, 0.1183291357831518E-27, 0.8602246198024520E-03, 0.1402897408133881E-02},
			info: 0,
		},
		{
			z:    []float64{0.6756942057876089E+00, 0.6579919202857232E+00, 0.8168130848194743E+00, 0.2483963371195050E+00, 0.2866280219451008E+00, 0.7172808579862799E-03, 0.2411898945446896E+00, 0.5534360794587689E-02, 0.3878248310175912E+00, 0.4256983490856848E+00, 0.7118512164413080E+00, 0.8762976594477806E+00, 0.6850772131038755E+00, 0.4603991232581716E+00, 0.6546622425118448E+00, 0.3103854094723100E+00, 0.7913005955185648E+00, 0.4864788053022765E+00, 0.6214717223070496E+00, 0.7079028836241165E+00, 0.2446362485461124E+00, 0.3062353821816216E+00, 0.8766788389923219E-01, 0.4381267847541642E+00, 0.4459912790878073E-01, 0.9432264671740954E+00, 0.3830574008366420E+00, 0.3381025391982955E+00, 0.6585260539196298E+00, 0.1004337858849739E+00, 0.9320890528134474E+00, 0.3322930037041504E+00, 0.7282334590214989E-02, 0.8367863647557111E+00, 0.9736104021592913E+00, 0.1613168640897539E+00, 0.6409545774446285E+00, 0.8441218931570360E-01, 0.9793937208019250E+00, 0.8303681522642254E+00, 0.1022316826967604E+00, 0.3021608972738009E+00, 0.1238379032543057E+00, 0.3839961366681232E+00, 0.8871553342676223E+00, 0.2909019998361990E+00, 0.7955411369197152E-01, 0.6476593477808288E+00, 0.1739219408315845E+00, 0.6498034164943688E+00, 0.3405210004266120E+00, 0.5459816771585898E+00, 0.2574285667818188E-01, 0.8691155035114945E+00, 0.4320450067500660E+00, 0.6967311316847606E+00, 0.9186339034853966E+00, 0.9491091410944081E+00, 0.2046793386144015E+00, 0.4114169869296881E+00, 0.9170199522695309E+00, 0.1514288291023066E+00, 0.2203325773257230E+00, 0.7891126546748299E+00, 0.4738113210301682E+00, 0.6729225031084131E+00, 0.2382373075875677E+00, 0.4180455054749687E+00, 0.9185761229203592E+00, 0.8030412579733187E+00, 0.4789612026295414E+00, 0.5038982716350845E+00, 0.9959829239394621E+00, 0.3087119711461705E-01, 0.6615773194242809E+00, 0.4461280013353816E+00, 0.1982838436639324E+00, 0.7583555141388076E+00, 0.7946309632008934E+00, 0.6749216068143323E+00, 0.9948312782688508E+00, 0.2854132428873886E+00, 0.8405396060835862E+00, 0.3412432527462372E-01},
			n:    21,
			zOut: []float64{0.2393193272196224E+01, 0.2018136210599260E+01, 0.1995725670346335E+01, 0.1962470040389077E+01, 0.1880885332732760E+01, 0.1640866198530974E+01, 0.1381237309251015E+01, 0.1349378962361067E+01, 0.9454836406584929E+00, 0.7636867441308103E+00, 0.6820433637129892E+00, 0.6091604279437030E+00, 0.5856982990037397E+00, 0.4957297472312800E+00, 0.2553870576469542E+00, 0.2360257010865909E+00, 0.1371029529316844E+00, 0.8102847375475340E-01, 0.5521596879227348E-01, 0.9135542089709688E-02, 0.7426079491367038E-04, 0.1040308236944517E-14, 0.7301351780522185E-10, 0.7774740781328628E-10, 0.1381237309251015E+01, 0.2169367489357782E-29, 0.6020076428884570E-10, 0.5781767346304832E-10, 0.1349378962361067E+01, 0.1885108385497271E-16, 0.6878646083705261E-10, 0.6530919047819172E-10, 0.9454836406584929E+00, 0.5778430642028005E-17, 0.9207835097245334E-10, 0.7637719283517905E-10, 0.7636867441308103E+00, 0.1387778780781446E-16, 0.1259777267883337E-09, 0.1656656563258317E-09, 0.6820433637129892E+00, 0.0000000000000000E+00, 0.1947766517618461E+02, 0.1947766517618461E+02, 0.1070000000000000E+03, 0.3197278911564626E+01, 0.3738317757009346E+01, 0.5169611983503601E-09, 0.5856982990037397E+00, 0.2460562659624748E-15, 0.4628473083370263E-08, 0.9345989594028561E-08, 0.4957297472312800E+00, 0.2615714135774493E-18, 0.2264320058754590E-06, 0.2929148984854635E-06, 0.2553870576469542E+00, 0.1029104326196136E-18, 0.2640422579955958E-07, 0.2108896012270181E-07, 0.2360257010865909E+00, 0.3005450327091600E-17, 0.6822175517251369E-08, 0.8519370047734521E-08, 0.1371029529316844E+00, 0.3395257336071235E-24, 0.2687435952714477E-08, 0.2177398712709520E-08, 0.8102847375475340E-01, 0.0000000000000000E+00, 0.5343374415472690E-09, 0.5384901342094013E-09, 0.5521596879227348E-01, 0.2914620163738177E-24, 0.6457803167637477E-08, 0.2258738450739571E-07, 0.9135542089709688E-02, 0.5326388832052272E-25, 0.9576323906120589E-06, 0.3243967823107194E-05, 0.7426079491367038E-04, 0.3164925372525892E-18, 0.6752170208452239E-05, 0.3302115586194402E-04},
			info: 0,
		},
		{
			z:    []float64{0.1859840927709887E+00, 0.1326482709507555E+00, 0.1830793816152183E+00, 0.4955347339015181E+00, 0.7174832323677909E+00, 0.1239131234434754E-01, 0.8960455455727024E+00, 0.1883195594308053E+00, 0.3701760109540279E+00, 0.5189832590024491E+00, 0.8746518205444311E+00, 0.5127960794741733E+00, 0.2715029219143491E+00, 0.3130984190329523E+00, 0.6427011945330067E+00, 0.2728531634457195E+00, 0.4849915812764480E+00, 0.4474855195890282E-01, 0.1232624016522375E+00, 0.4054507321320544E+00, 0.8168182420396084E+00, 0.5779650563825861E+00, 0.3336575737681243E+00, 0.4793345429794458E+00, 0.3323762775436829E+00, 0.4013874798394268E-01, 0.3250840079396182E+00, 0.3339528328323188E+00, 0.6534745789256358E+00, 0.1592419685588734E+00, 0.4882641395765897E-01, 0.3378473153079410E+00, 0.1498873352536064E+00, 0.9366519936638189E+00, 0.8628314489246045E+00, 0.9035188141748132E+00, 0.3376603415628452E+00, 0.8754867721780122E+00, 0.4314313357227496E+00, 0.6689080802643724E+00, 0.7427192946427073E+00, 0.8408966332922378E+00, 0.3923135283693152E+00, 0.9856346160535667E+00, 0.5253798946684216E+00, 0.4589666030610524E+00, 0.3627813920321372E+00, 0.6650592661455799E+00, 0.5488555419923284E+00, 0.8723519563008553E+00, 0.5759829964509324E+00, 0.8314551283704829E+00, 0.8033910216639006E-01, 0.5533270198466882E+00, 0.4579854660280547E-01, 0.8805162884999327E+00, 0.7798601625088321E+00, 0.3998529604552336E+00, 0.2290634020126769E-01, 0.4087780821380217E+00, 0.8084384129630373E-01, 0.1651045317338867E+00, 0.5138876592921692E+00, 0.1502237554834691E+00, 0.3139451210283801E+00, 0.4951417832103121E+00, 0.4305502836911147E+00, 0.4221379680427187E+00, 0.6629276271381638E+00, 0.6032939209531929E+00, 0.7486429236694788E+00, 0.4603187644190643E+00, 0.9910999698454684E+00, 0.9830579802970648E+00, 0.1327788029232968E+00, 0.7906300391462903E+00, 0.4870686063503010E+00, 0.9107501765395865E-01, 0.5847999740468798E+00, 0.5237055378474927E+00, 0.1949033413503392E+00, 0.6706570830152104E+00, 0.4839742423553839E+00, 0.6933746141255293E+00},
			n:    21,
			zOut: []float64{0.2594702985921864E+01, 0.1963381202215380E+01, 0.1899121104965353E+01, 0.1852857657580134E+01, 0.1352179381057759E+01, 0.1316420368068040E+01, 0.1223907530772361E+01, 0.1104722051320913E+01, 0.1042814535040686E+01, 0.9392359492223399E+00, 0.5879944547218380E+00, 0.5343064059199153E+00, 0.3616892068472140E+00, 0.3474187565935542E+00, 0.3132932870269020E+00, 0.2394766491234008E+00, 0.1924620369224269E+00, 0.5796610832720670E-01, 0.4900919199482741E-01, 0.2087594658763894E-01, 0.6804292520862544E-03, 0.2186403793390500E-19, 0.2668268315891067E-16, 0.1216320938093713E-16, 0.1223907530772361E+01, 0.3077822220048293E-18, 0.9690290545884895E-18, 0.1677534925257981E-18, 0.1104722051320913E+01, 0.3294338264035954E-18, 0.1270509199570000E-13, 0.4678488073918506E-13, 0.1042814535040686E+01, 0.2740439675865377E-23, 0.1047809983111396E-11, 0.5781643460797313E-12, 0.9392359492223399E+00, 0.3379215452805658E-16, 0.8350045830190690E-10, 0.5730922211948388E-10, 0.5879944547218380E+00, 0.6938893903907228E-17, 0.1799451523948185E+02, 0.1799451523948184E+02, 0.1150000000000000E+03, 0.3578231292517007E+01, 0.6086956521739131E+01, 0.8630088713732289E-10, 0.3616892068472140E+00, 0.1664229231319872E-20, 0.9644621511794884E-10, 0.9354199679969641E-10, 0.3474187565935542E+00, 0.0000000000000000E+00, 0.1015403419286529E-09, 0.1046840761330470E-09, 0.3132932870269020E+00, 0.6366107505133565E-27, 0.3071873418074744E-11, 0.5342885575500614E-12, 0.2394766491234008E+00, 0.2599072586631376E-18, 0.1837414559126179E-09, 0.2303923049314304E-09, 0.1924620369224269E+00, 0.3400814535759169E-17, 0.9192916981159684E-08, 0.4803632388224207E-08, 0.5796610832720670E-01, 0.0000000000000000E+00, 0.1565526746702558E-05, 0.8408463868497472E-06, 0.4900919199482741E-01, 0.7888609052210118E-30, 0.1903794116022181E-04, 0.1014775117962413E-04, 0.2087594658763894E-01, 0.9734740470109229E-22, 0.4676468030348868E-06, 0.4248047233272650E-05, 0.6804292520862544E-03, 0.3161038140634776E-18, 0.1818811163938070E-02, 0.3653457173655013E-02},
			info: 0,
		},
		{
			z:    []float64{0.2029039404458555E-01, 0.4529725864625287E+00, 0.6412066909421453E+00, 0.6139288440336396E+00, 0.3661028502807454E+00, 0.9178254749803582E+00, 0.4307551173824371E+00, 0.4985808870906128E+00, 0.8171615510934167E+00, 0.8279058065788196E+00, 0.3035261047197422E+00, 0.7877147223127390E+00, 0.7688130520483442E+00, 0.4834409462671818E-01, 0.3121145202501183E+00, 0.5269232357652766E+00, 0.7600633371694686E+00, 0.5251092877920901E+00, 0.8263235554592012E+00, 0.6894073332954654E+00, 0.3521429062439027E+00, 0.6447189811681343E+00, 0.1105592548668879E+00, 0.1049454508922576E+00, 0.7381247076050562E+00, 0.7320584094686099E+00, 0.7003114013559573E+00, 0.9638534557041758E+00, 0.5727311775052889E+00, 0.3930768970870516E+00, 0.7640649329763241E+00, 0.3221407324893386E+00, 0.9725443946255905E+00, 0.6922592425787558E+00, 0.3449642467056757E-01, 0.1384627086161467E+00, 0.4398391723165511E+00, 0.8406456486574913E+00, 0.4126144976348992E+00, 0.5351779876797247E-01, 0.6669497026260232E+00, 0.5746288223886410E+00, 0.4589594741804619E+00, 0.7484833637036891E+00, 0.3274476209575072E-01, 0.7067960763848782E+00, 0.8759136475280368E+00, 0.3284828537280239E+00, 0.2536109230678856E+00, 0.8799264273691800E+00, 0.8159607815416694E+00, 0.5729670098854700E-01, 0.8884200881604851E+00, 0.5249116128877267E+00, 0.6498061599007876E-01, 0.6075292761444269E+00, 0.2428201413403417E+00, 0.2668460043818345E+00, 0.1429723702275580E+00, 0.6942338078813951E+00, 0.9271225003121289E+00, 0.8931957344289477E+00, 0.6420336733834545E+00, 0.3786225606645894E+00, 0.9421368123010410E-01, 0.9476439119845459E+00, 0.6497389318978118E+00, 0.3841806572586668E+00, 0.1329188317035728E+00, 0.6545507077812880E+00, 0.4430247695845350E+00, 0.1753161904868739E+00, 0.2413409779703910E+00, 0.1476409499735053E+00, 0.1179852848482287E+00, 0.3704631210033814E+00, 0.1967445635592234E+00, 0.2100970549835486E+00, 0.5744473538919733E+00, 0.1093206314301038E+00, 0.5132572818416121E+00, 0.4964557538937117E+00, 0.7518888363275744E-02, 0.9720615373003737E+00},
			n:    21,
			zOut: []float64{0.2563191374817756E+01, 0.2359364796169898E+01, 0.2296884847589588E+01, 0.2056943321577226E+01, 0.1949112826206492E+01, 0.1629835138023691E+01, 0.1562766390641567E+01, 0.1422208590143927E+01, 0.1368481726990307E+01, 0.1150114414099739E+01, 0.8482773777570239E+00, 0.7665061168435710E+00, 0.7135198036652551E+00, 0.3486547062556205E+00, 0.2456691453437445E+00, 0.2203009969871996E+00, 0.1404417091680581E+00, 0.1160909181211330E+00, 0.2112854362928769E-01, 0.5069559443178203E-02, 0.5650407111696261E-03, 0.3717112585401408E-26, 0.2316306425767847E-08, 0.1632477182259275E-08, 0.1562766390641567E+01, 0.6446192664231602E-24, 0.9634960729667980E-08, 0.6524712961691138E-08, 0.1422208590143927E+01, 0.1932709217791479E-28, 0.4932058767625490E-07, 0.7860293908453283E-07, 0.1368481726990307E+01, 0.1238511621196989E-27, 0.2778987342481399E-06, 0.5558918854266694E-06, 0.1150114414099739E+01, 0.2156683454185074E-19, 0.1662595987219070E-04, 0.2580299679298412E-04, 0.8482773777570239E+00, 0.7914569980025776E-17, 0.2178512734418544E+02, 0.2178512734418543E+02, 0.1000000000000000E+03, 0.2934240362811791E+01, 0.1000000000000000E+01, 0.1715129124192563E-05, 0.7135198036652551E+00, 0.1613909508212798E-19, 0.2836076293001813E-04, 0.1595224464977716E-04, 0.3486547062556205E+00, 0.3352658847189300E-29, 0.8812160938383347E-04, 0.1178320764067675E-03, 0.2456691453437445E+00, 0.0000000000000000E+00, 0.5251568647637462E-04, 0.4480459044047029E-04, 0.2203009969871996E+00, 0.6471183777709004E-25, 0.1114279427921990E-09, 0.2270576453660371E-10, 0.1404417091680581E+00, 0.2855676476900063E-27, 0.2886261726979772E-05, 0.5470704056451658E-06, 0.1160909181211330E+00, 0.3526208246337923E-27, 0.3588687949078309E-02, 0.5383294574404876E-02, 0.2112854362928769E-01, 0.2896602354076833E-17, 0.1732996915625309E-02, 0.1499310228751921E-02, 0.5069559443178203E-02, 0.1003594455767978E-17, 0.5081816639203317E-03, 0.1080637708244138E-02, 0.5650407111696261E-03, 0.0000000000000000E+00, 0.5066428376081889E-02, 0.3247326449492487E-02},
			info: 0,
		},
		{
			z:    []float64{0.7177806115181969E+00, 0.9619042524298227E+00, 0.9756290650460009E-01, 0.2695657197860104E+00, 0.5092382615580680E+00, 0.2216490915715645E+00, 0.7713962041593733E+00, 0.9019626308054164E+00, 0.1009043823109016E+00, 0.7164892891763703E+00, 0.6972336574704985E+00, 0.5604456603447616E-01, 0.4658944020177349E+00, 0.4677429402719713E+00, 0.9395972368396054E+00, 0.6567060928508761E-01, 0.6711470087762763E+00, 0.3572615865847095E-01, 0.7841795004355456E-01, 0.2100489732227784E+00, 0.1528790147771258E+00, 0.2944728557463859E+00, 0.6929796239262105E+00, 0.5593689694083593E+00, 0.5600220347177359E+00, 0.6885800745012647E+00, 0.1441376161827218E+00, 0.2468678391528564E+00, 0.2219056216366488E+00, 0.7828887069679423E+00, 0.1781536622262658E-01, 0.9712383516530864E+00, 0.1031071887927941E+00, 0.5428204596104211E+00, 0.9920375605622551E+00, 0.2231614148449633E+00, 0.1774197365731667E+00, 0.1667803739895276E+00, 0.2293956181863850E+00, 0.8453959149417756E+00, 0.2211895679989639E+00, 0.5610288802043042E+00, 0.3749535737701304E-01, 0.7418144847444434E+00, 0.2593245755300763E+00, 0.5358881543370908E-01, 0.9118629582226685E+00, 0.6483736004795430E+00, 0.7595140552166778E+00, 0.2981894989055883E+00, 0.6091779707233183E+00, 0.7798268953767704E+00, 0.1682114110436058E+00, 0.3801835867597201E+00, 0.6380508025759659E+00, 0.7892946000460455E-01, 0.7607559424299619E+00, 0.3271484239841950E+00, 0.7700240256278714E+00, 0.5894107219393652E+00, 0.6207117138108632E+00, 0.4725019923733027E+00, 0.3616574024313524E+00, 0.6744122205664662E+00, 0.5489632027065386E+00, 0.1532930834122788E+00, 0.1692473010839316E+00, 0.7515070367293223E+00, 0.1186783750638719E+00, 0.1028479850139022E+00, 0.3468642488228025E+00, 0.5278752643808988E+00, 0.3849250707234438E+00, 0.7889573536656195E+00, 0.4738712276132427E+00, 0.7058571493185843E+00, 0.8948240863202030E+00, 0.3494029097065446E+00, 0.4916843795342892E+00, 0.1993321046695898E+00, 0.3650043431665774E+00, 0.2403686388300026E+00, 0.5474155466298688E+00, 0.6157434374908201E+00},
			n:    21,
			zOut: []float64{0.1955717973339200E+01, 0.1915826728710446E+01, 0.1780143898982495E+01, 0.1751081083154649E+01, 0.1679171531306522E+01, 0.1398798787154148E+01, 0.1289579288306139E+01, 0.1171599632002814E+01, 0.9950381700653156E+00, 0.9008548767119643E+00, 0.7944341357698450E+00, 0.7347990252582277E+00, 0.4053391427425183E+00, 0.3415476013970046E+00, 0.2769292856200630E+00, 0.2086988709391624E+00, 0.7594980741000955E-01, 0.5721582845751132E-01, 0.5152297043277422E-01, 0.6186350115045150E-02, 0.5774958135565486E-05, 0.6722096271457334E-14, 0.6635533119216432E-06, 0.1048982691930868E-05, 0.1289579288306139E+01, 0.0000000000000000E+00, 0.9172850657907766E-08, 0.6527900227304331E-08, 0.1171599632002814E+01, 0.0000000000000000E+00, 0.3386861611239913E-06, 0.1115072165646295E-06, 0.9950381700653156E+00, 0.1044578056257455E-24, 0.5793689136913799E-11, 0.1625904890507595E-10, 0.9008548767119643E+00, 0.8883435102679349E-21, 0.3220671712964998E-09, 0.8453946299986560E-09, 0.7944341357698450E+00, 0.1653452457343241E-26, 0.1779044076283400E+02, 0.1779044076283399E+02, 0.1120000000000000E+03, 0.3267573696145125E+01, 0.3571428571428572E+01, 0.1211740269624600E-06, 0.4053391427425183E+00, 0.5298753945086544E-14, 0.1515410194583574E-06, 0.2069357347527298E-06, 0.3415476013970046E+00, 0.0000000000000000E+00, 0.2579333883676927E-05, 0.8435987645777294E-05, 0.2769292856200630E+00, 0.2465190328815662E-30, 0.1658852164453815E-08, 0.3207072615633299E-09, 0.2086988709391624E+00, 0.2017246551328580E-19, 0.6091457019925236E-05, 0.6297972233022177E-05, 0.7594980741000955E-01, 0.2553131970482608E-15, 0.2736366650719638E-04, 0.2348362771705736E-04, 0.5721582845751132E-01, 0.1141311335899276E-23, 0.6030351005218485E-04, 0.3385365250818500E-04, 0.5152297043277422E-01, 0.0000000000000000E+00, 0.4442856906399686E-04, 0.3980460155412817E-04, 0.6186350115045150E-02, 0.2711709361697228E-30, 0.1322896557202734E-03, 0.3209571887096705E-03, 0.5774958135565486E-05, 0.0000000000000000E+00, 0.9248295446812529E-03, 0.1317994001905306E-02},
			info: 0,
		},
		{
			z:    []float64{0.7669589114699304E-01, 0.2959120838012549E+00, 0.8617075527885690E+00, 0.3774472194278293E+00, 0.7430204868341380E+00, 0.3337487204025202E-01, 0.2115072947611989E+00, 0.8195300149220009E+00, 0.4610153087687133E+00, 0.1049580467270205E+00, 0.6553761031646719E+00, 0.2044425657813468E+00, 0.1419536829746254E+00, 0.8338399989758465E+00, 0.9917105482281005E+00, 0.3698082853107363E+00, 0.5656210986926330E+00, 0.5849265049217818E+00, 0.9018145207330325E+00, 0.7089264704350354E+00, 0.2562038341655152E+00, 0.1524378342317656E-01, 0.9206210351037002E+00, 0.1877259832276945E+00, 0.3637451600309541E+00, 0.9979742594017312E+00, 0.8919698496914060E+00, 0.3716780472994426E+00, 0.3142548907151147E+00, 0.4170439734847903E+00, 0.3197415298896636E+00, 0.7285365914169133E+00, 0.5784968714882697E+00, 0.6831919512327895E+00, 0.6057174197537164E+00, 0.6413693987683566E-01, 0.8827799219498907E+00, 0.4192891003797022E+00, 0.6040167945472836E+00, 0.6469271762278970E+00, 0.7359461974470041E+00, 0.5539488596393002E+00, 0.4023966166720030E+00, 0.9915288950117843E+00, 0.3043860170301459E+00, 0.4917889743094436E+00, 0.1179472550216760E+00, 0.9125202427370891E+00, 0.8786441005384636E+00, 0.4634730997209351E+00, 0.1080276946115265E+00, 0.6187727001119313E+00, 0.6709766557170562E+00, 0.3818949179452977E+00, 0.8906457783485427E+00, 0.8010289089804480E+00, 0.6931910498827129E+00, 0.8914072225833563E+00, 0.3822463769530011E+00, 0.5898102885291842E+00, 0.1092049166271940E+00, 0.8496011938807149E+00, 0.2304394290731624E+00, 0.5949006075515944E+00, 0.3290638194228065E+00, 0.1971699984758039E+00, 0.7443144303888384E+00, 0.4653555333092598E+00, 0.8207632269304853E+00, 0.4152482306441556E+00, 0.1689785233215235E+00, 0.1721084345877374E+00, 0.3931596107353640E+00, 0.5403345228237123E+00, 0.6387555392172577E+00, 0.1804790096977364E+00, 0.5783025205396422E+00, 0.7395837791384520E+00, 0.5701597209798811E+00, 0.1952734055752668E+00, 0.9661114908130567E+00, 0.4893113195434146E+00, 0.6562980654604210E+00, 0.1879394621701417E+00},
			n:    21,
			zOut: []float64{0.2390558120678448E+01, 0.2326196956724044E+01, 0.2071676597874671E+01, 0.1995424877356871E+01, 0.1973795189776371E+01, 0.1728273264358535E+01, 0.1492322130169846E+01, 0.1216694895544233E+01, 0.1149617898796123E+01, 0.1007011094632661E+01, 0.9207056739913982E+00, 0.7404938152461292E+00, 0.6082352700464410E+00, 0.5965144149455339E+00, 0.2276577044879657E+00, 0.1776272681811473E+00, 0.1662286875439616E+00, 0.8310804130734020E-01, 0.4604735683437314E-01, 0.1806926893972028E-01, 0.1257133375345320E-01, 0.3600386373101646E-15, 0.6280777483216298E-03, 0.1060573000233681E-02, 0.1492322130169846E+01, 0.1292469707114105E-25, 0.5799349861429450E-06, 0.5584709508928944E-06, 0.1216694895544233E+01, 0.2047280287874868E-18, 0.1299955932331242E-09, 0.8534229075832313E-10, 0.1149617898796123E+01, 0.5088714046955349E-16, 0.6001732451841635E-09, 0.4081989685050919E-09, 0.1007011094632661E+01, 0.5578719308559292E-21, 0.1498995168909837E-07, 0.5625367692005608E-08, 0.9207056739913982E+00, 0.2728742825015995E-21, 0.2094882986118927E+02, 0.2094882986118927E+02, 0.9800000000000000E+02, 0.3004535147392290E+01, 0.4081632653061225E+01, 0.6223967900246754E-04, 0.6082352700464410E+00, 0.4038967834731580E-26, 0.1122996207538654E-05, 0.1754126012223728E-05, 0.5965144149455339E+00, 0.4386207914826929E-19, 0.1334483593164094E-04, 0.6982705884252015E-05, 0.2276577044879657E+00, 0.3148538924591216E-18, 0.9085150621800872E-04, 0.4628114404518947E-04, 0.1776272681811473E+00, 0.1149284743081177E-19, 0.3934684725502468E-05, 0.5044161242313764E-05, 0.1662286875439616E+00, 0.3944304526105059E-30, 0.1149340461723492E-04, 0.8510588625472410E-05, 0.8310804130734020E-01, 0.1540743955509789E-32, 0.1285467894756909E-02, 0.9239857033877289E-03, 0.4604735683437314E-01, 0.2169367489357782E-29, 0.2808730938050547E-03, 0.3455059327639833E-03, 0.1806926893972028E-01, 0.4614836295542919E-28, 0.7683442768713957E-06, 0.1846813906304985E-05, 0.1257133375345320E-01, 0.2538554393001216E-26, 0.4028502005942143E-04, 0.8466712427867894E-04},
			info: 0,
		},
		{
			z:    []float64{0.5568197491282034E+00, 0.7338530212513171E+00, 0.1711270247791036E+00, 0.6696990680288049E+00, 0.1107644593582661E+00, 0.1487844153251054E+00, 0.6221478836712087E+00, 0.3739707210550620E+00, 0.6142936216832375E+00, 0.4504419047617665E+00, 0.1390832371836795E+00, 0.9602056283222130E+00, 0.4128383897877478E+00, 0.6202590221465013E+00, 0.5716294881431405E+00, 0.7009876531280159E+00, 0.8556346552408018E+00, 0.4300043005510307E+00, 0.5625488786064613E+00, 0.5236571943623558E+00, 0.2035297706440273E+00, 0.5324677179555473E+00, 0.9854023908952125E+00, 0.8088580870995794E+00, 0.7902887218423563E+00, 0.4196418643524230E+00, 0.5294914665193529E+00, 0.5947417442754066E-01, 0.2753919335549286E+00, 0.8807083336864044E+00, 0.6016279529290567E+00, 0.4144655693858035E-01, 0.4808953029705748E+00, 0.8506686680287934E-01, 0.2145404015834380E+00, 0.3020133878086849E+00, 0.8967140938263458E+00, 0.5344538485865815E+00, 0.2536120629120923E+00, 0.2552452828610304E+00, 0.9211096671641958E+00, 0.9028760680202415E+00, 0.3005599522329846E+00, 0.1197919551028395E+00, 0.3772579707039786E+00, 0.4157715274081910E+00, 0.2203607291065889E+00, 0.1084188329562713E+00, 0.5463193166123409E+00, 0.8107359194268960E+00, 0.6676190072779817E+00, 0.6711454730587799E+00, 0.6360708014875704E-01, 0.8242026716736568E+00, 0.3687006444230088E+00, 0.2216898043026083E-01, 0.4978612409817640E+00, 0.3144452507777135E+00, 0.3412289603988730E+00, 0.3266626310182044E+00, 0.4581662376370765E+00, 0.6985218227047190E+00, 0.6594479039269319E-01, 0.3938130402504401E+00, 0.9907371819490932E+00, 0.2478499678343852E+00, 0.3565435772734814E+00, 0.8420455744018336E+00, 0.2692247454903065E+00, 0.7327570909626056E+00, 0.1752776205164243E+00, 0.3569190164542581E+00, 0.2122621782757903E+00, 0.2056257129895300E+00, 0.6641092684756426E+00, 0.5386917539530447E+00, 0.8420713652275945E+00, 0.3587226239611645E+00, 0.9561844063661247E+00, 0.9126272291387975E+00, 0.5768542567999925E+00, 0.1972168939311342E+00, 0.5763304496236371E+00, 0.9478337554439876E+00},
			n:    21,
			zOut: []float64{0.2593664459139222E+01, 0.2338172956520739E+01, 0.1783072679058981E+01, 0.1768637196043293E+01, 0.1688808722758796E+01, 0.1535694950136988E+01, 0.1488691522387158E+01, 0.1290354539321546E+01, 0.1187570436840380E+01, 0.1077679082742636E+01, 0.7925423484101771E+00, 0.6549022645335129E+00, 0.6315653598883190E+00, 0.5184441284206909E+00, 0.3791491409939438E+00, 0.3105722740860607E+00, 0.1128189367670211E+00, 0.7036440937731858E-01, 0.4999153992819697E-01, 0.2730914559941030E-01, 0.7241089204639656E-03, 0.7344040350490801E-18, 0.1828200693350630E-07, 0.2226187377323797E-07, 0.1488691522387158E+01, 0.4586948101010510E-17, 0.3930142187345605E-07, 0.5206893736347286E-07, 0.1290354539321546E+01, 0.6658190007940994E-20, 0.1499933495859733E-06, 0.7910043275590841E-07, 0.1187570436840380E+01, 0.2370536763713858E-16, 0.1192422432592963E-05, 0.6065280015592916E-06, 0.1077679082742636E+01, 0.6803770690793804E-23, 0.5360141282897325E-04, 0.3254990522417721E-04, 0.7925423484101771E+00, 0.8404284270509473E-23, 0.2030073020187486E+02, 0.2030073020187486E+02, 0.9700000000000000E+02, 0.2975056689342404E+01, 0.1030927835051546E+01, 0.1762395614510388E-04, 0.6315653598883190E+00, 0.2034005362457094E-16, 0.3178492515415974E-08, 0.1141868824523976E-08, 0.5184441284206909E+00, 0.1631704231140345E-20, 0.1154759308961600E-04, 0.3555504775624820E-05, 0.3791491409939438E+00, 0.9251916923707874E-19, 0.1302644751785016E-04, 0.1874642271188771E-04, 0.3105722740860607E+00, 0.1235671721938193E-25, 0.8697586673438179E-05, 0.7595546859022038E-05, 0.1128189367670211E+00, 0.1175733067685735E-18, 0.2174688695641498E-03, 0.1411064498403114E-03, 0.7036440937731858E-01, 0.5916456789157589E-30, 0.4997004849851516E-03, 0.7625846015345374E-03, 0.4999153992819697E-01, 0.3710108611279200E-20, 0.2898431160817185E-04, 0.2329024376647231E-03, 0.2730914559941030E-01, 0.1593079290931123E-17, 0.5731904499848677E-04, 0.7071520517918782E-04, 0.7241089204639656E-03, 0.1043820749788443E-25, 0.2096656950463827E-03, 0.4915700720935459E-03},
			info: 0,
		},
		{
			z:    []float64{0.2129092067945103E+00, 0.6174796583507793E+00, 0.2209275310088966E+00, 0.1770681976597738E+00, 0.8468210663920229E+00, 0.7900600194799612E+00, 0.8319584509547915E+00, 0.8077183726155964E+00, 0.7964773509287093E+00, 0.2479492923952727E+00, 0.5169395370002006E-01, 0.6312063121285433E+00, 0.5688486960732374E+00, 0.3049379579822397E+00, 0.3779887662042721E+00, 0.4551165943302187E+00, 0.8807128836121972E+00, 0.1333615670826408E+00, 0.6901278826487529E+00, 0.8360350720177558E+00, 0.4190492004641316E+00, 0.9067077556412378E+00, 0.9762584494767094E+00, 0.9959763915912888E+00, 0.8503724779590973E+00, 0.6671116075685212E+00, 0.8841807167789617E+00, 0.6917331852931945E+00, 0.3414398718690443E+00, 0.2211364241075178E+00, 0.8057250135329493E+00, 0.4405475768508057E+00, 0.9321382353755537E+00, 0.6976358515081519E+00, 0.3254605598548291E+00, 0.1144237265035558E+00, 0.8547099978727912E-01, 0.3131314226614652E+00, 0.6940029038044814E+00, 0.2327469211945017E+00, 0.3523896044562020E+00, 0.6443116162172926E+00, 0.5382708884344491E+00, 0.9378966459649967E+00, 0.7274262519928346E+00, 0.4882092650189528E+00, 0.4497457043638882E+00, 0.8803511251742836E+00, 0.1077837527475992E+00, 0.4760275900323754E+00, 0.9725944612716640E+00, 0.4013370227296387E+00, 0.8188706727711300E+00, 0.7549128555022213E+00, 0.3045373312490215E+00, 0.9776368127163177E+00, 0.5516094182757485E+00, 0.8449472155130459E+00, 0.9309695118211208E+00, 0.6985346927019656E+00, 0.3270836933831586E+00, 0.2305805980953167E+00, 0.9135598850313159E+00, 0.2886818807914165E+00, 0.6607756604738726E+00, 0.7992086772120486E+00, 0.9191714969823802E+00, 0.9956222713515444E+00, 0.1646632133753003E+00, 0.8114783440114911E+00, 0.9650128806412454E+00, 0.1382218488636167E-01, 0.8464890579713791E+00, 0.2866238720970538E+00, 0.4485743907736028E-01, 0.5384705384174762E+00, 0.2996807710312754E-01, 0.7020716858392958E+00, 0.6280049430389886E+00, 0.2750308929159380E+00, 0.1648004007672321E-01, 0.8720570971091830E+00, 0.9962373992422624E+00, 0.8022759164194093E-01},
			n:    21,
			zOut: []float64{0.3179557858974794E+01, 0.2694788878259719E+01, 0.2281326093492730E+01, 0.2196001939032204E+01, 0.1828355236773963E+01, 0.1586247682528960E+01, 0.1454772213395289E+01, 0.1286809233158695E+01, 0.1269332746846940E+01, 0.1099355321362049E+01, 0.9971216943768625E+00, 0.8310914612054856E+00, 0.4047131081238095E+00, 0.3869245862172090E+00, 0.3328473924133467E+00, 0.2908704078018756E+00, 0.1880200745280021E+00, 0.5784266149949692E-01, 0.4139216712761701E-01, 0.1651732519409763E-01, 0.3149646326524349E-02, 0.0000000000000000E+00, 0.5227593374295754E-11, 0.1202138413496143E-10, 0.1454772213395289E+01, 0.1393199790273629E-16, 0.1121878363020271E-09, 0.5353121918082171E-10, 0.1286809233158695E+01, 0.8659547037664508E-24, 0.1645814283637519E-08, 0.8601359367531547E-09, 0.1269332746846940E+01, 0.2823844361652584E-22, 0.5929700276768749E-08, 0.1126902625809321E-07, 0.1099355321362049E+01, 0.2358945595467410E-22, 0.2764360194400850E-05, 0.1553001254800959E-05, 0.9971216943768625E+00, 0.4450651748190101E-21, 0.2242703772863968E+02, 0.2242703772863967E+02, 0.1000000000000000E+03, 0.3015873015873016E+01, 0.3000000000000000E+01, 0.2632919892231646E-04, 0.4047131081238095E+00, 0.5820888909768840E-10, 0.2209233784503674E-04, 0.4136287504408153E-04, 0.3869245862172090E+00, 0.6462348535570008E-26, 0.3088379129783557E-03, 0.1528857353804532E-03, 0.3328473924133467E+00, 0.5549599819233453E-15, 0.8893453420684614E-08, 0.8741044673362129E-07, 0.2908704078018756E+00, 0.2289589891313465E-25, 0.5532514209995071E-04, 0.2183152999074702E-03, 0.1880200745280021E+00, 0.6113672015462841E-28, 0.9851023787951559E-03, 0.2769770083415459E-02, 0.5784266149949692E-01, 0.3571962178840741E-24, 0.3360295855593957E-02, 0.3874576703777444E-02, 0.4139216712761701E-01, 0.1555002616371658E-25, 0.3047908004675713E-02, 0.4283880536865950E-02, 0.1651732519409763E-01, 0.1498533807074276E-21, 0.9218134151523294E-06, 0.6993690185531638E-05, 0.3149646326524349E-02, 0.5476587948406352E-25, 0.9434027266146764E-03, 0.1032678112098284E-02},
			info: 0,
		},
		{
			z:    []float64{0.5325973817915391E+00, 0.8075384328404875E+00, 0.2063920576624930E+00, 0.4895050697482526E+00, 0.3224859392283662E+00, 0.4597919629142447E+00, 0.5480240110168966E-01, 0.5228709929272914E+00, 0.9956836568223112E+00, 0.3978480622803672E+00, 0.9124816118920143E-01, 0.3247027355683270E+00, 0.9224658967837341E+00, 0.9984579111484309E+00, 0.8533112174943273E+00, 0.8907006920892125E+00, 0.3659539504394835E+00, 0.5561036191465061E+00, 0.7570794287369270E+00, 0.7781602931894585E+00, 0.8583979385494303E+00, 0.5883490719908564E+00, 0.2958788793535505E+00, 0.8441503484168636E+00, 0.7171074610327245E+00, 0.6324186559935309E+00, 0.6889002812298057E+00, 0.9679449699589238E-01, 0.8071604187540067E+00, 0.5036464295967858E+00, 0.3075050973746345E+00, 0.4354181299061508E+00, 0.5397482510302705E+00, 0.3266316370831254E+00, 0.7127302469392831E+00, 0.7822329043112050E+00, 0.1787867076882754E+00, 0.5059399114486356E+00, 0.1635357104384569E+00, 0.7923616792404551E+00, 0.8527619409672080E+00, 0.6624868870738707E+00, 0.6395827198572863E+00, 0.5045303635733072E+00, 0.5207053335438128E-01, 0.3872927271413512E+00, 0.8590236531191382E+00, 0.8644737695336893E+00, 0.5447620155822054E+00, 0.3420775682173337E+00, 0.1787889858336624E+00, 0.2962040723342502E+00, 0.1197091589896203E+00, 0.2582729465177200E+00, 0.8564224654785235E+00, 0.1003255861450059E+00, 0.5792751794645187E+00, 0.3553877787422808E+00, 0.3515051629979192E+00, 0.7026347035497048E+00, 0.7851727410421285E+00, 0.9451470000488936E+00, 0.1482591966327134E+00, 0.8035088707163867E+00, 0.8018983228501475E-02, 0.5992223740738118E+00, 0.3369698862429809E+00, 0.9971168486495201E+00, 0.7449267587097297E+00, 0.5929357935363798E+00, 0.8575900212486389E+00, 0.8115297023854162E+00, 0.3380359989630263E+00, 0.8328174310878843E+00, 0.8255713836908419E+00, 0.7399149789721748E+00, 0.1725333812438860E+00, 0.6828663265380140E+00, 0.6234367510076171E+00, 0.5628070366267671E+00, 0.8501068267959022E-01, 0.9315797050119701E+00, 0.1842029704669090E+00, 0.9504814303233714E+00},
			n:    21,
			zOut: []float64{0.2923425454171414E+01, 0.2535311958925162E+01, 0.2192080092128743E+01, 0.1925146455902099E+01, 0.1841700496719334E+01, 0.1753280727277585E+01, 0.1582442080550360E+01, 0.1581069533140902E+01, 0.1420113611638973E+01, 0.1070153147168429E+01, 0.9980373856163754E+00, 0.9531244298503998E+00, 0.7077665812605668E+00, 0.5659990522281487E+00, 0.3146390113583965E+00, 0.2516605740660456E+00, 0.1663945631562664E+00, 0.1260010860507715E+00, 0.4255237190875141E-01, 0.7100815132372204E-02, 0.1566331927046113E-03, 0.4122332130840440E-22, 0.4572750287882177E-06, 0.8150029788456509E-06, 0.1582442080550360E+01, 0.6246970091107327E-19, 0.1184058245159526E-05, 0.5957129004529113E-06, 0.1581069533140902E+01, 0.2712893915232508E-21, 0.4684400026881307E-05, 0.9374804625040887E-05, 0.1420113611638973E+01, 0.4814409269321697E-22, 0.2425457953812455E-04, 0.3059673263223507E-04, 0.1070153147168429E+01, 0.4457872866361589E-11, 0.2849702090992769E-05, 0.4079444611357521E-05, 0.9980373856163754E+00, 0.2148022614677702E-19, 0.2295815606144379E+02, 0.2295815606144380E+02, 0.9700000000000000E+02, 0.2981859410430839E+01, 0.2061855670103093E+01, 0.4336961109868880E-06, 0.7077665812605668E+00, 0.3355216099479021E-19, 0.2178215603335118E-03, 0.1596478432727446E-03, 0.5659990522281487E+00, 0.3777948407501058E-19, 0.1732350582817826E-03, 0.1687341431970761E-03, 0.3146390113583965E+00, 0.4215672677501087E-24, 0.8708765796225133E-03, 0.5753505295482449E-03, 0.2516605740660456E+00, 0.8872339250080977E-20, 0.2809442395187102E-02, 0.2330568324677941E-02, 0.1663945631562664E+00, 0.1848892746611746E-31, 0.7424955400801435E-06, 0.1353946060853610E-06, 0.1260010860507715E+00, 0.7099748146989106E-27, 0.5320431162642944E-03, 0.1085271179151602E-03, 0.4255237190875141E-01, 0.2808116620904140E-21, 0.1709028753951538E-02, 0.8528114742759547E-03, 0.7100815132372204E-02, 0.4930380657631324E-31, 0.2904056487242609E-02, 0.3026916823887611E-02, 0.1566331927046113E-03, 0.0000000000000000E+00, 0.2549645590833870E-02, 0.2592045976400073E-02},
			info: 0,
		},
		{
			z:    []float64{0.6141085085904113E+00, 0.3421028526671319E+00, 0.2523050636037050E-01, 0.6894939888024587E+00, 0.1106696902059259E+00, 0.3232796216228777E+00, 0.5026462674631121E+00, 0.4061906185674803E+00, 0.2489870821257593E+00, 0.5655528330838598E+00, 0.8215445521378404E+00, 0.6847122058387792E+00, 0.1058838690465073E+00, 0.3150742731806708E+00, 0.6299058795873502E-01, 0.2792122534089967E+00, 0.6156682980944264E+00, 0.6784924553414444E+00, 0.5548029179057333E+00, 0.9875334352131669E+00, 0.7114388615166174E+00, 0.4680838251435693E+00, 0.9939378516921551E+00, 0.9317334719497590E+00, 0.7166727913665860E+00, 0.9133735504519750E+00, 0.6652039093977162E+00, 0.8687642647047388E+00, 0.5616945762819415E+00, 0.3985759546461669E+00, 0.7431799484571856E+00, 0.7126935860242257E+00, 0.1006984098734091E+00, 0.3341210154632034E+00, 0.8752290131259591E+00, 0.9044047683690323E+00, 0.2748574881470656E+00, 0.9153414825361147E+00, 0.1879700367171477E-01, 0.4139829057070108E+00, 0.6163962814716032E+00, 0.4758310984958897E+00, 0.7283265374189019E-01, 0.4293121596195046E+00, 0.1627494487778676E+00, 0.6477819718948226E+00, 0.1769525517070647E-01, 0.8315284238294101E+00, 0.3951374239126071E-01, 0.8607842403091001E+00, 0.4542147501441236E+00, 0.7533183963616871E+00, 0.3860688952521878E+00, 0.7194066659377851E+00, 0.5625319388225433E+00, 0.2666855367146020E+00, 0.1952117588514313E+00, 0.4351982558470283E+00, 0.5735131371596182E+00, 0.6135067797741378E-01, 0.6627925797553079E+00, 0.5958408703801275E+00, 0.1272760770553625E+00, 0.9764806486481753E+00, 0.3738230266231765E+00, 0.8498786384730982E+00, 0.9509621583287799E+00, 0.5442964253426497E+00, 0.1274761736897801E+00, 0.8749459887021185E+00, 0.5319902581507450E+00, 0.4050330112540623E-01, 0.4679445276719574E-01, 0.6398654173496683E+00, 0.1619398995392787E+00, 0.2834692561637309E-01, 0.7369067773846130E+00, 0.2655208002136908E+00, 0.1902643003388997E+00, 0.7054378485643029E+00, 0.8869999325574263E+00, 0.1700725746174213E+00, 0.7638558771240449E+00, 0.3965145662986846E+00},
			n:    21,
			zOut: []float64{0.3026266675500349E+01, 0.2485098580281992E+01, 0.2208517136068298E+01, 0.2089817225148914E+01, 0.1987605159106218E+01, 0.1671782272803828E+01, 0.1288709839574181E+01, 0.1226108028801468E+01, 0.1094715872236073E+01, 0.9893293804315401E+00, 0.9470004635866569E+00, 0.8354737908875075E+00, 0.7678170961167213E+00, 0.5139691139127377E+00, 0.3402258729602133E+00, 0.2640682732152568E+00, 0.1814642316568930E+00, 0.9210283120697343E-01, 0.5017295169510767E-01, 0.1311272184039149E-01, 0.2605821149103687E-06, 0.8821141249794507E-26, 0.1703621403540070E-06, 0.1022946688596755E-06, 0.1288709839574181E+01, 0.0000000000000000E+00, 0.7771811135093639E-06, 0.1129025861285597E-05, 0.1226108028801468E+01, 0.8850832554089860E-22, 0.6461044606060110E-07, 0.3214361230688769E-07, 0.1094715872236073E+01, 0.5589172803550431E-19, 0.3128365184192157E-08, 0.1088448901615084E-07, 0.9893293804315401E+00, 0.4249640396990678E-22, 0.2862531707476509E-05, 0.1047704834723032E-05, 0.9470004635866569E+00, 0.2336393132634786E-16, 0.2207335777761344E+02, 0.2207335777761343E+02, 0.1020000000000000E+03, 0.3043083900226757E+01, 0.9803921568627451E+00, 0.1031792375203169E-03, 0.7678170961167213E+00, 0.8633839815190009E-18, 0.1747705279299585E-04, 0.1213546251050483E-04, 0.5139691139127377E+00, 0.2016356967855187E-17, 0.7704963297163309E-04, 0.2063636533685701E-03, 0.3402258729602133E+00, 0.5916456789157589E-30, 0.2417908304509407E-03, 0.2207145059751390E-03, 0.2640682732152568E+00, 0.1479114197289397E-30, 0.1687694376453652E-02, 0.8888286778685235E-03, 0.1814642316568930E+00, 0.1064934658613210E-15, 0.3591290150683059E-02, 0.7604711042910346E-02, 0.9210283120697343E-01, 0.2042102137240287E-23, 0.1463034917919357E-02, 0.1882622569903463E-02, 0.5017295169510767E-01, 0.2551739098626718E-22, 0.3961160948815250E-03, 0.9438210467320612E-03, 0.1311272184039149E-01, 0.1972152263052530E-29, 0.6693623055505279E-03, 0.9046122259092353E-03, 0.2605821149103687E-06, 0.3100321587303656E-14, 0.3978735902689679E-02, 0.4253573989104679E-02},
			info: 0,
		},
		{
			z:    []float64{0.8778106868894964E+00, 0.4777882175260769E+00, 0.1820603962716897E+00, 0.7891460918977841E+00, 0.7131176407472852E+00, 0.2166796106605611E+00, 0.7093860568912320E+00, 0.6747559124413653E+00, 0.1755802732664898E+00, 0.2538529139601842E-01, 0.3839736304139417E+00, 0.4972278527690542E+00, 0.2219918935874782E+00, 0.6587468361459490E+00, 0.6959875726535614E+00, 0.8785259347357933E+00, 0.4385001119617188E+00, 0.2992124106963234E-01, 0.9853525010355352E+00, 0.1842422327604198E-01, 0.7939453301916002E+00, 0.8502867854292200E+00, 0.6683895767567380E+00, 0.8502503508041696E+00, 0.2705991244761674E-02, 0.2273590703395093E+00, 0.1996606140173212E+00, 0.5828845765712893E+00, 0.7928614954547730E+00, 0.1675870051229429E+00, 0.3078809727828886E+00, 0.6073537805183471E-01, 0.3498986727103111E+00, 0.4266607326484094E+00, 0.7790595486765419E+00, 0.6274781693151706E+00, 0.3468808495278470E+00, 0.4002308549977796E+00, 0.8927740819756569E+00, 0.5380209790008849E+00, 0.9944530064668308E+00, 0.8382845348910617E+00, 0.6578833875204938E+00, 0.6784609218227333E-01, 0.9880398702345439E-01, 0.1591591396324092E+00, 0.3625625292925116E-01, 0.7834491156107602E+00, 0.8935128283958205E-01, 0.6534249936105881E+00, 0.9608003610697393E+00, 0.9122649849010867E+00, 0.3064782192618438E+00, 0.6165812021330105E+00, 0.3942978367050161E+00, 0.2389206118855702E+00, 0.4357310309586615E+00, 0.6366539061757281E+00, 0.2487697657706114E+00, 0.3158114775243555E+00, 0.4359459725257834E+00, 0.8521387333399649E+00, 0.7376171834812397E+00, 0.7198918826028659E+00, 0.3787355957929660E+00, 0.5132345781976363E+00, 0.5880413889667653E+00, 0.8394688090887562E+00, 0.3673771632022454E+00, 0.1291928261630060E+00, 0.6552015571219838E+00, 0.7918941075374089E+00, 0.3759434916876330E+00, 0.2131514444401432E+00, 0.5997464643577372E+00, 0.8626895671971565E+00, 0.1962050188754781E+00, 0.6291330853690040E+00, 0.6873926281549517E+00, 0.8949666870308315E-01, 0.1384684788210775E+00, 0.8959871983479050E+00, 0.5967626810459625E+00, 0.6838507108780668E+00},
			n:    21,
			zOut: []float64{0.2344484145674817E+01, 0.2205594465347172E+01, 0.2180337141043826E+01, 0.1936893457336740E+01, 0.1819297464222486E+01, 0.1595314798543036E+01, 0.1497965290687764E+01, 0.1192635586559562E+01, 0.1109297670462514E+01, 0.1021894664697417E+01, 0.8474592078637809E+00, 0.7996737801504824E+00, 0.5357931354846659E+00, 0.5077230661246940E+00, 0.4005808269588222E+00, 0.2350964781455809E+00, 0.1732854339186359E+00, 0.5510322151786835E-01, 0.3865442753785928E-01, 0.1308309165665133E-01, 0.1986637888070920E-03, 0.1267902442921281E-19, 0.1632713022477316E-10, 0.9197304681744396E-11, 0.1497965290687764E+01, 0.6540946849634405E-22, 0.1644600422568816E-11, 0.2569557641484267E-11, 0.1192635586559562E+01, 0.1886570321166912E-23, 0.1155544514329289E-09, 0.3178796095468434E-10, 0.1109297670462514E+01, 0.2958228394578794E-30, 0.1814581278982536E-06, 0.5763166468812303E-07, 0.1021894664697417E+01, 0.1648133134119960E-16, 0.5817023408243420E-07, 0.2381874475610022E-07, 0.8474592078637809E+00, 0.5828670879282072E-15, 0.2051036601772319E+02, 0.2051036601772318E+02, 0.1250000000000000E+03, 0.3639455782312925E+01, 0.8800000000000001E+01, 0.1964091724818299E-07, 0.5357931354846659E+00, 0.3518203713570956E-14, 0.1200151291154190E-07, 0.2251426423705210E-07, 0.5077230661246940E+00, 0.6882401190382613E-24, 0.1803872146382089E-06, 0.4763453660743515E-06, 0.4005808269588222E+00, 0.1559915108945621E-18, 0.4978941410546894E-05, 0.5275156660739611E-05, 0.2350964781455809E+00, 0.1004043408045437E-17, 0.5647304839276210E-04, 0.2873291365503010E-04, 0.1732854339186359E+00, 0.2485883728081419E-23, 0.9907264920144563E-04, 0.1293007080503375E-03, 0.5510322151786835E-01, 0.0000000000000000E+00, 0.9623294785827948E-04, 0.7898847596644510E-04, 0.3865442753785928E-01, 0.0000000000000000E+00, 0.2279226510196514E-03, 0.3149968919986460E-03, 0.1308309165665133E-01, 0.4893402802699089E-27, 0.4145854568122489E-03, 0.3196372933697030E-03, 0.1986637888070920E-03, 0.1930680189793120E-17, 0.2183385452652873E-03, 0.3171631588309268E-03},
			info: 0,
		},
		{
			z:    []float64{0.3053504558052776E+00, 0.2234484687205406E+00, 0.6531459952011753E+00, 0.3122060875846019E+00, 0.9991431676363117E+00, 0.2727966396486101E+00, 0.9325466197866266E+00, 0.9368849148298319E+00, 0.8561644447997885E+00, 0.1054815122266185E+00, 0.4923718332155202E-01, 0.9268095624277023E+00, 0.1888864280722891E+00, 0.7655148720572605E+00, 0.1232767826602820E+00, 0.5903563091717590E+00, 0.9248190356569348E+00, 0.5395288290327440E+00, 0.9028402566412551E+00, 0.9500520891471187E-01, 0.8953008781613688E+00, 0.2933361506740548E+00, 0.8453264440546789E+00, 0.5481580781657376E+00, 0.3587646761786403E+00, 0.5176903549368316E+00, 0.7562422108585272E+00, 0.9675296263022207E+00, 0.8986594456301542E+00, 0.5421338625853966E-01, 0.1201688601088308E+00, 0.6190380352364769E+00, 0.5038255434020116E+00, 0.6721007162690862E+00, 0.5702376374225947E+00, 0.9284095682432886E+00, 0.3695163217482177E+00, 0.6039359484498441E-01, 0.4652604684312921E+00, 0.8651698808632446E-01, 0.7546245127264203E+00, 0.7397909213681499E+00, 0.1212064081570199E+00, 0.3507242515197573E+00, 0.1903823153835104E+00, 0.7217789084869874E-01, 0.4631739813227773E+00, 0.4692891642215747E+00, 0.4583968848602870E+00, 0.1862358222844885E+00, 0.2939086301666586E+00, 0.5739509914073745E+00, 0.2602639918141684E+00, 0.7265362515535626E+00, 0.2180031380994948E+00, 0.1132710387386646E+00, 0.7997129355612416E+00, 0.7503609043894605E+00, 0.7921549516519859E+00, 0.2820213454373731E+00, 0.3757885220124352E+00, 0.7700990985029622E+00, 0.7233316318551536E+00, 0.1305854233773539E+00, 0.7245124401476205E+00, 0.5999815977622437E+00, 0.3886468370487757E+00, 0.5608149286383455E+00, 0.4298380566755162E+00, 0.6436590616913128E+00, 0.3743035898091585E-01, 0.4516061963302198E-01, 0.3504820364503235E+00, 0.4814816352089226E+00, 0.5678245050510763E+00, 0.5711961955150159E+00, 0.9549329198325482E+00, 0.8527165734568565E-01, 0.4185783350739758E-01, 0.5702414021975877E-01, 0.2302271730247050E+00, 0.7442529870754924E+00, 0.5366636699040384E+00, 0.8085683101567075E+00},
			n:    21,
			zOut: []float64{0.2590499366480712E+01, 0.2541239603587953E+01, 0.2259909426243320E+01, 0.2181782566945434E+01, 0.1886341857097904E+01, 0.1651483296972034E+01, 0.1427705031981996E+01, 0.1248256205386413E+01, 0.1112814709657656E+01, 0.9343995210628773E+00, 0.8379224813887226E+00, 0.8181328249547786E+00, 0.6165308413237942E+00, 0.4939038343854810E+00, 0.4603621358717260E+00, 0.3639045182225083E+00, 0.3137819362841643E+00, 0.1577326270698492E+00, 0.8198100142281199E-01, 0.9578206270736807E-02, 0.4942793252781514E-03, 0.1292469707114106E-22, 0.1980335658106781E-04, 0.1681298419390782E-04, 0.1427705031981996E+01, 0.2584939414228211E-25, 0.8200031816835676E-05, 0.9102296987148495E-05, 0.1248256205386413E+01, 0.4930380657631324E-30, 0.4104488055426074E-09, 0.2576051753309951E-09, 0.1112814709657656E+01, 0.6452508963690269E-19, 0.2339325112754308E-08, 0.4010758969857423E-08, 0.9343995210628773E+00, 0.2041003537989240E-20, 0.4754834042401540E-06, 0.1343807701186157E-06, 0.8379224813887226E+00, 0.2942145729531872E-22, 0.2198875627193616E+02, 0.2198875627193615E+02, 0.1030000000000000E+03, 0.3158730158730159E+01, 0.5825242718446602E+01, 0.2325395492704734E-05, 0.6165308413237942E+00, 0.1475671084242936E-18, 0.4982404846255511E-06, 0.2497087821784679E-06, 0.4939038343854810E+00, 0.1295808538938520E-16, 0.6559194735079054E-05, 0.3408173383041987E-05, 0.4603621358717260E+00, 0.5811344985782108E-17, 0.9559725090946887E-05, 0.1189458663431952E-04, 0.3639045182225083E+00, 0.2913839059670682E-18, 0.4863259217476146E-04, 0.7411068946771063E-04, 0.3137819362841643E+00, 0.2338183723075079E-26, 0.4219303942492233E-03, 0.5983359002845232E-03, 0.1577326270698492E+00, 0.3451266460341927E-30, 0.4370512454739224E-03, 0.2953761682364422E-03, 0.8198100142281199E-01, 0.2482356093574565E-20, 0.7632038538424591E-04, 0.1389505478745866E-03, 0.9578206270736807E-02, 0.1680537859656637E-20, 0.1168521819007177E-01, 0.1200804951998557E-01, 0.4942793252781514E-03, 0.1967697241141140E-19, 0.1184783913560175E-01, 0.1590443390723593E-01},
			info: 0,
		},
		{
			z:    []float64{0.9721612501531746E+00, 0.9226086145236946E+00, 0.6294635256706198E+00, 0.3023176111891041E+00, 0.6049371291087220E+00, 0.2862933293136216E+00, 0.8819078559307578E+00, 0.5258944749310531E+00, 0.1024274135843443E+00, 0.4745865163816484E+00, 0.3569366663070145E+00, 0.5081536020377282E+00, 0.6994675492692721E+00, 0.3184330055525583E+00, 0.4062572718658636E+00, 0.8497577069296732E+00, 0.7686323719242980E+00, 0.8873781649289114E+00, 0.5445815352331720E+00, 0.5465168091072755E+00, 0.2655612328242208E+00, 0.7149116271405493E+00, 0.2779393895548951E+00, 0.1344927254674255E-01, 0.4544708454847078E+00, 0.5332505496545747E-01, 0.3223583007255554E+00, 0.4314145703973596E+00, 0.6194759383350579E+00, 0.9673744275195654E+00, 0.5445933229599376E-01, 0.2359803760372408E+00, 0.4349989237395262E+00, 0.3983243124260842E+00, 0.1549760127081995E+00, 0.6483062457720113E+00, 0.9796384288763583E+00, 0.2088098624466561E-01, 0.7266086220595113E+00, 0.7787182393455840E+00, 0.2714545750078465E+00, 0.3281766266219532E-01, 0.4443206765749519E+00, 0.7385116967559627E+00, 0.4742748966298079E+00, 0.2719801119596308E+00, 0.5535252469642412E+00, 0.6819653774052530E+00, 0.5802197659205470E+00, 0.8706014802153047E+00, 0.7899732601913489E-01, 0.6149130585720997E+00, 0.1184080138409910E+00, 0.7335155686110397E+00, 0.3161056790243246E+00, 0.4751009230144272E+00, 0.6074228340613933E+00, 0.1834459697241099E+00, 0.5602196728537751E+00, 0.6036792416269905E+00, 0.4996379576165632E+00, 0.5762117563802562E+00, 0.4313302593954876E+00, 0.2357090997139660E+00, 0.5266696040444221E+00, 0.4943568418910921E+00, 0.5194970579745682E+00, 0.8729901711287622E+00, 0.1253423978225398E+00, 0.3249303186427334E+00, 0.1476841747547597E+00, 0.8115004890141919E+00, 0.3358761522855614E+00, 0.6267425544884889E+00, 0.8684373664078195E+00, 0.3410509649937432E+00, 0.1223171934609151E+00, 0.6692100819077175E+00, 0.8714060240466885E+00, 0.8435328911476530E+00, 0.3613173587051273E+00, 0.5970888673100956E+00, 0.4814113673780336E+00, 0.9518076426657822E+00},
			n:    21,
			zOut: []float64{0.2554276088974017E+01, 0.2353122577016648E+01, 0.1866240795386955E+01, 0.1792222013488891E+01, 0.1776522631286352E+01, 0.1671843747376224E+01, 0.1608565757744856E+01, 0.1413448668253747E+01, 0.9978397039940720E+00, 0.9776727877490632E+00, 0.8701050228770406E+00, 0.6522479060818760E+00, 0.6295532782699175E+00, 0.4076513128864766E+00, 0.3045303763029391E+00, 0.2026925943270836E+00, 0.1597713759311400E+00, 0.1149331809326307E+00, 0.5501161781795570E-01, 0.3472328989596482E-02, 0.1615352262160075E-02, 0.2261810287929078E-15, 0.1434223756681676E-21, 0.1189989751754647E-20, 0.1608565757744856E+01, 0.1068494074045434E-17, 0.1018285523195584E-14, 0.1812446834981097E-15, 0.1413448668253747E+01, 0.8361925595342725E-26, 0.5806129122189708E-11, 0.1848580748818003E-11, 0.9978397039940720E+00, 0.5222833230866609E-21, 0.9842396145073251E-07, 0.4923018478143697E-07, 0.9776727877490632E+00, 0.9540591325261466E-21, 0.1504532523487471E-05, 0.7619936176069947E-06, 0.8701050228770406E+00, 0.2405710216561998E-25, 0.2041333911794964E+02, 0.2041333911794964E+02, 0.1050000000000000E+03, 0.3108843537414966E+01, 0.6666666666666667E+01, 0.1010063659896995E-05, 0.6295532782699175E+00, 0.2958228394578794E-30, 0.7231781167780556E-06, 0.8027785243996724E-06, 0.4076513128864766E+00, 0.2327139670401985E-28, 0.4311361064418766E-06, 0.4365718303993244E-06, 0.3045303763029391E+00, 0.3511198893363720E-17, 0.5813968676396444E-06, 0.4990623034093460E-06, 0.2026925943270836E+00, 0.5787194672316798E-22, 0.1414410322798341E-05, 0.1084116171450400E-05, 0.1597713759311400E+00, 0.3368436065293720E-27, 0.4438807230072854E-05, 0.3295442179394680E-05, 0.1149331809326307E+00, 0.3973698114522315E-22, 0.1450590122472916E-04, 0.1074561246336364E-04, 0.5501161781795570E-01, 0.1153335343143272E-20, 0.9954779132989017E-07, 0.6305695341872095E-07, 0.3472328989596482E-02, 0.7183968878891587E-15, 0.2465868832870137E-03, 0.3524118878925340E-03, 0.1615352262160075E-02, 0.1517768381645227E-26, 0.5410549418889337E-03, 0.6396403869804795E-03},
			info: 0,
		},
		{
			z:    []float64{0.9945523629868341E+00, 0.6311483525653909E+00, 0.7029190400645571E+00, 0.6452581068575963E+00, 0.9353553331334358E+00, 0.1547193049011500E+00, 0.5087511216875095E+00, 0.8821879483245458E+00, 0.5873463835171173E+00, 0.3391476900726194E+00, 0.3121839954924539E+00, 0.2777103145160921E+00, 0.2404463284577694E+00, 0.9114535183484027E+00, 0.4707329974301702E+00, 0.8965398451252625E+00, 0.6082343132413309E+00, 0.6509204493235183E+00, 0.1573499033176421E+00, 0.7416865701488552E+00, 0.1617007503826062E-01, 0.9896605969885931E+00, 0.3427350027671039E+00, 0.9677837053324486E+00, 0.1744936477417883E+00, 0.1063275349300943E+00, 0.7956875801511568E+00, 0.8998967378241718E-01, 0.9957418429488507E-01, 0.2587083053394715E+00, 0.3753593926504107E+00, 0.7537263520315082E+00, 0.1726167393116790E+00, 0.1420490749456633E+00, 0.1475360304745418E+00, 0.3281044300775052E+00, 0.4011897283558843E+00, 0.5039659382748809E+00, 0.8444659776686902E-01, 0.6685320205580927E+00, 0.8425776598223370E+00, 0.6334616337862548E+00, 0.6754035119469365E+00, 0.3594856315188868E+00, 0.4308279252773916E+00, 0.2170629535054914E+00, 0.5040120070238915E+00, 0.3780998968579012E+00, 0.9176188652711103E+00, 0.1603892455353655E+00, 0.2475008720560291E+00, 0.4414989878913728E+00, 0.3466658552370731E+00, 0.3098329951977107E+00, 0.8940921934240968E+00, 0.6686136942966417E+00, 0.4049936818549904E-01, 0.1308695446239941E+00, 0.8770734618597430E+00, 0.7917220872288512E+00, 0.5736823795257117E+00, 0.5474219864141461E+00, 0.3320663514826834E+00, 0.5824816531032581E+00, 0.6748067573570548E+00, 0.8139348519761679E+00, 0.1984641509226878E+00, 0.5557729841117627E+00, 0.1101812804319026E+00, 0.2169710965518002E+00, 0.5846617154920911E+00, 0.9784843983810533E+00, 0.8650267562586114E+00, 0.4041298540058539E+00, 0.9548165813531374E+00, 0.3839697059508718E+00, 0.4296171922210591E+00, 0.9617664542372570E+00, 0.2102779850721345E+00, 0.6135130181658475E+00, 0.3331715196673474E+00, 0.3177082868916457E+00, 0.2836520073686416E+00, 0.4452894940247868E+00},
			n:    21,
			zOut: []float64{0.2413112525759640E+01, 0.2309698811549223E+01, 0.1971065496840386E+01, 0.1839093033734385E+01, 0.1586321309986888E+01, 0.1384287501933647E+01, 0.1357349915568794E+01, 0.1342137050614370E+01, 0.1098398899504136E+01, 0.1013688381736709E+01, 0.8838922773071145E+00, 0.7023193003094995E+00, 0.6083268670387161E+00, 0.3967105732701523E+00, 0.3295901758335739E+00, 0.2610566591740404E+00, 0.2254040681461952E+00, 0.1107925081061023E+00, 0.7084089717166595E-01, 0.5790734022898784E-02, 0.1162539701808497E-05, 0.2217532248565769E-17, 0.1400266087055540E-09, 0.1806586784837133E-09, 0.1357349915568794E+01, 0.1160047396826257E-18, 0.1994113072724655E-16, 0.4228494360227003E-16, 0.1342137050614370E+01, 0.1172187884313513E-15, 0.1255075745859590E-14, 0.2503873040081153E-15, 0.1098398899504136E+01, 0.3711182370878063E-20, 0.5744299067797088E-08, 0.4873476684669810E-08, 0.1013688381736709E+01, 0.1283916659070448E-18, 0.9737119066622311E-08, 0.8382615528986061E-08, 0.8838922773071145E+00, 0.3385457556489824E-15, 0.1990987815014785E+02, 0.1990987815014784E+02, 0.1050000000000000E+03, 0.3324263038548753E+01, 0.5714285714285714E+01, 0.5531678096648157E-07, 0.6083268670387161E+00, 0.9327491343333244E-26, 0.3094358165149238E-06, 0.4992693882721107E-06, 0.3967105732701523E+00, 0.1615587133892632E-26, 0.5013465632663534E-06, 0.4474558637993164E-06, 0.3295901758335739E+00, 0.7527343574232552E-22, 0.6917755140723670E-06, 0.9191878134168280E-06, 0.2610566591740404E+00, 0.8992644540970212E-27, 0.3283328599477859E-08, 0.8980185396558612E-09, 0.2254040681461952E+00, 0.2761013168273541E-29, 0.1566859737304233E-05, 0.9651044230345755E-05, 0.1107925081061023E+00, 0.7853700447084989E-18, 0.2494389596499284E-03, 0.3103855763358142E-03, 0.7084089717166595E-01, 0.7910121402691465E-20, 0.9848486231185840E-03, 0.1285663115805527E-02, 0.5790734022898784E-02, 0.1925929944387236E-33, 0.1531134660067926E-02, 0.1103959511478054E-02, 0.1162539701808497E-05, 0.0000000000000000E+00, 0.3553926119432854E-02, 0.1161337136649311E-01},
			info: 0,
		},
		{
			z:    []float64{0.6912525216559012E+00, 0.7535513505900934E+00, 0.9471932644608124E+00, 0.3186447229836621E+00, 0.5324891284853083E+00, 0.8568362673509099E+00, 0.7683599820389093E+00, 0.5584553004793524E+00, 0.5571013534938567E+00, 0.9300192917069305E-01, 0.8580829209571639E+00, 0.7019126526839274E+00, 0.6637712341629651E+00, 0.1211000088170944E+00, 0.6340253546080570E+00, 0.8089912896293909E+00, 0.9056005723453709E+00, 0.8259750504337368E+00, 0.7787230128973186E+00, 0.5552567089885843E+00, 0.1575217143336131E+00, 0.6258385609742768E+00, 0.3943745586872103E+00, 0.2414131111537543E-02, 0.1491577729788018E-01, 0.3231570165180105E+00, 0.6629837458829935E+00, 0.3692715436026686E+00, 0.4217957028148089E+00, 0.9587355452830710E+00, 0.2048474370070816E+00, 0.9514275263988156E+00, 0.6824099372375696E+00, 0.4368303647429550E-01, 0.1779948598152153E+00, 0.8894462775192779E+00, 0.5101271266317630E+00, 0.9448652224292025E+00, 0.2265718177112338E+00, 0.3995811139403516E+00, 0.8380789267629041E+00, 0.6621441253102893E+00, 0.1548712624835822E+00, 0.6028850196944033E+00, 0.6938032609628844E+00, 0.6847684356722229E+00, 0.5160938472467536E+00, 0.2489894297094153E+00, 0.1603949345362032E-01, 0.8469682453172568E+00, 0.1332301650751776E+00, 0.4964641576870391E+00, 0.8955382322295532E+00, 0.8913224930710517E-01, 0.1632933980261509E+00, 0.7671629642421702E+00, 0.1918190416387825E+00, 0.4660642407804531E+00, 0.3421851388276426E+00, 0.6793284417353496E-03, 0.4856583571413033E+00, 0.6596584184483706E+00, 0.3999565195667565E-01, 0.4265147212719423E+00, 0.1721559491949431E-01, 0.4779030554385033E-01, 0.9628058707102425E+00, 0.4575395386200597E+00, 0.3526890078762277E+00, 0.7271726865364773E+00, 0.1741804541232641E+00, 0.3524640041347155E+00, 0.2143698389021181E+00, 0.9553499123803968E+00, 0.7803670081719427E+00, 0.7906509834069840E+00, 0.5939624443460740E+00, 0.5439294231719827E+00, 0.4026522884432556E+00, 0.2400403086558794E+00, 0.1695045416332096E+00, 0.8774151073165960E+00, 0.9528799076371641E-01, 0.6357307125994049E-01},
			n:    21,
			zOut: []float64{0.2768602537214622E+01, 0.2449987435840808E+01, 0.2103617452638331E+01, 0.2091492233971247E+01, 0.2040261542124423E+01, 0.1984511273424848E+01, 0.1507960384149227E+01, 0.1491386484166052E+01, 0.1283295633334833E+01, 0.1065640575633715E+01, 0.9760063516861833E+00, 0.9535053549649605E+00, 0.6117733318422185E+00, 0.5156356643703764E+00, 0.4217391763517565E+00, 0.2514332616796913E+00, 0.1056943786041289E+00, 0.6556299181229375E-01, 0.2554189252987301E-01, 0.1599343544217954E-01, 0.7248128851221257E-03, 0.1757582096832414E-26, 0.8938568486456872E-12, 0.7801064326234513E-12, 0.1507960384149227E+01, 0.1792073872399063E-19, 0.2167105506392983E-08, 0.2566615495401212E-08, 0.1491386484166052E+01, 0.2939740300877914E-17, 0.1513553473151668E-08, 0.1292098164447321E-08, 0.1283295633334833E+01, 0.5002192774679673E-18, 0.3422853236655498E-09, 0.3087733884567774E-09, 0.1065640575633715E+01, 0.2649562899583917E-24, 0.1824851422869684E-09, 0.1969539993879423E-09, 0.9760063516861833E+00, 0.4849588587138365E-15, 0.2273036620466689E+02, 0.2273036620466689E+02, 0.1110000000000000E+03, 0.3401360544217687E+01, 0.6306306306306307E+01, 0.7012703734578152E-09, 0.6117733318422185E+00, 0.2636779683484747E-15, 0.8832037497597917E-08, 0.1377654767646391E-07, 0.5156356643703764E+00, 0.0000000000000000E+00, 0.4403364729024244E-07, 0.2921450058151117E-07, 0.4217391763517565E+00, 0.5116160632510206E-16, 0.4627032593211780E-06, 0.6780935482482274E-06, 0.2514332616796913E+00, 0.3477119313860159E-18, 0.8396751469149006E-05, 0.1191099168408868E-04, 0.1056943786041289E+00, 0.3125614770312806E-14, 0.3873238345753621E-04, 0.1564927337192913E-03, 0.6556299181229375E-01, 0.2145521217817673E-17, 0.9476510350371504E-03, 0.9104754488359111E-03, 0.2554189252987301E-01, 0.1508279173816323E-23, 0.1093077353977330E-02, 0.3191708039391989E-03, 0.1599343544217954E-01, 0.1319576415787468E-18, 0.1003863779460019E-02, 0.1271197433268307E-02, 0.7248128851221257E-03, 0.5169135658374832E-21, 0.1128681157512944E-02, 0.4574374031799749E-02},
			info: 0,
		},
		{
			z:    []float64{0.8514967554120231E+00, 0.1903564904561825E+00, 0.9315134585755248E+00, 0.6865565499502658E+00, 0.9874780893034819E+00, 0.4243732836164926E-01, 0.4493254789963134E+00, 0.1218497220886967E+00, 0.3032420422448456E-01, 0.4472006833337416E+00, 0.7764061193501844E+00, 0.2765680149425037E+00, 0.1540380964603436E+00, 0.8227084597523057E+00, 0.1281873340097673E+00, 0.2797045229662546E+00, 0.1723485697940061E+00, 0.5407468670158238E+00, 0.3703892253237170E+00, 0.3275545813562609E+00, 0.1384768695152372E+00, 0.8683661690082241E+00, 0.2417038393999236E+00, 0.1573965961648072E+00, 0.7549703050710239E+00, 0.4197135409081310E+00, 0.5565257749798536E-01, 0.7494783707245707E+00, 0.8342833407960080E+00, 0.7207413539297494E+00, 0.3936911272324267E+00, 0.8307872407436112E-01, 0.5696677800088412E-01, 0.2580569153061851E-01, 0.3976149332803511E+00, 0.4082340292723502E+00, 0.3620800929220623E+00, 0.2078006460748600E+00, 0.6810562489816266E+00, 0.8414616295760738E+00, 0.2053319202340824E+00, 0.7745582586901971E+00, 0.8243869237197288E-01, 0.7513634029365589E+00, 0.8658814490552211E+00, 0.7014202441287165E-01, 0.8786892218281559E+00, 0.2415414918645031E-01, 0.1648367770784782E+00, 0.5852629690413500E+00, 0.6558826026732849E+00, 0.3505171007322015E+00, 0.5959744796861849E+00, 0.6000199298251039E+00, 0.5156931460104713E+00, 0.1010243586499049E-01, 0.2804608587164645E+00, 0.7088599878489240E+00, 0.3159377781412633E+00, 0.1196632783263437E+00, 0.8053993759743566E+00, 0.9537751230670044E+00, 0.6890540876884894E+00, 0.7327967809533995E+00, 0.1752177926983489E-01, 0.1176294169286665E+00, 0.1417165464977135E+00, 0.1128117051181837E+00, 0.4431718138373526E+00, 0.6383761487884151E+00, 0.4394388749172047E+00, 0.4496324720961747E+00, 0.2336818223819791E+00, 0.8283237197456694E+00, 0.5968446077081969E+00, 0.2123688346867825E-03, 0.2316409292069651E+00, 0.4041203764036865E+00, 0.5989557684304492E-01, 0.1751475090786662E+00, 0.3300532749209990E+00, 0.5257653351639615E+00, 0.9423173850128755E+00, 0.5110067889308013E+00},
			n:    21,
			zOut: []float64{0.2298367449128524E+01, 0.2183508003369306E+01, 0.1768578955898148E+01, 0.1497740650820804E+01, 0.1451044171656134E+01, 0.1248562072819587E+01, 0.1096241131844911E+01, 0.1047761158237631E+01, 0.1036921434023463E+01, 0.1001878203872264E+01, 0.6028997776208512E+00, 0.5966927468392602E+00, 0.3787547972649039E+00, 0.3164597916816877E+00, 0.2770907680746649E+00, 0.1378928396715414E+00, 0.1008983167754413E+00, 0.7523836829300481E-01, 0.4549396861483578E-01, 0.2905908538794771E-01, 0.1164397397714791E-04, 0.2593449025859535E-11, 0.4028878361793870E-19, 0.4523833078775027E-19, 0.1096241131844911E+01, 0.2254118867995285E-20, 0.6781252758190456E-19, 0.6132859990189332E-19, 0.1047761158237631E+01, 0.2208810534618833E-28, 0.7941399228881527E-19, 0.8755467647203911E-19, 0.1036921434023463E+01, 0.2697904295855860E-27, 0.1193508568348155E-18, 0.1375302458176128E-18, 0.1001878203872264E+01, 0.1808754486438286E-19, 0.1585460437287443E-17, 0.7110442921296878E-18, 0.6028997776208512E+00, 0.1121217470921487E-23, 0.1719109533586889E+02, 0.1719109533586889E+02, 0.1020000000000000E+03, 0.3002267573696145E+01, 0.7843137254901960E+01, 0.6126841018734423E-14, 0.3787547972649039E+00, 0.2514494135391975E-29, 0.3694268515215640E-12, 0.2142184401680204E-12, 0.3164597916816877E+00, 0.2495880173472380E-22, 0.1052194788283320E-11, 0.1801542652052392E-11, 0.2770907680746649E+00, 0.2042405059827892E-23, 0.3416315088185822E-10, 0.1784450556067769E-10, 0.1378928396715414E+00, 0.2231720057459842E-17, 0.5936850868223322E-09, 0.2792901329469146E-09, 0.1008983167754413E+00, 0.4009937753419447E-25, 0.1438568455567306E-07, 0.6376284135652434E-08, 0.7523836829300481E-01, 0.1673899732006569E-23, 0.3811420638095229E-06, 0.1677978014843256E-06, 0.4549396861483578E-01, 0.4108428387966160E-20, 0.9969899927958168E-05, 0.2238705130236833E-04, 0.2905908538794771E-01, 0.5621738354967019E-25, 0.1622697124847995E-03, 0.1340889241809050E-03, 0.1164397397714791E-04, 0.4733165431326071E-29, 0.2205086015306072E-02, 0.1109731617446849E-02},
			info: 0,
		},
		{
			z:    []float64{0.9566341326420642E+00, 0.1368890791339401E+00, 0.1041033686995702E+00, 0.1843288381431366E+00, 0.9983016155690853E+00, 0.1786866722264480E+00, 0.3401743522788719E+00, 0.3857333052582377E+00, 0.5001620282112933E+00, 0.9013417473823726E+00, 0.9483009872337210E+00, 0.8066933876879568E+00, 0.5299205541713562E+00, 0.1746250429119467E-01, 0.9676426134678857E+00, 0.9451210738656991E+00, 0.4964770237154315E+00, 0.7837324008636358E+00, 0.9355959680316486E+00, 0.6160646886859618E+00, 0.3400424465599898E+00, 0.5332164761252830E+00, 0.3086260018745091E+00, 0.7161810741898500E+00, 0.9586563909920308E+00, 0.5072089025298385E+00, 0.5345428289191819E+00, 0.3564033548208417E+00, 0.5073508962435702E-01, 0.5457190349920044E+00, 0.5924351087865416E+00, 0.4551284321479383E+00, 0.1212070325395470E+00, 0.4136992969230449E+00, 0.9668715153070689E+00, 0.5158905905227948E+00, 0.3815356588296225E+00, 0.9643666102790432E+00, 0.2243124357596349E-01, 0.1865949320717719E+00, 0.7081123336136068E+00, 0.8847286955309422E+00, 0.8075237017566694E+00, 0.7058935338718864E+00, 0.8930194548226011E-01, 0.5968683124630468E+00, 0.6423123320238798E+00, 0.4264046421741549E+00, 0.6535060636263267E+00, 0.8848885990305457E+00, 0.1727496532173951E+00, 0.2233116713730792E+00, 0.5179656160482806E+00, 0.8833427955005373E+00, 0.5463392573098168E+00, 0.2339495215057854E+00, 0.2531599269911875E+00, 0.8693933445310196E+00, 0.9904673923253784E+00, 0.5001765262594373E+00, 0.8475066901059425E+00, 0.7747140538451288E+00, 0.8097617518836179E+00, 0.6177833422496881E+00, 0.4294196301939426E+00, 0.8754149021622922E+00, 0.1154777972031917E+00, 0.4002319777705416E+00, 0.8948519482752643E+00, 0.2680092666078437E-01, 0.4982362166797213E+00, 0.8946688039099321E+00, 0.6004540842339657E+00, 0.7864972660235388E+00, 0.5343742612900907E+00, 0.6804570163829448E+00, 0.8534533212497195E+00, 0.6588190749186591E+00, 0.3987256626502571E+00, 0.9641448737959348E+00, 0.8049200655531427E+00, 0.8429466638354991E+00, 0.3173537092974966E+00, 0.2969758130071765E+00},
			n:    21,
			zOut: []float64{0.2691092090580979E+01, 0.2651656165852700E+01, 0.2273892452779949E+01, 0.2046491657483657E+01, 0.1856589708219707E+01, 0.1474606896148613E+01, 0.1449970763396692E+01, 0.1157826920722246E+01, 0.1131422765724169E+01, 0.1103316690199991E+01, 0.1081571209422854E+01, 0.8863514547656711E+00, 0.7487688552248520E+00, 0.4946957959671176E+00, 0.3261665336572834E+00, 0.2704116388140369E+00, 0.1035778920187539E+00, 0.8274904671505576E-01, 0.4712842456001845E-01, 0.3463118683574901E-01, 0.5254769424391846E-04, 0.2373014771939845E-22, 0.2238209824420786E-09, 0.1516830937674482E-09, 0.1449970763396692E+01, 0.6496080927593321E-18, 0.4705401432274776E-09, 0.6877602609145650E-09, 0.1157826920722246E+01, 0.9485735114056482E-11, 0.6765447972194981E-08, 0.9075269590011981E-08, 0.1131422765724169E+01, 0.0000000000000000E+00, 0.2115471642425146E-07, 0.1586124232313620E-07, 0.1103316690199991E+01, 0.1685550134725853E-21, 0.9651672077875321E-13, 0.7184159896265294E-12, 0.1081571209422854E+01, 0.5505416081326749E-23, 0.2191297069678434E+02, 0.2191297069678434E+02, 0.1080000000000000E+03, 0.3308390022675737E+01, 0.4629629629629630E+01, 0.3615019372146533E-05, 0.7487688552248520E+00, 0.2711157159063573E-25, 0.1185348717221526E-04, 0.1538970705040117E-04, 0.4946957959671176E+00, 0.7711572887860419E-23, 0.5228762878644057E-04, 0.2140066476035995E-04, 0.3261665336572834E+00, 0.1027012207743053E-16, 0.1058102716324823E-03, 0.1097923206605328E-03, 0.2704116388140369E+00, 0.1419949629397821E-27, 0.3075139383959133E-03, 0.2041016121130936E-03, 0.1035778920187539E+00, 0.1569288801032274E-18, 0.2245437021601444E-03, 0.2161294383881767E-03, 0.8274904671505576E-01, 0.1322963914266329E-22, 0.4098429023844965E-03, 0.2685224923308173E-03, 0.4712842456001845E-01, 0.1972152263052530E-27, 0.1445743106447529E-03, 0.3591079434142046E-04, 0.3463118683574901E-01, 0.1109335647967048E-30, 0.1752604588321185E-04, 0.1168961103966469E-04, 0.5254769424391846E-04, 0.0000000000000000E+00, 0.7596773881065606E-02, 0.6101958366533248E-02},
			info: 0,
		},
		{
			z:    []float64{0.8353120998595153E+00, 0.2762691216596785E+00, 0.4005263075088337E+00, 0.2013066591993183E+00, 0.7159087664363259E+00, 0.6826958051178053E+00, 0.3940626911632166E+00, 0.4990834556070268E+00, 0.6760530863202496E-02, 0.5490926358579395E-01, 0.1673564807759778E+00, 0.6884305772394704E+00, 0.3902830106720901E+00, 0.9394502319309270E+00, 0.1453732258871473E+00, 0.6913382271536774E+00, 0.4797608951630542E+00, 0.8813081041297155E+00, 0.7179153476502249E+00, 0.8684015860527726E+00, 0.4912919359735076E+00, 0.2734537604788182E+00, 0.6687443482830383E+00, 0.1554406884207427E+00, 0.2919860915775756E+00, 0.2642733542167044E+00, 0.8429147529046822E+00, 0.2888320698097314E+00, 0.8826173246454528E+00, 0.8977386905269689E+00, 0.7688524109189127E+00, 0.9833521147086286E+00, 0.9216258197795713E-01, 0.4886212819310254E-01, 0.2266462269156035E+00, 0.3411440109054314E+00, 0.3008601997419669E-01, 0.6174499545648404E+00, 0.5004878789063633E+00, 0.6820549240543587E+00, 0.2697209119346977E-01, 0.5561910637447186E+00, 0.5496066776611622E+00, 0.5185759009159874E+00, 0.1533641991284778E+00, 0.9896294286837857E+00, 0.3752499036369343E+00, 0.6748924776803822E-01, 0.2328081038279309E+00, 0.6721552304404512E+00, 0.5472950056303504E+00, 0.9949773164461492E+00, 0.5646629720515278E+00, 0.4280389734679516E+00, 0.4488723317267336E+00, 0.7263390986042261E+00, 0.2371171913738990E-01, 0.2980895305269504E-01, 0.2561842200752050E+00, 0.2554059142478259E+00, 0.5952758446711104E-01, 0.7921877816291150E+00, 0.2045719212105177E+00, 0.9215889426626128E+00, 0.7130733301970400E+00, 0.1420514661017834E+00, 0.8292002693849506E+00, 0.5327095640926792E+00, 0.9857821333545339E+00, 0.6686900428841760E+00, 0.1963263366455671E+00, 0.2881353528210300E+00, 0.7077457532403378E+00, 0.9075905470440924E+00, 0.1600253949359417E+00, 0.1037543259304946E+00, 0.3343707004174388E+00, 0.7287034220491817E-01, 0.8402900560224008E+00, 0.1343632243665364E+00, 0.8746515522185468E+00, 0.4102434537961821E+00, 0.1922774302984470E+00, 0.7386986269659255E+00},
			n:    21,
			zOut: []float64{0.2705701250380865E+01, 0.2517516394575531E+01, 0.1823532784589120E+01, 0.1787562059778101E+01, 0.1543666662169401E+01, 0.1517718677289282E+01, 0.1380407836894782E+01, 0.1248997815711926E+01, 0.1005808190406691E+01, 0.8278526045363968E+00, 0.6708485238929687E+00, 0.6082664705845506E+00, 0.5949155529076431E+00, 0.3419285270749396E+00, 0.2930861537880754E+00, 0.2677590880174420E+00, 0.1897681878114907E+00, 0.4498262444281959E-01, 0.4027859329477797E-01, 0.1212960387301111E-02, 0.4788271754899720E-05, 0.2113479533223326E-17, 0.1072835726919963E-07, 0.2769573896400322E-08, 0.1380407836894782E+01, 0.2075077381359190E-19, 0.2509615479685493E-12, 0.8472237738633739E-13, 0.1248997815711926E+01, 0.8862433277732312E-18, 0.5565142852935547E-10, 0.1115246324111066E-09, 0.1005808190406691E+01, 0.1508363342119467E-20, 0.4766364300104371E-06, 0.1686414207934524E-06, 0.8278526045363968E+00, 0.2958228394578794E-30, 0.1339811587116281E-04, 0.9618739022097639E-05, 0.6708485238929687E+00, 0.5096041447727736E-27, 0.1941181574680586E+02, 0.1941181574680586E+02, 0.1100000000000000E+03, 0.3301587301587301E+01, 0.6363636363636363E+01, 0.2769598741382734E-05, 0.5949155529076431E+00, 0.1660823573641619E-23, 0.5008194680845295E-05, 0.3644576454263968E-05, 0.3419285270749396E+00, 0.8597611808520244E-19, 0.8197192371775147E-05, 0.1495511835801176E-04, 0.2930861537880754E+00, 0.6452008777913547E-22, 0.7613853051650506E-05, 0.9176969716117157E-05, 0.2677590880174420E+00, 0.1626694295438144E-23, 0.2172962611604878E-06, 0.9910144268854854E-06, 0.1897681878114907E+00, 0.5698278443334010E-17, 0.3051734820342141E-03, 0.3112883552318466E-03, 0.4498262444281959E-01, 0.6022708464481807E-24, 0.1205042765713422E-04, 0.1759326252954196E-04, 0.4027859329477797E-01, 0.1150776253123047E-20, 0.1426316484726719E-04, 0.9508927802446297E-05, 0.1212960387301111E-02, 0.3136662420452545E-23, 0.9384394253406367E-03, 0.4268700521479601E-03, 0.4788271754899720E-05, 0.0000000000000000E+00, 0.1227549320701017E-01, 0.5132215250850077E-02},
			info: 0,
		},
		{
			z:    []float64{0.6118263281058917E+00, 0.9095172470423057E+00, 0.3600259528878788E+00, 0.2816648927398081E+00, 0.1399164597780194E+00, 0.8570391268113827E+00, 0.6056165732881742E+00, 0.8852338629006543E+00, 0.5421670517489531E+00, 0.6780884449288360E+00, 0.8758212925230515E+00, 0.8288674605669587E+00, 0.6440226363310471E+00, 0.5894956271378661E+00, 0.5783475018163303E+00, 0.3156005123186516E+00, 0.8703974350876685E+00, 0.2617715240366199E+00, 0.8370578409687427E+00, 0.5560169751702698E+00, 0.7455275952513329E-01, 0.7082615127868872E+00, 0.1297707357428401E-02, 0.3064709066297203E+00, 0.8391571848933236E+00, 0.9253805128965463E+00, 0.1130984054668048E+00, 0.8319072101853607E+00, 0.7799276064931965E+00, 0.8579163369971375E+00, 0.5543120979067712E+00, 0.1549966158329672E+00, 0.2643029344048516E+00, 0.8314141615883028E+00, 0.5782452249644381E+00, 0.6319732449892587E+00, 0.4892864093075822E+00, 0.1584418066385224E+00, 0.1120577745932696E+00, 0.6617954786615949E+00, 0.1159882395495345E+00, 0.2876505945591634E+00, 0.7061058070010036E+00, 0.7252986915993946E+00, 0.6775944839786487E+00, 0.7230010551819884E+00, 0.6571938557215073E+00, 0.7482251296401533E+00, 0.9684494841571512E-01, 0.1940071502680237E+00, 0.8896785746779762E+00, 0.9170179284973872E+00, 0.8053995841969270E+00, 0.9321965525342302E+00, 0.7288973738137661E-01, 0.3849028957514653E+00, 0.4410947536600551E+00, 0.4770808300783691E-01, 0.3690162722923055E+00, 0.1908498348358704E+00, 0.8094133574022262E+00, 0.4809437445509840E+00, 0.1405808283203624E+00, 0.4482801388864215E+00, 0.7023724161235537E+00, 0.5041894616412674E+00, 0.2984594028552712E+00, 0.8510121345026274E+00, 0.7612412832182369E-01, 0.8890771840141870E+00, 0.8611357776125709E+00, 0.7702458212424681E+00, 0.8813091992976040E+00, 0.3838156614335277E+00, 0.1536911508311884E+00, 0.2763378258408019E+00, 0.7613349010649187E+00, 0.4228084592762277E+00, 0.5894002697827440E+00, 0.5744691724607953E-01, 0.3014767112059925E+00, 0.7221203248495245E-01, 0.7619553096876313E+00, 0.5154237548276358E+00},
			n:    21,
			zOut: []float64{0.2723457737035386E+01, 0.2426437846006860E+01, 0.2169822787916940E+01, 0.1997417054976096E+01, 0.1928996526776896E+01, 0.1891356815602834E+01, 0.1803418741268920E+01, 0.1442378108645599E+01, 0.1026854281138765E+01, 0.9766368850685674E+00, 0.8449303421617025E+00, 0.7575761304896602E+00, 0.7013909577306906E+00, 0.6362631409338654E+00, 0.3605885444762012E+00, 0.1912852882240109E+00, 0.1837209447646653E+00, 0.1221129878112117E+00, 0.2486838916706533E-01, 0.9765050532318768E-02, 0.3171294864219318E-06, 0.1371236688474761E-19, 0.2987011438498780E-15, 0.8087574764480620E-15, 0.1803418741268920E+01, 0.5451390847569379E-21, 0.7270236634641202E-11, 0.2896404140496822E-11, 0.1442378108645599E+01, 0.6954964380428437E-14, 0.4131395561099005E-08, 0.7275670618661885E-08, 0.1026854281138765E+01, 0.1292469707114106E-25, 0.1838064307199299E-06, 0.1238204560933622E-06, 0.9766368850685674E+00, 0.3388131789017201E-20, 0.3880169268725981E-06, 0.5681289313967714E-06, 0.8449303421617025E+00, 0.6785686819474261E-17, 0.2221927887785774E+02, 0.2221927887785774E+02, 0.1080000000000000E+03, 0.3244897959183673E+01, 0.4629629629629630E+01, 0.3025393053677348E-08, 0.7013909577306906E+00, 0.1341063538875720E-28, 0.3767849368753714E-07, 0.3184959175038880E-07, 0.6362631409338654E+00, 0.8874685183736383E-29, 0.5999352088776256E-07, 0.7114794232343430E-07, 0.3605885444762012E+00, 0.1582265649256097E-24, 0.2391907791779699E-03, 0.3344095791507466E-03, 0.1912852882240109E+00, 0.1130832107634320E-25, 0.4040482702647743E-03, 0.3567248396804949E-03, 0.1837209447646653E+00, 0.4930380657631324E-31, 0.1016317982961682E-02, 0.8970313840717793E-03, 0.1221129878112117E+00, 0.2076836816009399E-16, 0.1892596629513954E-02, 0.2329446640559320E-02, 0.2486838916706533E-01, 0.1713835026472478E-23, 0.3568672048225553E-04, 0.1599561599209249E-03, 0.9765050532318768E-02, 0.0000000000000000E+00, 0.3650779499194616E-03, 0.5712243726145556E-03, 0.3171294864219318E-06, 0.8599783076848839E-19, 0.2181668718924628E-02, 0.9598847336904798E-03},
			info: 0,
		},
		{
			z:    []float64{0.1133237741366075E+00, 0.6519297783748091E+00, 0.1419059404596466E+00, 0.6942573823108688E+00, 0.4899507169502382E-03, 0.4408585439270172E+00, 0.4089484333544855E+00, 0.8087151633238087E+00, 0.7299241994604067E-01, 0.7772039339909241E+00, 0.6160100823712519E+00, 0.1679565169487562E+00, 0.3276744683398485E+00, 0.8481565553508272E+00, 0.5614085421003249E+00, 0.5889702994117808E+00, 0.9865181317193454E+00, 0.5686958983255483E+00, 0.2173616127956169E+00, 0.6617740807852496E+00, 0.1324295788322836E+00, 0.5932751564887147E+00, 0.4417023572495267E+00, 0.8308006832392040E+00, 0.6963900525231828E+00, 0.7023649910778577E+00, 0.2540087053190563E+00, 0.5116605124111596E+00, 0.4580501208371790E+00, 0.9900881058795367E+00, 0.6233054379975522E+00, 0.5093215893644945E+00, 0.8805051397371310E+00, 0.5918749676473295E+00, 0.8024628157090720E+00, 0.1089848141144264E+00, 0.9385822032209372E+00, 0.9676954608704463E+00, 0.7567544231269694E+00, 0.7763472508927070E+00, 0.7544500156824916E+00, 0.7080578938468585E-02, 0.8622695184492650E+00, 0.1268681530179145E+00, 0.2408963669487220E+00, 0.1909503620360825E+00, 0.9251648557888137E-01, 0.4447315055648083E+00, 0.1504005134322962E+00, 0.6245252202415555E+00, 0.8432800589533600E+00, 0.1408178180805120E+00, 0.8538656063298837E+00, 0.6959250082173740E-01, 0.7452290984944354E-01, 0.8186895700534982E+00, 0.4817344889163616E+00, 0.8941802149688474E+00, 0.8277272846279742E+00, 0.3122760515919010E+00, 0.4631046821883726E+00, 0.9939521405533804E+00, 0.5067334594578294E+00, 0.3251044449945518E+00, 0.4514296322650755E+00, 0.9520015267726308E+00, 0.3811787795610073E+00, 0.6466473208499942E+00, 0.1035647165455822E-01, 0.3768739780085785E+00, 0.7301131130949717E+00, 0.6341150979687762E+00, 0.2006486990564113E+00, 0.7995641372737874E+00, 0.3501522989293299E+00, 0.7212117404145123E+00, 0.3724384783514179E+00, 0.2704745802524587E+00, 0.5954948132025037E+00, 0.8796174543237598E+00, 0.2533217117523398E+00, 0.2406431629179949E+00, 0.4238947154458014E+00, 0.1002294321190256E+00},
			n:    21,
			zOut: []float64{0.2763722715135843E+01, 0.2473753815483350E+01, 0.2367461218794171E+01, 0.2245079256722998E+01, 0.1833576927640889E+01, 0.1670289128561736E+01, 0.1457654822263786E+01, 0.1350493275779479E+01, 0.1277854548609735E+01, 0.1156481310792204E+01, 0.1106963375568883E+01, 0.8704957369830484E+00, 0.6845380809859644E+00, 0.4950109522946930E+00, 0.4198272059610658E+00, 0.2867476262636770E+00, 0.2180239597616727E+00, 0.1578141753335409E+00, 0.1153511714179556E+00, 0.2506658150747594E-01, 0.5048803894682267E-08, 0.8470329472540474E-20, 0.3080709680328647E-07, 0.2263725078605666E-07, 0.1457654822263786E+01, 0.6882142696441190E-21, 0.4511645736903135E-07, 0.5799182381327367E-07, 0.1350493275779479E+01, 0.3225542408567476E-17, 0.3673248914261325E-08, 0.2431637013515395E-08, 0.1277854548609735E+01, 0.1479114197289397E-30, 0.1457855695816495E-07, 0.1029378003116242E-07, 0.1156481310792204E+01, 0.9441087313685069E-24, 0.4178859307872738E-07, 0.3166226192774972E-07, 0.1106963375568883E+01, 0.1745440919131533E-19, 0.2297620589091097E+02, 0.2297620589091097E+02, 0.9600000000000000E+02, 0.3092970521541950E+01, 0.3125000000000000E+01, 0.2193086683474567E-06, 0.6845380809859644E+00, 0.6815668447404457E-13, 0.1173742870849872E-05, 0.2415236474615244E-05, 0.4950109522946930E+00, 0.4604423331594002E-25, 0.1089056660110428E-04, 0.1249035785040860E-04, 0.4198272059610658E+00, 0.2034864306780130E-21, 0.1807219455044531E-04, 0.1582973453084106E-04, 0.2867476262636770E+00, 0.1157544870450433E-16, 0.1788428346885188E-06, 0.1048647571833164E-05, 0.2180239597616727E+00, 0.2820966597070338E-26, 0.2675556367648033E-04, 0.1965631735302374E-04, 0.1578141753335409E+00, 0.4632909363938796E-19, 0.6657143729168815E-04, 0.2311126907684945E-04, 0.1153511714179556E+00, 0.7089566920435294E-17, 0.2519028315687161E-03, 0.1126597449160621E-03, 0.2506658150747594E-01, 0.1511791266627385E-18, 0.1819059120658101E-02, 0.1223404988252185E-02, 0.5048803894682267E-08, 0.5048803894682267E-08, 0.1176483960189416E-01, 0.7561758166988619E-02},
			info: 0,
		},
		{
			z:    []float64{0.6007483099411913E+00, 0.8757604992442719E+00, 0.3498665716095440E+00, 0.4974125174432915E+00, 0.3942177430818359E+00, 0.4655097178512825E+00, 0.7062255099400584E+00, 0.5955461614709365E+00, 0.5705117921786074E+00, 0.2580859610386106E+00, 0.5213970441770472E+00, 0.3227485215512098E+00, 0.8200431655548257E-01, 0.9521784777464899E+00, 0.9768302381970198E+00, 0.5401259459761943E+00, 0.2876249345369413E+00, 0.9928912621382637E+00, 0.9068004769556834E+00, 0.8136746797372185E+00, 0.6557906174128045E+00, 0.7593059043472016E+00, 0.7306718357524993E+00, 0.7872142480398446E+00, 0.2169453387448870E+00, 0.1236189910537125E+00, 0.4376414329540577E+00, 0.2475974410903709E+00, 0.5743712937757064E+00, 0.8451940172770922E+00, 0.2954362279165403E+00, 0.9333330371592371E+00, 0.7261884685452219E+00, 0.2905925760133674E+00, 0.3022535752338255E+00, 0.6907049310391071E+00, 0.2616870809800952E+00, 0.3758121789769751E+00, 0.7112612645989228E+00, 0.9602233026485287E+00, 0.7212734445945013E+00, 0.9266335266312882E+00, 0.7829804394259932E-01, 0.6683939253194693E+00, 0.3539007443404153E+00, 0.1629995906155713E+00, 0.2470024645126471E+00, 0.7250237828800384E+00, 0.9725422312806663E+00, 0.1983270881451602E+00, 0.2167236969811593E+00, 0.2406674239058443E+00, 0.3811046126380717E+00, 0.6358718370494476E-01, 0.6038903598347725E+00, 0.2622945348536884E+00, 0.4871669697876424E+00, 0.1853788955105938E+00, 0.3471794920367192E+00, 0.9520318129455714E+00, 0.9596274379126818E+00, 0.5893179749928269E+00, 0.8560240909608330E-01, 0.9435308110759013E+00, 0.2058424446004554E+00, 0.2877465111691004E+00, 0.6254635389850627E+00, 0.5518846905281638E-01, 0.1132608874973561E+00, 0.8191427866511807E+00, 0.1031178242713734E+00, 0.1909407653284053E+00, 0.5428362177906626E+00, 0.2973328401307228E+00, 0.6079142025065691E+00, 0.1448090006401505E+00, 0.5807148716343321E+00, 0.2207276122221573E+00, 0.4617526470734991E+00, 0.2133930631010816E-01, 0.8719991154365724E+00, 0.4224743304865443E+00, 0.7012315055540164E+00, 0.9970966526798359E+00},
			n:    21,
			zOut: []float64{0.2875734378878489E+01, 0.2344615299508293E+01, 0.2292398069608056E+01, 0.2200897502376554E+01, 0.2016889876654445E+01, 0.1997885510503641E+01, 0.1786196162676624E+01, 0.1505330793574375E+01, 0.1193246823746953E+01, 0.1153847743209180E+01, 0.9864509519829328E+00, 0.8076407462080507E+00, 0.6427729668463894E+00, 0.4808392927305387E+00, 0.3994667257867982E+00, 0.3273321022655395E+00, 0.1654363687174887E+00, 0.1176242694536732E+00, 0.4639514928819789E-01, 0.1592499252046327E-01, 0.3521629889976334E-03, 0.1784216669875992E-16, 0.7121431862167897E-10, 0.2812798494121821E-10, 0.1786196162676624E+01, 0.4108568523994154E-17, 0.4438281217193102E-07, 0.2274689950145052E-07, 0.1505330793574375E+01, 0.7499168293791558E-11, 0.5226296435976747E-05, 0.8641809444662033E-05, 0.1193246823746953E+01, 0.0000000000000000E+00, 0.2628223783350490E-04, 0.3987870710471355E-04, 0.1153847743209180E+01, 0.5865702946789929E-16, 0.1241308452869419E-07, 0.3894882997253744E-08, 0.9864509519829328E+00, 0.7673863643253590E-18, 0.2335727788952568E+02, 0.2335727788952568E+02, 0.1040000000000000E+03, 0.3058956916099773E+01, 0.2884615384615385E+01, 0.3948721236408985E-04, 0.6427729668463894E+00, 0.1028674620408199E-26, 0.2391232675375281E-03, 0.2086678697444007E-03, 0.4808392927305387E+00, 0.3130051437900657E-19, 0.1426864276581422E-03, 0.1522473256880454E-03, 0.3994667257867982E+00, 0.2101777886998446E-24, 0.1526280399103512E-05, 0.7002070790936748E-06, 0.3273321022655395E+00, 0.1242239255406691E-15, 0.1253374424171423E-04, 0.2848254240665514E-04, 0.1654363687174887E+00, 0.2455402723507254E-17, 0.1659752591164777E-02, 0.8476830847116842E-03, 0.1176242694536732E+00, 0.1344218982496604E-26, 0.1195138829435961E-02, 0.1583065192028977E-02, 0.4639514928819789E-01, 0.2205655090997949E-26, 0.2795062437264197E-03, 0.9604240981222648E-03, 0.1592499252046327E-01, 0.6574583373762050E-21, 0.7008887495297913E-02, 0.4078689360266482E-02, 0.3521629889976334E-03, 0.2981894221735425E-26, 0.7785429009218255E-02, 0.8697865294154498E-02},
			info: 0,
		},
		{
			z:    []float64{0.3565428176835072E+00, 0.1947118362244993E+00, 0.7741156920152520E+00, 0.2422304602782238E+00, 0.6344560478781652E+00, 0.6454994272905118E+00, 0.1782236022643452E+00, 0.1761708296759242E+00, 0.4902923959906477E+00, 0.3501715044277058E+00, 0.8606282659620700E+00, 0.4671519599404039E-03, 0.9177061631942818E+00, 0.9869535184930249E+00, 0.9615604930014852E+00, 0.2283553018405985E+00, 0.2470322441623238E+00, 0.5789642646481346E+00, 0.9797717987449011E+00, 0.2628090453859127E+00, 0.1063995753795172E+00, 0.9446865880365994E+00, 0.8109285339456792E+00, 0.4534650679402863E-01, 0.9560542860825706E+00, 0.7338084401822241E+00, 0.4719995459594307E+00, 0.8981162685661518E+00, 0.6088366551058919E+00, 0.4061662134958367E+00, 0.6141507801286759E+00, 0.7683300427463863E+00, 0.2030033710532269E+00, 0.5109903203799075E+00, 0.1295039929316932E-01, 0.1313823831505738E+00, 0.4600541675974348E+00, 0.8052912479215125E+00, 0.7315674921483241E+00, 0.1311059905375703E+00, 0.9372521946935908E-01, 0.2804058432106148E+00, 0.1426401168120447E+00, 0.1302494454623315E-01, 0.9936953989155247E+00, 0.7109841901452127E+00, 0.8438933393801377E+00, 0.5599779486257196E+00, 0.6224094357036947E-01, 0.3722277721927331E+00, 0.2128496963914419E+00, 0.2132595627624145E+00, 0.1761474673379855E+00, 0.5003566260610350E+00, 0.5448249098276142E+00, 0.5701738373290982E+00, 0.5196161461339291E+00, 0.4684002904633344E-01, 0.3091316721875115E+00, 0.9311143262655500E+00, 0.5165589926587729E+00, 0.4230658512580241E+00, 0.1580095970610971E+00, 0.9213403630754643E+00, 0.8395102804227733E+00, 0.6279936637330286E-01, 0.2118369865058433E+00, 0.5654902136914255E+00, 0.8603345552081203E+00, 0.6355651085391135E+00, 0.9504788095745560E+00, 0.4928822316317183E+00, 0.8546738890428598E+00, 0.1506739079739667E+00, 0.7955004699434830E+00, 0.8623803468730415E+00, 0.3328976137523824E+00, 0.6282681071703561E+00, 0.5253150183233042E+00, 0.5508861108006231E+00, 0.8482246434328706E+00, 0.3438184421622726E+00, 0.4546750663905129E+00, 0.2702842147468514E+00},
			n:    21,
			zOut: []float64{0.2599043716461175E+01, 0.2392272222815941E+01, 0.1945154273268249E+01, 0.1882117601080557E+01, 0.1808214125087810E+01, 0.1701027479657859E+01, 0.1674631474676720E+01, 0.1432830957842129E+01, 0.1245386641860508E+01, 0.1003047986420639E+01, 0.5925499318269597E+00, 0.5737323778075372E+00, 0.4619197085153934E+00, 0.3795861721644865E+00, 0.2896752472940557E+00, 0.2024598813435771E+00, 0.1753335138277014E+00, 0.8865983947068434E-01, 0.4088280438555415E-01, 0.2271023207656241E-01, 0.3207012114254504E-03, 0.1328658858913301E-22, 0.6931093261583948E-12, 0.1962539051213948E-11, 0.1674631474676720E+01, 0.1972152263052530E-30, 0.1003060686577864E-09, 0.7786971971568132E-10, 0.1432830957842129E+01, 0.3552283923653146E-20, 0.1335875663368593E-08, 0.1098996752778210E-08, 0.1245386641860508E+01, 0.6078417508770261E-16, 0.6005521677860925E-08, 0.4565873133117504E-08, 0.1003047986420639E+01, 0.4907378230941548E-20, 0.6042756667162823E-07, 0.1337733817342293E-06, 0.5925499318269597E+00, 0.7849166006949067E-28, 0.2051155688909552E+02, 0.2051155688909552E+02, 0.1190000000000000E+03, 0.3632653061224490E+01, 0.6722689075630252E+01, 0.9823469248695270E-04, 0.4619197085153934E+00, 0.3582825487703231E-19, 0.4408755124729528E-13, 0.6425665938157879E-14, 0.3795861721644865E+00, 0.3049769924381334E-19, 0.2649002399507504E-08, 0.1284693180187630E-07, 0.2896752472940557E+00, 0.2730342256278547E-24, 0.3857186710932351E-04, 0.7729247272182434E-05, 0.2024598813435771E+00, 0.2411244474478362E-13, 0.9974443273860175E-04, 0.7715885583051736E-04, 0.1753335138277014E+00, 0.1003583340528496E-19, 0.1275804036051463E-05, 0.1384195629265487E-05, 0.8865983947068434E-01, 0.1976477763343393E-17, 0.9250271469637329E-05, 0.1838436200742544E-04, 0.4088280438555415E-01, 0.1570480975080192E-18, 0.1665813744256122E-03, 0.9368787794988337E-04, 0.2271023207656241E-01, 0.1120555887990158E-19, 0.1174000031728850E-02, 0.1217958466428532E-03, 0.3207012114254504E-03, 0.0000000000000000E+00, 0.1126152474028795E-01, 0.4219515300090987E-02},
			info: 0,
		},
		{
			z:    []float64{0.1431540885741494E+00, 0.8417471667062104E+00, 0.3721480902981644E-01, 0.9023772086815021E+00, 0.7374866167630612E-01, 0.8092071841305744E+00, 0.8019394206305236E+00, 0.4350675566733520E+00, 0.8704440278245326E+00, 0.8549104680538234E+00, 0.1422025173760633E+00, 0.8968456705882693E+00, 0.1647179447265090E-01, 0.8994077105608889E-01, 0.8596481603583839E-02, 0.3900328616396558E+00, 0.4012630844305533E+00, 0.8262908556770313E+00, 0.9026062467432411E+00, 0.6485345137550630E+00, 0.5598932241080331E+00, 0.6084992568220661E+00, 0.5039450314115669E+00, 0.3490854750626526E+00, 0.9921372250281116E+00, 0.1522469068827847E-01, 0.9395473455386270E+00, 0.3928498832803473E-01, 0.1672951108814763E+00, 0.9480670302336689E+00, 0.8743929741539138E+00, 0.9201765927311700E-01, 0.8641536356563365E+00, 0.8867562289118487E+00, 0.7156446870854873E+00, 0.5025184567500440E-01, 0.4878799838415181E+00, 0.5568327059782646E+00, 0.4596548235310455E+00, 0.6857902774944131E+00, 0.4795565553491499E-01, 0.4752906527216701E+00, 0.9288351773531449E+00, 0.7419018575576386E+00, 0.9987802353476521E+00, 0.8896105755435116E+00, 0.3190450046252536E+00, 0.9685806853440787E+00, 0.3396383774694021E+00, 0.9164401886915974E+00, 0.1269879571285023E+00, 0.7912318128907188E+00, 0.9987805345221650E+00, 0.4107567957671243E+00, 0.4798441760727139E+00, 0.2357309197085595E+00, 0.3404494185276096E+00, 0.7067865773496134E+00, 0.2931542927088892E+00, 0.6654441577727066E+00, 0.9279529535770864E+00, 0.9667208022109988E+00, 0.5518920376663735E+00, 0.1657487215954807E+00, 0.3897636058410684E+00, 0.9489706352102197E+00, 0.9626390247015670E+00, 0.1020775856599213E+00, 0.3545784768064359E+00, 0.3098089336816090E+00, 0.3791089991258285E+00, 0.6519489570740321E+00, 0.4763523952410913E+00, 0.3754524631080590E+00, 0.9792813089520809E+00, 0.1998301915332230E+00, 0.7618060057813935E+00, 0.1923436350325713E+00, 0.2507012719817848E+00, 0.9704520069999245E-01, 0.2277384851702763E+00, 0.7790727508043876E+00, 0.9105177383528497E+00, 0.9714469561729607E+00},
			n:    21,
			zOut: []float64{0.2499720115178021E+01, 0.2371028651009860E+01, 0.2298220416774619E+01, 0.1883765650093571E+01, 0.1735426237683963E+01, 0.1616050162695453E+01, 0.1545275083087796E+01, 0.1187762610175739E+01, 0.1136805332133151E+01, 0.9576839460725933E+00, 0.9184986446054146E+00, 0.7702442640767845E+00, 0.6433137908930019E+00, 0.6160049176831036E+00, 0.3208789970551323E+00, 0.1900871628722621E+00, 0.9670499780074775E-01, 0.7460708611062000E-01, 0.6486083888512703E-01, 0.9966295048259487E-02, 0.4462615269650136E-07, 0.2765885173224186E-23, 0.3446029508568731E-11, 0.2660256849760007E-11, 0.1545275083087796E+01, 0.1351675564459578E-15, 0.5238898174323723E-11, 0.6640890994279970E-11, 0.1187762610175739E+01, 0.1020402561847179E-21, 0.5048392892414130E-10, 0.4263729817754413E-10, 0.1136805332133151E+01, 0.1429794613494979E-24, 0.1140378416083416E-09, 0.8548580688939750E-10, 0.9576839460725933E+00, 0.1302866745903472E-12, 0.1207358062805810E-07, 0.1106598026337004E-07, 0.9184986446054146E+00, 0.3538135823224864E-23, 0.2093690524456137E+02, 0.2093690524456137E+02, 0.1130000000000000E+03, 0.3331065759637188E+01, 0.7964601769911504E+01, 0.4249897555997367E-08, 0.6433137908930019E+00, 0.4627357327667341E-16, 0.5702383148705814E-08, 0.8656383665454304E-08, 0.6160049176831036E+00, 0.0000000000000000E+00, 0.4551987331271016E-07, 0.8434005257711676E-07, 0.3208789970551323E+00, 0.2939872200043623E-13, 0.4894684795516590E-05, 0.8200734504818474E-05, 0.1900871628722621E+00, 0.9080407486043539E-23, 0.2659953011765320E-05, 0.2713538128245964E-05, 0.9670499780074775E-01, 0.1949336478520425E-16, 0.1287973346598181E-04, 0.3266830247798530E-04, 0.7460708611062000E-01, 0.0000000000000000E+00, 0.5588793940970657E-03, 0.8233963212342685E-03, 0.6486083888512703E-01, 0.1680331387617770E-17, 0.5510428852120968E-02, 0.6166917183278490E-02, 0.9966295048259487E-02, 0.9229672591085838E-28, 0.3019065027713110E-04, 0.7674568090533565E-04, 0.4462615269650136E-07, 0.1609276246650864E-27, 0.6212935138424260E-03, 0.6196304337665942E-02},
			info: 0,
		},
		{
			z:    []float64{0.2117689827794353E+00, 0.3469390325266096E+00, 0.8556157513093896E+00, 0.4757820823688597E-01, 0.8174618541686680E+00, 0.7502623797079592E+00, 0.3473865910309967E+00, 0.1102061280439136E+00, 0.3838366632541155E+00, 0.4335750892772081E+00, 0.8350483733926333E+00, 0.8472841724322083E+00, 0.7001411292294977E+00, 0.1602971622162627E+00, 0.8475208106005836E+00, 0.1606731136869689E+00, 0.1282020202063155E+00, 0.7021142414638628E+00, 0.6509542291315448E+00, 0.3403267711628978E+00, 0.2783715946342622E+00, 0.2391807909006515E+00, 0.9326061845277929E+00, 0.8200594229325990E+00, 0.5326312099842309E+00, 0.4390518762985745E+00, 0.9087332134146432E+00, 0.8586305843825174E+00, 0.5289458618923581E+00, 0.6098713986238019E+00, 0.4124733909450122E+00, 0.2730411694523917E+00, 0.8437528550291284E+00, 0.6776322145297138E+00, 0.6772761566308538E+00, 0.6457716484911746E+00, 0.4290441069454122E+00, 0.9486890113840626E+00, 0.1145429178800543E+00, 0.8512453512490206E+00, 0.6435458910126746E+00, 0.6571563420730420E-01, 0.6473933144723745E+00, 0.2355560081089225E+00, 0.1262112692360912E+00, 0.6892345322103945E+00, 0.9088034154559810E-01, 0.5725023743105110E+00, 0.8924086653580375E+00, 0.6645221244511262E+00, 0.7287357579298158E+00, 0.6462379994906295E+00, 0.8254375000545862E+00, 0.2402799002378904E+00, 0.2312499677892260E+00, 0.7164295349077132E-01, 0.1216505240532725E+00, 0.5279275209153104E+00, 0.2010432640871422E+00, 0.2335833224032272E+00, 0.7053869472451330E+00, 0.1271428628255256E-01, 0.5920854820119847E-01, 0.1973220586788875E+00, 0.9024202349843203E+00, 0.2696040474399301E+00, 0.2399100427530529E+00, 0.4936945945994136E+00, 0.7377918193661529E+00, 0.4454167008139350E+00, 0.6822934143143184E+00, 0.1980205964978332E+00, 0.9382259163150929E+00, 0.7952650736079295E+00, 0.2147267423149601E+00, 0.2160850567828982E+00, 0.2063316967709339E-02, 0.4613555480251579E+00, 0.2983741464914229E+00, 0.7653433710630531E+00, 0.9319255930520460E+00, 0.9624604653562387E+00, 0.7921958309630666E+00, 0.3349247737517954E+00},
			n:    21,
			zOut: []float64{0.2554012710728313E+01, 0.2350385283314499E+01, 0.2336543487584985E+01, 0.2078331003959482E+01, 0.1819938506810449E+01, 0.1751888604422140E+01, 0.1642765030717598E+01, 0.1391850148176030E+01, 0.1265614265169072E+01, 0.1192296172100378E+01, 0.1118290194286764E+01, 0.7471001949454404E+00, 0.6602310515087256E+00, 0.5223373748122110E+00, 0.3101567956620342E+00, 0.2634177676934227E+00, 0.1468471914237014E+00, 0.9669691105737251E-01, 0.6862527639473751E-01, 0.2251868425683227E-01, 0.2442899974694956E-02, 0.0000000000000000E+00, 0.3247526756203984E-07, 0.3301672636248404E-07, 0.1642765030717598E+01, 0.2849742554922193E-13, 0.2093860354423170E-09, 0.9504157619668633E-09, 0.1391850148176030E+01, 0.5471978423050590E-17, 0.2765846142955850E-06, 0.3763175370346248E-06, 0.1265614265169072E+01, 0.7099748146989106E-29, 0.8629250150360319E-06, 0.7218261937815504E-06, 0.1192296172100378E+01, 0.0000000000000000E+00, 0.1792585742344500E-05, 0.1391262462240339E-05, 0.1118290194286764E+01, 0.2158808112824856E-22, 0.2234228955499889E+02, 0.2234228955499888E+02, 0.1130000000000000E+03, 0.3253968253968254E+01, 0.7079646017699115E+01, 0.1273852622682608E-03, 0.6602310515087256E+00, 0.1370219837932689E-23, 0.6841106514156665E-04, 0.6915431692027581E-04, 0.5223373748122110E+00, 0.8086955524743426E-16, 0.3151324787101243E-03, 0.4936870820060986E-03, 0.3101567956620342E+00, 0.1690864291947714E-17, 0.5342613094053112E-03, 0.4525307423692253E-03, 0.2634177676934227E+00, 0.1569544193046020E-19, 0.8085628512848309E-03, 0.5320256570924908E-03, 0.1468471914237014E+00, 0.1725633230170963E-30, 0.2171574910172427E-04, 0.7393923467130197E-05, 0.9669691105737251E-01, 0.8185725058097063E-19, 0.6259416979424432E-04, 0.7993629240914341E-04, 0.6862527639473751E-01, 0.1144304443709360E-17, 0.8363460972147480E-05, 0.2145990923718838E-04, 0.2251868425683227E-01, 0.3423656328659191E-26, 0.4258124540743150E-02, 0.4456054742147899E-02, 0.2442899974694956E-02, 0.7553929676107889E-20, 0.1657093833399320E-01, 0.2123214045479585E-01},
			info: 0,
		},
		{
			z:    []float64{0.5143465832338334E-01, 0.9671338481452775E+00, 0.2594126021369855E+00, 0.4887933538896223E+00, 0.4809357095031462E+00, 0.1648852435705869E+00, 0.5382296047886386E+00, 0.2404093745709619E+00, 0.5749849863516346E+00, 0.8755417117410101E+00, 0.6479432781744229E+00, 0.8462387170859478E+00, 0.1030360983310576E+00, 0.2790419505795448E+00, 0.2643216914883461E+00, 0.5072751399665939E+00, 0.9813013168637370E+00, 0.5270325239254721E+00, 0.1341185443667956E+00, 0.1666757001787883E+00, 0.8421762037577841E+00, 0.6333707418037352E+00, 0.9714076539879462E+00, 0.8133030640726957E-01, 0.5961674478430062E+00, 0.9962348063653491E+00, 0.9690228851917282E-02, 0.4933763527820312E-01, 0.1005560439675963E+00, 0.9794838233988775E+00, 0.8412211319105695E+00, 0.4530689238841823E+00, 0.9824397821984348E+00, 0.9435184376619912E+00, 0.5899369853858024E+00, 0.3347062982138894E+00, 0.6634688706474758E+00, 0.7781534216536603E+00, 0.5213269680559092E+00, 0.2415498257414400E+00, 0.4153017903771141E+00, 0.6142268391585892E-01, 0.1582256219487426E+00, 0.5180630943028907E+00, 0.2531248086553088E+00, 0.4588063905612799E+00, 0.9850645955475579E-01, 0.4386688206276981E+00, 0.6723312445641013E+00, 0.2417358982121058E+00, 0.9102202020018957E+00, 0.4790528316718639E+00, 0.8645283277006380E+00, 0.5896385561561974E+00, 0.3547186358320912E+00, 0.2570702739547999E+00, 0.8872414069371877E+00, 0.7977277358714381E+00, 0.2283476919840042E+00, 0.5481349489617042E+00, 0.9565429487676439E+00, 0.3941453871538936E-01, 0.2005355998230810E+00, 0.6815820255993522E+00, 0.8827641753074383E+00, 0.3721138471690448E+00, 0.3473678385941080E+00, 0.5958558861949184E+00, 0.5289126915593071E+00, 0.7848410946975201E+00, 0.9380898148044153E+00, 0.4238368524709774E+00, 0.9095202310640094E-01, 0.5205916750079600E+00, 0.3388678441932766E+00, 0.9218715971576251E+00, 0.2785357832665022E+00, 0.7353529037048490E+00, 0.3430576699659011E+00, 0.1583386336396764E-03, 0.1574217518812531E+00, 0.2808549182652627E+00, 0.9165809855463820E+00, 0.2729923295285186E-01},
			n:    21,
			zOut: []float64{0.2740279952468604E+01, 0.2296796691149573E+01, 0.2162929780566866E+01, 0.2022438316208162E+01, 0.1971555203048247E+01, 0.1712998698314224E+01, 0.1596820997274643E+01, 0.1536702996895977E+01, 0.1112897445143912E+01, 0.8259900943746278E+00, 0.7751147071242228E+00, 0.6231241133597348E+00, 0.5263220334234422E+00, 0.4481765711163059E+00, 0.4215577615769598E+00, 0.1440581718410880E+00, 0.1296344814538628E+00, 0.5340807481344281E-01, 0.1871489185523312E-01, 0.3434703213692513E-02, 0.2376961512853157E-03, 0.2255238469879073E-22, 0.1711146112720571E-14, 0.5183409008081618E-15, 0.1596820997274643E+01, 0.0000000000000000E+00, 0.1075504130778587E-11, 0.4005497363076952E-12, 0.1536702996895977E+01, 0.1981618593915182E-26, 0.4700742807330894E-10, 0.1164621641126979E-09, 0.1112897445143912E+01, 0.4726602108594632E-24, 0.9459556816770867E-09, 0.1692486098434743E-08, 0.8259900943746278E+00, 0.0000000000000000E+00, 0.4245107391668450E-08, 0.4612248550244559E-08, 0.7751147071242228E+00, 0.8284213834597216E-21, 0.2112319338137411E+02, 0.2112319338137411E+02, 0.1000000000000000E+03, 0.2931972789115646E+01, 0.2000000000000000E+01, 0.1818678155237499E-07, 0.5263220334234422E+00, 0.4246772729828520E-23, 0.1119198730079850E-06, 0.4602199017952358E-06, 0.4481765711163059E+00, 0.1035379938102578E-29, 0.2940687130487931E-04, 0.4282452030972280E-04, 0.4215577615769598E+00, 0.5276311108751726E-20, 0.8277389737129399E-05, 0.6004055084531256E-05, 0.1440581718410880E+00, 0.0000000000000000E+00, 0.8355340113184283E-05, 0.1023986211128489E-04, 0.1296344814538628E+00, 0.1552281967638183E-19, 0.6343521312985942E-05, 0.6697527987420460E-05, 0.5340807481344281E-01, 0.6040133837432384E-17, 0.1556990792688111E-03, 0.3593183434334682E-03, 0.1871489185523312E-01, 0.1153562559317783E-18, 0.8237486079760113E-03, 0.4973426416094711E-03, 0.3434703213692513E-02, 0.0000000000000000E+00, 0.1008741049678482E-02, 0.1888495698860201E-02, 0.2376961512853157E-03, 0.2773339119917620E-31, 0.6098206042824776E-05, 0.1474144332182451E-03},
			info: 0,
		},
		{
			z:    []float64{0.1099031735712288E+00, 0.5724174842808358E+00, 0.3122550743498126E+00, 0.7231231021981217E+00, 0.6175771623314344E+00, 0.5814310193624382E+00, 0.6423493353686718E+00, 0.8309609356975209E+00, 0.6400193102877816E-01, 0.8728840273137624E+00, 0.8682274216885312E+00, 0.7829529646382269E+00, 0.4658588928084396E+00, 0.4997300418007194E+00, 0.5750522308672811E+00, 0.5282811502940782E+00, 0.7877382060309071E+00, 0.2242905591148447E-01, 0.9672168595892823E+00, 0.9229185948074735E+00, 0.7120977511666169E+00, 0.9513613307832994E+00, 0.6495350224590044E+00, 0.1796094611119584E+00, 0.6238202085518583E+00, 0.4056553414984787E-01, 0.2863733551163383E-01, 0.7740758620163646E-01, 0.1652368945937387E+00, 0.5017774327613052E+00, 0.6445125305703996E+00, 0.5670374114090049E-01, 0.1147723121006077E+00, 0.5126855432858525E+00, 0.9526888308153642E+00, 0.1724272641761350E+00, 0.9682495468226523E+00, 0.2579549866163475E+00, 0.5365442422680400E+00, 0.7055455249537391E+00, 0.6510869341601188E+00, 0.6910931676909811E+00, 0.8245356137796966E+00, 0.2467134555988069E+00, 0.2932965007679977E-01, 0.3174819406165880E+00, 0.1008447548759832E+00, 0.7998426243165985E+00, 0.6780870841109060E+00, 0.2896057920811994E+00, 0.9303619290394473E+00, 0.3607716194531815E+00, 0.5211419140438559E+00, 0.1046120770241560E+00, 0.7683871590101677E+00, 0.2147990043395044E+00, 0.6702055898403114E+00, 0.2446830800671760E+00, 0.1275992810549672E+00, 0.9234458715873217E+00, 0.6974016894347440E+00, 0.2461420250226486E+00, 0.2448954779360711E+00, 0.1956976010946661E+00, 0.8608131397075515E+00, 0.7387865940798233E-01, 0.5053692911917109E+00, 0.3354069180988316E+00, 0.4024076719528004E+00, 0.5167351886997615E+00, 0.7829885112810930E-01, 0.3735275576161866E+00, 0.9546718920527260E+00, 0.5321377922787256E+00, 0.6121034951800618E+00, 0.8028222695558183E+00, 0.3295254582685344E+00, 0.5907471536091985E+00, 0.8247143229089846E+00, 0.1329486496858557E+00, 0.9484305946158008E+00, 0.4510190965747323E+00, 0.1159598040227069E+00, 0.7290524870797256E+00},
			n:    21,
			zOut: []float64{0.2759844065448739E+01, 0.2469029941309663E+01, 0.2170232676307031E+01, 0.1960369893365743E+01, 0.1773994559660628E+01, 0.1654180064160601E+01, 0.1454572475268221E+01, 0.1398795476174613E+01, 0.1260788731803824E+01, 0.1011169812035994E+01, 0.9032283997666860E+00, 0.7344998632955592E+00, 0.6915613254033124E+00, 0.3097922830021908E+00, 0.2074097203460417E+00, 0.1750683611049406E+00, 0.1335859329381489E+00, 0.1070304609992355E+00, 0.6466361262660925E-01, 0.1024517112836532E-01, 0.4658519939343828E-03, 0.5916456789157589E-30, 0.2414839775472174E-07, 0.1563475838564645E-07, 0.1454572475268221E+01, 0.5710570490785520E-12, 0.1598956453701674E-05, 0.9938094799470723E-06, 0.1398795476174613E+01, 0.7078291130367095E-25, 0.1301105456376455E-06, 0.7650088290739393E-07, 0.1260788731803824E+01, 0.6660939091008025E-19, 0.3865722666202593E-07, 0.5454440035759107E-07, 0.1011169812035994E+01, 0.5414223942436753E-17, 0.3344051754361652E-04, 0.2635566037358100E-04, 0.9032283997666860E+00, 0.5272968833424410E-16, 0.2125052867814009E+02, 0.2125052867814008E+02, 0.1090000000000000E+03, 0.3179138321995465E+01, 0.2752293577981651E+01, 0.8667028517906969E-06, 0.6915613254033124E+00, 0.6016648435007899E-23, 0.2921486269254270E-05, 0.9773987035049091E-05, 0.3097922830021908E+00, 0.6500213859021137E-26, 0.6786203908623285E-05, 0.1231548092049969E-04, 0.2074097203460417E+00, 0.0000000000000000E+00, 0.7819652657144090E-06, 0.1217347772587842E-05, 0.1750683611049406E+00, 0.6582255393164123E-26, 0.1458540666485106E-06, 0.3789967922095048E-07, 0.1335859329381489E+00, 0.1232595164407831E-31, 0.4615373247666497E-05, 0.3177332815969559E-05, 0.1070304609992355E+00, 0.5947376905467984E-20, 0.9914288367235413E-05, 0.1492808588608919E-04, 0.6466361262660925E-01, 0.5865227530883812E-22, 0.3814164765172745E-05, 0.1095283410948310E-05, 0.1024517112836532E-01, 0.3761300629007363E-24, 0.3159730783704485E-04, 0.5907442037524074E-05, 0.4658519939343828E-03, 0.2094421310984266E-20, 0.1426408794386696E-02, 0.1350744215392459E-02},
			info: 0,
		},
		{
			z:    []float64{0.1337238926382778E+00, 0.3106335653974364E+00, 0.8951406789337562E+00, 0.2282076226383426E+00, 0.3559900270036094E+00, 0.1333708564991680E+00, 0.9225357561814306E+00, 0.6383337981459615E+00, 0.7455817572577521E+00, 0.9604635037498299E-01, 0.1362822625756884E+00, 0.7894982141438661E+00, 0.5371956247017955E+00, 0.2757551089818502E+00, 0.9298698100988576E+00, 0.8755382224220718E+00, 0.2542040210765101E+00, 0.1287904754285962E+00, 0.6810846780454294E+00, 0.6526638813043742E+00, 0.5955845877046446E+00, 0.4670096896697419E+00, 0.5475207984291373E+00, 0.6684162034597571E+00, 0.5235410539200527E+00, 0.7337617639827179E+00, 0.8903194008545992E+00, 0.8072217495300122E+00, 0.3881183186356038E+00, 0.9675156751063907E+00, 0.4869149855955625E+00, 0.4861129300486463E+00, 0.5200718178308463E+00, 0.7592798686765068E+00, 0.7809192005748844E+00, 0.9939952535261979E+00, 0.4314053553650160E+00, 0.3880099380329156E+00, 0.9349849386467751E+00, 0.5291872418075704E+00, 0.7655812944336832E+00, 0.7077741471317599E+00, 0.5275519688342463E+00, 0.1415092279196126E+00, 0.1904335663118598E+00, 0.3699272115054363E+00, 0.7445484920377089E+00, 0.8400049615316296E+00, 0.8734700661006028E+00, 0.8683217518669656E+00, 0.8515176384024258E+00, 0.6202080985416659E+00, 0.3604998919630682E+00, 0.2403115969743893E+00, 0.9554672796937618E+00, 0.1693873282831592E+00, 0.5994144523705257E+00, 0.1384422941659869E-01, 0.6596251817370012E+00, 0.5636231004812753E+00, 0.5389166856740800E+00, 0.2607457441602791E+00, 0.5920539595206202E+00, 0.2966504229935458E+00, 0.5816050348468149E+00, 0.9328164339475690E+00, 0.5805507576670368E+00, 0.6771712743677023E+00, 0.9823651336988842E+00, 0.9793282488782439E+00, 0.4667245203343464E+00, 0.7134684364121646E+00, 0.4880671978756712E-01, 0.8383078993270572E+00, 0.7936419849408084E+00, 0.5385800856643647E+00, 0.4634644976924968E+00, 0.3931501879746360E+00, 0.1248524879525928E+00, 0.3152310060096000E+00, 0.9363438862709352E+00, 0.3520470715344234E+00, 0.1429338846554544E+00, 0.1326946626315275E+00},
			n:    21,
			zOut: []float64{0.2657315323020249E+01, 0.2572905823883155E+01, 0.2192663626292596E+01, 0.2071095653426940E+01, 0.2044494413445205E+01, 0.1999166174223334E+01, 0.1788414821296782E+01, 0.1468335212647637E+01, 0.1398203839606685E+01, 0.1347197234417596E+01, 0.1032902470665773E+01, 0.6608948302945201E+00, 0.5911878145202527E+00, 0.5378924504889658E+00, 0.3142273934265400E+00, 0.2842363451796606E+00, 0.2019484989227465E+00, 0.1206949829709270E+00, 0.7426269792367300E-01, 0.2567909008435554E-01, 0.2199972943425967E-02, 0.1467781105980630E-22, 0.1166962122444304E-12, 0.3923264640094201E-13, 0.1788414821296782E+01, 0.4047127357551305E-19, 0.1945479305169457E-09, 0.9334440121571419E-10, 0.1468335212647637E+01, 0.0000000000000000E+00, 0.9722951216797492E-08, 0.1689489155058893E-07, 0.1398203839606685E+01, 0.0000000000000000E+00, 0.7669133926546234E-07, 0.4506074380002258E-07, 0.1347197234417596E+01, 0.5319335283638860E-18, 0.5446647026201709E-11, 0.3163064243829271E-10, 0.1032902470665773E+01, 0.4430813007685208E-16, 0.2338591866968101E+02, 0.2338591866968102E+02, 0.1020000000000000E+03, 0.3054421768707483E+01, 0.2941176470588236E+01, 0.1546723845733498E-03, 0.5911878145202527E+00, 0.3652829585034170E-19, 0.1702044288822486E-03, 0.2177596193354961E-03, 0.5378924504889658E+00, 0.1350864780951217E-19, 0.2554902108144373E-06, 0.3466691719880105E-07, 0.3142273934265400E+00, 0.2387932220615686E-11, 0.6740109938598231E-04, 0.1019437079103270E-03, 0.2842363451796606E+00, 0.4338734978715565E-28, 0.4438729696644533E-03, 0.6983868198931385E-03, 0.2019484989227465E+00, 0.2612058427213185E-17, 0.1559412966194519E-03, 0.4467949014161081E-04, 0.1206949829709270E+00, 0.7367960854764250E-27, 0.5572085070843593E-03, 0.6756387538188178E-03, 0.7426269792367300E-01, 0.4936220984925504E-18, 0.5280924471496804E-02, 0.4881438130898638E-02, 0.2567909008435554E-01, 0.4675428386128583E-22, 0.7885638578366546E-02, 0.6344653647355715E-02, 0.2199972943425967E-02, 0.1043187069080892E-15, 0.1144606019330652E-02, 0.5299576363060843E-02},
			info: 0,
		},
		{
			z:    []float64{0.2890643798273074E-01, 0.8912209063965303E+00, 0.7091793456137876E+00, 0.8245039023218966E+00, 0.5191833323605244E+00, 0.9894484508239253E+00, 0.5658659268235589E-01, 0.1289734762146678E+00, 0.3341027001588562E+00, 0.6587235353882195E+00, 0.7097532970603662E+00, 0.5025058684824796E+00, 0.5130835936402325E+00, 0.6692657853529054E+00, 0.2879265169895939E+00, 0.8181258306911487E-01, 0.2380388408780171E+00, 0.1056636973917433E+00, 0.4227669640232596E+00, 0.8862610364887391E+00, 0.3174262171725216E+00, 0.4110634725577023E+00, 0.6279625314936111E+00, 0.2325114979496041E+00, 0.5638889949786475E+00, 0.7596856267416864E+00, 0.8240081240110467E+00, 0.9973320050910897E+00, 0.5587421997876323E+00, 0.4288334013103122E+00, 0.8314043340979348E+00, 0.5650188659610876E+00, 0.2964679109455457E+00, 0.8407175605087841E+00, 0.1058970658656445E+00, 0.6883026296612506E+00, 0.9166830669002658E+00, 0.7992763288082885E+00, 0.5373538962118501E+00, 0.8167960120885054E-01, 0.9998997916538879E+00, 0.1975674901149639E+00, 0.7582212604798201E+00, 0.2860808530483561E+00, 0.5862675745609662E+00, 0.4334284293442484E+00, 0.9545191575190999E+00, 0.5542276060509370E+00, 0.9135493345204525E+00, 0.9597250519108470E+00, 0.4020434518393716E+00, 0.2147209304773570E+00, 0.8396072187357826E+00, 0.5371494282057584E+00, 0.9930506172699961E+00, 0.3818967451326610E+00, 0.8312670748053316E+00, 0.7518558266385443E+00, 0.7897281313428761E+00, 0.4898691976861856E+00, 0.9259118998644267E+00, 0.2411762039549979E+00, 0.3003769542607895E+00, 0.9868194969589331E+00, 0.1841003538786936E+00, 0.1211293694641126E+00, 0.7968556025054743E+00, 0.5500013093928259E+00, 0.1705581246551358E-01, 0.5695318221187962E-02, 0.1626907733376057E+00, 0.3178865594423397E-01, 0.6489383021126400E+00, 0.3661200730578906E-01, 0.9937238435410299E+00, 0.8693948290233534E+00, 0.8108974262339508E-01, 0.4858895290851201E+00, 0.5417752479164495E+00, 0.3742432440371155E+00, 0.4593696265617863E+00, 0.7058043658513553E+00, 0.3505553117224474E+00, 0.9647100571711842E+00},
			n:    21,
			zOut: []float64{0.2698643664756728E+01, 0.2430046536911446E+01, 0.2385739474636754E+01, 0.2028138720182182E+01, 0.1901894919230603E+01, 0.1736372633374239E+01, 0.1325608228839813E+01, 0.1300504895564844E+01, 0.1261345481029282E+01, 0.1136052475360616E+01, 0.1031365869480280E+01, 0.7689316276636458E+00, 0.5233321832080760E+00, 0.4817169321317593E+00, 0.3418348106633131E+00, 0.2652183763191593E+00, 0.1871042345023881E+00, 0.9160826173360032E-01, 0.4317601760647842E-01, 0.3303497213448481E-02, 0.1231458285303120E-03, 0.9226440439387837E-18, 0.2555628409159789E-14, 0.2440793084913596E-14, 0.1325608228839813E+01, 0.0000000000000000E+00, 0.5589422112493051E-13, 0.3085898188945034E-13, 0.1300504895564844E+01, 0.7444625512849138E-23, 0.1759654299048522E-12, 0.9781103208866350E-13, 0.1261345481029282E+01, 0.6802337789204302E-18, 0.3650786119252150E-16, 0.3306223266172732E-17, 0.1136052475360616E+01, 0.9063201483067583E-22, 0.4434055300668524E-09, 0.2614666143660316E-09, 0.1031365869480280E+01, 0.2213258132402469E-23, 0.2194206198623719E+02, 0.2194206198623719E+02, 0.1100000000000000E+03, 0.3090702947845805E+01, 0.2727272727272727E+01, 0.1493749053307339E-05, 0.5233321832080760E+00, 0.6795157417200880E-17, 0.4382394072074649E-04, 0.6871024518289093E-04, 0.4817169321317593E+00, 0.6227468015696590E-20, 0.2332676874935389E-03, 0.1644354889791431E-03, 0.3418348106633131E+00, 0.8455074291031222E-24, 0.5211421342985135E-03, 0.2133578004049512E-03, 0.2652183763191593E+00, 0.4572942646937810E-19, 0.1472671843690071E-04, 0.1084821769416695E-04, 0.1871042345023881E+00, 0.4930380657631324E-31, 0.2208706795255935E-03, 0.1179477310012027E-03, 0.9160826173360032E-01, 0.2210123199165121E-23, 0.1806566548732391E-03, 0.1927419907906541E-03, 0.4317601760647842E-01, 0.5416053917983313E-23, 0.1030445327331746E-02, 0.1902956416897231E-02, 0.3303497213448481E-02, 0.8065145389241560E-19, 0.2031248978330205E-02, 0.9946626682395031E-03, 0.1231458285303120E-03, 0.1613264727387662E-23, 0.1026826157383968E-01, 0.7685291223570192E-02},
			info: 0,
		},
		{
			z:    []float64{0.9180725347208115E+00, 0.2301506372544524E+00, 0.2276838071944142E+00, 0.3342400207211600E+00, 0.9639463843013925E+00, 0.5437724527901152E+00, 0.1580993234156807E+00, 0.7877490287014776E+00, 0.3507533849023979E+00, 0.8833112450245699E+00, 0.3185731691946991E-01, 0.4118134066759526E+00, 0.8069686780110494E+00, 0.8681061806796153E-01, 0.6859948674552803E+00, 0.3663178920382139E+00, 0.3720291416615038E+00, 0.8903287156706564E+00, 0.1324429304812152E+00, 0.1008541223140741E+00, 0.8373883396136816E-01, 0.6715628313581425E+00, 0.1594467086953334E+00, 0.5413131938455139E+00, 0.1818140526389971E+00, 0.8685248287187100E+00, 0.5132298455242696E+00, 0.6886095101515632E+00, 0.4005049001067575E+00, 0.6084237383755163E+00, 0.7877676299774720E+00, 0.7102138904901485E+00, 0.6423372229000449E+00, 0.5613660527307726E+00, 0.9158966894941798E+00, 0.4933711427035579E+00, 0.7338439214039514E+00, 0.9908978412238557E+00, 0.5798575644928616E-01, 0.8977291910051136E+00, 0.9253772304099678E+00, 0.3904911857418987E+00, 0.4422593302764081E+00, 0.8489177311215401E+00, 0.7212670509555793E+00, 0.9615617648004515E+00, 0.8226716687395509E+00, 0.9612871412036802E+00, 0.1291974168797438E-01, 0.4488531482063538E+00, 0.2469769340642032E+00, 0.3614360453900619E+00, 0.2074351009622584E+00, 0.7016685284881871E+00, 0.1548060648394445E-01, 0.7311687113133817E+00, 0.9296728393442250E+00, 0.9396166536686752E+00, 0.8501954423645642E+00, 0.4336964896981472E+00, 0.6496017727538979E+00, 0.4464862127725214E+00, 0.4878335970157573E+00, 0.3486784051750773E+00, 0.7455411052079824E+00, 0.5444043691163374E+00, 0.3832092245164885E+00, 0.2030848069834680E+00, 0.9785068556455849E+00, 0.4981273340936699E+00, 0.2968547769748885E+00, 0.8139137104995479E+00, 0.5979876314385963E+00, 0.7890959952347512E+00, 0.1396638856742113E+00, 0.6423356747667288E+00, 0.4204101650717844E+00, 0.3861511203891811E+00, 0.5255853602620404E+00, 0.8105960955427686E+00, 0.1919308380441287E-01, 0.5308532240470543E-01, 0.3836913823972644E+00, 0.5510156269652182E+00},
			n:    21,
			zOut: []float64{0.2573005090179951E+01, 0.2214810120454235E+01, 0.1922733254389488E+01, 0.1915272191491466E+01, 0.1796031325760082E+01, 0.1645710570695390E+01, 0.1570469265391087E+01, 0.1354841927194508E+01, 0.1294466452193999E+01, 0.1181212352080131E+01, 0.1084541549607215E+01, 0.7996017680870424E+00, 0.7671322760534200E+00, 0.5413194973828868E+00, 0.4525283650095193E+00, 0.2478901170080125E+00, 0.2160828479985811E+00, 0.1019188530491480E+00, 0.3154199687907010E-01, 0.6039114283538317E-02, 0.2585297601905087E-05, 0.1842779074596284E-25, 0.3846373205279405E-07, 0.1879918241657939E-07, 0.1570469265391087E+01, 0.5301607341005493E-20, 0.5291292043249480E-06, 0.9680604951014117E-06, 0.1354841927194508E+01, 0.4298088920574052E-25, 0.7835178780587823E-04, 0.5534134534297650E-04, 0.1294466452193999E+01, 0.1009741958682895E-27, 0.1198198722705974E-04, 0.2465537083987539E-04, 0.1181212352080131E+01, 0.6309019138365174E-17, 0.1614759348306573E-06, 0.9017894254302745E-07, 0.1084541549607215E+01, 0.2228464273087314E-11, 0.2171715152048637E+02, 0.2171715152048638E+02, 0.1110000000000000E+03, 0.3170068027210884E+01, 0.4504504504504505E+01, 0.2430910851731611E-05, 0.7671322760534200E+00, 0.2706512346053632E-23, 0.6377722122869933E-05, 0.4589261521941547E-05, 0.5413194973828868E+00, 0.6745076284001739E-23, 0.1892848367477632E-04, 0.2486261833151780E-04, 0.4525283650095193E+00, 0.4943696629711454E-24, 0.8902582280300076E-04, 0.6878116767917398E-04, 0.2478901170080125E+00, 0.2843433355651033E-22, 0.3448699135776589E-03, 0.2736188045956721E-03, 0.2160828479985811E+00, 0.6002787666751380E-18, 0.2577148216451160E-04, 0.6049628369117275E-04, 0.1019188530491480E+00, 0.1341063538875720E-28, 0.1448550797531803E-02, 0.1280708514177034E-02, 0.3154199687907010E-01, 0.6162975822039155E-32, 0.1183522807025889E-02, 0.1098677799367788E-02, 0.6039114283538317E-02, 0.2067951531379678E-24, 0.3575820084217755E-02, 0.2195872556012757E-02, 0.2585297601905087E-05, 0.1137373342260413E-23, 0.1063898509008824E-02, 0.9861644892506420E-03},
			info: 0,
		},
		{
			z:    []float64{0.9985296910189717E+00, 0.8831740589213332E+00, 0.2324113848518039E+00, 0.7533238559688371E+00, 0.3064992106368385E+00, 0.3932363212439300E+00, 0.5064743635322475E+00, 0.2720810167440399E+00, 0.1759478514683438E+00, 0.1390889491193166E+00, 0.1132483424706631E+00, 0.7690628781223635E+00, 0.9618570638759271E+00, 0.5440138404508711E+00, 0.4137500693527351E+00, 0.1092888165750733E+00, 0.3873107100859553E+00, 0.6511277223354393E+00, 0.2155123791160785E-02, 0.9398729888067584E+00, 0.6720352720616287E+00, 0.1535603342192503E+00, 0.1531061200685264E+00, 0.7545904619621174E+00, 0.7663253741916968E+00, 0.9523625338296001E+00, 0.7564452546028717E+00, 0.3871338859213032E+00, 0.8414924166180344E+00, 0.9656526120147547E+00, 0.8983142486291256E+00, 0.4931225086134777E+00, 0.2402429523967143E+00, 0.2566457485714786E+00, 0.6291482026295649E+00, 0.3102713820139572E+00, 0.8907835976003569E+00, 0.7576089622962475E+00, 0.4268794706004859E+00, 0.9679759265393412E+00, 0.8761096912299335E+00, 0.8367482459498927E+00, 0.6924039391525048E+00, 0.1390797616032549E+00, 0.4788493402292462E+00, 0.4619292003991998E+00, 0.6108276748986761E+00, 0.4236594210420574E-01, 0.1642264441572165E+00, 0.3019681501044491E+00, 0.6872347566608095E-01, 0.4797791709329366E+00, 0.3191437336153862E+00, 0.4231397520665843E+00, 0.9123119461602809E+00, 0.4032629658718287E+00, 0.3761450089120290E+00, 0.1152232563042578E+00, 0.9954630682646954E+00, 0.5373824202779058E+00, 0.6106178691724040E-01, 0.8364420731639322E-01, 0.8933845816890847E+00, 0.5581653322395922E+00, 0.2899731736068278E-01, 0.1770979452363362E+00, 0.5004901441174880E-01, 0.7090935149513330E+00, 0.6196074778714252E+00, 0.7620558730426811E+00, 0.3663579087238161E+00, 0.7799033559850937E+00, 0.5196335588513137E+00, 0.2674625162438964E+00, 0.2943854187199277E+00, 0.7770221064689762E+00, 0.1654873860303906E+00, 0.1664508117665333E+00, 0.4846990191351895E+00, 0.8900931539882498E+00, 0.1193560423223905E-01, 0.7974122776570580E+00, 0.1940680748090599E+00, 0.6554519288132757E+00},
			n:    21,
			zOut: []float64{0.2834962035267250E+01, 0.2457739633394971E+01, 0.2412615740112653E+01, 0.2205350258735166E+01, 0.2102070731548960E+01, 0.1691268502125131E+01, 0.1682341944223941E+01, 0.1365229052855181E+01, 0.1159748452086257E+01, 0.1104152264055237E+01, 0.9082896055849982E+00, 0.8350561719565637E+00, 0.6006322679527780E+00, 0.5944104151258589E+00, 0.3054714512857917E+00, 0.1934898777304675E+00, 0.1070106456037005E+00, 0.7977404013852835E-01, 0.4467492857453391E-01, 0.1796045849894196E-01, 0.1273912617121224E-04, 0.5392013224803094E-11, 0.7047397633000775E-04, 0.4975950204285101E-04, 0.1682341944223941E+01, 0.0000000000000000E+00, 0.1776257168565356E-04, 0.2478687660984077E-04, 0.1365229052855181E+01, 0.2904595710528663E-17, 0.2723859519581362E-08, 0.9106491605280572E-08, 0.1159748452086257E+01, 0.1011100673149366E-15, 0.1332628661445848E-04, 0.5871026575773219E-05, 0.1104152264055237E+01, 0.8308334390451641E-18, 0.8332699889408906E-05, 0.9212653513444334E-05, 0.9082896055849982E+00, 0.3518078502828044E-18, 0.2270226121598308E+02, 0.2270226121598308E+02, 0.1240000000000000E+03, 0.3505668934240363E+01, 0.5645161290322581E+01, 0.2479678732743906E-05, 0.6006322679527780E+00, 0.0000000000000000E+00, 0.7879659138115500E-06, 0.7199236433698600E-06, 0.5944104151258589E+00, 0.3024585909035025E-20, 0.8203337497349541E-06, 0.8765068833865939E-06, 0.3054714512857917E+00, 0.4733165431326071E-26, 0.3737214231038995E-06, 0.3923468144379782E-06, 0.1934898777304675E+00, 0.4260044778133448E-20, 0.3232147542049518E-06, 0.3376352641996194E-06, 0.1070106456037005E+00, 0.3308722450212072E-23, 0.5070207875063323E-06, 0.4107488228647395E-06, 0.7977404013852835E-01, 0.6162975822039155E-32, 0.1529214528521930E-08, 0.6332986682625979E-08, 0.4467492857453391E-01, 0.1074272623509048E-20, 0.2583088473438835E-04, 0.1663016813733052E-04, 0.1796045849894196E-01, 0.7053209140113412E-22, 0.1477449551257491E-03, 0.9547349784916723E-04, 0.1273912617121224E-04, 0.0000000000000000E+00, 0.2066074227952428E-02, 0.2903362394245088E-02},
			info: 0,
		},
		{
			z:    []float64{0.9711900205601530E+00, 0.1638861523026915E+00, 0.1833187318057834E+00, 0.9072755753374127E+00, 0.2283578849017619E+00, 0.7520216236020175E+00, 0.3999380553956507E+00, 0.2624503967963340E+00, 0.4831766333071612E+00, 0.5439216550253211E+00, 0.6414566703448968E+00, 0.4958417903972772E+00, 0.6484413600580574E+00, 0.7544273119845675E+00, 0.7061059155531373E+00, 0.9606326194792012E+00, 0.5067069535242709E+00, 0.3029510615023524E+00, 0.6987030441702768E+00, 0.9783205951302908E+00, 0.1548952654742521E+00, 0.9315036439248046E+00, 0.5516706717897903E+00, 0.5168328579759034E+00, 0.3596295330063849E+00, 0.4476574932894306E-01, 0.9741728325615778E+00, 0.2804678894509116E-01, 0.3902916772468864E+00, 0.8530466945184888E+00, 0.2969904800916723E-01, 0.1860020491995459E+00, 0.7523094177753908E-01, 0.1049676044274535E+00, 0.9260152605248726E+00, 0.4316992548719201E+00, 0.7469073285915341E+00, 0.9833774639100495E+00, 0.4974597252506658E+00, 0.4177634673241238E+00, 0.2822911131407428E+00, 0.8441017127353869E+00, 0.3634452570924520E+00, 0.3139202807660985E+00, 0.3006600055674361E+00, 0.3009046530704764E+00, 0.5749841775931973E+00, 0.8312625997674838E+00, 0.7353697939061690E+00, 0.5347208541042444E+00, 0.3745659295783104E+00, 0.2530774659476339E+00, 0.4186076940004666E+00, 0.9926397860987635E+00, 0.9744627902125954E+00, 0.5156862870399348E-01, 0.9609885826474012E+00, 0.1340044668459911E+00, 0.4269295726396694E+00, 0.1323606439861225E+00, 0.9547002490565327E+00, 0.3235500895339022E+00, 0.9408694232081565E+00, 0.5256315334453321E-01, 0.5435043012268493E+00, 0.8290694048290136E+00, 0.3690485876123201E+00, 0.1315626820984572E+00, 0.7782838224801890E+00, 0.4824080201008953E+00, 0.4904940544366522E+00, 0.7125669967024466E+00, 0.2079010288312985E+00, 0.3154779356788950E+00, 0.8039212621301359E+00, 0.1606576564727077E+00, 0.9470564071979923E+00, 0.9189825217362574E+00, 0.4332760074849629E+00, 0.2823180730446854E+00, 0.8869162317432342E+00, 0.9632623909785668E+00, 0.5010388463829042E+00, 0.4287967421178640E+00},
			n:    21,
			zOut: []float64{0.2578695081979337E+01, 0.2397185171350384E+01, 0.2122428254475021E+01, 0.1845603260474165E+01, 0.1680081614592041E+01, 0.1588339204725779E+01, 0.1309467974766432E+01, 0.1299839243529426E+01, 0.1209324813016549E+01, 0.1078225499420584E+01, 0.1021574489264402E+01, 0.8212075951361281E+00, 0.6252200086865959E+00, 0.5691142568406321E+00, 0.4167987738801949E+00, 0.2624912719101158E+00, 0.1530089161378868E+00, 0.6867147881453296E-01, 0.2505166907249862E-01, 0.2349256929716605E-02, 0.7151879759250424E-03, 0.1577721810442024E-29, 0.4511324342265551E-10, 0.2432911773654117E-10, 0.1309467974766432E+01, 0.0000000000000000E+00, 0.2950215161815511E-09, 0.4606013118502744E-09, 0.1299839243529426E+01, 0.0000000000000000E+00, 0.1717787641934460E-08, 0.1104907179077728E-08, 0.1209324813016549E+01, 0.2227112107619961E-25, 0.8047601721100021E-14, 0.3064349892439287E-13, 0.1078225499420584E+01, 0.1519533811285116E-12, 0.1082897652895214E-09, 0.2849698526408556E-09, 0.1021574489264402E+01, 0.9160719227198137E-18, 0.2107539302297835E+02, 0.2107539302297835E+02, 0.1110000000000000E+03, 0.3317460317460318E+01, 0.3603603603603604E+01, 0.1689224645784323E-04, 0.6252200086865959E+00, 0.1615587133892632E-26, 0.8276137867398754E-04, 0.1155304449854771E-03, 0.5691142568406321E+00, 0.1852884571782062E-20, 0.2098842504693311E-03, 0.1630830816462844E-03, 0.4167987738801949E+00, 0.7013543182763534E-16, 0.5750632050789642E-07, 0.3137720527277731E-06, 0.2624912719101158E+00, 0.1295300917428640E-14, 0.8049639874935534E-04, 0.1584688939577041E-03, 0.1530089161378868E+00, 0.3636550670570672E-17, 0.1206768167404478E-03, 0.2128462525136164E-03, 0.6867147881453296E-01, 0.4176548246217679E-19, 0.3626791442965051E-04, 0.6269592722343766E-04, 0.2505166907249862E-01, 0.8628166150854817E-31, 0.1144078296038157E-02, 0.1062123559861287E-02, 0.2349256929716605E-02, 0.1853343815383365E-22, 0.1348818609641394E-03, 0.1209095310981735E-04, 0.7151879759250424E-03, 0.2331746618088476E-23, 0.3639492207606013E-03, 0.1002254703741813E-03},
			info: 0,
		},
		{
			z:    []float64{0.5145122753774686E+00, 0.7453791781450037E+00, 0.4937919697078275E-01, 0.9600912461125237E+00, 0.5558575009096904E+00, 0.9202496102671529E+00, 0.1803962777705234E+00, 0.6732582413383450E+00, 0.3646300420523177E+00, 0.4351523227625338E+00, 0.5258615952095320E+00, 0.2618724553334706E-01, 0.6600255279090639E+00, 0.4323183918305084E+00, 0.2184526174816337E+00, 0.7758998738943313E+00, 0.3700044944947983E+00, 0.9075309916724112E+00, 0.7873590137849384E-02, 0.6422780210226944E+00, 0.7012374235504933E+00, 0.3225317189575561E-01, 0.9626413579028830E+00, 0.2957325301869758E+00, 0.9393912496594896E+00, 0.3122772343061888E+00, 0.3372582329155205E-01, 0.6694758703662361E+00, 0.5160501222108982E-03, 0.9689575293997177E-01, 0.3780626069308365E+00, 0.1243877469635841E+00, 0.3753610862140089E+00, 0.7686454249086220E+00, 0.8958999262270639E+00, 0.9484254659678725E+00, 0.7229120985603210E-01, 0.7282046499931183E+00, 0.1602916509824523E+00, 0.3708722215113680E+00, 0.5328491413896952E+00, 0.7547173918812869E+00, 0.4313535833192655E+00, 0.5849601017371611E+00, 0.5030126650939655E+00, 0.3345267020968724E+00, 0.8264894134567216E+00, 0.3997968176112487E+00, 0.7977852660508306E+00, 0.3214021056175872E+00, 0.4636510192825659E+00, 0.6796405156053968E+00, 0.6279823358542748E+00, 0.1573806073104234E-01, 0.6382889890358745E-01, 0.5417145287154319E+00, 0.2343304174607242E+00, 0.7662339681589646E+00, 0.6291880658725347E+00, 0.5139706179483768E+00, 0.1583956600060378E+00, 0.9391868842090931E+00, 0.2645051239078070E+00, 0.7814123021217941E+00, 0.8945112708258808E+00, 0.7464898217422987E+00, 0.3770460515219444E+00, 0.3622886345531428E+00, 0.2208273216958112E-01, 0.7102940678247979E+00, 0.2735151315490909E+00, 0.4039400600381490E+00, 0.9317363313474538E+00, 0.5754441557390587E+00, 0.5886999087747160E+00, 0.3030246870939485E+00, 0.8507721951782303E+00, 0.3043440840669203E+00, 0.6021496993262363E+00, 0.4505272689007913E+00, 0.3874595231430722E+00, 0.2566155965014733E+00, 0.2521494866083717E+00, 0.7091848125152684E+00},
			n:    21,
			zOut: []float64{0.2432728498446405E+01, 0.2127807422546525E+01, 0.1810603620342434E+01, 0.1754551066335428E+01, 0.1442239744319712E+01, 0.1358471286041730E+01, 0.1268104094970585E+01, 0.1166598518949770E+01, 0.1142749842956014E+01, 0.8038795081997708E+00, 0.7842646316799070E+00, 0.7340488722153766E+00, 0.6406325830516326E+00, 0.6326463399427553E+00, 0.4829120798428922E+00, 0.4021501496449236E+00, 0.3628067219099092E+00, 0.1315077965186685E-01, 0.3447035773652165E-02, 0.9829838131645125E-03, 0.5542446906743292E-07, 0.3726944967513355E-18, 0.7775649827661083E-15, 0.4707679240410040E-15, 0.1268104094970585E+01, 0.4204313080465905E-25, 0.2081271826241763E-14, 0.3549560565109884E-14, 0.1166598518949770E+01, 0.0000000000000000E+00, 0.3900682806423283E-13, 0.2423637210496787E-13, 0.1142749842956014E+01, 0.1408856084028437E-16, 0.9605918786201080E-13, 0.6043593726409546E-13, 0.8038795081997708E+00, 0.8160364745046778E-12, 0.9063031760991450E-12, 0.1255592191298957E-11, 0.7842646316799070E+00, 0.2274746684496083E-23, 0.1936477583605892E+02, 0.1936477583605892E+02, 0.1150000000000000E+03, 0.3673469387755102E+01, 0.7826086956521739E+01, 0.1055105767279683E-11, 0.6406325830516326E+00, 0.7534418653829021E-12, 0.2026915084594321E-10, 0.1938443898507311E-10, 0.6326463399427553E+00, 0.7428954317778491E-22, 0.1822158222411987E-10, 0.2310105649781206E-10, 0.4829120798428922E+00, 0.6818472407494482E-21, 0.4806621018591031E-08, 0.3430760494577760E-08, 0.4021501496449236E+00, 0.9407638730741502E-13, 0.4934881466368944E-07, 0.7682525823543034E-07, 0.3628067219099092E+00, 0.4641733924717305E-17, 0.3006332568302334E-06, 0.4841414710438772E-06, 0.1315077965186685E-01, 0.0000000000000000E+00, 0.4649166787994157E-03, 0.6126205080519521E-03, 0.3447035773652165E-02, 0.0000000000000000E+00, 0.8654037114492079E-03, 0.1518309118546661E-02, 0.9829838131645125E-03, 0.2948446519354054E-25, 0.2136365904689920E-10, 0.9493111427282873E-09, 0.5542446906743292E-07, 0.0000000000000000E+00, 0.8210503952597113E-02, 0.1866280792966489E-03},
			info: 0,
		},
		{
			z:    []float64{0.1382149566425072E+00, 0.6575474867299980E+00, 0.6348472674088155E+00, 0.2171300387013793E+00, 0.8255967246227341E+00, 0.3006539504418347E+00, 0.2997341048143420E+00, 0.8472159875058289E+00, 0.6738978429295159E+00, 0.5424646868297128E+00, 0.4120014569672862E+00, 0.8343380028244362E+00, 0.3003911043596160E+00, 0.4710045367840567E+00, 0.4862652198104067E-01, 0.7471894777046170E+00, 0.1793049886163456E+00, 0.7023511329140396E+00, 0.7338019696975340E-02, 0.6396982308318109E+00, 0.9907774162255600E-01, 0.9295761484947797E+00, 0.9648060362942152E+00, 0.9280158186031068E+00, 0.2707915909911263E-01, 0.3838735516497237E+00, 0.1682677327524866E+00, 0.2113105759433432E+00, 0.9045861188128749E+00, 0.7471063495001966E+00, 0.8342180430544732E-01, 0.1412979915701199E+00, 0.8279925714974835E+00, 0.1155895711297670E+00, 0.9231783969041454E+00, 0.3728782651185074E+00, 0.6830046202574890E+00, 0.9069008465878121E+00, 0.9754648591267837E+00, 0.8893150707687680E-01, 0.6771275111272185E+00, 0.9913978119103384E+00, 0.7253101374756428E+00, 0.9967636737323063E+00, 0.8627812047783159E+00, 0.3179382291892185E-01, 0.3971194218951257E+00, 0.1929798662828303E+00, 0.8983406738830656E+00, 0.8696986427722838E+00, 0.1626550211323166E+00, 0.5446758738044107E+00, 0.7791897500950196E+00, 0.5495589328822522E+00, 0.2598162756138545E+00, 0.4878822167962611E+00, 0.1063554708684851E+00, 0.9205986422545666E+00, 0.2525019548845375E+00, 0.6202404757545382E+00, 0.4398012123976176E+00, 0.3367811528280380E+00, 0.2846930465290612E+00, 0.3637854092863660E+00, 0.2816012898088578E+00, 0.8253973364224151E+00, 0.1068693035499244E+00, 0.6335234891165292E+00, 0.4457012705011152E+00, 0.3822128202770037E-01, 0.8044679788500431E-02, 0.1644333417433916E+00, 0.8243501026702226E+00, 0.3724822902257573E+00, 0.3284619434713851E+00, 0.8284230195063381E+00, 0.9834160248288059E+00, 0.5390643127606534E+00, 0.4874602694948874E+00, 0.3393327340444893E+00, 0.9478697182248690E+00, 0.9005921446100461E+00, 0.6870582101064761E+00, 0.8753727823211164E+00},
			n:    21,
			zOut: []float64{0.2520205603715150E+01, 0.2467591671122728E+01, 0.2156156465592955E+01, 0.1830603796963098E+01, 0.1688980285573207E+01, 0.1475729352347043E+01, 0.1457403847234258E+01, 0.1216096979409916E+01, 0.1020221539309743E+01, 0.9109792709218910E+00, 0.7786685137433046E+00, 0.6943983149907305E+00, 0.5952732619368740E+00, 0.5747064141061984E+00, 0.4193783470766816E+00, 0.2849493660130012E+00, 0.2000335007304226E+00, 0.1879518797669419E+00, 0.9250201866750386E-01, 0.6720326503698931E-01, 0.2522304426258675E-08, 0.0000000000000000E+00, 0.6146929522129176E-12, 0.1808448588630822E-11, 0.1457403847234258E+01, 0.4890937612370273E-28, 0.1829381779873047E-10, 0.6063479072736104E-11, 0.1216096979409916E+01, 0.3707772472283591E-24, 0.7129364201262979E-09, 0.6841733798178873E-09, 0.1020221539309743E+01, 0.2138261883449577E-21, 0.5257033338238628E-09, 0.5072654023074907E-09, 0.9109792709218910E+00, 0.7953858577580093E-22, 0.4258628064439383E-09, 0.4402488234184115E-09, 0.7786685137433046E+00, 0.2050608176139579E-11, 0.2063903369678094E+02, 0.2063903369678095E+02, 0.1070000000000000E+03, 0.3256235827664399E+01, 0.7476635514018692E+01, 0.2680505535092421E-09, 0.5952732619368740E+00, 0.3490920289236852E-22, 0.2767075275933556E-09, 0.2579359628684292E-09, 0.5747064141061984E+00, 0.6550779105727953E-17, 0.5565909614599560E-09, 0.4340922452181151E-09, 0.4193783470766816E+00, 0.3307688474446419E-20, 0.4344481061547019E-08, 0.2950068756973984E-08, 0.2849493660130012E+00, 0.2114500635677851E-23, 0.2105683914952377E-07, 0.1386409244069912E-07, 0.2000335007304226E+00, 0.2552627671550359E-24, 0.1231406445198996E-06, 0.7724199195397155E-07, 0.1879518797669419E+00, 0.2762617809804602E-19, 0.3259696859933768E-06, 0.5428848805276070E-06, 0.9250201866750386E-01, 0.2465190328815662E-31, 0.5308529636258477E-04, 0.3674504775836459E-04, 0.6720326503698931E-01, 0.4910144116114772E-20, 0.8312610965835068E-03, 0.3682729181838852E-03, 0.2522304426258675E-08, 0.0000000000000000E+00, 0.2850162708346282E-01, 0.1272924532804709E-01},
			info: 0,
		},
		{
			z:    []float64{0.8107097465650837E+00, 0.6284212325752714E-02, 0.8801609468679439E-01, 0.1317925189426976E+00, 0.7816238529622888E+00, 0.5138440552812117E+00, 0.8491910920486024E+00, 0.5354109734403600E-01, 0.9922209940223927E+00, 0.5302203883752287E+00, 0.7634020308181488E-01, 0.1679415943232965E+00, 0.2329711998757648E+00, 0.1414305919938111E+00, 0.1841480420952157E-01, 0.3417588803467335E+00, 0.2555749764808386E+00, 0.5793635436480560E-01, 0.5867939998055902E+00, 0.9699556000711096E+00, 0.2545002439558014E+00, 0.6318762794888357E+00, 0.3917398730545906E+00, 0.5624711052874073E+00, 0.4151753215255505E-01, 0.2323556867603191E+00, 0.8346423132579881E+00, 0.4193569213002444E+00, 0.5070048505400149E+00, 0.9880568155195324E-01, 0.6350898210108906E+00, 0.2370382383359541E+00, 0.3299228061150288E+00, 0.4349239764921071E+00, 0.5262500692361848E+00, 0.5263549923651836E+00, 0.8161052184432745E+00, 0.5568267671289201E+00, 0.7852669932205985E+00, 0.4481607592551184E-01, 0.6290146498792036E+00, 0.9736787804665343E+00, 0.5369941621471508E+00, 0.6750777504837204E+00, 0.9452511173220645E+00, 0.3252402467696922E+00, 0.8149000487070703E+00, 0.8087485664240622E+00, 0.8643101435258340E+00, 0.4683483752594740E+00, 0.5054760923588362E+00, 0.9646412930917844E+00, 0.1611953384510251E+00, 0.6426851941066625E-01, 0.3732266796993424E+00, 0.6384029126799168E+00, 0.3844644469584264E+00, 0.6635273633462982E+00, 0.2553934870916856E+00, 0.7037835913610216E+00, 0.7783327991001192E+00, 0.7381327592761330E+00, 0.2765915485450557E+00, 0.9846188872724937E-01, 0.1992562101416243E+00, 0.2440052895970430E+00, 0.3301477999473963E+00, 0.5338134839029927E+00, 0.5498293871712301E+00, 0.1974401363114953E+00, 0.7407533997655510E+00, 0.6821638940995054E+00, 0.9814777780110351E+00, 0.4891456709117443E+00, 0.6868139367703679E+00, 0.4181531426430654E+00, 0.6623070699739451E+00, 0.9202771282456093E+00, 0.2973080397025486E+00, 0.4403051603022431E+00, 0.1603780540263299E+00, 0.4910562348998989E+00, 0.7838834745165658E+00, 0.1931809149066596E+00},
			n:    21,
			zOut: []float64{0.2163023058341640E+01, 0.1906105992609667E+01, 0.1885682431608721E+01, 0.1629672232824707E+01, 0.1530256904983391E+01, 0.1307296833197682E+01, 0.1220997281181084E+01, 0.9674288853031870E+00, 0.8179252376135864E+00, 0.6735586841320580E+00, 0.6320597056120635E+00, 0.5126868322939253E+00, 0.4828683228163860E+00, 0.4745823594217251E+00, 0.3685025909289511E+00, 0.2947763441663148E+00, 0.8912041525268403E-01, 0.6456597576955210E-01, 0.6259461554584676E-01, 0.1853533814873023E-01, 0.2023108580404890E-03, 0.8671199725506423E-13, 0.1722304485286045E-09, 0.1666294529992108E-09, 0.1220997281181084E+01, 0.1774302413926244E-21, 0.1370178691819834E-09, 0.1399340123848638E-09, 0.9674288853031870E+00, 0.4276620702127187E-22, 0.3173068333548179E-09, 0.2274343348931791E-09, 0.8179252376135864E+00, 0.7888609052210118E-30, 0.5644994943584269E-09, 0.8117631577107350E-09, 0.6735586841320580E+00, 0.2962158340816585E-16, 0.9564613436107998E-08, 0.6775718950029066E-08, 0.6320597056120635E+00, 0.4964115825120276E-16, 0.1710244235260994E+02, 0.1710244235260994E+02, 0.1120000000000000E+03, 0.3265306122448980E+01, 0.6250000000000000E+01, 0.5299633299932952E-07, 0.4828683228163860E+00, 0.2680954403830288E-19, 0.1311205982337445E-05, 0.9790063018353157E-06, 0.4745823594217251E+00, 0.5160286279848935E-24, 0.1890785247579508E-05, 0.1394729630636152E-05, 0.3685025909289511E+00, 0.4693722386065020E-28, 0.6031882082954460E-05, 0.2415384446316590E-05, 0.2947763441663148E+00, 0.1882813323774143E-21, 0.7814092950831008E-04, 0.8799876387761915E-04, 0.8912041525268403E-01, 0.1005797654156790E-28, 0.3017726164846613E-06, 0.1278833151586840E-06, 0.6456597576955210E-01, 0.4635851024992188E-19, 0.2982580055136947E-07, 0.4759731822531787E-07, 0.6259461554584676E-01, 0.4490807166402986E-22, 0.9170799331172770E-06, 0.3025792024297308E-06, 0.1853533814873023E-01, 0.4985600920996795E-27, 0.9814614537388964E-04, 0.3116488226164178E-03, 0.2023108580404890E-03, 0.0000000000000000E+00, 0.9878507322060364E-03, 0.9125548412055403E-03},
			info: 0,
		},
		{
			z:    []float64{0.4943635351465083E+00, 0.1295760812930432E+00, 0.7491711827268356E+00, 0.8583007697336751E+00, 0.8597988452333791E+00, 0.9020619105716389E+00, 0.2915077046929524E+00, 0.2667690273281318E+00, 0.2640508133196529E-01, 0.6592981688042527E+00, 0.5004017812459820E+00, 0.7056211227388411E+00, 0.2254860841977465E+00, 0.8570117342825790E+00, 0.6863131135194549E+00, 0.6029337708611949E+00, 0.6631267757314846E+00, 0.5332167960220966E+00, 0.3047497072553197E+00, 0.3019927731659721E+00, 0.6244714872283929E+00, 0.3204566006732108E+00, 0.9437738905138152E+00, 0.4462876020196550E+00, 0.6693016494515551E+00, 0.5687239407477801E+00, 0.5130358959101252E+00, 0.1634623710980615E+00, 0.6356065394794165E+00, 0.9187737904031696E+00, 0.8612817405760382E+00, 0.1602125514297330E+00, 0.5297426144233635E+00, 0.3699984812614213E+00, 0.3455622529703073E+00, 0.9962779456692529E+00, 0.3564481259772661E+00, 0.2073475342531472E-01, 0.5839325153640170E+00, 0.6856088547326490E+00, 0.1606020836922776E+00, 0.2475573692197572E+00, 0.8169030104559274E+00, 0.1727021335202554E+00, 0.8794153095346040E+00, 0.1052546999951709E-01, 0.4332903851899277E+00, 0.4302943891085251E+00, 0.6360909664440160E+00, 0.1924830187263967E+00, 0.6792019925733094E+00, 0.7328300669715262E+00, 0.8282900281692296E+00, 0.1586820387506954E+00, 0.5999731076117462E+00, 0.4873494502000296E+00, 0.4446262392098149E+00, 0.6430807880043099E-01, 0.4949024386471711E+00, 0.7324156973660632E+00, 0.4547667791304575E+00, 0.7542985454738853E+00, 0.3755946849636591E+00, 0.7135765429513076E+00, 0.7888131931751277E+00, 0.6384460726826821E+00, 0.4685063200928931E+00, 0.1343379262734882E+00, 0.8857603211032115E+00, 0.5798608180272281E+00, 0.8765662925729737E+00, 0.2107645345329755E+00, 0.7032612644639602E-01, 0.2672549085142431E+00, 0.5322118352980683E+00, 0.4697779217142906E-01, 0.1959612687762308E+00, 0.7346288295330818E+00, 0.4373432557584073E+00, 0.2514178299506654E+00, 0.7845430350805238E+00, 0.7823648156079857E+00, 0.6098686444303012E+00, 0.2687771415020881E+00},
			n:    21,
			zOut: []float64{0.2638973127799410E+01, 0.2299941431214051E+01, 0.2280019421450360E+01, 0.2103910562764817E+01, 0.1753406216295798E+01, 0.1675269066003242E+01, 0.1419658234290438E+01, 0.1365959224584141E+01, 0.1187696680558471E+01, 0.1126073702945326E+01, 0.8926412592525027E+00, 0.7715773240739381E+00, 0.5428647577479762E+00, 0.4329885186763779E+00, 0.3589145864992100E+00, 0.2461266947628802E+00, 0.1802035742271595E+00, 0.1083562693061272E+00, 0.7909884021715781E-01, 0.2853616071040153E-01, 0.1859995500862765E-03, 0.5169878828456423E-25, 0.7099412452593561E-13, 0.1129160994681896E-12, 0.1419658234290438E+01, 0.7523863534922662E-17, 0.1645873945742296E-11, 0.1162641799741134E-11, 0.1365959224584141E+01, 0.3171195595439500E-23, 0.6168272840237550E-11, 0.8582511762369885E-11, 0.1187696680558471E+01, 0.8519789992808060E-16, 0.1934844654065141E-10, 0.2568816579510325E-10, 0.1126073702945326E+01, 0.8564308176121538E-22, 0.5690968370753065E-10, 0.7456610067231580E-10, 0.8926412592525027E+00, 0.4500097668303756E-13, 0.2149240165292988E+02, 0.2149240165292987E+02, 0.1030000000000000E+03, 0.2950113378684807E+01, 0.4854368932038835E+01, 0.1146501122497315E-08, 0.5428647577479762E+00, 0.2966117003631004E-27, 0.2887486457338918E-08, 0.3484100742696835E-08, 0.4329885186763779E+00, 0.9478952637135678E-26, 0.8311221870318503E-08, 0.6953572191921854E-08, 0.3589145864992100E+00, 0.2375820297137957E-16, 0.1629548458200623E-07, 0.1368379043565619E-07, 0.2461266947628802E+00, 0.1012567897168591E-16, 0.3168042687452701E-07, 0.3807114536183845E-07, 0.1802035742271595E+00, 0.8688408444571280E-16, 0.7017919615597339E-07, 0.9241973720004823E-07, 0.1083562693061272E+00, 0.3747089299799806E-28, 0.3320058009883818E-06, 0.6266972429387508E-06, 0.7909884021715781E-01, 0.9269115636346889E-29, 0.6452918276099027E-05, 0.1508443954686445E-04, 0.2853616071040153E-01, 0.2499111347740165E-24, 0.3360677328899296E-03, 0.1546059694047028E-03, 0.1859995500862765E-03, 0.1972152263052530E-28, 0.5448826922428762E-03, 0.1724316467181159E-02},
			info: 0,
		},
		{
			z:    []float64{0.5641760158025995E+00, 0.2227931624220032E+00, 0.1040729150876691E+00, 0.7228520011975436E+00, 0.2490915552877161E+00, 0.4463107957028827E+00, 0.6882435259034004E+00, 0.2173580561342540E+00, 0.8146865135594571E+00, 0.3630549548424193E+00, 0.2813912484076728E+00, 0.7560034280086618E+00, 0.7800835186160227E-01, 0.9984316063995433E+00, 0.2057310185040729E+00, 0.4428139729521006E+00, 0.1946530136132755E+00, 0.8953866467149748E+00, 0.5910093704925600E+00, 0.9736056943014803E+00, 0.7342559355497796E+00, 0.5630651274913434E+00, 0.3586901690989497E+00, 0.1859950386974873E-01, 0.4012559193852120E+00, 0.5767984160414075E-01, 0.3379735138652440E+00, 0.6337834884220164E+00, 0.8830566914548466E-01, 0.2985048049064926E+00, 0.9820684138520455E+00, 0.1144932752938616E+00, 0.5905532956519596E+00, 0.4757653978696945E+00, 0.1997612463043563E-01, 0.6356348606671800E+00, 0.8478495728965164E+00, 0.4997410778119411E+00, 0.6861774168985810E+00, 0.6912919871514965E+00, 0.1276032443467088E+00, 0.2695430969348002E+00, 0.9446688115451145E+00, 0.5104603739010690E+00, 0.8863068797328323E+00, 0.3347723354379355E+00, 0.2977917172311930E+00, 0.9464871506253476E+00, 0.4279639349386329E+00, 0.6378279454376028E+00, 0.2778621310060349E+00, 0.6971058794896068E-01, 0.6309805553452471E+00, 0.3447281367207256E-01, 0.8480855191050544E+00, 0.6700112511333863E+00, 0.1616260465745026E+00, 0.7744510887458210E+00, 0.8408544531170117E+00, 0.8444763755373497E-01, 0.3286971575766536E+00, 0.8244867044222992E+00, 0.8036107174439190E+00, 0.4654902702844697E+00, 0.8300602115903392E+00, 0.5911511948645825E+00, 0.5734805041784674E+00, 0.6046912223886362E+00, 0.4400460907177143E+00, 0.4069795298963303E+00, 0.2413714645456244E+00, 0.7838758775665940E+00, 0.1371288591293247E+00, 0.8435169936355635E-01, 0.4867567406491472E+00, 0.2914527168883616E+00, 0.7540094160758792E+00, 0.6792178828893637E+00, 0.8924641252729165E+00, 0.2188072300242774E+00, 0.1899571860965691E+00, 0.7806965639263314E+00, 0.8706240780095753E-01, 0.7170634647198669E+00},
			n:    21,
			zOut: []float64{0.2553381332090116E+01, 0.2166390759033473E+01, 0.1737362408504838E+01, 0.1528419999405254E+01, 0.1507484707859341E+01, 0.1334735890703865E+01, 0.1222641790395049E+01, 0.1186722126839839E+01, 0.1073660052771764E+01, 0.9488489095119069E+00, 0.9141860725715962E+00, 0.7726247731371300E+00, 0.6075856291402457E+00, 0.5409244297850613E+00, 0.4230641503671385E+00, 0.2356839373494127E+00, 0.1583434617720056E+00, 0.3187519618749328E-01, 0.2747107248999187E-01, 0.1480095837386152E-02, 0.5569185181399456E-04, 0.3215260779752862E-17, 0.2284971140234050E-13, 0.6915955898032105E-13, 0.1222641790395049E+01, 0.9860761315262648E-31, 0.9831000121605633E-10, 0.3802561298586086E-10, 0.1186722126839839E+01, 0.2066815571679051E-27, 0.6142487424703478E-09, 0.1524532000885779E-08, 0.1073660052771764E+01, 0.3155443620884047E-29, 0.1933670198753400E-07, 0.4311109498288097E-07, 0.9488489095119069E+00, 0.6613963504021090E-17, 0.7125410558010122E-07, 0.6137903750393884E-07, 0.9141860725715962E+00, 0.2713620929442760E-22, 0.1897294248760472E+02, 0.1897294248760472E+02, 0.1070000000000000E+03, 0.3233560090702948E+01, 0.4672897196261682E+01, 0.2313496546014482E-05, 0.6075856291402457E+00, 0.2008094738046662E-26, 0.6354651604911740E-04, 0.4633045629030685E-04, 0.5409244297850613E+00, 0.2465190328815662E-30, 0.8082164119218066E-03, 0.5151838483992701E-03, 0.4230641503671385E+00, 0.3299331849996360E-25, 0.9881103085087938E-03, 0.1036550305023510E-02, 0.2356839373494127E+00, 0.2111622871095604E-25, 0.1203513003711909E-02, 0.1300398061184096E-02, 0.1583434617720056E+00, 0.1737664936697394E-23, 0.8260928074113359E-03, 0.9484297179228856E-03, 0.3187519618749328E-01, 0.0000000000000000E+00, 0.1785275423606748E-03, 0.4485462762371783E-03, 0.2747107248999187E-01, 0.8804949879714635E-25, 0.5200632154389635E-08, 0.3234666977722410E-08, 0.1480095837386152E-02, 0.5399595124056782E-24, 0.5661901580915454E-02, 0.5860254353052324E-02, 0.5569185181399456E-04, 0.0000000000000000E+00, 0.1190560893471898E-01, 0.9131404251707014E-02},
			info: 0,
		},
		{
			z:    []float64{0.7290518158132969E+00, 0.5196659176766039E-03, 0.6083923160589086E+00, 0.3459319602615154E+00, 0.3277021139736517E+00, 0.6020592719504614E-01, 0.4608784204502808E+00, 0.8430478676154098E+00, 0.3856212930081542E+00, 0.6602823264317716E+00, 0.4451601253283373E+00, 0.4130313322309198E-01, 0.7414750648249350E+00, 0.1630235543068703E-01, 0.6004925370465983E+00, 0.6953287054569501E+00, 0.7809046354974044E+00, 0.7892746585788727E+00, 0.8348704064067982E+00, 0.5999411739898802E+00, 0.5502575606089829E+00, 0.4392206918044746E+00, 0.7984758158185817E+00, 0.9084185319619590E+00, 0.1786837110145214E+00, 0.2414921532322530E+00, 0.5630913108462098E+00, 0.7820323704275389E+00, 0.5523326888742272E+00, 0.9015512185421630E+00, 0.6659841765947113E+00, 0.2845604319423287E+00, 0.3491764093356571E+00, 0.9628247275953050E-01, 0.6976897044251388E+00, 0.2960314575787897E+00, 0.5554565606281359E-01, 0.3558757052257084E+00, 0.9341850945519581E+00, 0.6665938574616496E+00, 0.9614102712845256E+00, 0.8607214960584614E+00, 0.4038740733880183E+00, 0.8195623831081322E+00, 0.6129183541897144E+00, 0.7215327066311872E-01, 0.5031970254369666E-01, 0.1396452298202989E+00, 0.9662061833347503E+00, 0.1270920838674061E+00, 0.4311106155362532E+00, 0.8052749852152479E+00, 0.4015794631208873E+00, 0.9090852867289334E-01, 0.6606775367976888E+00, 0.1481179966505828E+00, 0.9638580924515038E+00, 0.9709844996525795E+00, 0.1150673643193312E-01, 0.1429293205078334E+00, 0.2900042758147406E+00, 0.9072348766684573E+00, 0.5781273728635536E+00, 0.5436950045999074E+00, 0.1857622262967125E+00, 0.9164714647876027E+00, 0.1370982861482903E+00, 0.4992756875010576E+00, 0.1554652709586491E+00, 0.3126479857720679E+00, 0.4107075699045055E+00, 0.2895846225773548E+00, 0.6273125726015746E+00, 0.2838905730735005E+00, 0.5809021463377108E+00, 0.7647718231557297E+00, 0.2069816461533343E+00, 0.3140055757156704E+00, 0.1235726138058405E+00, 0.4368723232060789E-01, 0.7675622822273089E+00, 0.9898297711455730E+00, 0.5845844813092991E+00, 0.3724928805727834E+00},
			n:    21,
			zOut: []float64{0.2656606156715505E+01, 0.2361088067613179E+01, 0.2270460825471249E+01, 0.2122871466419806E+01, 0.1864972075992565E+01, 0.1483766917780093E+01, 0.1313593002893207E+01, 0.1121546276208008E+01, 0.1100149868182268E+01, 0.9659264313206680E+00, 0.7624972132947719E+00, 0.7289080649182198E+00, 0.6752163576811447E+00, 0.6611764720975886E+00, 0.5004340625462813E+00, 0.2440674815088013E+00, 0.2171809202435314E+00, 0.1235497606154792E+00, 0.3737121186344041E-01, 0.2834840406904476E-01, 0.5846755428136478E-02, 0.7443896006885402E-16, 0.5016677150024004E-07, 0.3707799983650212E-07, 0.1313593002893207E+01, 0.2958228394578794E-29, 0.1581957950672119E-06, 0.2298083114979900E-06, 0.1121546276208008E+01, 0.1915993820645148E-16, 0.4749045639556273E-06, 0.4145613346073258E-06, 0.1100149868182268E+01, 0.9026685982667287E-18, 0.2821283116535875E-06, 0.3158894087015811E-06, 0.9659264313206680E+00, 0.2037048580685469E-20, 0.2246093425341190E-06, 0.2076005360058006E-06, 0.7624972132947719E+00, 0.1234566306928925E-21, 0.2124557779286299E+02, 0.2124557779286298E+02, 0.1010000000000000E+03, 0.3002267573696145E+01, 0.2970297029702970E+01, 0.4897987464996691E-04, 0.6752163576811447E+00, 0.7461812581271996E-18, 0.7761124913531668E-04, 0.7483244468930916E-04, 0.6611764720975886E+00, 0.1038114253693458E-19, 0.1582094808245393E-03, 0.1235322388201987E-03, 0.5004340625462813E+00, 0.1010613662291919E-16, 0.2610534203903574E-03, 0.4450812691210283E-03, 0.2440674815088013E+00, 0.1110435042389887E-20, 0.5668320309921573E-04, 0.5311544381771655E-04, 0.2171809202435314E+00, 0.4733165431326071E-27, 0.6599010146996593E-04, 0.7896621041843452E-04, 0.1235497606154792E+00, 0.3155385912111625E-18, 0.1746230697391169E-01, 0.1331928313395539E-01, 0.3737121186344041E-01, 0.7395570986446986E-31, 0.1364698415102933E-02, 0.9400709901879650E-03, 0.2834840406904476E-01, 0.1581043861486090E-22, 0.9826835453665868E-03, 0.1307033404080312E-02, 0.5846755428136478E-02, 0.5593364605928029E-23, 0.6675960788692170E-02, 0.6853670175678725E-02},
			info: 0,
		},
		{
			z:    []float64{0.4059771234161322E+00, 0.9607215623246773E+00, 0.2458768097786967E-01, 0.2349999394464353E-01, 0.5195584553698999E+00, 0.4267565401751807E+00, 0.5653505726897825E+00, 0.1993982232067181E+00, 0.7123509462851438E+00, 0.7322091078130321E+00, 0.6397865818692764E+00, 0.7795245712118530E+00, 0.9924677795119503E+00, 0.9446777193650685E+00, 0.9161440637569995E+00, 0.4356325057978820E+00, 0.6968317286853016E+00, 0.8623411680463823E+00, 0.8789901991689911E+00, 0.3964880116495537E+00, 0.3452640838632506E+00, 0.2457053330236874E+00, 0.3088650440709877E+00, 0.3819576657222301E+00, 0.3428589632713919E-01, 0.5108638376023356E-01, 0.1646491337519898E+00, 0.4402398340108123E+00, 0.6741907107293844E+00, 0.8399512836956711E+00, 0.5689849385173426E+00, 0.5208530271268840E+00, 0.1800472375386978E+00, 0.7659410457647264E+00, 0.2320702606914905E+00, 0.1105800266048680E+00, 0.9555355666039600E-02, 0.1859192596152662E+00, 0.7825627400019242E-02, 0.5286418396370182E+00, 0.2569279638014341E+00, 0.5549104171411018E+00, 0.7406835450501670E+00, 0.8319568443444482E+00, 0.1523286720686696E+00, 0.6741822248281739E+00, 0.7686049385045001E+00, 0.7046090895325479E+00, 0.5187887151753254E+00, 0.1082987708911324E+00, 0.1267838014025160E+00, 0.1115865001832446E+00, 0.9799554493413084E+00, 0.2112830058825086E+00, 0.1414836977037485E+00, 0.6416690587599562E+00, 0.7257743058080667E+00, 0.5941533679852271E+00, 0.2889670048515264E+00, 0.2352763591376699E+00, 0.4250778140801711E+00, 0.1017573282390372E-01, 0.7592168685641336E+00, 0.1236630021486789E+00, 0.3794258137484455E+00, 0.3894249799986492E+00, 0.3467719607731882E+00, 0.5651244918121399E+00, 0.7941305753999619E+00, 0.7160004674307343E+00, 0.6194397297375442E+00, 0.1453439098617406E+00, 0.3817157280391337E-03, 0.9381666563671931E+00, 0.8543357988390619E+00, 0.4675645670082479E+00, 0.8133557631373793E+00, 0.1160066189475003E+00, 0.6551894700973839E+00, 0.6157191087148942E+00, 0.4651705119652626E+00, 0.2935400080400429E+00, 0.4317648586994048E+00, 0.8522331953027451E-01},
			n:    21,
			zOut: []float64{0.3090946182879871E+01, 0.2427731098824810E+01, 0.2178358732328273E+01, 0.1944657148720347E+01, 0.1388313875420951E+01, 0.1366009016577358E+01, 0.1254539813854158E+01, 0.1096743376160052E+01, 0.8976114875705017E+00, 0.7925182471468240E+00, 0.7633415317747125E+00, 0.5940562138773567E+00, 0.3406702196684873E+00, 0.2541261981011808E+00, 0.2081990106830526E+00, 0.1967266434974441E+00, 0.8212572008660911E-01, 0.6231598248219099E-01, 0.2121611392818772E-01, 0.6627852439078377E-02, 0.2020564063973749E-05, 0.5156511541642804E-17, 0.3563959083802911E-09, 0.2193664839500192E-09, 0.1254539813854158E+01, 0.5107396913706617E-17, 0.2245013704648247E-08, 0.3514139913607919E-08, 0.1096743376160052E+01, 0.1770166510860316E-21, 0.1137128600534393E-07, 0.7589514151253922E-08, 0.8976114875705017E+00, 0.3631375006656261E-16, 0.1027683956127825E-09, 0.1703981899256795E-09, 0.7925182471468240E+00, 0.0000000000000000E+00, 0.2692111438183261E-08, 0.1491447500870467E-08, 0.7633415317747125E+00, 0.3655079503368504E-18, 0.1896683648658551E+02, 0.1896683648658551E+02, 0.9800000000000000E+02, 0.2941043083900227E+01, 0.3061224489795918E+01, 0.1473194721443778E-07, 0.3406702196684873E+00, 0.8487843002529896E-19, 0.1107295924775035E-05, 0.6897939613884539E-06, 0.2541261981011808E+00, 0.8699892772041781E-19, 0.4010969039165872E-05, 0.5564243023106877E-05, 0.2081990106830526E+00, 0.1258849778182470E-17, 0.2192657884798391E-05, 0.1921286515322920E-05, 0.1967266434974441E+00, 0.1498835719919922E-28, 0.3181611995531238E-06, 0.2328457188758453E-05, 0.8212572008660911E-01, 0.9162137014768053E-16, 0.4655593170815260E-04, 0.2906458517213153E-04, 0.6231598248219099E-01, 0.7111877894410489E-19, 0.1084455050840344E-03, 0.4940544045928809E-04, 0.2121611392818772E-01, 0.1296887328183343E-24, 0.1059468102013224E-02, 0.9250155264543548E-03, 0.6627852439078377E-02, 0.2014616961124939E-22, 0.1082171160654875E-03, 0.1415095048461561E-03, 0.2020564063973749E-05, 0.0000000000000000E+00, 0.6572087989553770E-04, 0.6490933881766271E-04},
			info: 0,
		},
		{
			z:    []float64{0.3515867373170093E+00, 0.6900024519663606E-01, 0.2562617636787797E+00, 0.4448144612254716E+00, 0.1882829208975508E+00, 0.5473764707914393E+00, 0.2385465140832452E+00, 0.1069888484826550E+00, 0.6650479699038470E+00, 0.5577458834974632E+00, 0.3827016483759793E-01, 0.2430476485682098E-01, 0.4730689827049095E-01, 0.7028807845337007E+00, 0.2183345614483843E+00, 0.4504620631941809E+00, 0.4989666416356805E+00, 0.3524179690092576E+00, 0.2245980971932510E+00, 0.4235501615333767E+00, 0.5262756658101302E+00, 0.9543218824354944E+00, 0.7932768470961233E+00, 0.6115989702914060E-01, 0.5090745247561921E+00, 0.6231416396947956E+00, 0.5848645563028462E+00, 0.4261243979220120E-01, 0.7488050001299923E+00, 0.1371419590365067E+00, 0.2276573560104874E+00, 0.3333510148295767E+00, 0.8801005866377587E+00, 0.1783788042757909E+00, 0.7167665061134074E+00, 0.8328416386285911E+00, 0.3239179761297745E+00, 0.3396256067647054E+00, 0.3124818554244235E-01, 0.3558253686357221E+00, 0.3434895378785284E+00, 0.5623459587025392E-01, 0.2638452014850361E+00, 0.8291744484791080E+00, 0.1974847584119865E+00, 0.7319795333910079E+00, 0.2610606436005336E+00, 0.5710127034520260E+00, 0.1439281438102160E+00, 0.9086761346521347E+00, 0.8079247890865698E+00, 0.9018578929430654E+00, 0.4900454364738460E+00, 0.8138610668520670E+00, 0.6711229676830790E+00, 0.5055629692453131E+00, 0.7743169767236130E+00, 0.4534345901632091E+00, 0.9064089158232752E+00, 0.4827899713942460E+00, 0.7358703780808750E+00, 0.9369729458478693E-02, 0.8597279323370060E+00, 0.7253136203352766E+00, 0.9571052925069841E+00, 0.2164676786228616E+00, 0.5574107740713430E-05, 0.7154716238932374E+00, 0.1141135068223202E+00, 0.6209438587853456E+00, 0.7851847140042024E+00, 0.4215540797089058E+00, 0.9132020640887749E+00, 0.1510454496157759E+00, 0.4172986352579149E+00, 0.5037450189692434E+00, 0.4485087403424131E+00, 0.5740668076879305E+00, 0.2978796610784261E+00, 0.6581099078338648E-01, 0.8930273171304047E+00, 0.8152477689300778E+00, 0.1069556485483132E+00, 0.6932157626522990E+00},
			n:    21,
			zOut: []float64{0.2205969711876647E+01, 0.1949821527895322E+01, 0.1555841877124327E+01, 0.1396030697440207E+01, 0.1339074151471965E+01, 0.1304928492707023E+01, 0.1082549957806062E+01, 0.9163938874206132E+00, 0.8887122139609793E+00, 0.7285328536138327E+00, 0.6136134334950112E+00, 0.5962413214799335E+00, 0.4250735423840991E+00, 0.3604939377250871E+00, 0.2295486599338548E+00, 0.1443459206161604E+00, 0.1184188267216090E+00, 0.7075746168152415E-01, 0.2006576919306350E-01, 0.3756273174580983E-02, 0.4503973951446061E-03, 0.2214970142320351E-19, 0.1258326061053631E-10, 0.9003557400980630E-11, 0.1082549957806062E+01, 0.1308878013942703E-25, 0.3368372626791194E-10, 0.2572022209508410E-10, 0.9163938874206132E+00, 0.7614331624527915E-16, 0.1049500061330476E-08, 0.2928487158224854E-08, 0.8887122139609793E+00, 0.1203547791264655E-21, 0.7079518778653070E-07, 0.2057204985301009E-06, 0.7285328536138327E+00, 0.1084769825180869E-21, 0.6504687546036223E-07, 0.2011551828019932E-07, 0.6136134334950112E+00, 0.1678480469223895E-16, 0.1595062091511705E+02, 0.1595062091511705E+02, 0.1020000000000000E+03, 0.3156462585034014E+01, 0.2941176470588236E+01, 0.1336032003974816E-08, 0.4250735423840991E+00, 0.1033222323690773E-11, 0.1230907576002882E-05, 0.1377773980605759E-05, 0.3604939377250871E+00, 0.5435093612356237E-21, 0.5904438579725169E-05, 0.1247176535660427E-04, 0.2295486599338548E+00, 0.3794809073678436E-22, 0.8654598111714590E-03, 0.7607233838455693E-03, 0.1443459206161604E+00, 0.2335329392584174E-18, 0.9486551585203510E-04, 0.5848864829043123E-04, 0.1184188267216090E+00, 0.2421802979028506E-27, 0.4440929582264130E-04, 0.3669242548994226E-04, 0.7075746168152415E-01, 0.1891688450719986E-26, 0.5237855019006247E-04, 0.2468105637453731E-04, 0.2006576919306350E-01, 0.1180787648903814E-27, 0.8134126440113327E-03, 0.8865157422262554E-03, 0.3756273174580983E-02, 0.2581152881883151E-26, 0.5819544721666204E-02, 0.3152501351079271E-02, 0.4503973951446061E-03, 0.0000000000000000E+00, 0.5880303886253363E-04, 0.1585743718762697E-03},
			info: 0,
		},
		{
			z:    []float64{0.5610544265871693E+00, 0.3865731523333528E+00, 0.8792353035941016E+00, 0.8523516652694250E+00, 0.1444314813898040E-01, 0.8826603847895033E+00, 0.6868176286586357E+00, 0.1885799699020001E+00, 0.8756021324147915E+00, 0.9000748303342890E+00, 0.6057362077051875E+00, 0.7116572505145777E+00, 0.4651180401287146E+00, 0.2732222796277717E-01, 0.4226433912686857E+00, 0.2137613193249559E+00, 0.8364980217841237E+00, 0.7760779616422476E+00, 0.4204777290352190E+00, 0.6224635225380051E+00, 0.2764474942580449E+00, 0.3783169947455127E+00, 0.5180995217194659E+00, 0.3065081068358929E+00, 0.7865831746887696E+00, 0.1478697978858728E+00, 0.9439830303880672E+00, 0.8253272390565236E-01, 0.3067445628749279E+00, 0.7129198739696287E+00, 0.1655397340526111E+00, 0.1744497852178990E+00, 0.8123176080018479E-01, 0.7468152661528306E+00, 0.8702942511684653E-01, 0.2665897748347560E+00, 0.9213749341388631E+00, 0.1523897006311256E+00, 0.9645030292913545E+00, 0.9612618327522493E-01, 0.4802656873918926E+00, 0.9400504426452867E+00, 0.4625574582408437E+00, 0.7387695442245192E+00, 0.7695082904503155E+00, 0.1394488941705607E+00, 0.4086909510206329E+00, 0.8420266381115991E+00, 0.9564685490270811E+00, 0.7091850518571272E+00, 0.5799173993611073E+00, 0.1780919033120022E+00, 0.4196947685163247E+00, 0.9892865772966106E+00, 0.5502405891255741E+00, 0.7805747931381838E+00, 0.8443114386354278E-01, 0.3818004737775779E+00, 0.5006744900666209E+00, 0.8330036981446504E+00, 0.6972526215524509E+00, 0.5997317354738497E+00, 0.6745658299087246E+00, 0.2369187863883299E+00, 0.4162907871251388E+00, 0.3987550165661536E+00, 0.3127240434721901E+00, 0.9587921004875174E+00, 0.9734600215022703E+00, 0.9507333301136496E+00, 0.3445022862066339E+00, 0.6240012410156072E+00, 0.3036264632031094E-01, 0.6999839547669153E+00, 0.4672138296892350E+00, 0.4669965382645248E+00, 0.7561275857160221E-01, 0.9250454776547237E+00, 0.8620177110728862E+00, 0.5266728805873626E+00, 0.6377535363121675E+00, 0.6098026285363055E+00, 0.8075088997828720E+00, 0.8486821693163010E+00},
			n:    21,
			zOut: []float64{0.2468092632282687E+01, 0.2093427673862981E+01, 0.2055392019191517E+01, 0.1647867143267642E+01, 0.1615849876515927E+01, 0.1589328968238607E+01, 0.1271214388518367E+01, 0.1121382381594411E+01, 0.1069461264755295E+01, 0.1002295596318809E+01, 0.8815764834354939E+00, 0.8051252546404786E+00, 0.6374343329196452E+00, 0.5469723867709347E+00, 0.4015178604185544E+00, 0.3707602923265790E+00, 0.1795105850679969E+00, 0.1400595531050117E+00, 0.2548349514639473E-01, 0.1561366451013735E-02, 0.1155321273825210E-02, 0.3137752941767923E-22, 0.1448569525610494E-09, 0.1682297708320945E-10, 0.1271214388518367E+01, 0.3747089299799806E-29, 0.3127891707973917E-07, 0.1432924100618686E-07, 0.1121382381594411E+01, 0.1829309924661015E-19, 0.1414348814679943E-06, 0.1338467241717145E-06, 0.1069461264755295E+01, 0.2814655709828570E-26, 0.1421706541245010E-06, 0.1494636997707923E-06, 0.1002295596318809E+01, 0.2555408239179452E-16, 0.1745096068338993E-06, 0.1639215185470380E-06, 0.8815764834354939E+00, 0.5195421261043265E-21, 0.1992546887610216E+02, 0.1992546887610217E+02, 0.1040000000000000E+03, 0.3263038548752835E+01, 0.3846153846153846E+01, 0.5490557769901970E-06, 0.6374343329196452E+00, 0.1696050946225175E-28, 0.4391959465127471E-06, 0.4441570425561870E-06, 0.5469723867709347E+00, 0.4122920742224575E-16, 0.4792107785920969E-06, 0.5244422797027597E-06, 0.4015178604185544E+00, 0.4437342591868191E-30, 0.1189624719598388E-05, 0.1545254324980458E-05, 0.3707602923265790E+00, 0.0000000000000000E+00, 0.3830671301160838E-05, 0.5436396467135237E-05, 0.1795105850679969E+00, 0.1558228073167050E-14, 0.5423210427919237E-04, 0.1228583690085413E-03, 0.1400595531050117E+00, 0.8832743148296122E-19, 0.1395228897382397E-04, 0.1537585917376726E-04, 0.2548349514639473E-01, 0.0000000000000000E+00, 0.8277350694369951E-03, 0.9528953717632958E-03, 0.1561366451013735E-02, 0.2465190328815662E-31, 0.7095009159403311E-03, 0.6952609237045969E-03, 0.1155321273825210E-02, 0.1309509102666880E-27, 0.9046555260897671E-03, 0.1226313636978587E-02},
			info: 0,
		},
		{
			z:    []float64{0.1242522906682814E+00, 0.6390609620209767E+00, 0.4481822861180138E+00, 0.4850355616354706E+00, 0.4018997557884576E+00, 0.7075817751574985E+00, 0.3076496509717662E+00, 0.9845619472128853E+00, 0.4421050939804582E+00, 0.3472138710835164E+00, 0.7694977672753175E+00, 0.4786944038481857E+00, 0.6974278852457209E+00, 0.5004517708240744E+00, 0.7658513494839985E+00, 0.7508522623862479E-01, 0.8645614962374704E-02, 0.2513671118506070E+00, 0.9355687048630774E+00, 0.2836435232395548E+00, 0.8653687008315966E+00, 0.3043727382738487E+00, 0.8973655399535756E+00, 0.7135517870607274E+00, 0.4956863425678929E+00, 0.8362506849216409E+00, 0.9829450079482006E+00, 0.9413718361369072E+00, 0.2542921002853715E-01, 0.5951158954104814E+00, 0.9317747763384295E+00, 0.9085271053958202E+00, 0.9993466426511500E+00, 0.2497009223772528E-01, 0.8726460080387569E+00, 0.3868463012727243E+00, 0.5820757557404177E-01, 0.7974577837432384E-01, 0.8242609714155934E+00, 0.1887913799920241E+00, 0.3268225406700475E+00, 0.2264596835393817E+00, 0.4020160264981738E+00, 0.8597685353848280E+00, 0.5429598215579996E+00, 0.4134507486351844E+00, 0.8781720494573462E+00, 0.3655827200857601E+00, 0.2876763179222336E+00, 0.9193112804533413E+00, 0.3958413207165046E-01, 0.7980401670837647E+00, 0.5101422689515223E-01, 0.2194050847732673E+00, 0.6115669547384739E-02, 0.9036470382476453E+00, 0.6696681747677364E+00, 0.1689100553906829E+00, 0.9284887836539969E+00, 0.8748192507086142E+00, 0.7181738874264668E+00, 0.8990747837549429E-01, 0.7166811698029575E+00, 0.8712804839027968E+00, 0.2571898936326318E+00, 0.2264160460654618E+00, 0.6364724288557849E+00, 0.9550337894922362E+00, 0.8995892333665090E+00, 0.3728949594240628E+00, 0.6496276036419958E+00, 0.7218580021816370E+00, 0.2620053580399828E+00, 0.8492826434932900E+00, 0.1090756692624820E+00, 0.1230806328807070E+00, 0.5326342483791896E+00, 0.3425391815117244E+00, 0.4714474424671373E+00, 0.9044440560476328E+00, 0.2298320061743346E+00, 0.7771450381369926E+00, 0.4623588499686800E+00, 0.8996441805847293E+00},
			n:    21,
			zOut: []float64{0.2749036998648791E+01, 0.2657147938200912E+01, 0.2156780738808330E+01, 0.2057290308234013E+01, 0.1910481192038085E+01, 0.1550197528741744E+01, 0.1531663815430223E+01, 0.1307411568456853E+01, 0.1180055477995477E+01, 0.1145160315453326E+01, 0.8462599780670359E+00, 0.8196023258344234E+00, 0.7505058481318937E+00, 0.3802378569170251E+00, 0.3493658644293456E+00, 0.2733852715996189E+00, 0.1920071438579977E+00, 0.3602127141457542E-01, 0.1503353393017178E-01, 0.5047074669949666E-02, 0.4814167031169938E-03, 0.1615587133892632E-26, 0.3701495167228324E-14, 0.8166164251788096E-14, 0.1531663815430223E+01, 0.5400326177233737E-21, 0.8345943934662111E-13, 0.3845954584916692E-13, 0.1307411568456853E+01, 0.2270604060615577E-17, 0.1098158814550649E-11, 0.6011774632275279E-12, 0.1180055477995477E+01, 0.6617444900424221E-23, 0.1039362721731823E-10, 0.1780733867152956E-10, 0.1145160315453326E+01, 0.3193234771665464E-17, 0.8582967920523009E-10, 0.1451582492090454E-09, 0.8462599780670359E+00, 0.4268426102972081E-16, 0.2191317346756291E+02, 0.2191317346756291E+02, 0.1100000000000000E+03, 0.3303854875283447E+01, 0.6363636363636363E+01, 0.9730504757902699E-07, 0.7505058481318937E+00, 0.1718125187449148E-14, 0.1758342086676891E-05, 0.2642640058729254E-05, 0.3802378569170251E+00, 0.9047523974206004E-14, 0.3754566725544834E-04, 0.5673943129185001E-04, 0.3493658644293456E+00, 0.2726303288443817E-26, 0.2006689219611138E-03, 0.1777791597062721E-03, 0.2733852715996189E+00, 0.1852245405458936E-26, 0.4273484804153724E-06, 0.2289494629295668E-05, 0.1920071438579977E+00, 0.7673537210642012E-17, 0.3631464818839244E-02, 0.3052322482564877E-02, 0.3602127141457542E-01, 0.1402889840288648E-17, 0.2065820498676818E-02, 0.7907462477517949E-03, 0.1503353393017178E-01, 0.3588277137446075E-19, 0.9842420036565967E-03, 0.4295497278130008E-03, 0.5047074669949666E-02, 0.1100487290795119E-18, 0.2169791314757801E-02, 0.2746294001732303E-02, 0.4814167031169938E-03, 0.9926167349879280E-23, 0.2524480967032520E-02, 0.1685323473608901E-02},
			info: 0,
		},
		{
			z:    []float64{0.4452569544189223E+00, 0.3712416527218666E+00, 0.4540009468556715E+00, 0.9149505778015055E-01, 0.9825460881415958E+00, 0.8144387623295611E+00, 0.4257438310114644E+00, 0.4055209729019219E+00, 0.1358301754544711E-01, 0.6660365069195171E+00, 0.4791028777450690E-01, 0.3657122138300755E+00, 0.9244230955293443E+00, 0.2570776992045346E+00, 0.5718524872194095E+00, 0.6959027703199671E+00, 0.8022093165313721E+00, 0.4495167304148069E+00, 0.4533829141769831E+00, 0.4399857636092745E+00, 0.8343772053001745E+00, 0.5841091089133705E+00, 0.9225724116024000E+00, 0.3646730358635919E+00, 0.4342402996301750E+00, 0.1979669913465428E+00, 0.2710080252534286E+00, 0.4064878156937679E+00, 0.1329017000110605E+00, 0.3577863781516848E+00, 0.7737638611946227E+00, 0.3628629851888825E+00, 0.6191378950237536E+00, 0.9181610808896479E-01, 0.3164967052049129E+00, 0.1800354212961807E+00, 0.5261304159866632E+00, 0.2095779664090124E+00, 0.4734470785970568E+00, 0.8233700542774806E+00, 0.1405149000531442E+00, 0.5739228235076023E+00, 0.4484023224779539E+00, 0.7953415183519565E+00, 0.5683020712105596E+00, 0.3828815589637512E+00, 0.9478512018818865E+00, 0.9398576744287683E+00, 0.6799918835962694E+00, 0.9795978835389229E+00, 0.9415167003494995E+00, 0.9804631144197878E+00, 0.4529282233852833E+00, 0.6987441800620822E+00, 0.1071819561656476E+00, 0.6287695276127018E+00, 0.2573913359217608E+00, 0.5283299892256954E-01, 0.1312057002484920E+00, 0.2566822109082798E+00, 0.5022596747022927E+00, 0.3755634037822867E+00, 0.7687685406410046E+00, 0.9286257388850563E+00, 0.2460950353669618E+00, 0.4615682752011302E+00, 0.2668978314403934E+00, 0.9526334220011422E+00, 0.7113266960956600E+00, 0.8033610237680390E+00, 0.2049912091857929E+00, 0.9104136201043411E+00, 0.9905140155095393E+00, 0.5284892163797259E+00, 0.4617116226676921E-01, 0.7638541825257228E+00, 0.8058860223281950E-01, 0.6562405799504624E+00, 0.4217948031372573E+00, 0.8444441663937204E+00, 0.4135300708069887E+00, 0.3992676961789670E+00, 0.9443059736340937E+00, 0.4697907769676380E+00},
			n:    21,
			zOut: []float64{0.2339652518244840E+01, 0.2127240999798084E+01, 0.2122097736373912E+01, 0.1640409192349226E+01, 0.1522936721900112E+01, 0.1513900850773956E+01, 0.1190428480890818E+01, 0.1098839257111754E+01, 0.9060499827420537E+00, 0.8483833633939547E+00, 0.7423625631343986E+00, 0.6441873762582893E+00, 0.5561064852023264E+00, 0.5332801819376103E+00, 0.3323799891685881E+00, 0.2356658160961767E+00, 0.1910001776595044E+00, 0.1052937995306845E+00, 0.3493206724380617E-01, 0.1582963148293342E-01, 0.1362410302334481E-03, 0.2902907641693595E-19, 0.1235088787739235E-18, 0.5938156039510254E-18, 0.1190428480890818E+01, 0.2366582715663035E-29, 0.1736669751992271E-12, 0.4479686287025860E-12, 0.1098839257111754E+01, 0.4930380657631324E-31, 0.6845104399398104E-11, 0.1631652613834293E-10, 0.9060499827420537E+00, 0.3623410171350593E-13, 0.4397239464622790E-08, 0.8783797026819435E-08, 0.8483833633939547E+00, 0.1680831004707752E-20, 0.6292544501750849E-07, 0.1207646260809090E-06, 0.7423625631343986E+00, 0.1044536639065691E-17, 0.1870111343232326E+02, 0.1870111343232326E+02, 0.1030000000000000E+03, 0.3002267573696145E+01, 0.3883495145631068E+01, 0.8796985420666049E-06, 0.5561064852023264E+00, 0.6232001151245993E-28, 0.4765675419124029E-06, 0.5281771988202168E-06, 0.5332801819376103E+00, 0.5995342879679690E-28, 0.3211676631039122E-06, 0.3513325837054497E-06, 0.3323799891685881E+00, 0.2177256098409993E-25, 0.1826919937583927E-06, 0.1747056528730059E-06, 0.2356658160961767E+00, 0.1311326116357865E-17, 0.1784537561302489E-06, 0.2175474607275274E-06, 0.1910001776595044E+00, 0.3862262991962074E-24, 0.9127544222909640E-06, 0.4962248419758332E-06, 0.1052937995306845E+00, 0.1072850831100576E-26, 0.2872845137271569E-03, 0.1413151427897814E-03, 0.3493206724380617E-01, 0.8028654523313936E-17, 0.5779626756191757E-04, 0.4784930511330926E-04, 0.1582963148293342E-01, 0.9359045779542084E-26, 0.9339546012110267E-04, 0.6206712854083305E-04, 0.1362410302334481E-03, 0.4535950205020818E-29, 0.1799327870469576E-03, 0.1621654177500190E-02},
			info: 0,
		},
		{
			z:    []float64{0.6327743759434090E-01, 0.8874473183212240E+00, 0.1587228549487630E+00, 0.1869853022948459E+00, 0.1852952724381735E+00, 0.2190328352455733E+00, 0.7936705141328082E+00, 0.1573124860628259E+00, 0.2446308768539528E+00, 0.2441044501798444E+00, 0.4435884001235265E+00, 0.1783624365771731E+00, 0.6874700271616803E+00, 0.4796486989431838E+00, 0.2471425348644392E+00, 0.4391077480264999E+00, 0.9973269002312380E+00, 0.4770343529783838E+00, 0.3451566405930041E+00, 0.5011834710046675E+00, 0.8072188861666797E+00, 0.5748577907200507E+00, 0.2986003422862493E+00, 0.4760318526964945E+00, 0.5838957192070238E-01, 0.1760621214885278E+00, 0.7926085978219721E+00, 0.3379849703418611E+00, 0.6012719579585296E+00, 0.1130783704430351E+00, 0.5590675745254436E-01, 0.8616930340961984E+00, 0.9917564091315376E+00, 0.4219575368674273E+00, 0.1044382974319413E+00, 0.1886528536659160E-01, 0.9337085742573710E+00, 0.6447952074628677E+00, 0.7158824182577913E+00, 0.3883216757352448E+00, 0.2116568255149501E+00, 0.9819105316462939E+00, 0.8471687522868032E-01, 0.2785996362910685E+00, 0.8775405646426044E+00, 0.5671584270354416E+00, 0.4912868754910720E+00, 0.5767395813214109E+00, 0.7327323379759062E+00, 0.8184048812627022E+00, 0.3325270745990432E+00, 0.3026500963479654E+00, 0.7228591188664935E+00, 0.1094677581261434E+00, 0.5280482398171430E+00, 0.9974727380694979E+00, 0.5087630734843742E+00, 0.6953603638889684E+00, 0.9103967979863506E+00, 0.2584730811693510E+00, 0.5498750728524477E+00, 0.9758543899455846E+00, 0.8309066632970131E+00, 0.4311646442586020E+00, 0.4732818688792167E+00, 0.4096051739313099E+00, 0.7479697576436509E+00, 0.9369473766445329E+00, 0.7380538090811954E+00, 0.1199951604231148E+00, 0.5672128274092054E+00, 0.7150763241893471E+00, 0.6134240647991112E+00, 0.7095501841622953E-01, 0.1510080432732774E-01, 0.2912426074708729E+00, 0.1611746025295486E+00, 0.3946903662654964E+00, 0.7408818971566200E-01, 0.3978128748254128E+00, 0.3580504147215868E+00, 0.2603799996304434E-01, 0.8990784999913340E+00, 0.1985602490013758E+00},
			n:    21,
			zOut: []float64{0.2139906798692060E+01, 0.2104052265745164E+01, 0.2080474521860817E+01, 0.1784368581097839E+01, 0.1501145759273247E+01, 0.1395549430074518E+01, 0.1236063891814824E+01, 0.1123736402991315E+01, 0.8211474325690415E+00, 0.7411912844479571E+00, 0.7336682913123216E+00, 0.5274366057326734E+00, 0.3715621485686555E+00, 0.3171624722931449E+00, 0.2809062056956198E+00, 0.1668884413843382E+00, 0.9744092162322979E-01, 0.7613224124753509E-01, 0.1665106039654164E-01, 0.4700366860927376E-02, 0.1401918312945850E-02, 0.1084683744678891E-29, 0.4021673570416241E-08, 0.3931910860092540E-08, 0.1236063891814824E+01, 0.1439956010332256E-19, 0.3626387646080745E-08, 0.3543555828237728E-08, 0.1123736402991315E+01, 0.3006506681978320E-24, 0.3371387381785702E-08, 0.3448854867663729E-08, 0.8211474325690415E+00, 0.6197291271416269E-26, 0.2831422041774839E-08, 0.2886341750444270E-08, 0.7411912844479571E+00, 0.0000000000000000E+00, 0.2880306038760265E-08, 0.2936321908308739E-08, 0.7336682913123216E+00, 0.2406371799476454E-20, 0.1752158704199472E+02, 0.1752158704199472E+02, 0.1050000000000000E+03, 0.3111111111111111E+01, 0.7619047619047619E+01, 0.3652487989473621E-08, 0.3715621485686555E+00, 0.1451701975030564E-21, 0.4296538801302010E-08, 0.3992227758256396E-08, 0.3171624722931449E+00, 0.6162975822039155E-31, 0.5151357395533313E-08, 0.5974032086218965E-08, 0.2809062056956198E+00, 0.2039658356197591E-16, 0.1134165265363298E-07, 0.1513934018844884E-07, 0.1668884413843382E+00, 0.2400943524542200E-24, 0.1781584505522029E-06, 0.1218300442987205E-06, 0.9744092162322979E-01, 0.6162975822039155E-31, 0.9557876398738343E-06, 0.5872690889712619E-06, 0.7613224124753509E-01, 0.5074727415413836E-17, 0.2309052370169074E-04, 0.5924498773169641E-04, 0.1665106039654164E-01, 0.1400228106767296E-28, 0.5116509326795711E-04, 0.5154480234673206E-04, 0.4700366860927376E-02, 0.1956855682120442E-20, 0.1061141342429108E-03, 0.9755516139369119E-04, 0.1401918312945850E-02, 0.0000000000000000E+00, 0.1229085807195224E-02, 0.8009097549502667E-02},
			info: 0,
		},
		{
			z:    []float64{0.7225493913848618E+00, 0.3268825038779278E+00, 0.2580957584922012E+00, 0.4222896610254372E+00, 0.8243187922209068E+00, 0.3973391932422579E+00, 0.3700698452657641E+00, 0.2073422474585492E+00, 0.3071746797249822E+00, 0.2370168833330754E+00, 0.7779583030993944E+00, 0.4350651843345934E+00, 0.7867543754352369E+00, 0.3439673345654075E+00, 0.3736479358698906E+00, 0.8450641599616520E+00, 0.3880138308747846E+00, 0.2479782080134303E+00, 0.3220864609073060E+00, 0.8460149388339712E+00, 0.6082870752479030E+00, 0.3396261004129498E+00, 0.1231305479976765E+00, 0.6514493424347317E+00, 0.4987273421671501E-01, 0.2199653339283912E+00, 0.5626339029868159E+00, 0.6755574738917364E+00, 0.7016850168758351E+00, 0.7957347983005405E+00, 0.3394687111682781E-01, 0.1490794513934588E+00, 0.9837873705641814E+00, 0.5038299137267350E+00, 0.8394265886439850E+00, 0.1438624872728633E-01, 0.4444190467253504E+00, 0.1489248400425094E+00, 0.6410535698206483E+00, 0.2739267916732867E+00, 0.6522097474411971E+00, 0.9086492388715542E+00, 0.1949071722314141E+00, 0.7176724675018002E+00, 0.9247176136838016E+00, 0.4929580602677628E+00, 0.3824418236647242E+00, 0.2425574232540663E+00, 0.7085287909144397E+00, 0.2574302555167504E+00, 0.6093160252921369E+00, 0.2417561032032995E+00, 0.4795262420397247E+00, 0.4658200993830509E+00, 0.9033699918994937E+00, 0.1295021719769064E+00, 0.6219991535812029E+00, 0.7040800557612208E+00, 0.8644629036591867E-01, 0.5129394685202899E+00, 0.7988983132437487E+00, 0.8322467913624354E+00, 0.9905487172695071E-01, 0.7093832076693246E+00, 0.1992462440538834E-01, 0.6378263512182120E+00, 0.5823949252324323E-02, 0.7826279628890456E+00, 0.5919828344466795E+00, 0.3815528510244244E+00, 0.6537355893096064E-01, 0.6459825623191251E-01, 0.4466210788758083E+00, 0.6620124961190184E+00, 0.6318809895815939E+00, 0.8145861476435108E+00, 0.5938169865214039E+00, 0.8092284578140500E+00, 0.5378701643349270E+00, 0.1648044346935199E+00, 0.9874291179811017E+00, 0.3210667070512012E+00, 0.6809988920516613E+00, 0.6410080963943865E+00},
			n:    21,
			zOut: []float64{0.2108032684782400E+01, 0.1971253922847928E+01, 0.1952878543820764E+01, 0.1839982524288513E+01, 0.1730503177731785E+01, 0.1509287569836986E+01, 0.1316544049683026E+01, 0.1117715356834192E+01, 0.9058841126016787E+00, 0.8452427684353671E+00, 0.7582219581243467E+00, 0.6392786878439012E+00, 0.5749674919024932E+00, 0.5111167278611346E+00, 0.3859485208834002E+00, 0.2295319067023061E+00, 0.2262113890497363E+00, 0.1122658486667464E+00, 0.7234153262002044E-01, 0.4527048542558528E-01, 0.8319414807792288E-04, 0.2547288901631409E-13, 0.9052716547280089E-04, 0.7818093042872877E-04, 0.1316544049683026E+01, 0.0000000000000000E+00, 0.2537869832861900E-04, 0.2058109204222074E-04, 0.1117715356834192E+01, 0.0000000000000000E+00, 0.1154377886785102E-04, 0.1215574110748570E-04, 0.9058841126016787E+00, 0.8843729980912407E-17, 0.1391102994778336E-08, 0.5248885105740662E-08, 0.8452427684353671E+00, 0.2445468806185137E-28, 0.2192473584471834E-06, 0.3792344586441173E-06, 0.7582219581243467E+00, 0.6480317173221865E-17, 0.1885256245409039E+02, 0.1885256245409039E+02, 0.1090000000000000E+03, 0.3303854875283447E+01, 0.3669724770642202E+01, 0.3715603111278862E-08, 0.5749674919024932E+00, 0.4928548884551519E-18, 0.2515491583997865E-07, 0.9138389520203617E-08, 0.5111167278611346E+00, 0.6587911462745685E-21, 0.7857712130631377E-05, 0.2298529163657988E-05, 0.3859485208834002E+00, 0.6162975822039155E-32, 0.1022329296046741E-03, 0.8680151473672892E-04, 0.2295319067023061E+00, 0.0000000000000000E+00, 0.4616464530663241E-04, 0.3877924919889212E-04, 0.2262113890497363E+00, 0.5522026336547083E-29, 0.2365660345682357E-03, 0.1990352708186489E-03, 0.1122658486667464E+00, 0.9015102222917331E-20, 0.6825572508936817E-03, 0.6443371781720532E-03, 0.7234153262002044E-01, 0.1168271446196110E-24, 0.1518594231951364E-05, 0.3554999772664582E-05, 0.4527048542558528E-01, 0.1023373475125114E-24, 0.5037539097182833E-02, 0.4761613742874584E-02, 0.8319414807792288E-04, 0.1479114197289397E-30, 0.9105689905446296E-02, 0.4825995872157137E-02},
			info: 0,
		},
		{
			z:    []float64{0.3256696334281521E+00, 0.7530856859911462E+00, 0.8853989127988440E+00, 0.4058112879440673E+00, 0.4296659748141172E+00, 0.5414662225246791E-01, 0.7727106089758434E+00, 0.8219533891865206E+00, 0.8804898818241804E+00, 0.2701661776792790E+00, 0.3062100113053869E+00, 0.7616550925245322E+00, 0.7441861437457686E+00, 0.1208643477044503E+00, 0.1894387751691341E+00, 0.5502297763300060E+00, 0.7033314588214317E+00, 0.9049640752657374E+00, 0.7642474001418834E+00, 0.6461873559439021E+00, 0.7323555348666727E+00, 0.6137344390498661E+00, 0.4469041036073067E+00, 0.6120974783290001E+00, 0.3695185251601272E+00, 0.9332999793731735E+00, 0.3981597952563400E+00, 0.5926926060619504E+00, 0.2218776156238759E+00, 0.5556127447847847E-01, 0.2642041684868913E+00, 0.3100431241269808E-01, 0.7617158805289858E+00, 0.5576464290832684E+00, 0.1499291988028689E+00, 0.6071627267784354E+00, 0.2903444085740193E+00, 0.8982141027722228E+00, 0.3437740722392461E+00, 0.5257340992985249E-02, 0.9772197173932363E+00, 0.2743313505008338E-01, 0.5939995532981283E+00, 0.8125099455585232E+00, 0.8394050677385213E+00, 0.2410326841076476E+00, 0.6066214991817382E+00, 0.1828025638429278E+00, 0.5406675263078469E+00, 0.1108622672142920E+00, 0.2412280709874803E+00, 0.5713495470758625E+00, 0.6315445401927943E+00, 0.2035563756883153E+00, 0.8696814083183412E+00, 0.2313134367709406E+00, 0.4474447231170641E+00, 0.6238312383525342E+00, 0.4961806049996582E+00, 0.8121574484576765E+00, 0.8702938949430352E+00, 0.4187164629520156E+00, 0.9204090241956668E+00, 0.8587176140225750E-02, 0.5171057705188283E+00, 0.5195595031109387E+00, 0.1704600468811621E+00, 0.3205951022793651E+00, 0.5643494948239225E-01, 0.9895063272544601E+00, 0.5554681247022339E+00, 0.6098243547887802E+00, 0.4730289261335907E+00, 0.1850396134174421E+00, 0.9997475966289492E+00, 0.6812817712215841E+00, 0.8515433432907883E+00, 0.7377047940023331E+00, 0.2280516830782600E+00, 0.7204549837953970E+00, 0.6096260713045146E+00, 0.9381128478076188E+00, 0.8446173293023428E+00, 0.6255387686605924E+00},
			n:    21,
			zOut: []float64{0.2655567567233002E+01, 0.2307219644143945E+01, 0.2068899090176533E+01, 0.2040332602619700E+01, 0.1742115921396612E+01, 0.1681856112031183E+01, 0.1680515965675234E+01, 0.1379580829731510E+01, 0.1074312270120212E+01, 0.9816962393879983E+00, 0.7217300335876665E+00, 0.6931029681515243E+00, 0.5680300928567689E+00, 0.5098239945058581E+00, 0.4280073603039590E+00, 0.2868581986899945E+00, 0.1650069923584772E+00, 0.9685136354131559E-01, 0.5953175886938158E-01, 0.1010639579838301E-01, 0.1940920539235378E-02, 0.0000000000000000E+00, 0.6683246174917478E-10, 0.4267275295503314E-10, 0.1680515965675234E+01, 0.3510431028233503E-28, 0.1660414363340253E-09, 0.1048896312812690E-09, 0.1379580829731510E+01, 0.1462018530271286E-15, 0.3421106985799154E-06, 0.1788748688597153E-06, 0.1074312270120212E+01, 0.6708481546523056E-14, 0.2133146148219686E-04, 0.4629406838917969E-04, 0.9816962393879983E+00, 0.1747667198202037E-19, 0.5113942524303023E-03, 0.6287577911928029E-03, 0.7217300335876665E+00, 0.3213860550383917E-17, 0.2115308632171850E+02, 0.2115308632171849E+02, 0.1090000000000000E+03, 0.3217687074829932E+01, 0.4587155963302752E+01, 0.4202140396420195E-04, 0.5680300928567689E+00, 0.4489765112701642E-21, 0.3523329095202763E-09, 0.1167512512099963E-08, 0.5098239945058581E+00, 0.1652551767516096E-21, 0.2872281528375301E-06, 0.1040291448945555E-06, 0.4280073603039590E+00, 0.3262461795428160E-16, 0.6249760558326680E-04, 0.5475920486023645E-04, 0.2868581986899945E+00, 0.1568256893218114E-20, 0.1414968677825121E-03, 0.1814031293335571E-03, 0.1650069923584772E+00, 0.8677469957431130E-28, 0.6720068122740940E-03, 0.4975589868187537E-03, 0.9685136354131559E-01, 0.5127595883936577E-29, 0.9260272777115331E-03, 0.1730290225790419E-02, 0.5953175886938158E-01, 0.7227833914448031E-23, 0.5968169418548035E-04, 0.1670361992731526E-04, 0.1010639579838301E-01, 0.1295106991551472E-19, 0.1213518994781533E-03, 0.2534883535393275E-03, 0.1940920539235378E-02, 0.0000000000000000E+00, 0.1005634135102979E-02, 0.2318000467665894E-02},
			info: 0,
		},
		{
			z:    []float64{0.1944566011999833E+00, 0.2181910996636003E+00, 0.3150099357031281E+00, 0.2091722274105127E+00, 0.6593858021341683E+00, 0.9426815336068497E+00, 0.5288011644568881E+00, 0.7784091503839152E+00, 0.1751638954124558E+00, 0.6164772800195180E+00, 0.6225196859922683E+00, 0.2786175486675760E+00, 0.1945914328217568E+00, 0.5779085626077999E+00, 0.5568218939451671E+00, 0.5170566254269496E+00, 0.5742861029155353E+00, 0.6923056150383281E+00, 0.5876692361605969E+00, 0.4295639666640205E+00, 0.8612105519992207E+00, 0.9963836880013683E+00, 0.2692934182903581E+00, 0.6181020857251435E+00, 0.1251902537663725E+00, 0.2838625146280274E+00, 0.2999174442521634E+00, 0.7258439193048929E+00, 0.2048904835805728E+00, 0.1174987481558037E+00, 0.6887427109309550E+00, 0.7409107864964065E+00, 0.5306892441542433E+00, 0.6214581212205206E-01, 0.1234501953117273E-01, 0.2038789370511589E+00, 0.6180963044354376E+00, 0.2820746320472540E+00, 0.4643004781082601E+00, 0.2642696366861867E+00, 0.4102198269957935E-01, 0.1495900075045802E+00, 0.2060405615637534E+00, 0.3399183002928583E+00, 0.7918555051917199E+00, 0.9699614514519834E+00, 0.2922678000248715E+00, 0.1376674512222148E+00, 0.2496993200590716E+00, 0.3432539555227123E+00, 0.5835005548357151E+00, 0.4094205671447549E+00, 0.4876670000237163E+00, 0.7973523995663028E+00, 0.8851386592921134E+00, 0.1523554028450115E+00, 0.7049520832902729E+00, 0.4689643982524618E+00, 0.4347705823917429E+00, 0.8152140283905303E+00, 0.9259201525760803E+00, 0.3175789772106010E+00, 0.3950010049586650E+00, 0.7107346574834400E+00, 0.8109922991383097E+00, 0.6889072760433030E+00, 0.7861311781839575E-02, 0.7751963519893628E+00, 0.7382950156197396E+00, 0.9729693586970557E+00, 0.7283826758396350E+00, 0.1269651196972527E+00, 0.3530947260257147E+00, 0.2061448083515627E-01, 0.5931058475369740E+00, 0.9689379021952851E+00, 0.3254887771415839E+00, 0.2808523639662175E-01, 0.6211468186653845E+00, 0.6037463613076512E+00, 0.8746435454108382E+00, 0.5300695323546331E+00, 0.5276801989236730E+00, 0.2128951663669798E-01},
			n:    21,
			zOut: []float64{0.2424279264689105E+01, 0.2270324150901105E+01, 0.1935128898096250E+01, 0.1814500336114905E+01, 0.1538233405195664E+01, 0.1297421320254112E+01, 0.1267282488999314E+01, 0.1249158514677014E+01, 0.9155076211264116E+00, 0.8242950831432680E+00, 0.5949232544739386E+00, 0.5630079212779312E+00, 0.4530053256303904E+00, 0.3458193312768856E+00, 0.2146623096050983E+00, 0.1686454472303586E+00, 0.1155214217797334E+00, 0.5180948935108346E-01, 0.2789744048901975E-01, 0.8268186305777192E-02, 0.6680158028153346E-04, 0.0000000000000000E+00, 0.3986945646893946E-08, 0.7023072825762287E-08, 0.1267282488999314E+01, 0.0000000000000000E+00, 0.2355158572444734E-07, 0.1877711213087836E-07, 0.1249158514677014E+01, 0.3384506411488746E-20, 0.2648360770785381E-11, 0.1741394977127960E-10, 0.9155076211264116E+00, 0.3299177303725225E-12, 0.2487861145673591E-05, 0.4028407591944186E-05, 0.8242950831432680E+00, 0.2441420600100056E-16, 0.1869515956083083E-05, 0.1842279235203962E-05, 0.5949232544739386E+00, 0.2024938139529812E-20, 0.1807975801219765E+02, 0.1807975801219765E+02, 0.9300000000000000E+02, 0.2836734693877551E+01, 0.1075268817204301E+01, 0.9521662195121400E-05, 0.4530053256303904E+00, 0.1259501632163109E-23, 0.1468124594065504E-05, 0.4670591953230941E-05, 0.3458193312768856E+00, 0.7975064420889996E-22, 0.1308295990671250E-06, 0.6690001990389955E-07, 0.2146623096050983E+00, 0.1927976052360153E-26, 0.3812574759844485E-05, 0.3580340604691736E-05, 0.1686454472303586E+00, 0.3739200690747596E-27, 0.6844384387279355E-05, 0.5557405125771409E-05, 0.1155214217797334E+00, 0.7506098603421864E-21, 0.2139339757899294E-04, 0.3026575107963825E-04, 0.5180948935108346E-01, 0.3886068668379982E-20, 0.1983680974127293E-03, 0.1336830557694390E-03, 0.2789744048901975E-01, 0.1991084924777834E-26, 0.4443273484877918E-03, 0.6821022035819646E-03, 0.8268186305777192E-02, 0.1921747358113822E-20, 0.1778955160858100E-05, 0.3048292256883238E-06, 0.6680158028153346E-04, 0.7915528772828604E-21, 0.4499131616906370E-02, 0.1547765737453591E-02},
			info: 0,
		},
		{
			z:    []float64{0.3849545441014558E+00, 0.9201984078647510E+00, 0.4611893507512446E+00, 0.9426704990067624E+00, 0.4513996483425642E+00, 0.3546824995764211E+00, 0.2673932938327498E+00, 0.2250322436282782E+00, 0.3003875396759296E+00, 0.8637078150569484E+00, 0.2463125523040188E-01, 0.6381498516846796E+00, 0.9054059378226459E+00, 0.9211368359293561E+00, 0.6899055937753183E+00, 0.6469675067081565E+00, 0.2524386712411693E+00, 0.4816130873217878E+00, 0.8437265349008123E+00, 0.8843672501609817E+00, 0.3917808232071412E+00, 0.4109107468556734E+00, 0.6294664210780233E+00, 0.5231064834871043E+00, 0.6006449315760222E+00, 0.2321330919598237E+00, 0.9857216391074231E-01, 0.5246894901632879E+00, 0.2429723268334654E+00, 0.3385437000243748E+00, 0.1784805178646218E+00, 0.8008314854810928E+00, 0.3388059467143087E+00, 0.4957241256656372E+00, 0.6108069077503485E+00, 0.8350989375447051E+00, 0.1087391555565628E+00, 0.5847974581188020E+00, 0.8316850716702742E+00, 0.8921011586703487E+00, 0.1947070853010260E+00, 0.1394928972102342E+00, 0.7929088249487071E+00, 0.4078510186300481E+00, 0.6849316610029904E+00, 0.1436003498260631E+00, 0.5673845335495399E+00, 0.7998164012861997E-01, 0.6667892725157643E+00, 0.7612117828169483E+00, 0.8195812739875934E+00, 0.1507277483884870E+00, 0.6683936863561600E+00, 0.1889659251016290E+00, 0.7979363461805246E+00, 0.9481151487908223E+00, 0.3017570036067704E+00, 0.2677628114499495E+00, 0.4479716416042271E+00, 0.3773062273227097E+00, 0.1756267436681674E+00, 0.6251009447636316E+00, 0.3010990216120933E+00, 0.6044916624270988E+00, 0.3012752666004853E+00, 0.7113839134297870E+00, 0.7355680503868338E+00, 0.1830572310895536E+00, 0.2065116792960049E+00, 0.2432895898830973E+00, 0.9615089470298006E+00, 0.3137165160357294E+00, 0.5335631824040450E+00, 0.8324142824771229E+00, 0.9749825715503555E+00, 0.3108065469391890E+00, 0.2584483510692804E+00, 0.5679665509067858E+00, 0.9084113884935793E+00, 0.3061643504016712E+00, 0.1856860631474661E+00, 0.4086356189591027E+00, 0.2761517708925276E+00, 0.9203155374121844E+00},
			n:    21,
			zOut: []float64{0.2651616639993538E+01, 0.2278042999629812E+01, 0.2198077975076974E+01, 0.2185608493228643E+01, 0.1866238785250178E+01, 0.1694317687161655E+01, 0.1314780820581521E+01, 0.1267230115137409E+01, 0.1165102366599241E+01, 0.1094106513205574E+01, 0.9375639320451692E+00, 0.7228525903025090E+00, 0.5063995930233953E+00, 0.4433597755655621E+00, 0.3782868675227946E+00, 0.2710178897462153E+00, 0.1447594933682902E+00, 0.1060101050946834E+00, 0.8449698031014202E-01, 0.1468593250969371E-01, 0.8406928034951682E-06, 0.3591986573571771E-20, 0.3864386735331416E-10, 0.3252595492372177E-10, 0.1314780820581521E+01, 0.1447952803137974E-21, 0.5374081097788385E-10, 0.4849713226733927E-10, 0.1267230115137409E+01, 0.2958228394578794E-29, 0.8017357434413146E-10, 0.7063808972759981E-10, 0.1165102366599241E+01, 0.1139115147139141E-26, 0.1961041727821929E-09, 0.1374026522849091E-09, 0.1094106513205574E+01, 0.3915388343685865E-19, 0.1756082891845291E-08, 0.3338672481507716E-08, 0.9375639320451692E+00, 0.4870203640784043E-16, 0.2132455639604580E+02, 0.2132455639604580E+02, 0.9900000000000000E+02, 0.2972789115646258E+01, 0.5050505050505050E+01, 0.1300256830939750E-04, 0.5063995930233953E+00, 0.4338734978715565E-29, 0.3205424704587136E-04, 0.3879495859639124E-04, 0.4433597755655621E+00, 0.9466330862652142E-25, 0.6653798840082298E-04, 0.6266988927201412E-04, 0.3782868675227946E+00, 0.2274100449667269E-22, 0.3059080514749475E-04, 0.2450157710629126E-04, 0.2710178897462153E+00, 0.2739429933906694E-24, 0.4451877567018950E-05, 0.2739439060253447E-05, 0.1447594933682902E+00, 0.1897052704875489E-25, 0.3311215248867682E-05, 0.2822048496282046E-05, 0.1060101050946834E+00, 0.7924889161564951E-19, 0.2588042287900903E-05, 0.3305698843095029E-05, 0.8449698031014202E-01, 0.2523395625690618E-18, 0.1425730854724969E-04, 0.2612802087529877E-04, 0.1468593250969371E-01, 0.7520086356827106E-20, 0.6278930605158724E-03, 0.1193309863751681E-02, 0.8406928034951682E-06, 0.0000000000000000E+00, 0.8769018979884540E-02, 0.6748334476252631E-02},
			info: 0,
		},
		{
			z:    []float64{0.5539290016733247E+00, 0.6936589108803458E+00, 0.1862325586311042E-01, 0.3903593359823143E+00, 0.1387446022374186E+00, 0.9230050933369500E+00, 0.1655735508788293E+00, 0.5464594833562775E+00, 0.4407574799078734E+00, 0.7597516703282015E+00, 0.3240675365298944E+00, 0.8527429657828770E+00, 0.6134024974884296E+00, 0.1359668624923763E+00, 0.8589771621484943E+00, 0.8334002673394481E+00, 0.3811010712979018E+00, 0.4518439634289880E+00, 0.4121953913957921E-01, 0.1499929777106017E+00, 0.7537932319194001E+00, 0.1137770685080763E+00, 0.9362285670837264E+00, 0.2284833451474525E+00, 0.4661006612092690E+00, 0.3461611111488332E+00, 0.1608705680575839E-01, 0.9250298701911358E+00, 0.5983544857783111E+00, 0.9400090024445320E+00, 0.6595514287179831E+00, 0.2725099566160494E+00, 0.6509556024164401E+00, 0.8851211780351773E+00, 0.5925872091724521E+00, 0.5318402341230010E+00, 0.3225952236300995E+00, 0.6233031538827258E+00, 0.1806586091116282E+00, 0.9476369741031940E+00, 0.6784219735316235E+00, 0.6934023884718178E+00, 0.5000312772557033E+00, 0.6725383579734943E+00, 0.6771923299216058E+00, 0.9125469473100194E+00, 0.9862018367238429E+00, 0.7259311136907298E+00, 0.9021849324334038E+00, 0.6032549715715884E+00, 0.9017706724408630E+00, 0.8975979926873651E+00, 0.5949035726420406E+00, 0.6903449880442312E+00, 0.7574844360343417E+00, 0.2889632382233942E-01, 0.9428474184445177E+00, 0.5555118914598791E+00, 0.8663544108664935E+00, 0.6853450780608091E+00, 0.1464483859238053E+00, 0.6491672315887742E+00, 0.2994712877436206E+00, 0.3101752077576794E+00, 0.4920466664329196E+00, 0.2135103260181662E+00, 0.3809190441316870E+00, 0.8437350743416491E+00, 0.5443983884818225E+00, 0.7426189539459086E+00, 0.1055227287563778E+00, 0.3059118205598027E+00, 0.8189910523272392E+00, 0.9773505795713493E+00, 0.7305661438576656E+00, 0.6062516615534109E+00, 0.4660033490547544E+00, 0.5413353206637471E+00, 0.2388208915142312E+00, 0.6428463909118429E+00, 0.2982699820336984E+00, 0.2856298024316706E-01, 0.5487207914459959E+00, 0.4464180688275057E+00},
			n:    21,
			zOut: []float64{0.2448593467642387E+01, 0.2141330855004301E+01, 0.2126430153924173E+01, 0.1907639051889359E+01, 0.1812500110390200E+01, 0.1510069013602020E+01, 0.1441897976798092E+01, 0.1263171337642769E+01, 0.1138107892295268E+01, 0.1088450894719558E+01, 0.9398511957468885E+00, 0.8468227425873083E+00, 0.6124964740092524E+00, 0.5436278175488470E+00, 0.4066256939946141E+00, 0.3972422051503922E+00, 0.2390787026361968E+00, 0.4848296808782174E-01, 0.2905803980602126E-01, 0.1302961362478940E-02, 0.3616542154739030E-05, 0.3187755557432356E-18, 0.8836722322771784E-05, 0.6065531139006927E-05, 0.1441897976798092E+01, 0.7730421460348975E-16, 0.3574427637238573E-07, 0.1935178018527985E-07, 0.1263171337642769E+01, 0.8058524406205312E-18, 0.1418632976070110E-05, 0.4069026170914173E-05, 0.1138107892295268E+01, 0.1371633476674845E-23, 0.4721385684453755E-04, 0.3623692584267971E-04, 0.1088450894719558E+01, 0.1790018845564752E-20, 0.4055026077404345E-04, 0.2446096735206838E-04, 0.9398511957468885E+00, 0.3178330447896772E-25, 0.2094278317138010E+02, 0.2094278317138010E+02, 0.1040000000000000E+03, 0.3031746031746032E+01, 0.4807692307692307E+01, 0.8138465546053722E-05, 0.6124964740092524E+00, 0.1019617235038814E-22, 0.4736809894069740E-03, 0.1915264360566004E-03, 0.5436278175488470E+00, 0.2004410489406572E-21, 0.1259085689345719E-03, 0.2338760337361548E-03, 0.4066256939946141E+00, 0.1231392444306089E-15, 0.6226403214695822E-04, 0.7644683470656998E-04, 0.3972422051503922E+00, 0.5914962592392188E-17, 0.1887068321547840E-03, 0.1345267246510674E-03, 0.2390787026361968E+00, 0.1936824017247308E-20, 0.7333182971373067E-03, 0.1062997506116107E-02, 0.4848296808782174E-01, 0.3799154119544393E-26, 0.1269933545612727E-03, 0.1297822869175177E-03, 0.2905803980602126E-01, 0.8465676581597317E-24, 0.5642309440908588E-03, 0.2732446225021143E-03, 0.1302961362478940E-02, 0.2467375723434259E-18, 0.9177892725485271E-03, 0.8034707353774361E-03, 0.3616542154739030E-05, 0.2761013168273541E-28, 0.6531167933367503E-03, 0.6727959129527735E-03},
			info: 0,
		},
		{
			z:    []float64{0.7455810318731756E+00, 0.1881309445499726E+00, 0.6290241532486281E+00, 0.9426231091333456E+00, 0.1402641401457146E+00, 0.5545071521563689E+00, 0.3467970399972181E+00, 0.6378935846273492E-01, 0.8187242451993508E+00, 0.8444039716090014E+00, 0.8807815832461214E+00, 0.6871811349512845E+00, 0.6041217734774926E+00, 0.2285364565760638E+00, 0.6287288909172152E+00, 0.5441550558534458E+00, 0.6062929607453951E+00, 0.1528830611582420E+00, 0.7289323622783690E+00, 0.8693274251763169E+00, 0.1210720262902459E+00, 0.4752572018677603E+00, 0.8160358228459934E+00, 0.5003926181135285E+00, 0.2800920281530351E+00, 0.3817159580569316E+00, 0.1419563352692587E+00, 0.9738793587569783E+00, 0.2402077997739175E+00, 0.5021080238100061E+00, 0.5325521311583831E+00, 0.7002793445871702E+00, 0.2004913666518293E-01, 0.6858750037076770E+00, 0.5705320248969311E+00, 0.6000416876176061E+00, 0.9254592880635680E+00, 0.2987366812581649E+00, 0.8838368946481180E+00, 0.7495294261248863E+00, 0.2516401660161148E+00, 0.5770724542103510E+00, 0.8689432882806168E+00, 0.3108844333247283E+00, 0.7611598373381380E+00, 0.9533545884676758E+00, 0.6146629999183371E+00, 0.7337933880625785E+00, 0.1335018938357140E+00, 0.4054745880121539E+00, 0.9816031767048012E+00, 0.5190257866591561E+00, 0.9457220484783406E+00, 0.2563725588490263E+00, 0.8953616129834293E+00, 0.1343673038869742E+00, 0.1198867110907023E+00, 0.7765966504091196E+00, 0.1685346783514826E+00, 0.9322265874533907E+00, 0.4968937019786546E+00, 0.3933065437909874E+00, 0.7046190939244956E-01, 0.5772052710604483E+00, 0.1220174671595003E+00, 0.3586914192309758E+00, 0.4743117898783903E+00, 0.1205436116155321E+00, 0.2068106627971966E-01, 0.5035688415619853E+00, 0.2656792568844590E-01, 0.4951625786650252E+00, 0.1600264513881963E+00, 0.4218870487180432E+00, 0.4847863747130776E-01, 0.9478135093620923E+00, 0.1811694594266104E+00, 0.5311488460048615E+00, 0.2296061187775216E+00, 0.9932681440344262E+00, 0.2007921586496573E+00, 0.9684478357621775E+00, 0.9322927111902295E+00, 0.6234102172880590E+00},
			n:    21,
			zOut: []float64{0.2617080551859897E+01, 0.2305187568066598E+01, 0.1942735806791493E+01, 0.1834812272403632E+01, 0.1760060933961288E+01, 0.1720372963502770E+01, 0.1676173250234917E+01, 0.1530300579122039E+01, 0.1214313538868506E+01, 0.1146550683576815E+01, 0.9555081169996513E+00, 0.8235353939361046E+00, 0.7281321711646775E+00, 0.5964515885643180E+00, 0.4105953672636856E+00, 0.2154672622243388E+00, 0.1469861329023305E+00, 0.1350631498164301E+00, 0.8120368457133148E-01, 0.1548517647968005E-01, 0.1861612641330624E-04, 0.6389970231972139E-21, 0.6259229043737505E-18, 0.1065975851985723E-17, 0.1676173250234917E+01, 0.1398996249446581E-17, 0.1485385179581525E-16, 0.5390878377546583E-16, 0.1530300579122039E+01, 0.1934286939601921E-26, 0.4995235293268131E-13, 0.1480097232724468E-13, 0.1214313538868506E+01, 0.1393981231259665E-11, 0.4031414073293921E-10, 0.9065983401759033E-10, 0.1146550683576815E+01, 0.7754818242684634E-25, 0.9068533370171412E-09, 0.1932480064220526E-08, 0.9555081169996513E+00, 0.5937598997073433E-12, 0.2185603480843691E+02, 0.2185603480843692E+02, 0.1010000000000000E+03, 0.3183673469387755E+01, 0.2970297029702970E+01, 0.4106209724092858E-05, 0.7281321711646775E+00, 0.2221432309102369E-26, 0.2012416535197213E-04, 0.1179395510019159E-04, 0.5964515885643180E+00, 0.6492360922940637E-17, 0.6055695087058826E-05, 0.1676327789285107E-04, 0.4105953672636856E+00, 0.1174834768927548E-24, 0.8526024871833974E-05, 0.9608606324255413E-05, 0.2154672622243388E+00, 0.1186446801452402E-26, 0.9887524717240814E-05, 0.1072842889916555E-04, 0.1469861329023305E+00, 0.0000000000000000E+00, 0.1666923382276663E-04, 0.2139923074663986E-04, 0.1350631498164301E+00, 0.4860988391194038E-18, 0.4968094868839375E-08, 0.1325007303324406E-08, 0.8120368457133148E-01, 0.5476819983069490E-18, 0.5501364953991437E-04, 0.1068949633822309E-04, 0.1548517647968005E-01, 0.6621786174648700E-13, 0.4934492558904634E-03, 0.7815729865935395E-03, 0.1861612641330624E-04, 0.0000000000000000E+00, 0.2047621357235247E-01, 0.1218736604744046E-01},
			info: 0,
		},
		{
			z:    []float64{0.1913768959569714E+00, 0.7347223265181069E+00, 0.9087350122086006E+00, 0.8876448886267929E-02, 0.1794058368310475E+00, 0.3375830657319635E+00, 0.4153249303964603E+00, 0.3079259326141542E+00, 0.5095638682609168E+00, 0.6300946130469818E+00, 0.5367870439046851E-02, 0.6082577193589970E+00, 0.6096872650047096E+00, 0.8098482943683755E+00, 0.4018780482667224E+00, 0.1612961166583111E+00, 0.4165836183710623E+00, 0.6711774659609234E+00, 0.1871437048914691E+00, 0.2043648411447756E+00, 0.4603921058522200E+00, 0.5138380788557162E-01, 0.3299651081607302E+00, 0.5178779891127856E+00, 0.8488474652006981E+00, 0.4000528745818374E+00, 0.7038372186752184E-01, 0.1091866126814279E+00, 0.4601541814804277E-01, 0.2814862519917873E+00, 0.5969661695911915E+00, 0.1111325580672384E+00, 0.1377964900539917E+00, 0.4488655138014651E+00, 0.5793089439934721E+00, 0.4068390675279384E+00, 0.3141858292757815E-01, 0.3803058398371814E+00, 0.6865263808463873E-01, 0.6565571596516916E+00, 0.4227763797508006E+00, 0.9281332433572439E+00, 0.2549706813172651E+00, 0.1472316879439791E+00, 0.8389980826186394E+00, 0.4949206978487660E+00, 0.8778524239605636E+00, 0.8125876339222501E+00, 0.6384442407604712E+00, 0.6297206683503800E+00, 0.1074594814776241E+00, 0.4635106216187717E+00, 0.2149027083261391E+00, 0.2926633791426133E+00, 0.8718806127632718E+00, 0.9358351753143842E+00, 0.5812389276262170E+00, 0.8361764419241092E+00, 0.1334582685582402E+00, 0.6700349085889619E+00, 0.1370175035793201E+00, 0.2605729802823288E+00, 0.7055670307426516E+00, 0.1974656950667419E-01, 0.9516894704106690E+00, 0.7509460514650641E+00, 0.9770872584819335E-01, 0.1679929405438133E+00, 0.2605432695744189E+00, 0.7255281751885829E+00, 0.2060091110826470E+00, 0.5123028703888126E+00, 0.5392241233948379E+00, 0.3215743887975069E+00, 0.4306560982435532E+00, 0.9326432909148183E+00, 0.1891146429259456E+00, 0.5585690444839775E+00, 0.8103752159402208E+00, 0.3850798219907741E+00, 0.6027394925107610E-01, 0.6960376568363590E+00, 0.6132631218829975E+00, 0.5859904896405407E+00},
			n:    21,
			zOut: []float64{0.1948845699194504E+01, 0.1757754653408917E+01, 0.1739740003489702E+01, 0.1427189755042558E+01, 0.1354312190162734E+01, 0.1300861212306309E+01, 0.1144086431430055E+01, 0.9728683413380136E+00, 0.7924660843958955E+00, 0.7190108168957414E+00, 0.5334843173898538E+00, 0.4234595469843583E+00, 0.3840099969172215E+00, 0.2310167712576353E+00, 0.1706582281146164E+00, 0.1481114719972525E+00, 0.9844570800522298E-01, 0.8643465546688130E-01, 0.2116850399857309E-01, 0.1490679637756442E-02, 0.9506251872618699E-05, 0.0000000000000000E+00, 0.3139247800552953E-11, 0.1114956279480716E-10, 0.1144086431430055E+01, 0.2611946364919696E-16, 0.1910542599593587E-05, 0.1867627468366115E-05, 0.9728683413380136E+00, 0.3548331964791209E-13, 0.9422609200466975E-06, 0.3861796048985941E-05, 0.7924660843958955E+00, 0.4543838814073028E-27, 0.1545075054473009E-06, 0.2314398440634665E-06, 0.7190108168957414E+00, 0.1033975765689929E-24, 0.9239329035775786E-07, 0.1002180390059042E-06, 0.5334843173898538E+00, 0.0000000000000000E+00, 0.1525542457368567E+02, 0.1525542457368567E+02, 0.1120000000000000E+03, 0.3308390022675737E+01, 0.5357142857142857E+01, 0.1175851227353848E-05, 0.3840099969172215E+00, 0.1009741958682895E-27, 0.6285056589702800E-05, 0.7777675415159152E-05, 0.2310167712576353E+00, 0.4642054879586336E-19, 0.2707022647265855E-04, 0.1462219920606713E-04, 0.1706582281146164E+00, 0.0000000000000000E+00, 0.1442808298792521E-03, 0.1499267314065209E-03, 0.1481114719972525E+00, 0.1784642171583157E-21, 0.9060208842449760E-06, 0.2604899516157433E-05, 0.9844570800522298E-01, 0.3285517707584525E-17, 0.4009882974094740E-08, 0.9813054658490708E-08, 0.8643465546688130E-01, 0.1441804758967627E-17, 0.1005867330167897E-05, 0.3548177460196556E-06, 0.2116850399857309E-01, 0.1161145423573808E-16, 0.4264476922944564E-04, 0.6523019800084465E-04, 0.1490679637756442E-02, 0.5282426871388410E-23, 0.4120097727637851E-03, 0.5256559484655862E-03, 0.9506251872618699E-05, 0.7858215819253763E-23, 0.4122429400478702E-03, 0.4195795458123320E-03},
			info: 0,
		},
		{
			z:    []float64{0.8707866955342474E+00, 0.7817747598907121E-01, 0.3395074897480099E-01, 0.4299950297507704E+00, 0.8088090483755312E+00, 0.8059665333356031E+00, 0.8715468955879169E-01, 0.7726332912417854E+00, 0.8113440783694211E+00, 0.3884398998833286E+00, 0.2376236729669906E+00, 0.1163685212573567E+00, 0.9165435854442110E+00, 0.6267984817484763E+00, 0.9624876615834328E+00, 0.3034180669232360E+00, 0.2061796047621195E+00, 0.5768621491208505E+00, 0.9925917678590149E+00, 0.6443157924817072E+00, 0.5967958277754652E+00, 0.6529032825080359E+00, 0.2463148548215904E+00, 0.3434579503978075E+00, 0.9563439871443443E+00, 0.2121028540747388E+00, 0.7195547303990343E+00, 0.2508441805243417E+00, 0.1006260142041168E+00, 0.8893309100691102E+00, 0.5028754540045453E+00, 0.8271029410166070E+00, 0.1744058577319449E+00, 0.1497291367846938E+00, 0.9760674104574272E+00, 0.1217237467828105E+00, 0.5182991497218187E+00, 0.9638099804000833E-01, 0.3920754919448237E+00, 0.5405769677178972E+00, 0.2099631365577761E+00, 0.7305783624173950E+00, 0.5230361650507421E+00, 0.3249231142539155E+00, 0.6943021078034506E+00, 0.8039502137729849E+00, 0.5430939865698426E+00, 0.5802454813209417E+00, 0.4521353811622019E+00, 0.6022646695138674E+00, 0.5085578584349214E+00, 0.2084880202460930E+00, 0.7893764544854661E+00, 0.8499187425465698E+00, 0.6507732600828079E+00, 0.6030189256946239E+00, 0.2748834677816949E+00, 0.7991444375081345E+00, 0.3953548021356785E+00, 0.8063610109500122E+00, 0.9322494176165492E+00, 0.2295756744488692E+00, 0.9380170922026533E+00, 0.6349173046266506E+00, 0.8138238710526029E+00, 0.1723717901392456E+00, 0.7436778507578146E+00, 0.9748296910217159E+00, 0.9719004334667378E+00, 0.1097372829332979E+00, 0.6782829454939702E+00, 0.5097404868982898E+00, 0.9485025358778609E+00, 0.7313319469015522E-01, 0.6156780373408383E+00, 0.8764490831370680E+00, 0.2737186188000360E+00, 0.3500606337779347E+00, 0.7029831161979777E+00, 0.1747477098480691E+00, 0.8887436470613648E+00, 0.8441051781034927E+00, 0.3534934185139980E+00, 0.9278581973721739E+00},
			n:    21,
			zOut: []float64{0.2399590582568773E+01, 0.2223029747781157E+01, 0.2028465681071774E+01, 0.1928601105940012E+01, 0.1734632538043284E+01, 0.1705587556623443E+01, 0.1335586809119933E+01, 0.1201248979220471E+01, 0.1103973785860777E+01, 0.9516762968928613E+00, 0.8501027578146555E+00, 0.7334218559333251E+00, 0.5694462065979470E+00, 0.4616704914677346E+00, 0.3869349008726894E+00, 0.2370229392379994E+00, 0.1895639186923105E+00, 0.6837204156138489E-01, 0.3336561258733767E-01, 0.4796873010443177E-02, 0.8309969413683648E-03, 0.1974829088984998E-20, 0.1178494024155901E-08, 0.8095343015842228E-09, 0.1335586809119933E+01, 0.2220446049250313E-15, 0.1020576889344504E-06, 0.1802026823995536E-06, 0.1201248979220471E+01, 0.0000000000000000E+00, 0.6243063866214613E-06, 0.3862955415798747E-06, 0.1103973785860777E+01, 0.6981113564263453E-22, 0.3544097463307990E-10, 0.1357364267827356E-10, 0.9516762968928613E+00, 0.9072531498765813E-25, 0.7979152396248553E-09, 0.3946058728874109E-09, 0.8501027578146555E+00, 0.3130538746543124E-14, 0.2014792167783968E+02, 0.2014792167783968E+02, 0.1020000000000000E+03, 0.3129251700680272E+01, 0.5882352941176471E+01, 0.5818593578868780E-05, 0.5694462065979470E+00, 0.1588643710672175E-16, 0.4031382989989102E-05, 0.4054337523014161E-05, 0.4616704914677346E+00, 0.4437342591868191E-30, 0.3922192530785082E-05, 0.4209867122852495E-05, 0.3869349008726894E+00, 0.1490631566505624E-25, 0.4185837521330395E-06, 0.8450171931154743E-06, 0.2370229392379994E+00, 0.9509832565716118E-20, 0.2486232683772935E-03, 0.1020105306066691E-03, 0.1895639186923105E+00, 0.4152563805083406E-26, 0.4122104994398350E-03, 0.2603466333914367E-03, 0.6837204156138489E-01, 0.2482703040911568E-24, 0.1451022916339763E-02, 0.8718585538326752E-03, 0.3336561258733767E-01, 0.7888609052210118E-30, 0.3124612722893384E-02, 0.2800832977531187E-02, 0.4796873010443177E-02, 0.5945360652724886E-23, 0.1330631700030230E-06, 0.1700369028500688E-07, 0.8309969413683648E-03, 0.0000000000000000E+00, 0.1665788949608015E-02, 0.1927727749653244E-02},
			info: 0,
		},
		{
			z:    []float64{0.5557143146495949E+00, 0.5989968816903124E+00, 0.2940668070493605E+00, 0.6865591553636113E+00, 0.2538362881116337E+00, 0.8562930572057048E-01, 0.8935270452331215E+00, 0.2427258280604294E-01, 0.8872146530392093E+00, 0.8383596820641517E+00, 0.7298494135137694E+00, 0.5456106579379609E+00, 0.5284508278981448E+00, 0.4551015440756836E+00, 0.8310297436868961E+00, 0.2533050402685522E+00, 0.8204820449217906E+00, 0.3961557150240700E+00, 0.9768702237400030E+00, 0.2370292739142171E+00, 0.2559318188841475E-02, 0.9750517337670606E-01, 0.4332045279801727E+00, 0.9393777930670477E+00, 0.1401411199977110E-01, 0.6412771440674316E+00, 0.3992540155849937E+00, 0.6710909099058460E+00, 0.7286319488310415E+00, 0.2796269361967505E+00, 0.7441899058930037E+00, 0.6626826587793098E+00, 0.9214594500725222E+00, 0.3161807402236700E+00, 0.5522479249937589E+00, 0.8328895958825197E+00, 0.3188368475648113E+00, 0.7495883496609020E+00, 0.7525354981370723E+00, 0.4819658709067065E-01, 0.8655147680740739E+00, 0.9438778396406793E+00, 0.2488475915004000E+00, 0.6823535213934872E+00, 0.1937525404626215E+00, 0.5005903164733780E+00, 0.8863816362271992E-01, 0.8279647895785316E+00, 0.7576076924432378E+00, 0.7026499641621615E+00, 0.4430381204856144E+00, 0.7250369983032848E+00, 0.6848785243425647E+00, 0.4460397323983623E+00, 0.4708648248777212E+00, 0.2715064773162834E+00, 0.3027887490966231E+00, 0.7428389405348396E+00, 0.4450045451144832E+00, 0.4683793136386452E+00, 0.2207684946192057E+00, 0.2427030064074031E+00, 0.9554840783351275E+00, 0.4627007756736100E+00, 0.2498171419709001E+00, 0.6457883819068694E+00, 0.9852852516910280E+00, 0.1549706320399313E-01, 0.5863698153730145E+00, 0.8639664213849532E+00, 0.3535803218615043E+00, 0.5300900503312423E+00, 0.6390014357018353E+00, 0.7961857807391770E+00, 0.2691637990251612E+00, 0.5302467438659471E+00, 0.3050412123368091E+00, 0.6256487701731338E+00, 0.3334512397543001E+00, 0.4160884062357342E+00, 0.1255438899512573E+00, 0.1645317210625844E+00, 0.3238197294562333E+00, 0.8765325383521217E+00},
			n:    21,
			zOut: []float64{0.2427342887003738E+01, 0.2350720031505609E+01, 0.2023223336640000E+01, 0.1998000565720247E+01, 0.1766392007527096E+01, 0.1571733737439641E+01, 0.1560415852981921E+01, 0.1417948678305651E+01, 0.1258973312710257E+01, 0.1056655934390461E+01, 0.8835393928887348E+00, 0.8723851974669989E+00, 0.7820423442972025E+00, 0.7419840433476598E+00, 0.4998280519834369E+00, 0.4045332331022956E+00, 0.9795545665796999E-01, 0.6816976350034422E-01, 0.5040003524610775E-01, 0.3066447317814986E-01, 0.1707038609221110E-04, 0.0000000000000000E+00, 0.7788285758834551E-10, 0.4967814553062384E-10, 0.1560415852981921E+01, 0.2584939414223288E-25, 0.2197123097785426E-09, 0.1273532664170934E-09, 0.1417948678305651E+01, 0.1666074231826777E-26, 0.1275101209751314E-13, 0.3690153580151478E-13, 0.1258973312710257E+01, 0.3205217501722064E-17, 0.9823174457391989E-10, 0.2521354047663225E-09, 0.1056655934390461E+01, 0.7258232531419596E-14, 0.3712131332588551E-06, 0.9470624651125662E-06, 0.8835393928887348E+00, 0.1013078509970455E-14, 0.2186292540627962E+02, 0.2186292540627962E+02, 0.1200000000000000E+03, 0.3639455782312925E+01, 0.8333333333333334E+01, 0.8044662327724623E-11, 0.7820423442972025E+00, 0.1609007833004439E-16, 0.1435272653873044E-05, 0.2709954877521105E-06, 0.7419840433476598E+00, 0.7040565915227063E-17, 0.3841536774665796E-05, 0.2871402006582635E-05, 0.4998280519834369E+00, 0.2100142105025380E-22, 0.2079096157763567E-04, 0.3245111531694086E-04, 0.4045332331022956E+00, 0.1391829401523057E-17, 0.6737785910120600E-04, 0.9764140649613147E-04, 0.9795545665796999E-01, 0.0000000000000000E+00, 0.6779864796214124E-03, 0.3130513090482015E-03, 0.6816976350034422E-01, 0.2326445472805242E-24, 0.3547436816484852E-02, 0.2240007954218211E-02, 0.5040003524610775E-01, 0.1487168151587342E-22, 0.3710761871988309E-06, 0.2590766399899962E-07, 0.3066447317814986E-01, 0.7395570986446986E-31, 0.2806228106658499E-03, 0.1300265719680322E-02, 0.1707038609221110E-04, 0.5107323294632100E-21, 0.5584126334680110E-02, 0.2863707539154273E-03},
			info: 0,
		},
		{
			z:    []float64{0.8091224458140820E+00, 0.8008972278830493E+00, 0.2578176398184776E+00, 0.1256248089219657E+00, 0.6581359192692769E+00, 0.2304361454551056E+00, 0.7582551898966047E+00, 0.1757884371165862E+00, 0.3186400696162690E+00, 0.7479165768101737E+00, 0.3726824047207358E+00, 0.8797387692666324E+00, 0.5750380180879821E+00, 0.5660816893967801E+00, 0.4438713938433396E+00, 0.1758644078670230E+00, 0.5631776732547016E+00, 0.3584358301396408E+00, 0.9436101806256550E+00, 0.5626737326978751E+00, 0.6424649400467324E+00, 0.8481112336470399E+00, 0.5655978701315552E+00, 0.8705727446437589E+00, 0.2336253154027212E-01, 0.5296192722406249E+00, 0.2416414888807097E+00, 0.8901032371078870E+00, 0.6213866196646989E+00, 0.7130035148592161E+00, 0.7957781913464947E+00, 0.8476792055481847E+00, 0.9456821938668590E+00, 0.9119251640705257E+00, 0.6793496748825844E+00, 0.1959111178023272E+00, 0.1712010119398182E+00, 0.1826738976471362E+00, 0.2526159624017289E+00, 0.1636253698204285E+00, 0.5186411299293459E+00, 0.9760895321955887E-01, 0.6893646610477002E+00, 0.9919227436537437E-01, 0.3790249398464486E+00, 0.9060767708258410E+00, 0.1329141344440885E+00, 0.8748053076614649E+00, 0.3613991427617499E+00, 0.1317424493721138E+00, 0.1676759484196766E-01, 0.7195688942861501E+00, 0.7508197149131951E+00, 0.1084062966598669E+00, 0.7799888885196686E+00, 0.8731271868042269E+00, 0.7281667421703623E+00, 0.5506096404209293E+00, 0.2344365678709384E+00, 0.8253982006984435E+00, 0.6925029659724733E+00, 0.2137159289768908E+00, 0.6304949708624844E+00, 0.2263541700174215E+00, 0.3731966957484361E+00, 0.4056710443895672E-01, 0.1518575777839423E+00, 0.7000182640684851E+00, 0.5207195688041865E+00, 0.3546513077756396E+00, 0.5223091585165126E+00, 0.1312526670646489E+00, 0.6075518716591177E+00, 0.7815133160786110E+00, 0.1466092133856621E+00, 0.6908403097208041E+00, 0.5289383481177163E+00, 0.3060973141340825E+00, 0.2719751878023001E+00, 0.4367099092855503E+00, 0.1678669773704986E+00, 0.2384522838837465E+00, 0.7578135131488263E+00, 0.2260651503779529E+00},
			n:    21,
			zOut: []float64{0.3001718609114466E+01, 0.2461580649918252E+01, 0.2139800724137603E+01, 0.2110198913058292E+01, 0.1773285881581521E+01, 0.1731854037718996E+01, 0.1434868444776101E+01, 0.1292334845681036E+01, 0.1202898865215905E+01, 0.9074690574580867E+00, 0.8145281637318031E+00, 0.6833997197084192E+00, 0.6303801796404412E+00, 0.5115989792032534E+00, 0.4365028215455523E+00, 0.3347059637856380E+00, 0.2704761845997483E+00, 0.1007567293499011E+00, 0.7404356572986662E-01, 0.2225224658576705E-01, 0.1003499792335912E-03, 0.1163915500165708E-19, 0.5702083577853931E-06, 0.6235005324467382E-06, 0.1434868444776101E+01, 0.7352183636659830E-27, 0.3221030091629643E-04, 0.6932009635225803E-04, 0.1292334845681036E+01, 0.1547654926086715E-20, 0.2859096049606110E-03, 0.1689403705904475E-03, 0.1202898865215905E+01, 0.5187952724823741E-19, 0.3457740509453089E-04, 0.1548723988845099E-04, 0.9074690574580867E+00, 0.1883346903736854E-18, 0.4924250006037621E-06, 0.3064183963004930E-06, 0.8145281637318031E+00, 0.7436271633171000E-16, 0.2193475493251988E+02, 0.2193475493251988E+02, 0.9400000000000000E+02, 0.2913832199546485E+01, 0.3191489361702128E+01, 0.6928384721316203E-06, 0.6303801796404412E+00, 0.4344707644406440E-17, 0.8243004381261832E-06, 0.1242944619712133E-05, 0.5115989792032534E+00, 0.1321342016245195E-28, 0.6830327722875946E-05, 0.4610227507804771E-05, 0.4365028215455523E+00, 0.7922538128148438E-16, 0.5185315150043044E-04, 0.9574812800552337E-04, 0.3347059637856380E+00, 0.4042912139257686E-29, 0.6190562688247171E-03, 0.1111720505727282E-02, 0.2704761845997483E+00, 0.1659171176970474E-17, 0.3895087988050344E-03, 0.1929449868539905E-03, 0.1007567293499011E+00, 0.2919870821923328E-23, 0.2088900906409071E-03, 0.1786849109209585E-03, 0.7404356572986662E-01, 0.6310887241768094E-28, 0.1266951898160773E-03, 0.1217799054154213E-03, 0.2225224658576705E-01, 0.3697785493223493E-31, 0.1828672832615040E-03, 0.2196698921083079E-03, 0.1003499792335912E-03, 0.0000000000000000E+00, 0.1256202468809242E-02, 0.9158129895566399E-03},
			info: 0,
		},
		{
			z:    []float64{0.7473165994595915E+00, 0.9383144044637891E+00, 0.9705766820209645E+00, 0.3093136353727625E+00, 0.7158638842786125E+00, 0.8927477442087851E+00, 0.5846837014009232E-02, 0.2823227977987499E+00, 0.4656399961948744E+00, 0.1783480145929806E-01, 0.8517241739910267E+00, 0.2968236367983081E+00, 0.3828703962512207E+00, 0.1618551942929359E+00, 0.8422815023410849E+00, 0.9667213467147939E+00, 0.1872774458350456E+00, 0.9673785609387944E+00, 0.1668393270938346E+00, 0.1793363294699661E+00, 0.3882509817575230E+00, 0.9322497724034516E+00, 0.1678634979113953E+00, 0.4000730353927472E+00, 0.1370854040387474E+00, 0.1383728853360747E+00, 0.8728086099072359E+00, 0.7142605368034651E+00, 0.1298497843576266E+00, 0.4890695005398649E+00, 0.6561831309305255E+00, 0.2551502145039813E+00, 0.3137603328050955E+00, 0.4725573834392259E+00, 0.1864194559915778E+00, 0.2476903242862807E+00, 0.3016682498100809E-01, 0.6028908129301045E+00, 0.9699532639034771E+00, 0.6804787742392342E+00, 0.5085224418131328E+00, 0.8160312832212636E+00, 0.4537844508149779E+00, 0.7698002058576117E+00, 0.9102837491884663E-01, 0.7957963901796614E+00, 0.4129806172970929E+00, 0.1183929303892000E+00, 0.7231979382284242E+00, 0.6908050716155305E+00, 0.9129211251463741E+00, 0.8407226028232299E+00, 0.7511043357392063E+00, 0.6614037096770188E+00, 0.5816383867961591E+00, 0.9508597818692400E+00, 0.6671834981267941E+00, 0.5353731249555691E+00, 0.8270836995328532E+00, 0.4748649574057349E+00, 0.7310961648034807E+00, 0.5329698192925740E+00, 0.5537463638220994E+00, 0.8202244086458278E+00, 0.3089778794417031E-01, 0.8347421543151022E+00, 0.7341688304615497E+00, 0.6293051268804178E+00, 0.9685330131023850E-01, 0.2218306173097127E+00, 0.4005982063157664E+00, 0.8004209862846372E+00, 0.9322471342744776E-02, 0.5252230232113276E+00, 0.8078446044346348E+00, 0.8663675770089085E-02, 0.4174348230921995E+00, 0.5316704871571061E+00, 0.8080873214304936E+00, 0.3191328650837369E+00, 0.2265528787275422E+00, 0.1299687059183835E+00, 0.5606006275148095E+00, 0.2716326223111132E+00},
			n:    21,
			zOut: []float64{0.2557856639061994E+01, 0.2221569823028635E+01, 0.2113175506056959E+01, 0.1791034203034858E+01, 0.1626866366713385E+01, 0.1553853827140545E+01, 0.1378038738404736E+01, 0.1313711478279983E+01, 0.1103021858691488E+01, 0.8924372074434684E+00, 0.7465727975101516E+00, 0.5658880123536838E+00, 0.5427616116084486E+00, 0.3934406603234193E+00, 0.3139076008306549E+00, 0.2922956394268174E+00, 0.1324157743308766E+00, 0.8492495339426936E-01, 0.1726019296564064E-01, 0.1478900502211735E-02, 0.3047316799617542E-04, 0.0000000000000000E+00, 0.2209423397088722E-18, 0.1230851536694111E-18, 0.1378038738404736E+01, 0.4653537180464338E-22, 0.4724643033910209E-17, 0.2943893377745906E-17, 0.1313711478279983E+01, 0.2193156221990361E-16, 0.1119006893822957E-16, 0.1741431705208298E-16, 0.1103021858691488E+01, 0.2831227495869906E-12, 0.2860829853751284E-15, 0.4129571258389818E-15, 0.8924372074434684E+00, 0.1711229892219076E-22, 0.2249832739645017E-14, 0.3254005212313171E-14, 0.7465727975101516E+00, 0.5231751935973153E-19, 0.1964254226427022E+02, 0.1964254226427022E+02, 0.1110000000000000E+03, 0.3285714285714286E+01, 0.8108108108108109E+01, 0.1373504512438542E-12, 0.5427616116084486E+00, 0.0000000000000000E+00, 0.2663664432452481E-11, 0.1451131289962852E-11, 0.3934406603234193E+00, 0.3021704178298789E-14, 0.8991877456137337E-10, 0.1576401145017008E-09, 0.3139076008306549E+00, 0.2274554202459952E-24, 0.7202855721215940E-08, 0.4214854283718170E-08, 0.2922956394268174E+00, 0.3975858962313899E-27, 0.6024353396932969E-07, 0.1022672043122323E-06, 0.1324157743308766E+00, 0.6969933701296916E-21, 0.1098572510799548E-05, 0.1404105143401733E-05, 0.8492495339426936E-01, 0.6511321020566649E-24, 0.5604469692434887E-06, 0.6104659421650381E-06, 0.1726019296564064E-01, 0.9860761315262648E-31, 0.7747760678878360E-06, 0.1026260244983655E-05, 0.1478900502211735E-02, 0.1334496117765360E-28, 0.5102973157105362E-05, 0.1258609455819239E-04, 0.3047316799617542E-04, 0.0000000000000000E+00, 0.4753862314998056E-03, 0.1863315015192667E-02},
			info: 0,
		},
		{
			z:    []float64{0.2548672807470598E-01, 0.1495421137012933E+00, 0.6637277283391485E-01, 0.6233200993667132E-01, 0.4958061473131723E+00, 0.9804571594196556E+00, 0.5741929121502225E+00, 0.9319350517768583E+00, 0.5154498317218548E-01, 0.4165283024496188E+00, 0.9821854184367015E+00, 0.7510033276340442E+00, 0.2648151368970406E+00, 0.9698327797382789E-01, 0.1433813304773114E+00, 0.1687421369042604E+00, 0.6230509959313209E+00, 0.8106153730403777E+00, 0.5719487513699320E+00, 0.5761793262532514E+00, 0.4983158430615118E+00, 0.8860830618783583E+00, 0.7136722399358287E+00, 0.6310350013419201E+00, 0.5509601824732582E+00, 0.5806336147624858E-01, 0.1818648959869262E+00, 0.2346005241916651E+00, 0.5346851174706903E+00, 0.4444466121668155E+00, 0.8418378722202517E+00, 0.6164532479196363E+00, 0.4298553148227666E+00, 0.2296849794057675E+00, 0.3620982682376475E+00, 0.3645096805309075E+00, 0.9980228064491022E+00, 0.7106112428543574E+00, 0.6651697421094976E+00, 0.9830151455980368E+00, 0.3483046917749841E+00, 0.3668589729383628E+00, 0.5655614401040915E+00, 0.8891526728148491E+00, 0.9164649086142118E+00, 0.4065736846475075E+00, 0.1470436468080603E+00, 0.5802557267939668E+00, 0.8272597245263099E+00, 0.7163617706554632E+00, 0.3488790096540706E+00, 0.6404227109073246E+00, 0.1472796557920839E+00, 0.8764536863696604E-02, 0.5020645462548778E-01, 0.6906675094909457E+00, 0.4668434450047272E+00, 0.7886435616506033E+00, 0.8775817704321313E+00, 0.4246952359284411E+00, 0.1646609549881360E+00, 0.6697681588293083E+00, 0.7914692739599321E+00, 0.1396674072195736E-01, 0.5167232939537366E+00, 0.9374340964740459E+00, 0.4228531365925756E+00, 0.9727064517088465E+00, 0.5765501520393458E+00, 0.5648508784605104E+00, 0.3049637657479770E+00, 0.3063190445223624E+00, 0.5204528216889723E+00, 0.8180178920731451E+00, 0.4938773599630263E+00, 0.8975563911549045E-01, 0.5983688838646251E+00, 0.3698925131813697E+00, 0.2992530973981942E+00, 0.7463539101278609E+00, 0.8005980511688328E+00, 0.7944834752945218E-01, 0.3132925544163919E+00, 0.8866951016419623E+00},
			n:    21,
			zOut: []float64{0.2548744028948828E+01, 0.2439584200134194E+01, 0.2276898263864132E+01, 0.2083856178014486E+01, 0.2039342663979975E+01, 0.1949418505713503E+01, 0.1425638870874207E+01, 0.1087729835971203E+01, 0.9667696375323971E+00, 0.8022433247488755E+00, 0.5846254874613492E+00, 0.5355025899480791E+00, 0.3940805795947064E+00, 0.2663780336330241E+00, 0.2413081251593111E+00, 0.1817740817404329E+00, 0.1073144197241267E+00, 0.4327520903789019E-01, 0.3141654146246065E-01, 0.2017433957164045E-01, 0.3181705377209195E-03, 0.2778526523712400E-12, 0.1839247883012458E-10, 0.2248848580297948E-10, 0.1425638870874207E+01, 0.4660214926831417E-16, 0.5684400877302917E-10, 0.6387162891082867E-10, 0.1087729835971203E+01, 0.6729715978266918E-13, 0.1741585337702246E-09, 0.2256709509706825E-09, 0.9667696375323971E+00, 0.8583633357857125E-17, 0.5122248377584688E-09, 0.7095864943104643E-09, 0.8022433247488755E+00, 0.1479114197289397E-30, 0.3545102092401686E-08, 0.2547082836533246E-08, 0.5846254874613492E+00, 0.1684748530176799E-17, 0.2002639308765254E+02, 0.2002639308765254E+02, 0.9900000000000000E+02, 0.2829931972789116E+01, 0.2020202020202020E+01, 0.1450844904575395E-08, 0.3940805795947064E+00, 0.8431345355002174E-26, 0.2834737824482197E-07, 0.1323731756234205E-07, 0.2663780336330241E+00, 0.1205202152489745E-19, 0.2553256673132341E-06, 0.1983391491151363E-06, 0.2413081251593111E+00, 0.5052150994009667E-21, 0.4174664103103917E-06, 0.5361294231508838E-06, 0.1817740817404329E+00, 0.4777592300362271E-19, 0.1103166278762495E-05, 0.1411552925870387E-05, 0.1073144197241267E+00, 0.1369462531463676E-25, 0.4868310689385493E-05, 0.3749559753873054E-05, 0.4327520903789019E-01, 0.6517810995642209E-18, 0.3308901711413922E-04, 0.5021439046381042E-04, 0.3141654146246065E-01, 0.3980258690751375E-19, 0.2091355707842741E-03, 0.1263714817696205E-03, 0.2017433957164045E-01, 0.9608259223012324E-19, 0.2907989762339014E-03, 0.3270695437261140E-04, 0.3181705377209195E-03, 0.2078018254813854E-20, 0.5046681645946245E-02, 0.5971971384809690E-02},
			info: 0,
		},
		{
			z:    []float64{0.7556699778432414E+00, 0.9264145479756024E+00, 0.2732571748073197E+00, 0.6411921070422063E+00, 0.9885815056247259E+00, 0.3723918099656685E+00, 0.8347397513753253E+00, 0.1449609854320509E+00, 0.7237973270068254E+00, 0.7937879503069174E+00, 0.1580314367294657E+00, 0.6183123759968190E-01, 0.8526562423479169E+00, 0.4515796037285547E+00, 0.4273610709781387E+00, 0.9916977661043136E-01, 0.6106106148524300E+00, 0.8331371477897991E+00, 0.3226596180395027E+00, 0.9411873968460633E+00, 0.4713107215814810E+00, 0.2408559451890391E-01, 0.8159283865403915E+00, 0.2484819964854622E+00, 0.9053457264816500E-01, 0.7765873412573800E+00, 0.6903393919382517E-01, 0.4737861094943234E+00, 0.2953181034757211E+00, 0.1455852388723616E+00, 0.3595484224812872E+00, 0.7546693773577520E+00, 0.2176499262400425E+00, 0.2180463762362660E+00, 0.4619066216292487E+00, 0.9195384505328125E+00, 0.7817199663072332E+00, 0.7252058550809327E+00, 0.7231496659924924E+00, 0.4997135548263443E+00, 0.7857040551775740E-01, 0.9096915224111880E+00, 0.5048166916632318E+00, 0.9737715836828489E+00, 0.6597329703373522E+00, 0.1262659377119990E-01, 0.3525536903406803E+00, 0.6167857219758321E+00, 0.7408468432196044E+00, 0.7867196986400673E+00, 0.2313289425583596E+00, 0.8524506859175643E+00, 0.1392061690435467E+00, 0.4799927308227143E+00, 0.5988201473645773E+00, 0.8148615971957359E+00, 0.9991849415795491E+00, 0.4062458282025133E+00, 0.8021012361110058E-01, 0.7041715844729369E+00, 0.9319571339573152E-01, 0.9303801779957639E+00, 0.6825635414858495E+00, 0.2651022334979941E+00, 0.2966022780498856E-01, 0.2756350604456157E+00, 0.3996117016882065E+00, 0.4084803031952775E+00, 0.7693093419592285E+00, 0.5425247266459055E+00, 0.7317803042615013E+00, 0.9398853135227920E+00, 0.5923013015298344E+00, 0.6749323490534131E-01, 0.2677632908794654E+00, 0.1468441510303787E+00, 0.1088650133568649E+00, 0.2179701521651023E+00, 0.8673093102161434E+00, 0.5667697713082290E+00, 0.5061732946886482E+00, 0.4707781162637231E+00, 0.4572852405093119E+00, 0.2817384205908577E+00},
			n:    21,
			zOut: []float64{0.2549866140639053E+01, 0.2289692363298807E+01, 0.2042824675422646E+01, 0.1807091836494319E+01, 0.1608583829794490E+01, 0.1518595775727545E+01, 0.1431682400757094E+01, 0.1278141352537376E+01, 0.1218077200015686E+01, 0.1115746594519385E+01, 0.9489661418934578E+00, 0.7950491688285616E+00, 0.6705557578092293E+00, 0.4223587911677574E+00, 0.3211641543143434E+00, 0.1395786344205778E+00, 0.9463219952875908E-01, 0.4563841563552573E-01, 0.3718976888616738E-01, 0.2766660139323238E-01, 0.2861060880329385E-03, 0.1489369389057270E-25, 0.4009456273839574E-11, 0.5913965075566282E-11, 0.1431682400757094E+01, 0.9808641529204798E-18, 0.1658075466067222E-10, 0.2375182467377229E-10, 0.1278141352537376E+01, 0.1561716996498461E-20, 0.5563872147228742E-10, 0.4003249215027731E-10, 0.1218077200015686E+01, 0.8198555146385643E-18, 0.1047304739332037E-09, 0.1509388346471184E-09, 0.1115746594519385E+01, 0.6804594514014344E-21, 0.1132515351796206E-08, 0.7305070647547987E-09, 0.9489661418934578E+00, 0.6324297261214257E-19, 0.2036338790917205E+02, 0.2036338790917205E+02, 0.1050000000000000E+03, 0.3147392290249433E+01, 0.2857142857142857E+01, 0.1188392639382130E-06, 0.6705557578092293E+00, 0.2287929633195320E-19, 0.1283168907741916E-05, 0.1048613283787068E-05, 0.4223587911677574E+00, 0.1189072130544977E-23, 0.1798535669474704E-05, 0.2477367923932174E-05, 0.3211641543143434E+00, 0.8187795594567816E-23, 0.5813187502570164E-05, 0.4274698539216166E-05, 0.1395786344205778E+00, 0.4387796946624011E-14, 0.1512969274311187E-04, 0.4383088766938462E-04, 0.9463219952875908E-01, 0.2802074325023381E-22, 0.1713816584845726E-03, 0.1715892923902221E-03, 0.4563841563552573E-01, 0.8331483361628599E-18, 0.1806157324013782E-06, 0.1193242630586832E-05, 0.3718976888616738E-01, 0.1396316416906445E-20, 0.3480495660704905E-03, 0.4402048010276426E-03, 0.2766660139323238E-01, 0.1504073756330590E-24, 0.6353091961964474E-03, 0.1317363577877200E-03, 0.2861060880329385E-03, 0.4589883047388968E-23, 0.3883745848947868E-02, 0.5358907697572753E-02},
			info: 0,
		},
		{
			z:    []float64{0.8456683250225028E-01, 0.3058015837199188E+00, 0.4317902183064470E-01, 0.3671979469896071E+00, 0.3595487854233631E+00, 0.8791524368561765E+00, 0.1256853689373587E+00, 0.3494000337993621E+00, 0.2131672180923028E+00, 0.4488324680055523E+00, 0.9452502344450827E+00, 0.7197446096567599E+00, 0.8918071015747069E+00, 0.5551949519753436E+00, 0.5627883747367335E+00, 0.6930858113379974E+00, 0.7026202611250518E+00, 0.3856840921343180E+00, 0.4453224782611998E+00, 0.9627551339340559E+00, 0.6255469039377685E+00, 0.6093210861157965E+00, 0.1297243619022892E-01, 0.2745425059216151E+00, 0.5902573930769862E+00, 0.3658719785914519E+00, 0.8179101736936932E+00, 0.4251816548465440E+00, 0.3319568042078717E+00, 0.8294675656951641E+00, 0.1906193594777114E+00, 0.6862699206380130E+00, 0.6832362040809048E+00, 0.8968537122850908E+00, 0.3753636108676777E+00, 0.9926770667520373E+00, 0.7473790867948205E-01, 0.3653635034012445E+00, 0.6553100464759598E-01, 0.1045856631958887E+00, 0.2265961870023414E+00, 0.3205967446655300E-01, 0.1845082884370131E+00, 0.2250092498055734E+00, 0.5876866067653603E+00, 0.2146512915654570E+00, 0.3937206506545190E+00, 0.4110924225795073E+00, 0.3228836105384403E+00, 0.3531543308069818E-01, 0.9472843614842231E+00, 0.2348052564069324E+00, 0.1315455878409307E+00, 0.2246729768637089E+00, 0.8445861542610614E+00, 0.9528962673528295E+00, 0.4907499555456967E+00, 0.9420172239501383E+00, 0.4896535244350841E-01, 0.5088622551995382E+00, 0.1789066474661110E+00, 0.3566730680524121E+00, 0.4279936896698711E-01, 0.5241586506933626E-02, 0.9191361995860731E+00, 0.7666474814508011E+00, 0.8302596584023555E+00, 0.4738482866502817E+00, 0.2085735695866969E+00, 0.3976808123484415E+00, 0.4364253757027529E+00, 0.2407360056333678E+00, 0.4002772343201099E+00, 0.7717338386755229E+00, 0.5023514841326452E+00, 0.8988902694804358E+00, 0.4259852339101408E+00, 0.2067093317010162E+00, 0.8456128611542781E+00, 0.1290027044731792E+00, 0.2598901078235174E+00, 0.5836006873243835E+00, 0.9777637054617091E+00, 0.4896357840832322E+00},
			n:    21,
			zOut: []float64{0.2625813250937807E+01, 0.2316814186652984E+01, 0.2239421861777163E+01, 0.1853308891402210E+01, 0.1827033391537137E+01, 0.1481398580531558E+01, 0.1385699025778513E+01, 0.1101045596651604E+01, 0.1077490875294086E+01, 0.6790358065311344E+00, 0.6361857378195889E+00, 0.5095707085100130E+00, 0.4739579905788711E+00, 0.4209497658895578E+00, 0.3012142858983374E+00, 0.2432742659092710E+00, 0.2088674538811938E+00, 0.1664409530858738E+00, 0.2606656040091844E-01, 0.1205810631995507E-01, 0.9325512290661112E-07, 0.1292469707114106E-24, 0.1719039818430849E-04, 0.1845080438772963E-04, 0.1385699025778513E+01, 0.2224123229986848E-19, 0.6819642419856913E-05, 0.9079053758699490E-05, 0.1101045596651604E+01, 0.6054420446023063E-14, 0.5392861994521241E-06, 0.8422816920290823E-06, 0.1077490875294086E+01, 0.7330726620037819E-25, 0.4290042344637304E-06, 0.3471234449409943E-06, 0.6790358065311344E+00, 0.1969273319091676E-18, 0.8635424232755040E-07, 0.8294124288205587E-07, 0.6361857378195889E+00, 0.4466775307674635E-22, 0.1958564738864290E+02, 0.1958564738864290E+02, 0.9700000000000000E+02, 0.2927437641723356E+01, 0.2061855670103093E+01, 0.4284285350948107E-09, 0.4739579905788711E+00, 0.4844247598072386E-17, 0.1501956526577450E-07, 0.4599310764980267E-07, 0.4209497658895578E+00, 0.5697556379758414E-13, 0.1468884761079027E-05, 0.1935721204206928E-05, 0.3012142858983374E+00, 0.3414189997796539E-26, 0.7129394687116921E-05, 0.9262068822983554E-05, 0.2432742659092710E+00, 0.2465190328815662E-31, 0.3320148255009530E-04, 0.2563439348523979E-04, 0.2088674538811938E+00, 0.8874685183736383E-29, 0.9377671959730327E-04, 0.7166613903737452E-04, 0.1664409530858738E+00, 0.1175218723307338E-19, 0.5883141447281398E-03, 0.4140832085136277E-03, 0.2606656040091844E-01, 0.2634795423438179E-27, 0.4797918408258123E-02, 0.4833036765864798E-02, 0.1205810631995507E-01, 0.6530371216255944E-19, 0.7811358433513861E-02, 0.1392611548870054E-01, 0.9325512290661112E-07, 0.1190420191421466E-12, 0.6274023555637595E-01, 0.1241063086176471E-01},
			info: 0,
		},
		{
			z:    []float64{0.6345014789904082E+00, 0.9828013844807603E+00, 0.5292373213190880E+00, 0.3345627500234365E+00, 0.9345959669050260E+00, 0.5873733336147081E+00, 0.8888462128293518E+00, 0.1044020545901757E+00, 0.9375280272458696E+00, 0.1233411973951652E+00, 0.3330283913003940E+00, 0.2432208394685875E-01, 0.6106490172620382E+00, 0.9136489477065954E+00, 0.8250850334045809E+00, 0.7866479843500775E+00, 0.1629924039124250E+00, 0.2349971551206693E+00, 0.1024684774133804E+00, 0.7852727749024598E+00, 0.1427984573527686E+00, 0.4315181982028241E+00, 0.6564047419338838E+00, 0.5555590030741936E+00, 0.8205569452257172E+00, 0.4368031922318115E+00, 0.2897575605849890E+00, 0.6163386054890784E+00, 0.7016794584968877E+00, 0.7265422079630662E+00, 0.6393115149906525E+00, 0.7552597718778955E+00, 0.5029129969011173E+00, 0.4874732646810529E+00, 0.5094624150009296E+00, 0.4766369914172104E+00, 0.9003818070855094E+00, 0.6813014247956244E+00, 0.2988296447575162E+00, 0.6597790416796705E+00, 0.5646192150605487E+00, 0.3275024686793908E+00, 0.2725061169408716E-01, 0.8098076662320616E+00, 0.3345276190659139E+00, 0.1812780354242056E+00, 0.1997802571871318E+00, 0.5440765633504190E+00, 0.9774510594766147E+00, 0.1863483792916641E+00, 0.5910097330685633E+00, 0.2734489448477760E+00, 0.4395818444416557E+00, 0.9022786559097939E+00, 0.1029995124123335E+00, 0.9437933734967648E+00, 0.1226746192846208E+00, 0.1122541771291640E+00, 0.1756516455268763E+00, 0.9109529416432673E+00, 0.3050345572159197E-01, 0.1746678233544180E+00, 0.3750993781268622E+00, 0.4476224977918100E+00, 0.5437446694835568E+00, 0.3729575052852190E+00, 0.3542378566336749E+00, 0.7449948633772266E+00, 0.3157555204945224E+00, 0.7753984984296890E+00, 0.3284038993889418E+00, 0.5159361949232710E-01, 0.4529339074400007E-01, 0.5693581512511349E+00, 0.2213156237057901E+00, 0.9395831428155459E+00, 0.3843047271617620E+00, 0.2968653921042081E+00, 0.6156090467550724E+00, 0.3170556612853570E-01, 0.8265655132088339E+00, 0.3165750369023552E+00, 0.7055602587055542E+00, 0.8365600455841105E+00},
			n:    21,
			zOut: []float64{0.2473859321883559E+01, 0.2422834159796051E+01, 0.2342770718537345E+01, 0.2190393245355733E+01, 0.2048748874742353E+01, 0.1880327408972340E+01, 0.1569014269461521E+01, 0.1302496773444914E+01, 0.1193514796722045E+01, 0.1150651682141956E+01, 0.8969794111087930E+00, 0.8226581338194529E+00, 0.6917042463755413E+00, 0.5972139808466055E+00, 0.3480375932216391E+00, 0.2932789944580818E+00, 0.2090971942801221E+00, 0.1221616224423389E+00, 0.1000478595340333E+00, 0.3412207557320287E-01, 0.3160927987836547E-03, 0.2386777554836693E-25, 0.1787847554062023E-08, 0.1930667016131883E-08, 0.1569014269461521E+01, 0.5039432348921582E-18, 0.3047636948536222E-07, 0.1223330280849913E-07, 0.1302496773444914E+01, 0.0000000000000000E+00, 0.7408697490815375E-06, 0.1498788057488595E-05, 0.1193514796722045E+01, 0.6273021918317486E-26, 0.8867922405978688E-08, 0.3710956866722375E-08, 0.1150651682141956E+01, 0.1462883977034466E-17, 0.2681277488017278E-06, 0.6188429230273071E-06, 0.8969794111087930E+00, 0.4930380657631324E-31, 0.2269022845551642E+02, 0.2269022845551641E+02, 0.1110000000000000E+03, 0.3215419501133787E+01, 0.4504504504504505E+01, 0.4606971433208740E-07, 0.6917042463755413E+00, 0.7123549154360197E-18, 0.1197511514786163E-07, 0.1386949181837661E-07, 0.5972139808466055E+00, 0.4652335183636722E-20, 0.1082703076660205E-07, 0.1003906263016573E-07, 0.3480375932216391E+00, 0.3573127201018094E-16, 0.2388210318135170E-06, 0.5694782222336181E-06, 0.2932789944580818E+00, 0.7601337464964823E-24, 0.7791755144624492E-05, 0.3242296896441773E-05, 0.2090971942801221E+00, 0.1663630656286764E-22, 0.9916420504550920E-06, 0.3916254919710180E-05, 0.1221616224423389E+00, 0.1406614427791829E-18, 0.1356917845842480E-04, 0.1379449187478977E-04, 0.1000478595340333E+00, 0.2800456213534592E-28, 0.3213444959426999E-04, 0.2790230345312924E-04, 0.3412207557320287E-01, 0.1075846717720530E-17, 0.8531299184272352E-03, 0.1419520770966347E-02, 0.3160927987836547E-03, 0.1848892746611746E-31, 0.7376316906583622E-02, 0.6546496744187779E-02},
			info: 0,
		},
		{
			z:    []float64{0.7781991870339823E+00, 0.3799726947828087E+00, 0.8225406656461727E+00, 0.4322410258771066E+00, 0.4965834581857734E+00, 0.1642548945285990E+00, 0.4773703559733889E+00, 0.7580746486223987E+00, 0.9220155673403277E+00, 0.1624062232083541E+00, 0.5664470130564820E+00, 0.7367013127895712E+00, 0.4720756942873998E+00, 0.8695958825780965E+00, 0.6045812450089678E+00, 0.8503421320137978E+00, 0.7291568033584502E+00, 0.9563602214514951E+00, 0.1245752972610169E+00, 0.2621011209284912E+00, 0.8642265339858576E+00, 0.9528021036615359E+00, 0.9515862784178062E+00, 0.2632504807072628E+00, 0.8585529108378003E+00, 0.3025834513231899E+00, 0.7125877353473579E+00, 0.9221275053156179E+00, 0.9787992369955746E+00, 0.8789465750552752E+00, 0.9318675804398889E+00, 0.8827440896392771E+00, 0.3124337393280541E+00, 0.3124656972853590E-01, 0.5860766227864144E+00, 0.9812515177915087E+00, 0.4480000254042209E+00, 0.5121903109069345E+00, 0.1837298514334742E+00, 0.2680410429176278E+00, 0.5529294510898309E+00, 0.3627557909974974E+00, 0.9282219359125773E+00, 0.2173723036967439E+00, 0.7593665130145739E+00, 0.6245533364325671E+00, 0.1767680905493787E+00, 0.3823356866170564E+00, 0.5171578058722374E+00, 0.5376155024979712E+00, 0.5280269975706942E-01, 0.7947058949878518E+00, 0.1214716509814368E+00, 0.6840110600217510E+00, 0.7041243787921371E+00, 0.6945194787199669E+00, 0.8184844284849127E+00, 0.3440855810599863E+00, 0.1619958520504678E+00, 0.6990470088709023E+00, 0.3532225659848430E+00, 0.8229090368317697E+00, 0.8349696782028652E+00, 0.6932180266797183E-01, 0.6880274120498576E+00, 0.7566065163897195E+00, 0.2981422921509080E-01, 0.8636153925759227E-02, 0.7928093803058089E-01, 0.3096144259285381E+00, 0.6829886066202427E+00, 0.8246144682759358E+00, 0.2007259621364732E+00, 0.8710862365466575E+00, 0.6898755422119236E+00, 0.9105030198433044E+00, 0.8974818523094739E+00, 0.8909991694059211E+00, 0.8084341913662618E-01, 0.6072211485624930E+00, 0.9680950749726419E+00, 0.3932275197607872E+00, 0.8781208296025552E+00, 0.5527043030768199E+00},
			n:    21,
			zOut: []float64{0.3158195987040060E+01, 0.2674397094280163E+01, 0.2611167345511754E+01, 0.2117792657212280E+01, 0.2026395303611904E+01, 0.1943316400161173E+01, 0.1877195462180231E+01, 0.1801757023901512E+01, 0.1273111869927408E+01, 0.1088705127135854E+01, 0.9559661931705944E+00, 0.9013607966201198E+00, 0.6445742634037150E+00, 0.5446175180211763E+00, 0.4486902226406620E+00, 0.3854842198344371E+00, 0.2741584921777233E+00, 0.1210105287753585E+00, 0.6940868812740908E-01, 0.2059029850850508E-01, 0.3673564803695461E-02, 0.8077935669463161E-25, 0.2195808487174199E-10, 0.8449955132778232E-10, 0.1877195462180231E+01, 0.1633831636278255E-19, 0.3497961877980620E-08, 0.7650955873935623E-08, 0.1801757023901512E+01, 0.4072241987713803E-24, 0.2126842707416970E-06, 0.1130018646589442E-06, 0.1273111869927408E+01, 0.1740941751301712E-20, 0.3388039768888790E-03, 0.2863573324805332E-03, 0.1088705127135854E+01, 0.2858227958137079E-16, 0.2905468688553340E-03, 0.2787380120464217E-03, 0.9559661931705944E+00, 0.1315475667900737E-21, 0.2494156905704573E+02, 0.2494156905704573E+02, 0.1010000000000000E+03, 0.3074829931972789E+01, 0.1980198019801980E+01, 0.2647338938369395E-05, 0.6445742634037150E+00, 0.2341846562030201E-22, 0.2667503836084351E-03, 0.1510510164587073E-03, 0.5446175180211763E+00, 0.2242636890234710E-24, 0.5495840118850513E-05, 0.4810476319752364E-05, 0.4486902226406620E+00, 0.3552400222016096E-16, 0.2740332708518062E-05, 0.2562621073339741E-05, 0.3854842198344371E+00, 0.1036641484462207E-22, 0.2248949759797675E-05, 0.2492010005917310E-05, 0.2741584921777233E+00, 0.6071156729318302E-20, 0.4130294607665845E-03, 0.1141603094124651E-03, 0.1210105287753585E+00, 0.2468629837810345E-16, 0.7230132634032991E-02, 0.6618694284253705E-02, 0.6940868812740908E-01, 0.5487745597049798E-23, 0.4151017130761703E-02, 0.4525758324635968E-02, 0.2059029850850508E-01, 0.6083695301064443E-26, 0.3095027772299370E-02, 0.3218440937246976E-02, 0.3673564803695461E-02, 0.6106717417722413E-22, 0.6443405665983428E-02, 0.1622720765285148E-02},
			info: 0,
		},
		{
			z:    []float64{0.1985414918961074E+00, 0.4386372064833388E-01, 0.9272679997999695E+00, 0.9276564660006267E+00, 0.9509373652834754E+00, 0.6247274629358034E+00, 0.4838472987754385E+00, 0.6222976111124274E+00, 0.4248800474043221E+00, 0.9755567793536163E+00, 0.8912410754920851E+00, 0.2313492132656925E-01, 0.1563015923526294E+00, 0.7905214363406186E+00, 0.4927898149840948E+00, 0.5370919594301193E+00, 0.4644799081842790E+00, 0.9960946978984735E+00, 0.2493095982292803E+00, 0.3814725574373231E+00, 0.2940816225707878E+00, 0.6535898793246008E+00, 0.2084037192546966E+00, 0.5132959253232522E+00, 0.4251209694264122E+00, 0.9378579353373483E+00, 0.9786807729708236E+00, 0.7781441628535176E+00, 0.3951333405907657E+00, 0.3920742203259495E+00, 0.1500883321285087E+00, 0.9121803806188729E+00, 0.8516361075320147E+00, 0.6824417668614835E-01, 0.1390781730568696E+00, 0.3112124735573820E+00, 0.2535919561468232E+00, 0.7644352497405866E+00, 0.9218252733114815E+00, 0.9901458352558505E+00, 0.3279511469507669E+00, 0.8365860218696105E+00, 0.3198677553816687E+00, 0.6133665370063144E+00, 0.2580491557527305E+00, 0.6586714927743139E+00, 0.3933698458458805E+00, 0.7194383869235133E-01, 0.2213903901048253E-02, 0.1496106697134404E+00, 0.6766438239122079E+00, 0.7197724372602395E+00, 0.4797148651858846E+00, 0.5998154083597278E+00, 0.5590962662612089E+00, 0.6336887670809047E+00, 0.1072453740736307E+00, 0.7505878626477551E+00, 0.1427362910235584E+00, 0.6016163404777434E+00, 0.9488238354107547E+00, 0.4206748665323531E+00, 0.4473756257202280E+00, 0.2658295729210566E+00, 0.2340075619597504E+00, 0.9617320234413022E+00, 0.5962761690405762E+00, 0.8519099982463364E+00, 0.5571747680961063E+00, 0.1982921114470393E-01, 0.2779304562486129E-01, 0.7924439389736488E+00, 0.9672136382383878E-01, 0.8417095886148459E+00, 0.7473598542298929E+00, 0.9746159458640847E+00, 0.5038819873190444E+00, 0.9169242656654556E+00, 0.5252626769192049E+00, 0.5100979601115528E+00, 0.2790149607164115E+00, 0.4554892778467722E+00, 0.9762420639370454E+00, 0.9832519154713986E+00},
			n:    21,
			zOut: []float64{0.2773247209998878E+01, 0.2684570079202721E+01, 0.2535407140485478E+01, 0.2251849067745984E+01, 0.2036533658934636E+01, 0.1904369707782458E+01, 0.1446171755624304E+01, 0.1299751831669502E+01, 0.1269533063987320E+01, 0.1007664938847636E+01, 0.8251797533661941E+00, 0.5818474246453018E+00, 0.5486201168977923E+00, 0.3772420051451134E+00, 0.3380340897724159E+00, 0.2023767003460897E+00, 0.1840234711841343E+00, 0.9225167203340243E-01, 0.4232592978815707E-01, 0.2771262583534443E-01, 0.7321455618534900E-04, 0.2977163580658995E-22, 0.3752078887401622E-10, 0.4179376226086309E-10, 0.1446171755624304E+01, 0.2236367039610280E-12, 0.1299965948722022E-10, 0.1595321259646779E-10, 0.1299751831669502E+01, 0.9926167350636332E-23, 0.8604643146515003E-10, 0.1864418628665438E-09, 0.1269533063987320E+01, 0.1306353659045996E-24, 0.1991121186029356E-08, 0.8983175437487846E-09, 0.1007664938847636E+01, 0.1213184768518325E-22, 0.3444432736029153E-07, 0.6437805745712839E-07, 0.8251797533661941E+00, 0.7772022937681214E-16, 0.2242878545784906E+02, 0.2242878545784905E+02, 0.1000000000000000E+03, 0.3024943310657596E+01, 0.3000000000000000E+01, 0.1709907812443784E-06, 0.5486201168977923E+00, 0.2972680326362443E-23, 0.1350081364219064E-06, 0.1255933647347444E-06, 0.3772420051451134E+00, 0.8046381233254320E-28, 0.7022416754922569E-08, 0.2768431074740138E-07, 0.3380340897724159E+00, 0.1719716773381806E-27, 0.7079953450794871E-07, 0.6648251035830866E-07, 0.2023767003460897E+00, 0.1940724044588524E-24, 0.4764709677102475E-07, 0.4597049352360845E-07, 0.1840234711841343E+00, 0.3459682911934042E-21, 0.4438757837191216E-07, 0.4402069254647340E-07, 0.9225167203340243E-01, 0.2596900924099520E-17, 0.8125925620122143E-07, 0.6770863964779914E-07, 0.4232592978815707E-01, 0.1280067631025782E-15, 0.3584234088053806E-06, 0.4822962810861105E-06, 0.2771262583534443E-01, 0.1012934875847350E-18, 0.1556396151524861E-05, 0.2916279508701459E-05, 0.7321455618534900E-04, 0.1540743955509789E-32, 0.4190434742116713E-02, 0.1362677535010021E-02},
			info: 0,
		},
		{
			z:    []float64{0.6703344952482478E+00, 0.3577116069291627E+00, 0.8432263018920840E+00, 0.9584130033491455E+00, 0.2861685401430958E+00, 0.9668570623620085E+00, 0.6957796127156375E+00, 0.7603022900919236E+00, 0.7677099838587329E+00, 0.2702658617954818E+00, 0.9607212720070546E-01, 0.3973573502375266E+00, 0.8455511814548733E+00, 0.6862903202354254E-01, 0.1789860417278408E+00, 0.1112926642215833E+00, 0.8320621955974556E+00, 0.2637090148175287E+00, 0.7109403389502641E+00, 0.6494529820495407E+00, 0.8828985636942858E+00, 0.9098303542318168E+00, 0.1777489928582532E+00, 0.8977650882542143E-01, 0.7154356412046237E-01, 0.2693107133832517E-01, 0.7956794415059365E-02, 0.8852594037406200E+00, 0.2162284138438216E+00, 0.9216829080200467E+00, 0.9874282279096652E+00, 0.6087745498247736E+00, 0.7694028142872190E+00, 0.5918349824988693E+00, 0.1915561020798640E+00, 0.1135335598824661E-01, 0.6670902390154858E+00, 0.6601113994758983E+00, 0.7779237148990382E+00, 0.9023999203058302E+00, 0.1848842750705929E+00, 0.6120355291150891E+00, 0.7209828136916797E+00, 0.3124354740483853E+00, 0.9520753904353354E+00, 0.9694533109968584E+00, 0.6869212426966154E+00, 0.3510392134251344E+00, 0.7696249856763533E+00, 0.9203915575905880E+00, 0.4616165896255233E+00, 0.6796681592119477E+00, 0.9261196886552191E+00, 0.4511929507295228E+00, 0.5162093776469306E+00, 0.3093101029632945E+00, 0.1162653262260159E+00, 0.1951366360676013E+00, 0.7945099687545387E+00, 0.9707806531485516E+00, 0.4161172559602336E+00, 0.5813461713597171E+00, 0.8442974147064882E+00, 0.9135490286183924E+00, 0.3261094875410662E+00, 0.7080588637364270E+00, 0.3833080381406134E+00, 0.9095051028453779E+00, 0.6909172438403831E+00, 0.5187928284332267E+00, 0.5783778059525303E+00, 0.9615928062069277E+00, 0.2127624116065856E+00, 0.7254035177440832E+00, 0.2587362009618467E+00, 0.7570212065831576E+00, 0.2401238007004233E+00, 0.2744934873404078E+00, 0.9420426561890750E+00, 0.7098059460475599E+00, 0.2691363643891775E+00, 0.4536316870833661E+00, 0.6839445343274818E+00, 0.9336823543434762E+00},
			n:    21,
			zOut: []float64{0.2634467575625416E+01, 0.2594635276829839E+01, 0.2463070407655619E+01, 0.2293127065694029E+01, 0.2072673505384417E+01, 0.1459936798288049E+01, 0.1378497856793854E+01, 0.1333239471494390E+01, 0.1102542278566289E+01, 0.8930469384549427E+00, 0.8520787727779708E+00, 0.6953298055150813E+00, 0.6847648677749263E+00, 0.2185116940651705E+00, 0.2008337976176149E+00, 0.1757551361146239E+00, 0.1271484717317025E+00, 0.5049064582277562E-01, 0.3515271339576785E-01, 0.7966224551514293E-02, 0.1685389559757755E-03, 0.1635721388548769E-22, 0.3551032227485648E-11, 0.3089325791958004E-11, 0.1378497856793854E+01, 0.6463020148652660E-17, 0.4439365591922079E-11, 0.5247211770454983E-11, 0.1333239471494390E+01, 0.8616343098037817E-25, 0.4343651596456738E-10, 0.2452653826174233E-10, 0.1102542278566289E+01, 0.9876517884224792E-16, 0.1549700306365952E-08, 0.1012434062383302E-08, 0.8930469384549427E+00, 0.4606319384654338E-19, 0.1115188986357716E-08, 0.1237013304894199E-08, 0.8520787727779708E+00, 0.1447566071967798E-23, 0.2127343784310997E+02, 0.2127343784310997E+02, 0.1270000000000000E+03, 0.3675736961451247E+01, 0.9448818897637794E+01, 0.2782364756724027E-11, 0.6847648677749263E+00, 0.6100457017578579E-23, 0.1535488071926541E-09, 0.5733950833190260E-09, 0.2185116940651705E+00, 0.6436934318898579E-12, 0.5207333190112868E-06, 0.4302907440331721E-06, 0.2008337976176149E+00, 0.1972152263052530E-30, 0.2421587237749379E-06, 0.2920528765870267E-06, 0.1757551361146239E+00, 0.8101601496619791E-27, 0.1434500575863456E-11, 0.9045851929625007E-13, 0.1271484717317025E+00, 0.1560107593833772E-17, 0.1620731838764033E-06, 0.1206839073349329E-06, 0.5049064582277562E-01, 0.3155443620884047E-29, 0.7221494757183839E-05, 0.4279630756550309E-05, 0.3515271339576785E-01, 0.0000000000000000E+00, 0.2092539651031481E-04, 0.3641666945464839E-04, 0.7966224551514293E-02, 0.1018583906139065E-23, 0.4276808715243643E-05, 0.6380573645111411E-06, 0.1685389559757755E-03, 0.0000000000000000E+00, 0.2828744147621957E-02, 0.2649548441232189E-02},
			info: 0,
		},
		{
			z:    []float64{0.7647749557731113E+00, 0.1439440544256733E+00, 0.7500486933916728E+00, 0.6414832886183546E+00, 0.8122572937542115E+00, 0.9529234674941257E+00, 0.3352470221802866E+00, 0.6936226545338587E+00, 0.9931729574752579E-01, 0.3187856199297912E-01, 0.1671537284832858E+00, 0.6799274061730229E+00, 0.8177446911383146E+00, 0.5321042755342652E+00, 0.1611635863708949E+00, 0.1556305029012977E+00, 0.8064987994430675E+00, 0.6086425937675999E-01, 0.9539387771768112E+00, 0.5483826791143146E+00, 0.1624656726546599E+00, 0.8604262189788422E+00, 0.3348601405085260E+00, 0.6512790728986305E+00, 0.3357978279833680E+00, 0.7104789129438286E+00, 0.4161092786312924E+00, 0.8790797243791704E+00, 0.1469866181354552E+00, 0.7277776395098579E-01, 0.4317417806367096E-01, 0.8731968970940820E+00, 0.4856440529809196E+00, 0.2315660100703048E+00, 0.2955911335168215E+00, 0.4119299294790694E+00, 0.4565975215473586E+00, 0.8915397570999283E+00, 0.1639735129501203E+00, 0.5055946074157277E+00, 0.9530743160791620E+00, 0.9836351183281582E+00, 0.2913429799811792E+00, 0.9289008959282562E+00, 0.5996674664338365E+00, 0.6609996028219567E+00, 0.7317528474660470E+00, 0.9903648665746729E+00, 0.3982305249886617E+00, 0.3563571411857538E+00, 0.8851540908639802E+00, 0.8253410045707608E+00, 0.8085297859672740E+00, 0.3918185712378985E+00, 0.7922842941861202E+00, 0.7487878715845681E+00, 0.3117744132747395E+00, 0.7135433852632408E+00, 0.8969220948763633E+00, 0.5840741001028926E+00, 0.3261515465187765E+00, 0.5368137772779212E+00, 0.6371156779602756E+00, 0.5575308178040213E+00, 0.1319054467962255E+00, 0.6274305773732142E+00, 0.9230553933157697E+00, 0.3589158151867838E+00, 0.1429059780004028E+00, 0.6433214640390554E+00, 0.9159704795536289E+00, 0.8998439182451070E+00, 0.5665265465351250E+00, 0.4290736239553739E+00, 0.4923118412965143E+00, 0.3733795042755129E+00, 0.4008465132226234E+00, 0.8630963810875470E+00, 0.2835407905147469E+00, 0.8431337072866282E+00, 0.7450545748778526E+00, 0.2948369760127234E+00, 0.6896108484670013E+00, 0.1073791710993542E+00},
			n:    21,
			zOut: []float64{0.2492440020550355E+01, 0.1951232214041701E+01, 0.1924880150346451E+01, 0.1800337146913507E+01, 0.1731259644349016E+01, 0.1515025732591961E+01, 0.1392131403063820E+01, 0.1337121843604905E+01, 0.1335944341051471E+01, 0.9695979508906304E+00, 0.8602855628657217E+00, 0.7598736716193354E+00, 0.5663438341779535E+00, 0.4991982558529897E+00, 0.3900147250500268E+00, 0.2612685334229408E+00, 0.1040703325016068E+00, 0.5095987426708586E-01, 0.4270469018444285E-01, 0.6313950453962554E-02, 0.4526318588114864E-04, 0.1623810366421630E-16, 0.5786287269352850E-19, 0.2440824886543042E-19, 0.1392131403063820E+01, 0.1972152263052530E-30, 0.1305662432930495E-17, 0.2687054746502436E-17, 0.1337121843604905E+01, 0.6807276388656606E-23, 0.2125450465832215E-16, 0.4166517172824519E-16, 0.1335944341051471E+01, 0.3971441341244662E-25, 0.3131313418539388E-15, 0.1598422579069803E-15, 0.9695979508906304E+00, 0.2612707318091991E-24, 0.3703169268599905E-13, 0.5897085302084673E-13, 0.8602855628657217E+00, 0.3944304526105059E-30, 0.1999104914098575E+02, 0.1999104914098576E+02, 0.1050000000000000E+03, 0.3185941043083900E+01, 0.4761904761904762E+01, 0.3815530651290367E-11, 0.5663438341779535E+00, 0.4442864618204738E-26, 0.2314135864505331E-10, 0.3280099800133426E-10, 0.4991982558529897E+00, 0.2524354896707238E-28, 0.8792792965154248E-10, 0.6284153656438893E-10, 0.3900147250500268E+00, 0.2716911819042188E-17, 0.6371083763388994E-09, 0.9063994743331422E-09, 0.2612685334229408E+00, 0.7682295356186976E-14, 0.2615630848198228E-07, 0.4934681155229334E-07, 0.1040703325016068E+00, 0.3134368286722418E-21, 0.1338644268367565E-05, 0.2571041054206291E-05, 0.5095987426708586E-01, 0.6829203137199389E-20, 0.1734538060272007E-04, 0.9160403953625247E-05, 0.4270469018444285E-01, 0.5620633949699709E-29, 0.1970936154220316E-07, 0.7875060113736862E-07, 0.6313950453962554E-02, 0.1074365444038600E-22, 0.1950342748319920E-02, 0.1960570817697857E-02, 0.4526318588114864E-04, 0.5045801736573469E-22, 0.2719687288170513E-02, 0.7229547441545878E-02},
			info: 0,
		},
		{
			z:    []float64{0.6830594937182136E+00, 0.2806903427821971E+00, 0.3842580501524426E+00, 0.7113293946876786E+00, 0.6120481717427926E+00, 0.4087522624125913E+00, 0.2683959794755927E+00, 0.1982680719912362E+00, 0.5503761648124630E+00, 0.9947639519912582E+00, 0.4817851020535202E-01, 0.1841410650783141E+00, 0.7364485595361944E+00, 0.2204876163244174E+00, 0.8785531172976315E+00, 0.3900984044089164E+00, 0.9898399332392419E+00, 0.4652324815840650E+00, 0.6574114801218780E+00, 0.6641201888165797E+00, 0.5371222655177923E+00, 0.2241732155112269E+00, 0.6726821610374456E-01, 0.2361953456198683E+00, 0.5602287124209401E+00, 0.7836687903368798E+00, 0.2633616918428828E+00, 0.9669553142730151E+00, 0.3692579170869618E+00, 0.5395054150347700E-01, 0.4776047537674022E-01, 0.5691410116455676E+00, 0.1304181940038973E+00, 0.2107738642513712E+00, 0.5434523182333528E-01, 0.4019375962886824E+00, 0.7125153526157032E+00, 0.5305712841430671E+00, 0.1164846974244694E+00, 0.2037577925839799E+00, 0.2045617511271008E+00, 0.6122054957321945E+00, 0.5114531146106966E+00, 0.2882478115017332E+00, 0.5304873168821023E+00, 0.1796380649103803E+00, 0.9949092378946756E-01, 0.5448372048900864E+00, 0.1971761401002603E+00, 0.9487011110186927E+00, 0.8526405089060669E+00, 0.4533764477818548E+00, 0.8410819408489681E+00, 0.7592016862970913E+00, 0.8131142938762341E+00, 0.1445496351282255E+00, 0.6614974180377753E+00, 0.3394841721225367E+00, 0.4878107636737276E+00, 0.5499609605641745E+00, 0.1789798083030991E+00, 0.2008397619575961E+00, 0.5247567247775218E+00, 0.7394478154071386E+00, 0.6713478520024037E+00, 0.3894295843862983E+00, 0.9099978858741096E+00, 0.9872931573704680E-01, 0.5370559804237263E+00, 0.8492599321050560E+00, 0.4671993983916084E+00, 0.6397070306147701E+00, 0.2884512208824136E+00, 0.5002881141083532E+00, 0.3356213102526899E+00, 0.4102730364689955E+00, 0.8172446941984062E+00, 0.1463391840674726E+00, 0.6374760821459636E+00, 0.2986995049911381E+00, 0.6932950085590186E+00, 0.8475246239619540E+00, 0.3550038162879430E+00, 0.9747731646460122E+00},
			n:    21,
			zOut: []float64{0.2302037253547019E+01, 0.1904432769194928E+01, 0.1838629010091233E+01, 0.1679631021950300E+01, 0.1638588328999231E+01, 0.1556939208705248E+01, 0.1154775776884602E+01, 0.1123025367728587E+01, 0.1005963210310813E+01, 0.7892040895461047E+00, 0.6171474041407791E+00, 0.5107385873854940E+00, 0.4745466376069911E+00, 0.2987072329916590E+00, 0.2518976395749056E+00, 0.1562926917018191E+00, 0.1208002043434920E+00, 0.1117929648180551E+00, 0.3025264937486962E-01, 0.5461437067005637E-02, 0.3901591662109151E-04, 0.2028008633963607E-17, 0.1197298285059118E-05, 0.1163352934520667E-05, 0.1154775776884602E+01, 0.2502221352972909E-22, 0.7438201518083012E-06, 0.8784873727121712E-06, 0.1123025367728587E+01, 0.2368236188315088E-19, 0.3676741199329849E-11, 0.3039307086935086E-11, 0.1005963210310813E+01, 0.2065553394230697E-24, 0.6686270095375819E-11, 0.8550052948388210E-11, 0.7892040895461047E+00, 0.1239058396467615E-20, 0.2067843352216928E-06, 0.6687212675645054E-07, 0.6171474041407791E+00, 0.2620094590259913E-21, 0.1757090250187976E+02, 0.1757090250187975E+02, 0.9500000000000000E+02, 0.2798185941043084E+01, 0.1052631578947368E+01, 0.6997553999213272E-08, 0.4745466376069911E+00, 0.5994838008700348E-24, 0.1052996378646298E-04, 0.6312169743031761E-04, 0.2987072329916590E+00, 0.9860761315262648E-30, 0.1466875513591431E-03, 0.1024839055944592E-03, 0.2518976395749056E+00, 0.7516632826070909E-17, 0.5524256710713082E-08, 0.1151290676113078E-07, 0.1562926917018191E+00, 0.1348952147927930E-27, 0.1222984298841777E-06, 0.3042067118561864E-06, 0.1208002043434920E+00, 0.4760428464210509E-23, 0.9492582004636672E-05, 0.3402727365723720E-04, 0.1117929648180551E+00, 0.7018716354804804E-22, 0.2431231599464108E-03, 0.1315672135071063E-03, 0.3025264937486962E-01, 0.4377821080194221E-21, 0.1333575496853992E-02, 0.6988231628130009E-03, 0.5461437067005637E-02, 0.2193916840728260E-24, 0.1845211740639344E-02, 0.8881725247692956E-03, 0.3901591662109151E-04, 0.2067951531382569E-23, 0.6469870796864946E-02, 0.1027022788337614E-01},
			info: 0,
		},
		{
			z:    []float64{0.3270384527330652E+00, 0.4068565573647237E+00, 0.5445258191923925E+00, 0.9571802305689175E+00, 0.1840747434207346E-01, 0.8759261930487382E+00, 0.3018283835341639E+00, 0.4754674889513868E+00, 0.9620709342523251E+00, 0.3596342978433105E+00, 0.1820397917245298E+00, 0.9562190617892271E+00, 0.8004073710305811E+00, 0.9503584926946729E+00, 0.1634422399578277E+00, 0.3053134184266467E+00, 0.3871438286377282E+00, 0.1225252880124038E+00, 0.5955538022158587E+00, 0.9831372787057571E+00, 0.2130378801829313E+00, 0.1502296787149880E+00, 0.5120891507658710E+00, 0.2769158347903287E+00, 0.6740978559159013E+00, 0.6671510755324899E+00, 0.5252429573239310E+00, 0.7312875683466940E+00, 0.6359518339326128E+00, 0.3635213813950466E-01, 0.7509704695539338E+00, 0.2796455326190517E+00, 0.3929933998036216E+00, 0.2441550702438192E+00, 0.5181887041338979E+00, 0.2149631476903890E-01, 0.1922157358558590E+00, 0.8352708839021310E+00, 0.2143118694803675E+00, 0.8099207585987700E+00, 0.4013415573783083E+00, 0.8990854368069680E+00, 0.2274365144039603E-01, 0.1574221848735892E+00, 0.5021896402707141E+00, 0.6811320055313477E+00, 0.4008235511220088E+00, 0.2607115591315068E+00, 0.8396807341169598E+00, 0.3393611517447541E+00, 0.3095200315094494E+00, 0.8849677836408325E+00, 0.3309280721041051E+00, 0.2177939175852258E+00, 0.7396054658394657E-02, 0.5387313739545944E+00, 0.5592732178453728E+00, 0.4422242955757759E+00, 0.6010599951901635E+00, 0.2729214395073326E+00, 0.8861412244709392E+00, 0.3303460134234409E+00, 0.3230906405176623E+00, 0.5979281304041633E+00, 0.1747480524852378E+00, 0.5019201598522602E+00, 0.3041629115671023E+00, 0.7344252813174572E+00, 0.5145018596354390E+00, 0.7032668540687012E+00, 0.4542245378490408E+00, 0.6883469599293311E+00, 0.7365046871633816E+00, 0.6892200550994384E+00, 0.9612097768268251E+00, 0.3687306903044788E+00, 0.4902139977664450E+00, 0.4004732628576805E+00, 0.2121831569752699E+00, 0.3483027732062219E+00, 0.4140454821962284E+00, 0.5590476647584736E+00, 0.1191647727722587E+00, 0.7126611758207085E+00},
			n:    21,
			zOut: []float64{0.2451808019367049E+01, 0.2128252216129400E+01, 0.1884015768998821E+01, 0.1793403527862625E+01, 0.1731698897481475E+01, 0.1605979322764491E+01, 0.1346135733575790E+01, 0.1222183737737495E+01, 0.1054517637460965E+01, 0.8541696661721662E+00, 0.8335553929836017E+00, 0.7643627371640215E+00, 0.6493538020248019E+00, 0.5117674171078171E+00, 0.4801902398267136E+00, 0.1830481952424627E+00, 0.1336751260833609E+00, 0.7548204386515513E-01, 0.4276641694583261E-01, 0.1152852330305948E-01, 0.4825291327526518E-04, 0.1046558665109411E-21, 0.5148060409033168E-10, 0.1742874021361670E-10, 0.1346135733575790E+01, 0.3944304526105059E-30, 0.1177385120842054E-08, 0.5739784145251797E-09, 0.1222183737737495E+01, 0.5228352308675503E-16, 0.2817945521555932E-11, 0.1891433639936970E-11, 0.1054517637460965E+01, 0.1775328283321620E-16, 0.9506244555381454E-11, 0.6780149876052939E-11, 0.8541696661721662E+00, 0.4289572169691169E-11, 0.3560204585171959E-10, 0.5552509181949502E-10, 0.8335553929836017E+00, 0.8925278809447169E-21, 0.1975794267501038E+02, 0.1975794267501038E+02, 0.9900000000000000E+02, 0.3029478458049887E+01, 0.2020202020202020E+01, 0.4057773458399609E-09, 0.6493538020248019E+00, 0.3196600703119960E-17, 0.4936238682205707E-06, 0.2172173957444601E-06, 0.5117674171078171E+00, 0.2475160268480207E-20, 0.1127880382626992E-04, 0.1031817493791828E-04, 0.4801902398267136E+00, 0.2327139670401985E-28, 0.8359957951383868E-05, 0.8045173068477057E-05, 0.1830481952424627E+00, 0.2763487036072314E-23, 0.1049648761427048E-04, 0.8607871065540926E-05, 0.1336751260833609E+00, 0.6392286336815055E-18, 0.5478798785309189E-04, 0.3178071334079827E-04, 0.7548204386515513E-01, 0.1886324196564483E-25, 0.4604839715747456E-05, 0.3206078546562559E-04, 0.4276641694583261E-01, 0.2130807968310579E-24, 0.2160396768988393E-02, 0.2401524776593251E-02, 0.1152852330305948E-01, 0.4597153189491485E-23, 0.1577121702032026E-02, 0.1548884890976950E-02, 0.4825291327526518E-04, 0.0000000000000000E+00, 0.1144436370534878E-01, 0.4805213857273134E-02},
			info: 0,
		},
		{
			z:    []float64{0.1204296195656108E+00, 0.2644532009534747E+00, 0.2981945708465356E+00, 0.3379851046426219E-01, 0.5872157683391875E+00, 0.3661495849020495E+00, 0.2783310889774259E+00, 0.8471009797765506E+00, 0.6087926721027869E+00, 0.7227374835926721E+00, 0.1157491272838733E+00, 0.5565144099551486E+00, 0.8949691253636921E+00, 0.4271566773654165E+00, 0.6387841251831913E+00, 0.3125047228733280E-01, 0.2092736013478247E-01, 0.9188324418961444E+00, 0.2275180796997284E+00, 0.4519662960462961E+00, 0.1155877493866434E+00, 0.3574940735178409E+00, 0.1489772525039219E+00, 0.3847908470095970E+00, 0.4780110568180811E+00, 0.3489586723075616E+00, 0.2559758537216194E+00, 0.7015930720521131E+00, 0.6512181083044030E+00, 0.1619173394771481E+00, 0.5094142654629881E+00, 0.9037662631759247E+00, 0.8829671919577533E+00, 0.5739200804260277E+00, 0.6584413366512047E+00, 0.7174965427568090E+00, 0.2705911606916550E+00, 0.6519148889036277E+00, 0.8284421625450799E+00, 0.1417109678591228E+00, 0.8472265674806589E-01, 0.3638971914001953E+00, 0.8802661885427909E-01, 0.5730596475071164E+00, 0.6156880168413905E+00, 0.3769532609458294E+00, 0.6302840487845919E+00, 0.8652028697564479E+00, 0.5886269364510055E-01, 0.9901772381773877E-01, 0.9679961670969370E+00, 0.9025028618130532E+00, 0.2858592550914936E+00, 0.4523960301168739E-01, 0.9961073801516410E+00, 0.5404176614913075E+00, 0.2161216977420954E+00, 0.6517643909270738E+00, 0.1408193879882935E-01, 0.2323315528058443E+00, 0.4040647670376405E+00, 0.5063393857452272E+00, 0.2149141182072994E+00, 0.8405813011731451E+00, 0.8044826512704509E+00, 0.5364711289689184E+00, 0.2922476360373625E+00, 0.9518940562302493E+00, 0.1980063292193738E+00, 0.6243757640305569E-01, 0.3641799792979717E+00, 0.4781083740929701E+00, 0.3488268687432241E+00, 0.8284107864073382E+00, 0.5855039308834178E+00, 0.2593007284605915E+00, 0.9998845891621441E+00, 0.5048349794999019E+00, 0.5828201446782487E+00, 0.8921301000231074E+00, 0.2672402237051515E+00, 0.3352853759971494E+00, 0.5977528180071631E+00, 0.1151194604047919E+00},
			n:    21,
			zOut: []float64{0.2557315455353142E+01, 0.2083235685800309E+01, 0.1875944452104717E+01, 0.1852714062580819E+01, 0.1643995946812320E+01, 0.1314218355716144E+01, 0.1293789198407225E+01, 0.1108624029644386E+01, 0.1060866236520538E+01, 0.6893140608436983E+00, 0.6265017468006577E+00, 0.6169803536834606E+00, 0.4472638275156540E+00, 0.3909824987049766E+00, 0.2990543016857948E+00, 0.1644755201813542E+00, 0.1145158516686298E+00, 0.6240413476421734E-01, 0.2540980528285533E-01, 0.1114684260186565E-01, 0.3077034058364147E-04, 0.1514612938024343E-27, 0.1068838715614041E-10, 0.1816953714834036E-10, 0.1293789198407225E+01, 0.1634858280249726E-16, 0.1546152805249061E-09, 0.7139985980503436E-10, 0.1108624029644386E+01, 0.1852295205932338E-17, 0.2931760870340471E-08, 0.5717158524063146E-08, 0.1060866236520538E+01, 0.1656986554198631E-24, 0.3897819821179759E-07, 0.2050786216968739E-07, 0.6893140608436983E+00, 0.1054613376713825E-21, 0.1114796130440952E-06, 0.1754727955273104E-06, 0.6265017468006577E+00, 0.0000000000000000E+00, 0.1823878313701335E+02, 0.1823878313701335E+02, 0.1020000000000000E+03, 0.3172335600907029E+01, 0.3921568627450980E+01, 0.4944285541926277E-08, 0.4472638275156540E+00, 0.7188113412291574E-19, 0.5360570179815202E-09, 0.5055207828345367E-09, 0.3909824987049766E+00, 0.7011182861561962E-17, 0.5424596001050196E-08, 0.1149191082508281E-07, 0.2990543016857948E+00, 0.1397964075969312E-24, 0.2148922978775398E-05, 0.1039924072201963E-05, 0.1644755201813542E+00, 0.1424715207546021E-19, 0.1286356681922344E-03, 0.6709371685087021E-04, 0.1145158516686298E+00, 0.1923558431290915E-24, 0.1040681698574874E-03, 0.9690504707177204E-04, 0.6240413476421734E-01, 0.3021337266996475E-27, 0.3740811812377314E-06, 0.8725750691496404E-06, 0.2540980528285533E-01, 0.6622271595176623E-28, 0.2937391640354928E-03, 0.1148070460566611E-03, 0.1114684260186565E-01, 0.3845696912952433E-29, 0.9582588059437623E-03, 0.1284334386264186E-02, 0.3077034058364147E-04, 0.1610185014413679E-23, 0.1713355521956925E-01, 0.1560043716021792E-01},
			info: 0,
		},
		{
			z:    []float64{0.6550531738065145E+00, 0.9006303757888976E+00, 0.9204609778380736E+00, 0.4658341923257867E+00, 0.5436644687290008E+00, 0.2908556755036122E+00, 0.1209811495723523E+00, 0.6227378829506942E+00, 0.3843019792679099E+00, 0.4431829610151485E+00, 0.4903210952175132E+00, 0.4210264133022774E+00, 0.3200750839632230E+00, 0.2252794777033682E+00, 0.5128630958928748E-01, 0.5136099492114639E+00, 0.4906286803922755E+00, 0.1747340137845675E+00, 0.6367238354860758E+00, 0.7515407582129341E+00, 0.4275500565044866E+00, 0.2014278558825494E+00, 0.7137603260056157E+00, 0.8686875192977085E+00, 0.7169591605793454E+00, 0.9050302202813415E+00, 0.9899533400874350E-01, 0.9969046692447475E+00, 0.6121260436585757E+00, 0.6518683331715626E+00, 0.5607404970528083E+00, 0.4000687152662108E-03, 0.1221862374388689E+00, 0.4484923714483016E+00, 0.4074629631039243E+00, 0.2322333097032281E-01, 0.4732551077558298E+00, 0.7342979327909982E+00, 0.3458749207719752E+00, 0.7729568887054280E+00, 0.2047858477824707E+00, 0.7991048365911296E+00, 0.5370237056253078E+00, 0.2233168849800682E-01, 0.5135288197983614E+00, 0.9129664417816657E+00, 0.9407166870649586E+00, 0.4321958081203685E+00, 0.5563774896630214E-01, 0.9494906037764664E+00, 0.9674063566885983E+00, 0.2302624676329315E+00, 0.1506811640032519E+00, 0.4445855507514112E+00, 0.6038782945102905E-01, 0.4499742519338505E+00, 0.1940991655604004E+00, 0.6493052482066142E-01, 0.2902929850901331E+00, 0.2876163569026815E+00, 0.9140475482126542E+00, 0.7006705577674063E+00, 0.2819386709216735E+00, 0.9125816873773608E+00, 0.9433170896498744E-01, 0.1632821600392161E+00, 0.3150556796654537E+00, 0.4051206017420083E+00, 0.2687493393481850E+00, 0.5768057516945001E+00, 0.4306858988251338E+00, 0.3827211690774009E+00, 0.3593520435650671E+00, 0.5610694397953092E+00, 0.7151178950427733E+00, 0.7138830385076256E-01, 0.5281951083990266E-01, 0.4880677722651299E+00, 0.9334247489037772E+00, 0.6764242544288053E+00, 0.8551178347453967E+00, 0.1876923010834376E+00, 0.7408118952029578E+00, 0.7967219280126981E+00},
			n:    21,
			zOut: []float64{0.2493394334366539E+01, 0.2460477172068809E+01, 0.2083159145455880E+01, 0.1779228959307519E+01, 0.1724095029781113E+01, 0.1560812668654360E+01, 0.1120482054552078E+01, 0.1088744749350826E+01, 0.9613598859526896E+00, 0.9456175481954929E+00, 0.9119552728538921E+00, 0.7767939725863090E+00, 0.6893167116879397E+00, 0.4408717103973492E+00, 0.2706330695081346E+00, 0.2343515836463984E+00, 0.6795401725828518E-01, 0.6600498464192311E-01, 0.2004873521291848E-01, 0.1421097732958122E-01, 0.4015560238058376E-03, 0.2226481018895784E-25, 0.2564499425740104E-12, 0.2658329977886514E-12, 0.1120482054552078E+01, 0.4315813836253463E-23, 0.1335501643346940E-12, 0.1368307862991113E-12, 0.1088744749350826E+01, 0.3926080427856238E-18, 0.1185565636052547E-12, 0.1214013431095007E-12, 0.9613598859526896E+00, 0.2719992401202049E-26, 0.9906513980413714E-13, 0.1012398200894782E-12, 0.9456175481954929E+00, 0.3937993638863291E-26, 0.9389079133840153E-13, 0.9189527811944332E-13, 0.9119552728538921E+00, 0.1580250204306566E-21, 0.1970991413883184E+02, 0.1970991413883184E+02, 0.1080000000000000E+03, 0.3401360544217687E+01, 0.8333333333333334E+01, 0.6496201881821226E-13, 0.6893167116879397E+00, 0.2040319821514109E-19, 0.6886568712781699E-13, 0.6342531805840532E-13, 0.4408717103973492E+00, 0.9116462109840799E-16, 0.6575726430673280E-12, 0.1146915438924438E-11, 0.2706330695081346E+00, 0.6100742213737356E-14, 0.5548784478521690E-10, 0.9411844035493366E-10, 0.2343515836463984E+00, 0.2464780121144947E-24, 0.1267470050081896E-08, 0.7549586177318002E-09, 0.6795401725828518E-01, 0.6058451752097371E-27, 0.2494439518946058E-07, 0.4030144639288306E-07, 0.6600498464192311E-01, 0.1931466728993349E-20, 0.1742718098912744E-06, 0.1063050845015484E-06, 0.2004873521291848E-01, 0.2078695185822955E-22, 0.1365398583912438E-05, 0.6056776980192580E-06, 0.1421097732958122E-01, 0.3944304526105059E-30, 0.8436693551699617E-06, 0.8772483974218289E-06, 0.4015560238058376E-03, 0.5424456586666688E-20, 0.1372662528173007E-02, 0.2323379424199439E-03},
			info: 0,
		},
		{
			z:    []float64{0.5955158028190186E+00, 0.9601282390728449E-01, 0.2944266054820921E+00, 0.3924250378682403E+00, 0.4462256781176310E-01, 0.5690757232665151E+00, 0.7855339329019884E-01, 0.8681545225840360E+00, 0.7288006955920397E+00, 0.2963229749240101E+00, 0.2728029241156386E+00, 0.4237536504971896E+00, 0.9003368942847588E+00, 0.7357957067011865E+00, 0.4901129309073038E-01, 0.4716171621744532E+00, 0.4890099406800087E+00, 0.3570165784460172E+00, 0.7160897857565474E-01, 0.6109824519907316E+00, 0.1221913259189585E+00, 0.1847302629402909E+00, 0.4009366884871202E-01, 0.9703472440233364E+00, 0.6369052870991366E+00, 0.5580400433416689E+00, 0.9024815855223057E+00, 0.1227045689636573E+00, 0.9668897094172768E+00, 0.5989409973498315E+00, 0.9709139844325040E+00, 0.9532564522700616E+00, 0.6239574483953332E+00, 0.7056703505253071E+00, 0.9506691238320553E-01, 0.7998586962365666E+00, 0.6556519136464994E+00, 0.4181856697120553E+00, 0.9478314170226607E+00, 0.3832663309972284E-01, 0.4257218688507075E+00, 0.2049443327129853E+00, 0.9168853967010917E+00, 0.5423988966788751E-02, 0.3002879152136316E+00, 0.7743151012217909E+00, 0.5154350917338609E+00, 0.9658560730800175E+00, 0.6314599250676610E+00, 0.5584937104443134E+00, 0.4741445335250092E+00, 0.3157508358915576E+00, 0.4349529093465575E+00, 0.7860003715032308E+00, 0.8695087804152180E-02, 0.3920445280700808E+00, 0.3765485221235618E+00, 0.4334604109656897E+00, 0.1636117026415890E+00, 0.3006093728066712E+00, 0.2129148932459900E+00, 0.3092975700919495E+00, 0.1986761620686717E-01, 0.9075541031649185E+00, 0.9781911433730479E+00, 0.6945955075731600E+00, 0.9959584477684137E+00, 0.7505944883792014E+00, 0.7627854018352902E+00, 0.1267035334952504E+00, 0.8056780234474171E+00, 0.4507060196586773E+00, 0.1799397964631048E+00, 0.7554691089798000E+00, 0.3012534688386570E+00, 0.9612172487804931E+00, 0.4056708256608614E+00, 0.6648221051068217E+00, 0.1966051207552482E+00, 0.6661008672469584E+00, 0.6875730168317549E+00, 0.9591133633640367E+00, 0.8995318056907078E+00, 0.1435403265941513E+00},
			n:    21,
			zOut: []float64{0.2826953331143077E+01, 0.2184946373215138E+01, 0.2011432985807930E+01, 0.1983939371799505E+01, 0.1783403572375380E+01, 0.1527291743334771E+01, 0.1218359458122959E+01, 0.1031625203087357E+01, 0.9141017580644314E+00, 0.8935891545682184E+00, 0.7995009441187808E+00, 0.6760292370761557E+00, 0.5375939998175604E+00, 0.5031272745805969E+00, 0.4136950654384539E+00, 0.4077555149266558E+00, 0.1977184484214556E+00, 0.1172157517521918E+00, 0.4197144528461013E-01, 0.1336078057720851E-01, 0.3545889291755145E-06, 0.2544082546855865E-16, 0.2583036406189355E-05, 0.3206950450307273E-05, 0.1218359458122959E+01, 0.6512835633504673E-25, 0.5241867641497365E-06, 0.5644106780290774E-06, 0.1031625203087357E+01, 0.0000000000000000E+00, 0.4305359229680111E-07, 0.8271940560154021E-07, 0.9141017580644314E+00, 0.0000000000000000E+00, 0.3911233321019150E-06, 0.2241550799736728E-06, 0.8935891545682184E+00, 0.2385737520619250E-22, 0.4206034452718400E-11, 0.9899461762377114E-11, 0.7995009441187808E+00, 0.9331359083488707E-16, 0.2008361176810137E+02, 0.2008361176810137E+02, 0.1130000000000000E+03, 0.3444444444444445E+01, 0.6194690265486726E+01, 0.1651034509856568E-04, 0.5375939998175604E+00, 0.1421716677825516E-23, 0.1528222338287967E-04, 0.2433829126418849E-04, 0.5031272745805969E+00, 0.6632003279205155E-19, 0.4757657931696731E-05, 0.2751276356124619E-05, 0.4136950654384539E+00, 0.1680371209901916E-16, 0.1318069984220972E-06, 0.1659409431372840E-06, 0.4077555149266558E+00, 0.2554525986432683E-22, 0.5944173486813264E-06, 0.1429245864393051E-05, 0.1977184484214556E+00, 0.1494891415393817E-27, 0.8465737022131437E-03, 0.4160790916597339E-03, 0.1172157517521918E+00, 0.6780166836549887E-20, 0.3655496023749206E-04, 0.5562211290196199E-04, 0.4197144528461013E-01, 0.1803076020780872E-22, 0.2337686132102848E-02, 0.5241185550833300E-02, 0.1336078057720851E-01, 0.3234329711406148E-27, 0.6369386968947829E-02, 0.2727172056719641E-02, 0.3545889291755145E-06, 0.0000000000000000E+00, 0.6135698199493001E-02, 0.4453215245051444E-02},
			info: 0,
		},
		{
			z:    []float64{0.5256994769563229E+00, 0.4751970954150097E+00, 0.5550798296722601E+00, 0.3565746805143110E+00, 0.3971523551134015E-01, 0.7088262788771817E+00, 0.6369448567868402E+00, 0.6372974860231623E+00, 0.5963599363854956E+00, 0.7996832763398349E+00, 0.7266271077857295E+00, 0.3813393981851099E+00, 0.8884393254050771E-01, 0.8685934834267716E+00, 0.3797066359545181E+00, 0.4541910736411658E+00, 0.6285354031372432E+00, 0.5295256159852654E+00, 0.8908170817260027E+00, 0.5793264969983637E+00, 0.6056871126785802E+00, 0.6045507124525907E+00, 0.9734230310286895E-01, 0.9482358811543057E+00, 0.8804820874856442E+00, 0.4472263119108183E+00, 0.5843860139759072E+00, 0.5372922689923049E+00, 0.5799351791336661E+00, 0.5116182356749631E+00, 0.3816355069915517E+00, 0.6475808331559241E+00, 0.8746013736579017E-02, 0.8411650861050215E+00, 0.2992737481736951E-01, 0.6246901010621124E+00, 0.3123078564554325E+00, 0.7411164234121632E+00, 0.5960432842954070E+00, 0.8102594121204127E+00, 0.3552957522458323E+00, 0.6235337320803771E+00, 0.5962195242077326E+00, 0.4845790335098474E+00, 0.4045953379382708E+00, 0.2452522545656534E+00, 0.2743318280596185E+00, 0.2023077390748810E+00, 0.4206973723636350E+00, 0.4039949452544738E+00, 0.9114680949622015E+00, 0.7779482986378454E+00, 0.6812819304160683E+00, 0.5580962104642140E-01, 0.4287711156774094E+00, 0.9817581515129085E+00, 0.4431516884275333E+00, 0.6237912221093689E-01, 0.8364254576520689E+00, 0.8759899245723605E+00, 0.8787473635901457E+00, 0.4702211826699049E+00, 0.8284521047414805E+00, 0.2041855966018515E+00, 0.8060130309143703E+00, 0.5501723689038956E+00, 0.2188842792675016E-01, 0.5672241818001204E+00, 0.6450459353754012E+00, 0.2431902854401001E+00, 0.5193698672717711E+00, 0.1962120228423043E+00, 0.6361488000943472E-01, 0.5342761659162559E+00, 0.1758994081846166E-01, 0.7459187699999678E+00, 0.5953394313659540E+00, 0.9379918174958790E+00, 0.4032831592210619E+00, 0.7704238877222783E+00, 0.9410981148731864E+00, 0.8355998775704846E+00, 0.7904637124553195E+00, 0.4324126135288506E+00},
			n:    21,
			zOut: []float64{0.2360605380677641E+01, 0.2349812365456326E+01, 0.2259072978559188E+01, 0.2074669549844899E+01, 0.1734529465065066E+01, 0.1635336277856887E+01, 0.1535245292061319E+01, 0.1453596007978833E+01, 0.1156130097482147E+01, 0.1084746900314168E+01, 0.1034520391904089E+01, 0.8201881896309742E+00, 0.6191782955028541E+00, 0.5267433243518804E+00, 0.4355086546946592E+00, 0.4155619327338856E+00, 0.3396295360604018E+00, 0.9317218785919168E-01, 0.5773665727788814E-01, 0.1842462732727454E-01, 0.2018232369069597E-07, 0.4517585523147273E-24, 0.2167663685989129E-10, 0.1523023581682742E-10, 0.1535245292061319E+01, 0.1340432450151543E-25, 0.8069500146896454E-10, 0.1279555645700667E-09, 0.1453596007978833E+01, 0.6673291668607916E-16, 0.1337219714947293E-07, 0.4537550993048733E-08, 0.1156130097482147E+01, 0.2388039732285047E-25, 0.6170125127971897E-05, 0.1353762614268721E-04, 0.1084746900314168E+01, 0.3862470231561244E-18, 0.2338329578753200E-04, 0.2318969913953679E-04, 0.1034520391904089E+01, 0.6409494854920721E-30, 0.2200440813282189E+02, 0.2200440813282190E+02, 0.9400000000000000E+02, 0.2857142857142857E+01, 0.0000000000000000E+00, 0.2156912239043876E-04, 0.6191782955028541E+00, 0.4311940021813490E-19, 0.2098085305628239E-04, 0.2047701462113856E-04, 0.5267433243518804E+00, 0.7292491488692259E-18, 0.2483658471590867E-04, 0.2278231239253353E-04, 0.4355086546946592E+00, 0.6335396825622830E-16, 0.5560492899787359E-04, 0.8068766398366361E-04, 0.4155619327338856E+00, 0.7336406418555410E-28, 0.3220618566867426E-03, 0.1956958209756436E-03, 0.3396295360604018E+00, 0.1297676189088564E-27, 0.4121897802138222E-02, 0.2587205720417875E-02, 0.9317218785919168E-01, 0.7651950780643815E-28, 0.2826512581544380E-02, 0.2814359488753958E-02, 0.5773665727788814E-01, 0.4923437167052441E-21, 0.1821079370679042E-04, 0.2969056479875054E-04, 0.1842462732727454E-01, 0.4842851680813876E-21, 0.4217326450522519E-02, 0.5738852897157041E-02, 0.2018232369069597E-07, 0.2018232369069597E-07, 0.3702854219118327E-02, 0.7091388009151286E-02},
			info: 0,
		},
		{
			z:    []float64{0.5196081465664333E+00, 0.1450873999446859E+00, 0.1902498073897446E+00, 0.9459083122514356E-01, 0.6971960507146802E-01, 0.2126447963850432E+00, 0.1693138701795316E-01, 0.6615449591751588E-01, 0.6671851537110856E+00, 0.7116922416084167E+00, 0.7735840319989629E+00, 0.8593705742984634E+00, 0.6060015557980403E+00, 0.1071620560007485E+00, 0.6056582690297291E+00, 0.2064801115284116E+00, 0.4992099275808294E+00, 0.9056413330105546E+00, 0.6051575714193578E+00, 0.1395071235858423E+00, 0.3386142237730057E+00, 0.2065895571122717E+00, 0.2774285740015214E-01, 0.4649920904615868E+00, 0.9062834189437683E+00, 0.3022801499192989E+00, 0.2125071125007829E+00, 0.2366624610094756E+00, 0.3232052419625674E+00, 0.5879745505340277E+00, 0.3212955913685801E+00, 0.9110441809347004E+00, 0.2144184579231917E+00, 0.5617161155671668E-01, 0.9725855527757206E+00, 0.9798782363582490E+00, 0.9666637418953663E+00, 0.1842420381101751E+00, 0.9810363127661145E+00, 0.4061071205983764E+00, 0.4929437214282740E+00, 0.3087398230344144E+00, 0.8692083335886002E+00, 0.7401065606674918E+00, 0.6829833274325647E+00, 0.9797795404622264E+00, 0.6101846761247042E+00, 0.7796537878703258E+00, 0.4568641605073986E-01, 0.1752946998854403E+00, 0.7558659638245062E+00, 0.4081915619381170E+00, 0.5344465439920063E+00, 0.3199118710728778E+00, 0.3613720388280138E+00, 0.1409062127217534E+00, 0.2811520826338662E+00, 0.9462348779627723E+00, 0.4911007415294493E+00, 0.2488061765237712E+00, 0.1661818317483100E+00, 0.1203910774987040E+00, 0.4544163143251944E+00, 0.8767242907488393E+00, 0.6688224142601292E+00, 0.2497550267789769E+00, 0.2658845545571695E+00, 0.4072601097670568E+00, 0.5517137496493807E+00, 0.4206663434956901E+00, 0.3655154771809294E+00, 0.6509199815744504E-01, 0.6480911975614350E+00, 0.1271267318339999E+00, 0.8874102966539428E+00, 0.9607257650026479E+00, 0.8902777005093224E+00, 0.7074702462790970E+00, 0.5045879991550398E+00, 0.3357826713243962E+00, 0.9303671751018463E+00, 0.3667268693834904E+00, 0.7709773173263057E+00, 0.1274123220614284E+00},
			n:    21,
			zOut: []float64{0.2669666350125788E+01, 0.2458073928301015E+01, 0.1960648799383531E+01, 0.1674511937828339E+01, 0.1629444569956276E+01, 0.1605168511500282E+01, 0.1169506153633164E+01, 0.8378899015114316E+00, 0.8214470290652885E+00, 0.7290022816300561E+00, 0.5956393798847858E+00, 0.5147057006185641E+00, 0.3415078453317431E+00, 0.3316800239182673E+00, 0.2245847640277715E+00, 0.1981086483360383E+00, 0.1729134064527326E+00, 0.1081292683666610E+00, 0.4856623040087921E-01, 0.2705373732739987E-02, 0.9745444154776346E-03, 0.7298616502634276E-19, 0.4601636530183765E-14, 0.3001385089664378E-13, 0.1169506153633164E+01, 0.5522026336547083E-28, 0.9439817212889541E-12, 0.1534086358433234E-11, 0.8378899015114316E+00, 0.7174347921301560E-11, 0.1095021852663145E-10, 0.1580074435702411E-10, 0.8214470290652885E+00, 0.3131705799125763E-20, 0.2247124942125492E-10, 0.3228253356903498E-10, 0.7290022816300561E+00, 0.6063934311659721E-18, 0.3794147471177530E-13, 0.3318524623173314E-12, 0.5956393798847858E+00, 0.3765687391680646E-13, 0.1809487464842083E+02, 0.1809487464842083E+02, 0.1050000000000000E+03, 0.3215419501133787E+01, 0.3809523809523809E+01, 0.2077325438490571E-07, 0.3415078453317431E+00, 0.1645262238367970E-20, 0.7912752069326401E-07, 0.6053132028546428E-07, 0.3316800239182673E+00, 0.4184118241292247E-26, 0.1343987419983520E-06, 0.1754397466154310E-06, 0.2245847640277715E+00, 0.3340387958036406E-22, 0.6150231447604006E-06, 0.4780001259056422E-06, 0.1981086483360383E+00, 0.1070733604161610E-20, 0.1003229473607176E-05, 0.1287291281183416E-05, 0.1729134064527326E+00, 0.1298955481663118E-20, 0.1611116216931278E-05, 0.3807035614938680E-06, 0.1081292683666610E+00, 0.4901279578837721E-23, 0.3565927169328702E-03, 0.2312261838015572E-03, 0.4856623040087921E-01, 0.3325573155747123E-13, 0.4389787505319806E-03, 0.9715248586459800E-03, 0.2705373732739987E-02, 0.1348432029845013E-20, 0.7371401467208961E-04, 0.2975668602269332E-04, 0.9745444154776346E-03, 0.4135903062570168E-24, 0.7273370201297700E-04, 0.3602244970122637E-04},
			info: 0,
		},
		{
			z:    []float64{0.3441168249350421E+00, 0.7064572711980356E+00, 0.6589781374655990E+00, 0.8687841598825752E+00, 0.7811747459515918E+00, 0.7470750719206745E+00, 0.7591454338662803E+00, 0.7956934079357347E-01, 0.3416983743372617E+00, 0.3365394610260509E+00, 0.8635365040683821E+00, 0.4793906833418223E+00, 0.9394584418725971E+00, 0.7354375226759881E+00, 0.3959912256877245E+00, 0.4088558388864650E+00, 0.6899389113237457E+00, 0.6329180251180861E+00, 0.6760812245614891E+00, 0.1743991444245150E+00, 0.3881927716152306E+00, 0.2103848624556167E+00, 0.5309784956583943E+00, 0.5758384660199964E+00, 0.6542468116269122E+00, 0.2141275453071043E+00, 0.4917208240158387E+00, 0.6996271491850585E+00, 0.7074081025976766E+00, 0.1896811377589238E+00, 0.4872676647104760E+00, 0.8987347035581787E+00, 0.1651654313203957E+00, 0.7969431328829826E+00, 0.8979672856778195E+00, 0.2621352236509209E+00, 0.2994430908669790E+00, 0.8412000921937168E+00, 0.6910228781616401E-01, 0.9849719192333963E+00, 0.2835195798153757E+00, 0.6126955834278749E+00, 0.9957514180764350E+00, 0.4133859474205875E+00, 0.2126790023013077E+00, 0.9230696247878700E-01, 0.5165813646587417E+00, 0.9232126059423650E+00, 0.6160486127374017E+00, 0.5529065454633691E+00, 0.6862175458708878E+00, 0.2677675454795836E-01, 0.6729513992144193E+00, 0.8612993184178528E+00, 0.5455358966165057E+00, 0.2518716982253303E+00, 0.5312849864208884E+00, 0.7551228210749875E+00, 0.1095520838657484E+00, 0.8767983608629261E+00, 0.9578119141004069E+00, 0.6878921114137557E+00, 0.2890574974795965E+00, 0.9851780344062913E+00, 0.9392886821673129E+00, 0.4125982690623264E+00, 0.6817912182549608E+00, 0.7805388095263401E+00, 0.1567802887930034E+00, 0.1901362770322003E+00, 0.5155717108920093E+00, 0.5470641908290981E+00, 0.9496522047623522E+00, 0.4367251554723609E+00, 0.7016249991347693E+00, 0.6652534390290816E+00, 0.3892590437090979E+00, 0.1464591367640408E+00, 0.9179795596761495E+00, 0.1527313083317114E+00, 0.8661173016992874E+00, 0.8044520995141484E+00, 0.4399581275677011E+00, 0.6590078920726725E+00},
			n:    21,
			zOut: []float64{0.2654671686308588E+01, 0.2409077903364995E+01, 0.2090082174851482E+01, 0.1940605709090538E+01, 0.1909776782011245E+01, 0.1629990303184568E+01, 0.1566043989638403E+01, 0.1473218419964192E+01, 0.1321200673977518E+01, 0.1275771383585936E+01, 0.9826159405158056E+00, 0.8184731809209335E+00, 0.6715230178331039E+00, 0.5669018352800101E+00, 0.3641671875118937E+00, 0.2699135892930649E+00, 0.1446269090001342E+00, 0.8937359624216266E-01, 0.5860931848549461E-01, 0.3094108728329079E-01, 0.6182329613034853E-03, 0.9233403587623171E-22, 0.2552413399194827E-12, 0.1309204970235786E-12, 0.1566043989638403E+01, 0.2003762315848454E-18, 0.9222125989396634E-12, 0.1831545934750265E-11, 0.1473218419964192E+01, 0.1899959664962892E-11, 0.5682525570474145E-10, 0.1113952503246639E-09, 0.1321200673977518E+01, 0.5311918312726658E-16, 0.2115322762070563E-08, 0.1208354514490733E-08, 0.1275771383585936E+01, 0.1424261829852281E-16, 0.6059411460508015E-08, 0.1032477558436010E-07, 0.9826159405158056E+00, 0.5397353496908506E-22, 0.2226820292130466E+02, 0.2226820292130466E+02, 0.9500000000000000E+02, 0.2927437641723356E+01, 0.1052631578947368E+01, 0.2555943178691847E-06, 0.6715230178331039E+00, 0.1218922493912602E-18, 0.4789189736721279E-06, 0.5466231772857101E-06, 0.5669018352800101E+00, 0.1893266172530428E-27, 0.9952011217509916E-06, 0.1128113648538554E-05, 0.3641671875118937E+00, 0.1993230626440035E-24, 0.2427606592544749E-05, 0.2156829827516489E-05, 0.2699135892930649E+00, 0.2761013168273541E-29, 0.3900010091178873E-05, 0.3393761026267541E-05, 0.1446269090001342E+00, 0.2150416673893778E-15, 0.8242388755752233E-05, 0.1105863902029246E-04, 0.8937359624216266E-01, 0.2934562567422164E-27, 0.3455780897358792E-04, 0.2250328346305790E-04, 0.5860931848549461E-01, 0.8147325136863849E-21, 0.2346512312055719E-04, 0.8962661529405173E-05, 0.3094108728329079E-01, 0.2460979885389816E-17, 0.5238375287742530E-03, 0.1080192509223090E-02, 0.6182329613034853E-03, 0.8383483586160605E-23, 0.7400552677463828E-02, 0.1581674273722876E-01},
			info: 0,
		},
		{
			z:    []float64{0.7940281584071446E+00, 0.8540600349699839E+00, 0.8158431165852809E-01, 0.5431841788581357E+00, 0.3696613346727944E+00, 0.2343742079469738E+00, 0.4891909888056500E-01, 0.6769876984160987E+00, 0.4777767465052760E+00, 0.1867381312399053E+00, 0.2018744873845245E+00, 0.5511201479607295E+00, 0.6938788283912793E+00, 0.8167542438070282E+00, 0.7904606414789531E+00, 0.9443564310071292E+00, 0.7287247677237652E-01, 0.8645122013586991E+00, 0.1884651475116826E+00, 0.3844755283611681E+00, 0.9959264361467982E+00, 0.6424370932833342E+00, 0.1972122925077952E+00, 0.2842024247377670E+00, 0.9819646913482807E+00, 0.9118347224008859E+00, 0.8184691845197246E+00, 0.7051587281589254E+00, 0.7604703230109544E+00, 0.6312964755149379E+00, 0.5240863862347888E+00, 0.3442050916384676E-01, 0.2415614308212055E+00, 0.2814868323669945E+00, 0.6529284673126197E+00, 0.3727305084153835E+00, 0.5033733868757848E+00, 0.2317122058804952E+00, 0.7555584130128312E+00, 0.5854566742645219E+00, 0.5481204696337160E+00, 0.8479425268049923E+00, 0.2310874615764000E+00, 0.1250993726775007E-01, 0.6243285982203539E-01, 0.8533587246073391E+00, 0.9203815588639257E+00, 0.9256849509751471E+00, 0.6691405057262187E+00, 0.8847091531299658E+00, 0.6783572983386376E+00, 0.4701257141291857E+00, 0.8976078424378102E+00, 0.8575018884445876E+00, 0.4119363561363949E+00, 0.2824477027676924E+00, 0.2787507690368071E+00, 0.7994878185780909E+00, 0.6141832897278305E+00, 0.6772728066124333E+00, 0.1568652581579784E+00, 0.8025492691231176E+00, 0.2609459151100056E+00, 0.4956700691019098E+00, 0.1008839464621498E+00, 0.6129709499983976E+00, 0.4551038858718992E-02, 0.8382785474023564E+00, 0.9327452694814308E+00, 0.9710431593941808E+00, 0.3785578217695214E+00, 0.9620839159000718E+00, 0.3183561960196257E-01, 0.9167635157854341E+00, 0.8989971039988554E+00, 0.2723769512210017E-01, 0.4176537489735596E+00, 0.9619881273217982E+00, 0.8761769579995293E+00, 0.6385245520487358E+00, 0.6821739872929905E+00, 0.3927943300877799E+00, 0.3299501391296433E-01, 0.6026481165267817E+00},
			n:    21,
			zOut: []float64{0.2841529467847260E+01, 0.2556740368064117E+01, 0.2029069210305357E+01, 0.1816492749229813E+01, 0.1807397947918166E+01, 0.1724050761249482E+01, 0.1394683696862690E+01, 0.1363282300837870E+01, 0.1235317529564993E+01, 0.1007821728138393E+01, 0.9728972767837659E+00, 0.8110838192609224E+00, 0.6190683239156912E+00, 0.3314348466978195E+00, 0.2969816495631171E+00, 0.2506421226611442E+00, 0.1551624233480766E+00, 0.1141049603637759E+00, 0.6806098279643324E-01, 0.4065442795375918E-01, 0.1509783792061944E-04, 0.2745446733802996E-19, 0.2334033424405603E-14, 0.7232047343832039E-15, 0.1394683696862690E+01, 0.7652022750203868E-17, 0.1502464319787860E-09, 0.5990324067027169E-10, 0.1363282300837870E+01, 0.5511008105786353E-19, 0.1215529748555577E-07, 0.3576186028975079E-08, 0.1235317529564993E+01, 0.6271335389136957E-17, 0.2589527811059762E-05, 0.3563684464760320E-05, 0.1007821728138393E+01, 0.6928731028484552E-15, 0.2286607739649456E-04, 0.2916328836696281E-04, 0.9728972767837659E+00, 0.4013724285764508E-25, 0.2143649169120057E+02, 0.2143649169120056E+02, 0.1080000000000000E+03, 0.3174603174603174E+01, 0.5555555555555555E+01, 0.5006788488237952E-04, 0.6190683239156912E+00, 0.1272779738919789E-24, 0.5392273695714327E-05, 0.6586751690782050E-05, 0.3314348466978195E+00, 0.9952016744778614E-24, 0.4588114650322476E-05, 0.4104030919118472E-05, 0.2969816495631171E+00, 0.1254277227268884E-20, 0.1579210272170037E-05, 0.2523508055026877E-05, 0.2506421226611442E+00, 0.1514835081255253E-22, 0.9117975197333799E-05, 0.1415311657319248E-04, 0.1551624233480766E+00, 0.3272195034856757E-26, 0.3128414626489953E-04, 0.4770966856082524E-04, 0.1141049603637759E+00, 0.7575261888623807E-20, 0.6277758538102519E-05, 0.4644422903178713E-05, 0.6806098279643324E-01, 0.3715389965619895E-21, 0.5956965112658162E-05, 0.4447744978930882E-05, 0.4065442795375918E-01, 0.1972152263052530E-30, 0.2965586553650948E-04, 0.1900611263569203E-04, 0.1509783792061944E-04, 0.1009741958682895E-26, 0.1608958133772104E-02, 0.9583670521235791E-03},
			info: 0,
		},
	} {
		z := make([]float64, len(test.z))
		copy(z, test.z)

		info := impl.Dlasq2(test.n, z)
		if !floats.EqualApprox(test.zOut, z, dTol) {
			diff := make([]float64, len(z))
			floats.SubTo(diff, z, test.zOut)
			for i := range diff {
				diff[i] = math.Abs(diff[i])
			}
			t.Errorf("Case %v, Z Mismatch", c)
		}
		if test.info != info {
			t.Errorf("Info mismatch. Want %v, got %v", test.info, info)
		}
	}

	rnd := rand.New(rand.NewSource(1))
	// Perform a bunch of random tests to check for access out of bounds or
	// infinite loops.
	// TODO(btracey): Implement direct tests.
	// bi := blas64.Implementation()
	for _, n := range []int{5, 8, 20, 25} {
		for k := 0; k < 10; k++ {
			z := make([]float64, 4*n)
			for i := range z {
				z[i] = rnd.Float64()
			}
			zCopy := make([]float64, len(z))
			copy(zCopy, z)

			// Compute the eigenvalues
			impl.Dlasq2(n, z)

			// Below is the code to test the eigenvalues. Eventually implement
			// real tests.
			// The code below is missing the transformation from L and U into
			// the symmetric tridiagonal matrix.
			// See discussion http://icl.cs.utk.edu/lapack-forum/viewtopic.php?f=5&t=4839
			// for format.

			/*
				ldl := n
				ldu := n
				u := make([]float64, n*n)
				for i := 0; i < n; i++ {
					u[i*ldu+i] = zCopy[2*i]
					if i != n-1 {
						u[i*ldu+i+1] = 1
					}
				}
				l := make([]float64, n*n)
				for i := 0; i < n; i++ {
					l[i*ldl+i] = 1
					if i != n-1 {
						l[(i+1)*ldl+i] = zCopy[2*i+1]
					}
				}

				ldTriDi := n
				triDi := make([]float64, n*n)
				bi.Dgemm(blas.NoTrans, blas.NoTrans, n, n, n, 1, l, ldl, u, ldu, 0, triDi, ldTriDi)

				tridi2 := make([]float64, n*n)
				bi.Dgemm(blas.Trans, blas.NoTrans, n, n, n, 1, triDi, n, triDi, n, 0, tridi2, n)

				// Eigenvalues have the property that det(A - lambda I ) = 0
				triDiCopy := make([]float64, len(triDi))
				copy(triDiCopy, triDi)
				for _, lambda := range z[1:n] {
					copy(triDi, triDiCopy)
					for i := 0; i < n; i++ {
						triDi[i*n+i] -= lambda
					}

					// Compute LU
					//ipiv := make([]int, n)
					//impl.Dgetrf(n, n, triDi, n, ipiv)
					ok := impl.Dpotrf(blas.Upper, n, triDi, n)
					fmt.Println(ok)

					var det float64
					for i := 0; i < n; i++ {
						det += math.Log(math.Abs(triDi[i*n+i]))
					}
					fmt.Println("det = ", math.Exp(det))
				}
			*/
		}
	}
}
