/*

  Nosuid TCP/IP ping 1.6 beta by Michal Zalewski <lcamtuf@coredump.cx>
  --------------------------------------------------------------------
 
  The Nosuid TCP/IP ping and related utilities are free software; you
  can redistribute it and/or modify it under the terms of the GNU Library
  General Public License as published by the Free Software Foundation;
  either version 2 of the License, or (at your option) any later version.
 
*/

#include <stdio.h>
#include <unistd.h>
#include <sys/param.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <netdb.h>
#include <stdarg.h>
#include <errno.h>
#include <signal.h>
#include <getopt.h>
#include <stdlib.h>
#include <string.h>

#define DEFAULT_PORT  65535
#define DEFAULT_TMOUT 4
#define DEFAULT_REP   0
#define DEFAULT_DELAY 1
#define MS_THRES      500


#define debug(x...) fprintf(stderr,x)

#define fatal(x...) do { \
    debug(x); \
    exit(1); \
  } while (0)

#define pfatal(x) do { \
    perror(x); \
    exit(1); \
  } while (0)


static char iptext[20],got_alrm;
static unsigned int received,sent,max_rtt,total_rtt;
static unsigned int min_rtt=(unsigned int)-1;


static void sig_alarm(int dummy) {
  got_alrm=1; 
}


static void do_stats(int x) {

  printf("\n-- %s ping statistics --\n",iptext);
  printf("%d packet(s) transmitted, %d packet(s) "
         "received, %d%% packets lost.\n",sent,
         received,(sent-received)*100/sent);

  printf("round-trip statistics min/avg/max: ");

  if (received) {

     printf("%.1f/%.1f/%.1f ms",(float)min_rtt/1000,
            (float)total_rtt/received/1000,(float)max_rtt/1000);

     if (min_rtt < MS_THRES) printf(" (%d/%d/%d usec)",min_rtt,
                                    total_rtt/received,max_rtt);
     puts(".");

  } else printf("(not available due to 100%% packets lost).\n");

  exit(0);
}


static void usage(void) {
  fatal("Usage: poink [ -i delay ] [ -c count ] [ -t timeout ] host\n");
}


int main(int argc,char* argv[]) {

  unsigned int count=DEFAULT_REP,
               delay=DEFAULT_DELAY,
               timeout=DEFAULT_TMOUT;

  unsigned int n=0,port=DEFAULT_PORT;

  char ch;
  unsigned char *iaddr;
  struct hostent* hp;
  struct sockaddr_in s;
  struct timeval tv1,tv2;
  
  printf("Nosuid TCP/IP ping 1.6 by <lcamtuf@coredump.cx>\n");

  while ((ch=getopt(argc,argv,"i:t:c:")) != EOF)
    switch(ch) {

      case 'c':
        if (sscanf(optarg,"%u",&count)!=1)
          fatal("poink: invalid number of pings.\n");
        break;

      case 'i':
        if (sscanf(optarg,"%u",&delay)!=1 || delay < 1 || delay > 1000)
          fatal("poink: invalid ping delay.\n");
        break;

      case 't':
        if (sscanf(optarg,"%u",&timeout)!=1 || timeout < 1 || timeout > 100)
          fatal("poink: invalid ping timeout.\n");
        break;

      default:
        usage();
    }
    
  if (argc-optind!=1) usage();
  
  hp=gethostbyname(argv[optind]);
  
  if (!hp) fatal("ping: host not found.\n");

  iaddr=(unsigned char*)hp->h_addr;
  
  // Eeevil.
  sprintf(iptext,"%u.%u.%u.%u",iaddr[0],iaddr[1],iaddr[2],iaddr[3]);
	  
  signal(SIGINT,do_stats);
  
  if (count) 
    printf("Pinging %s (%s) - %d packets, delay %d sec(s), "
           "timeout %d sec(s).\n",hp->h_name,iptext,count,delay,timeout);
  else
    printf("Pinging %s (%s) - delay %d sec(s), timeout %d "
           "sec(s).\n",hp->h_name,iptext,delay,timeout);

  while (!count || n++ < count) {
    int csock;
    char skew=0;
    struct sigaction siga;
    unsigned int delayed;

    // First, let's get a socket.
    memcpy((void*)&s.sin_addr,hp->h_addr,hp->h_length);
    s.sin_family=hp->h_addrtype;
    s.sin_port=htons(port);
    if ((csock=socket(AF_INET,SOCK_STREAM,0))<0) pfatal("socket");

    // Get ready for SIGALRM... We need this signal not to restart
    // the syscall, and this seems to be a good way to do it
    siga.sa_handler=sig_alarm;  
    siga.sa_flags=0;
    sigaction(SIGALRM,&siga,NULL);
    alarm(timeout);

    // Get time of day...
    gettimeofday(&tv1,NULL);

    sent++;

    if (connect(csock,(struct sockaddr*)&s,sizeof(s))) {
      if (got_alrm) {
        printf("No ping reply from %s within %d second(s)...\n",iptext,
               timeout);
        got_alrm=0;
        shutdown(csock,2);
        close(csock);
        continue;
      } else {
        if (errno != ECONNREFUSED) pfatal("connect");
        received++;
      }
    } else { 
      received++; 
      skew=1; 
      port--;
    }

    alarm(0);

    gettimeofday(&tv2,NULL);

    // Can overflow with extremely large delays, but c'mon...
    delayed=(tv2.tv_sec-tv1.tv_sec)*1000000 + (tv2.tv_usec - tv1.tv_usec);

    if (delayed >= MS_THRES)
      printf("Ping reply from %s: seq=%d, time=%.1f ms",iptext,sent-1,
             (float)delayed/1000);
    else
      printf("Ping reply from %s: seq=%d, time<0.1 ms (%d usec)",iptext,
             sent-1,delayed);

    if (skew) printf(" (skewed!)\n"); else puts("");

    total_rtt += delayed;
    if (delayed > max_rtt) max_rtt = delayed;
    if (delayed < min_rtt) min_rtt = delayed;

    // Be nice.
    shutdown(csock,2);
    close(csock);

    if (!count || n != count) usleep(delay*1000000);

  }

  do_stats(0);
  return 0;

}

