// SPDX-FileCopyrightText: Copyright 2015-2025 go-swagger maintainers
// SPDX-License-Identifier: Apache-2.0

package runtime

import (
	"context"
	"io"
	"net/http"

	"github.com/go-openapi/strfmt"
)

// OperationHandlerFunc an adapter for a function to the OperationHandler interface
type OperationHandlerFunc func(any) (any, error)

// Handle implements the operation handler interface
func (s OperationHandlerFunc) Handle(data any) (any, error) {
	return s(data)
}

// OperationHandler a handler for a swagger operation
type OperationHandler interface {
	Handle(any) (any, error)
}

// ConsumerFunc represents a function that can be used as a consumer
type ConsumerFunc func(io.Reader, any) error

// Consume consumes the reader into the data parameter
func (fn ConsumerFunc) Consume(reader io.Reader, data any) error {
	return fn(reader, data)
}

// Consumer implementations know how to bind the values on the provided interface to
// data provided by the request body
type Consumer interface {
	// Consume performs the binding of request values
	Consume(io.Reader, any) error
}

// ProducerFunc represents a function that can be used as a producer
type ProducerFunc func(io.Writer, any) error

// Produce produces the response for the provided data
func (f ProducerFunc) Produce(writer io.Writer, data any) error {
	return f(writer, data)
}

// Producer implementations know how to turn the provided interface into a valid
// HTTP response
type Producer interface {
	// Produce writes to the http response
	Produce(io.Writer, any) error
}

// AuthenticatorFunc turns a function into an authenticator
type AuthenticatorFunc func(any) (bool, any, error)

// Authenticate authenticates the request with the provided data
func (f AuthenticatorFunc) Authenticate(params any) (bool, any, error) {
	return f(params)
}

// Authenticator represents an authentication strategy
// implementations of Authenticator know how to authenticate the
// request data and translate that into a valid principal object or an error
type Authenticator interface {
	Authenticate(any) (bool, any, error)
}

// AuthorizerFunc turns a function into an authorizer
type AuthorizerFunc func(*http.Request, any) error

// Authorize authorizes the processing of the request for the principal
func (f AuthorizerFunc) Authorize(r *http.Request, principal any) error {
	return f(r, principal)
}

// Authorizer represents an authorization strategy
// implementations of Authorizer know how to authorize the principal object
// using the request data and returns error if unauthorized
type Authorizer interface {
	Authorize(*http.Request, any) error
}

// Validatable types implementing this interface allow customizing their validation
// this will be used instead of the reflective validation based on the spec document.
// the implementations are assumed to have been generated by the swagger tool so they should
// contain all the validations obtained from the spec
type Validatable interface {
	Validate(strfmt.Registry) error
}

// ContextValidatable types implementing this interface allow customizing their validation
// this will be used instead of the reflective validation based on the spec document.
// the implementations are assumed to have been generated by the swagger tool so they should
// contain all the context validations obtained from the spec
type ContextValidatable interface {
	ContextValidate(context.Context, strfmt.Registry) error
}
