// Copyright 2025 Google LLC
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package packageindex_test

import (
	"testing"

	"github.com/google/go-cmp/cmp"
	"github.com/google/go-cmp/cmp/cmpopts"
	"github.com/google/osv-scalibr/extractor"
	"github.com/google/osv-scalibr/extractor/filesystem/language/javascript/packagejson"
	"github.com/google/osv-scalibr/extractor/filesystem/language/python/wheelegg"
	"github.com/google/osv-scalibr/packageindex"
	"github.com/google/osv-scalibr/purl"
)

var (
	pkgLess = func(i1, i2 *extractor.Package) bool {
		return i1.Name < i2.Name
	}
	sortPKGs        = cmpopts.SortSlices(pkgLess)
	allowUnexported = cmp.AllowUnexported(packagejson.Extractor{}, wheelegg.Extractor{})
)

func TestGetAll(t *testing.T) {
	pkgs := []*extractor.Package{
		{Name: "software1", PURLType: purl.TypeNPM},
		{Name: "software2", PURLType: purl.TypePyPi},
		{Name: "software3", PURLType: purl.TypePyPi},
		{Name: "software-no-purl"},
	}
	want := pkgs

	px, err := packageindex.New(pkgs)
	if err != nil {
		t.Fatalf("packageindex.New(%v): %v", pkgs, err)
	}

	got := px.GetAll()
	if diff := cmp.Diff(want, got, sortPKGs, allowUnexported); diff != "" {
		t.Errorf("packageindex.New(%v).GetAll(): unexpected package (-want +got):\n%s", pkgs, diff)
	}
}

func TestGetAllOfType(t *testing.T) {
	pkgs := []*extractor.Package{
		{Name: "software1", PURLType: purl.TypeNPM},
		{Name: "software2", PURLType: purl.TypePyPi},
		{Name: "software3", PURLType: purl.TypePyPi},
		{Name: "software-no-purl"},
	}
	want := []*extractor.Package{
		{Name: "software2", PURLType: purl.TypePyPi},
		{Name: "software3", PURLType: purl.TypePyPi},
	}

	px, err := packageindex.New(pkgs)
	if err != nil {
		t.Fatalf("packageindex.New(%v): %v", pkgs, err)
	}

	got := px.GetAllOfType("pypi")
	if diff := cmp.Diff(want, got, sortPKGs, allowUnexported); diff != "" {
		t.Errorf("packageindex.New(%v).GetAllOfType(pypi): unexpected package (-want +got):\n%s", pkgs, diff)
	}
}

func TestGetSpecific(t *testing.T) {
	pkg1 := &extractor.Package{Name: "software1", Version: "1.2.3", PURLType: purl.TypeNPM}
	pkg2 := &extractor.Package{Name: "software2", Version: "1.2.3", PURLType: purl.TypePyPi}
	pkg3 := &extractor.Package{Name: "software3", PURLType: purl.TypePyPi}
	pkg4v123 := &extractor.Package{Name: "software4", Version: "1.2.3", PURLType: purl.TypeNPM}
	pkg4v456 := &extractor.Package{Name: "software4", Version: "4.5.6", PURLType: purl.TypeNPM}
	pkgNoPURL := &extractor.Package{Name: "software-no-purl", Version: "1.2.3"}
	pkgs := []*extractor.Package{pkg1, pkg2, pkg3, pkg4v123, pkg4v456, pkgNoPURL}

	testCases := []struct {
		desc    string
		pkgType string
		pkgName string
		want    []*extractor.Package
	}{
		{
			desc:    "No version or namespace",
			pkgType: "pypi",
			pkgName: "software3",
			want:    []*extractor.Package{pkg3},
		},
		{
			desc:    "software with version",
			pkgType: "pypi",
			pkgName: "software2",
			want:    []*extractor.Package{pkg2},
		},
		{
			desc:    "software with namespace",
			pkgType: "npm",
			pkgName: "software1",
			want:    []*extractor.Package{pkg1},
		},
		{
			desc:    "multiple versions",
			pkgType: "npm",
			pkgName: "software4",
			want:    []*extractor.Package{pkg4v123, pkg4v456},
		},
		{
			desc:    "no purl type",
			pkgType: "",
			pkgName: "software-no-purl",
			want:    []*extractor.Package{pkgNoPURL},
		},
	}

	for _, tc := range testCases {
		t.Run(tc.desc, func(t *testing.T) {
			px, err := packageindex.New(pkgs)
			if err != nil {
				t.Fatalf("packageindex.New(%v): %v", pkgs, err)
			}

			got := px.GetSpecific(tc.pkgName, tc.pkgType)
			if diff := cmp.Diff(tc.want, got, sortPKGs, allowUnexported); diff != "" {
				t.Errorf("packageindex.New(%v).GetSpecific(%s, %s): unexpected package (-want +got):\n%s", pkgs, tc.pkgName, tc.pkgType, diff)
			}
		})
	}
}
