// // This file is part of MinIO DirectPV
// // Copyright (c) 2021, 2022 MinIO, Inc.
// //
// // This program is free software: you can redistribute it and/or modify
// // it under the terms of the GNU Affero General Public License as published by
// // the Free Software Foundation, either version 3 of the License, or
// // (at your option) any later version.
// //
// // This program is distributed in the hope that it will be useful,
// // but WITHOUT ANY WARRANTY; without even the implied warranty of
// // MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// // GNU Affero General Public License for more details.
// //
// // You should have received a copy of the GNU Affero General Public License
// // along with this program.  If not, see <http://www.gnu.org/licenses/>.

package installer

import (
	"bytes"
	"compress/gzip"
	"fmt"
	"io"
	"strings"
)

func bindata_read(data []byte, name string) ([]byte, error) {
	gz, err := gzip.NewReader(bytes.NewBuffer(data))
	if err != nil {
		return nil, fmt.Errorf("Read %q: %v", name, err)
	}

	var buf bytes.Buffer
	_, err = io.Copy(&buf, gz)
	gz.Close()

	if err != nil {
		return nil, fmt.Errorf("Read %q: %v", name, err)
	}

	return buf.Bytes(), nil
}

var _config_crd_direct_csi_min_io_directcsidrives_yaml = []byte("\x1f\x8b\x08\x00\x00\x00\x00\x00\x00\xff\xec\x5a\x5f\x73\x1b\xb7\x11\x7f\xd7\xa7\xd8\x61\x3b\x63\xc9\x25\x4f\x96\xdc\x49\x13\xce\x78\x3c\xae\x54\x75\x34\x89\x63\x8d\xa5\xe4\xa1\xa2\xda\x80\x77\x4b\x12\xd1\x1d\x70\x01\x70\x94\x98\x4e\xbf\x7b\x67\x81\x3b\xfe\xbd\x3b\x91\xac\x5d\xc7\xce\xe2\x89\x3c\x00\x8b\xc5\x62\xff\x01\xbf\xed\xf5\x7a\x07\x22\x97\x3f\xa2\xb1\x52\xab\x3e\x88\x5c\xe2\xa3\x43\x45\xff\x6c\x74\xff\xb5\x8d\xa4\x3e\x9e\x9e\x1c\xdc\x4b\x95\xf4\xe1\xac\xb0\x4e\x67\xef\xd1\xea\xc2\xc4\x78\x8e\x23\xa9\xa4\x93\x5a\x1d\x64\xe8\x44\x22\x9c\xe8\x1f\x00\x08\xa5\xb4\x13\xf4\xd9\xd2\x5f\x80\x58\x2b\x67\x74\x9a\xa2\xe9\x8d\x51\x45\xf7\xc5\x10\x87\x85\x4c\x13\x34\x9e\x78\xb5\xf4\xf4\x45\xf4\x4d\x74\x7a\x00\x10\x1b\xf4\xd3\x6f\x64\x86\xd6\x89\x2c\xef\x83\x2a\xd2\xf4\x00\x40\x89\x0c\xfb\x90\x48\x83\xb1\x8b\xad\x4c\x8c\x9c\xa2\x8d\xc2\xff\x28\xb6\x32\xca\xa4\x8a\xa4\x3e\xb0\x39\xc6\xb4\xf6\xd8\xe8\x22\xaf\x26\x2c\x0f\x08\xa4\x4a\xfe\xc2\xde\xce\xfd\xa0\xb3\xeb\xcb\x73\xa2\xea\x3b\x52\x69\xdd\xb7\x35\x9d\xdf\x49\xeb\xfc\x80\x3c\x2d\x8c\x48\x37\x38\xf2\x7d\x56\xaa\x71\x91\x0a\xb3\xde\x7b\x00\x60\x63\x9d\x63\x1f\xce\xd2\xc2\x3a\x34\x07\x00\xa5\x0c\x3c\x3f\xbd\x72\x97\xd3\x13\x91\xe6\x13\x71\x12\x88\xc5\x13\xcc\x44\x60\x17\x40\xe7\xa8\xde\x5c\x5d\xfe\xf8\xf2\x7a\xe5\x33\x40\x82\x36\x36\x32\x77\x5e\x9e\xab\x3c\x43\x82\x4a\x3b\xb4\xe0\x99\x80\xb3\xf7\xe7\xa0\x87\x3f\x93\x58\xe6\xb3\x73\xa3\x73\x34\x4e\x56\x72\x09\x6d\x49\x3b\x96\xbe\xae\xad\xf5\x8c\xd8\x09\xa3\x20\x21\xb5\x40\x0b\x6e\x82\xd5\xc6\x30\x29\x77\x00\x7a\x04\x6e\x22\x2d\x18\xcc\x0d\x5a\x54\x41\x51\x56\x08\x03\x0d\x12\xaa\x62\x0f\xae\xd1\x10\x19\xb0\x13\x5d\xa4\x09\x69\xd3\x14\x8d\x03\x83\xb1\x1e\x2b\xf9\xeb\x9c\xb6\x05\xa7\xfd\xa2\xa9\x70\x58\x1e\xd0\xa2\x49\xe5\xd0\x28\x91\xc2\x54\xa4\x05\x76\x41\xa8\x04\x32\x31\x03\x83\xb4\x0a\x14\x6a\x89\x9e\x1f\x62\x23\x78\xab\x0d\x82\x54\x23\xdd\x87\x89\x73\xb9\xed\x1f\x1f\x8f\xa5\xab\xac\x22\xd6\x59\x56\x28\xe9\x66\xc7\x5e\xc1\xe5\xb0\x70\xda\xd8\xe3\x04\xa7\x98\x1e\x5b\x39\xee\x09\x13\x4f\xa4\xc3\xd8\x15\x06\x8f\x45\x2e\x7b\x9e\x75\xe5\x2d\x23\xca\x92\x3f\x98\xd2\x8e\xec\xb3\x15\x5e\xdd\x8c\x94\xc3\x3a\x23\xd5\x78\xa9\xc3\x6b\x69\xcb\x09\x90\xa2\x82\xb4\x20\xca\xa9\x61\x17\x0b\x41\xd3\x27\x92\xce\xfb\xbf\x5d\xdf\x40\xb5\xb4\x3f\x8c\x75\xe9\x7b\xb9\x2f\x26\xda\xc5\x11\x90\xc0\xa4\x1a\xa1\x09\x87\x38\x32\x3a\xf3\x34\x51\x25\xb9\x96\xca\xf9\x3f\x71\x2a\x51\xad\x8b\xdf\x16\xc3\x4c\x3a\x3a\xf7\x5f\x0a\xb4\x8e\xce\x2a\x82\x33\xef\x2a\x60\x88\x50\xe4\x89\x70\x98\x44\x70\xa9\xe0\x4c\x64\x98\x9e\x09\x8b\x1f\xfd\x00\x48\xd2\xb6\x47\x82\xdd\xee\x08\x96\xbd\xdc\xfa\xe0\x20\xb5\xa5\x8e\xca\x07\x35\x9c\xd7\xaa\x75\x5e\xe7\x18\xaf\x59\x28\xcd\x97\x23\x19\x7b\x03\x89\x56\x08\xd5\x1b\xaa\x5f\xa2\xa2\xfa\xee\x41\x61\xb2\xde\xbb\xc6\x02\x9d\x85\x34\x98\x6c\x8c\x0a\x3b\x1a\x6a\x9d\xa2\x58\xb7\x4d\xcf\xdc\x8d\x90\xca\x6d\x52\x17\x49\xe2\xc3\x81\x48\xaf\x1a\x39\x6c\x11\x6f\xab\x38\xa9\x95\xca\x83\xc9\x85\x36\x99\xa8\x61\x60\x65\x7b\xef\x57\x47\xaf\x89\x77\x14\x3e\x96\x24\xbd\x92\xd1\x87\x0d\x59\xb7\xcb\x9b\xda\x48\xa6\x68\x67\xd6\x61\x56\xd7\xfb\xc4\x6e\x81\x18\x89\xb1\x6d\x66\xfd\x39\x50\xcb\x74\xa1\xdc\xbb\x7c\x29\xd4\xae\x37\xe9\x30\x6b\xe8\x7a\x92\xb1\x6a\x80\x30\x46\xcc\x6a\xfb\x1f\x7b\x14\xcb\x8d\x42\x87\xb6\x47\xc1\xb2\x57\xce\x70\x3a\x93\x71\x13\xc3\xde\x53\xec\x25\xaa\xbc\x30\xe3\xbd\x44\xd5\xa8\x53\x95\x09\xac\x12\xed\xad\xd9\xd1\x56\xe6\xee\x84\x2b\xec\xf6\x06\xef\x87\xaf\xe9\x64\xa3\x12\x36\x2b\xa0\x48\x53\x1d\x93\xeb\x3c\x13\xb9\x88\xa5\x9b\x6d\x8a\x27\xd0\xec\x53\x04\xfc\xea\xcf\x0d\xa2\xa1\xe8\x38\xf6\xa9\xc8\x72\x8b\xb5\x0a\x06\x5d\xa3\x42\x8d\x9a\xb5\xb2\xe9\xce\x59\x45\xc2\x67\x81\x42\x2a\xda\xb3\x13\x32\xb5\xc4\x17\x68\x85\x20\xc8\xd3\xb9\x90\x19\x20\xc4\x85\x31\x9b\xe1\x63\x21\x63\x9c\xa7\x10\x6f\xae\x2e\xa1\x4a\x45\x23\xe8\xf5\x7a\x70\x43\x9f\xad\x33\x45\xec\x28\x12\xd2\xa6\x54\x82\x89\x5f\x29\x9c\x68\x2d\xd9\xc2\x12\x13\x94\x72\x78\x55\x07\x11\xe2\xd8\x48\x62\x9a\x40\x2e\xdc\x04\xa2\x70\xba\xd1\x42\x20\x11\xc0\x85\x36\x80\x8f\x22\xcb\x53\xec\x36\xea\x24\x5c\x68\x5d\x9e\x75\x60\xec\xdf\x70\x7c\x0c\xef\xe7\xb1\xd5\xaf\xa4\x87\x16\xcd\x34\xa4\xcc\x3e\xf9\x81\x91\xd6\xcf\xd6\xe3\x72\x79\x26\x41\x3e\x41\x16\x11\x11\xfb\x56\xe9\x07\x55\xc7\xa2\x5f\x5f\x18\xec\xc3\xa0\xf3\x66\x2a\x64\x2a\x86\x29\x0e\x3a\xf5\xcc\x0e\x3a\x57\x46\x8f\x0d\x5a\xca\x5b\x07\x9d\x90\x20\x0d\x3a\xe7\x38\x36\x22\xc1\x64\xd0\xa1\xa5\xfe\x94\x0b\x17\x4f\xde\xa2\x19\xe3\xb7\x38\x7b\xe5\x17\x98\x7f\xbe\x76\x46\x38\x1c\xcf\x5e\x65\xd4\x5f\xbb\x08\x8d\x25\x3f\x71\x33\xcb\xf1\x55\x26\xf2\xf9\x87\xb7\x22\x9f\x13\x9c\xab\x8c\x85\xdb\x3b\x0a\xb9\xd3\x93\x68\xfe\xad\x96\xec\x4f\x3f\x5b\xad\xfa\x83\xce\x62\xef\x5d\x9d\x91\x82\xe6\x6e\x36\xe8\xc0\x0a\x77\xfd\x41\xc7\xf3\x57\x7d\xaf\x36\xd3\x1f\x74\x68\xf5\x41\xa7\x76\x85\xdc\x68\xa7\x87\xc5\xa8\x3f\xe8\x0c\x67\x0e\x6d\xf7\xa4\x6b\x30\xef\x52\xa6\xfe\x6a\xb1\xea\xa0\xf3\x13\x0c\x14\x6d\x4a\xbb\x09\x9a\xa0\x41\x16\xfe\x53\x47\xb3\x3d\xa6\x00\xa4\xc2\xba\x1b\x23\x94\x95\xd5\x55\xa8\xc9\x8d\xaf\x18\xdc\xe6\x34\xb2\x84\x90\x15\x5b\x07\x8e\x3e\x78\x33\x6b\x15\x28\x35\x37\xa7\x42\x16\x44\x99\x1e\x19\x6b\xd0\x32\xca\xb4\x85\xf2\x9b\x8c\x4a\xab\x0b\xc9\xf9\x10\xe1\x61\x82\x2d\x44\x27\x08\x85\x4a\xd0\xa4\x33\xca\x47\xe3\x85\x77\x98\x08\x35\xa6\x04\x10\x2e\xc9\xbc\x85\x37\x60\x4a\x0e\xef\x49\xbb\xbb\x34\xb1\x99\x6a\x61\xab\xe4\xd6\xef\x8f\x38\xf0\xff\xc8\x43\x04\x2b\x2e\xc9\xfb\xfc\x38\x8e\x31\x77\x64\x0a\x9b\x81\x3e\xb4\xca\x61\x52\x4a\xda\x23\x8a\xfb\xc6\xcf\x0c\xad\x15\x4d\x11\x6b\xed\xe0\xca\xb1\x21\x83\x9f\x14\x99\x50\x60\x50\x24\xc4\xe7\xa2\x4f\x25\x3e\x1f\x6c\x58\x2e\xd0\x0c\xce\x55\x0c\x75\x11\xdc\xd8\xe2\x1c\xcb\xa3\xa2\x24\x7e\x88\xe4\xee\xbc\x81\x94\x1b\x68\x12\x46\x26\x1e\xbf\x43\x35\x76\x93\x3e\xbc\x3c\xfd\xcb\x57\x5f\xef\x2b\x8b\xe0\xe3\x30\xf9\x3b\x2a\x34\xde\xd5\x6d\x25\x96\xcd\x69\x4b\x17\x13\xbf\xbf\xa8\xca\xca\xa3\xf1\x7c\x4c\x8b\xfe\x95\xce\x7d\xa1\x79\x0f\xc2\x82\x45\x07\x43\x61\x31\x81\x22\x27\x39\x91\x6b\x97\xca\x3a\xa1\x62\xec\x82\x1c\xed\xb6\x88\xb4\x95\x97\x4e\x67\x70\x72\xda\x85\x61\x79\x14\x9b\x3e\xfa\xf6\xf1\x2e\xda\xdc\x62\x1b\xe5\x6f\xba\x6b\xfc\x4b\x0b\x74\xd4\x7a\xe4\xf5\x15\x1e\xa4\x9b\xd0\xf5\xce\xc7\xd4\xf2\x42\xdc\x16\x53\x61\x35\xae\xe2\x7c\xdf\x4f\x59\x47\x7d\x3a\x11\x5a\x26\x95\xcc\x8a\xac\x0f\x2f\x5a\xd5\xa5\x3e\xeb\x08\xcd\xa0\xb0\x5b\xea\x48\x18\xba\x48\x30\x04\x39\xd7\xb1\x11\x19\xa5\x52\x31\xc8\x84\xae\x7c\x23\x89\x66\x1b\x03\x22\x11\x94\x04\x29\x6d\x58\x91\xf5\x33\x5b\x7a\xd1\x25\x93\xba\x32\x3a\x29\x62\x34\xf5\xd1\x1a\xc2\x6b\x46\x75\x97\x5b\x3a\x36\x7f\xf7\xf4\xb6\x18\xde\x4b\x00\x1f\xe9\xc8\xe6\xaf\x0f\x14\x7f\x1b\x49\x66\x28\x94\x54\x63\x5b\xb2\x48\x57\x71\x72\x73\x21\x68\x3f\x4c\xd0\x47\x1f\xff\xfe\x52\xd2\x32\x7e\x17\x56\x26\x58\x77\xdf\xab\x9a\x80\x71\x21\x8c\x50\x0e\x31\x21\xe7\x49\x0e\xa3\xa4\xb1\xe4\xe0\xc5\xe2\x86\xfe\x84\xef\x80\xe0\x70\x82\x0b\xa6\xad\x96\xb7\x7d\xef\x77\xb6\x70\x38\x27\x2f\x4e\x5b\x34\x6c\x3e\xaa\x61\x48\x2e\x9c\x43\xa3\xfa\xf0\xcf\xdb\x37\xbd\x7f\x88\xde\xaf\x77\x87\xe5\x8f\x17\xbd\x6f\xfe\xd5\xed\xdf\x3d\x5f\xfa\x7b\x77\xf4\xfa\x8f\xfb\xba\xb6\xba\xd4\x7f\xd1\x56\x54\xb5\x0c\x9f\x55\xae\x5b\x69\x43\xd7\xc7\x56\x3d\x82\x1b\x53\x60\x17\x2e\x44\x6a\xb1\x0b\x3f\x28\x1f\xfc\x9a\x04\x85\xaa\x68\xb8\x71\xd2\x0d\xa6\x43\xa4\xea\x93\x19\xdf\xed\xd7\x68\xee\x2f\xd7\xfe\x9f\x6e\x8e\xdb\x08\xc4\x67\x7c\x7a\xb4\xec\xcf\x96\x5e\x80\xc0\xfb\x61\xca\x86\xa3\x32\xd3\x8e\x62\x9d\x1d\x2f\x5e\x88\x1a\x15\x8f\xae\x03\x6f\x85\x9a\xc1\xc2\xd9\x86\x7c\x78\xdd\x22\xac\xa3\x6c\x5a\xc4\x46\x5b\x3b\x7f\x16\x6b\x36\xe6\x54\xde\x23\xcc\x93\xe9\xe0\xda\x87\x18\x0b\x7f\x87\x30\x43\xe9\x8c\x30\xb3\xa5\x8b\x13\xc4\x42\xf9\x07\x2e\x8b\xa3\x22\x6d\x24\x7b\x68\x11\x21\x52\x3a\xc1\xcd\x18\x71\x14\x3c\xbe\x18\xca\x54\xba\x19\xf9\xf4\x04\x63\xad\x46\xa9\xf4\xd7\x9c\xe6\x60\x91\xe5\xda\x38\xa1\x5c\x30\x63\x83\x63\x7c\x04\xe9\x20\xa3\xb4\x17\x2d\x05\x8e\xc3\x44\xd9\x93\x93\xd3\x97\xd7\xc5\x30\xd1\x99\x90\xea\x22\x73\xc7\x47\xaf\x0f\x7f\x29\x44\x4a\x1e\x33\xf9\x5e\x64\x78\x91\xb9\xa3\x2d\x92\x83\x93\xaf\x9e\xb4\xc3\xc3\xdb\x60\x6d\x77\x87\xb7\xbd\xf2\xd7\xf3\xea\xd3\xd1\xeb\xc3\x41\xd4\xda\x7f\xf4\x9c\x58\x5b\xb2\xe1\xbb\xdb\xde\xc2\x80\xa3\xbb\xe7\x47\xaf\x97\xfa\x8e\xf6\x34\xe7\xfa\x17\x81\xd0\x7a\x35\xe9\x75\xed\xb0\x32\x61\xab\xed\x0b\xc1\xa5\xb6\x2b\x1c\x7d\x6d\x17\x71\xdd\xf8\x92\x56\xfb\x58\xd6\xfe\x7c\xb3\xf9\x74\x93\x89\xbc\x77\x8f\xb3\x1a\x3f\xd6\xb0\x7a\xd3\xeb\x4f\x26\xf2\xba\x37\xc3\xeb\x06\x2f\xb9\xfa\x48\xd2\xf8\x36\x52\x9a\x45\xc3\x26\x6b\x8f\xb3\xed\x61\xae\x6d\x9a\x41\xfc\x18\xaf\x29\xa9\x1e\xcb\x58\xa4\x7f\x4d\x75\x7c\x7f\x2d\x7f\xad\xf1\x8f\xfb\xd3\xce\x74\x82\xe9\xf7\x45\x36\x44\xb3\xd3\x5e\xdb\x5f\x10\x1b\xdf\x78\xb6\x78\xc0\xdd\x56\xed\x5a\x5e\x0c\xdb\x5e\x0b\x5b\x38\x20\x2f\x4a\x7e\x6b\xa7\x49\xb9\x30\xce\xdb\xf4\xf7\x75\x41\xb5\x4d\xf4\xb9\x70\x93\xdd\x96\x9a\xcc\xec\x47\x53\x04\xa3\xb5\xbb\xaa\xf6\xb2\x13\x5b\x16\x8d\x14\xfb\xe8\x90\xd3\xb9\x4e\xf5\xb8\xc6\x56\x3e\x36\x1e\xe0\xb4\x13\xe9\x87\x37\xd5\xa6\x47\x61\x3a\xe9\xa7\x9f\x82\x37\x67\xf7\xe6\xc0\xd1\xd2\x27\xba\x12\x1c\x34\x12\x0a\x37\xc2\x3e\x38\x53\x04\xc7\x6b\x9d\x36\x62\x8c\x7d\x18\x51\xde\xb6\x02\x13\x0f\xd1\x31\x4a\xcc\x28\x71\x68\x8c\x12\x33\x4a\x5c\x36\x46\x89\x19\x25\x06\x46\x89\x37\x67\xfe\x0e\x51\xe2\x38\x46\x6b\x6f\x64\x5d\x66\xb7\xb2\xfc\x9b\xf9\xc0\xf9\xa2\x61\x2e\x38\x89\x66\xa7\xdb\x17\x23\xd3\x8c\x4c\x33\x32\xcd\xc8\x34\x23\xd3\x8c\x4c\x03\x23\xd3\x8c\x4c\x33\x32\xcd\xc8\x34\x23\xd3\xc0\xc8\x34\x23\xd3\x8c\x4c\x33\x32\xbd\xd9\x18\x99\x06\x46\xa6\x19\x99\xae\xe6\x31\x32\xed\x1b\x23\xd3\x8c\x4c\x57\xed\x33\x44\xa6\x4f\xc3\x20\x46\xa6\x19\x99\x66\x64\x9a\x91\x69\xdf\x18\x99\x66\x64\x1a\x18\x99\xde\x9c\xc9\xc8\x74\xe3\xf2\x8c\x4c\x33\x32\xcd\xc8\x34\x23\xd3\x8c\x4c\x33\x32\xcd\xc8\x74\xab\x58\x18\x99\x5e\xa3\xcc\xc8\x34\x30\x32\xcd\xc8\x34\x23\xd3\x8c\x4c\x33\x32\xcd\xc8\x34\x23\xd3\x8c\x4c\x7f\x68\x64\x7a\x3e\xed\x87\x1f\x2e\xcf\xbf\x7c\x50\x5b\xfc\xac\x4d\x13\x20\xb9\x44\xf6\xe5\xe9\x6e\x64\xa5\xfa\x28\x64\x19\x82\x9f\xb7\xff\x3b\x04\x5f\xce\xdc\xd9\x2c\x18\xbc\x67\xf0\xfe\x93\x83\xf7\x2f\xc3\x20\x06\xef\x19\xbc\x67\xf0\x9e\xc1\x7b\xdf\x18\xbc\x67\xf0\x1e\x18\xbc\xdf\x9c\xc9\xe0\x7d\xe3\xf2\x0c\xde\x33\x78\xcf\xe0\x3d\x83\xf7\x0c\xde\x33\x78\xcf\xe0\x7d\xab\x58\x18\xbc\x5f\xa3\xcc\xe0\x3d\x30\x78\xcf\xe0\x3d\x83\xf7\x0c\xde\x33\x78\xcf\xe0\x3d\x83\xf7\x9f\x03\x78\x9f\xed\x8c\x31\x26\xbb\x23\xe7\x5c\x22\x10\x7a\x7f\x57\x25\x02\xc2\xba\x5d\x61\xfc\x64\x67\x81\x73\x21\xc2\x6f\xb7\x10\xe1\x86\xe2\xf0\x4d\x6d\xba\xb1\xcd\xcc\x3d\x0a\x11\x3e\x45\xf1\x43\x39\xb3\x2e\x2c\xb5\xbd\xb6\xff\xc6\xaa\x26\x50\x24\xef\x54\x5a\xe3\xb3\xda\xf6\xf0\x09\x6a\x2d\xec\x83\xc8\xdf\x35\xae\x55\xcf\xe6\x17\x57\x9f\x01\x50\xe0\x14\x95\xbb\xb8\xde\x59\x5f\xc3\xc4\x6b\x2f\xfe\x9d\x26\x4e\x51\x25\x7a\xb7\xb3\x9a\x4a\xe3\x8a\xe6\x65\xea\x0f\xeb\xe1\x41\x36\x5a\x52\xcd\x2a\x9f\x61\xa9\x4a\x38\x69\x2e\x55\xe1\x52\x15\x2e\x55\xe1\x52\x95\xd0\xb8\x54\x85\x4b\x55\x80\x4b\x55\x36\x67\x72\xa9\x4a\xe3\xf2\x5c\xaa\xc2\xa5\x2a\x5c\xaa\xc2\xa5\x2a\x5c\xaa\xc2\xa5\x2a\x5c\xaa\xd2\x2a\x16\x2e\x55\x59\xa3\xcc\xa5\x2a\xc0\xa5\x2a\x5c\xaa\xc2\xa5\x2a\x5c\xaa\xc2\xa5\x2a\x5c\xaa\xc2\xa5\x2a\x5c\xaa\x52\x4d\xe1\x52\x15\xe0\x52\x95\xe5\x39\x5c\xaa\xb2\x25\xc5\xcf\xb4\x54\xc5\x3f\x40\xbc\xa5\x05\xed\xa5\x1a\xe9\x7d\x1f\x29\xdf\xad\x92\x59\x5c\x24\xfc\x5e\xfc\x4b\xf0\x1e\x0f\x38\x4f\x03\x02\x4f\x40\x02\x5b\xa4\x59\x4f\xc3\x02\x7b\x00\x03\x4f\x43\x03\xdb\x62\x57\x5b\xed\xe2\x03\x86\xb0\x96\x8d\x71\x55\xd3\x07\xaf\x6a\x8a\xe5\x15\x57\x42\xed\x55\x09\xb5\x34\xf1\x3b\xad\xc6\x5c\x46\xc5\x65\x54\x8b\xf6\x25\x95\x51\xf9\x2f\xff\x0d\x00\x00\xff\xff\xea\xd3\x24\x38\xff\xa5\x00\x00")

func config_crd_direct_csi_min_io_directcsidrives_yaml() ([]byte, error) {
	return bindata_read(
		_config_crd_direct_csi_min_io_directcsidrives_yaml,
		"config/crd/direct.csi.min.io_directcsidrives.yaml",
	)
}

var _config_crd_direct_csi_min_io_directcsivolumes_yaml = []byte("\x1f\x8b\x08\x00\x00\x00\x00\x00\x00\xff\xec\x98\x5f\x6f\x1b\xb9\x11\xc0\xdf\xfd\x29\x06\x6a\x81\xd8\xa9\x76\x1d\x3b\x87\xeb\x45\x40\x10\x04\x4e\x53\x04\xb9\x14\x41\xec\xe6\xa1\x96\xdb\x1b\xed\x8e\x56\x3c\xef\x92\x1b\x92\xab\x58\x57\xf4\xbb\x17\x43\xae\xa4\x95\xb4\x52\x94\xa0\xd7\xbc\x0c\x9f\x24\x92\x3b\x24\x87\xf3\x8f\xbf\x24\x49\x4e\xb0\x56\x1f\xc9\x3a\x65\xf4\x08\xb0\x56\xf4\xe0\x49\xf3\x3f\x97\xde\xff\xe4\x52\x65\xce\xe7\x17\x27\xf7\x4a\xe7\x23\xb8\x6a\x9c\x37\xd5\x07\x72\xa6\xb1\x19\xbd\xa2\xa9\xd2\xca\x2b\xa3\x4f\x2a\xf2\x98\xa3\xc7\xd1\x09\x00\x6a\x6d\x3c\x72\xb7\xe3\xbf\x00\x99\xd1\xde\x9a\xb2\x24\x9b\x14\xa4\xd3\xfb\x66\x42\x93\x46\x95\x39\xd9\x20\x7c\xb9\xf4\xfc\x49\xfa\x2c\xbd\x3c\x01\xc8\x2c\x85\xcf\x6f\x54\x45\xce\x63\x55\x8f\x40\x37\x65\x79\x02\xa0\xb1\xa2\x11\xe4\xca\x52\xe6\x33\xa7\xe6\xa6\x6c\x2a\x72\x69\xec\x48\x33\xa7\xd2\x4a\xe9\x54\x99\x13\x57\x53\xc6\x8b\x17\xd6\x34\xf5\xf2\x8b\xee\x84\x28\xab\xdd\x60\x3c\xdc\xab\x30\xe9\xea\xfa\xcd\xc7\x20\x36\x8c\x94\xca\xf9\xb7\x7d\xa3\x3f\x2b\xe7\xc3\x8c\xba\x6c\x2c\x96\xbb\x9b\x0a\x83\x4e\xe9\xa2\x29\xd1\xee\x0c\x9f\x00\xb8\xcc\xd4\x34\x82\xab\xb2\x71\x9e\xec\x09\x40\xab\x88\xb0\xa7\xa4\x3d\xea\xfc\x02\xcb\x7a\x86\x17\x51\x5a\x36\xa3\x0a\xe3\x96\x01\x4c\x4d\xfa\xe5\xfb\x37\x1f\x9f\x5e\x6f\x74\x03\xe4\xe4\x32\xab\x6a\x1f\x94\xba\xb5\x6d\xc8\x49\x1b\x4f\x0e\xe2\x36\xe0\xea\xc3\x2b\x30\x93\x5f\x59\x39\xab\xef\x6b\x6b\x6a\xb2\x5e\x2d\xb5\x13\x5b\xc7\x48\x3a\xbd\x5b\xab\x3d\xe2\x0d\xc5\x59\x90\xb3\x75\x90\x03\x3f\xa3\xe5\xd1\x28\x6f\xcf\x00\x66\x0a\x7e\xa6\x1c\x58\xaa\x2d\x39\xd2\xd1\x5e\x36\x04\x03\x4f\x42\xbd\xdc\x1e\x5c\x93\x65\x31\xe0\x66\xa6\x29\x73\x36\xaa\x39\x59\x0f\x96\x32\x53\x68\xf5\xdb\x4a\xb6\x03\x6f\xc2\xa2\x25\x7a\x6a\x2f\x69\xdd\x94\xf6\x64\x35\x96\x30\xc7\xb2\xa1\x21\xa0\xce\xa1\xc2\x05\x58\xe2\x55\xa0\xd1\x1d\x79\x61\x8a\x4b\xe1\x9d\xb1\x04\x4a\x4f\xcd\x08\x66\xde\xd7\x6e\x74\x7e\x5e\x28\xbf\x74\x8e\xcc\x54\x55\xa3\x95\x5f\x9c\x07\x3b\x57\x93\xc6\x1b\xeb\xce\x73\x9a\x53\x79\xee\x54\x91\xa0\xcd\x66\xca\x53\xe6\x1b\x4b\xe7\x58\xab\x24\x6c\x5d\x07\x07\x49\xab\xfc\x0f\xb6\x75\x27\xf7\x68\x63\xaf\x7e\xc1\xe6\xe1\xbc\x55\xba\xe8\x0c\x04\x5b\x3d\x70\x03\x6c\xad\xa0\x1c\x60\xfb\x69\x3c\xc5\x5a\xd1\xdc\xc5\xda\xf9\xf0\x97\xeb\x1b\x58\x2e\x1d\x2e\x63\x5b\xfb\x41\xef\xeb\x0f\xdd\xfa\x0a\x58\x61\x4a\x4f\xc9\xc6\x4b\x9c\x5a\x53\x05\x99\xa4\xf3\xda\x28\xed\xc3\x9f\xac\x54\xa4\xb7\xd5\xef\x9a\x49\xa5\x3c\xdf\xfb\xa7\x86\x9c\xe7\xbb\x4a\xe1\x2a\x44\x0c\x98\x10\x34\x75\x8e\x9e\xf2\x14\xde\x68\xb8\xc2\x8a\xca\x2b\x74\xf4\xbb\x5f\x00\x6b\xda\x25\xac\xd8\xe3\xae\xa0\x1b\xec\xb6\x27\x47\xad\x75\x06\x9c\x47\xdf\xb8\x03\x37\xb6\xe5\xa1\xd7\x61\xfe\xb6\x9f\xf2\xe1\x6d\x15\x9c\x24\xdd\x10\xd5\xef\xac\xdc\x70\x8e\xaa\xc4\x49\x49\x57\x58\x63\xa6\xfc\x62\x7b\x02\x40\x94\x39\x62\xa7\xf8\xf1\x87\x9d\xd1\x78\x20\x76\x98\x22\xc4\xa7\x6e\xcb\x8c\xce\x55\x27\xc4\x77\x9b\xf2\x54\xf5\x74\x6f\x1d\x7b\x70\xb5\x14\x11\xf2\x03\x2a\xcd\x87\xf6\xa8\x4a\xc7\xfb\x02\xa3\x09\x90\xc3\xb8\x8f\xc1\x82\x20\x6b\xac\xdd\xb5\xa8\xb5\x96\x69\x15\x55\x5e\xbe\x7f\x03\xcb\x24\x95\x42\x92\x24\x70\xc3\xdd\xce\xdb\x26\xf3\xec\x1c\x7c\x28\x9d\x53\x1e\x56\x8a\xa1\xb9\x57\x6c\xe3\x78\x13\x1c\x85\xd0\x5a\x5c\x00\x46\xd3\x9e\x2a\x2a\x73\xa8\xd1\xcf\x20\x8d\xf7\x9b\xae\x15\x92\x02\xbc\x36\x16\xe8\x01\xab\xba\xa4\x61\xaf\x5c\x56\x2d\xbc\x36\xa6\xbd\xec\xb8\xb1\x7f\xc3\xf9\x39\x7c\x58\xb9\x5b\x58\xc9\x4c\x1c\xd9\x79\x4c\xa6\x21\x1e\xc2\xd4\x98\x47\xdb\xae\xda\xde\x49\xd4\x4f\xd4\x45\xca\xc2\xde\x6a\xf3\x59\xf7\x6d\x31\xac\x8f\x96\x46\x30\x1e\xbc\x5c\xda\xc9\x78\xd0\xbf\xd9\xf1\xe0\xbd\x35\x85\x25\xc7\xd9\x6c\x3c\x88\x31\x73\x3c\x78\x45\x85\xc5\x9c\xf2\xf1\x80\x97\xfa\x53\x8d\x3e\x9b\xbd\x23\x5b\xd0\x5b\x5a\x3c\x0f\x0b\xac\xba\xaf\xbd\x45\x4f\xc5\xe2\x79\xc5\xe3\xbd\x8b\xf0\x5c\x4e\xb6\x37\x8b\x9a\x9e\x57\x58\xaf\x3a\xde\x61\xbd\x12\xb8\x32\x19\x07\xb7\x77\xec\x85\xf3\x8b\x74\xd5\xd7\x2b\xf6\x97\x5f\x9d\xd1\xa3\xf1\x60\x7d\xf6\xa1\xa9\xd8\x40\x6b\xbf\x18\x0f\x60\x63\x77\xa3\xf1\x20\xec\x6f\xd9\xbf\x3c\xcc\x68\x3c\xe0\xd5\xc7\x83\xde\x15\x6a\x6b\xbc\x99\x34\xd3\xd1\x78\x30\x59\x78\x72\xc3\x8b\xa1\xa5\x7a\xc8\xe9\xfb\xf9\x7a\xd5\xf1\xe0\x17\x18\x6b\x3e\x94\xf1\x33\xb2\xd1\x82\x1c\xfc\xa7\x4f\xe6\x7e\x97\x8e\xad\x44\xe7\x6f\x2c\x6a\xa7\x96\x45\x52\xff\xbc\x2d\x87\xdb\xfd\x8c\x3d\x21\x26\x4a\xe7\xc1\x73\x47\x70\xb3\x83\x0a\xe5\xe6\x57\x52\xd8\x83\x38\xf8\xb3\xb3\x46\x2b\xe3\xe4\x8b\x3a\x1c\x32\x6d\xbd\x2e\xe6\xeb\x09\xc1\xe7\x19\x1d\x10\x3a\x23\x68\x74\x4e\xb6\x5c\x70\x8a\xca\xd6\xd1\x61\x86\xba\xe0\x9c\x00\x6f\xd8\xbd\x31\x38\x30\xe7\x8b\x7b\xb6\xee\x21\x7f\xb8\x5f\x6a\xe3\x96\xf9\x2e\x9c\x8f\x77\x10\xfe\x71\x84\x88\x5e\xdc\x8a\x0f\x29\x33\xcb\xa8\xf6\xec\x0a\xe9\x1e\x81\xcb\x80\xc9\x59\x2a\x61\x89\x7b\xe6\xed\x49\x1c\xeb\x56\x91\x73\x58\x1c\x77\x71\xed\xdc\x98\xd4\x67\x4d\x85\x1a\x2c\x61\xce\xfb\x5c\x8f\xe9\x5c\x65\xe8\xf7\x2d\x17\x65\xc6\xe0\x8a\x13\xd3\xc4\x30\xb6\xbe\xc7\xf6\xaa\x38\xaf\x4f\x88\xc3\x5d\x70\x90\xf6\x00\xfb\x94\x51\xe1\xc3\xcf\xa4\x0b\x3f\x1b\xc1\xd3\xcb\x3f\xff\xf8\xd3\xb7\xea\x22\xc6\x38\xca\xff\x4a\x9a\x6c\x08\x75\x47\xa9\x65\xf7\xb3\x4e\xad\x12\xce\x97\x2e\x13\x75\x5a\xac\xe6\x1c\xb0\xbf\x36\xb8\xaf\x2d\xef\x33\x3a\x70\xe4\x61\x82\x8e\x72\x68\x6a\xd6\x13\x87\x76\xa5\x9d\x47\x9d\xd1\x10\xd4\xf4\xeb\x16\x51\x6e\x19\xa5\xcb\x05\x5c\x5c\x0e\x61\xd2\x5e\xc5\x6e\x8c\xbe\x7d\xb8\x4b\x77\x8f\x78\x48\xf2\xb3\xe1\xd6\xfe\x95\x03\xbe\x6a\x33\x0d\xf6\x0a\x9f\x95\x9f\x71\xc5\x17\x72\x6a\x5b\x23\x1f\xca\xa9\xb0\x99\x57\x69\x75\xee\x2f\x79\x47\x7f\x39\x11\x5b\xa5\xb4\xaa\x9a\x6a\x04\x4f\x0e\x9a\x4b\x7f\xd5\x11\x9b\x25\x74\x47\xda\x48\x9c\xba\x2e\x30\x90\x83\x6b\x61\xb1\xe2\x52\x2a\x03\x95\x73\x15\x38\x55\x64\x8f\x71\x20\x56\x41\x2b\x90\xcb\x86\x0d\x5d\x3f\x72\x6d\x14\xed\xb8\xd4\x7b\x6b\xf2\x26\x23\xdb\x9f\xad\x21\x3e\x70\xf8\x36\xd4\x54\x65\x9d\x6b\x0b\xe5\x68\xf0\xc5\xf8\x84\x02\x7a\xe0\x2b\x5b\x3d\x48\x38\xff\xee\x15\x59\x11\x6a\xa5\x0b\xd7\x6e\x91\xab\x73\x0e\x73\x31\x69\x7f\x9e\x51\xc8\x3e\xe1\x49\xd6\xca\xb2\xe1\x14\x4e\xe5\x64\x69\xbf\x58\x84\xa2\x41\x8b\xda\x13\xe5\x1c\x3c\x39\x60\xb4\x32\x3a\x01\x1e\xd7\x45\xfb\x17\x62\x07\xc4\x80\x13\x43\x30\x1f\xb5\x7d\x00\x84\xb8\x73\x44\xc0\xb9\x78\x72\x79\xc0\xc2\x56\xb3\xf6\x4c\xa9\xd1\xf3\x2b\x70\x04\xff\xbc\x7d\x99\xfc\x03\x93\xdf\xee\x4e\xdb\x1f\x4f\x92\x67\xff\x1a\x8e\xee\x1e\x77\xfe\xde\x9d\xbd\xf8\xe3\xb7\x86\xb6\xbe\xe2\x7f\xdd\x36\x4c\xb5\x4d\x9f\xcb\x5a\x77\x69\x0d\xc3\x90\x5b\xcd\x14\x6e\x2c\x3f\x57\x5f\x63\xe9\x68\x08\x7f\xd7\x21\xf9\xed\x53\x14\xe9\xa6\xda\xb7\x68\x02\x03\x16\xd5\x5f\xcc\x84\xe1\xb0\xc6\xfe\xf1\x76\xed\x6f\x55\x49\x98\x70\x8c\x42\x42\xc5\x67\xa6\xdd\x78\xd6\x79\x14\x42\x88\xc3\x5c\x0d\xa7\x6d\xa5\x9d\x66\xa6\x3a\x5f\x3f\x1a\xf7\x1a\x1e\x3f\x07\xde\xa1\x5e\xc0\x3a\xd8\xc6\x7a\x78\xdb\x23\x9c\xe7\x6a\x1a\x33\x6b\x9c\x5b\xbd\x94\xf7\x3b\x73\xa9\xee\x09\x56\xc5\x74\x0c\xed\x13\xca\x30\xbc\x21\xec\x44\x79\x8b\x76\xd1\x79\x38\x41\x86\x3a\xbc\x79\x1d\x4d\x9b\x72\xaf\xd8\x53\x47\x04\xa9\x36\x39\xed\xe6\x88\xb3\x18\xf1\x71\xa2\x4a\xe5\x17\x1c\xd3\x73\xca\x8c\x9e\x96\x2a\x3c\x73\xf6\x27\x8b\xaa\x36\xd6\xa3\xf6\xd1\x8d\x2d\x15\xf4\x00\xca\x43\xc5\x65\x2f\x39\x4e\x1c\xa7\xb9\x76\x17\x17\x97\x4f\xaf\x9b\x49\x6e\x2a\x54\xfa\x75\xe5\xcf\xcf\x5e\x9c\x7e\x6a\xb0\xe4\x88\x99\xff\x0d\x2b\x7a\x5d\xf9\xb3\x23\x8a\x83\x8b\x1f\xbf\xe8\x87\xa7\xb7\xd1\xdb\xee\x4e\x6f\x93\xf6\xd7\xe3\x65\xd7\xd9\x8b\xd3\x71\x7a\x70\xfc\xec\x31\x6f\xad\xe3\xc3\x77\xb7\xc9\xda\x81\xd3\xbb\xc7\x67\x2f\x3a\x63\x67\xdf\xe8\xce\x96\x3e\x35\xca\x52\xde\x67\xbd\x49\x4f\x79\xdd\x3b\xad\x2d\xd8\x7a\xc7\x62\x72\xe9\x1d\x8a\x57\xdf\x3b\xc4\xbb\xee\x19\xd8\x83\x23\xba\x83\xe1\x4d\xbb\x33\xf6\x90\xdc\x37\x13\xb2\x9a\x3c\xb9\x84\x5f\x60\x49\x85\x75\x72\x4f\x8b\x9e\x38\xb6\x67\xf5\x5d\x11\x71\xc1\x0a\xeb\x5d\x8e\xc0\x99\x99\xec\x7b\xf4\xb3\x5d\xf9\x07\x6e\x24\xb7\x6a\xde\x13\x48\x0e\x7c\x31\x33\xce\x7f\xf5\x32\xec\x78\x6c\xea\x5f\xf5\x91\xf3\x58\x28\x5d\x7c\xf5\x62\xde\x78\x2c\x7f\x0f\x5c\xd3\x38\xca\xff\xf7\x72\x7b\x4d\x6c\xd7\x4b\x92\x15\x30\x3b\xd9\xfb\x65\xac\x73\x47\xe0\x6d\x13\xcd\xc9\x79\x63\xf9\x81\x04\x53\xce\x46\x1b\x44\x7c\x42\x5e\x80\xb8\x00\xf1\x65\x13\x20\x2e\x40\xbc\xd3\x04\x88\xaf\xb4\x2c\x40\x5c\x80\xf8\x76\x13\x20\x2e\x40\x5c\x80\x78\x2b\x53\x80\xb8\x00\x71\x01\xe2\x02\xc4\x05\x88\x0b\x10\x17\x20\x2e\x40\x5c\x80\xf8\x66\x13\x20\x0e\x02\xc4\x05\x88\x1f\x2d\xf7\x3b\x02\xf1\x4b\x01\xe2\x02\xc4\x63\x13\x20\x2e\x40\xbc\xd3\x04\x88\xaf\xb4\x2c\x40\x5c\x80\xf8\x76\x13\x20\x2e\x40\x5c\x80\x78\x2b\x53\x80\xb8\x00\x71\x01\xe2\x02\xc4\x05\x88\x0b\x10\x17\x20\x2e\x40\x5c\x80\xf8\x66\x13\x20\x0e\x02\xc4\x05\x88\x1f\x2d\xf7\x3b\x02\xf1\xa7\x02\xc4\x05\x88\xc7\x26\x40\x5c\x80\x78\xa7\x09\x10\x5f\x69\x59\x80\xb8\x00\xf1\xed\x26\x40\x5c\x80\xb8\x00\xf1\x56\xa6\x00\x71\x01\xe2\x02\xc4\x05\x88\x0b\x10\x17\x20\x2e\x40\x5c\x80\xb8\x00\xf1\xcd\x26\x40\x1c\x04\x88\x0b\x10\x3f\x5a\xee\x77\x04\xe2\x3f\x08\x10\x17\x20\x1e\x9b\x00\x71\x01\xe2\x9d\x26\x40\x7c\xa5\x65\x01\xe2\x02\xc4\xb7\x9b\x00\x71\x01\xe2\x02\xc4\x5b\x99\x02\xc4\x05\x88\x0b\x10\x17\x20\x2e\x40\x5c\x80\xb8\x00\x71\x01\xe2\x02\xc4\x37\x9b\x00\x71\x10\x20\x2e\x40\xfc\x68\xb9\xff\x17\x20\x1e\x7a\xfe\x1b\x00\x00\xff\xff\x54\x0c\xf8\x33\xea\x74\x00\x00")

func config_crd_direct_csi_min_io_directcsivolumes_yaml() ([]byte, error) {
	return bindata_read(
		_config_crd_direct_csi_min_io_directcsivolumes_yaml,
		"config/crd/direct.csi.min.io_directcsivolumes.yaml",
	)
}

// Asset loads and returns the asset for the given name.
// It returns an error if the asset could not be found or
// could not be loaded.
func Asset(name string) ([]byte, error) {
	cannonicalName := strings.Replace(name, "\\", "/", -1)
	if f, ok := _bindata[cannonicalName]; ok {
		return f()
	}
	return nil, fmt.Errorf("Asset %s not found", name)
}

// AssetNames returns the names of the assets.
func AssetNames() []string {
	names := make([]string, 0, len(_bindata))
	for name := range _bindata {
		names = append(names, name)
	}
	return names
}

// _bindata is a table, holding each asset generator, mapped to its name.
var _bindata = map[string]func() ([]byte, error){
	"config/crd/direct.csi.min.io_directcsidrives.yaml":  config_crd_direct_csi_min_io_directcsidrives_yaml,
	"config/crd/direct.csi.min.io_directcsivolumes.yaml": config_crd_direct_csi_min_io_directcsivolumes_yaml,
}

// AssetDir returns the file names below a certain
// directory embedded in the file by go-bindata.
// For example if you run go-bindata on data/... and data contains the
// following hierarchy:
//     data/
//       foo.txt
//       img/
//         a.png
//         b.png
// then AssetDir("data") would return []string{"foo.txt", "img"}
// AssetDir("data/img") would return []string{"a.png", "b.png"}
// AssetDir("foo.txt") and AssetDir("notexist") would return an error
// AssetDir("") will return []string{"data"}.
func AssetDir(name string) ([]string, error) {
	node := _bintree
	if len(name) != 0 {
		cannonicalName := strings.Replace(name, "\\", "/", -1)
		pathList := strings.Split(cannonicalName, "/")
		for _, p := range pathList {
			node = node.Children[p]
			if node == nil {
				return nil, fmt.Errorf("Asset %s not found", name)
			}
		}
	}
	if node.Func != nil {
		return nil, fmt.Errorf("Asset %s not found", name)
	}
	rv := make([]string, 0, len(node.Children))
	for name := range node.Children {
		rv = append(rv, name)
	}
	return rv, nil
}

type _bintree_t struct {
	Func     func() ([]byte, error)
	Children map[string]*_bintree_t
}

var _bintree = &_bintree_t{nil, map[string]*_bintree_t{
	"config": {nil, map[string]*_bintree_t{
		"crd": {nil, map[string]*_bintree_t{
			"direct.csi.min.io_directcsidrives.yaml":  {config_crd_direct_csi_min_io_directcsidrives_yaml, map[string]*_bintree_t{}},
			"direct.csi.min.io_directcsivolumes.yaml": {config_crd_direct_csi_min_io_directcsivolumes_yaml, map[string]*_bintree_t{}},
		}},
	}},
}}
