"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.OpenSearchRoutesService = void 0;
exports.registerOpenSearchRoutes = registerOpenSearchRoutes;
var _configSchema = require("@osd/config-schema");
var _lodash = require("lodash");
var _semver = _interopRequireDefault(require("semver"));
var _common = require("../../common");
var _helpers = require("./helpers");
var _helpers2 = require("../utils/helpers");
function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
function _defineProperty(e, r, t) { return (r = _toPropertyKey(r)) in e ? Object.defineProperty(e, r, { value: t, enumerable: !0, configurable: !0, writable: !0 }) : e[r] = t, e; }
function _toPropertyKey(t) { var i = _toPrimitive(t, "string"); return "symbol" == typeof i ? i : i + ""; }
function _toPrimitive(t, r) { if ("object" != typeof t || !t) return t; var e = t[Symbol.toPrimitive]; if (void 0 !== e) { var i = e.call(t, r || "default"); if ("object" != typeof i) return i; throw new TypeError("@@toPrimitive must return a primitive value."); } return ("string" === r ? String : Number)(t); } /*
                                                                                                                                                                                                                                                                                                                          * Copyright OpenSearch Contributors
                                                                                                                                                                                                                                                                                                                          * SPDX-License-Identifier: Apache-2.0
                                                                                                                                                                                                                                                                                                                          */
/**
 * Server-side routes to process OpenSearch-related node API calls and execute the
 * corresponding API calls against the OpenSearch cluster.
 */
function registerOpenSearchRoutes(router, opensearchRoutesService) {
  router.get({
    path: `${_common.CAT_INDICES_NODE_API_PATH}/{pattern}`,
    validate: {
      params: _configSchema.schema.object({
        pattern: _configSchema.schema.string()
      })
    }
  }, opensearchRoutesService.catIndices);
  router.get({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/opensearch/catIndices/{pattern}`,
    validate: {
      params: _configSchema.schema.object({
        pattern: _configSchema.schema.string(),
        data_source_id: _configSchema.schema.string()
      })
    }
  }, opensearchRoutesService.catIndices);
  router.get({
    path: `${_common.GET_MAPPINGS_NODE_API_PATH}/{index}`,
    validate: {
      params: _configSchema.schema.object({
        index: _configSchema.schema.string()
      })
    }
  }, opensearchRoutesService.getMappings);
  router.get({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/opensearch/mappings/{index}`,
    validate: {
      params: _configSchema.schema.object({
        index: _configSchema.schema.string(),
        data_source_id: _configSchema.schema.string()
      })
    }
  }, opensearchRoutesService.getMappings);
  router.get({
    path: `${_common.GET_INDEX_NODE_API_PATH}/{index}`,
    validate: {
      params: _configSchema.schema.object({
        index: _configSchema.schema.string()
      })
    }
  }, opensearchRoutesService.getIndex);
  router.get({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/opensearch/getIndex/{index}`,
    validate: {
      params: _configSchema.schema.object({
        index: _configSchema.schema.string(),
        data_source_id: _configSchema.schema.string()
      })
    }
  }, opensearchRoutesService.getIndex);
  router.post({
    path: _common.SEARCH_INDEX_NODE_API_PATH,
    validate: {
      body: _configSchema.schema.any(),
      query: _configSchema.schema.object({
        verbose: _configSchema.schema.boolean(),
        data_source_version: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, opensearchRoutesService.searchIndex);
  router.post({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/opensearch/search`,
    validate: {
      params: _configSchema.schema.object({
        data_source_id: _configSchema.schema.string()
      }),
      body: _configSchema.schema.any(),
      query: _configSchema.schema.object({
        verbose: _configSchema.schema.boolean(),
        data_source_version: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, opensearchRoutesService.searchIndex);
  router.post({
    path: `${_common.SEARCH_INDEX_NODE_API_PATH}/{index}`,
    validate: {
      params: _configSchema.schema.object({
        index: _configSchema.schema.string()
      }),
      body: _configSchema.schema.any(),
      query: _configSchema.schema.object({
        verbose: _configSchema.schema.boolean(),
        data_source_version: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, opensearchRoutesService.searchIndex);
  router.post({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/opensearch/search/{index}`,
    validate: {
      params: _configSchema.schema.object({
        index: _configSchema.schema.string(),
        data_source_id: _configSchema.schema.string()
      }),
      body: _configSchema.schema.any(),
      query: _configSchema.schema.object({
        verbose: _configSchema.schema.boolean(),
        data_source_version: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, opensearchRoutesService.searchIndex);
  router.post({
    path: `${_common.SEARCH_INDEX_NODE_API_PATH}/{index}/{search_pipeline}`,
    validate: {
      params: _configSchema.schema.object({
        index: _configSchema.schema.string(),
        search_pipeline: _configSchema.schema.string()
      }),
      body: _configSchema.schema.any(),
      query: _configSchema.schema.object({
        verbose: _configSchema.schema.boolean(),
        data_source_version: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, opensearchRoutesService.searchIndex);
  router.post({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/opensearch/search/{index}/{search_pipeline}`,
    validate: {
      params: _configSchema.schema.object({
        index: _configSchema.schema.string(),
        search_pipeline: _configSchema.schema.string(),
        data_source_id: _configSchema.schema.string()
      }),
      body: _configSchema.schema.any(),
      query: _configSchema.schema.object({
        verbose: _configSchema.schema.boolean(),
        data_source_version: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, opensearchRoutesService.searchIndex);
  router.put({
    path: `${_common.INGEST_NODE_API_PATH}/{index}`,
    validate: {
      params: _configSchema.schema.object({
        index: _configSchema.schema.string()
      }),
      body: _configSchema.schema.any()
    }
  }, opensearchRoutesService.ingest);
  router.put({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/opensearch/ingest/{index}`,
    validate: {
      params: _configSchema.schema.object({
        index: _configSchema.schema.string(),
        data_source_id: _configSchema.schema.string()
      }),
      body: _configSchema.schema.any()
    }
  }, opensearchRoutesService.ingest);
  router.post({
    path: `${_common.BULK_NODE_API_PATH}/{pipeline}`,
    validate: {
      params: _configSchema.schema.object({
        pipeline: _configSchema.schema.string()
      }),
      body: _configSchema.schema.any()
    }
  }, opensearchRoutesService.bulk);
  router.post({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/opensearch/bulk/{pipeline}`,
    validate: {
      params: _configSchema.schema.object({
        pipeline: _configSchema.schema.string(),
        data_source_id: _configSchema.schema.string()
      }),
      body: _configSchema.schema.any()
    }
  }, opensearchRoutesService.bulk);
  router.post({
    path: _common.BULK_NODE_API_PATH,
    validate: {
      body: _configSchema.schema.any()
    }
  }, opensearchRoutesService.bulk);
  router.post({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/opensearch/bulk`,
    validate: {
      params: _configSchema.schema.object({
        data_source_id: _configSchema.schema.string()
      }),
      body: _configSchema.schema.any()
    }
  }, opensearchRoutesService.bulk);
  router.post({
    path: _common.SIMULATE_PIPELINE_NODE_API_PATH,
    validate: {
      body: _configSchema.schema.object({
        pipeline: _configSchema.schema.any(),
        docs: _configSchema.schema.any()
      }),
      query: _configSchema.schema.object({
        verbose: _configSchema.schema.boolean()
      })
    }
  }, opensearchRoutesService.simulatePipeline);
  router.post({
    path: `${_common.SIMULATE_PIPELINE_NODE_API_PATH}/{pipeline_id}`,
    validate: {
      body: _configSchema.schema.object({
        docs: _configSchema.schema.any()
      }),
      params: _configSchema.schema.object({
        pipeline_id: _configSchema.schema.string()
      }),
      query: _configSchema.schema.object({
        verbose: _configSchema.schema.boolean()
      })
    }
  }, opensearchRoutesService.simulatePipeline);
  router.post({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/opensearch/simulatePipeline`,
    validate: {
      params: _configSchema.schema.object({
        data_source_id: _configSchema.schema.string()
      }),
      body: _configSchema.schema.object({
        pipeline: _configSchema.schema.any(),
        docs: _configSchema.schema.any()
      }),
      query: _configSchema.schema.object({
        verbose: _configSchema.schema.boolean()
      })
    }
  }, opensearchRoutesService.simulatePipeline);
  router.post({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/opensearch/simulatePipeline/{pipeline_id}`,
    validate: {
      params: _configSchema.schema.object({
        data_source_id: _configSchema.schema.string(),
        pipeline_id: _configSchema.schema.string()
      }),
      body: _configSchema.schema.object({
        docs: _configSchema.schema.any()
      }),
      query: _configSchema.schema.object({
        verbose: _configSchema.schema.boolean()
      })
    }
  }, opensearchRoutesService.simulatePipeline);
  router.get({
    path: `${_common.INGEST_PIPELINE_NODE_API_PATH}/{pipeline_id}`,
    validate: {
      params: _configSchema.schema.object({
        pipeline_id: _configSchema.schema.string()
      })
    }
  }, opensearchRoutesService.getIngestPipeline);
  router.get({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/opensearch/getIngestPipeline/{pipeline_id}`,
    validate: {
      params: _configSchema.schema.object({
        pipeline_id: _configSchema.schema.string(),
        data_source_id: _configSchema.schema.string()
      })
    }
  }, opensearchRoutesService.getIngestPipeline);
  router.get({
    path: `${_common.SEARCH_PIPELINE_NODE_API_PATH}/{pipeline_id}`,
    validate: {
      params: _configSchema.schema.object({
        pipeline_id: _configSchema.schema.string()
      })
    }
  }, opensearchRoutesService.getSearchPipeline);
  router.get({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/opensearch/getSearchPipeline/{pipeline_id}`,
    validate: {
      params: _configSchema.schema.object({
        pipeline_id: _configSchema.schema.string(),
        data_source_id: _configSchema.schema.string()
      })
    }
  }, opensearchRoutesService.getSearchPipeline);
  router.get({
    path: _common.GET_SEARCH_TEMPLATES_NODE_API_PATH,
    validate: {}
  }, opensearchRoutesService.getSearchTemplates);
  router.get({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/opensearch/getSearchTemplates`,
    validate: {
      params: _configSchema.schema.object({
        data_source_id: _configSchema.schema.string()
      })
    }
  }, opensearchRoutesService.getSearchTemplates);
}
class OpenSearchRoutesService {
  constructor(client, dataSourceEnabled) {
    _defineProperty(this, "client", void 0);
    _defineProperty(this, "dataSourceEnabled", void 0);
    _defineProperty(this, "catIndices", async (context, req, res) => {
      const {
        pattern
      } = req.params;
      const {
        data_source_id = ''
      } = req.params;
      try {
        const callWithRequest = (0, _helpers2.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        const response = await callWithRequest('cat.indices', {
          index: pattern,
          format: 'json',
          h: 'health,index'
        });

        // re-formatting the index results to match Index
        const cleanedIndices = response.map(index => ({
          name: index.index,
          health: index.health
        }));
        return res.ok({
          body: cleanedIndices
        });
      } catch (err) {
        return (0, _helpers.generateCustomError)(res, err);
      }
    });
    _defineProperty(this, "getMappings", async (context, req, res) => {
      const {
        index
      } = req.params;
      const {
        data_source_id = ''
      } = req.params;
      try {
        var _response$index;
        const callWithRequest = (0, _helpers2.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        const response = await callWithRequest('indices.getMapping', {
          index
        });

        // Response will be a dict with key being the index name. Attempt to
        // pull out the mappings. If any errors found (missing index, etc.), an error
        // will be thrown.
        const mappings = (_response$index = response[index]) === null || _response$index === void 0 ? void 0 : _response$index.mappings;
        return res.ok({
          body: mappings
        });
      } catch (err) {
        return (0, _helpers.generateCustomError)(res, err);
      }
    });
    _defineProperty(this, "getIndex", async (context, req, res) => {
      const {
        index
      } = req.params;
      const {
        data_source_id = ''
      } = req.params;
      try {
        const callWithRequest = (0, _helpers2.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        const response = await callWithRequest('indices.get', {
          index
        });
        // re-formatting the results to match IndexResponse
        const cleanedIndexDetails = Object.entries(response).map(([indexName, indexDetails]) => ({
          indexName,
          indexDetails
        }));
        return res.ok({
          body: cleanedIndexDetails
        });
      } catch (err) {
        return (0, _helpers.generateCustomError)(res, err);
      }
    });
    _defineProperty(this, "searchIndex", async (context, req, res) => {
      const {
        index,
        search_pipeline
      } = req.params;
      const {
        data_source_id = ''
      } = req.params;
      const {
        verbose = false,
        data_source_version = undefined
      } = req.query;
      const isPreV219 = data_source_version !== undefined ? _semver.default.lt(data_source_version, _common.MINIMUM_FULL_SUPPORTED_VERSION) : false;
      const body = req.body;
      try {
        const callWithRequest = (0, _helpers2.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        let response;
        // If verbose is false/undefined, or the version isn't eligible, omit the verbose param when searching.
        if (!verbose || isPreV219) {
          response = await callWithRequest('search', {
            index: index || '',
            body,
            search_pipeline
          });
        } else {
          response = await callWithRequest('search', {
            index,
            body,
            search_pipeline,
            verbose_pipeline: verbose
          });
        }
        return res.ok({
          body: response
        });
      } catch (err) {
        return (0, _helpers.generateCustomError)(res, err);
      }
    });
    _defineProperty(this, "ingest", async (context, req, res) => {
      const {
        data_source_id = ''
      } = req.params;
      const {
        index
      } = req.params;
      const doc = req.body;
      try {
        const callWithRequest = (0, _helpers2.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        const response = await callWithRequest('index', {
          index,
          body: doc
        });
        return res.ok({
          body: response
        });
      } catch (err) {
        return (0, _helpers.generateCustomError)(res, err);
      }
    });
    _defineProperty(this, "bulk", async (context, req, res) => {
      const {
        data_source_id = ''
      } = req.params;
      const {
        pipeline
      } = req.params;
      const body = req.body;
      try {
        const callWithRequest = (0, _helpers2.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        const response = await callWithRequest('bulk', {
          body,
          pipeline
        });
        return res.ok({
          body: response
        });
      } catch (err) {
        return (0, _helpers.generateCustomError)(res, err);
      }
    });
    _defineProperty(this, "simulatePipeline", async (context, req, res) => {
      const {
        data_source_id = '',
        pipeline_id = ''
      } = req.params;
      const {
        pipeline,
        docs
      } = req.body;
      const {
        verbose = false
      } = req.query;
      try {
        const callWithRequest = (0, _helpers2.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        let response = undefined;
        if (!(0, _lodash.isEmpty)(pipeline_id)) {
          response = await callWithRequest('ingest.simulate', {
            body: {
              docs
            },
            id: pipeline_id,
            verbose
          });
        } else {
          response = await callWithRequest('ingest.simulate', {
            body: {
              docs,
              pipeline
            },
            verbose
          });
        }
        return res.ok({
          body: {
            docs: response.docs
          }
        });
      } catch (err) {
        return (0, _helpers.generateCustomError)(res, err);
      }
    });
    _defineProperty(this, "getIngestPipeline", async (context, req, res) => {
      const {
        pipeline_id
      } = req.params;
      const {
        data_source_id = ''
      } = req.params;
      try {
        const callWithRequest = (0, _helpers2.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        const response = await callWithRequest('ingest.getPipeline', {
          id: pipeline_id
        });
        // re-formatting the results to match IngestPipelineResponse
        const cleanedIngestPipelineDetails = Object.entries(response).map(([pipelineId, ingestPipelineDetails]) => ({
          pipelineId,
          ingestPipelineDetails
        }));
        return res.ok({
          body: cleanedIngestPipelineDetails
        });
      } catch (err) {
        return (0, _helpers.generateCustomError)(res, err);
      }
    });
    _defineProperty(this, "getSearchPipeline", async (context, req, res) => {
      const {
        pipeline_id
      } = req.params;
      const {
        data_source_id = ''
      } = req.params;
      try {
        const callWithRequest = (0, _helpers2.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        const response = await callWithRequest('coreClient.getSearchPipeline', {
          pipeline_id: pipeline_id
        });

        // re-formatting the results to match SearchPipelineResponse
        const cleanedSearchPipelineDetails = Object.entries(response).map(([pipelineId, searchPipelineDetails]) => ({
          pipelineId,
          searchPipelineDetails
        }));
        return res.ok({
          body: cleanedSearchPipelineDetails
        });
      } catch (err) {
        return (0, _helpers.generateCustomError)(res, err);
      }
    });
    _defineProperty(this, "getSearchTemplates", async (context, req, res) => {
      const {
        data_source_id = ''
      } = req.params;
      try {
        var _response$metadata;
        const callWithRequest = (0, _helpers2.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        const response = await callWithRequest('cluster.state', {
          metric: 'metadata',
          filter_path: '**.stored_scripts'
        });
        const searchTemplates = (response === null || response === void 0 || (_response$metadata = response.metadata) === null || _response$metadata === void 0 ? void 0 : _response$metadata.stored_scripts) || {};
        return res.ok({
          body: searchTemplates
        });
      } catch (err) {
        return (0, _helpers.generateCustomError)(res, err);
      }
    });
    this.client = client;
    this.dataSourceEnabled = dataSourceEnabled;
  }
}
exports.OpenSearchRoutesService = OpenSearchRoutesService;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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