MACRO (SKIP_AWS_SDK MSG)
  SET(${RETVAL} OFF PARENT_SCOPE)
  SET(${REASON} ${MSG} PARENT_SCOPE)
  RETURN()
ENDMACRO()

FUNCTION (CHECK_AWS_SDK RETVAL REASON)
  # AWS_SDK_EXTERNAL_PROJECT must be ON
  IF(NOT AWS_SDK_EXTERNAL_PROJECT)
    SKIP_AWS_SDK("AWS_SDK_EXTERNAL_PROJECT is not ON")
  ENDIF()
  IF(NOT NOT_FOR_DISTRIBUTION)
    SKIP_AWS_SDK("AWS SDK has Apache 2.0 License which is not complatible with GPLv2. Set -DNOT_FOR_DISTRIBUTION=ON if you need it")
  ENDIF()
  # Check compiler support
  IF(CMAKE_CXX_COMPILER_ID MATCHES "GNU")
    EXECUTE_PROCESS(COMMAND ${CMAKE_CXX_COMPILER} -dumpversion OUTPUT_VARIABLE GCC_VERSION)
    IF (GCC_VERSION VERSION_LESS 4.8)
      SKIP_AWS_SDK("GCC VERSION too old (${GCC_VERSION}, required is 4.8 or later")
    ENDIF()
  ELSEIF (CMAKE_CXX_COMPILER_ID MATCHES "Clang")
    IF ((CMAKE_CXX_COMPILER_VERSION AND CMAKE_CXX_COMPILER_VERSION VERSION_LESS  3.3) OR
        (CLANG_VERSION_STRING AND CLANG_VERSION_STRING VERSION_LESS  3.3))
      SKIP_AWS_SDK("Clang version too old, required is 3.3 or later")
    ENDIF()
  ELSEIF(MSVC)
    IF (MSVC_VERSION LESS 1800)
      SKIP_AWS_SDK("MSVC version too old, required is VS2015 or later")
    ENDIF()
  ELSE()
    SKIP_AWS_SDK("Unsupported compiler")
  ENDIF()

  # Check OS support
  IF (NOT(WIN32 OR APPLE  OR (CMAKE_SYSTEM_NAME MATCHES "Linux")))
    SKIP_AWS_SDK("OS unsupported by AWS SDK")
  ENDIF()

  # Build from source, using ExternalProject_Add
  # AWS C++ SDK requires cmake 2.8.12
  IF(CMAKE_VERSION VERSION_LESS "2.8.12")
    SKIP_AWS_SDK("CMake is too old")
  ENDIF()

  IF(UNIX)
    # Check librairies required for building SDK
    FIND_PACKAGE(CURL)
    IF(NOT CURL_FOUND)
      SKIP_AWS_SDK("AWS C++ SDK requires libcurl development package")
    ENDIF()
    FIND_PATH(UUID_INCLUDE_DIR uuid/uuid.h)
    IF(NOT UUID_INCLUDE_DIR)
      SKIP_AWS_SDK("AWS C++ SDK requires uuid development package")
    ENDIF()
    IF(NOT APPLE)
      FIND_LIBRARY(UUID_LIBRARIES uuid)
      IF(NOT UUID_LIBRARIES)
        SKIP_AWS_SDK("AWS C++ SDK requires uuid development package")
      ENDIF()
      FIND_PACKAGE(OpenSSL)
      IF(NOT OPENSSL_FOUND)
        SKIP_AWS_SDK("AWS C++ SDK requires openssl development package")
      ENDIF()
    ENDIF()
  ENDIF()
  SET(${RETVAL} ON PARENT_SCOPE)
ENDFUNCTION()


# USE_AWS_SDK_LIBS(target sdk_component1 ... sdk_component_N)
# Example usage
# USE_AWS_SDK_LIBS(aws_key_management kms s3)
FUNCTION(USE_AWS_SDK_LIBS)
  SET(SDK_COMPONENTS ${ARGN})
  LIST(GET SDK_COMPONENTS 0 target)
  IF(NOT TARGET ${target})
    MESSAGE(FATAL_ERROR "${target} is not a valid target")
  ENDIF()
  SET(NON_DISTRIBUTABLE_WARNING "Apache 2.0" CACHE INTERNAL "")
  LIST(REMOVE_AT SDK_COMPONENTS 0)
  FOREACH(comp ${SDK_COMPONENTS})
    SET_PROPERTY(GLOBAL PROPERTY AWS_SDK_LIBS ${comp} APPEND)
    TARGET_LINK_LIBRARIES(${target} aws-cpp-sdk-${comp})
  ENDFOREACH()
  TARGET_LINK_LIBRARIES(${target} aws-cpp-sdk-core)
  TARGET_INCLUDE_DIRECTORIES(${target} PRIVATE ${PROJECT_BINARY_DIR}/extra/aws_sdk/aws_sdk_cpp/include)
  # Link OS libraries that AWS SDK depends on
  IF(WIN32)
    TARGET_LINK_LIBRARIES(${target} bcrypt winhttp wininet userenv version)
  ELSE()
    FIND_PACKAGE(CURL REQUIRED)
    FIND_PACKAGE(OpenSSL REQUIRED)
    TARGET_LINK_LIBRARIES(${target} ${OPENSSL_LIBRARIES} ${CURL_LIBRARIES} ${UUID_LIBRARIES})
  ENDIF()
ENDFUNCTION()
