/*
Name:            random.h
Description:     random number generation  
Library:         random
Authors:         Sven Thommesen, Nelson Minar, Roger Burkhart, Manor Askenazi
Date:            1996-12-11
Modified:        1997-01-15
*/

#import <defobj.h>

//
// InternalState --
//   An object containing internal state that may be retrieved or reset
//   using a representation of binary data of some fixed size.
//
@protocol InternalState
- (unsigned)	getStateSize;
- (void)	getState: (void *) state;
- (void)	setState: (void *) state;
@end

//
// RandomNumberGenerator --
//   A predefined machine-dependent algorithm for generating
//   non-negative integral values uniformly distributed across a
//   architecture defined interval.
//
@protocol RandomNumberGenerator <Create, Drop, InternalState> 
CREATING
+ 		create: aZone setStateFromSeed: (unsigned) seed;  // !added
- (void)	setStateFromSeed: (unsigned)seed;
USING
- (unsigned)	getUnsignedMax;
- (unsigned)	getUnsignedSample;
@end

//
// PMMLCG1, PMMLCG2, PMMLCG2, SWB1, SWB2, SWB3 --
//   predefined random number generators recommended for use (additional
//   generators could also be defined, but only if well documented with
//   reasons to consider using them along with references)
//
@protocol PMMLCG1  <RandomNumberGenerator, CREATABLE> 
@end
@protocol PMMLCG2  <RandomNumberGenerator, CREATABLE> 
@end
@protocol PMMLCG3  <RandomNumberGenerator, CREATABLE> 
@end
@protocol SWB1     <RandomNumberGenerator, CREATABLE> 
@end
@protocol SWB2     <RandomNumberGenerator, CREATABLE> 
@end
@protocol SWB3     <RandomNumberGenerator, CREATABLE> 
@end

//
// ACG, SCG, LCG1, LCG2, LCG3 --
//   predefined random number generators, not recommended for serious use;
//   included for historical reasons (compatibility with earlier releases).
// 
@protocol ACG      <RandomNumberGenerator, CREATABLE> 
@end
@protocol SCG      <RandomNumberGenerator, CREATABLE> 
@end
@protocol LCG1     <RandomNumberGenerator, CREATABLE> 
@end
@protocol LCG2     <RandomNumberGenerator, CREATABLE> 
@end
@protocol LCG3     <RandomNumberGenerator, CREATABLE> 
@end

//
// ProbabilityDistribution --
//   A process for generating a sequence of random numbers matching the
//   frequencies defined by a specified probability distribution.  The
//   process is driven by input from a generator of discrete random numbers.
//
@protocol ProbabilityDistribution <Create, Drop, InternalState>   // !changed
CREATING
+		create: aZone setGenerator: generator;
- (void)	setGenerator: generator;
USING
-               getGenerator;
- (BOOL)	getOptionsInitialized;
@end

//
// IntegerDistribution --
//   A probability distribution that returns integer sample values.
//
@protocol IntegerDistribution <ProbabilityDistribution> 
- (int)		getIntegerSample;
@end

//
// UnsignedDistribution --
//   A probability distribution that returns non-negative integer sample values.
//
@protocol UnsignedDistribution <ProbabilityDistribution> 
- (unsigned)	getUnsignedSample;
@end

//
// DoubleDistribution --
//   A probability distribution that returns an approximation of continuous
//   values as represented by double-precision floating point values.
//
@protocol DoubleDistribution <ProbabilityDistribution>
- (double)	getDoubleSample;
@end

//
// RandomBitDistribution --
//   A generator that returns randomly distributed single bit values
//   (i.e. fair coin tosses).
//
@protocol RandomBitDistribution <IntegerDistribution, CREATABLE>
USING
- (BOOL)	getCoinToss;
@end

//
// UniformInteger --
//   A generator of integral values uniformly distributed across an interval.
//   The interval includes both its endpoints.
//
@protocol UniformInteger <IntegerDistribution, CREATABLE>
CREATING
+		create: aZone setGenerator: generator
                setIntegerMin: (int) minValue setMax: (int) maxValue;

- (void)	setIntegerMin: (int) minValue setMax: (int) maxValue;
USING
- (int)		getIntegerMin;
- (int)		getIntegerMax;

- (int)		getIntegerWithMin: (int) minValue withMax: (int) maxValue;
@end

//
// UniformUnsigned --
//   A generator of non-negative integral values uniformly distributed across
//   an interval.  The interval includes both its endpoints.
//
@protocol UniformUnsigned <UnsignedDistribution, CREATABLE> 
CREATING
+		create: aZone setGenerator: generator
                setUnsignedMin: (unsigned) minValue setMax: (unsigned) maxValue;

- (void)	setUnsignedMin: (unsigned) minValue setMax: (unsigned) maxValue;
USING
- (unsigned)	getUnsignedMin;
- (unsigned)	getUnsignedMax;

- (unsigned)	getUnsignedWithMin: (unsigned) minVal 
			withMax: (unsigned) maxVal;
@end

//
// UniformDouble --
//   A generator of floating point values uniformly distributed
//   across an interval.  The interval includes the lower endpoint but
//   excludes the upper endpoint.
//
@protocol UniformDouble <DoubleDistribution, CREATABLE> 
CREATING
+		create: aZone setGenerator: generator
                setDoubleMin: (double) minValue setMax: (double) maxValue;

- (void)	setDoubleMin: (double) minValue setMax: (double) maxValue;
USING
- (double)	getDoubleMin;
- (double)	getDoubleMax;

- (double)	getDoubleWithMin: (double) minValue withMax: (double) maxValue;
@end

//
// NormalDistribution -- a well-known continuous probability distribution
// LogNormalDistribution -- also well-known ...
//
@protocol Normal <DoubleDistribution> 
CREATING
+		create: aZone setGenerator: generator
                setMean: (double) mean setVariance: (double) variance;

- (void)	setMean: (double) mean setVariance: (double) variance;
USING
- (double)	getMean;
- (double)	getVariance;
- (double)	getStdDev;

- (double)	getSampleWithMean: (double) mean 
			withVariance: (double) variance;
@end
@protocol NormalDistribution <Normal, CREATABLE> 
@end
@protocol LogNormalDistribution <Normal, CREATABLE> 
@end

//
// ExponentialDistribution -- a well-known continuous probability distribution
//
@protocol Exponential <DoubleDistribution, CREATABLE> 
CREATING
+		create: aZone setGenerator: generator
                setMean: (double) mean;

- (void)	setMean: (double) mean;
USING
- (double)	getMean;

- (double)	getSampleWithMean: (double) mean ;

@end

//
// GammaDistribution -- a well-known continuous probability distribution
//
@protocol GammaDistribution <DoubleDistribution, CREATABLE> 
CREATING
+		create: aZone setGenerator: generator
		setAlpha: (double) alpha setBeta: (double) beta;

- (void)        setAlpha: (double) alpha setBeta: (double) beta;
USING
- (double)	getAlpha;
- (double)	getBeta;

- (double)	getSampleWithAlpha: (double) alpha 
			withBeta: (double) beta;
@end

/*
plus other useful continuous probability distributions, each with
their own defining parameters ...

Generators defined by libg++:
  Binomial<n, p>, Erlang<mean, variance>, Geometric<mean>,
  HyperGeometric<mean, variance>, NegativeExpntl<mean>,
  Normal<mean, variance>, LogNormal<mean, variance>, 
  Poisson<mean>, DiscreteUniform<low, high>, Uniform<low, high>,
  Weibull<alpha, beta>, RandomInteger<low, high>

In addition to the above, V1.0 of Swarm should probably have at least a
Binomial (which includes Bernoulli as a special case, at least from a user
point of view), along with a Poisson; other distributions can be added
according to anybody's interest or contribution, but we'll have to decide
which to bundle in the base Swarm distribution vs. to make available as
contributed libraries.
*/

//
// Include declarations of type factories for each protocol marked CREATABLE
// (type factories can be defined either as class names or external id's)
//
#import <random/RandomDefs.h>

@class ACG;
@class SCG;
@class SWB1;
@class SWB2;
@class SWB3;
@class LCG1;
@class LCG2;
@class LCG3;
@class PMMLCG1;
@class PMMLCG2;
@class PMMLCG3;
@class RandomBitDistribution;
@class UniformInteger;
@class UniformUnsigned;
@class UniformDouble;
@class NormalDistribution;
@class LogNormalDistribution;
@class ExponentialDistribution;
@class GammaDistribution;

