/* Copyright (c) 2007-2009 Axel Wachtler
   All rights reserved.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:

   * Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
   * Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
   * Neither the name of the authors nor the names of its contributors
     may be used to endorse or promote products derived from this software
     without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
   AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
   INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
   CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
   POSSIBILITY OF SUCH DAMAGE. */

/* $Id: board.h,v 1.25 2009/06/25 20:45:43 awachtler Exp $ */
/**
 * @file
 * @brief Interface for @ref grpBoard.
 *
 */
#ifndef BOARD_H
#define BOARD_H

#ifndef F_CPU
# error "F_CPU is undefined"
#endif

#include <stdlib.h>
#include <stdint.h>
#include <inttypes.h>
#include <avr/io.h>
#include <avr/sleep.h>
#include <avr/interrupt.h>
#include <util/delay.h>
#include "const.h"


#if defined(stkm16)
# include "boards/board_stkm16.h"
#elif defined(stkm8)
# include "boards/board_stkm8.h"
#elif defined(ics230_11) || defined(icm230_11)
# include "boards/board_ict_11.h"
#elif defined(ict230)
# include "boards/board_ict230.h"
#elif defined(rdk230) || defined(rdk230b) || defined(rdk212) || defined(rdk231)
# include "boards/board_rdk230.h"
#elif defined(psk230) || defined(psk230b) ||\
      defined(psk231) || defined(psk212)
# include "boards/board_stk541.h"
#elif defined(stkt26)
# include "boards/board_stkt26.h"
#elif defined(wdba1281) || defined(mnb900)
# include "boards/board_wdba1281.h"
#elif defined(tiny230)
# include "boards/board_tiny230.h"
#elif defined(stb230) || defined(stb230b) || defined(stb231) || defined(stb212)
# include "boards/board_stb.h"
#elif defined(rbb230) || defined(rbb230b) || defined(rbb231) || defined(rbb212)
# include "boards/board_rbb.h"
#elif defined(rzusb)
# include "boards/board_rzusb.h"
#elif defined(ravrf)
# include "boards/board_ravrf.h"
#elif defined(zgbl230) || defined(zgbl231) || defined(zgbl212) ||\
      defined(zgbh230) || defined(zgbh231) || defined(zgbh212)
# include "boards/board_zgbl.h"
#elif defined(lgee231)
# include "boards/board_lgee.h"
/* #elif defined(...) */
#else
# error "BOARD_TYPE is not defined or wrong"
#endif /* #if defined(BOARD_SHORT) */


/**
 * @brief Macro for delays with us resolution.
 *
 * The avr-libc internal _delay_us() function
 * allows delays up to 255us. Since the radio
 * needs some delays, which are above this value,
 * _delay_ms is used. As long as the argument for
 * the DELAY_US  macro is a compile time constant,
 * no large overhead is produced, because the compiler
 * performs the division.
 *
 */
#define DELAY_US(x)  _delay_ms(x/1000.0)
/**
 * @brief Macro for delays with ms resolution.
 */
#define DELAY_MS(x)  _delay_ms(x)

#ifndef PULLUP_KEYS
  /** The internal pull-up resistors in the MCU are used, if this macro is defined
   * to 1 in the board definition file
   */
# define PULLUP_KEYS (0)
#endif


/**
 * Set MCU into idle mode.
 */
#define SLEEP_ON_IDLE()\
        do{\
            set_sleep_mode(SLEEP_MODE_IDLE);\
            sleep_mode();\
        }while(0);


#ifdef NO_TIMER
//# define HAVE_MALLOC_TIMERS
/** Macro is defined, if there is no TIMER definition in apropriate board*.h */
# define TIMER_POOL_SIZE  (0)
# define TIMER_INIT() do{}while(0)
# define TIMER_IRQ   TIMER1_OVF_vect
#endif

#ifndef HIF_TYPE
/** Macro is defined, if there is no HIF definition in apropriate board*.h */
# define NO_HIF (1)
# define HIF_TYPE (HIF_NONE)
#endif

#ifndef HIF_IO_ENABLE
/** Thiis macro is used to enable the interface circuit of the HIF */
# define HIF_IO_ENABLE() do{}while(0)
#endif

#define HIF_TYPE_IS_UART  ((HIF_TYPE >= HIF_UART_0) && ( HIF_TYPE <= HIF_UART_1))
#define HIF_TYPE_IS_USB   ((HIF_TYPE == HIF_FT245) || (HIF_TYPE == HIF_AT90USB))

/* === Radio Control Pins === */
#ifndef TRX_RESET_INIT
  /** RESET pin IO initialization */
# define TRX_RESET_INIT() DDR_TRX_RESET |= MASK_TRX_RESET
#endif

#ifndef TRX_RESET_HIGH
  /** set RESET pin to high level */
# define TRX_RESET_HIGH() PORT_TRX_RESET |= MASK_TRX_RESET
#endif

#ifndef TRX_RESET_LOW
  /** set RESET pin to low level */
# define TRX_RESET_LOW()  PORT_TRX_RESET &= ~MASK_TRX_RESET
#endif

#ifndef TRX_SLPTR_INIT
  /** SLEEP_TR pin IO initialization */
# define TRX_SLPTR_INIT() DDR_TRX_SLPTR |= MASK_TRX_SLPTR
#endif

#ifndef TRX_SLPTR_HIGH
  /** set SLEEP_TR pin to high level */
# define TRX_SLPTR_HIGH() PORT_TRX_SLPTR |= MASK_TRX_SLPTR
#endif

#ifndef TRX_SLPTR_LOW
  /** set SLEEP_TR pin to low level */
# define TRX_SLPTR_LOW()  PORT_TRX_SLPTR &= ~MASK_TRX_SLPTR
#endif


#if defined (DBG_PORT) && defined (DBG_DDR) && defined (DBG_PIN)
# define DBG_INIT() do{DBG_DDR |= DBG_PIN; DBG_PORT &= ~DBG_PIN;}while(0)
# define DBG_SET() do{DBG_PORT |= DBG_PIN;}while(0)
# define DBG_CLR() do{DBG_PORT &= ~DBG_PIN;}while(0)
# define DBG_TOGGLE() do{DBG_PORT ^= DBG_PIN;}while(0)
#else
# define DBG_INIT() do{}while(0)
# define DBG_SET() do{}while(0)
# define DBG_CLR() do{}while(0)
# define DBG_TOGGLE() do{}while(0)
#endif

#endif /* #ifndef BOARD_H */
