/* Copyright (c) 2007-2009 Axel Wachtler
   All rights reserved.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:

   * Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
   * Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
   * Neither the name of the authors nor the names of its contributors
     may be used to endorse or promote products derived from this software
     without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
   AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
   INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
   CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
   POSSIBILITY OF SUCH DAMAGE. */

/* $Id: radio_rf230.c,v 1.9 2009/06/30 19:27:22 awachtler Exp $ */
/**
 * @file
 * @brief
 * Implementation of high level radio functions for AT86RF230 chip
 *
 */

/* === includes ============================================================ */
#include <avr/pgmspace.h>
#include "radio.h"

/* === globals ============================================================= */
static radio_status_t radiostatus;
//trx_param_t PROGMEM radio_cfg_flash = RADIO_CFG_DATA;

/* === prototypes ========================================================== */
void radio_irq_handler(uint8_t cause);


/* === functions ============================================================ */

/* === internal functions ====================================================*/
/**
 * @brief Error handler
 *
 * @param  err error value (see enumeration radio_error_t)
 * @ingroup grpRadio
 */

void radio_error(radio_error_t err)
{
    sei();
    usr_radio_error(err);
    while(1);
}

/**
 * @brief Frame reception
 *
 */
void radio_receive_frame(void)
{

uint8_t len, lqi, i,  *frm;
uint16_t crc;
    len = trx_frame_read(radiostatus.rxframe, radiostatus.rxframesz, &lqi);
    len &= ~0x80;

    #if defined(SR_RX_CRC_VALID)
        crc = trx_bit_read(SR_RX_CRC_VALID) ? 0 : 1;
    #else
        crc = 0;
        frm = radiostatus.rxframe;
        for (i=0; i < len; i++)
        {
            crc = CRC_CCITT_UPDATE(crc, *frm++);
        }
    #endif
    radiostatus.rxframe = usr_radio_receive_frame(len, radiostatus.rxframe,
                                                  lqi, radiostatus.rxrssi, crc);
    radiostatus.rxrssi = VOID_RSSI;
}

/**
 * @brief IRQ handler for radio functions.
 *
 * This function is called in the transceiver interrupt routine.
 * Keep the implementation of the callback functions
 * (usr_radio_irq, usr_radio_receive_frame) short and efficient.
 *
 * @parm cause  value of the interrupt status register
 *
 */
void radio_irq_handler(uint8_t cause)
{

    if (cause & TRX_IRQ_RX_START)
    {
        radiostatus.rxrssi = trx_bit_read(SR_RSSI);
    }
    if (cause & TRX_IRQ_TRX_END)
    {
        if (radiostatus.state == STATE_RX)
        {
            radio_receive_frame();
        }
        else if (radiostatus.state == STATE_TX)
        {
            usr_radio_tx_done(TX_OK);
            if (radiostatus.rxon_idle == 1)
            {
                radio_set_state(STATE_RX);
            }
            else if (radiostatus.rxon_idle == 0)
            {
              /** @todo think about to stay in TX */
                radio_set_state(STATE_OFF);
            } /* none is done, continous TX */
        }
    }
    usr_radio_irq(cause);
}


/* === external functions ====================================================*/

void radio_init(uint8_t * rxbuf, uint8_t rxbufsz)
{
trx_regval_t status;
    /* init cpu peripherals and global IRQ enable */
    radiostatus.rxframe = rxbuf;
    radiostatus.rxframesz = rxbufsz;
    trx_io_init(DEFAULT_SPI_RATE);
    trx_set_irq_handler(radio_irq_handler);
    /* transceiver initialization */

    TRX_RESET_LOW();
    TRX_SLPTR_LOW();
    DELAY_US(TRX_RESET_TIME_US);
    #if defined(CUSTOM_RESET_TIME_MS)
        DELAY_MS(CUSTOM_RESET_TIME_MS);
    #endif
    TRX_RESET_HIGH();

    /* disable IRQ and clear any pending IRQs */
    trx_reg_write(RG_IRQ_MASK, 0);
    trx_reg_read(RG_IRQ_STATUS);
    trx_bit_write(SR_TRX_CMD, CMD_TRX_OFF);
    DELAY_US(510);
    status = trx_bit_read(SR_TRX_STATUS);
    if (status != TRX_OFF)
    {
        radio_error(STATE_SET_FAILED);
    }
    trx_bit_write(SR_TX_AUTO_CRC_ON, 1);
    trx_reg_write(RG_IRQ_MASK, TRX_IRQ_RX_START | TRX_IRQ_TRX_END);
}


void radio_set_state(radio_state_t state)
{
volatile trx_regval_t cmd, expstatus, currstatus;
uint8_t retries;

//    trx_bit_write(SR_TRX_CMD, CMD_FORCE_TRX_OFF);
    switch(state)
    {
        case STATE_OFF:
            expstatus = TRX_OFF;
            cmd = CMD_TRX_OFF;
            break;

        case STATE_RX:
            expstatus = RX_ON;
            cmd = CMD_RX_ON;
            break;

        case STATE_TX:
            expstatus = PLL_ON;
            cmd = CMD_PLL_ON;
            break;

        case STATE_RXAUTO:
            expstatus = RX_AACK_ON;
            cmd = CMD_RX_AACK_ON;
            break;

        case STATE_TXAUTO:
            expstatus = TX_ARET_ON;
            cmd = CMD_TX_ARET_ON;
            break;

        default:
            radio_error(GENERAL_ERROR);
            expstatus = TRX_OFF;
            cmd = CMD_TRX_OFF;
            break;

     }

#if 0
    currstatus = trx_bit_read(SR_TRX_STATUS);
    if (expstatus != currstatus)
    {
        trx_bit_write(SR_TRX_CMD, cmd);
        retries = 10;
        while (retries--)
        {
            currstatus = trx_bit_read(SR_TRX_STATUS);
            if (expstatus == currstatus)
            {
                break;
            }
            DELAY_US(6);
        }
    }
#else
    trx_bit_write(SR_TRX_CMD, cmd);
    retries = 128;
    do
    {
        currstatus = trx_bit_read(SR_TRX_STATUS);
        if (expstatus == currstatus)
        {
            break;
        }
        DELAY_US(32);
    }
    while (retries--);

#endif
    if (expstatus != currstatus)
    {
        radio_error(STATE_SET_FAILED);
    }

    radiostatus.state = state;
}

void radio_set_param(radio_attribute_t attr, radio_param_t parm)
{
    switch (attr)
    {
        case phyCurrentChannel:
            if (((int)parm.channel >= TRX_MIN_CHANNEL) &&
                ((int)parm.channel <= TRX_MAX_CHANNEL))
            {
                trx_bit_write(SR_CHANNEL, parm.channel);
                radiostatus.channel = parm.channel;
            }
            else
            {
                radio_error(SET_PARM_FAILED);
            }
            break;

        case phyTransmitPower:
            if (parm.tx_pwr <= 15)
            {
                radiostatus.tx_pwr = parm.tx_pwr;
                trx_bit_write(SR_TX_PWR, radiostatus.tx_pwr);
            }
            else
            {
                radio_error(SET_PARM_FAILED);
            }
            break;

        case phyCCAMode:
            if (parm.cca_mode <= 3)
            {
                radiostatus.cca_mode = parm.cca_mode;
                trx_bit_write(SR_CCA_MODE, radiostatus.cca_mode);
            }
            else
            {
                radio_error(SET_PARM_FAILED);
            }
            break;

        case phyRxOnIdle:
            radiostatus.rxon_idle = parm.rxon_idle;
            if (radiostatus.rxon_idle == 1)
            {
                radio_set_state(STATE_RX);
            }
            else if (radiostatus.rxon_idle == 0)
            {
                radio_set_state(STATE_OFF);
            } /* otherwise we are continous tx mode */
            break;

        case phyChannelsSupported:
            break;

        default:
            radio_error(SET_PARM_FAILED);
            break;
    }
}


void radio_send_frame(uint8_t len, uint8_t *frm, uint8_t compcrc)
{
    TRX_SLPTR_HIGH();
    TRX_SLPTR_LOW();
    trx_frame_write(len, frm);
}

radio_cca_t radio_do_cca(void)
{
uint8_t tmp, trxcmd, trxstatus;
radio_cca_t ret = CCA_FREE;

    trxcmd = trx_reg_read(RG_TRX_STATE);
    trx_reg_write(RG_TRX_STATE, CMD_RX_ON);
    tmp = 130;
    do
    {
         trxstatus = trx_bit_read(SR_TRX_STATUS);
         if ((trxstatus == RX_ON) || (trxstatus == BUSY_RX))
         {
            break;
         }
         DELAY_US(32); /* wait for one octett */
    }
    while(tmp-- > 0);

    trx_reg_write(RG_TRX_STATE, CMD_PLL_ON);
    trx_reg_write(RG_TRX_STATE, CMD_RX_ON);

    trx_bit_write(SR_CCA_REQUEST,1);
    DELAY_US(140);
    /* we need to read the whole status register
     * because CCA_DONE and CCA_STATUS are valid
     * only for one read, after the read they are reset
     */
    tmp = trx_reg_read(RG_TRX_STATUS);

    if((tmp & 0x80) == 0)
    {
        ret = CCA_FAIL;
    }
    else if (tmp & 0x40)
    {
        ret = CCA_FREE;
    }
    else
    {
        ret = CCA_BUSY;
    }

    trx_reg_write(RG_TRX_STATE, trxcmd);

    return ret;
}

/* EOF */
