#   Copyright (c) 2008 Axel Wachtler
#   All rights reserved.
#
#   Redistribution and use in source and binary forms, with or without
#   modification, are permitted provided that the following conditions
#   are met:
#
#   * Redistributions of source code must retain the above copyright
#     notice, this list of conditions and the following disclaimer.
#   * Redistributions in binary form must reproduce the above copyright
#     notice, this list of conditions and the following disclaimer in the
#     documentation and/or other materials provided with the distribution.
#   * Neither the name of the authors nor the names of its contributors
#     may be used to endorse or promote products derived from this software
#     without specific prior written permission.
#
#   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
#   AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
#   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
#   ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
#   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
#   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
#   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
#   INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
#   CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
#   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
#   POSSIBILITY OF SUCH DAMAGE.

# $Id: ieee802154_io.py,v 1.9 2009/02/04 21:27:59 awachtler Exp $
##
# @file
# @ingroup grpContribCapture
# @brief File/Serial port io related classes.
#

# === import ==================================================================
import sys, serial, time, threading, Queue, struct, os
from ieee802154_base import PcapBase, UtilBase

# === globals =================================================================

# === functions ===============================================================

# === classes =================================================================
##
# @brief Capture file reader.
#
# This class reads packets from a capture file,
# that was previously stored with tshark/wireshark.
#
# @todo This class reads the entire file (@ref open()) at once.
# So it would be a good idea, to read the file block by block.
#
class FileIn(PcapBase, UtilBase):

    ## Open and read a capture file.
    # @param fname name of the file
    def open(self, fname):
        self.fname = fname
        self.fh = open(fname,'r')
        self.FCNT = 0
        d = self.fh.read()
        self.HEADER,self.frames = self.pcap_parse_data(d)
        print "nb frames", len(self.frames)
        self.frameidx = 0

    ## close the file handle
    def close(self):
        self.fh.close()

    ## return file header
    def pcap_get_header(self):
        return self.HEADER

    def info(self):
        afn = os.path.abspath(self.fh.name)
        ret = {'type'   : 'file',
               'file'   : os.path.basename(afn),
               'dir'    : os.path.dirname(afn),
               'frames' : self.FCNT,
                }
        return ret

    ##
    # Setting of a channel is is unsupported for a file
    def set_channel(self, channel):
        self.error("can not set channel on a file")

    ## read a frame f
    def read_packet(self):
        try:
            #print "read frame %d" % self.frameidx
            ret = self.frames[self.frameidx]
            self.frameidx += 1
            self.FCNT += 1
            print map(hex,map(ord,ret))
        except IndexError:
            ret = None
        except:
            import traceback
            traceback.print_exc()
            ret = None
        return ret

##
# @brief Serial Interface Reader.
#
class PortIn(PcapBase, UtilBase):
    TMO = 1
    FCNT = 0
    UNSYNC = 0
    SYNCED = 1
    IDXERR = 0
    BAUDRATE = 38400
    def __init__(self):
        self.RxThread = None
        self.RxQueue = Queue.Queue()
        self.TextQueue = Queue.Queue()
        self.channel = 0
        self.clist = []
        self.state = self.UNSYNC
        self.maxpackets = -1
    ## @brief Open the serial port.
    def open(self, fname):
        if self.RxThread:
            self.RxThread.join()
        self.fname = fname
        self.sport = serial.Serial(fname, self.BAUDRATE)
        self.sport.open()
        self.RxThread=threading.Thread(target = self.__rx__)
        self.RxThread.setDaemon(1)
        self.RxThread.start()
        self.init()

    def init(self):
        self.sport.write("\n\nidle\n")
        self.sport.write("parms\n")
        time.sleep(.5)
        for l in self.get_text_queue():
            if l.find("SUPP_CMSK")>=0:
                self.clist = self.get_channels(eval(l.split(":")[1]))
            if l.find("CURR_CHAN")>=0:
                self.channel = eval(l.split(":")[1])
            if l.find("TIMER_SCALE")>=0:
                self.tscale = eval(l.split(":")[1])
            if l.find("TICK_NUMBER")>=0:
                self.ticknb = eval(l.split(":")[1])
        t = int(time.time())
        self.sport.write("timeset %s\n" % t)
        self.timebase = t
        self.sport.write("sniff\n")

    def get_text_queue(self):
        ret = ""
        while not self.TextQueue.empty():
            ret += self.TextQueue.get()
        ret = ret.replace('\x0b',"")
        return ret.split("\n")

    def get_channels(self,cmask):
        ret = []
        cidx = 0
        while cmask != 0:
            if cmask & 1:
                ret.append(cidx)
            cidx += 1
            cmask /=2
        return ret

    def close(self):
        if self.RxThread != None:
            self.RxThread.join(self.TMO)
            self.RxThread = None
        self.sport.close()

    def info(self):
        ret = {'type'   : 'port',
               'chan'   : self.channel,
               'port'   : self.sport.port,
               'clist'  : self.clist,
               'tscale' : self.tscale,
               'ticknb' : self.ticknb,
                }
        return ret

    def __rx__(self):
        frm = ""
        while 1:
            sdata = self.sport.read()
            frm += sdata
            if self.state == self.UNSYNC:
                self.TextQueue.put(sdata)
                p = self.sync_search(frm)
                if type(p) != None:
                    frm = frm[p:]
                    self.state = self.SYNCED
            if self.state == self.SYNCED:
                self.state,frm = self.packetizer(frm)
                self.message(2,"state sync after packetizer(), state=%d, len_frm=%d",self.state, len(frm))

    def sync_search(self,frm):
        ret = None
        p = 0
        nbstartpos = frm.count('\x01')
        dlen = len(frm)
        if nbstartpos:
            self.message(2,"syncsearch : dlen=%d, nbstarts=%d" ,dlen,nbstartpos)
            for idx in range(nbstartpos):
                try:
                    p += frm[p:].index('\x01')
                    plen = ord(frm[p+1])
                    pe = p + plen + 2
                    self.message(2,"syncing : idx=%d, packet=%d:%d, plen=%d dlen=%d", idx,p,pe,plen,dlen)

                    if pe <= dlen:
                        self.message(2,"packet : %s " , str(map(lambda i,f=frm: hex(ord(f[i])), (p,p+1,pe-1,pe) )))

                        if(frm[pe] == '\x04'):
                            ret = p
                            self.message(1,"synced : idx=%d, packet=%d:%d, plen=%d dlen=%d", idx,p,pe,plen,dlen)
                            raise "Synced"
                    p += 1
                except IndexError:
                    # this catches the blind access in line "l = ord(frm[p+1])"
                    break
                except "Synced":

                    break
                except:
                    self.exc_handler("sync_search")
                    break
        return ret

    def packetizer(self,frm):
        state = self.SYNCED
        while 1:
            frmlen = len(frm)
            if len(frm) < 3:
                # incomplete data
                break
            if frm[0] != '\x01':
                state = self.UNSYNC
                break
            pktlen = ord(frm[1])
            if (pktlen+3) > frmlen:
                # incomplete data
                break
            if frm[pktlen+2] != '\x04':
                state = self.UNSYNC
                break
            packet,frm = frm[:pktlen+3],frm[pktlen+3:]

            ## XXX refactor this hack
            # convert frame from serial line to pcap format
            # u8   length
            # u64  ts
            # u8[] frm
            #
            # packet data is prefixed with byte STX (1) + length (1)
            # and suffixed with EOT char (4)
            fl = pktlen - 8
            ticks,pkt = packet[2:10], packet[10:-1]
            ticks = struct.unpack('LL',ticks)
            tstamp = ((ticks[0]*self.ticknb) + ticks[1]) * self.tscale
            t_sec = int(tstamp)
            t_usec = int((tstamp-t_sec)*1.0e6)
            ts = struct.pack('LL',(t_sec+self.timebase),t_usec)
            lenfield = struct.pack("LL",fl,fl)
            packet = ts+lenfield+pkt

            if self.FCNT < self.maxpackets or self.maxpackets == 0:
                self.RxQueue.put(packet)
                self.FCNT += 1
                self.message(1,"Found Packet l=%d qsize: %d:\npkt: %s",
                        pktlen, self.RxQueue.qsize(),
                        " ".join(map(lambda s: '%02x' %s,map(ord,packet))))
            else:
                self.message(1,"Discard Packet l=%d qsize: %d", pktlen,self.RxQueue.qsize())
        return state,frm

    def set_channel(self, channel):
        if channel in self.clist:
            self.channel = channel
            time.sleep(0.1) # this sleep is somehow needed for rzusb stick.
            self.sport.write("chan %d\n" % channel)
        else:
            print "unsupported channel %d not in %s" % (channel,self.clist)

    def read_packet(self):
        if self.RxQueue.empty():
            ret = None
        else:
            ret = self.RxQueue.get()
        return ret

# === init ====================================================================
