/* Copyright (c) 2012 - 2013 Samuel Klipphahn
   All rights reserved.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:

   * Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
   * Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
   * Neither the name of the authors nor the names of its contributors
     may be used to endorse or promote products derived from this software
     without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
   AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
   INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
   CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
   POSSIBILITY OF SUCH DAMAGE. */

/* $Id: $ */
/**
 * @file
 * @brief Simple Routing Protocoll.
 * @_addtogroup grplwMesh
 */


/* === includes ============================================================ */
#include "lw_mac.h"
#include "lw_route.h"
#include "lw_mesh.h"
#include "lw_cmd.h"
#include "timer.h"

/* === macros ============================================================== */

/* === types =============================================================== */

/* === globals ============================================================= */
static NWK_RouteTableEntry_t routes[NWK_ROUTE_TABLE_SIZE];
static NWK_DuplicateRejectionTableEntry_t
    duplicates[NWK_DUPLICATE_REJECTION_TABLE_SIZE];

/* === prototypes ========================================================== */
static void clear_duplicates();
static time_t duplicates_ttl_update();

static void clear_routes();
static void add_or_update_route(uint16_t dst_addr, uint16_t next_hop_addr,
        uint8_t lqi);
/** find routing table entry for @a dst_addr
 *  @param dst_addr Address of destination node
 *  @retval id of routing table entry or @ref NWK_NO_ROUTE_ID
 *  if no entry is found */
static uint16_t get_route_id(uint16_t dst_addr);

/* === functions =========================================================== */

void lwr_init()
{
    clear_routes();
    clear_duplicates();
    /*start TTL update of duplicates table entries*/
    timer_start(duplicates_ttl_update,
            MSEC(NWK_DUPLICATES_REJECTION_TIMER_INTERVAL), 0);
}

uint16_t lwr_get_next_hop_addr(uint16_t dst_addr)
{
    uint16_t next_route_id = get_route_id(dst_addr);
    uint16_t next_hop_addr;

    if(next_route_id != NWK_NO_ROUTE_ID)
    {
        next_hop_addr = routes[next_route_id].nextHopAddr;
    } else {
        next_hop_addr = NWK_BROADCAST_ADDR;
    }

    return next_hop_addr;
}

void lwr_save_routes(uint16_t lw_src_addr, uint16_t m_src_addr,
        uint8_t lqi)
{
    /* add/update direct route */
    add_or_update_route(m_src_addr, m_src_addr, lqi);

    if(lw_src_addr != m_src_addr && NWK_IS_ROUTER_ADDR(m_src_addr))
    {
        /* add/update indirect route */
        add_or_update_route(lw_src_addr, m_src_addr, lqi);
    }
}

bool lwr_prep_r_frame(NWK_Tasklist_t *task)
{
    bool send_frame;
    uint16_t route_id = get_route_id(task->frame->lw_dstAddr);
    if(route_id != NWK_NO_ROUTE_ID)
    {
        /* prepare frame for routing, lw_mesh sends frame */
        task->frame->m_dstAddr = routes[route_id].nextHopAddr;
        send_frame = true;
    } else {
        /* self send route error frame */
        task->info.req.dstAddr       = task->frame->lw_srcAddr;
        task->info.req.dstEndpoint   = NWK_ENDPOINT_ID_LW_COMMAND;
        task->info.req.srcEndpoint   = NWK_ENDPOINT_ID_LW_COMMAND;
        task->info.req.options       = 0;
        lwc_prep_r_err_pl(task->frame,
                (NWK_cmd_r_err_t *) task->frame->lw_payload);
        task->info.req.data          = task->frame->lw_payload;
        task->info.req.size          = sizeof(NWK_cmd_r_err_t);
        task->info.req.confirm       = NULL; /* no application but routing */
        /* put route error frame to queue */
        lw_mesh_data_req(&task->info.req);

        /* clear this buffer */
        free(task->frame);
        task->type = NWK_TASK_NO_TASK;

        /* call task handler to send frame */
        lw_mesh_task_handler();
        send_frame = false;
    }
    return send_frame;
}

/* is called every time a frame was sent successfully or sending failed */
void lwr_data_confirm(NWK_DataReq_t *req)
{
    uint16_t route_id = get_route_id(req->dstAddr);

    if(route_id != NWK_NO_ROUTE_ID)
    {
        /* update routing table */
        switch(req->status)
        {
        case NWK_SUCCESS_STATUS:
            routes[route_id].score = NWK_ROUTE_DEFAULT_SCORE;
            break;

        default:
            routes[route_id].score--;
            if(routes[route_id].score == 0)
            {
                /* drop this entry */
                routes[route_id].dstAddr = NWK_BROADCAST_ADDR;
            }
            break;
        }
    }
}

bool lwr_is_duplicate_frame(uint8_t lw_seq, uint16_t lw_src_addr)
{
    bool have_duplicate = false;
    bool table_full = false;
    uint8_t nr = 0;
    for(; nr < NWK_DUPLICATE_REJECTION_TABLE_SIZE; nr++)
    {
        if(duplicates[nr].ttl != 0
                && duplicates[nr].lw_seq == lw_seq
                && duplicates[nr].lw_src_addr == lw_src_addr)
        {
            have_duplicate = true;
            break;
        }
    }
    if(!have_duplicate)
    {
        /* add to table */
        nr = 0;
        table_full = true;
        for(; nr < NWK_DUPLICATE_REJECTION_TABLE_SIZE; nr++)
        {
            if(duplicates[nr].ttl == 0)
            {
                duplicates[nr].lw_seq       = lw_seq;
                duplicates[nr].lw_src_addr  = lw_src_addr;
                duplicates[nr].ttl          = NWK_DUPLICATE_REJECTION_TTL;
                table_full = false;
                break;
            }
        }
    }
    return (have_duplicate || table_full);
}

static void clear_duplicates()
{
    uint8_t i;
    for(i = 0; i < NWK_DUPLICATE_REJECTION_TABLE_SIZE; i++)
    {
        duplicates[i].ttl = 0;
    }
}

static time_t duplicates_ttl_update()
{
    uint8_t i;
    for(i = 0; i < NWK_DUPLICATE_REJECTION_TABLE_SIZE; i++)
    {
        if(duplicates[i].ttl > 0)
        {
            /* reduce TTL but prevent negative values*/
            if(duplicates[i].ttl < NWK_DUPLICATES_REJECTION_TIMER_INTERVAL)
            {
                duplicates[i].ttl = 0;
            } else {
                duplicates[i].ttl -= NWK_DUPLICATES_REJECTION_TIMER_INTERVAL;
            }
        }
    }
    /* restart timer */
    return MSEC(NWK_DUPLICATES_REJECTION_TIMER_INTERVAL);
}

static void clear_routes()
{
    uint8_t i;
    for(i = 0; i < NWK_ROUTE_TABLE_SIZE; i++)
    {
        routes[i].dstAddr = NWK_BROADCAST_ADDR;
    }
}

static void add_or_update_route(uint16_t dst_addr, uint16_t next_hop_addr,
        uint8_t lqi)
{
    uint16_t route_id = get_route_id(dst_addr);
    if(route_id != NWK_NO_ROUTE_ID)
    {
        /* we already know a route to this node         ==> update */
        if(routes[route_id].nextHopAddr != next_hop_addr
            && routes[route_id].lqi < lqi)
        {
            /* take new route with better lqi */
            routes[route_id].nextHopAddr    = next_hop_addr;
            routes[route_id].score          = NWK_ROUTE_DEFAULT_SCORE;
            routes[route_id].lqi            = lqi;
        }
    } else {
        /* we do not know any route to this dst_addr    ==> add */
        uint8_t free_line = 0;
        for(; free_line < NWK_ROUTE_TABLE_SIZE; free_line++)
        {
            /* find first free line */
            if(routes[free_line].dstAddr == NWK_BROADCAST_ADDR)
            {
                break;
            }
        }

        if (free_line != NWK_ROUTE_TABLE_SIZE)
        {
            routes[free_line].dstAddr       = dst_addr;
            routes[free_line].nextHopAddr   = next_hop_addr;
            routes[free_line].score         = NWK_ROUTE_DEFAULT_SCORE;
            routes[free_line].lqi           = lqi;
        }
        else
        {
            /*TODO: what to do if table is full? */
        }
    }
}

void lwr_r_err_received(uint16_t src_addr, uint16_t dst_addr)
{
    /*TODO: Why do we need the src_addr here?*/
    uint16_t route_id = get_route_id(dst_addr);
    if(route_id != NWK_NO_ROUTE_ID)
    {
        /* remove this entry */
        routes[route_id].dstAddr = NWK_BROADCAST_ADDR;
    }
}

static uint16_t get_route_id(uint16_t dst_addr)
{
    uint16_t routing_id = 0;
    for(; routing_id < NWK_ROUTE_TABLE_SIZE; routing_id++)
    {
        if(routes[routing_id].dstAddr == dst_addr)
        {
            break;
        }
    }
    if(routing_id == NWK_ROUTE_TABLE_SIZE)
    {
        /* no entry was found */
        routing_id = NWK_NO_ROUTE_ID;
    }
    return routing_id;
}
