// This application will take a set of files in a directory and generate an "embedded" resource file containing the
// compressed contents of those files and related metadata.
package main

import (
	"bufio"
	"bytes"
	"compress/gzip"
	"encoding/base64"
	"flag"
	"fmt"
	"io/ioutil"
	"log"
	"os"
	"path"
)

var (
	inputDir    string
	outputFile  string
	packageName string
)

func main() {
	flag.StringVar(&inputDir, "in", "", "The directory to read resources from.")
	flag.StringVar(&outputFile, "out", "resources.go", "The go file to write the resources to.")
	flag.StringVar(&packageName, "pkg", "main", "The package for the resource module.")
	flag.Parse()

	files, err := ioutil.ReadDir(inputDir)
	if err != nil {
		log.Fatal(err)
	}

	fd, err := os.Create(outputFile)
	if err != nil {
		log.Fatal(err)
	}

	defer fd.Close()

	w := bufio.NewWriter(fd)
	defer w.Flush()

	w.WriteString(fmt.Sprintf(`// AUTOGENERATED CODE. DO NOT EDIT.

package %s

import (
	"bytes"
	"compress/gzip"
	"encoding/base64"
	"fmt"
	"io"
)

`, packageName))

	w.WriteString("var embeddedResources = map[string]string{\n")

	for _, file := range files {
		if file.Name() != "scalars.json" && path.Ext(file.Name()) != ".tmpl" {
			continue
		}

		compressed, err := compressFile(path.Join(inputDir, file.Name()))
		if err != nil {
			log.Fatal(err)
		}

		w.WriteString(fmt.Sprintf("\t\"%s\": \"%s\",\n", file.Name(), compressed))
	}

	w.WriteString("}\n")

	w.WriteString(`
func fetchResource(name string) ([]byte, error) {
	raw, ok := embeddedResources[name]
	if !ok {
		return nil, fmt.Errorf("Could not find resource for '%s'", name)
	}

	compressed, err := base64.StdEncoding.DecodeString(raw)
	if err != nil {
		return nil, err
	}

	var out bytes.Buffer
	buf := bytes.NewBuffer(compressed)
	
	r, err := gzip.NewReader(buf)
	if err != nil {
		return nil, err
	}

	if _, err := io.Copy(&out, r); err != nil {
		return nil, err
	}

	return out.Bytes(), nil
}
`)
}

func compressFile(path string) (string, error) {
	var gBuf bytes.Buffer

	file, err := ioutil.ReadFile(path)
	if err != nil {
		return "", err
	}

	w := gzip.NewWriter(&gBuf)
	if _, err = w.Write(file); err != nil {
		return "", err
	}

	if err = w.Close(); err != nil {
		return "", err
	}

	return format(&gBuf), nil
}

func format(b *bytes.Buffer) string {
	var bBuf bytes.Buffer
	bw := base64.NewEncoder(base64.StdEncoding, &bBuf)
	bw.Write(b.Bytes())
	bw.Close()

	return bBuf.String()
}
