/*
 * Copyright (c) 2020-2024 Estonian Information System Authority
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#pragma once

#include "flag-set-cpp/flag_set.hpp"

#include <memory>
#include <vector>
#include <limits>

// The rule of five (C++ Core guidelines C.21).
#define PCSC_CPP_DISABLE_COPY_MOVE(Class)                                                          \
    Class(const Class&) = delete;                                                                  \
    Class& operator=(const Class&) = delete;                                                       \
    Class(Class&&) = delete;                                                                       \
    Class& operator=(Class&&) = delete

#ifdef WIN32
#define PCSC_CPP_WARNING_PUSH __pragma(warning(push))
#define PCSC_CPP_WARNING_POP __pragma(warning(pop))
#define PCSC_CPP_WARNING_DISABLE_CLANG(text)
#define PCSC_CPP_WARNING_DISABLE_GCC(text)
#define PCSC_CPP_WARNING_DISABLE_MSVC(number) __pragma(warning(disable : number))
#else
#define PCSC_CPP_DO_PRAGMA(text) _Pragma(#text)
#define PCSC_CPP_WARNING_PUSH PCSC_CPP_DO_PRAGMA(GCC diagnostic push)
#define PCSC_CPP_WARNING_POP PCSC_CPP_DO_PRAGMA(GCC diagnostic pop)
#if __clang__
#define PCSC_CPP_WARNING_DISABLE_CLANG(text) PCSC_CPP_DO_PRAGMA(clang diagnostic ignored text)
#else
#define PCSC_CPP_WARNING_DISABLE_CLANG(text)
#endif
#define PCSC_CPP_WARNING_DISABLE_GCC(text) PCSC_CPP_DO_PRAGMA(GCC diagnostic ignored text)
#define PCSC_CPP_WARNING_DISABLE_MSVC(text)
#endif

namespace pcsc_cpp
{

using byte_type = unsigned char;
using byte_vector = std::vector<byte_type>;
#ifdef _WIN32
using string_t = std::wstring;
#else
using string_t = std::string;
#endif

/** Opaque class that wraps the PC/SC resource manager context. */
class Context;
using ContextPtr = std::shared_ptr<Context>;

/** Returns the value of the response status bytes SW1 and SW2 as a single status word SW. */
constexpr uint16_t toSW(byte_type sw1, byte_type sw2) noexcept
{
    return uint16_t(sw1 << 8) | sw2;
}

/** Struct that wraps response APDUs. */
struct ResponseApdu
{
    enum Status {
        OK = 0x90,
        MORE_DATA_AVAILABLE = 0x61,
        VERIFICATION_FAILED = 0x63,
        VERIFICATION_CANCELLED = 0x64,
        WRONG_LENGTH = 0x67,
        COMMAND_NOT_ALLOWED = 0x69,
        WRONG_PARAMETERS = 0x6a,
        WRONG_LE_LENGTH = 0x6c
    };

    byte_type sw1 {};
    byte_type sw2 {};

    byte_vector data;

    static constexpr size_t MAX_DATA_SIZE = 256;
    static constexpr size_t MAX_SIZE = MAX_DATA_SIZE + 2; // + sw1 and sw2

    ResponseApdu(byte_type s1, byte_type s2, byte_vector d = {}) :
        sw1(s1), sw2(s2), data(std::move(d))
    {
    }

    ResponseApdu() = default;

    static ResponseApdu fromBytes(byte_vector data)
    {
        if (data.size() < 2) {
            throw std::invalid_argument("Need at least 2 bytes for creating ResponseApdu");
        }

        PCSC_CPP_WARNING_PUSH
        PCSC_CPP_WARNING_DISABLE_GCC("-Warray-bounds") // avoid GCC 13 false positive warning
        byte_type sw1 = data[data.size() - 2];
        byte_type sw2 = data[data.size() - 1];
        data.resize(data.size() - 2);
        PCSC_CPP_WARNING_POP

        // SW1 and SW2 are in the end
        return {sw1, sw2, std::move(data)};
    }

    byte_vector toBytes() const
    {
        // makes a copy, valid both if data is empty or full
        auto bytes = data;

        bytes.push_back(sw1);
        bytes.push_back(sw2);

        return bytes;
    }

    constexpr uint16_t toSW() const noexcept { return pcsc_cpp::toSW(sw1, sw2); }

    constexpr bool isOK() const noexcept { return sw1 == OK && sw2 == 0x00; }

    // TODO: friend function toString() in utilities.hpp
};

/** Struct that wraps command APDUs. */
struct CommandApdu
{
    byte_type cla;
    byte_type ins;
    byte_type p1;
    byte_type p2;
    unsigned short le;
    // Lc is data.size()
    byte_vector data;

    static const size_t MAX_DATA_SIZE = 255;
    static const unsigned short LE_UNUSED = std::numeric_limits<unsigned short>::max();

    CommandApdu(byte_type c, byte_type i, byte_type pp1, byte_type pp2, byte_vector d = {},
                unsigned short l = LE_UNUSED) :
        cla(c), ins(i), p1(pp1), p2(pp2), le(l), data(std::move(d))
    {
    }

    CommandApdu(const CommandApdu& other, byte_vector d) :
        CommandApdu(other.cla, other.ins, other.p1, other.p2, std::move(d), other.le)
    {
    }

    constexpr bool isLeSet() const noexcept { return le != LE_UNUSED; }

    byte_vector toBytes() const
    {
        if (data.size() > MAX_DATA_SIZE) {
            throw std::invalid_argument("Command chaining not supported");
        }

        auto bytes = byte_vector {cla, ins, p1, p2};

        if (!data.empty()) {
            bytes.push_back(static_cast<byte_type>(data.size()));
            bytes.insert(bytes.end(), data.cbegin(), data.cend());
        }

        if (isLeSet()) {
            // TODO: EstEID spec: the maximum value of Le is 0xFE
            if (le > ResponseApdu::MAX_DATA_SIZE)
                throw std::invalid_argument("LE larger than response size");
            bytes.push_back(static_cast<byte_type>(le));
        }

        return bytes;
    }
};

/** Opaque class that wraps the PC/SC smart card resources like card handle and I/O protocol. */
class CardImpl;
using CardImplPtr = std::unique_ptr<CardImpl>;

/** PIN pad PIN entry timer timeout */
constexpr uint8_t PIN_PAD_PIN_ENTRY_TIMEOUT = 90; // 1 minute, 30 seconds

/** SmartCard manages bidirectional input/output to an ISO 7816 smart card. */
class SmartCard
{
public:
    enum class Protocol { UNDEFINED, T0, T1 }; // AUTO = T0 | T1

    using ptr = std::unique_ptr<SmartCard>;

    class TransactionGuard
    {
    public:
        TransactionGuard(const CardImpl& CardImpl, bool& inProgress);
        ~TransactionGuard();
        PCSC_CPP_DISABLE_COPY_MOVE(TransactionGuard);

    private:
        const CardImpl& card;
        bool& inProgress;
    };

    SmartCard(const ContextPtr& context, const string_t& readerName, byte_vector atr);
    SmartCard(); // Null object constructor.
    ~SmartCard();
    PCSC_CPP_DISABLE_COPY_MOVE(SmartCard);

    TransactionGuard beginTransaction();
    ResponseApdu transmit(const CommandApdu& command) const;
    ResponseApdu transmitCTL(const CommandApdu& command, uint16_t lang, uint8_t minlen) const;
    bool readerHasPinPad() const;

    Protocol protocol() const { return _protocol; }
    const byte_vector& atr() const { return _atr; }

private:
    CardImplPtr card;
    byte_vector _atr;
    Protocol _protocol = Protocol::UNDEFINED;
    bool transactionInProgress = false;
};

/** Reader provides card reader information, status and gives access to the smart card in it. */
class Reader
{
public:
    enum class Status {
        UNAWARE,
        IGNORE,
        CHANGED,
        UNKNOWN,
        UNAVAILABLE,
        EMPTY,
        PRESENT,
        ATRMATCH,
        EXCLUSIVE,
        INUSE,
        MUTE,
        UNPOWERED,
        _
    };

    Reader(ContextPtr context, string_t name, byte_vector cardAtr, flag_set<Status> status);

    SmartCard::ptr connectToCard() const { return std::make_unique<SmartCard>(ctx, name, cardAtr); }

    bool isCardInserted() const { return status[Status::PRESENT]; }

    std::string statusString() const;

    const string_t name;
    const byte_vector cardAtr;
    const flag_set<Status> status;

private:
    ContextPtr ctx;
};

/**
 * Access system smart card readers, entry point to the library.
 *
 * @throw ScardError, SystemError
 */
std::vector<Reader> listReaders();

// Utility functions.

extern const byte_vector APDU_RESPONSE_OK;

/** Convert bytes to hex string. */
std::string bytes2hexstr(const byte_vector& bytes);

/** Transmit APDU command and verify that expected response is received. */
void transmitApduWithExpectedResponse(const SmartCard& card, const CommandApdu& command,
                                      const byte_vector& expectedResponseBytes = APDU_RESPONSE_OK);

/** Read data length from currently selected file header, file must be ASN.1-encoded. */
size_t readDataLengthFromAsn1(const SmartCard& card);

/** Read lenght bytes from currently selected binary file in blockLength-sized chunks. */
byte_vector readBinary(const SmartCard& card, const size_t length, const size_t blockLength);

// Errors.

/** Base class for all pcsc-cpp errors. */
class Error : public std::runtime_error
{
public:
    using std::runtime_error::runtime_error;
};

/** Programming or system errors. */
class SystemError : public Error
{
public:
    using Error::Error;
};

/** Base class for all SCard API errors. */
class ScardError : public Error
{
public:
    using Error::Error;
};

/** Thrown when the PC/SC service is not running. */
class ScardServiceNotRunningError : public ScardError
{
public:
    using ScardError::ScardError;
};

/** Thrown when no card readers are connected to the system. */
class ScardNoReadersError : public ScardError
{
public:
    using ScardError::ScardError;
};

/** Thrown when no card is connected to the selected reader. */
class ScardNoCardError : public ScardError
{
public:
    using ScardError::ScardError;
};

/** Thrown when communication with the card or reader fails. */
class ScardCardCommunicationFailedError : public ScardError
{
public:
    using ScardError::ScardError;
};

/** Thrown when the card is removed from the selected reader. */
class ScardCardRemovedError : public ScardError
{
public:
    using ScardError::ScardError;
};

/** Thrown when the card transaction fails. */
class ScardTransactionFailedError : public ScardError
{
public:
    using ScardError::ScardError;
};

} // namespace pcsc_cpp
