package grpc_server

import (
	"crypto/tls"
	"net"
	"os"

	"fmt"
	"reflect"

	"errors"

	"github.com/tedsuo/ifrit"
	"google.golang.org/grpc"
	"google.golang.org/grpc/credentials"
)

type grpcServerRunner struct {
	listenAddress   string
	handler         interface{}
	serverRegistrar interface{}
	tlsConfig       *tls.Config
}

// NewGRPCServer returns an ifrit.Runner for your GRPC server process, given artifacts normally generated from a
// protobuf service definition by protoc.
//
// tlsConfig is optional.  If nil the server will run insecure.
//
// handler must be an implementation of the interface generated by protoc.
//
// serverRegistrar must be the "RegisterXXXServer" method generated by protoc.
//
// Type checking occurs at runtime.  Poorly typed `handler` or `serverRegistrar` parameters will result in an error
// when the Runner is Invoked.
func NewGRPCServer(listenAddress string, tlsConfig *tls.Config, handler, serverRegistrar interface{}) ifrit.Runner {
	return &grpcServerRunner{
		listenAddress:   listenAddress,
		handler:         handler,
		serverRegistrar: serverRegistrar,
		tlsConfig:       tlsConfig,
	}
}

func (s *grpcServerRunner) Validate() error {
	if s.serverRegistrar == nil || s.handler == nil {
		return errors.New("NewGRPCServer: `serverRegistrar` and `handler` must be non nil")
	}

	vServerRegistrar := reflect.ValueOf(s.serverRegistrar)
	vHandler := reflect.ValueOf(s.handler)

	registrarType := vServerRegistrar.Type()
	handlerType := vHandler.Type()

	// registrar type must be `func(*grpc.Server, X)`
	if registrarType.Kind() != reflect.Func {
		return fmt.Errorf("NewGRPCServer: `serverRegistrar` should be %s but is %s",
			reflect.Func, registrarType.Kind())
	}
	if registrarType.NumIn() != 2 {
		return fmt.Errorf("NewGRPCServer: `serverRegistrar` should have 2 parameters but it has %d parameters",
			registrarType.NumIn())
	}
	if registrarType.NumOut() != 0 {
		return fmt.Errorf("NewGRPCServer: `serverRegistrar` should return no value but it returns %d values",
			registrarType.NumOut())
	}

	// registrar's first parameter type must be a grpc server
	if reflect.TypeOf((*grpc.Server)(nil)) != registrarType.In(0) {
		return fmt.Errorf("NewGRPCServer: type of `serverRegistrar`'s first parameter must be `*grpc.Server` but is %s",
			registrarType.In(0))
	}

	// registrar's second parameter type must be implemented by handler type.
	if registrarType.In(1).Kind() != reflect.Interface || !handlerType.Implements(registrarType.In(1)) {
		return fmt.Errorf("NewGRPCServer: type of `serverRegistrar`'s second parameter %s is not implemented by `handler` type %s",
			registrarType.In(1), handlerType)
	}
	return nil
}

func (s *grpcServerRunner) Run(signals <-chan os.Signal, ready chan<- struct{}) error {
	err := s.Validate()
	if err != nil {
		return err
	}

	vServerRegistrar := reflect.ValueOf(s.serverRegistrar)
	vHandler := reflect.ValueOf(s.handler)

	lis, err := net.Listen("tcp", s.listenAddress)
	if err != nil {
		return err
	}

	opts := []grpc.ServerOption{}
	if s.tlsConfig != nil {
		opts = append(opts, grpc.Creds(credentials.NewTLS(s.tlsConfig)))
	}
	server := grpc.NewServer(opts...)
	args := []reflect.Value{reflect.ValueOf(server), vHandler}
	vServerRegistrar.Call(args)

	errCh := make(chan error)
	go func() {
		errCh <- server.Serve(lis)
	}()

	close(ready)

	select {
	case <-signals:
	case err = <-errCh:
	}

	server.GracefulStop()
	return err
}
