// Copyright 2025 Google LLC.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

// Code generated file. DO NOT EDIT.

// Package drive provides access to the Google Drive API.
//
// For product documentation, see: https://developers.google.com/workspace/drive/
//
// # Library status
//
// These client libraries are officially supported by Google. However, this
// library is considered complete and is in maintenance mode. This means
// that we will address critical bugs and security issues but will not add
// any new features.
//
// When possible, we recommend using our newer
// [Cloud Client Libraries for Go](https://pkg.go.dev/cloud.google.com/go)
// that are still actively being worked and iterated on.
//
// # Creating a client
//
// Usage example:
//
//	import "google.golang.org/api/drive/v2"
//	...
//	ctx := context.Background()
//	driveService, err := drive.NewService(ctx)
//
// In this example, Google Application Default Credentials are used for
// authentication. For information on how to create and obtain Application
// Default Credentials, see https://developers.google.com/identity/protocols/application-default-credentials.
//
// # Other authentication options
//
// By default, all available scopes (see "Constants") are used to authenticate.
// To restrict scopes, use [google.golang.org/api/option.WithScopes]:
//
//	driveService, err := drive.NewService(ctx, option.WithScopes(drive.DriveScriptsScope))
//
// To use an API key for authentication (note: some APIs do not support API
// keys), use [google.golang.org/api/option.WithAPIKey]:
//
//	driveService, err := drive.NewService(ctx, option.WithAPIKey("AIza..."))
//
// To use an OAuth token (e.g., a user token obtained via a three-legged OAuth
// flow, use [google.golang.org/api/option.WithTokenSource]:
//
//	config := &oauth2.Config{...}
//	// ...
//	token, err := config.Exchange(ctx, ...)
//	driveService, err := drive.NewService(ctx, option.WithTokenSource(config.TokenSource(ctx, token)))
//
// See [google.golang.org/api/option.ClientOption] for details on options.
package drive // import "google.golang.org/api/drive/v2"

import (
	"bytes"
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"log/slog"
	"net/http"
	"net/url"
	"strconv"
	"strings"

	"github.com/googleapis/gax-go/v2/internallog"
	googleapi "google.golang.org/api/googleapi"
	internal "google.golang.org/api/internal"
	gensupport "google.golang.org/api/internal/gensupport"
	option "google.golang.org/api/option"
	internaloption "google.golang.org/api/option/internaloption"
	htransport "google.golang.org/api/transport/http"
)

// Always reference these packages, just in case the auto-generated code
// below doesn't.
var _ = bytes.NewBuffer
var _ = strconv.Itoa
var _ = fmt.Sprintf
var _ = json.NewDecoder
var _ = io.Copy
var _ = url.Parse
var _ = gensupport.MarshalJSON
var _ = googleapi.Version
var _ = errors.New
var _ = strings.Replace
var _ = context.Canceled
var _ = internaloption.WithDefaultEndpoint
var _ = internal.Version
var _ = internallog.New

const apiId = "drive:v2"
const apiName = "drive"
const apiVersion = "v2"
const basePath = "https://www.googleapis.com/drive/v2/"
const basePathTemplate = "https://www.UNIVERSE_DOMAIN/drive/v2/"
const mtlsBasePath = "https://www.mtls.googleapis.com/drive/v2/"

// OAuth2 scopes used by this API.
const (
	// See, edit, create, and delete all of your Google Drive files
	DriveScope = "https://www.googleapis.com/auth/drive"

	// See, create, and delete its own configuration data in your Google Drive
	DriveAppdataScope = "https://www.googleapis.com/auth/drive.appdata"

	// View your Google Drive apps
	DriveAppsReadonlyScope = "https://www.googleapis.com/auth/drive.apps.readonly"

	// See, edit, create, and delete only the specific Google Drive files you use
	// with this app
	DriveFileScope = "https://www.googleapis.com/auth/drive.file"

	// See and download your Google Drive files that were created or edited by
	// Google Meet.
	DriveMeetReadonlyScope = "https://www.googleapis.com/auth/drive.meet.readonly"

	// View and manage metadata of files in your Google Drive
	DriveMetadataScope = "https://www.googleapis.com/auth/drive.metadata"

	// See information about your Google Drive files
	DriveMetadataReadonlyScope = "https://www.googleapis.com/auth/drive.metadata.readonly"

	// View the photos, videos and albums in your Google Photos
	DrivePhotosReadonlyScope = "https://www.googleapis.com/auth/drive.photos.readonly"

	// See and download all your Google Drive files
	DriveReadonlyScope = "https://www.googleapis.com/auth/drive.readonly"

	// Modify your Google Apps Script scripts' behavior
	DriveScriptsScope = "https://www.googleapis.com/auth/drive.scripts"
)

// NewService creates a new Service.
func NewService(ctx context.Context, opts ...option.ClientOption) (*Service, error) {
	scopesOption := internaloption.WithDefaultScopes(
		"https://www.googleapis.com/auth/drive",
		"https://www.googleapis.com/auth/drive.appdata",
		"https://www.googleapis.com/auth/drive.apps.readonly",
		"https://www.googleapis.com/auth/drive.file",
		"https://www.googleapis.com/auth/drive.meet.readonly",
		"https://www.googleapis.com/auth/drive.metadata",
		"https://www.googleapis.com/auth/drive.metadata.readonly",
		"https://www.googleapis.com/auth/drive.photos.readonly",
		"https://www.googleapis.com/auth/drive.readonly",
		"https://www.googleapis.com/auth/drive.scripts",
	)
	// NOTE: prepend, so we don't override user-specified scopes.
	opts = append([]option.ClientOption{scopesOption}, opts...)
	opts = append(opts, internaloption.WithDefaultEndpoint(basePath))
	opts = append(opts, internaloption.WithDefaultEndpointTemplate(basePathTemplate))
	opts = append(opts, internaloption.WithDefaultMTLSEndpoint(mtlsBasePath))
	opts = append(opts, internaloption.EnableNewAuthLibrary())
	client, endpoint, err := htransport.NewClient(ctx, opts...)
	if err != nil {
		return nil, err
	}
	s := &Service{client: client, BasePath: basePath, logger: internaloption.GetLogger(opts)}
	s.About = NewAboutService(s)
	s.Apps = NewAppsService(s)
	s.Changes = NewChangesService(s)
	s.Channels = NewChannelsService(s)
	s.Children = NewChildrenService(s)
	s.Comments = NewCommentsService(s)
	s.Drives = NewDrivesService(s)
	s.Files = NewFilesService(s)
	s.Parents = NewParentsService(s)
	s.Permissions = NewPermissionsService(s)
	s.Properties = NewPropertiesService(s)
	s.Replies = NewRepliesService(s)
	s.Revisions = NewRevisionsService(s)
	s.Teamdrives = NewTeamdrivesService(s)
	if endpoint != "" {
		s.BasePath = endpoint
	}
	return s, nil
}

// New creates a new Service. It uses the provided http.Client for requests.
//
// Deprecated: please use NewService instead.
// To provide a custom HTTP client, use option.WithHTTPClient.
// If you are using google.golang.org/api/googleapis/transport.APIKey, use option.WithAPIKey with NewService instead.
func New(client *http.Client) (*Service, error) {
	if client == nil {
		return nil, errors.New("client is nil")
	}
	return NewService(context.TODO(), option.WithHTTPClient(client))
}

type Service struct {
	client    *http.Client
	logger    *slog.Logger
	BasePath  string // API endpoint base URL
	UserAgent string // optional additional User-Agent fragment

	About *AboutService

	Apps *AppsService

	Changes *ChangesService

	Channels *ChannelsService

	Children *ChildrenService

	Comments *CommentsService

	Drives *DrivesService

	Files *FilesService

	Parents *ParentsService

	Permissions *PermissionsService

	Properties *PropertiesService

	Replies *RepliesService

	Revisions *RevisionsService

	Teamdrives *TeamdrivesService
}

func (s *Service) userAgent() string {
	if s.UserAgent == "" {
		return googleapi.UserAgent
	}
	return googleapi.UserAgent + " " + s.UserAgent
}

func NewAboutService(s *Service) *AboutService {
	rs := &AboutService{s: s}
	return rs
}

type AboutService struct {
	s *Service
}

func NewAppsService(s *Service) *AppsService {
	rs := &AppsService{s: s}
	return rs
}

type AppsService struct {
	s *Service
}

func NewChangesService(s *Service) *ChangesService {
	rs := &ChangesService{s: s}
	return rs
}

type ChangesService struct {
	s *Service
}

func NewChannelsService(s *Service) *ChannelsService {
	rs := &ChannelsService{s: s}
	return rs
}

type ChannelsService struct {
	s *Service
}

func NewChildrenService(s *Service) *ChildrenService {
	rs := &ChildrenService{s: s}
	return rs
}

type ChildrenService struct {
	s *Service
}

func NewCommentsService(s *Service) *CommentsService {
	rs := &CommentsService{s: s}
	return rs
}

type CommentsService struct {
	s *Service
}

func NewDrivesService(s *Service) *DrivesService {
	rs := &DrivesService{s: s}
	return rs
}

type DrivesService struct {
	s *Service
}

func NewFilesService(s *Service) *FilesService {
	rs := &FilesService{s: s}
	return rs
}

type FilesService struct {
	s *Service
}

func NewParentsService(s *Service) *ParentsService {
	rs := &ParentsService{s: s}
	return rs
}

type ParentsService struct {
	s *Service
}

func NewPermissionsService(s *Service) *PermissionsService {
	rs := &PermissionsService{s: s}
	return rs
}

type PermissionsService struct {
	s *Service
}

func NewPropertiesService(s *Service) *PropertiesService {
	rs := &PropertiesService{s: s}
	return rs
}

type PropertiesService struct {
	s *Service
}

func NewRepliesService(s *Service) *RepliesService {
	rs := &RepliesService{s: s}
	return rs
}

type RepliesService struct {
	s *Service
}

func NewRevisionsService(s *Service) *RevisionsService {
	rs := &RevisionsService{s: s}
	return rs
}

type RevisionsService struct {
	s *Service
}

func NewTeamdrivesService(s *Service) *TeamdrivesService {
	rs := &TeamdrivesService{s: s}
	return rs
}

type TeamdrivesService struct {
	s *Service
}

// About: An item with user information and settings.
type About struct {
	// AdditionalRoleInfo: Information about supported additional roles per file
	// type. The most specific type takes precedence.
	AdditionalRoleInfo []*AboutAdditionalRoleInfo `json:"additionalRoleInfo,omitempty"`
	// CanCreateDrives: Whether the user can create shared drives.
	CanCreateDrives bool `json:"canCreateDrives,omitempty"`
	// CanCreateTeamDrives: Deprecated: Use `canCreateDrives` instead.
	CanCreateTeamDrives bool `json:"canCreateTeamDrives,omitempty"`
	// DomainSharingPolicy: The domain sharing policy for the current user.
	// Possible values are: * `allowed` * `allowedWithWarning` * `incomingOnly` *
	// `disallowed`
	DomainSharingPolicy string `json:"domainSharingPolicy,omitempty"`
	// DriveThemes: A list of themes that are supported for shared drives.
	DriveThemes []*AboutDriveThemes `json:"driveThemes,omitempty"`
	// Etag: The ETag of the item.
	Etag string `json:"etag,omitempty"`
	// ExportFormats: The allowable export formats.
	ExportFormats []*AboutExportFormats `json:"exportFormats,omitempty"`
	// Features: List of additional features enabled on this account.
	Features []*AboutFeatures `json:"features,omitempty"`
	// FolderColorPalette: The palette of allowable folder colors as RGB hex
	// strings.
	FolderColorPalette []string `json:"folderColorPalette,omitempty"`
	// ImportFormats: The allowable import formats.
	ImportFormats []*AboutImportFormats `json:"importFormats,omitempty"`
	// IsCurrentAppInstalled: A boolean indicating whether the authenticated app is
	// installed by the authenticated user.
	IsCurrentAppInstalled bool `json:"isCurrentAppInstalled,omitempty"`
	// Kind: This is always `drive#about`.
	Kind string `json:"kind,omitempty"`
	// LanguageCode: The user's language or locale code, as defined by BCP 47, with
	// some extensions from Unicode's LDML format
	// (http://www.unicode.org/reports/tr35/).
	LanguageCode string `json:"languageCode,omitempty"`
	// LargestChangeId: The largest change id.
	LargestChangeId int64 `json:"largestChangeId,omitempty,string"`
	// MaxUploadSizes: List of max upload sizes for each file type. The most
	// specific type takes precedence.
	MaxUploadSizes []*AboutMaxUploadSizes `json:"maxUploadSizes,omitempty"`
	// Name: The name of the current user.
	Name string `json:"name,omitempty"`
	// PermissionId: The current user's ID as visible in the permissions
	// collection.
	PermissionId string `json:"permissionId,omitempty"`
	// QuotaBytesByService: The amount of storage quota used by different Google
	// services.
	QuotaBytesByService []*AboutQuotaBytesByService `json:"quotaBytesByService,omitempty"`
	// QuotaBytesTotal: The total number of quota bytes. This is only relevant when
	// quotaType is LIMITED.
	QuotaBytesTotal int64 `json:"quotaBytesTotal,omitempty,string"`
	// QuotaBytesUsed: The number of quota bytes used by Google Drive.
	QuotaBytesUsed int64 `json:"quotaBytesUsed,omitempty,string"`
	// QuotaBytesUsedAggregate: The number of quota bytes used by all Google apps
	// (Drive, Picasa, etc.).
	QuotaBytesUsedAggregate int64 `json:"quotaBytesUsedAggregate,omitempty,string"`
	// QuotaBytesUsedInTrash: The number of quota bytes used by trashed items.
	QuotaBytesUsedInTrash int64 `json:"quotaBytesUsedInTrash,omitempty,string"`
	// QuotaType: The type of the user's storage quota. Possible values are: *
	// `LIMITED` * `UNLIMITED`
	QuotaType string `json:"quotaType,omitempty"`
	// RemainingChangeIds: The number of remaining change ids, limited to no more
	// than 2500.
	RemainingChangeIds int64 `json:"remainingChangeIds,omitempty,string"`
	// RootFolderId: The id of the root folder.
	RootFolderId string `json:"rootFolderId,omitempty"`
	// SelfLink: A link back to this item.
	SelfLink string `json:"selfLink,omitempty"`
	// TeamDriveThemes: Deprecated: Use `driveThemes` instead.
	TeamDriveThemes []*AboutTeamDriveThemes `json:"teamDriveThemes,omitempty"`
	// User: The authenticated user.
	User *User `json:"user,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "AdditionalRoleInfo") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AdditionalRoleInfo") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s About) MarshalJSON() ([]byte, error) {
	type NoMethod About
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type AboutAdditionalRoleInfo struct {
	// RoleSets: The supported additional roles per primary role.
	RoleSets []*AboutAdditionalRoleInfoRoleSets `json:"roleSets,omitempty"`
	// Type: The content type that this additional role info applies to.
	Type string `json:"type,omitempty"`
	// ForceSendFields is a list of field names (e.g. "RoleSets") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "RoleSets") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AboutAdditionalRoleInfo) MarshalJSON() ([]byte, error) {
	type NoMethod AboutAdditionalRoleInfo
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type AboutAdditionalRoleInfoRoleSets struct {
	// AdditionalRoles: The supported additional roles with the primary role.
	AdditionalRoles []string `json:"additionalRoles,omitempty"`
	// PrimaryRole: A primary permission role.
	PrimaryRole string `json:"primaryRole,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AdditionalRoles") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AdditionalRoles") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AboutAdditionalRoleInfoRoleSets) MarshalJSON() ([]byte, error) {
	type NoMethod AboutAdditionalRoleInfoRoleSets
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type AboutDriveThemes struct {
	// BackgroundImageLink: A link to this theme's background image.
	BackgroundImageLink string `json:"backgroundImageLink,omitempty"`
	// ColorRgb: The color of this theme as an RGB hex string.
	ColorRgb string `json:"colorRgb,omitempty"`
	// Id: The ID of the theme.
	Id string `json:"id,omitempty"`
	// ForceSendFields is a list of field names (e.g. "BackgroundImageLink") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "BackgroundImageLink") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AboutDriveThemes) MarshalJSON() ([]byte, error) {
	type NoMethod AboutDriveThemes
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type AboutExportFormats struct {
	// Source: The content type to convert from.
	Source string `json:"source,omitempty"`
	// Targets: The possible content types to convert to.
	Targets []string `json:"targets,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Source") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Source") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AboutExportFormats) MarshalJSON() ([]byte, error) {
	type NoMethod AboutExportFormats
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type AboutFeatures struct {
	// FeatureName: The name of the feature.
	FeatureName string `json:"featureName,omitempty"`
	// FeatureRate: The request limit rate for this feature, in queries per second.
	FeatureRate float64 `json:"featureRate,omitempty"`
	// ForceSendFields is a list of field names (e.g. "FeatureName") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "FeatureName") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AboutFeatures) MarshalJSON() ([]byte, error) {
	type NoMethod AboutFeatures
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *AboutFeatures) UnmarshalJSON(data []byte) error {
	type NoMethod AboutFeatures
	var s1 struct {
		FeatureRate gensupport.JSONFloat64 `json:"featureRate"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.FeatureRate = float64(s1.FeatureRate)
	return nil
}

type AboutImportFormats struct {
	// Source: The imported file's content type to convert from.
	Source string `json:"source,omitempty"`
	// Targets: The possible content types to convert to.
	Targets []string `json:"targets,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Source") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Source") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AboutImportFormats) MarshalJSON() ([]byte, error) {
	type NoMethod AboutImportFormats
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type AboutMaxUploadSizes struct {
	// Size: The max upload size for this type.
	Size int64 `json:"size,omitempty,string"`
	// Type: The file type.
	Type string `json:"type,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Size") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Size") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AboutMaxUploadSizes) MarshalJSON() ([]byte, error) {
	type NoMethod AboutMaxUploadSizes
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type AboutQuotaBytesByService struct {
	// BytesUsed: The storage quota bytes used by the service.
	BytesUsed int64 `json:"bytesUsed,omitempty,string"`
	// ServiceName: The service's name, e.g. DRIVE, GMAIL, or PHOTOS.
	ServiceName string `json:"serviceName,omitempty"`
	// ForceSendFields is a list of field names (e.g. "BytesUsed") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "BytesUsed") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AboutQuotaBytesByService) MarshalJSON() ([]byte, error) {
	type NoMethod AboutQuotaBytesByService
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type AboutTeamDriveThemes struct {
	// BackgroundImageLink: Deprecated: Use `driveThemes/backgroundImageLink`
	// instead.
	BackgroundImageLink string `json:"backgroundImageLink,omitempty"`
	// ColorRgb: Deprecated: Use `driveThemes/colorRgb` instead.
	ColorRgb string `json:"colorRgb,omitempty"`
	// Id: Deprecated: Use `driveThemes/id` instead.
	Id string `json:"id,omitempty"`
	// ForceSendFields is a list of field names (e.g. "BackgroundImageLink") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "BackgroundImageLink") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AboutTeamDriveThemes) MarshalJSON() ([]byte, error) {
	type NoMethod AboutTeamDriveThemes
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// App: The apps resource provides a list of the apps that a user has
// installed, with information about each app's supported MIME types, file
// extensions, and other details. Some resource methods (such as `apps.get`)
// require an `appId`. Use the `apps.list` method to retrieve the ID for an
// installed application.
type App struct {
	// Authorized: Whether the app is authorized to access data on the user's
	// Drive.
	Authorized bool `json:"authorized,omitempty"`
	// CreateInFolderTemplate: The template url to create a new file with this app
	// in a given folder. The template will contain {folderId} to be replaced by
	// the folder to create the new file in.
	CreateInFolderTemplate string `json:"createInFolderTemplate,omitempty"`
	// CreateUrl: The url to create a new file with this app.
	CreateUrl string `json:"createUrl,omitempty"`
	// HasDriveWideScope: Whether the app has drive-wide scope. An app with
	// drive-wide scope can access all files in the user's drive.
	HasDriveWideScope bool `json:"hasDriveWideScope,omitempty"`
	// Icons: The various icons for the app.
	Icons []*AppIcons `json:"icons,omitempty"`
	// Id: The ID of the app.
	Id string `json:"id,omitempty"`
	// Installed: Whether the app is installed.
	Installed bool `json:"installed,omitempty"`
	// Kind: This is always `drive#app`.
	Kind string `json:"kind,omitempty"`
	// LongDescription: A long description of the app.
	LongDescription string `json:"longDescription,omitempty"`
	// Name: The name of the app.
	Name string `json:"name,omitempty"`
	// ObjectType: The type of object this app creates (e.g. Chart). If empty, the
	// app name should be used instead.
	ObjectType string `json:"objectType,omitempty"`
	// OpenUrlTemplate: The template url for opening files with this app. The
	// template will contain `{ids}` and/or `{exportIds}` to be replaced by the
	// actual file ids. See Open Files for the full documentation.
	OpenUrlTemplate string `json:"openUrlTemplate,omitempty"`
	// PrimaryFileExtensions: The list of primary file extensions.
	PrimaryFileExtensions []string `json:"primaryFileExtensions,omitempty"`
	// PrimaryMimeTypes: The list of primary mime types.
	PrimaryMimeTypes []string `json:"primaryMimeTypes,omitempty"`
	// ProductId: The ID of the product listing for this app.
	ProductId string `json:"productId,omitempty"`
	// ProductUrl: A link to the product listing for this app.
	ProductUrl string `json:"productUrl,omitempty"`
	// SecondaryFileExtensions: The list of secondary file extensions.
	SecondaryFileExtensions []string `json:"secondaryFileExtensions,omitempty"`
	// SecondaryMimeTypes: The list of secondary mime types.
	SecondaryMimeTypes []string `json:"secondaryMimeTypes,omitempty"`
	// ShortDescription: A short description of the app.
	ShortDescription string `json:"shortDescription,omitempty"`
	// SupportsCreate: Whether this app supports creating new objects.
	SupportsCreate bool `json:"supportsCreate,omitempty"`
	// SupportsImport: Whether this app supports importing from Docs Editors.
	SupportsImport bool `json:"supportsImport,omitempty"`
	// SupportsMultiOpen: Whether this app supports opening more than one file.
	SupportsMultiOpen bool `json:"supportsMultiOpen,omitempty"`
	// SupportsOfflineCreate: Whether this app supports creating new files when
	// offline.
	SupportsOfflineCreate bool `json:"supportsOfflineCreate,omitempty"`
	// UseByDefault: Whether the app is selected as the default handler for the
	// types it supports.
	UseByDefault bool `json:"useByDefault,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Authorized") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Authorized") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s App) MarshalJSON() ([]byte, error) {
	type NoMethod App
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type AppIcons struct {
	// Category: Category of the icon. Allowed values are: * `application` - icon
	// for the application * `document` - icon for a file associated with the app *
	// `documentShared` - icon for a shared file associated with the app
	Category string `json:"category,omitempty"`
	// IconUrl: URL for the icon.
	IconUrl string `json:"iconUrl,omitempty"`
	// Size: Size of the icon. Represented as the maximum of the width and height.
	Size int64 `json:"size,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Category") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Category") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AppIcons) MarshalJSON() ([]byte, error) {
	type NoMethod AppIcons
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// AppList: A list of third-party applications which the user has installed or
// given access to Google Drive.
type AppList struct {
	// DefaultAppIds: List of app IDs that the user has specified to use by
	// default. The list is in reverse-priority order (lowest to highest).
	DefaultAppIds []string `json:"defaultAppIds,omitempty"`
	// Etag: The ETag of the list.
	Etag string `json:"etag,omitempty"`
	// Items: The list of apps.
	Items []*App `json:"items,omitempty"`
	// Kind: This is always `drive#appList`.
	Kind string `json:"kind,omitempty"`
	// SelfLink: A link back to this list.
	SelfLink string `json:"selfLink,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "DefaultAppIds") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "DefaultAppIds") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AppList) MarshalJSON() ([]byte, error) {
	type NoMethod AppList
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Change: Representation of a change to a file or shared drive.
type Change struct {
	// ChangeType: The type of the change. Possible values are `file` and `drive`.
	ChangeType string `json:"changeType,omitempty"`
	// Deleted: Whether the file or shared drive has been removed from this list of
	// changes, for example by deletion or loss of access.
	Deleted bool `json:"deleted,omitempty"`
	// Drive: The updated state of the shared drive. Present if the changeType is
	// drive, the user is still a member of the shared drive, and the shared drive
	// has not been deleted.
	Drive *Drive `json:"drive,omitempty"`
	// DriveId: The ID of the shared drive associated with this change.
	DriveId string `json:"driveId,omitempty"`
	// File: The updated state of the file. Present if the type is file and the
	// file has not been removed from this list of changes.
	File *File `json:"file,omitempty"`
	// FileId: The ID of the file associated with this change.
	FileId string `json:"fileId,omitempty"`
	// Id: The ID of the change.
	Id int64 `json:"id,omitempty,string"`
	// Kind: This is always `drive#change`.
	Kind string `json:"kind,omitempty"`
	// ModificationDate: The time of this modification.
	ModificationDate string `json:"modificationDate,omitempty"`
	// SelfLink: A link back to this change.
	SelfLink string `json:"selfLink,omitempty"`
	// TeamDrive: Deprecated: Use `drive` instead.
	TeamDrive *TeamDrive `json:"teamDrive,omitempty"`
	// TeamDriveId: Deprecated: Use `driveId` instead.
	TeamDriveId string `json:"teamDriveId,omitempty"`
	// Type: Deprecated: Use `changeType` instead.
	Type string `json:"type,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "ChangeType") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ChangeType") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Change) MarshalJSON() ([]byte, error) {
	type NoMethod Change
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ChangeList: A list of changes for a user.
type ChangeList struct {
	// Etag: The ETag of the list.
	Etag string `json:"etag,omitempty"`
	// Items: The list of changes. If nextPageToken is populated, then this list
	// may be incomplete and an additional page of results should be fetched.
	Items []*Change `json:"items,omitempty"`
	// Kind: This is always `drive#changeList`.
	Kind string `json:"kind,omitempty"`
	// LargestChangeId: The current largest change ID.
	LargestChangeId int64 `json:"largestChangeId,omitempty,string"`
	// NewStartPageToken: The starting page token for future changes. This will be
	// present only if the end of the current changes list has been reached.
	NewStartPageToken string `json:"newStartPageToken,omitempty"`
	// NextLink: A link to the next page of changes.
	NextLink string `json:"nextLink,omitempty"`
	// NextPageToken: The page token for the next page of changes. This will be
	// absent if the end of the changes list has been reached. If the token is
	// rejected for any reason, it should be discarded, and pagination should be
	// restarted from the first page of results.
	NextPageToken string `json:"nextPageToken,omitempty"`
	// SelfLink: A link back to this list.
	SelfLink string `json:"selfLink,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Etag") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Etag") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ChangeList) MarshalJSON() ([]byte, error) {
	type NoMethod ChangeList
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Channel: A notification channel used to watch for resource changes.
type Channel struct {
	// Address: The address where notifications are delivered for this channel.
	Address string `json:"address,omitempty"`
	// Expiration: Date and time of notification channel expiration, expressed as a
	// Unix timestamp, in milliseconds. Optional.
	Expiration int64 `json:"expiration,omitempty,string"`
	// Id: A UUID or similar unique string that identifies this channel.
	Id string `json:"id,omitempty"`
	// Kind: Identifies this as a notification channel used to watch for changes to
	// a resource, which is `api#channel`.
	Kind string `json:"kind,omitempty"`
	// Params: Additional parameters controlling delivery channel behavior.
	// Optional.
	Params map[string]string `json:"params,omitempty"`
	// Payload: A Boolean value to indicate whether payload is wanted. Optional.
	Payload bool `json:"payload,omitempty"`
	// ResourceId: An opaque ID that identifies the resource being watched on this
	// channel. Stable across different API versions.
	ResourceId string `json:"resourceId,omitempty"`
	// ResourceUri: A version-specific identifier for the watched resource.
	ResourceUri string `json:"resourceUri,omitempty"`
	// Token: An arbitrary string delivered to the target address with each
	// notification delivered over this channel. Optional.
	Token string `json:"token,omitempty"`
	// Type: The type of delivery mechanism used for this channel. Valid values are
	// "web_hook" or "webhook".
	Type string `json:"type,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Address") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Address") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Channel) MarshalJSON() ([]byte, error) {
	type NoMethod Channel
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ChildList: A list of children of a file.
type ChildList struct {
	// Etag: The ETag of the list.
	Etag string `json:"etag,omitempty"`
	// Items: The list of children. If nextPageToken is populated, then this list
	// may be incomplete and an additional page of results should be fetched.
	Items []*ChildReference `json:"items,omitempty"`
	// Kind: This is always `drive#childList`.
	Kind string `json:"kind,omitempty"`
	// NextLink: A link to the next page of children.
	NextLink string `json:"nextLink,omitempty"`
	// NextPageToken: The page token for the next page of children. This will be
	// absent if the end of the children list has been reached. If the token is
	// rejected for any reason, it should be discarded, and pagination should be
	// restarted from the first page of results.
	NextPageToken string `json:"nextPageToken,omitempty"`
	// SelfLink: A link back to this list.
	SelfLink string `json:"selfLink,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Etag") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Etag") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ChildList) MarshalJSON() ([]byte, error) {
	type NoMethod ChildList
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ChildReference: A reference to a folder's child. Some resource methods (such
// as `children.get`) require a `childId`. Use the `children.list` method to
// retrieve the ID of the child.
type ChildReference struct {
	// ChildLink: Output only. A link to the child.
	ChildLink string `json:"childLink,omitempty"`
	// Id: The ID of the child.
	Id string `json:"id,omitempty"`
	// Kind: Output only. This is always `drive#childReference`.
	Kind string `json:"kind,omitempty"`
	// SelfLink: Output only. A link back to this reference.
	SelfLink string `json:"selfLink,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "ChildLink") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ChildLink") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ChildReference) MarshalJSON() ([]byte, error) {
	type NoMethod ChildReference
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Comment: A comment on a file in Google Drive. Some resource methods (such as
// `comments.update`) require a `commentId`. Use the `comments.list` method to
// retrieve the ID for a comment in a file.
type Comment struct {
	// Anchor: A region of the document represented as a JSON string. For details
	// on defining anchor properties, refer to Add comments and replies
	// (https://developers.google.com/workspace/drive/api/v2/manage-comments).
	Anchor string `json:"anchor,omitempty"`
	// Author: Output only. The author of the comment. The author's email address
	// and permission ID will not be populated.
	Author *User `json:"author,omitempty"`
	// CommentId: Output only. The ID of the comment.
	CommentId string `json:"commentId,omitempty"`
	// Content: The plain text content used to create this comment. This is not
	// HTML safe and should only be used as a starting point to make edits to a
	// comment's content.
	Content string `json:"content,omitempty"`
	// Context: The context of the file which is being commented on.
	Context *CommentContext `json:"context,omitempty"`
	// CreatedDate: The date when this comment was first created.
	CreatedDate string `json:"createdDate,omitempty"`
	// Deleted: Output only. Whether this comment has been deleted. If a comment
	// has been deleted the content will be cleared and this will only represent a
	// comment that once existed.
	Deleted bool `json:"deleted,omitempty"`
	// FileId: Output only. The file which this comment is addressing.
	FileId string `json:"fileId,omitempty"`
	// FileTitle: Output only. The title of the file which this comment is
	// addressing.
	FileTitle string `json:"fileTitle,omitempty"`
	// HtmlContent: Output only. HTML formatted content for this comment.
	HtmlContent string `json:"htmlContent,omitempty"`
	// Kind: Output only. This is always `drive#comment`.
	Kind string `json:"kind,omitempty"`
	// ModifiedDate: The date when this comment or any of its replies were last
	// modified.
	ModifiedDate string `json:"modifiedDate,omitempty"`
	// Replies: Output only. Replies to this post.
	Replies []*CommentReply `json:"replies,omitempty"`
	// SelfLink: Output only. A link back to this comment.
	SelfLink string `json:"selfLink,omitempty"`
	// Status: Output only. The status of this comment. Status can be changed by
	// posting a reply to a comment with the desired status. * `open` - The comment
	// is still open. * `resolved` - The comment has been resolved by one of its
	// replies.
	Status string `json:"status,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Anchor") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Anchor") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Comment) MarshalJSON() ([]byte, error) {
	type NoMethod Comment
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// CommentContext: The context of the file which is being commented on.
type CommentContext struct {
	// Type: The MIME type of the context snippet.
	Type string `json:"type,omitempty"`
	// Value: Data representation of the segment of the file being commented on. In
	// the case of a text file for example, this would be the actual text that the
	// comment is about.
	Value string `json:"value,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Type") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Type") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s CommentContext) MarshalJSON() ([]byte, error) {
	type NoMethod CommentContext
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// CommentList: A list of comments on a file in Google Drive.
type CommentList struct {
	// Items: The list of comments. If nextPageToken is populated, then this list
	// may be incomplete and an additional page of results should be fetched.
	Items []*Comment `json:"items,omitempty"`
	// Kind: This is always `drive#commentList`.
	Kind string `json:"kind,omitempty"`
	// NextLink: A link to the next page of comments.
	NextLink string `json:"nextLink,omitempty"`
	// NextPageToken: The page token for the next page of comments. This will be
	// absent if the end of the comments list has been reached. If the token is
	// rejected for any reason, it should be discarded, and pagination should be
	// restarted from the first page of results.
	NextPageToken string `json:"nextPageToken,omitempty"`
	// SelfLink: A link back to this list.
	SelfLink string `json:"selfLink,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Items") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Items") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s CommentList) MarshalJSON() ([]byte, error) {
	type NoMethod CommentList
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// CommentReply: A comment on a file in Google Drive. Some resource methods
// (such as `replies.update`) require a `replyId`. Use the `replies.list`
// method to retrieve the ID for a reply.
type CommentReply struct {
	// Author: Output only. The author of the reply. The author's email address and
	// permission ID will not be populated.
	Author *User `json:"author,omitempty"`
	// Content: The plain text content used to create this reply. This is not HTML
	// safe and should only be used as a starting point to make edits to a reply's
	// content. This field is required on inserts if no verb is specified
	// (resolve/reopen).
	Content string `json:"content,omitempty"`
	// CreatedDate: The date when this reply was first created.
	CreatedDate string `json:"createdDate,omitempty"`
	// Deleted: Output only. Whether this reply has been deleted. If a reply has
	// been deleted the content will be cleared and this will only represent a
	// reply that once existed.
	Deleted bool `json:"deleted,omitempty"`
	// HtmlContent: Output only. HTML formatted content for this reply.
	HtmlContent string `json:"htmlContent,omitempty"`
	// Kind: Output only. This is always `drive#commentReply`.
	Kind string `json:"kind,omitempty"`
	// ModifiedDate: The date when this reply was last modified.
	ModifiedDate string `json:"modifiedDate,omitempty"`
	// ReplyId: Output only. The ID of the reply.
	ReplyId string `json:"replyId,omitempty"`
	// Verb: The action this reply performed to the parent comment. When creating a
	// new reply this is the action to be perform to the parent comment. Possible
	// values are: * `resolve` - To resolve a comment. * `reopen` - To reopen
	// (un-resolve) a comment.
	Verb string `json:"verb,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Author") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Author") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s CommentReply) MarshalJSON() ([]byte, error) {
	type NoMethod CommentReply
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// CommentReplyList: A list of replies to a comment on a file in Google Drive.
type CommentReplyList struct {
	// Items: The list of replies. If nextPageToken is populated, then this list
	// may be incomplete and an additional page of results should be fetched.
	Items []*CommentReply `json:"items,omitempty"`
	// Kind: This is always `drive#commentReplyList`.
	Kind string `json:"kind,omitempty"`
	// NextLink: A link to the next page of replies.
	NextLink string `json:"nextLink,omitempty"`
	// NextPageToken: The page token for the next page of replies. This will be
	// absent if the end of the replies list has been reached. If the token is
	// rejected for any reason, it should be discarded, and pagination should be
	// restarted from the first page of results.
	NextPageToken string `json:"nextPageToken,omitempty"`
	// SelfLink: A link back to this list.
	SelfLink string `json:"selfLink,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Items") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Items") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s CommentReplyList) MarshalJSON() ([]byte, error) {
	type NoMethod CommentReplyList
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ContentRestriction: A restriction for accessing the content of the file.
type ContentRestriction struct {
	// OwnerRestricted: Whether the content restriction can only be modified or
	// removed by a user who owns the file. For files in shared drives, any user
	// with `organizer` capabilities can modify or remove this content restriction.
	OwnerRestricted bool `json:"ownerRestricted,omitempty"`
	// ReadOnly: Whether the content of the file is read-only. If a file is
	// read-only, a new revision of the file may not be added, comments may not be
	// added or modified, and the title of the file may not be modified.
	ReadOnly bool `json:"readOnly,omitempty"`
	// Reason: Reason for why the content of the file is restricted. This is only
	// mutable on requests that also set `readOnly=true`.
	Reason string `json:"reason,omitempty"`
	// RestrictingUser: Output only. The user who set the content restriction. Only
	// populated if `readOnly` is true.
	RestrictingUser *User `json:"restrictingUser,omitempty"`
	// RestrictionDate: The time at which the content restriction was set
	// (formatted RFC 3339 timestamp). Only populated if readOnly is true.
	RestrictionDate string `json:"restrictionDate,omitempty"`
	// SystemRestricted: Output only. Whether the content restriction was applied
	// by the system, for example due to an esignature. Users cannot modify or
	// remove system restricted content restrictions.
	SystemRestricted bool `json:"systemRestricted,omitempty"`
	// Type: Output only. The type of the content restriction. Currently the only
	// possible value is `globalContentRestriction`.
	Type string `json:"type,omitempty"`
	// ForceSendFields is a list of field names (e.g. "OwnerRestricted") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "OwnerRestricted") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ContentRestriction) MarshalJSON() ([]byte, error) {
	type NoMethod ContentRestriction
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Drive: Representation of a shared drive. Some resource methods (such as
// `drives.update`) require a `driveId`. Use the `drives.list` method to
// retrieve the ID for a shared drive.
type Drive struct {
	// BackgroundImageFile: An image file and cropping parameters from which a
	// background image for this shared drive is set. This is a write only field;
	// it can only be set on `drive.drives.update` requests that don't set
	// `themeId`. When specified, all fields of the `backgroundImageFile` must be
	// set.
	BackgroundImageFile *DriveBackgroundImageFile `json:"backgroundImageFile,omitempty"`
	// BackgroundImageLink: Output only. A short-lived link to this shared drive's
	// background image.
	BackgroundImageLink string `json:"backgroundImageLink,omitempty"`
	// Capabilities: Output only. Capabilities the current user has on this shared
	// drive.
	Capabilities *DriveCapabilities `json:"capabilities,omitempty"`
	// ColorRgb: The color of this shared drive as an RGB hex string. It can only
	// be set on a `drive.drives.update` request that does not set `themeId`.
	ColorRgb string `json:"colorRgb,omitempty"`
	// CreatedDate: The time at which the shared drive was created (RFC 3339
	// date-time).
	CreatedDate string `json:"createdDate,omitempty"`
	// Hidden: Whether the shared drive is hidden from default view.
	Hidden bool `json:"hidden,omitempty"`
	// Id: Output only. The ID of this shared drive which is also the ID of the top
	// level folder of this shared drive.
	Id string `json:"id,omitempty"`
	// Kind: Output only. This is always `drive#drive`
	Kind string `json:"kind,omitempty"`
	// Name: The name of this shared drive.
	Name string `json:"name,omitempty"`
	// OrgUnitId: Output only. The organizational unit of this shared drive. This
	// field is only populated on `drives.list` responses when the
	// `useDomainAdminAccess` parameter is set to `true`.
	OrgUnitId string `json:"orgUnitId,omitempty"`
	// Restrictions: A set of restrictions that apply to this shared drive or items
	// inside this shared drive.
	Restrictions *DriveRestrictions `json:"restrictions,omitempty"`
	// ThemeId: The ID of the theme from which the background image and color will
	// be set. The set of possible `driveThemes` can be retrieved from a
	// `drive.about.get` response. When not specified on a `drive.drives.insert`
	// request, a random theme is chosen from which the background image and color
	// are set. This is a write-only field; it can only be set on requests that
	// don't set `colorRgb` or `backgroundImageFile`.
	ThemeId string `json:"themeId,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "BackgroundImageFile") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "BackgroundImageFile") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Drive) MarshalJSON() ([]byte, error) {
	type NoMethod Drive
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// DriveBackgroundImageFile: An image file and cropping parameters from which a
// background image for this shared drive is set. This is a write only field;
// it can only be set on `drive.drives.update` requests that don't set
// `themeId`. When specified, all fields of the `backgroundImageFile` must be
// set.
type DriveBackgroundImageFile struct {
	// Id: The ID of an image file in Google Drive to use for the background image.
	Id string `json:"id,omitempty"`
	// Width: The width of the cropped image in the closed range of 0 to 1. This
	// value represents the width of the cropped image divided by the width of the
	// entire image. The height is computed by applying a width to height aspect
	// ratio of 80 to 9. The resulting image must be at least 1280 pixels wide and
	// 144 pixels high.
	Width float64 `json:"width,omitempty"`
	// XCoordinate: The X coordinate of the upper left corner of the cropping area
	// in the background image. This is a value in the closed range of 0 to 1. This
	// value represents the horizontal distance from the left side of the entire
	// image to the left side of the cropping area divided by the width of the
	// entire image.
	XCoordinate float64 `json:"xCoordinate,omitempty"`
	// YCoordinate: The Y coordinate of the upper left corner of the cropping area
	// in the background image. This is a value in the closed range of 0 to 1. This
	// value represents the vertical distance from the top side of the entire image
	// to the top side of the cropping area divided by the height of the entire
	// image.
	YCoordinate float64 `json:"yCoordinate,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Id") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Id") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s DriveBackgroundImageFile) MarshalJSON() ([]byte, error) {
	type NoMethod DriveBackgroundImageFile
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *DriveBackgroundImageFile) UnmarshalJSON(data []byte) error {
	type NoMethod DriveBackgroundImageFile
	var s1 struct {
		Width       gensupport.JSONFloat64 `json:"width"`
		XCoordinate gensupport.JSONFloat64 `json:"xCoordinate"`
		YCoordinate gensupport.JSONFloat64 `json:"yCoordinate"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.Width = float64(s1.Width)
	s.XCoordinate = float64(s1.XCoordinate)
	s.YCoordinate = float64(s1.YCoordinate)
	return nil
}

// DriveCapabilities: Output only. Capabilities the current user has on this
// shared drive.
type DriveCapabilities struct {
	// CanAddChildren: Output only. Whether the current user can add children to
	// folders in this shared drive.
	CanAddChildren bool `json:"canAddChildren,omitempty"`
	// CanChangeCopyRequiresWriterPermissionRestriction: Output only. Whether the
	// current user can change the `copyRequiresWriterPermission` restriction of
	// this shared drive.
	CanChangeCopyRequiresWriterPermissionRestriction bool `json:"canChangeCopyRequiresWriterPermissionRestriction,omitempty"`
	// CanChangeDomainUsersOnlyRestriction: Output only. Whether the current user
	// can change the `domainUsersOnly` restriction of this shared drive.
	CanChangeDomainUsersOnlyRestriction bool `json:"canChangeDomainUsersOnlyRestriction,omitempty"`
	// CanChangeDriveBackground: Output only. Whether the current user can change
	// the background of this shared drive.
	CanChangeDriveBackground bool `json:"canChangeDriveBackground,omitempty"`
	// CanChangeDriveMembersOnlyRestriction: Output only. Whether the current user
	// can change the `driveMembersOnly` restriction of this shared drive.
	CanChangeDriveMembersOnlyRestriction bool `json:"canChangeDriveMembersOnlyRestriction,omitempty"`
	// CanChangeSharingFoldersRequiresOrganizerPermissionRestriction: Output only.
	// Whether the current user can change the
	// `sharingFoldersRequiresOrganizerPermission` restriction of this shared
	// drive.
	CanChangeSharingFoldersRequiresOrganizerPermissionRestriction bool `json:"canChangeSharingFoldersRequiresOrganizerPermissionRestriction,omitempty"`
	// CanComment: Output only. Whether the current user can comment on files in
	// this shared drive.
	CanComment bool `json:"canComment,omitempty"`
	// CanCopy: Output only. Whether the current user can copy files in this shared
	// drive.
	CanCopy bool `json:"canCopy,omitempty"`
	// CanDeleteChildren: Output only. Whether the current user can delete children
	// from folders in this shared drive.
	CanDeleteChildren bool `json:"canDeleteChildren,omitempty"`
	// CanDeleteDrive: Output only. Whether the current user can delete this shared
	// drive. Attempting to delete the shared drive may still fail if there are
	// untrashed items inside the shared drive.
	CanDeleteDrive bool `json:"canDeleteDrive,omitempty"`
	// CanDownload: Output only. Whether the current user can download files in
	// this shared drive.
	CanDownload bool `json:"canDownload,omitempty"`
	// CanEdit: Output only. Whether the current user can edit files in this shared
	// drive
	CanEdit bool `json:"canEdit,omitempty"`
	// CanListChildren: Output only. Whether the current user can list the children
	// of folders in this shared drive.
	CanListChildren bool `json:"canListChildren,omitempty"`
	// CanManageMembers: Output only. Whether the current user can add members to
	// this shared drive or remove them or change their role.
	CanManageMembers bool `json:"canManageMembers,omitempty"`
	// CanReadRevisions: Output only. Whether the current user can read the
	// revisions resource of files in this shared drive.
	CanReadRevisions bool `json:"canReadRevisions,omitempty"`
	// CanRename: Output only. Whether the current user can rename files or folders
	// in this shared drive.
	CanRename bool `json:"canRename,omitempty"`
	// CanRenameDrive: Output only. Whether the current user can rename this shared
	// drive.
	CanRenameDrive bool `json:"canRenameDrive,omitempty"`
	// CanResetDriveRestrictions: Output only. Whether the current user can reset
	// the shared drive restrictions to defaults.
	CanResetDriveRestrictions bool `json:"canResetDriveRestrictions,omitempty"`
	// CanShare: Output only. Whether the current user can share files or folders
	// in this shared drive.
	CanShare bool `json:"canShare,omitempty"`
	// CanTrashChildren: Output only. Whether the current user can trash children
	// from folders in this shared drive.
	CanTrashChildren bool `json:"canTrashChildren,omitempty"`
	// ForceSendFields is a list of field names (e.g. "CanAddChildren") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CanAddChildren") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s DriveCapabilities) MarshalJSON() ([]byte, error) {
	type NoMethod DriveCapabilities
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// DriveRestrictions: A set of restrictions that apply to this shared drive or
// items inside this shared drive.
type DriveRestrictions struct {
	// AdminManagedRestrictions: Whether administrative privileges on this shared
	// drive are required to modify restrictions.
	AdminManagedRestrictions bool `json:"adminManagedRestrictions,omitempty"`
	// CopyRequiresWriterPermission: Whether the options to copy, print, or
	// download files inside this shared drive, should be disabled for readers and
	// commenters. When this restriction is set to `true`, it will override the
	// similarly named field to `true` for any file inside this shared drive.
	CopyRequiresWriterPermission bool `json:"copyRequiresWriterPermission,omitempty"`
	// DomainUsersOnly: Whether access to this shared drive and items inside this
	// shared drive is restricted to users of the domain to which this shared drive
	// belongs. This restriction may be overridden by other sharing policies
	// controlled outside of this shared drive.
	DomainUsersOnly bool `json:"domainUsersOnly,omitempty"`
	// DriveMembersOnly: Whether access to items inside this shared drive is
	// restricted to its members.
	DriveMembersOnly bool `json:"driveMembersOnly,omitempty"`
	// SharingFoldersRequiresOrganizerPermission: If true, only users with the
	// organizer role can share folders. If false, users with either the organizer
	// role or the file organizer role can share folders.
	SharingFoldersRequiresOrganizerPermission bool `json:"sharingFoldersRequiresOrganizerPermission,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AdminManagedRestrictions")
	// to unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AdminManagedRestrictions") to
	// include in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s DriveRestrictions) MarshalJSON() ([]byte, error) {
	type NoMethod DriveRestrictions
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// DriveList: A list of shared drives.
type DriveList struct {
	// Items: The list of shared drives. If nextPageToken is populated, then this
	// list may be incomplete and an additional page of results should be fetched.
	Items []*Drive `json:"items,omitempty"`
	// Kind: This is always `drive#driveList`
	Kind string `json:"kind,omitempty"`
	// NextPageToken: The page token for the next page of shared drives. This will
	// be absent if the end of the list has been reached. If the token is rejected
	// for any reason, it should be discarded, and pagination should be restarted
	// from the first page of results.
	NextPageToken string `json:"nextPageToken,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Items") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Items") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s DriveList) MarshalJSON() ([]byte, error) {
	type NoMethod DriveList
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// File: The metadata for a file. Some resource methods (such as
// `files.update`) require a `fileId`. Use the `files.list` method to retrieve
// the ID for a file.
type File struct {
	// AlternateLink: Output only. A link for opening the file in a relevant Google
	// editor or viewer.
	AlternateLink string `json:"alternateLink,omitempty"`
	// AppDataContents: Output only. Whether this file is in the Application Data
	// folder.
	AppDataContents bool `json:"appDataContents,omitempty"`
	// CanComment: Output only. Deprecated: Use `capabilities/canComment` instead.
	CanComment bool `json:"canComment,omitempty"`
	// CanReadRevisions: Output only. Deprecated: Use
	// `capabilities/canReadRevisions` instead.
	CanReadRevisions bool `json:"canReadRevisions,omitempty"`
	// Capabilities: Output only. Capabilities the current user has on this file.
	// Each capability corresponds to a fine-grained action that a user may take.
	Capabilities *FileCapabilities `json:"capabilities,omitempty"`
	// ContentRestrictions: Restrictions for accessing the content of the file.
	// Only populated if such a restriction exists.
	ContentRestrictions []*ContentRestriction `json:"contentRestrictions,omitempty"`
	// CopyRequiresWriterPermission: Whether the options to copy, print, or
	// download this file, should be disabled for readers and commenters.
	CopyRequiresWriterPermission bool `json:"copyRequiresWriterPermission,omitempty"`
	// Copyable: Output only. Deprecated: Use `capabilities/canCopy` instead.
	Copyable bool `json:"copyable,omitempty"`
	// CreatedDate: Create time for this file (formatted RFC 3339 timestamp).
	CreatedDate string `json:"createdDate,omitempty"`
	// DefaultOpenWithLink: Output only. A link to open this file with the user's
	// default app for this file. Only populated when the drive.apps.readonly scope
	// is used.
	DefaultOpenWithLink string `json:"defaultOpenWithLink,omitempty"`
	// Description: A short description of the file.
	Description string `json:"description,omitempty"`
	// DownloadUrl: Output only. Short lived download URL for the file. This field
	// is only populated for files with content stored in Google Drive; it is not
	// populated for Google Docs or shortcut files.
	DownloadUrl string `json:"downloadUrl,omitempty"`
	// DriveId: Output only. ID of the shared drive the file resides in. Only
	// populated for items in shared drives.
	DriveId string `json:"driveId,omitempty"`
	// Editable: Output only. Deprecated: Use `capabilities/canEdit` instead.
	Editable bool `json:"editable,omitempty"`
	// EmbedLink: Output only. A link for embedding the file.
	EmbedLink string `json:"embedLink,omitempty"`
	// Etag: Output only. ETag of the file.
	Etag string `json:"etag,omitempty"`
	// ExplicitlyTrashed: Output only. Whether this file has been explicitly
	// trashed, as opposed to recursively trashed.
	ExplicitlyTrashed bool `json:"explicitlyTrashed,omitempty"`
	// ExportLinks: Output only. Links for exporting Docs Editors files to specific
	// formats.
	ExportLinks map[string]string `json:"exportLinks,omitempty"`
	// FileExtension: Output only. The final component of `fullFileExtension` with
	// trailing text that does not appear to be part of the extension removed. This
	// field is only populated for files with content stored in Google Drive; it is
	// not populated for Docs Editors or shortcut files.
	FileExtension string `json:"fileExtension,omitempty"`
	// FileSize: Output only. Size in bytes of blobs and first party editor files.
	// Won't be populated for files that have no size, like shortcuts and folders.
	FileSize int64 `json:"fileSize,omitempty,string"`
	// FolderColorRgb: Folder color as an RGB hex string if the file is a folder or
	// a shortcut to a folder. The list of supported colors is available in the
	// folderColorPalette field of the About resource. If an unsupported color is
	// specified, it will be changed to the closest color in the palette.
	FolderColorRgb string `json:"folderColorRgb,omitempty"`
	// FullFileExtension: Output only. The full file extension; extracted from the
	// title. May contain multiple concatenated extensions, such as "tar.gz".
	// Removing an extension from the title does not clear this field; however,
	// changing the extension on the title does update this field. This field is
	// only populated for files with content stored in Google Drive; it is not
	// populated for Docs Editors or shortcut files.
	FullFileExtension string `json:"fullFileExtension,omitempty"`
	// HasAugmentedPermissions: Output only. Whether there are permissions directly
	// on this file. This field is only populated for items in shared drives.
	HasAugmentedPermissions bool `json:"hasAugmentedPermissions,omitempty"`
	// HasThumbnail: Output only. Whether this file has a thumbnail. This does not
	// indicate whether the requesting app has access to the thumbnail. To check
	// access, look for the presence of the thumbnailLink field.
	HasThumbnail bool `json:"hasThumbnail,omitempty"`
	// HeadRevisionId: Output only. The ID of the file's head revision. This field
	// is only populated for files with content stored in Google Drive; it is not
	// populated for Docs Editors or shortcut files.
	HeadRevisionId string `json:"headRevisionId,omitempty"`
	// IconLink: Output only. A link to the file's icon.
	IconLink string `json:"iconLink,omitempty"`
	// Id: The ID of the file.
	Id string `json:"id,omitempty"`
	// ImageMediaMetadata: Output only. Metadata about image media. This will only
	// be present for image types, and its contents will depend on what can be
	// parsed from the image content.
	ImageMediaMetadata *FileImageMediaMetadata `json:"imageMediaMetadata,omitempty"`
	// IndexableText: Indexable text attributes for the file (can only be written)
	IndexableText *FileIndexableText `json:"indexableText,omitempty"`
	// InheritedPermissionsDisabled: Whether this file has inherited permissions
	// disabled. Inherited permissions are enabled by default.
	InheritedPermissionsDisabled bool `json:"inheritedPermissionsDisabled,omitempty"`
	// IsAppAuthorized: Output only. Whether the file was created or opened by the
	// requesting app.
	IsAppAuthorized bool `json:"isAppAuthorized,omitempty"`
	// Kind: Output only. The type of file. This is always `drive#file`.
	Kind string `json:"kind,omitempty"`
	// LabelInfo: Output only. An overview of the labels on the file.
	LabelInfo *FileLabelInfo `json:"labelInfo,omitempty"`
	// Labels: A group of labels for the file.
	Labels *FileLabels `json:"labels,omitempty"`
	// LastModifyingUser: Output only. The last user to modify this file. This
	// field is only populated when the last modification was performed by a
	// signed-in user.
	LastModifyingUser *User `json:"lastModifyingUser,omitempty"`
	// LastModifyingUserName: Output only. Name of the last user to modify this
	// file.
	LastModifyingUserName string `json:"lastModifyingUserName,omitempty"`
	// LastViewedByMeDate: Last time this file was viewed by the user (formatted
	// RFC 3339 timestamp).
	LastViewedByMeDate string `json:"lastViewedByMeDate,omitempty"`
	// LinkShareMetadata: Contains details about the link URLs that clients are
	// using to refer to this item.
	LinkShareMetadata *FileLinkShareMetadata `json:"linkShareMetadata,omitempty"`
	// MarkedViewedByMeDate: Deprecated.
	MarkedViewedByMeDate string `json:"markedViewedByMeDate,omitempty"`
	// Md5Checksum: Output only. An MD5 checksum for the content of this file. This
	// field is only populated for files with content stored in Google Drive; it is
	// not populated for Docs Editors or shortcut files.
	Md5Checksum string `json:"md5Checksum,omitempty"`
	// MimeType: The MIME type of the file. This is only mutable on update when
	// uploading new content. This field can be left blank, and the mimetype will
	// be determined from the uploaded content's MIME type.
	MimeType string `json:"mimeType,omitempty"`
	// ModifiedByMeDate: Last time this file was modified by the user (formatted
	// RFC 3339 timestamp). Note that setting modifiedDate will also update the
	// modifiedByMe date for the user which set the date.
	ModifiedByMeDate string `json:"modifiedByMeDate,omitempty"`
	// ModifiedDate: Last time this file was modified by anyone (formatted RFC 3339
	// timestamp). This is only mutable on update when the setModifiedDate
	// parameter is set.
	ModifiedDate string `json:"modifiedDate,omitempty"`
	// OpenWithLinks: Output only. A map of the id of each of the user's apps to a
	// link to open this file with that app. Only populated when the
	// drive.apps.readonly scope is used.
	OpenWithLinks map[string]string `json:"openWithLinks,omitempty"`
	// OriginalFilename: The original filename of the uploaded content if
	// available, or else the original value of the `title` field. This is only
	// available for files with binary content in Google Drive.
	OriginalFilename string `json:"originalFilename,omitempty"`
	// OwnedByMe: Output only. Whether the file is owned by the current user. Not
	// populated for items in shared drives.
	OwnedByMe bool `json:"ownedByMe,omitempty"`
	// OwnerNames: Output only. Name(s) of the owner(s) of this file. Not populated
	// for items in shared drives.
	OwnerNames []string `json:"ownerNames,omitempty"`
	// Owners: Output only. The owner of this file. Only certain legacy files may
	// have more than one owner. This field isn't populated for items in shared
	// drives.
	Owners []*User `json:"owners,omitempty"`
	// Parents: The ID of the parent folder containing the file. A file can only
	// have one parent folder; specifying multiple parents isn't supported. If not
	// specified as part of an insert request, the file is placed directly in the
	// user's My Drive folder. If not specified as part of a copy request, the file
	// inherits any discoverable parent of the source file. Update requests must
	// use the `addParents` and `removeParents` parameters to modify the parents
	// list.
	Parents []*ParentReference `json:"parents,omitempty"`
	// PermissionIds: Output only. List of permission IDs for users with access to
	// this file.
	PermissionIds []string `json:"permissionIds,omitempty"`
	// Permissions: Output only. The list of permissions for users with access to
	// this file. Not populated for items in shared drives.
	Permissions []*Permission `json:"permissions,omitempty"`
	// Properties: The list of properties.
	Properties []*Property `json:"properties,omitempty"`
	// QuotaBytesUsed: Output only. The number of quota bytes used by this file.
	QuotaBytesUsed int64 `json:"quotaBytesUsed,omitempty,string"`
	// ResourceKey: Output only. A key needed to access the item via a shared link.
	ResourceKey string `json:"resourceKey,omitempty"`
	// SelfLink: Output only. A link back to this file.
	SelfLink string `json:"selfLink,omitempty"`
	// Sha1Checksum: Output only. The SHA1 checksum associated with this file, if
	// available. This field is only populated for files with content stored in
	// Google Drive; it is not populated for Docs Editors or shortcut files.
	Sha1Checksum string `json:"sha1Checksum,omitempty"`
	// Sha256Checksum: Output only. The SHA256 checksum associated with this file,
	// if available. This field is only populated for files with content stored in
	// Google Drive; it is not populated for Docs Editors or shortcut files.
	Sha256Checksum string `json:"sha256Checksum,omitempty"`
	// Shareable: Output only. Deprecated: Use `capabilities/canShare` instead.
	Shareable bool `json:"shareable,omitempty"`
	// Shared: Output only. Whether the file has been shared. Not populated for
	// items in shared drives.
	Shared bool `json:"shared,omitempty"`
	// SharedWithMeDate: Time at which this file was shared with the user
	// (formatted RFC 3339 timestamp).
	SharedWithMeDate string `json:"sharedWithMeDate,omitempty"`
	// SharingUser: Output only. User that shared the item with the current user,
	// if available.
	SharingUser *User `json:"sharingUser,omitempty"`
	// ShortcutDetails: Shortcut file details. Only populated for shortcut files,
	// which have the mimeType field set to `application/vnd.google-apps.shortcut`.
	// Can only be set on `files.insert` requests.
	ShortcutDetails *FileShortcutDetails `json:"shortcutDetails,omitempty"`
	// Spaces: Output only. The list of spaces which contain the file. Supported
	// values are `drive`, `appDataFolder` and `photos`.
	Spaces []string `json:"spaces,omitempty"`
	// TeamDriveId: Output only. Deprecated: Use `driveId` instead.
	TeamDriveId string `json:"teamDriveId,omitempty"`
	// Thumbnail: A thumbnail for the file. This will only be used if a standard
	// thumbnail cannot be generated.
	Thumbnail *FileThumbnail `json:"thumbnail,omitempty"`
	// ThumbnailLink: Output only. A short-lived link to the file's thumbnail, if
	// available. Typically lasts on the order of hours. Not intended for direct
	// usage on web applications due to Cross-Origin Resource Sharing (CORS)
	// (https://developer.mozilla.org/en-US/docs/Web/HTTP/CORS), consider using a
	// proxy server. Only populated when the requesting app can access the file's
	// content. If the file isn't shared publicly, the URL returned in
	// `Files.thumbnailLink` must be fetched using a credentialed request.
	ThumbnailLink string `json:"thumbnailLink,omitempty"`
	// ThumbnailVersion: Output only. The thumbnail version for use in thumbnail
	// cache invalidation.
	ThumbnailVersion int64 `json:"thumbnailVersion,omitempty,string"`
	// Title: The title of this file. Note that for immutable items such as the top
	// level folders of shared drives, My Drive root folder, and Application Data
	// folder the title is constant.
	Title string `json:"title,omitempty"`
	// TrashedDate: The time that the item was trashed (formatted RFC 3339
	// timestamp). Only populated for items in shared drives.
	TrashedDate string `json:"trashedDate,omitempty"`
	// TrashingUser: Output only. If the file has been explicitly trashed, the user
	// who trashed it. Only populated for items in shared drives.
	TrashingUser *User `json:"trashingUser,omitempty"`
	// UserPermission: Output only. The permissions for the authenticated user on
	// this file.
	UserPermission *Permission `json:"userPermission,omitempty"`
	// Version: Output only. A monotonically increasing version number for the
	// file. This reflects every change made to the file on the server, even those
	// not visible to the requesting user.
	Version int64 `json:"version,omitempty,string"`
	// VideoMediaMetadata: Output only. Metadata about video media. This will only
	// be present for video types.
	VideoMediaMetadata *FileVideoMediaMetadata `json:"videoMediaMetadata,omitempty"`
	// WebContentLink: Output only. A link for downloading the content of the file
	// in a browser using cookie based authentication. In cases where the content
	// is shared publicly, the content can be downloaded without any credentials.
	WebContentLink string `json:"webContentLink,omitempty"`
	// WebViewLink: Output only. A link only available on public folders for
	// viewing their static web assets (HTML, CSS, JS, etc) via Google Drive's
	// Website Hosting.
	WebViewLink string `json:"webViewLink,omitempty"`
	// WritersCanShare: Whether writers can share the document with other users.
	// Not populated for items in shared drives.
	WritersCanShare bool `json:"writersCanShare,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "AlternateLink") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AlternateLink") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s File) MarshalJSON() ([]byte, error) {
	type NoMethod File
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// FileCapabilities: Output only. Capabilities the current user has on this
// file. Each capability corresponds to a fine-grained action that a user may
// take.
type FileCapabilities struct {
	// CanAcceptOwnership: Output only. Whether the current user is the pending
	// owner of the file. Not populated for shared drive files.
	CanAcceptOwnership bool `json:"canAcceptOwnership,omitempty"`
	// CanAddChildren: Output only. Whether the current user can add children to
	// this folder. This is always false when the item is not a folder.
	CanAddChildren bool `json:"canAddChildren,omitempty"`
	// CanAddFolderFromAnotherDrive: Output only. Whether the current user can add
	// a folder from another drive (different shared drive or My Drive) to this
	// folder. This is false when the item is not a folder. Only populated for
	// items in shared drives.
	CanAddFolderFromAnotherDrive bool `json:"canAddFolderFromAnotherDrive,omitempty"`
	// CanAddMyDriveParent: Output only. Whether the current user can add a parent
	// for the item without removing an existing parent in the same request. Not
	// populated for shared drive files.
	CanAddMyDriveParent bool `json:"canAddMyDriveParent,omitempty"`
	// CanChangeCopyRequiresWriterPermission: Output only. Whether the current user
	// can change the `copyRequiresWriterPermission` restriction of this file.
	CanChangeCopyRequiresWriterPermission bool `json:"canChangeCopyRequiresWriterPermission,omitempty"`
	// CanChangeRestrictedDownload: Output only. Deprecated.
	CanChangeRestrictedDownload bool `json:"canChangeRestrictedDownload,omitempty"`
	// CanChangeSecurityUpdateEnabled: Output only. Whether the current user can
	// change the securityUpdateEnabled field on link share metadata.
	CanChangeSecurityUpdateEnabled bool `json:"canChangeSecurityUpdateEnabled,omitempty"`
	// CanComment: Output only. Whether the current user can comment on this file.
	CanComment bool `json:"canComment,omitempty"`
	// CanCopy: Output only. Whether the current user can copy this file. For an
	// item in a shared drive, whether the current user can copy non-folder
	// descendants of this item, or this item itself if it is not a folder.
	CanCopy bool `json:"canCopy,omitempty"`
	// CanDelete: Output only. Whether the current user can delete this file.
	CanDelete bool `json:"canDelete,omitempty"`
	// CanDeleteChildren: Output only. Whether the current user can delete children
	// of this folder. This is false when the item is not a folder. Only populated
	// for items in shared drives.
	CanDeleteChildren bool `json:"canDeleteChildren,omitempty"`
	// CanDisableInheritedPermissions: Output only. Whether a user can disable
	// inherited permissions.
	CanDisableInheritedPermissions bool `json:"canDisableInheritedPermissions,omitempty"`
	// CanDownload: Output only. Whether the current user can download this file.
	CanDownload bool `json:"canDownload,omitempty"`
	// CanEdit: Output only. Whether the current user can edit this file. Other
	// factors may limit the type of changes a user can make to a file. For
	// example, see `canChangeCopyRequiresWriterPermission` or `canModifyContent`.
	CanEdit bool `json:"canEdit,omitempty"`
	// CanEnableInheritedPermissions: Output only. Whether a user can re-enable
	// inherited permissions.
	CanEnableInheritedPermissions bool `json:"canEnableInheritedPermissions,omitempty"`
	// CanListChildren: Output only. Whether the current user can list the children
	// of this folder. This is always false when the item is not a folder.
	CanListChildren bool `json:"canListChildren,omitempty"`
	// CanModifyContent: Output only. Whether the current user can modify the
	// content of this file.
	CanModifyContent bool `json:"canModifyContent,omitempty"`
	// CanModifyContentRestriction: Deprecated: Output only. Use one of
	// `canModifyEditorContentRestriction`, `canModifyOwnerContentRestriction` or
	// `canRemoveContentRestriction`.
	CanModifyContentRestriction bool `json:"canModifyContentRestriction,omitempty"`
	// CanModifyEditorContentRestriction: Output only. Whether the current user can
	// add or modify content restrictions on the file which are editor restricted.
	CanModifyEditorContentRestriction bool `json:"canModifyEditorContentRestriction,omitempty"`
	// CanModifyLabels: Output only. Whether the current user can modify the labels
	// on the file.
	CanModifyLabels bool `json:"canModifyLabels,omitempty"`
	// CanModifyOwnerContentRestriction: Output only. Whether the current user can
	// add or modify content restrictions which are owner restricted.
	CanModifyOwnerContentRestriction bool `json:"canModifyOwnerContentRestriction,omitempty"`
	// CanMoveChildrenOutOfDrive: Output only. Whether the current user can move
	// children of this folder outside of the shared drive. This is false when the
	// item is not a folder. Only populated for items in shared drives.
	CanMoveChildrenOutOfDrive bool `json:"canMoveChildrenOutOfDrive,omitempty"`
	// CanMoveChildrenOutOfTeamDrive: Output only. Deprecated: Use
	// `canMoveChildrenOutOfDrive` instead.
	CanMoveChildrenOutOfTeamDrive bool `json:"canMoveChildrenOutOfTeamDrive,omitempty"`
	// CanMoveChildrenWithinDrive: Output only. Whether the current user can move
	// children of this folder within this drive. This is false when the item is
	// not a folder. Note that a request to move the child may still fail depending
	// on the current user's access to the child and to the destination folder.
	CanMoveChildrenWithinDrive bool `json:"canMoveChildrenWithinDrive,omitempty"`
	// CanMoveChildrenWithinTeamDrive: Output only. Deprecated: Use
	// `canMoveChildrenWithinDrive` instead.
	CanMoveChildrenWithinTeamDrive bool `json:"canMoveChildrenWithinTeamDrive,omitempty"`
	// CanMoveItemIntoTeamDrive: Output only. Deprecated: Use
	// `canMoveItemOutOfDrive` instead.
	CanMoveItemIntoTeamDrive bool `json:"canMoveItemIntoTeamDrive,omitempty"`
	// CanMoveItemOutOfDrive: Output only. Whether the current user can move this
	// item outside of this drive by changing its parent. Note that a request to
	// change the parent of the item may still fail depending on the new parent
	// that is being added.
	CanMoveItemOutOfDrive bool `json:"canMoveItemOutOfDrive,omitempty"`
	// CanMoveItemOutOfTeamDrive: Output only. Deprecated: Use
	// `canMoveItemOutOfDrive` instead.
	CanMoveItemOutOfTeamDrive bool `json:"canMoveItemOutOfTeamDrive,omitempty"`
	// CanMoveItemWithinDrive: Output only. Whether the current user can move this
	// item within this drive. Note that a request to change the parent of the item
	// may still fail depending on the new parent that is being added and the
	// parent that is being removed.
	CanMoveItemWithinDrive bool `json:"canMoveItemWithinDrive,omitempty"`
	// CanMoveItemWithinTeamDrive: Output only. Deprecated: Use
	// `canMoveItemWithinDrive` instead.
	CanMoveItemWithinTeamDrive bool `json:"canMoveItemWithinTeamDrive,omitempty"`
	// CanMoveTeamDriveItem: Output only. Deprecated: Use `canMoveItemWithinDrive`
	// or `canMoveItemOutOfDrive` instead.
	CanMoveTeamDriveItem bool `json:"canMoveTeamDriveItem,omitempty"`
	// CanReadDrive: Output only. Whether the current user can read the shared
	// drive to which this file belongs. Only populated for items in shared drives.
	CanReadDrive bool `json:"canReadDrive,omitempty"`
	// CanReadLabels: Output only. Whether the current user can read the labels on
	// the file.
	CanReadLabels bool `json:"canReadLabels,omitempty"`
	// CanReadRevisions: Output only. Whether the current user can read the
	// revisions resource of this file. For a shared drive item, whether revisions
	// of non-folder descendants of this item, or this item itself if it is not a
	// folder, can be read.
	CanReadRevisions bool `json:"canReadRevisions,omitempty"`
	// CanReadTeamDrive: Output only. Deprecated: Use `canReadDrive` instead.
	CanReadTeamDrive bool `json:"canReadTeamDrive,omitempty"`
	// CanRemoveChildren: Output only. Whether the current user can remove children
	// from this folder. This is always false when the item is not a folder. For a
	// folder in a shared drive, use `canDeleteChildren` or `canTrashChildren`
	// instead.
	CanRemoveChildren bool `json:"canRemoveChildren,omitempty"`
	// CanRemoveContentRestriction: Output only. Whether there is a content
	// restriction on the file that can be removed by the current user.
	CanRemoveContentRestriction bool `json:"canRemoveContentRestriction,omitempty"`
	// CanRemoveMyDriveParent: Output only. Whether the current user can remove a
	// parent from the item without adding another parent in the same request. Not
	// populated for shared drive files.
	CanRemoveMyDriveParent bool `json:"canRemoveMyDriveParent,omitempty"`
	// CanRename: Output only. Whether the current user can rename this file.
	CanRename bool `json:"canRename,omitempty"`
	// CanShare: Output only. Whether the current user can modify the sharing
	// settings for this file.
	CanShare bool `json:"canShare,omitempty"`
	// CanTrash: Output only. Whether the current user can move this file to trash.
	CanTrash bool `json:"canTrash,omitempty"`
	// CanTrashChildren: Output only. Whether the current user can trash children
	// of this folder. This is false when the item is not a folder. Only populated
	// for items in shared drives.
	CanTrashChildren bool `json:"canTrashChildren,omitempty"`
	// CanUntrash: Output only. Whether the current user can restore this file from
	// trash.
	CanUntrash bool `json:"canUntrash,omitempty"`
	// ForceSendFields is a list of field names (e.g. "CanAcceptOwnership") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CanAcceptOwnership") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s FileCapabilities) MarshalJSON() ([]byte, error) {
	type NoMethod FileCapabilities
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// FileImageMediaMetadata: Output only. Metadata about image media. This will
// only be present for image types, and its contents will depend on what can be
// parsed from the image content.
type FileImageMediaMetadata struct {
	// Aperture: Output only. The aperture used to create the photo (f-number).
	Aperture float64 `json:"aperture,omitempty"`
	// CameraMake: Output only. The make of the camera used to create the photo.
	CameraMake string `json:"cameraMake,omitempty"`
	// CameraModel: Output only. The model of the camera used to create the photo.
	CameraModel string `json:"cameraModel,omitempty"`
	// ColorSpace: Output only. The color space of the photo.
	ColorSpace string `json:"colorSpace,omitempty"`
	// Date: Output only. The date and time the photo was taken (EXIF format
	// timestamp).
	Date string `json:"date,omitempty"`
	// ExposureBias: Output only. The exposure bias of the photo (APEX value).
	ExposureBias float64 `json:"exposureBias,omitempty"`
	// ExposureMode: Output only. The exposure mode used to create the photo.
	ExposureMode string `json:"exposureMode,omitempty"`
	// ExposureTime: Output only. The length of the exposure, in seconds.
	ExposureTime float64 `json:"exposureTime,omitempty"`
	// FlashUsed: Output only. Whether a flash was used to create the photo.
	FlashUsed bool `json:"flashUsed,omitempty"`
	// FocalLength: Output only. The focal length used to create the photo, in
	// millimeters.
	FocalLength float64 `json:"focalLength,omitempty"`
	// Height: Output only. The height of the image in pixels.
	Height int64 `json:"height,omitempty"`
	// IsoSpeed: Output only. The ISO speed used to create the photo.
	IsoSpeed int64 `json:"isoSpeed,omitempty"`
	// Lens: Output only. The lens used to create the photo.
	Lens string `json:"lens,omitempty"`
	// Location: Output only. Geographic location information stored in the image.
	Location *FileImageMediaMetadataLocation `json:"location,omitempty"`
	// MaxApertureValue: Output only. The smallest f-number of the lens at the
	// focal length used to create the photo (APEX value).
	MaxApertureValue float64 `json:"maxApertureValue,omitempty"`
	// MeteringMode: Output only. The metering mode used to create the photo.
	MeteringMode string `json:"meteringMode,omitempty"`
	// Rotation: Output only. The number of clockwise 90 degree rotations applied
	// from the image's original orientation.
	Rotation int64 `json:"rotation,omitempty"`
	// Sensor: Output only. The type of sensor used to create the photo.
	Sensor string `json:"sensor,omitempty"`
	// SubjectDistance: Output only. The distance to the subject of the photo, in
	// meters.
	SubjectDistance int64 `json:"subjectDistance,omitempty"`
	// WhiteBalance: Output only. The white balance mode used to create the photo.
	WhiteBalance string `json:"whiteBalance,omitempty"`
	// Width: Output only. The width of the image in pixels.
	Width int64 `json:"width,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Aperture") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Aperture") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s FileImageMediaMetadata) MarshalJSON() ([]byte, error) {
	type NoMethod FileImageMediaMetadata
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *FileImageMediaMetadata) UnmarshalJSON(data []byte) error {
	type NoMethod FileImageMediaMetadata
	var s1 struct {
		Aperture         gensupport.JSONFloat64 `json:"aperture"`
		ExposureBias     gensupport.JSONFloat64 `json:"exposureBias"`
		ExposureTime     gensupport.JSONFloat64 `json:"exposureTime"`
		FocalLength      gensupport.JSONFloat64 `json:"focalLength"`
		MaxApertureValue gensupport.JSONFloat64 `json:"maxApertureValue"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.Aperture = float64(s1.Aperture)
	s.ExposureBias = float64(s1.ExposureBias)
	s.ExposureTime = float64(s1.ExposureTime)
	s.FocalLength = float64(s1.FocalLength)
	s.MaxApertureValue = float64(s1.MaxApertureValue)
	return nil
}

// FileImageMediaMetadataLocation: Output only. Geographic location information
// stored in the image.
type FileImageMediaMetadataLocation struct {
	// Altitude: Output only. The altitude stored in the image.
	Altitude float64 `json:"altitude,omitempty"`
	// Latitude: Output only. The latitude stored in the image.
	Latitude float64 `json:"latitude,omitempty"`
	// Longitude: Output only. The longitude stored in the image.
	Longitude float64 `json:"longitude,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Altitude") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Altitude") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s FileImageMediaMetadataLocation) MarshalJSON() ([]byte, error) {
	type NoMethod FileImageMediaMetadataLocation
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *FileImageMediaMetadataLocation) UnmarshalJSON(data []byte) error {
	type NoMethod FileImageMediaMetadataLocation
	var s1 struct {
		Altitude  gensupport.JSONFloat64 `json:"altitude"`
		Latitude  gensupport.JSONFloat64 `json:"latitude"`
		Longitude gensupport.JSONFloat64 `json:"longitude"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.Altitude = float64(s1.Altitude)
	s.Latitude = float64(s1.Latitude)
	s.Longitude = float64(s1.Longitude)
	return nil
}

// FileIndexableText: Indexable text attributes for the file (can only be
// written)
type FileIndexableText struct {
	// Text: The text to be indexed for this file.
	Text string `json:"text,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Text") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Text") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s FileIndexableText) MarshalJSON() ([]byte, error) {
	type NoMethod FileIndexableText
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// FileLabelInfo: Output only. An overview of the labels on the file.
type FileLabelInfo struct {
	// Labels: Output only. The set of labels on the file as requested by the label
	// IDs in the `includeLabels` parameter. By default, no labels are returned.
	Labels []*Label `json:"labels,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Labels") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Labels") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s FileLabelInfo) MarshalJSON() ([]byte, error) {
	type NoMethod FileLabelInfo
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// FileLabels: A group of labels for the file.
type FileLabels struct {
	// Hidden: Output only. Deprecated.
	Hidden bool `json:"hidden,omitempty"`
	// Modified: Output only. Whether the file has been modified by this user.
	Modified bool `json:"modified,omitempty"`
	// Restricted: Output only. Deprecated: Use `copyRequiresWriterPermission`
	// instead.
	Restricted bool `json:"restricted,omitempty"`
	// Starred: Whether this file is starred by the user.
	Starred bool `json:"starred,omitempty"`
	// Trashed: Whether this file has been trashed. This label applies to all users
	// accessing the file; however, only owners are allowed to see and untrash
	// files.
	Trashed bool `json:"trashed,omitempty"`
	// Viewed: Whether this file has been viewed by this user.
	Viewed bool `json:"viewed,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Hidden") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Hidden") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s FileLabels) MarshalJSON() ([]byte, error) {
	type NoMethod FileLabels
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// FileLinkShareMetadata: Contains details about the link URLs that clients are
// using to refer to this item.
type FileLinkShareMetadata struct {
	// SecurityUpdateEligible: Output only. Whether the file is eligible for
	// security update.
	SecurityUpdateEligible bool `json:"securityUpdateEligible,omitempty"`
	// SecurityUpdateEnabled: Output only. Whether the security update is enabled
	// for this file.
	SecurityUpdateEnabled bool `json:"securityUpdateEnabled,omitempty"`
	// ForceSendFields is a list of field names (e.g. "SecurityUpdateEligible") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "SecurityUpdateEligible") to
	// include in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s FileLinkShareMetadata) MarshalJSON() ([]byte, error) {
	type NoMethod FileLinkShareMetadata
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// FileShortcutDetails: Shortcut file details. Only populated for shortcut
// files, which have the mimeType field set to
// `application/vnd.google-apps.shortcut`. Can only be set on `files.insert`
// requests.
type FileShortcutDetails struct {
	// TargetId: The ID of the file that this shortcut points to. Can only be set
	// on `files.insert` requests.
	TargetId string `json:"targetId,omitempty"`
	// TargetMimeType: Output only. The MIME type of the file that this shortcut
	// points to. The value of this field is a snapshot of the target's MIME type,
	// captured when the shortcut is created.
	TargetMimeType string `json:"targetMimeType,omitempty"`
	// TargetResourceKey: Output only. The ResourceKey for the target file.
	TargetResourceKey string `json:"targetResourceKey,omitempty"`
	// ForceSendFields is a list of field names (e.g. "TargetId") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "TargetId") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s FileShortcutDetails) MarshalJSON() ([]byte, error) {
	type NoMethod FileShortcutDetails
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// FileThumbnail: A thumbnail for the file. This will only be used if a
// standard thumbnail cannot be generated.
type FileThumbnail struct {
	// Image: The URL-safe Base64 encoded bytes of the thumbnail image. It should
	// conform to RFC 4648 section 5.
	Image string `json:"image,omitempty"`
	// MimeType: The MIME type of the thumbnail.
	MimeType string `json:"mimeType,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Image") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Image") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s FileThumbnail) MarshalJSON() ([]byte, error) {
	type NoMethod FileThumbnail
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// FileVideoMediaMetadata: Output only. Metadata about video media. This will
// only be present for video types.
type FileVideoMediaMetadata struct {
	// DurationMillis: Output only. The duration of the video in milliseconds.
	DurationMillis int64 `json:"durationMillis,omitempty,string"`
	// Height: Output only. The height of the video in pixels.
	Height int64 `json:"height,omitempty"`
	// Width: Output only. The width of the video in pixels.
	Width int64 `json:"width,omitempty"`
	// ForceSendFields is a list of field names (e.g. "DurationMillis") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "DurationMillis") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s FileVideoMediaMetadata) MarshalJSON() ([]byte, error) {
	type NoMethod FileVideoMediaMetadata
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// FileList: A list of files.
type FileList struct {
	// Etag: The ETag of the list.
	Etag string `json:"etag,omitempty"`
	// IncompleteSearch: Whether the search process was incomplete. If true, then
	// some search results may be missing, since all documents were not searched.
	// This may occur when searching multiple drives with the "allDrives" corpora,
	// but all corpora could not be searched. When this happens, it is suggested
	// that clients narrow their query by choosing a different corpus such as
	// "default" or "drive".
	IncompleteSearch bool `json:"incompleteSearch,omitempty"`
	// Items: The list of files. If nextPageToken is populated, then this list may
	// be incomplete and an additional page of results should be fetched.
	Items []*File `json:"items,omitempty"`
	// Kind: This is always `drive#fileList`.
	Kind string `json:"kind,omitempty"`
	// NextLink: A link to the next page of files.
	NextLink string `json:"nextLink,omitempty"`
	// NextPageToken: The page token for the next page of files. This will be
	// absent if the end of the files list has been reached. If the token is
	// rejected for any reason, it should be discarded, and pagination should be
	// restarted from the first page of results.
	NextPageToken string `json:"nextPageToken,omitempty"`
	// SelfLink: A link back to this list.
	SelfLink string `json:"selfLink,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Etag") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Etag") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s FileList) MarshalJSON() ([]byte, error) {
	type NoMethod FileList
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GeneratedIds: A list of generated IDs which can be provided in insert
// requests
type GeneratedIds struct {
	// Ids: The IDs generated for the requesting user in the specified space.
	Ids []string `json:"ids,omitempty"`
	// Kind: This is always `drive#generatedIds`
	Kind string `json:"kind,omitempty"`
	// Space: The type of file that can be created with these IDs.
	Space string `json:"space,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Ids") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Ids") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GeneratedIds) MarshalJSON() ([]byte, error) {
	type NoMethod GeneratedIds
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Label: Representation of a label and label fields.
type Label struct {
	// Fields: A map of the fields on the label, keyed by the field's ID.
	Fields map[string]LabelField `json:"fields,omitempty"`
	// Id: The ID of the label.
	Id string `json:"id,omitempty"`
	// Kind: This is always `drive#label`
	Kind string `json:"kind,omitempty"`
	// RevisionId: The revision ID of the label.
	RevisionId string `json:"revisionId,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Fields") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Fields") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Label) MarshalJSON() ([]byte, error) {
	type NoMethod Label
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// LabelField: Representation of field, which is a typed key-value pair.
type LabelField struct {
	// DateString: Only present if valueType is dateString. RFC 3339 formatted
	// date: YYYY-MM-DD.
	DateString []string `json:"dateString,omitempty"`
	// Id: The identifier of this label field.
	Id string `json:"id,omitempty"`
	// Integer: Only present if `valueType` is `integer`.
	Integer googleapi.Int64s `json:"integer,omitempty"`
	// Kind: This is always `drive#labelField`.
	Kind string `json:"kind,omitempty"`
	// Selection: Only present if `valueType` is `selection`
	Selection []string `json:"selection,omitempty"`
	// Text: Only present if `valueType` is `text`.
	Text []string `json:"text,omitempty"`
	// User: Only present if `valueType` is `user`.
	User []*User `json:"user,omitempty"`
	// ValueType: The field type. While new values may be supported in the future,
	// the following are currently allowed: * `dateString` * `integer` *
	// `selection` * `text` * `user`
	ValueType string `json:"valueType,omitempty"`
	// ForceSendFields is a list of field names (e.g. "DateString") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "DateString") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s LabelField) MarshalJSON() ([]byte, error) {
	type NoMethod LabelField
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// LabelFieldModification: A modification to a label's field.
type LabelFieldModification struct {
	// FieldId: The ID of the field to be modified.
	FieldId string `json:"fieldId,omitempty"`
	// Kind: This is always `drive#labelFieldModification`.
	Kind string `json:"kind,omitempty"`
	// SetDateValues: Replaces the value of a dateString Field with these new
	// values. The string must be in the RFC 3339 full-date format: YYYY-MM-DD.
	SetDateValues []string `json:"setDateValues,omitempty"`
	// SetIntegerValues: Replaces the value of an `integer` field with these new
	// values.
	SetIntegerValues googleapi.Int64s `json:"setIntegerValues,omitempty"`
	// SetSelectionValues: Replaces a `selection` field with these new values.
	SetSelectionValues []string `json:"setSelectionValues,omitempty"`
	// SetTextValues: Sets the value of a `text` field.
	SetTextValues []string `json:"setTextValues,omitempty"`
	// SetUserValues: Replaces a `user` field with these new values. The values
	// must be valid email addresses.
	SetUserValues []string `json:"setUserValues,omitempty"`
	// UnsetValues: Unsets the values for this field.
	UnsetValues bool `json:"unsetValues,omitempty"`
	// ForceSendFields is a list of field names (e.g. "FieldId") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "FieldId") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s LabelFieldModification) MarshalJSON() ([]byte, error) {
	type NoMethod LabelFieldModification
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// LabelList: A list of labels applied to a file.
type LabelList struct {
	// Items: The list of labels.
	Items []*Label `json:"items,omitempty"`
	// Kind: This is always `drive#labelList`
	Kind string `json:"kind,omitempty"`
	// NextPageToken: The page token for the next page of labels. This field will
	// be absent if the end of the list has been reached. If the token is rejected
	// for any reason, it should be discarded, and pagination should be restarted
	// from the first page of results.
	NextPageToken string `json:"nextPageToken,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Items") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Items") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s LabelList) MarshalJSON() ([]byte, error) {
	type NoMethod LabelList
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// LabelModification: A modification to a label on a file. A LabelModification
// can be used to apply a label to a file, update an existing label on a file,
// or remove a label from a file.
type LabelModification struct {
	// FieldModifications: The list of modifications to this label's fields.
	FieldModifications []*LabelFieldModification `json:"fieldModifications,omitempty"`
	// Kind: This is always `drive#labelModification`.
	Kind string `json:"kind,omitempty"`
	// LabelId: The ID of the label to modify.
	LabelId string `json:"labelId,omitempty"`
	// RemoveLabel: If true, the label will be removed from the file.
	RemoveLabel bool `json:"removeLabel,omitempty"`
	// ForceSendFields is a list of field names (e.g. "FieldModifications") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "FieldModifications") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s LabelModification) MarshalJSON() ([]byte, error) {
	type NoMethod LabelModification
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ModifyLabelsRequest: A request to modify the set of labels on a file. This
// request may contain many modifications that will either all succeed or all
// fail atomically.
type ModifyLabelsRequest struct {
	// Kind: This is always `drive#modifyLabelsRequest`.
	Kind string `json:"kind,omitempty"`
	// LabelModifications: The list of modifications to apply to the labels on the
	// file.
	LabelModifications []*LabelModification `json:"labelModifications,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Kind") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Kind") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ModifyLabelsRequest) MarshalJSON() ([]byte, error) {
	type NoMethod ModifyLabelsRequest
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ModifyLabelsResponse: Response to a ModifyLabels request. This contains only
// those labels which were added or updated by the request.
type ModifyLabelsResponse struct {
	// Kind: This is always `drive#modifyLabelsResponse`
	Kind string `json:"kind,omitempty"`
	// ModifiedLabels: The list of labels which were added or updated by the
	// request.
	ModifiedLabels []*Label `json:"modifiedLabels,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Kind") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Kind") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ModifyLabelsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ModifyLabelsResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ParentList: A list of a file's parents.
type ParentList struct {
	// Etag: The ETag of the list.
	Etag string `json:"etag,omitempty"`
	// Items: The list of parents.
	Items []*ParentReference `json:"items,omitempty"`
	// Kind: This is always `drive#parentList`.
	Kind string `json:"kind,omitempty"`
	// SelfLink: A link back to this list.
	SelfLink string `json:"selfLink,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Etag") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Etag") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ParentList) MarshalJSON() ([]byte, error) {
	type NoMethod ParentList
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ParentReference: A reference to a file's parent. A file can only have one
// parent folder; specifying multiple parents isn't supported. Some resource
// methods (such as `parents.get`) require a `parentId`. Use the `parents.list`
// method to retrieve the ID for a parent.
type ParentReference struct {
	// Id: The ID of the parent.
	Id string `json:"id,omitempty"`
	// IsRoot: Output only. Whether or not the parent is the root folder.
	IsRoot bool `json:"isRoot,omitempty"`
	// Kind: Output only. This is always `drive#parentReference`.
	Kind string `json:"kind,omitempty"`
	// ParentLink: Output only. A link to the parent.
	ParentLink string `json:"parentLink,omitempty"`
	// SelfLink: Output only. A link back to this reference.
	SelfLink string `json:"selfLink,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Id") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Id") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ParentReference) MarshalJSON() ([]byte, error) {
	type NoMethod ParentReference
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Permission: A permission for a file. A permission grants a user, group,
// domain, or the world access to a file or a folder hierarchy. Some resource
// methods (such as `permissions.update`) require a `permissionId`. Use the
// `permissions.list` method to retrieve the ID for a file, folder, or shared
// drive.
type Permission struct {
	// AdditionalRoles: Additional roles for this user. Only `commenter` is
	// currently allowed, though more may be supported in the future.
	AdditionalRoles []string `json:"additionalRoles,omitempty"`
	// AuthKey: Output only. Deprecated.
	AuthKey string `json:"authKey,omitempty"`
	// Deleted: Output only. Whether the account associated with this permission
	// has been deleted. This field only pertains to user and group permissions.
	Deleted bool `json:"deleted,omitempty"`
	// Domain: Output only. The domain name of the entity this permission refers
	// to. This is an output-only field which is present when the permission type
	// is `user`, `group` or `domain`.
	Domain string `json:"domain,omitempty"`
	// EmailAddress: Output only. The email address of the user or group this
	// permission refers to. This is an output-only field which is present when the
	// permission type is `user` or `group`.
	EmailAddress string `json:"emailAddress,omitempty"`
	// Etag: Output only. The ETag of the permission.
	Etag string `json:"etag,omitempty"`
	// ExpirationDate: The time at which this permission will expire (RFC 3339
	// date-time). Expiration dates have the following restrictions: - They can
	// only be set on user and group permissions - The date must be in the future -
	// The date cannot be more than a year in the future - The date can only be set
	// on drive.permissions.update or drive.permissions.patch requests
	ExpirationDate string `json:"expirationDate,omitempty"`
	// Id: The ID of the user this permission refers to, and identical to the
	// `permissionId` in the About and Files resources. When making a
	// `drive.permissions.insert` request, exactly one of the `id` or `value`
	// fields must be specified unless the permission type is `anyone`, in which
	// case both `id` and `value` are ignored.
	Id string `json:"id,omitempty"`
	// InheritedPermissionsDisabled: When true, only organizers, owners, and users
	// with permissions added directly on the item can access it.
	InheritedPermissionsDisabled bool `json:"inheritedPermissionsDisabled,omitempty"`
	// Kind: Output only. This is always `drive#permission`.
	Kind string `json:"kind,omitempty"`
	// Name: Output only. The name for this permission.
	Name string `json:"name,omitempty"`
	// PendingOwner: Whether the account associated with this permission is a
	// pending owner. Only populated for `user` type permissions for files that are
	// not in a shared drive.
	PendingOwner bool `json:"pendingOwner,omitempty"`
	// PermissionDetails: Output only. Details of whether the permissions on this
	// item are inherited or directly on this item.
	PermissionDetails []*PermissionPermissionDetails `json:"permissionDetails,omitempty"`
	// PhotoLink: Output only. A link to the profile photo, if available.
	PhotoLink string `json:"photoLink,omitempty"`
	// Role: The primary role for this user. While new values may be supported in
	// the future, the following are currently allowed: * `owner` * `organizer` *
	// `fileOrganizer` * `writer` * `reader`
	Role string `json:"role,omitempty"`
	// SelfLink: Output only. A link back to this permission.
	SelfLink string `json:"selfLink,omitempty"`
	// TeamDrivePermissionDetails: Output only. Deprecated: Use `permissionDetails`
	// instead.
	TeamDrivePermissionDetails []*PermissionTeamDrivePermissionDetails `json:"teamDrivePermissionDetails,omitempty"`
	// Type: The account type. Allowed values are: * `user` * `group` * `domain` *
	// `anyone`
	Type string `json:"type,omitempty"`
	// Value: The email address or domain name for the entity. This is used during
	// inserts and is not populated in responses. When making a
	// `drive.permissions.insert` request, exactly one of the `id` or `value`
	// fields must be specified unless the permission type is `anyone`, in which
	// case both `id` and `value` are ignored.
	Value string `json:"value,omitempty"`
	// View: Indicates the view for this permission. Only populated for permissions
	// that belong to a view. published and metadata are the only supported values.
	// - published: The permission's role is published_reader. - metadata: The item
	// is only visible to the metadata view because the item has limited access and
	// the scope has at least read access to the parent. Note: The metadata view is
	// currently only supported on folders.
	View string `json:"view,omitempty"`
	// WithLink: Whether the link is required for this permission.
	WithLink bool `json:"withLink,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "AdditionalRoles") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AdditionalRoles") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Permission) MarshalJSON() ([]byte, error) {
	type NoMethod Permission
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type PermissionPermissionDetails struct {
	// AdditionalRoles: Output only. Additional roles for this user. Only
	// `commenter` is currently possible, though more may be supported in the
	// future.
	AdditionalRoles []string `json:"additionalRoles,omitempty"`
	// Inherited: Output only. Whether this permission is inherited. This field is
	// always populated.
	Inherited bool `json:"inherited,omitempty"`
	// InheritedFrom: Output only. The ID of the item from which this permission is
	// inherited. This is only populated for items in shared drives.
	InheritedFrom string `json:"inheritedFrom,omitempty"`
	// PermissionType: Output only. The permission type for this user. While new
	// values may be added in future, the following are currently possible: *
	// `file` * `member`
	PermissionType string `json:"permissionType,omitempty"`
	// Role: Output only. The primary role for this user. While new values may be
	// added in the future, the following are currently possible: * `organizer` *
	// `fileOrganizer` * `writer` * `reader`
	Role string `json:"role,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AdditionalRoles") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AdditionalRoles") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s PermissionPermissionDetails) MarshalJSON() ([]byte, error) {
	type NoMethod PermissionPermissionDetails
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type PermissionTeamDrivePermissionDetails struct {
	// AdditionalRoles: Output only. Deprecated: Use
	// `permissionDetails/additionalRoles` instead.
	AdditionalRoles []string `json:"additionalRoles,omitempty"`
	// Inherited: Output only. Deprecated: Use `permissionDetails/inherited`
	// instead.
	Inherited bool `json:"inherited,omitempty"`
	// InheritedFrom: Output only. Deprecated: Use
	// `permissionDetails/inheritedFrom` instead.
	InheritedFrom string `json:"inheritedFrom,omitempty"`
	// Role: Output only. Deprecated: Use `permissionDetails/role` instead.
	Role string `json:"role,omitempty"`
	// TeamDrivePermissionType: Output only. Deprecated: Use
	// `permissionDetails/permissionType` instead.
	TeamDrivePermissionType string `json:"teamDrivePermissionType,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AdditionalRoles") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AdditionalRoles") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s PermissionTeamDrivePermissionDetails) MarshalJSON() ([]byte, error) {
	type NoMethod PermissionTeamDrivePermissionDetails
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// PermissionId: An ID for a user or group as seen in Permission items.
type PermissionId struct {
	// Id: The permission ID.
	Id string `json:"id,omitempty"`
	// Kind: This is always `drive#permissionId`.
	Kind string `json:"kind,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Id") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Id") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s PermissionId) MarshalJSON() ([]byte, error) {
	type NoMethod PermissionId
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// PermissionList: A list of permissions associated with a file.
type PermissionList struct {
	// Etag: The ETag of the list.
	Etag string `json:"etag,omitempty"`
	// Items: The list of permissions.
	Items []*Permission `json:"items,omitempty"`
	// Kind: This is always `drive#permissionList`.
	Kind string `json:"kind,omitempty"`
	// NextPageToken: The page token for the next page of permissions. This field
	// will be absent if the end of the permissions list has been reached. If the
	// token is rejected for any reason, it should be discarded, and pagination
	// should be restarted from the first page of results.
	NextPageToken string `json:"nextPageToken,omitempty"`
	// SelfLink: A link back to this list.
	SelfLink string `json:"selfLink,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Etag") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Etag") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s PermissionList) MarshalJSON() ([]byte, error) {
	type NoMethod PermissionList
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Property: A key-value pair attached to a file that is either public or
// private to an application. The following limits apply to file properties: *
// Maximum of 100 properties total per file * Maximum of 30 private properties
// per app * Maximum of 30 public properties * Maximum of 124 bytes size limit
// on (key + value) string in UTF-8 encoding for a single property Some
// resource methods (such as `properties.update`) require a `propertyKey`. Use
// the `properties.list` method to retrieve the key for a property.
type Property struct {
	// Etag: Output only. ETag of the property.
	Etag string `json:"etag,omitempty"`
	// Key: The key of this property.
	Key string `json:"key,omitempty"`
	// Kind: Output only. This is always `drive#property`.
	Kind string `json:"kind,omitempty"`
	// SelfLink: Output only. The link back to this property.
	SelfLink string `json:"selfLink,omitempty"`
	// Value: The value of this property.
	Value string `json:"value,omitempty"`
	// Visibility: The visibility of this property. Allowed values are PRIVATE
	// (default) and PUBLIC. Private properties can only be retrieved using an
	// authenticated request. An authenticated request uses an access token
	// obtained with a OAuth 2 client ID. You cannot use an API key to retrieve
	// private properties.
	Visibility string `json:"visibility,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Etag") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Etag") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Property) MarshalJSON() ([]byte, error) {
	type NoMethod Property
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// PropertyList: A collection of properties, key-value pairs that are either
// public or private to an application.
type PropertyList struct {
	// Etag: The ETag of the list.
	Etag string `json:"etag,omitempty"`
	// Items: The list of properties.
	Items []*Property `json:"items,omitempty"`
	// Kind: This is always `drive#propertyList`.
	Kind string `json:"kind,omitempty"`
	// SelfLink: The link back to this list.
	SelfLink string `json:"selfLink,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Etag") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Etag") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s PropertyList) MarshalJSON() ([]byte, error) {
	type NoMethod PropertyList
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Revision: A revision of a file. Some resource methods (such as
// `revisions.update`) require a `revisionId`. Use the `revisions.list` method
// to retrieve the ID for a revision.
type Revision struct {
	// DownloadUrl: Output only. Short term download URL for the file. This will
	// only be populated on files with content stored in Drive.
	DownloadUrl string `json:"downloadUrl,omitempty"`
	// Etag: Output only. The ETag of the revision.
	Etag string `json:"etag,omitempty"`
	// ExportLinks: Output only. Links for exporting Docs Editors files to specific
	// formats.
	ExportLinks map[string]string `json:"exportLinks,omitempty"`
	// FileSize: Output only. The size of the revision in bytes. This will only be
	// populated on files with content stored in Drive.
	FileSize int64 `json:"fileSize,omitempty,string"`
	// Id: Output only. The ID of the revision.
	Id string `json:"id,omitempty"`
	// Kind: Output only. This is always `drive#revision`.
	Kind string `json:"kind,omitempty"`
	// LastModifyingUser: Output only. The last user to modify this revision. This
	// field is only populated when the last modification was performed by a
	// signed-in user.
	LastModifyingUser *User `json:"lastModifyingUser,omitempty"`
	// LastModifyingUserName: Output only. Name of the last user to modify this
	// revision.
	LastModifyingUserName string `json:"lastModifyingUserName,omitempty"`
	// Md5Checksum: Output only. An MD5 checksum for the content of this revision.
	// This will only be populated on files with content stored in Drive.
	Md5Checksum string `json:"md5Checksum,omitempty"`
	// MimeType: Output only. The MIME type of the revision.
	MimeType string `json:"mimeType,omitempty"`
	// ModifiedDate: Last time this revision was modified (formatted RFC 3339
	// timestamp).
	ModifiedDate string `json:"modifiedDate,omitempty"`
	// OriginalFilename: Output only. The original filename when this revision was
	// created. This will only be populated on files with content stored in Drive.
	OriginalFilename string `json:"originalFilename,omitempty"`
	// Pinned: Whether this revision is pinned to prevent automatic purging. If not
	// set, the revision is automatically purged 30 days after newer content is
	// uploaded. This field can only be modified on files with content stored in
	// Drive, excluding Docs Editors files. Revisions can also be pinned when they
	// are created through the drive.files.insert/update/copy by using the pinned
	// query parameter. Pinned revisions are stored indefinitely using additional
	// storage quota, up to a maximum of 200 revisions.
	Pinned bool `json:"pinned,omitempty"`
	// PublishAuto: Whether subsequent revisions will be automatically republished.
	// This is only populated and can only be modified for Docs Editors files.
	PublishAuto bool `json:"publishAuto,omitempty"`
	// Published: Whether this revision is published. This is only populated and
	// can only be modified for Docs Editors files.
	Published bool `json:"published,omitempty"`
	// PublishedLink: Output only. A link to the published revision. This is only
	// populated for Docs Editors files.
	PublishedLink string `json:"publishedLink,omitempty"`
	// PublishedOutsideDomain: Whether this revision is published outside the
	// domain. This is only populated and can only be modified for Docs Editors
	// files.
	PublishedOutsideDomain bool `json:"publishedOutsideDomain,omitempty"`
	// SelfLink: Output only. A link back to this revision.
	SelfLink string `json:"selfLink,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "DownloadUrl") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "DownloadUrl") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Revision) MarshalJSON() ([]byte, error) {
	type NoMethod Revision
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// RevisionList: A list of revisions of a file.
type RevisionList struct {
	// Etag: The ETag of the list.
	Etag string `json:"etag,omitempty"`
	// Items: The list of revisions. If nextPageToken is populated, then this list
	// may be incomplete and an additional page of results should be fetched.
	Items []*Revision `json:"items,omitempty"`
	// Kind: This is always `drive#revisionList`.
	Kind string `json:"kind,omitempty"`
	// NextPageToken: The page token for the next page of revisions. This field
	// will be absent if the end of the revisions list has been reached. If the
	// token is rejected for any reason, it should be discarded and pagination
	// should be restarted from the first page of results.
	NextPageToken string `json:"nextPageToken,omitempty"`
	// SelfLink: A link back to this list.
	SelfLink string `json:"selfLink,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Etag") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Etag") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s RevisionList) MarshalJSON() ([]byte, error) {
	type NoMethod RevisionList
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type StartPageToken struct {
	// Kind: Identifies what kind of resource this is. Value: the fixed string
	// "drive#startPageToken".
	Kind string `json:"kind,omitempty"`
	// StartPageToken: The starting page token for listing changes.
	StartPageToken string `json:"startPageToken,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Kind") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Kind") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s StartPageToken) MarshalJSON() ([]byte, error) {
	type NoMethod StartPageToken
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// TeamDrive: Deprecated: Use the `drive` collection instead.
type TeamDrive struct {
	// BackgroundImageFile: An image file and cropping parameters from which a
	// background image for this Team Drive is set. This is a write only field; it
	// can only be set on `drive.teamdrives.update` requests that don't set
	// `themeId`. When specified, all fields of the `backgroundImageFile` must be
	// set.
	BackgroundImageFile *TeamDriveBackgroundImageFile `json:"backgroundImageFile,omitempty"`
	// BackgroundImageLink: A short-lived link to this Team Drive's background
	// image.
	BackgroundImageLink string `json:"backgroundImageLink,omitempty"`
	// Capabilities: Capabilities the current user has on this Team Drive.
	Capabilities *TeamDriveCapabilities `json:"capabilities,omitempty"`
	// ColorRgb: The color of this Team Drive as an RGB hex string. It can only be
	// set on a `drive.teamdrives.update` request that does not set `themeId`.
	ColorRgb string `json:"colorRgb,omitempty"`
	// CreatedDate: The time at which the Team Drive was created (RFC 3339
	// date-time).
	CreatedDate string `json:"createdDate,omitempty"`
	// Id: The ID of this Team Drive which is also the ID of the top level folder
	// of this Team Drive.
	Id string `json:"id,omitempty"`
	// Kind: This is always `drive#teamDrive`
	Kind string `json:"kind,omitempty"`
	// Name: The name of this Team Drive.
	Name string `json:"name,omitempty"`
	// OrgUnitId: The organizational unit of this shared drive. This field is only
	// populated on `drives.list` responses when the `useDomainAdminAccess`
	// parameter is set to `true`.
	OrgUnitId string `json:"orgUnitId,omitempty"`
	// Restrictions: A set of restrictions that apply to this Team Drive or items
	// inside this Team Drive.
	Restrictions *TeamDriveRestrictions `json:"restrictions,omitempty"`
	// ThemeId: The ID of the theme from which the background image and color will
	// be set. The set of possible `teamDriveThemes` can be retrieved from a
	// `drive.about.get` response. When not specified on a
	// `drive.teamdrives.insert` request, a random theme is chosen from which the
	// background image and color are set. This is a write-only field; it can only
	// be set on requests that don't set `colorRgb` or `backgroundImageFile`.
	ThemeId string `json:"themeId,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "BackgroundImageFile") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "BackgroundImageFile") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s TeamDrive) MarshalJSON() ([]byte, error) {
	type NoMethod TeamDrive
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// TeamDriveBackgroundImageFile: An image file and cropping parameters from
// which a background image for this Team Drive is set. This is a write only
// field; it can only be set on `drive.teamdrives.update` requests that don't
// set `themeId`. When specified, all fields of the `backgroundImageFile` must
// be set.
type TeamDriveBackgroundImageFile struct {
	// Id: The ID of an image file in Drive to use for the background image.
	Id string `json:"id,omitempty"`
	// Width: The width of the cropped image in the closed range of 0 to 1. This
	// value represents the width of the cropped image divided by the width of the
	// entire image. The height is computed by applying a width to height aspect
	// ratio of 80 to 9. The resulting image must be at least 1280 pixels wide and
	// 144 pixels high.
	Width float64 `json:"width,omitempty"`
	// XCoordinate: The X coordinate of the upper left corner of the cropping area
	// in the background image. This is a value in the closed range of 0 to 1. This
	// value represents the horizontal distance from the left side of the entire
	// image to the left side of the cropping area divided by the width of the
	// entire image.
	XCoordinate float64 `json:"xCoordinate,omitempty"`
	// YCoordinate: The Y coordinate of the upper left corner of the cropping area
	// in the background image. This is a value in the closed range of 0 to 1. This
	// value represents the vertical distance from the top side of the entire image
	// to the top side of the cropping area divided by the height of the entire
	// image.
	YCoordinate float64 `json:"yCoordinate,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Id") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Id") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s TeamDriveBackgroundImageFile) MarshalJSON() ([]byte, error) {
	type NoMethod TeamDriveBackgroundImageFile
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

func (s *TeamDriveBackgroundImageFile) UnmarshalJSON(data []byte) error {
	type NoMethod TeamDriveBackgroundImageFile
	var s1 struct {
		Width       gensupport.JSONFloat64 `json:"width"`
		XCoordinate gensupport.JSONFloat64 `json:"xCoordinate"`
		YCoordinate gensupport.JSONFloat64 `json:"yCoordinate"`
		*NoMethod
	}
	s1.NoMethod = (*NoMethod)(s)
	if err := json.Unmarshal(data, &s1); err != nil {
		return err
	}
	s.Width = float64(s1.Width)
	s.XCoordinate = float64(s1.XCoordinate)
	s.YCoordinate = float64(s1.YCoordinate)
	return nil
}

// TeamDriveCapabilities: Capabilities the current user has on this Team Drive.
type TeamDriveCapabilities struct {
	// CanAddChildren: Whether the current user can add children to folders in this
	// Team Drive.
	CanAddChildren bool `json:"canAddChildren,omitempty"`
	// CanChangeCopyRequiresWriterPermissionRestriction: Whether the current user
	// can change the `copyRequiresWriterPermission` restriction of this Team
	// Drive.
	CanChangeCopyRequiresWriterPermissionRestriction bool `json:"canChangeCopyRequiresWriterPermissionRestriction,omitempty"`
	// CanChangeDomainUsersOnlyRestriction: Whether the current user can change the
	// `domainUsersOnly` restriction of this Team Drive.
	CanChangeDomainUsersOnlyRestriction bool `json:"canChangeDomainUsersOnlyRestriction,omitempty"`
	// CanChangeSharingFoldersRequiresOrganizerPermissionRestriction: Whether the
	// current user can change the `sharingFoldersRequiresOrganizerPermission`
	// restriction of this Team Drive.
	CanChangeSharingFoldersRequiresOrganizerPermissionRestriction bool `json:"canChangeSharingFoldersRequiresOrganizerPermissionRestriction,omitempty"`
	// CanChangeTeamDriveBackground: Whether the current user can change the
	// background of this Team Drive.
	CanChangeTeamDriveBackground bool `json:"canChangeTeamDriveBackground,omitempty"`
	// CanChangeTeamMembersOnlyRestriction: Whether the current user can change the
	// `teamMembersOnly` restriction of this Team Drive.
	CanChangeTeamMembersOnlyRestriction bool `json:"canChangeTeamMembersOnlyRestriction,omitempty"`
	// CanComment: Whether the current user can comment on files in this Team
	// Drive.
	CanComment bool `json:"canComment,omitempty"`
	// CanCopy: Whether the current user can copy files in this Team Drive.
	CanCopy bool `json:"canCopy,omitempty"`
	// CanDeleteChildren: Whether the current user can delete children from folders
	// in this Team Drive.
	CanDeleteChildren bool `json:"canDeleteChildren,omitempty"`
	// CanDeleteTeamDrive: Whether the current user can delete this Team Drive.
	// Attempting to delete the Team Drive may still fail if there are untrashed
	// items inside the Team Drive.
	CanDeleteTeamDrive bool `json:"canDeleteTeamDrive,omitempty"`
	// CanDownload: Whether the current user can download files in this Team Drive.
	CanDownload bool `json:"canDownload,omitempty"`
	// CanEdit: Whether the current user can edit files in this Team Drive
	CanEdit bool `json:"canEdit,omitempty"`
	// CanListChildren: Whether the current user can list the children of folders
	// in this Team Drive.
	CanListChildren bool `json:"canListChildren,omitempty"`
	// CanManageMembers: Whether the current user can add members to this Team
	// Drive or remove them or change their role.
	CanManageMembers bool `json:"canManageMembers,omitempty"`
	// CanReadRevisions: Whether the current user can read the revisions resource
	// of files in this Team Drive.
	CanReadRevisions bool `json:"canReadRevisions,omitempty"`
	// CanRemoveChildren: Deprecated: Use `canDeleteChildren` or `canTrashChildren`
	// instead.
	CanRemoveChildren bool `json:"canRemoveChildren,omitempty"`
	// CanRename: Whether the current user can rename files or folders in this Team
	// Drive.
	CanRename bool `json:"canRename,omitempty"`
	// CanRenameTeamDrive: Whether the current user can rename this Team Drive.
	CanRenameTeamDrive bool `json:"canRenameTeamDrive,omitempty"`
	// CanResetTeamDriveRestrictions: Whether the current user can reset the Team
	// Drive restrictions to defaults.
	CanResetTeamDriveRestrictions bool `json:"canResetTeamDriveRestrictions,omitempty"`
	// CanShare: Whether the current user can share files or folders in this Team
	// Drive.
	CanShare bool `json:"canShare,omitempty"`
	// CanTrashChildren: Whether the current user can trash children from folders
	// in this Team Drive.
	CanTrashChildren bool `json:"canTrashChildren,omitempty"`
	// ForceSendFields is a list of field names (e.g. "CanAddChildren") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CanAddChildren") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s TeamDriveCapabilities) MarshalJSON() ([]byte, error) {
	type NoMethod TeamDriveCapabilities
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// TeamDriveRestrictions: A set of restrictions that apply to this Team Drive
// or items inside this Team Drive.
type TeamDriveRestrictions struct {
	// AdminManagedRestrictions: Whether administrative privileges on this Team
	// Drive are required to modify restrictions.
	AdminManagedRestrictions bool `json:"adminManagedRestrictions,omitempty"`
	// CopyRequiresWriterPermission: Whether the options to copy, print, or
	// download files inside this Team Drive, should be disabled for readers and
	// commenters. When this restriction is set to `true`, it will override the
	// similarly named field to `true` for any file inside this Team Drive.
	CopyRequiresWriterPermission bool `json:"copyRequiresWriterPermission,omitempty"`
	// DomainUsersOnly: Whether access to this Team Drive and items inside this
	// Team Drive is restricted to users of the domain to which this Team Drive
	// belongs. This restriction may be overridden by other sharing policies
	// controlled outside of this Team Drive.
	DomainUsersOnly bool `json:"domainUsersOnly,omitempty"`
	// SharingFoldersRequiresOrganizerPermission: If true, only users with the
	// organizer role can share folders. If false, users with either the organizer
	// role or the file organizer role can share folders.
	SharingFoldersRequiresOrganizerPermission bool `json:"sharingFoldersRequiresOrganizerPermission,omitempty"`
	// TeamMembersOnly: Whether access to items inside this Team Drive is
	// restricted to members of this Team Drive.
	TeamMembersOnly bool `json:"teamMembersOnly,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AdminManagedRestrictions")
	// to unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AdminManagedRestrictions") to
	// include in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s TeamDriveRestrictions) MarshalJSON() ([]byte, error) {
	type NoMethod TeamDriveRestrictions
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// TeamDriveList: A list of Team Drives.
type TeamDriveList struct {
	// Items: The list of Team Drives.
	Items []*TeamDrive `json:"items,omitempty"`
	// Kind: This is always `drive#teamDriveList`
	Kind string `json:"kind,omitempty"`
	// NextPageToken: The page token for the next page of Team Drives.
	NextPageToken string `json:"nextPageToken,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Items") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Items") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s TeamDriveList) MarshalJSON() ([]byte, error) {
	type NoMethod TeamDriveList
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// User: Information about a Drive user.
type User struct {
	// DisplayName: Output only. A plain text displayable name for this user.
	DisplayName string `json:"displayName,omitempty"`
	// EmailAddress: Output only. The email address of the user.
	EmailAddress string `json:"emailAddress,omitempty"`
	// IsAuthenticatedUser: Output only. Whether this user is the same as the
	// authenticated user for whom the request was made.
	IsAuthenticatedUser bool `json:"isAuthenticatedUser,omitempty"`
	// Kind: Output only. This is always `drive#user`.
	Kind string `json:"kind,omitempty"`
	// PermissionId: Output only. The user's ID as visible in the permissions
	// collection.
	PermissionId string `json:"permissionId,omitempty"`
	// Picture: Output only. The user's profile picture.
	Picture *UserPicture `json:"picture,omitempty"`
	// ForceSendFields is a list of field names (e.g. "DisplayName") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "DisplayName") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s User) MarshalJSON() ([]byte, error) {
	type NoMethod User
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// UserPicture: Output only. The user's profile picture.
type UserPicture struct {
	// Url: Output only. A URL that points to a profile picture of this user.
	Url string `json:"url,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Url") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Url") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s UserPicture) MarshalJSON() ([]byte, error) {
	type NoMethod UserPicture
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type AboutGetCall struct {
	s            *Service
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets the information about the current user along with Drive API
// settings
func (r *AboutService) Get() *AboutGetCall {
	c := &AboutGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	return c
}

// IncludeSubscribed sets the optional parameter "includeSubscribed": Whether
// to count changes outside the My Drive hierarchy. When set to false, changes
// to files such as those in the Application Data folder or shared files which
// have not been added to My Drive will be omitted from the `maxChangeIdCount`.
func (c *AboutGetCall) IncludeSubscribed(includeSubscribed bool) *AboutGetCall {
	c.urlParams_.Set("includeSubscribed", fmt.Sprint(includeSubscribed))
	return c
}

// MaxChangeIdCount sets the optional parameter "maxChangeIdCount": Maximum
// number of remaining change IDs to count
func (c *AboutGetCall) MaxChangeIdCount(maxChangeIdCount int64) *AboutGetCall {
	c.urlParams_.Set("maxChangeIdCount", fmt.Sprint(maxChangeIdCount))
	return c
}

// StartChangeId sets the optional parameter "startChangeId": Change ID to
// start counting from when calculating number of remaining change IDs
func (c *AboutGetCall) StartChangeId(startChangeId int64) *AboutGetCall {
	c.urlParams_.Set("startChangeId", fmt.Sprint(startChangeId))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AboutGetCall) Fields(s ...googleapi.Field) *AboutGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AboutGetCall) IfNoneMatch(entityTag string) *AboutGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AboutGetCall) Context(ctx context.Context) *AboutGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AboutGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AboutGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "about")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.about.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.about.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *About.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AboutGetCall) Do(opts ...googleapi.CallOption) (*About, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &About{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.about.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AppsGetCall struct {
	s            *Service
	appId        string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets a specific app.
//
// - appId: The ID of the app.
func (r *AppsService) Get(appId string) *AppsGetCall {
	c := &AppsGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.appId = appId
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AppsGetCall) Fields(s ...googleapi.Field) *AppsGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AppsGetCall) IfNoneMatch(entityTag string) *AppsGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AppsGetCall) Context(ctx context.Context) *AppsGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AppsGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AppsGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "apps/{appId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"appId": c.appId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.apps.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.apps.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *App.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AppsGetCall) Do(opts ...googleapi.CallOption) (*App, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &App{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.apps.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type AppsListCall struct {
	s            *Service
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists a user's installed apps.
func (r *AppsService) List() *AppsListCall {
	c := &AppsListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	return c
}

// AppFilterExtensions sets the optional parameter "appFilterExtensions": A
// comma-separated list of file extensions for open with filtering. All apps
// within the given app query scope which can open any of the given file
// extensions will be included in the response. If `appFilterMimeTypes` are
// provided as well, the result is a union of the two resulting app lists.
func (c *AppsListCall) AppFilterExtensions(appFilterExtensions string) *AppsListCall {
	c.urlParams_.Set("appFilterExtensions", appFilterExtensions)
	return c
}

// AppFilterMimeTypes sets the optional parameter "appFilterMimeTypes": A
// comma-separated list of MIME types for open with filtering. All apps within
// the given app query scope which can open any of the given MIME types will be
// included in the response. If `appFilterExtensions` are provided as well, the
// result is a union of the two resulting app lists.
func (c *AppsListCall) AppFilterMimeTypes(appFilterMimeTypes string) *AppsListCall {
	c.urlParams_.Set("appFilterMimeTypes", appFilterMimeTypes)
	return c
}

// LanguageCode sets the optional parameter "languageCode": A language or
// locale code, as defined by BCP 47, with some extensions from Unicode's LDML
// format (http://www.unicode.org/reports/tr35/).
func (c *AppsListCall) LanguageCode(languageCode string) *AppsListCall {
	c.urlParams_.Set("languageCode", languageCode)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *AppsListCall) Fields(s ...googleapi.Field) *AppsListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *AppsListCall) IfNoneMatch(entityTag string) *AppsListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *AppsListCall) Context(ctx context.Context) *AppsListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *AppsListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *AppsListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "apps")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.apps.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.apps.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *AppList.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *AppsListCall) Do(opts ...googleapi.CallOption) (*AppList, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &AppList{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.apps.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ChangesGetCall struct {
	s            *Service
	changeId     string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Deprecated: Use `changes.getStartPageToken` and `changes.list` to
// retrieve recent changes.
//
// - changeId: The ID of the change.
func (r *ChangesService) Get(changeId string) *ChangesGetCall {
	c := &ChangesGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.changeId = changeId
	return c
}

// DriveId sets the optional parameter "driveId": The shared drive from which
// the change will be returned.
func (c *ChangesGetCall) DriveId(driveId string) *ChangesGetCall {
	c.urlParams_.Set("driveId", driveId)
	return c
}

// SupportsAllDrives sets the optional parameter "supportsAllDrives": Whether
// the requesting application supports both My Drives and shared drives.
func (c *ChangesGetCall) SupportsAllDrives(supportsAllDrives bool) *ChangesGetCall {
	c.urlParams_.Set("supportsAllDrives", fmt.Sprint(supportsAllDrives))
	return c
}

// SupportsTeamDrives sets the optional parameter "supportsTeamDrives":
// Deprecated: Use `supportsAllDrives` instead.
func (c *ChangesGetCall) SupportsTeamDrives(supportsTeamDrives bool) *ChangesGetCall {
	c.urlParams_.Set("supportsTeamDrives", fmt.Sprint(supportsTeamDrives))
	return c
}

// TeamDriveId sets the optional parameter "teamDriveId": Deprecated: Use
// `driveId` instead.
func (c *ChangesGetCall) TeamDriveId(teamDriveId string) *ChangesGetCall {
	c.urlParams_.Set("teamDriveId", teamDriveId)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ChangesGetCall) Fields(s ...googleapi.Field) *ChangesGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *ChangesGetCall) IfNoneMatch(entityTag string) *ChangesGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ChangesGetCall) Context(ctx context.Context) *ChangesGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ChangesGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ChangesGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "changes/{changeId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"changeId": c.changeId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.changes.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.changes.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *Change.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *ChangesGetCall) Do(opts ...googleapi.CallOption) (*Change, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Change{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.changes.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ChangesGetStartPageTokenCall struct {
	s            *Service
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// GetStartPageToken: Gets the starting pageToken for listing future changes.
func (r *ChangesService) GetStartPageToken() *ChangesGetStartPageTokenCall {
	c := &ChangesGetStartPageTokenCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	return c
}

// DriveId sets the optional parameter "driveId": The ID of the shared drive
// for which the starting pageToken for listing future changes from that shared
// drive will be returned.
func (c *ChangesGetStartPageTokenCall) DriveId(driveId string) *ChangesGetStartPageTokenCall {
	c.urlParams_.Set("driveId", driveId)
	return c
}

// SupportsAllDrives sets the optional parameter "supportsAllDrives": Whether
// the requesting application supports both My Drives and shared drives.
func (c *ChangesGetStartPageTokenCall) SupportsAllDrives(supportsAllDrives bool) *ChangesGetStartPageTokenCall {
	c.urlParams_.Set("supportsAllDrives", fmt.Sprint(supportsAllDrives))
	return c
}

// SupportsTeamDrives sets the optional parameter "supportsTeamDrives":
// Deprecated: Use `supportsAllDrives` instead.
func (c *ChangesGetStartPageTokenCall) SupportsTeamDrives(supportsTeamDrives bool) *ChangesGetStartPageTokenCall {
	c.urlParams_.Set("supportsTeamDrives", fmt.Sprint(supportsTeamDrives))
	return c
}

// TeamDriveId sets the optional parameter "teamDriveId": Deprecated: Use
// `driveId` instead.
func (c *ChangesGetStartPageTokenCall) TeamDriveId(teamDriveId string) *ChangesGetStartPageTokenCall {
	c.urlParams_.Set("teamDriveId", teamDriveId)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ChangesGetStartPageTokenCall) Fields(s ...googleapi.Field) *ChangesGetStartPageTokenCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *ChangesGetStartPageTokenCall) IfNoneMatch(entityTag string) *ChangesGetStartPageTokenCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ChangesGetStartPageTokenCall) Context(ctx context.Context) *ChangesGetStartPageTokenCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ChangesGetStartPageTokenCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ChangesGetStartPageTokenCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "changes/startPageToken")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.changes.getStartPageToken", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.changes.getStartPageToken" call.
// Any non-2xx status code is an error. Response headers are in either
// *StartPageToken.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *ChangesGetStartPageTokenCall) Do(opts ...googleapi.CallOption) (*StartPageToken, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &StartPageToken{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.changes.getStartPageToken", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ChangesListCall struct {
	s            *Service
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists the changes for a user or shared drive.
func (r *ChangesService) List() *ChangesListCall {
	c := &ChangesListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	return c
}

// DriveId sets the optional parameter "driveId": The shared drive from which
// changes will be returned. If specified the change IDs will be reflective of
// the shared drive; use the combined drive ID and change ID as an identifier.
func (c *ChangesListCall) DriveId(driveId string) *ChangesListCall {
	c.urlParams_.Set("driveId", driveId)
	return c
}

// IncludeCorpusRemovals sets the optional parameter "includeCorpusRemovals":
// Whether changes should include the file resource if the file is still
// accessible by the user at the time of the request, even when a file was
// removed from the list of changes and there will be no further change entries
// for this file.
func (c *ChangesListCall) IncludeCorpusRemovals(includeCorpusRemovals bool) *ChangesListCall {
	c.urlParams_.Set("includeCorpusRemovals", fmt.Sprint(includeCorpusRemovals))
	return c
}

// IncludeDeleted sets the optional parameter "includeDeleted": Whether to
// include changes indicating that items have been removed from the list of
// changes, for example by deletion or loss of access.
func (c *ChangesListCall) IncludeDeleted(includeDeleted bool) *ChangesListCall {
	c.urlParams_.Set("includeDeleted", fmt.Sprint(includeDeleted))
	return c
}

// IncludeItemsFromAllDrives sets the optional parameter
// "includeItemsFromAllDrives": Whether both My Drive and shared drive items
// should be included in results.
func (c *ChangesListCall) IncludeItemsFromAllDrives(includeItemsFromAllDrives bool) *ChangesListCall {
	c.urlParams_.Set("includeItemsFromAllDrives", fmt.Sprint(includeItemsFromAllDrives))
	return c
}

// IncludeLabels sets the optional parameter "includeLabels": A comma-separated
// list of IDs of labels to include in the `labelInfo` part of the response.
func (c *ChangesListCall) IncludeLabels(includeLabels string) *ChangesListCall {
	c.urlParams_.Set("includeLabels", includeLabels)
	return c
}

// IncludePermissionsForView sets the optional parameter
// "includePermissionsForView": Specifies which additional view's permissions
// to include in the response. Only `published` is supported.
func (c *ChangesListCall) IncludePermissionsForView(includePermissionsForView string) *ChangesListCall {
	c.urlParams_.Set("includePermissionsForView", includePermissionsForView)
	return c
}

// IncludeSubscribed sets the optional parameter "includeSubscribed": Whether
// to include changes outside the My Drive hierarchy in the result. When set to
// false, changes to files such as those in the Application Data folder or
// shared files which have not been added to My Drive will be omitted from the
// result.
func (c *ChangesListCall) IncludeSubscribed(includeSubscribed bool) *ChangesListCall {
	c.urlParams_.Set("includeSubscribed", fmt.Sprint(includeSubscribed))
	return c
}

// IncludeTeamDriveItems sets the optional parameter "includeTeamDriveItems":
// Deprecated: Use `includeItemsFromAllDrives` instead.
func (c *ChangesListCall) IncludeTeamDriveItems(includeTeamDriveItems bool) *ChangesListCall {
	c.urlParams_.Set("includeTeamDriveItems", fmt.Sprint(includeTeamDriveItems))
	return c
}

// MaxResults sets the optional parameter "maxResults": Maximum number of
// changes to return.
func (c *ChangesListCall) MaxResults(maxResults int64) *ChangesListCall {
	c.urlParams_.Set("maxResults", fmt.Sprint(maxResults))
	return c
}

// PageToken sets the optional parameter "pageToken": The token for continuing
// a previous list request on the next page. This should be set to the value of
// `nextPageToken` from the previous response or to the response from the
// getStartPageToken method.
func (c *ChangesListCall) PageToken(pageToken string) *ChangesListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Spaces sets the optional parameter "spaces": A comma-separated list of
// spaces to query. Supported values are `drive`, `appDataFolder` and `photos`.
func (c *ChangesListCall) Spaces(spaces string) *ChangesListCall {
	c.urlParams_.Set("spaces", spaces)
	return c
}

// StartChangeId sets the optional parameter "startChangeId": Deprecated: Use
// `pageToken` instead.
func (c *ChangesListCall) StartChangeId(startChangeId int64) *ChangesListCall {
	c.urlParams_.Set("startChangeId", fmt.Sprint(startChangeId))
	return c
}

// SupportsAllDrives sets the optional parameter "supportsAllDrives": Whether
// the requesting application supports both My Drives and shared drives.
func (c *ChangesListCall) SupportsAllDrives(supportsAllDrives bool) *ChangesListCall {
	c.urlParams_.Set("supportsAllDrives", fmt.Sprint(supportsAllDrives))
	return c
}

// SupportsTeamDrives sets the optional parameter "supportsTeamDrives":
// Deprecated: Use `supportsAllDrives` instead.
func (c *ChangesListCall) SupportsTeamDrives(supportsTeamDrives bool) *ChangesListCall {
	c.urlParams_.Set("supportsTeamDrives", fmt.Sprint(supportsTeamDrives))
	return c
}

// TeamDriveId sets the optional parameter "teamDriveId": Deprecated: Use
// `driveId` instead.
func (c *ChangesListCall) TeamDriveId(teamDriveId string) *ChangesListCall {
	c.urlParams_.Set("teamDriveId", teamDriveId)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ChangesListCall) Fields(s ...googleapi.Field) *ChangesListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *ChangesListCall) IfNoneMatch(entityTag string) *ChangesListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ChangesListCall) Context(ctx context.Context) *ChangesListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ChangesListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ChangesListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "changes")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.changes.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.changes.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ChangeList.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *ChangesListCall) Do(opts ...googleapi.CallOption) (*ChangeList, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ChangeList{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.changes.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *ChangesListCall) Pages(ctx context.Context, f func(*ChangeList) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type ChangesWatchCall struct {
	s          *Service
	channel    *Channel
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Watch: Subscribe to changes for a user.
func (r *ChangesService) Watch(channel *Channel) *ChangesWatchCall {
	c := &ChangesWatchCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.channel = channel
	return c
}

// DriveId sets the optional parameter "driveId": The shared drive from which
// changes will be returned. If specified the change IDs will be reflective of
// the shared drive; use the combined drive ID and change ID as an identifier.
func (c *ChangesWatchCall) DriveId(driveId string) *ChangesWatchCall {
	c.urlParams_.Set("driveId", driveId)
	return c
}

// IncludeCorpusRemovals sets the optional parameter "includeCorpusRemovals":
// Whether changes should include the file resource if the file is still
// accessible by the user at the time of the request, even when a file was
// removed from the list of changes and there will be no further change entries
// for this file.
func (c *ChangesWatchCall) IncludeCorpusRemovals(includeCorpusRemovals bool) *ChangesWatchCall {
	c.urlParams_.Set("includeCorpusRemovals", fmt.Sprint(includeCorpusRemovals))
	return c
}

// IncludeDeleted sets the optional parameter "includeDeleted": Whether to
// include changes indicating that items have been removed from the list of
// changes, for example by deletion or loss of access.
func (c *ChangesWatchCall) IncludeDeleted(includeDeleted bool) *ChangesWatchCall {
	c.urlParams_.Set("includeDeleted", fmt.Sprint(includeDeleted))
	return c
}

// IncludeItemsFromAllDrives sets the optional parameter
// "includeItemsFromAllDrives": Whether both My Drive and shared drive items
// should be included in results.
func (c *ChangesWatchCall) IncludeItemsFromAllDrives(includeItemsFromAllDrives bool) *ChangesWatchCall {
	c.urlParams_.Set("includeItemsFromAllDrives", fmt.Sprint(includeItemsFromAllDrives))
	return c
}

// IncludeLabels sets the optional parameter "includeLabels": A comma-separated
// list of IDs of labels to include in the `labelInfo` part of the response.
func (c *ChangesWatchCall) IncludeLabels(includeLabels string) *ChangesWatchCall {
	c.urlParams_.Set("includeLabels", includeLabels)
	return c
}

// IncludePermissionsForView sets the optional parameter
// "includePermissionsForView": Specifies which additional view's permissions
// to include in the response. Only `published` is supported.
func (c *ChangesWatchCall) IncludePermissionsForView(includePermissionsForView string) *ChangesWatchCall {
	c.urlParams_.Set("includePermissionsForView", includePermissionsForView)
	return c
}

// IncludeSubscribed sets the optional parameter "includeSubscribed": Whether
// to include changes outside the My Drive hierarchy in the result. When set to
// false, changes to files such as those in the Application Data folder or
// shared files which have not been added to My Drive will be omitted from the
// result.
func (c *ChangesWatchCall) IncludeSubscribed(includeSubscribed bool) *ChangesWatchCall {
	c.urlParams_.Set("includeSubscribed", fmt.Sprint(includeSubscribed))
	return c
}

// IncludeTeamDriveItems sets the optional parameter "includeTeamDriveItems":
// Deprecated: Use `includeItemsFromAllDrives` instead.
func (c *ChangesWatchCall) IncludeTeamDriveItems(includeTeamDriveItems bool) *ChangesWatchCall {
	c.urlParams_.Set("includeTeamDriveItems", fmt.Sprint(includeTeamDriveItems))
	return c
}

// MaxResults sets the optional parameter "maxResults": Maximum number of
// changes to return.
func (c *ChangesWatchCall) MaxResults(maxResults int64) *ChangesWatchCall {
	c.urlParams_.Set("maxResults", fmt.Sprint(maxResults))
	return c
}

// PageToken sets the optional parameter "pageToken": The token for continuing
// a previous list request on the next page. This should be set to the value of
// `nextPageToken` from the previous response or to the response from the
// getStartPageToken method.
func (c *ChangesWatchCall) PageToken(pageToken string) *ChangesWatchCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Spaces sets the optional parameter "spaces": A comma-separated list of
// spaces to query. Supported values are `drive`, `appDataFolder` and `photos`.
func (c *ChangesWatchCall) Spaces(spaces string) *ChangesWatchCall {
	c.urlParams_.Set("spaces", spaces)
	return c
}

// StartChangeId sets the optional parameter "startChangeId": Deprecated: Use
// `pageToken` instead.
func (c *ChangesWatchCall) StartChangeId(startChangeId int64) *ChangesWatchCall {
	c.urlParams_.Set("startChangeId", fmt.Sprint(startChangeId))
	return c
}

// SupportsAllDrives sets the optional parameter "supportsAllDrives": Whether
// the requesting application supports both My Drives and shared drives.
func (c *ChangesWatchCall) SupportsAllDrives(supportsAllDrives bool) *ChangesWatchCall {
	c.urlParams_.Set("supportsAllDrives", fmt.Sprint(supportsAllDrives))
	return c
}

// SupportsTeamDrives sets the optional parameter "supportsTeamDrives":
// Deprecated: Use `supportsAllDrives` instead.
func (c *ChangesWatchCall) SupportsTeamDrives(supportsTeamDrives bool) *ChangesWatchCall {
	c.urlParams_.Set("supportsTeamDrives", fmt.Sprint(supportsTeamDrives))
	return c
}

// TeamDriveId sets the optional parameter "teamDriveId": Deprecated: Use
// `driveId` instead.
func (c *ChangesWatchCall) TeamDriveId(teamDriveId string) *ChangesWatchCall {
	c.urlParams_.Set("teamDriveId", teamDriveId)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ChangesWatchCall) Fields(s ...googleapi.Field) *ChangesWatchCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ChangesWatchCall) Context(ctx context.Context) *ChangesWatchCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ChangesWatchCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ChangesWatchCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.channel)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "changes/watch")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.changes.watch", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.changes.watch" call.
// Any non-2xx status code is an error. Response headers are in either
// *Channel.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *ChangesWatchCall) Do(opts ...googleapi.CallOption) (*Channel, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Channel{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.changes.watch", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ChannelsStopCall struct {
	s          *Service
	channel    *Channel
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Stop: Stops watching resources through this channel.
func (r *ChannelsService) Stop(channel *Channel) *ChannelsStopCall {
	c := &ChannelsStopCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.channel = channel
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ChannelsStopCall) Fields(s ...googleapi.Field) *ChannelsStopCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ChannelsStopCall) Context(ctx context.Context) *ChannelsStopCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ChannelsStopCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ChannelsStopCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.channel)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "channels/stop")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.channels.stop", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.channels.stop" call.
func (c *ChannelsStopCall) Do(opts ...googleapi.CallOption) error {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if err != nil {
		return err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return gensupport.WrapError(err)
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.channels.stop", "response", internallog.HTTPResponse(res, nil))
	return nil
}

type ChildrenDeleteCall struct {
	s          *Service
	folderId   string
	childId    string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Delete: Removes a child from a folder.
//
// - childId: The ID of the child.
// - folderId: The ID of the folder.
func (r *ChildrenService) Delete(folderId string, childId string) *ChildrenDeleteCall {
	c := &ChildrenDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.folderId = folderId
	c.childId = childId
	return c
}

// EnforceSingleParent sets the optional parameter "enforceSingleParent":
// Deprecated: If an item is not in a shared drive and its last parent is
// removed, the item is placed under its owner's root.
func (c *ChildrenDeleteCall) EnforceSingleParent(enforceSingleParent bool) *ChildrenDeleteCall {
	c.urlParams_.Set("enforceSingleParent", fmt.Sprint(enforceSingleParent))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ChildrenDeleteCall) Fields(s ...googleapi.Field) *ChildrenDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ChildrenDeleteCall) Context(ctx context.Context) *ChildrenDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ChildrenDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ChildrenDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{folderId}/children/{childId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"folderId": c.folderId,
		"childId":  c.childId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.children.delete", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.children.delete" call.
func (c *ChildrenDeleteCall) Do(opts ...googleapi.CallOption) error {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if err != nil {
		return err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return gensupport.WrapError(err)
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.children.delete", "response", internallog.HTTPResponse(res, nil))
	return nil
}

type ChildrenGetCall struct {
	s            *Service
	folderId     string
	childId      string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets a specific child reference.
//
// - childId: The ID of the child.
// - folderId: The ID of the folder.
func (r *ChildrenService) Get(folderId string, childId string) *ChildrenGetCall {
	c := &ChildrenGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.folderId = folderId
	c.childId = childId
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ChildrenGetCall) Fields(s ...googleapi.Field) *ChildrenGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *ChildrenGetCall) IfNoneMatch(entityTag string) *ChildrenGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ChildrenGetCall) Context(ctx context.Context) *ChildrenGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ChildrenGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ChildrenGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{folderId}/children/{childId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"folderId": c.folderId,
		"childId":  c.childId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.children.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.children.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *ChildReference.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *ChildrenGetCall) Do(opts ...googleapi.CallOption) (*ChildReference, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ChildReference{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.children.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ChildrenInsertCall struct {
	s              *Service
	folderId       string
	childreference *ChildReference
	urlParams_     gensupport.URLParams
	ctx_           context.Context
	header_        http.Header
}

// Insert: Inserts a file into a folder.
//
// - folderId: The ID of the folder.
func (r *ChildrenService) Insert(folderId string, childreference *ChildReference) *ChildrenInsertCall {
	c := &ChildrenInsertCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.folderId = folderId
	c.childreference = childreference
	return c
}

// EnforceSingleParent sets the optional parameter "enforceSingleParent":
// Deprecated: Adding files to multiple folders is no longer supported. Use
// `shortcuts` instead.
func (c *ChildrenInsertCall) EnforceSingleParent(enforceSingleParent bool) *ChildrenInsertCall {
	c.urlParams_.Set("enforceSingleParent", fmt.Sprint(enforceSingleParent))
	return c
}

// SupportsAllDrives sets the optional parameter "supportsAllDrives": Whether
// the requesting application supports both My Drives and shared drives.
func (c *ChildrenInsertCall) SupportsAllDrives(supportsAllDrives bool) *ChildrenInsertCall {
	c.urlParams_.Set("supportsAllDrives", fmt.Sprint(supportsAllDrives))
	return c
}

// SupportsTeamDrives sets the optional parameter "supportsTeamDrives":
// Deprecated: Use `supportsAllDrives` instead.
func (c *ChildrenInsertCall) SupportsTeamDrives(supportsTeamDrives bool) *ChildrenInsertCall {
	c.urlParams_.Set("supportsTeamDrives", fmt.Sprint(supportsTeamDrives))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ChildrenInsertCall) Fields(s ...googleapi.Field) *ChildrenInsertCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ChildrenInsertCall) Context(ctx context.Context) *ChildrenInsertCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ChildrenInsertCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ChildrenInsertCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.childreference)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{folderId}/children")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"folderId": c.folderId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.children.insert", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.children.insert" call.
// Any non-2xx status code is an error. Response headers are in either
// *ChildReference.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *ChildrenInsertCall) Do(opts ...googleapi.CallOption) (*ChildReference, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ChildReference{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.children.insert", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ChildrenListCall struct {
	s            *Service
	folderId     string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists a folder's children.
//
// - folderId: The ID of the folder.
func (r *ChildrenService) List(folderId string) *ChildrenListCall {
	c := &ChildrenListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.folderId = folderId
	return c
}

// MaxResults sets the optional parameter "maxResults": Maximum number of
// children to return.
func (c *ChildrenListCall) MaxResults(maxResults int64) *ChildrenListCall {
	c.urlParams_.Set("maxResults", fmt.Sprint(maxResults))
	return c
}

// OrderBy sets the optional parameter "orderBy": A comma-separated list of
// sort keys. Valid keys are `createdDate`, `folder`, `lastViewedByMeDate`,
// `modifiedByMeDate`, `modifiedDate`, `quotaBytesUsed`, `recency`,
// `sharedWithMeDate`, `starred`, and `title`. Each key sorts ascending by
// default, but may be reversed with the `desc` modifier. Example usage:
// ?orderBy=folder,modifiedDate desc,title. Please note that there is a current
// limitation for users with approximately one million files in which the
// requested sort order is ignored.
func (c *ChildrenListCall) OrderBy(orderBy string) *ChildrenListCall {
	c.urlParams_.Set("orderBy", orderBy)
	return c
}

// PageToken sets the optional parameter "pageToken": Page token for children.
func (c *ChildrenListCall) PageToken(pageToken string) *ChildrenListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Q sets the optional parameter "q": Query string for searching children.
func (c *ChildrenListCall) Q(q string) *ChildrenListCall {
	c.urlParams_.Set("q", q)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ChildrenListCall) Fields(s ...googleapi.Field) *ChildrenListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *ChildrenListCall) IfNoneMatch(entityTag string) *ChildrenListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ChildrenListCall) Context(ctx context.Context) *ChildrenListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ChildrenListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ChildrenListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{folderId}/children")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"folderId": c.folderId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.children.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.children.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ChildList.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *ChildrenListCall) Do(opts ...googleapi.CallOption) (*ChildList, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ChildList{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.children.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *ChildrenListCall) Pages(ctx context.Context, f func(*ChildList) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type CommentsDeleteCall struct {
	s          *Service
	fileId     string
	commentId  string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Delete: Deletes a comment.
//
// - commentId: The ID of the comment.
// - fileId: The ID of the file.
func (r *CommentsService) Delete(fileId string, commentId string) *CommentsDeleteCall {
	c := &CommentsDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.commentId = commentId
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *CommentsDeleteCall) Fields(s ...googleapi.Field) *CommentsDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *CommentsDeleteCall) Context(ctx context.Context) *CommentsDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *CommentsDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *CommentsDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/comments/{commentId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId":    c.fileId,
		"commentId": c.commentId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.comments.delete", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.comments.delete" call.
func (c *CommentsDeleteCall) Do(opts ...googleapi.CallOption) error {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if err != nil {
		return err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return gensupport.WrapError(err)
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.comments.delete", "response", internallog.HTTPResponse(res, nil))
	return nil
}

type CommentsGetCall struct {
	s            *Service
	fileId       string
	commentId    string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets a comment by ID.
//
// - commentId: The ID of the comment.
// - fileId: The ID of the file.
func (r *CommentsService) Get(fileId string, commentId string) *CommentsGetCall {
	c := &CommentsGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.commentId = commentId
	return c
}

// IncludeDeleted sets the optional parameter "includeDeleted": If set, this
// will succeed when retrieving a deleted comment, and will include any deleted
// replies.
func (c *CommentsGetCall) IncludeDeleted(includeDeleted bool) *CommentsGetCall {
	c.urlParams_.Set("includeDeleted", fmt.Sprint(includeDeleted))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *CommentsGetCall) Fields(s ...googleapi.Field) *CommentsGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *CommentsGetCall) IfNoneMatch(entityTag string) *CommentsGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *CommentsGetCall) Context(ctx context.Context) *CommentsGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *CommentsGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *CommentsGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/comments/{commentId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId":    c.fileId,
		"commentId": c.commentId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.comments.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.comments.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *Comment.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *CommentsGetCall) Do(opts ...googleapi.CallOption) (*Comment, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Comment{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.comments.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type CommentsInsertCall struct {
	s          *Service
	fileId     string
	comment    *Comment
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Insert: Creates a new comment on the given file.
//
// - fileId: The ID of the file.
func (r *CommentsService) Insert(fileId string, comment *Comment) *CommentsInsertCall {
	c := &CommentsInsertCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.comment = comment
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *CommentsInsertCall) Fields(s ...googleapi.Field) *CommentsInsertCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *CommentsInsertCall) Context(ctx context.Context) *CommentsInsertCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *CommentsInsertCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *CommentsInsertCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.comment)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/comments")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId": c.fileId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.comments.insert", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.comments.insert" call.
// Any non-2xx status code is an error. Response headers are in either
// *Comment.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *CommentsInsertCall) Do(opts ...googleapi.CallOption) (*Comment, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Comment{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.comments.insert", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type CommentsListCall struct {
	s            *Service
	fileId       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists a file's comments.
//
// - fileId: The ID of the file.
func (r *CommentsService) List(fileId string) *CommentsListCall {
	c := &CommentsListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	return c
}

// IncludeDeleted sets the optional parameter "includeDeleted": If set, all
// comments and replies, including deleted comments and replies (with content
// stripped) will be returned.
func (c *CommentsListCall) IncludeDeleted(includeDeleted bool) *CommentsListCall {
	c.urlParams_.Set("includeDeleted", fmt.Sprint(includeDeleted))
	return c
}

// MaxResults sets the optional parameter "maxResults": The maximum number of
// discussions to include in the response, used for paging.
func (c *CommentsListCall) MaxResults(maxResults int64) *CommentsListCall {
	c.urlParams_.Set("maxResults", fmt.Sprint(maxResults))
	return c
}

// PageToken sets the optional parameter "pageToken": The continuation token,
// used to page through large result sets. To get the next page of results, set
// this parameter to the value of "nextPageToken" from the previous response.
func (c *CommentsListCall) PageToken(pageToken string) *CommentsListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// UpdatedMin sets the optional parameter "updatedMin": Only discussions that
// were updated after this timestamp will be returned. Formatted as an RFC 3339
// timestamp.
func (c *CommentsListCall) UpdatedMin(updatedMin string) *CommentsListCall {
	c.urlParams_.Set("updatedMin", updatedMin)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *CommentsListCall) Fields(s ...googleapi.Field) *CommentsListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *CommentsListCall) IfNoneMatch(entityTag string) *CommentsListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *CommentsListCall) Context(ctx context.Context) *CommentsListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *CommentsListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *CommentsListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/comments")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId": c.fileId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.comments.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.comments.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *CommentList.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *CommentsListCall) Do(opts ...googleapi.CallOption) (*CommentList, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &CommentList{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.comments.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *CommentsListCall) Pages(ctx context.Context, f func(*CommentList) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type CommentsPatchCall struct {
	s          *Service
	fileId     string
	commentId  string
	comment    *Comment
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Patch: Updates an existing comment.
//
// - commentId: The ID of the comment.
// - fileId: The ID of the file.
func (r *CommentsService) Patch(fileId string, commentId string, comment *Comment) *CommentsPatchCall {
	c := &CommentsPatchCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.commentId = commentId
	c.comment = comment
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *CommentsPatchCall) Fields(s ...googleapi.Field) *CommentsPatchCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *CommentsPatchCall) Context(ctx context.Context) *CommentsPatchCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *CommentsPatchCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *CommentsPatchCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.comment)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/comments/{commentId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PATCH", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId":    c.fileId,
		"commentId": c.commentId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.comments.patch", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.comments.patch" call.
// Any non-2xx status code is an error. Response headers are in either
// *Comment.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *CommentsPatchCall) Do(opts ...googleapi.CallOption) (*Comment, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Comment{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.comments.patch", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type CommentsUpdateCall struct {
	s          *Service
	fileId     string
	commentId  string
	comment    *Comment
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Update: Updates an existing comment.
//
// - commentId: The ID of the comment.
// - fileId: The ID of the file.
func (r *CommentsService) Update(fileId string, commentId string, comment *Comment) *CommentsUpdateCall {
	c := &CommentsUpdateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.commentId = commentId
	c.comment = comment
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *CommentsUpdateCall) Fields(s ...googleapi.Field) *CommentsUpdateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *CommentsUpdateCall) Context(ctx context.Context) *CommentsUpdateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *CommentsUpdateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *CommentsUpdateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.comment)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/comments/{commentId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PUT", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId":    c.fileId,
		"commentId": c.commentId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.comments.update", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.comments.update" call.
// Any non-2xx status code is an error. Response headers are in either
// *Comment.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *CommentsUpdateCall) Do(opts ...googleapi.CallOption) (*Comment, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Comment{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.comments.update", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type DrivesDeleteCall struct {
	s          *Service
	driveId    string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Delete: Permanently deletes a shared drive for which the user is an
// `organizer`. The shared drive cannot contain any untrashed items.
//
// - driveId: The ID of the shared drive.
func (r *DrivesService) Delete(driveId string) *DrivesDeleteCall {
	c := &DrivesDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.driveId = driveId
	return c
}

// AllowItemDeletion sets the optional parameter "allowItemDeletion": Whether
// any items inside the shared drive should also be deleted. This option is
// only supported when `useDomainAdminAccess` is also set to `true`.
func (c *DrivesDeleteCall) AllowItemDeletion(allowItemDeletion bool) *DrivesDeleteCall {
	c.urlParams_.Set("allowItemDeletion", fmt.Sprint(allowItemDeletion))
	return c
}

// UseDomainAdminAccess sets the optional parameter "useDomainAdminAccess":
// Issue the request as a domain administrator; if set to true, then the
// requester will be granted access if they are an administrator of the domain
// to which the shared drive belongs.
func (c *DrivesDeleteCall) UseDomainAdminAccess(useDomainAdminAccess bool) *DrivesDeleteCall {
	c.urlParams_.Set("useDomainAdminAccess", fmt.Sprint(useDomainAdminAccess))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *DrivesDeleteCall) Fields(s ...googleapi.Field) *DrivesDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *DrivesDeleteCall) Context(ctx context.Context) *DrivesDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *DrivesDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *DrivesDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "drives/{driveId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"driveId": c.driveId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.drives.delete", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.drives.delete" call.
func (c *DrivesDeleteCall) Do(opts ...googleapi.CallOption) error {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if err != nil {
		return err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return gensupport.WrapError(err)
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.drives.delete", "response", internallog.HTTPResponse(res, nil))
	return nil
}

type DrivesGetCall struct {
	s            *Service
	driveId      string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets a shared drive's metadata by ID.
//
// - driveId: The ID of the shared drive.
func (r *DrivesService) Get(driveId string) *DrivesGetCall {
	c := &DrivesGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.driveId = driveId
	return c
}

// UseDomainAdminAccess sets the optional parameter "useDomainAdminAccess":
// Issue the request as a domain administrator; if set to true, then the
// requester will be granted access if they are an administrator of the domain
// to which the shared drive belongs.
func (c *DrivesGetCall) UseDomainAdminAccess(useDomainAdminAccess bool) *DrivesGetCall {
	c.urlParams_.Set("useDomainAdminAccess", fmt.Sprint(useDomainAdminAccess))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *DrivesGetCall) Fields(s ...googleapi.Field) *DrivesGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *DrivesGetCall) IfNoneMatch(entityTag string) *DrivesGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *DrivesGetCall) Context(ctx context.Context) *DrivesGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *DrivesGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *DrivesGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "drives/{driveId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"driveId": c.driveId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.drives.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.drives.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *Drive.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *DrivesGetCall) Do(opts ...googleapi.CallOption) (*Drive, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Drive{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.drives.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type DrivesHideCall struct {
	s          *Service
	driveId    string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Hide: Hides a shared drive from the default view.
//
// - driveId: The ID of the shared drive.
func (r *DrivesService) Hide(driveId string) *DrivesHideCall {
	c := &DrivesHideCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.driveId = driveId
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *DrivesHideCall) Fields(s ...googleapi.Field) *DrivesHideCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *DrivesHideCall) Context(ctx context.Context) *DrivesHideCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *DrivesHideCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *DrivesHideCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "drives/{driveId}/hide")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"driveId": c.driveId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.drives.hide", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.drives.hide" call.
// Any non-2xx status code is an error. Response headers are in either
// *Drive.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *DrivesHideCall) Do(opts ...googleapi.CallOption) (*Drive, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Drive{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.drives.hide", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type DrivesInsertCall struct {
	s          *Service
	drive      *Drive
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Insert: Creates a new shared drive.
//
//   - requestId: An ID, such as a random UUID, which uniquely identifies this
//     user's request for idempotent creation of a shared drive. A repeated
//     request by the same user and with the same request ID will avoid creating
//     duplicates by attempting to create the same shared drive. If the shared
//     drive already exists a 409 error will be returned.
func (r *DrivesService) Insert(requestId string, drive *Drive) *DrivesInsertCall {
	c := &DrivesInsertCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.urlParams_.Set("requestId", requestId)
	c.drive = drive
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *DrivesInsertCall) Fields(s ...googleapi.Field) *DrivesInsertCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *DrivesInsertCall) Context(ctx context.Context) *DrivesInsertCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *DrivesInsertCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *DrivesInsertCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.drive)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "drives")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.drives.insert", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.drives.insert" call.
// Any non-2xx status code is an error. Response headers are in either
// *Drive.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *DrivesInsertCall) Do(opts ...googleapi.CallOption) (*Drive, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Drive{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.drives.insert", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type DrivesListCall struct {
	s            *Service
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List:  Lists the user's shared drives. This method accepts the `q`
// parameter, which is a search query combining one or more search terms. For
// more information, see the Search for shared drives
// (/workspace/drive/api/guides/search-shareddrives) guide.
func (r *DrivesService) List() *DrivesListCall {
	c := &DrivesListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	return c
}

// MaxResults sets the optional parameter "maxResults": Maximum number of
// shared drives to return per page.
func (c *DrivesListCall) MaxResults(maxResults int64) *DrivesListCall {
	c.urlParams_.Set("maxResults", fmt.Sprint(maxResults))
	return c
}

// PageToken sets the optional parameter "pageToken": Page token for shared
// drives.
func (c *DrivesListCall) PageToken(pageToken string) *DrivesListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Q sets the optional parameter "q": Query string for searching shared drives.
func (c *DrivesListCall) Q(q string) *DrivesListCall {
	c.urlParams_.Set("q", q)
	return c
}

// UseDomainAdminAccess sets the optional parameter "useDomainAdminAccess":
// Issue the request as a domain administrator; if set to true, then all shared
// drives of the domain in which the requester is an administrator are
// returned.
func (c *DrivesListCall) UseDomainAdminAccess(useDomainAdminAccess bool) *DrivesListCall {
	c.urlParams_.Set("useDomainAdminAccess", fmt.Sprint(useDomainAdminAccess))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *DrivesListCall) Fields(s ...googleapi.Field) *DrivesListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *DrivesListCall) IfNoneMatch(entityTag string) *DrivesListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *DrivesListCall) Context(ctx context.Context) *DrivesListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *DrivesListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *DrivesListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "drives")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.drives.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.drives.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *DriveList.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *DrivesListCall) Do(opts ...googleapi.CallOption) (*DriveList, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &DriveList{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.drives.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *DrivesListCall) Pages(ctx context.Context, f func(*DriveList) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type DrivesUnhideCall struct {
	s          *Service
	driveId    string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Unhide: Restores a shared drive to the default view.
//
// - driveId: The ID of the shared drive.
func (r *DrivesService) Unhide(driveId string) *DrivesUnhideCall {
	c := &DrivesUnhideCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.driveId = driveId
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *DrivesUnhideCall) Fields(s ...googleapi.Field) *DrivesUnhideCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *DrivesUnhideCall) Context(ctx context.Context) *DrivesUnhideCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *DrivesUnhideCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *DrivesUnhideCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "drives/{driveId}/unhide")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"driveId": c.driveId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.drives.unhide", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.drives.unhide" call.
// Any non-2xx status code is an error. Response headers are in either
// *Drive.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *DrivesUnhideCall) Do(opts ...googleapi.CallOption) (*Drive, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Drive{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.drives.unhide", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type DrivesUpdateCall struct {
	s          *Service
	driveId    string
	drive      *Drive
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Update: Updates the metadata for a shared drive.
//
// - driveId: The ID of the shared drive.
func (r *DrivesService) Update(driveId string, drive *Drive) *DrivesUpdateCall {
	c := &DrivesUpdateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.driveId = driveId
	c.drive = drive
	return c
}

// UseDomainAdminAccess sets the optional parameter "useDomainAdminAccess":
// Issue the request as a domain administrator; if set to true, then the
// requester will be granted access if they are an administrator of the domain
// to which the shared drive belongs.
func (c *DrivesUpdateCall) UseDomainAdminAccess(useDomainAdminAccess bool) *DrivesUpdateCall {
	c.urlParams_.Set("useDomainAdminAccess", fmt.Sprint(useDomainAdminAccess))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *DrivesUpdateCall) Fields(s ...googleapi.Field) *DrivesUpdateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *DrivesUpdateCall) Context(ctx context.Context) *DrivesUpdateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *DrivesUpdateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *DrivesUpdateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.drive)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "drives/{driveId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PUT", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"driveId": c.driveId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.drives.update", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.drives.update" call.
// Any non-2xx status code is an error. Response headers are in either
// *Drive.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *DrivesUpdateCall) Do(opts ...googleapi.CallOption) (*Drive, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Drive{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.drives.update", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type FilesCopyCall struct {
	s          *Service
	fileId     string
	file       *File
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Copy: Creates a copy of the specified file.
//
// - fileId: The ID of the file to copy.
func (r *FilesService) Copy(fileId string, file *File) *FilesCopyCall {
	c := &FilesCopyCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.file = file
	return c
}

// Convert sets the optional parameter "convert": Whether to convert this file
// to the corresponding Docs Editors format.
func (c *FilesCopyCall) Convert(convert bool) *FilesCopyCall {
	c.urlParams_.Set("convert", fmt.Sprint(convert))
	return c
}

// EnforceSingleParent sets the optional parameter "enforceSingleParent":
// Deprecated: Copying files into multiple folders is no longer supported. Use
// shortcuts instead.
func (c *FilesCopyCall) EnforceSingleParent(enforceSingleParent bool) *FilesCopyCall {
	c.urlParams_.Set("enforceSingleParent", fmt.Sprint(enforceSingleParent))
	return c
}

// IncludeLabels sets the optional parameter "includeLabels": A comma-separated
// list of IDs of labels to include in the `labelInfo` part of the response.
func (c *FilesCopyCall) IncludeLabels(includeLabels string) *FilesCopyCall {
	c.urlParams_.Set("includeLabels", includeLabels)
	return c
}

// IncludePermissionsForView sets the optional parameter
// "includePermissionsForView": Specifies which additional view's permissions
// to include in the response. Only `published` is supported.
func (c *FilesCopyCall) IncludePermissionsForView(includePermissionsForView string) *FilesCopyCall {
	c.urlParams_.Set("includePermissionsForView", includePermissionsForView)
	return c
}

// Ocr sets the optional parameter "ocr": Whether to attempt OCR on .jpg, .png,
// .gif, or .pdf uploads.
func (c *FilesCopyCall) Ocr(ocr bool) *FilesCopyCall {
	c.urlParams_.Set("ocr", fmt.Sprint(ocr))
	return c
}

// OcrLanguage sets the optional parameter "ocrLanguage": If `ocr` is true,
// hints at the language to use. Valid values are BCP 47 codes.
func (c *FilesCopyCall) OcrLanguage(ocrLanguage string) *FilesCopyCall {
	c.urlParams_.Set("ocrLanguage", ocrLanguage)
	return c
}

// Pinned sets the optional parameter "pinned": Whether to pin the head
// revision of the new copy. A file can have a maximum of 200 pinned revisions.
func (c *FilesCopyCall) Pinned(pinned bool) *FilesCopyCall {
	c.urlParams_.Set("pinned", fmt.Sprint(pinned))
	return c
}

// SupportsAllDrives sets the optional parameter "supportsAllDrives": Whether
// the requesting application supports both My Drives and shared drives.
func (c *FilesCopyCall) SupportsAllDrives(supportsAllDrives bool) *FilesCopyCall {
	c.urlParams_.Set("supportsAllDrives", fmt.Sprint(supportsAllDrives))
	return c
}

// SupportsTeamDrives sets the optional parameter "supportsTeamDrives":
// Deprecated: Use `supportsAllDrives` instead.
func (c *FilesCopyCall) SupportsTeamDrives(supportsTeamDrives bool) *FilesCopyCall {
	c.urlParams_.Set("supportsTeamDrives", fmt.Sprint(supportsTeamDrives))
	return c
}

// TimedTextLanguage sets the optional parameter "timedTextLanguage": The
// language of the timed text.
func (c *FilesCopyCall) TimedTextLanguage(timedTextLanguage string) *FilesCopyCall {
	c.urlParams_.Set("timedTextLanguage", timedTextLanguage)
	return c
}

// TimedTextTrackName sets the optional parameter "timedTextTrackName": The
// timed text track name.
func (c *FilesCopyCall) TimedTextTrackName(timedTextTrackName string) *FilesCopyCall {
	c.urlParams_.Set("timedTextTrackName", timedTextTrackName)
	return c
}

// Visibility sets the optional parameter "visibility": The visibility of the
// new file. This parameter is only relevant when the source is not a native
// Google Doc and convert=false.
//
// Possible values:
//
//	"DEFAULT" (default) - The visibility of the new file is determined by the
//
// user's default visibility/sharing policies.
//
//	"PRIVATE" - The new file will be visible to only the owner.
func (c *FilesCopyCall) Visibility(visibility string) *FilesCopyCall {
	c.urlParams_.Set("visibility", visibility)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *FilesCopyCall) Fields(s ...googleapi.Field) *FilesCopyCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *FilesCopyCall) Context(ctx context.Context) *FilesCopyCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *FilesCopyCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *FilesCopyCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.file)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/copy")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId": c.fileId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.files.copy", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.files.copy" call.
// Any non-2xx status code is an error. Response headers are in either
// *File.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *FilesCopyCall) Do(opts ...googleapi.CallOption) (*File, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &File{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.files.copy", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type FilesDeleteCall struct {
	s          *Service
	fileId     string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Delete: Permanently deletes a file owned by the user without moving it to
// the trash. If the file belongs to a shared drive, the user must be an
// `organizer` on the parent folder. If the target is a folder, all descendants
// owned by the user are also deleted.
//
// - fileId: The ID of the file to delete.
func (r *FilesService) Delete(fileId string) *FilesDeleteCall {
	c := &FilesDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	return c
}

// EnforceSingleParent sets the optional parameter "enforceSingleParent":
// Deprecated: If an item is not in a shared drive and its last parent is
// deleted but the item itself is not, the item is placed under its owner's
// root.
func (c *FilesDeleteCall) EnforceSingleParent(enforceSingleParent bool) *FilesDeleteCall {
	c.urlParams_.Set("enforceSingleParent", fmt.Sprint(enforceSingleParent))
	return c
}

// SupportsAllDrives sets the optional parameter "supportsAllDrives": Whether
// the requesting application supports both My Drives and shared drives.
func (c *FilesDeleteCall) SupportsAllDrives(supportsAllDrives bool) *FilesDeleteCall {
	c.urlParams_.Set("supportsAllDrives", fmt.Sprint(supportsAllDrives))
	return c
}

// SupportsTeamDrives sets the optional parameter "supportsTeamDrives":
// Deprecated: Use `supportsAllDrives` instead.
func (c *FilesDeleteCall) SupportsTeamDrives(supportsTeamDrives bool) *FilesDeleteCall {
	c.urlParams_.Set("supportsTeamDrives", fmt.Sprint(supportsTeamDrives))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *FilesDeleteCall) Fields(s ...googleapi.Field) *FilesDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *FilesDeleteCall) Context(ctx context.Context) *FilesDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *FilesDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *FilesDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId": c.fileId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.files.delete", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.files.delete" call.
func (c *FilesDeleteCall) Do(opts ...googleapi.CallOption) error {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if err != nil {
		return err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return gensupport.WrapError(err)
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.files.delete", "response", internallog.HTTPResponse(res, nil))
	return nil
}

type FilesEmptyTrashCall struct {
	s          *Service
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// EmptyTrash: Permanently deletes all of the user's trashed files.
func (r *FilesService) EmptyTrash() *FilesEmptyTrashCall {
	c := &FilesEmptyTrashCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	return c
}

// DriveId sets the optional parameter "driveId": If set, empties the trash of
// the provided shared drive.
func (c *FilesEmptyTrashCall) DriveId(driveId string) *FilesEmptyTrashCall {
	c.urlParams_.Set("driveId", driveId)
	return c
}

// EnforceSingleParent sets the optional parameter "enforceSingleParent":
// Deprecated: If an item is not in a shared drive and its last parent is
// deleted but the item itself is not, the item is placed under its owner's
// root.
func (c *FilesEmptyTrashCall) EnforceSingleParent(enforceSingleParent bool) *FilesEmptyTrashCall {
	c.urlParams_.Set("enforceSingleParent", fmt.Sprint(enforceSingleParent))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *FilesEmptyTrashCall) Fields(s ...googleapi.Field) *FilesEmptyTrashCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *FilesEmptyTrashCall) Context(ctx context.Context) *FilesEmptyTrashCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *FilesEmptyTrashCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *FilesEmptyTrashCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/trash")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.files.emptyTrash", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.files.emptyTrash" call.
func (c *FilesEmptyTrashCall) Do(opts ...googleapi.CallOption) error {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if err != nil {
		return err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return gensupport.WrapError(err)
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.files.emptyTrash", "response", internallog.HTTPResponse(res, nil))
	return nil
}

type FilesExportCall struct {
	s            *Service
	fileId       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Export: Exports a Google Workspace document to the requested MIME type and
// returns exported byte content. Note that the exported content is limited to
// 10MB.
//
// - fileId: The ID of the file.
// - mimeType: The MIME type of the format requested for this export.
func (r *FilesService) Export(fileId string, mimeType string) *FilesExportCall {
	c := &FilesExportCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.urlParams_.Set("mimeType", mimeType)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *FilesExportCall) Fields(s ...googleapi.Field) *FilesExportCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *FilesExportCall) IfNoneMatch(entityTag string) *FilesExportCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do and Download methods.
func (c *FilesExportCall) Context(ctx context.Context) *FilesExportCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *FilesExportCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *FilesExportCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/export")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId": c.fileId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.files.export", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Download fetches the API endpoint's "media" value, instead of the normal
// API response value. If the returned error is nil, the Response is guaranteed to
// have a 2xx status code. Callers must close the Response.Body as usual.
func (c *FilesExportCall) Download(opts ...googleapi.CallOption) (*http.Response, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("media")
	if err != nil {
		return nil, err
	}
	if err := googleapi.CheckResponse(res); err != nil {
		res.Body.Close()
		return nil, gensupport.WrapError(err)
	}
	return res, nil
}

// Do executes the "drive.files.export" call.
func (c *FilesExportCall) Do(opts ...googleapi.CallOption) error {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if err != nil {
		return err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return gensupport.WrapError(err)
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.files.export", "response", internallog.HTTPResponse(res, nil))
	return nil
}

type FilesGenerateIdsCall struct {
	s            *Service
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// GenerateIds: Generates a set of file IDs which can be provided in insert or
// copy requests.
func (r *FilesService) GenerateIds() *FilesGenerateIdsCall {
	c := &FilesGenerateIdsCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	return c
}

// MaxResults sets the optional parameter "maxResults": Maximum number of IDs
// to return.
func (c *FilesGenerateIdsCall) MaxResults(maxResults int64) *FilesGenerateIdsCall {
	c.urlParams_.Set("maxResults", fmt.Sprint(maxResults))
	return c
}

// Space sets the optional parameter "space": The space in which the IDs can be
// used to create new files. Supported values are `drive` and `appDataFolder`.
// (Default: `drive`)
func (c *FilesGenerateIdsCall) Space(space string) *FilesGenerateIdsCall {
	c.urlParams_.Set("space", space)
	return c
}

// Type sets the optional parameter "type": The type of items which the IDs can
// be used for. Supported values are `files` and `shortcuts`. Note that
// `shortcuts` are only supported in the `drive` `space`. (Default: `files`)
func (c *FilesGenerateIdsCall) Type(type_ string) *FilesGenerateIdsCall {
	c.urlParams_.Set("type", type_)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *FilesGenerateIdsCall) Fields(s ...googleapi.Field) *FilesGenerateIdsCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *FilesGenerateIdsCall) IfNoneMatch(entityTag string) *FilesGenerateIdsCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *FilesGenerateIdsCall) Context(ctx context.Context) *FilesGenerateIdsCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *FilesGenerateIdsCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *FilesGenerateIdsCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/generateIds")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.files.generateIds", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.files.generateIds" call.
// Any non-2xx status code is an error. Response headers are in either
// *GeneratedIds.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *FilesGenerateIdsCall) Do(opts ...googleapi.CallOption) (*GeneratedIds, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &GeneratedIds{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.files.generateIds", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type FilesGetCall struct {
	s            *Service
	fileId       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get:  Gets a file's metadata or content by ID. If you provide the URL
// parameter `alt=media`, then the response includes the file contents in the
// response body. Downloading content with `alt=media` only works if the file
// is stored in Drive. To download Google Docs, Sheets, and Slides use
// `files.export` (/workspace/drive/api/reference/rest/v2/files/export)
// instead. For more information, see Download & export files
// (/workspace/drive/api/guides/manage-downloads).
//
// - fileId: The ID for the file in question.
func (r *FilesService) Get(fileId string) *FilesGetCall {
	c := &FilesGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	return c
}

// AcknowledgeAbuse sets the optional parameter "acknowledgeAbuse": Whether the
// user is acknowledging the risk of downloading known malware or other abusive
// files. This is only applicable when the `alt` parameter is set to `media`
// and the user is the owner of the file or an organizer of the shared drive in
// which the file resides.
func (c *FilesGetCall) AcknowledgeAbuse(acknowledgeAbuse bool) *FilesGetCall {
	c.urlParams_.Set("acknowledgeAbuse", fmt.Sprint(acknowledgeAbuse))
	return c
}

// IncludeLabels sets the optional parameter "includeLabels": A comma-separated
// list of IDs of labels to include in the `labelInfo` part of the response.
func (c *FilesGetCall) IncludeLabels(includeLabels string) *FilesGetCall {
	c.urlParams_.Set("includeLabels", includeLabels)
	return c
}

// IncludePermissionsForView sets the optional parameter
// "includePermissionsForView": Specifies which additional view's permissions
// to include in the response. Only `published` is supported.
func (c *FilesGetCall) IncludePermissionsForView(includePermissionsForView string) *FilesGetCall {
	c.urlParams_.Set("includePermissionsForView", includePermissionsForView)
	return c
}

// Projection sets the optional parameter "projection": Deprecated: This
// parameter has no function.
//
// Possible values:
//
//	"BASIC" - Deprecated.
//	"FULL" - Deprecated.
func (c *FilesGetCall) Projection(projection string) *FilesGetCall {
	c.urlParams_.Set("projection", projection)
	return c
}

// RevisionId sets the optional parameter "revisionId": Specifies the Revision
// ID that should be downloaded. Ignored unless alt=media is specified.
func (c *FilesGetCall) RevisionId(revisionId string) *FilesGetCall {
	c.urlParams_.Set("revisionId", revisionId)
	return c
}

// SupportsAllDrives sets the optional parameter "supportsAllDrives": Whether
// the requesting application supports both My Drives and shared drives.
func (c *FilesGetCall) SupportsAllDrives(supportsAllDrives bool) *FilesGetCall {
	c.urlParams_.Set("supportsAllDrives", fmt.Sprint(supportsAllDrives))
	return c
}

// SupportsTeamDrives sets the optional parameter "supportsTeamDrives":
// Deprecated: Use `supportsAllDrives` instead.
func (c *FilesGetCall) SupportsTeamDrives(supportsTeamDrives bool) *FilesGetCall {
	c.urlParams_.Set("supportsTeamDrives", fmt.Sprint(supportsTeamDrives))
	return c
}

// UpdateViewedDate sets the optional parameter "updateViewedDate": Deprecated:
// Use `files.update` with `modifiedDateBehavior=noChange,
// updateViewedDate=true` and an empty request body.
func (c *FilesGetCall) UpdateViewedDate(updateViewedDate bool) *FilesGetCall {
	c.urlParams_.Set("updateViewedDate", fmt.Sprint(updateViewedDate))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *FilesGetCall) Fields(s ...googleapi.Field) *FilesGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *FilesGetCall) IfNoneMatch(entityTag string) *FilesGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do and Download methods.
func (c *FilesGetCall) Context(ctx context.Context) *FilesGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *FilesGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *FilesGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId": c.fileId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.files.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Download fetches the API endpoint's "media" value, instead of the normal
// API response value. If the returned error is nil, the Response is guaranteed to
// have a 2xx status code. Callers must close the Response.Body as usual.
func (c *FilesGetCall) Download(opts ...googleapi.CallOption) (*http.Response, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("media")
	if err != nil {
		return nil, err
	}
	if err := googleapi.CheckResponse(res); err != nil {
		res.Body.Close()
		return nil, gensupport.WrapError(err)
	}
	return res, nil
}

// Do executes the "drive.files.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *File.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *FilesGetCall) Do(opts ...googleapi.CallOption) (*File, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &File{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.files.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type FilesInsertCall struct {
	s          *Service
	file       *File
	urlParams_ gensupport.URLParams
	mediaInfo_ *gensupport.MediaInfo
	ctx_       context.Context
	header_    http.Header
}

// Insert:  Inserts a new file. This method supports an */upload* URI and
// accepts uploaded media with the following characteristics: - *Maximum file
// size:* 5,120 GB - *Accepted Media MIME types:*`*/*` Note: Specify a valid
// MIME type, rather than the literal `*/*` value. The literal `*/*` is only
// used to indicate that any valid MIME type can be uploaded. For more
// information on uploading files, see Upload file data
// (/workspace/drive/api/guides/manage-uploads). Apps creating shortcuts with
// `files.insert` must specify the MIME type
// `application/vnd.google-apps.shortcut`. Apps should specify a file extension
// in the `title` property when inserting files with the API. For example, an
// operation to insert a JPEG file should specify something like "title":
// "cat.jpg" in the metadata. Subsequent `GET` requests include the read-only
// `fileExtension` property populated with the extension originally specified
// in the `title` property. When a Google Drive user requests to download a
// file, or when the file is downloaded through the sync client, Drive builds a
// full filename (with extension) based on the title. In cases where the
// extension is missing, Drive attempts to determine the extension based on the
// file's MIME type.
func (r *FilesService) Insert(file *File) *FilesInsertCall {
	c := &FilesInsertCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.file = file
	return c
}

// Convert sets the optional parameter "convert": Whether to convert this file
// to the corresponding Docs Editors format.
func (c *FilesInsertCall) Convert(convert bool) *FilesInsertCall {
	c.urlParams_.Set("convert", fmt.Sprint(convert))
	return c
}

// EnforceSingleParent sets the optional parameter "enforceSingleParent":
// Deprecated: Creating files in multiple folders is no longer supported.
func (c *FilesInsertCall) EnforceSingleParent(enforceSingleParent bool) *FilesInsertCall {
	c.urlParams_.Set("enforceSingleParent", fmt.Sprint(enforceSingleParent))
	return c
}

// IncludeLabels sets the optional parameter "includeLabels": A comma-separated
// list of IDs of labels to include in the `labelInfo` part of the response.
func (c *FilesInsertCall) IncludeLabels(includeLabels string) *FilesInsertCall {
	c.urlParams_.Set("includeLabels", includeLabels)
	return c
}

// IncludePermissionsForView sets the optional parameter
// "includePermissionsForView": Specifies which additional view's permissions
// to include in the response. Only `published` is supported.
func (c *FilesInsertCall) IncludePermissionsForView(includePermissionsForView string) *FilesInsertCall {
	c.urlParams_.Set("includePermissionsForView", includePermissionsForView)
	return c
}

// Ocr sets the optional parameter "ocr": Whether to attempt OCR on .jpg, .png,
// .gif, or .pdf uploads.
func (c *FilesInsertCall) Ocr(ocr bool) *FilesInsertCall {
	c.urlParams_.Set("ocr", fmt.Sprint(ocr))
	return c
}

// OcrLanguage sets the optional parameter "ocrLanguage": If ocr is true, hints
// at the language to use. Valid values are BCP 47 codes.
func (c *FilesInsertCall) OcrLanguage(ocrLanguage string) *FilesInsertCall {
	c.urlParams_.Set("ocrLanguage", ocrLanguage)
	return c
}

// Pinned sets the optional parameter "pinned": Whether to pin the head
// revision of the uploaded file. A file can have a maximum of 200 pinned
// revisions.
func (c *FilesInsertCall) Pinned(pinned bool) *FilesInsertCall {
	c.urlParams_.Set("pinned", fmt.Sprint(pinned))
	return c
}

// SupportsAllDrives sets the optional parameter "supportsAllDrives": Whether
// the requesting application supports both My Drives and shared drives.
func (c *FilesInsertCall) SupportsAllDrives(supportsAllDrives bool) *FilesInsertCall {
	c.urlParams_.Set("supportsAllDrives", fmt.Sprint(supportsAllDrives))
	return c
}

// SupportsTeamDrives sets the optional parameter "supportsTeamDrives":
// Deprecated: Use `supportsAllDrives` instead.
func (c *FilesInsertCall) SupportsTeamDrives(supportsTeamDrives bool) *FilesInsertCall {
	c.urlParams_.Set("supportsTeamDrives", fmt.Sprint(supportsTeamDrives))
	return c
}

// TimedTextLanguage sets the optional parameter "timedTextLanguage": The
// language of the timed text.
func (c *FilesInsertCall) TimedTextLanguage(timedTextLanguage string) *FilesInsertCall {
	c.urlParams_.Set("timedTextLanguage", timedTextLanguage)
	return c
}

// TimedTextTrackName sets the optional parameter "timedTextTrackName": The
// timed text track name.
func (c *FilesInsertCall) TimedTextTrackName(timedTextTrackName string) *FilesInsertCall {
	c.urlParams_.Set("timedTextTrackName", timedTextTrackName)
	return c
}

// UseContentAsIndexableText sets the optional parameter
// "useContentAsIndexableText": Whether to use the content as indexable text.
func (c *FilesInsertCall) UseContentAsIndexableText(useContentAsIndexableText bool) *FilesInsertCall {
	c.urlParams_.Set("useContentAsIndexableText", fmt.Sprint(useContentAsIndexableText))
	return c
}

// Visibility sets the optional parameter "visibility": The visibility of the
// new file. This parameter is only relevant when convert=false.
//
// Possible values:
//
//	"DEFAULT" (default) - The visibility of the new file is determined by the
//
// user's default visibility/sharing policies.
//
//	"PRIVATE" - The new file will be visible to only the owner.
func (c *FilesInsertCall) Visibility(visibility string) *FilesInsertCall {
	c.urlParams_.Set("visibility", visibility)
	return c
}

// Media specifies the media to upload in one or more chunks. The chunk size
// may be controlled by supplying a MediaOption generated by
// googleapi.ChunkSize. The chunk size defaults to
// googleapi.DefaultUploadChunkSize.The Content-Type header used in the upload
// request will be determined by sniffing the contents of r, unless a
// MediaOption generated by googleapi.ContentType is supplied.
// At most one of Media and ResumableMedia may be set.
func (c *FilesInsertCall) Media(r io.Reader, options ...googleapi.MediaOption) *FilesInsertCall {
	c.mediaInfo_ = gensupport.NewInfoFromMedia(r, options)
	return c
}

// ResumableMedia specifies the media to upload in chunks and can be canceled
// with ctx.
//
// Deprecated: use Media instead.
//
// At most one of Media and ResumableMedia may be set. mediaType identifies the
// MIME media type of the upload, such as "image/png". If mediaType is "", it
// will be auto-detected. The provided ctx will supersede any context
// previously provided to the Context method.
func (c *FilesInsertCall) ResumableMedia(ctx context.Context, r io.ReaderAt, size int64, mediaType string) *FilesInsertCall {
	c.ctx_ = ctx
	c.mediaInfo_ = gensupport.NewInfoFromResumableMedia(r, size, mediaType)
	return c
}

// ProgressUpdater provides a callback function that will be called after every
// chunk. It should be a low-latency function in order to not slow down the
// upload operation. This should only be called when using ResumableMedia (as
// opposed to Media).
func (c *FilesInsertCall) ProgressUpdater(pu googleapi.ProgressUpdater) *FilesInsertCall {
	c.mediaInfo_.SetProgressUpdater(pu)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *FilesInsertCall) Fields(s ...googleapi.Field) *FilesInsertCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
// This context will supersede any context previously provided to the
// ResumableMedia method.
func (c *FilesInsertCall) Context(ctx context.Context) *FilesInsertCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *FilesInsertCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *FilesInsertCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.file)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files")
	if c.mediaInfo_ != nil {
		urls = googleapi.ResolveRelative(c.s.BasePath, "/upload/drive/v2/files")
		c.urlParams_.Set("uploadType", c.mediaInfo_.UploadType())
	}
	newBody, getBody, cleanup := c.mediaInfo_.UploadRequest(reqHeaders, body)
	defer cleanup()
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, newBody)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	req.GetBody = getBody
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.files.insert", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.files.insert" call.
// Any non-2xx status code is an error. Response headers are in either
// *File.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *FilesInsertCall) Do(opts ...googleapi.CallOption) (*File, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	rx := c.mediaInfo_.ResumableUpload(res.Header.Get("Location"))
	if rx != nil {
		rx.Client = c.s.client
		rx.UserAgent = c.s.userAgent()
		ctx := c.ctx_
		if ctx == nil {
			ctx = context.TODO()
		}
		res, err = rx.Upload(ctx)
		if err != nil {
			return nil, err
		}
		defer res.Body.Close()
		if err := googleapi.CheckResponse(res); err != nil {
			return nil, gensupport.WrapError(err)
		}
	}
	ret := &File{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.files.insert", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type FilesListCall struct {
	s            *Service
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List:  Lists the user's files. This method accepts the `q` parameter, which
// is a search query combining one or more search terms. For more information,
// see the Search for files & folders
// (/workspace/drive/api/guides/search-files) guide. *Note:* This method
// returns *all* files by default, including trashed files. If you don't want
// trashed files to appear in the list, use the `trashed=false` query parameter
// to remove trashed files from the results.
func (r *FilesService) List() *FilesListCall {
	c := &FilesListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	return c
}

// Corpora sets the optional parameter "corpora": Bodies of items
// (files/documents) to which the query applies. Supported bodies are
// `default`, `domain`, `drive` and `allDrives`. Prefer `default` or `drive` to
// `allDrives` for efficiency.
func (c *FilesListCall) Corpora(corpora string) *FilesListCall {
	c.urlParams_.Set("corpora", corpora)
	return c
}

// Corpus sets the optional parameter "corpus": Deprecated: The body of items
// (files/documents) to which the query applies. Use `corpora` instead.
//
// Possible values:
//
//	"DEFAULT" - The items that the user has accessed.
//	"DOMAIN" - Items shared to the user's domain.
func (c *FilesListCall) Corpus(corpus string) *FilesListCall {
	c.urlParams_.Set("corpus", corpus)
	return c
}

// DriveId sets the optional parameter "driveId": ID of the shared drive to
// search.
func (c *FilesListCall) DriveId(driveId string) *FilesListCall {
	c.urlParams_.Set("driveId", driveId)
	return c
}

// IncludeItemsFromAllDrives sets the optional parameter
// "includeItemsFromAllDrives": Whether both My Drive and shared drive items
// should be included in results.
func (c *FilesListCall) IncludeItemsFromAllDrives(includeItemsFromAllDrives bool) *FilesListCall {
	c.urlParams_.Set("includeItemsFromAllDrives", fmt.Sprint(includeItemsFromAllDrives))
	return c
}

// IncludeLabels sets the optional parameter "includeLabels": A comma-separated
// list of IDs of labels to include in the `labelInfo` part of the response.
func (c *FilesListCall) IncludeLabels(includeLabels string) *FilesListCall {
	c.urlParams_.Set("includeLabels", includeLabels)
	return c
}

// IncludePermissionsForView sets the optional parameter
// "includePermissionsForView": Specifies which additional view's permissions
// to include in the response. Only `published` is supported.
func (c *FilesListCall) IncludePermissionsForView(includePermissionsForView string) *FilesListCall {
	c.urlParams_.Set("includePermissionsForView", includePermissionsForView)
	return c
}

// IncludeTeamDriveItems sets the optional parameter "includeTeamDriveItems":
// Deprecated: Use `includeItemsFromAllDrives` instead.
func (c *FilesListCall) IncludeTeamDriveItems(includeTeamDriveItems bool) *FilesListCall {
	c.urlParams_.Set("includeTeamDriveItems", fmt.Sprint(includeTeamDriveItems))
	return c
}

// MaxResults sets the optional parameter "maxResults": The maximum number of
// files to return per page. Partial or empty result pages are possible even
// before the end of the files list has been reached.
func (c *FilesListCall) MaxResults(maxResults int64) *FilesListCall {
	c.urlParams_.Set("maxResults", fmt.Sprint(maxResults))
	return c
}

// OrderBy sets the optional parameter "orderBy": A comma-separated list of
// sort keys. Valid keys are: * `createdDate`: When the file was created. *
// `folder`: The folder ID. This field is sorted using alphabetical ordering. *
// `lastViewedByMeDate`: The last time the file was viewed by the user. *
// `modifiedByMeDate`: The last time the file was modified by the user. *
// `modifiedDate`: The last time the file was modified by anyone. *
// `quotaBytesUsed`: The number of storage quota bytes used by the file. *
// `recency`: The most recent timestamp from the file's date-time fields. *
// `sharedWithMeDate`: When the file was shared with the user, if applicable. *
// `starred`: Whether the user has starred the file. * `title`: The title of
// the file. This field is sorted using alphabetical ordering, so 1, 12, 2, 22.
// * `title_natural`: The title of the file. This field is sorted using natural
// sort ordering, so 1, 2, 12, 22. Each key sorts ascending by default, but can
// be reversed with the 'desc' modifier. Example usage:
// `?orderBy=folder,modifiedDate desc,title`. Note that there's a current
// limitation for users with approximately one million files in which the
// requested sort order is ignored.
func (c *FilesListCall) OrderBy(orderBy string) *FilesListCall {
	c.urlParams_.Set("orderBy", orderBy)
	return c
}

// PageToken sets the optional parameter "pageToken": Page token for files.
func (c *FilesListCall) PageToken(pageToken string) *FilesListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Projection sets the optional parameter "projection": Deprecated: This
// parameter has no function.
//
// Possible values:
//
//	"BASIC" - Deprecated.
//	"FULL" - Deprecated.
func (c *FilesListCall) Projection(projection string) *FilesListCall {
	c.urlParams_.Set("projection", projection)
	return c
}

// Q sets the optional parameter "q": Query string for searching files.
func (c *FilesListCall) Q(q string) *FilesListCall {
	c.urlParams_.Set("q", q)
	return c
}

// Spaces sets the optional parameter "spaces": A comma-separated list of
// spaces to query. Supported values are `drive`, and `appDataFolder`.
func (c *FilesListCall) Spaces(spaces string) *FilesListCall {
	c.urlParams_.Set("spaces", spaces)
	return c
}

// SupportsAllDrives sets the optional parameter "supportsAllDrives": Whether
// the requesting application supports both My Drives and shared drives.
func (c *FilesListCall) SupportsAllDrives(supportsAllDrives bool) *FilesListCall {
	c.urlParams_.Set("supportsAllDrives", fmt.Sprint(supportsAllDrives))
	return c
}

// SupportsTeamDrives sets the optional parameter "supportsTeamDrives":
// Deprecated: Use `supportsAllDrives` instead.
func (c *FilesListCall) SupportsTeamDrives(supportsTeamDrives bool) *FilesListCall {
	c.urlParams_.Set("supportsTeamDrives", fmt.Sprint(supportsTeamDrives))
	return c
}

// TeamDriveId sets the optional parameter "teamDriveId": Deprecated: Use
// `driveId` instead.
func (c *FilesListCall) TeamDriveId(teamDriveId string) *FilesListCall {
	c.urlParams_.Set("teamDriveId", teamDriveId)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *FilesListCall) Fields(s ...googleapi.Field) *FilesListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *FilesListCall) IfNoneMatch(entityTag string) *FilesListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *FilesListCall) Context(ctx context.Context) *FilesListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *FilesListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *FilesListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.files.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.files.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *FileList.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *FilesListCall) Do(opts ...googleapi.CallOption) (*FileList, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &FileList{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.files.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *FilesListCall) Pages(ctx context.Context, f func(*FileList) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type FilesListLabelsCall struct {
	s            *Service
	fileId       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// ListLabels: Lists the labels on a file.
//
// - fileId: The ID for the file.
func (r *FilesService) ListLabels(fileId string) *FilesListLabelsCall {
	c := &FilesListLabelsCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	return c
}

// MaxResults sets the optional parameter "maxResults": The maximum number of
// labels to return per page. When not set, defaults to 100.
func (c *FilesListLabelsCall) MaxResults(maxResults int64) *FilesListLabelsCall {
	c.urlParams_.Set("maxResults", fmt.Sprint(maxResults))
	return c
}

// PageToken sets the optional parameter "pageToken": The token for continuing
// a previous list request on the next page. This should be set to the value of
// `nextPageToken` from the previous response.
func (c *FilesListLabelsCall) PageToken(pageToken string) *FilesListLabelsCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *FilesListLabelsCall) Fields(s ...googleapi.Field) *FilesListLabelsCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *FilesListLabelsCall) IfNoneMatch(entityTag string) *FilesListLabelsCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *FilesListLabelsCall) Context(ctx context.Context) *FilesListLabelsCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *FilesListLabelsCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *FilesListLabelsCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/listLabels")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId": c.fileId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.files.listLabels", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.files.listLabels" call.
// Any non-2xx status code is an error. Response headers are in either
// *LabelList.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *FilesListLabelsCall) Do(opts ...googleapi.CallOption) (*LabelList, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &LabelList{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.files.listLabels", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *FilesListLabelsCall) Pages(ctx context.Context, f func(*LabelList) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type FilesModifyLabelsCall struct {
	s                   *Service
	fileId              string
	modifylabelsrequest *ModifyLabelsRequest
	urlParams_          gensupport.URLParams
	ctx_                context.Context
	header_             http.Header
}

// ModifyLabels: Modifies the set of labels applied to a file. Returns a list
// of the labels that were added or modified.
//
// - fileId: The ID of the file to which the labels belong.
func (r *FilesService) ModifyLabels(fileId string, modifylabelsrequest *ModifyLabelsRequest) *FilesModifyLabelsCall {
	c := &FilesModifyLabelsCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.modifylabelsrequest = modifylabelsrequest
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *FilesModifyLabelsCall) Fields(s ...googleapi.Field) *FilesModifyLabelsCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *FilesModifyLabelsCall) Context(ctx context.Context) *FilesModifyLabelsCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *FilesModifyLabelsCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *FilesModifyLabelsCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.modifylabelsrequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/modifyLabels")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId": c.fileId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.files.modifyLabels", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.files.modifyLabels" call.
// Any non-2xx status code is an error. Response headers are in either
// *ModifyLabelsResponse.ServerResponse.Header or (if a response was returned
// at all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *FilesModifyLabelsCall) Do(opts ...googleapi.CallOption) (*ModifyLabelsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ModifyLabelsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.files.modifyLabels", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type FilesPatchCall struct {
	s          *Service
	fileId     string
	file       *File
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Patch: Updates a file's metadata and/or content. When calling this method,
// only populate fields in the request that you want to modify. When updating
// fields, some fields might change automatically, such as modifiedDate. This
// method supports patch semantics.
//
// - fileId: The ID of the file to update.
func (r *FilesService) Patch(fileId string, file *File) *FilesPatchCall {
	c := &FilesPatchCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.file = file
	return c
}

// AddParents sets the optional parameter "addParents": Comma-separated list of
// parent IDs to add.
func (c *FilesPatchCall) AddParents(addParents string) *FilesPatchCall {
	c.urlParams_.Set("addParents", addParents)
	return c
}

// Convert sets the optional parameter "convert": Deprecated: This parameter
// has no function.
func (c *FilesPatchCall) Convert(convert bool) *FilesPatchCall {
	c.urlParams_.Set("convert", fmt.Sprint(convert))
	return c
}

// EnforceSingleParent sets the optional parameter "enforceSingleParent":
// Deprecated: Adding files to multiple folders is no longer supported. Use
// `shortcuts` instead.
func (c *FilesPatchCall) EnforceSingleParent(enforceSingleParent bool) *FilesPatchCall {
	c.urlParams_.Set("enforceSingleParent", fmt.Sprint(enforceSingleParent))
	return c
}

// IncludeLabels sets the optional parameter "includeLabels": A comma-separated
// list of IDs of labels to include in the `labelInfo` part of the response.
func (c *FilesPatchCall) IncludeLabels(includeLabels string) *FilesPatchCall {
	c.urlParams_.Set("includeLabels", includeLabels)
	return c
}

// IncludePermissionsForView sets the optional parameter
// "includePermissionsForView": Specifies which additional view's permissions
// to include in the response. Only `published` is supported.
func (c *FilesPatchCall) IncludePermissionsForView(includePermissionsForView string) *FilesPatchCall {
	c.urlParams_.Set("includePermissionsForView", includePermissionsForView)
	return c
}

// ModifiedDateBehavior sets the optional parameter "modifiedDateBehavior":
// Determines the behavior in which `modifiedDate` is updated. This overrides
// `setModifiedDate`.
//
// Possible values:
//
//	"fromBody" - Set `modifiedDate` to the value provided in the body of the
//
// request. No change if no value was provided.
//
//	"fromBodyIfNeeded" - Set `modifiedDate` to the value provided in the body
//
// of the request depending on other contents of the update.
//
//	"fromBodyOrNow" - Set modifiedDate to the value provided in the body of
//
// the request, or to the current time if no value was provided.
//
//	"noChange" - Maintain the previous value of `modifiedDate`.
//	"now" - Set `modifiedDate` to the current time.
//	"nowIfNeeded" - Set `modifiedDate` to the current time depending on
//
// contents of the update.
func (c *FilesPatchCall) ModifiedDateBehavior(modifiedDateBehavior string) *FilesPatchCall {
	c.urlParams_.Set("modifiedDateBehavior", modifiedDateBehavior)
	return c
}

// NewRevision sets the optional parameter "newRevision": Whether a blob upload
// should create a new revision. If false, the blob data in the current head
// revision is replaced. If true or not set, a new blob is created as head
// revision, and previous unpinned revisions are preserved for a short period
// of time. Pinned revisions are stored indefinitely, using additional storage
// quota, up to a maximum of 200 revisions. For details on how revisions are
// retained, see the Drive Help Center
// (https://support.google.com/drive/answer/2409045). Note that this field is
// ignored if there is no payload in the request.
func (c *FilesPatchCall) NewRevision(newRevision bool) *FilesPatchCall {
	c.urlParams_.Set("newRevision", fmt.Sprint(newRevision))
	return c
}

// Ocr sets the optional parameter "ocr": Whether to attempt OCR on .jpg, .png,
// .gif, or .pdf uploads.
func (c *FilesPatchCall) Ocr(ocr bool) *FilesPatchCall {
	c.urlParams_.Set("ocr", fmt.Sprint(ocr))
	return c
}

// OcrLanguage sets the optional parameter "ocrLanguage": If ocr is true, hints
// at the language to use. Valid values are BCP 47 codes.
func (c *FilesPatchCall) OcrLanguage(ocrLanguage string) *FilesPatchCall {
	c.urlParams_.Set("ocrLanguage", ocrLanguage)
	return c
}

// Pinned sets the optional parameter "pinned": Whether to pin the new
// revision. A file can have a maximum of 200 pinned revisions. Note that this
// field is ignored if there is no payload in the request.
func (c *FilesPatchCall) Pinned(pinned bool) *FilesPatchCall {
	c.urlParams_.Set("pinned", fmt.Sprint(pinned))
	return c
}

// RemoveParents sets the optional parameter "removeParents": Comma-separated
// list of parent IDs to remove.
func (c *FilesPatchCall) RemoveParents(removeParents string) *FilesPatchCall {
	c.urlParams_.Set("removeParents", removeParents)
	return c
}

// SetModifiedDate sets the optional parameter "setModifiedDate": Whether to
// set the modified date using the value supplied in the request body. Setting
// this field to `true` is equivalent to `modifiedDateBehavior=fromBodyOrNow`,
// and `false` is equivalent to `modifiedDateBehavior=now`. To prevent any
// changes to the modified date set `modifiedDateBehavior=noChange`.
func (c *FilesPatchCall) SetModifiedDate(setModifiedDate bool) *FilesPatchCall {
	c.urlParams_.Set("setModifiedDate", fmt.Sprint(setModifiedDate))
	return c
}

// SupportsAllDrives sets the optional parameter "supportsAllDrives": Whether
// the requesting application supports both My Drives and shared drives.
func (c *FilesPatchCall) SupportsAllDrives(supportsAllDrives bool) *FilesPatchCall {
	c.urlParams_.Set("supportsAllDrives", fmt.Sprint(supportsAllDrives))
	return c
}

// SupportsTeamDrives sets the optional parameter "supportsTeamDrives":
// Deprecated: Use `supportsAllDrives` instead.
func (c *FilesPatchCall) SupportsTeamDrives(supportsTeamDrives bool) *FilesPatchCall {
	c.urlParams_.Set("supportsTeamDrives", fmt.Sprint(supportsTeamDrives))
	return c
}

// TimedTextLanguage sets the optional parameter "timedTextLanguage": The
// language of the timed text.
func (c *FilesPatchCall) TimedTextLanguage(timedTextLanguage string) *FilesPatchCall {
	c.urlParams_.Set("timedTextLanguage", timedTextLanguage)
	return c
}

// TimedTextTrackName sets the optional parameter "timedTextTrackName": The
// timed text track name.
func (c *FilesPatchCall) TimedTextTrackName(timedTextTrackName string) *FilesPatchCall {
	c.urlParams_.Set("timedTextTrackName", timedTextTrackName)
	return c
}

// UpdateViewedDate sets the optional parameter "updateViewedDate": Whether to
// update the view date after successfully updating the file.
func (c *FilesPatchCall) UpdateViewedDate(updateViewedDate bool) *FilesPatchCall {
	c.urlParams_.Set("updateViewedDate", fmt.Sprint(updateViewedDate))
	return c
}

// UseContentAsIndexableText sets the optional parameter
// "useContentAsIndexableText": Whether to use the content as indexable text.
func (c *FilesPatchCall) UseContentAsIndexableText(useContentAsIndexableText bool) *FilesPatchCall {
	c.urlParams_.Set("useContentAsIndexableText", fmt.Sprint(useContentAsIndexableText))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *FilesPatchCall) Fields(s ...googleapi.Field) *FilesPatchCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *FilesPatchCall) Context(ctx context.Context) *FilesPatchCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *FilesPatchCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *FilesPatchCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.file)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PATCH", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId": c.fileId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.files.patch", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.files.patch" call.
// Any non-2xx status code is an error. Response headers are in either
// *File.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *FilesPatchCall) Do(opts ...googleapi.CallOption) (*File, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &File{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.files.patch", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type FilesTouchCall struct {
	s          *Service
	fileId     string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Touch: Set the file's updated time to the current server time.
//
// - fileId: The ID of the file to update.
func (r *FilesService) Touch(fileId string) *FilesTouchCall {
	c := &FilesTouchCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	return c
}

// IncludeLabels sets the optional parameter "includeLabels": A comma-separated
// list of IDs of labels to include in the `labelInfo` part of the response.
func (c *FilesTouchCall) IncludeLabels(includeLabels string) *FilesTouchCall {
	c.urlParams_.Set("includeLabels", includeLabels)
	return c
}

// IncludePermissionsForView sets the optional parameter
// "includePermissionsForView": Specifies which additional view's permissions
// to include in the response. Only `published` is supported.
func (c *FilesTouchCall) IncludePermissionsForView(includePermissionsForView string) *FilesTouchCall {
	c.urlParams_.Set("includePermissionsForView", includePermissionsForView)
	return c
}

// SupportsAllDrives sets the optional parameter "supportsAllDrives": Whether
// the requesting application supports both My Drives and shared drives.
func (c *FilesTouchCall) SupportsAllDrives(supportsAllDrives bool) *FilesTouchCall {
	c.urlParams_.Set("supportsAllDrives", fmt.Sprint(supportsAllDrives))
	return c
}

// SupportsTeamDrives sets the optional parameter "supportsTeamDrives":
// Deprecated: Use `supportsAllDrives` instead.
func (c *FilesTouchCall) SupportsTeamDrives(supportsTeamDrives bool) *FilesTouchCall {
	c.urlParams_.Set("supportsTeamDrives", fmt.Sprint(supportsTeamDrives))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *FilesTouchCall) Fields(s ...googleapi.Field) *FilesTouchCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *FilesTouchCall) Context(ctx context.Context) *FilesTouchCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *FilesTouchCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *FilesTouchCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/touch")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId": c.fileId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.files.touch", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.files.touch" call.
// Any non-2xx status code is an error. Response headers are in either
// *File.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *FilesTouchCall) Do(opts ...googleapi.CallOption) (*File, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &File{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.files.touch", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type FilesTrashCall struct {
	s          *Service
	fileId     string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Trash: Moves a file to the trash. The currently authenticated user must own
// the file or be at least a `fileOrganizer` on the parent for shared drive
// files.
//
// - fileId: The ID of the file to trash.
func (r *FilesService) Trash(fileId string) *FilesTrashCall {
	c := &FilesTrashCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	return c
}

// IncludeLabels sets the optional parameter "includeLabels": A comma-separated
// list of IDs of labels to include in the `labelInfo` part of the response.
func (c *FilesTrashCall) IncludeLabels(includeLabels string) *FilesTrashCall {
	c.urlParams_.Set("includeLabels", includeLabels)
	return c
}

// IncludePermissionsForView sets the optional parameter
// "includePermissionsForView": Specifies which additional view's permissions
// to include in the response. Only `published` is supported.
func (c *FilesTrashCall) IncludePermissionsForView(includePermissionsForView string) *FilesTrashCall {
	c.urlParams_.Set("includePermissionsForView", includePermissionsForView)
	return c
}

// SupportsAllDrives sets the optional parameter "supportsAllDrives": Whether
// the requesting application supports both My Drives and shared drives.
func (c *FilesTrashCall) SupportsAllDrives(supportsAllDrives bool) *FilesTrashCall {
	c.urlParams_.Set("supportsAllDrives", fmt.Sprint(supportsAllDrives))
	return c
}

// SupportsTeamDrives sets the optional parameter "supportsTeamDrives":
// Deprecated: Use `supportsAllDrives` instead.
func (c *FilesTrashCall) SupportsTeamDrives(supportsTeamDrives bool) *FilesTrashCall {
	c.urlParams_.Set("supportsTeamDrives", fmt.Sprint(supportsTeamDrives))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *FilesTrashCall) Fields(s ...googleapi.Field) *FilesTrashCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *FilesTrashCall) Context(ctx context.Context) *FilesTrashCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *FilesTrashCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *FilesTrashCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/trash")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId": c.fileId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.files.trash", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.files.trash" call.
// Any non-2xx status code is an error. Response headers are in either
// *File.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *FilesTrashCall) Do(opts ...googleapi.CallOption) (*File, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &File{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.files.trash", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type FilesUntrashCall struct {
	s          *Service
	fileId     string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Untrash: Restores a file from the trash. The currently authenticated user
// must own the file or be at least a `fileOrganizer` on the parent for shared
// drive files.
//
// - fileId: The ID of the file to untrash.
func (r *FilesService) Untrash(fileId string) *FilesUntrashCall {
	c := &FilesUntrashCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	return c
}

// IncludeLabels sets the optional parameter "includeLabels": A comma-separated
// list of IDs of labels to include in the `labelInfo` part of the response.
func (c *FilesUntrashCall) IncludeLabels(includeLabels string) *FilesUntrashCall {
	c.urlParams_.Set("includeLabels", includeLabels)
	return c
}

// IncludePermissionsForView sets the optional parameter
// "includePermissionsForView": Specifies which additional view's permissions
// to include in the response. Only `published` is supported.
func (c *FilesUntrashCall) IncludePermissionsForView(includePermissionsForView string) *FilesUntrashCall {
	c.urlParams_.Set("includePermissionsForView", includePermissionsForView)
	return c
}

// SupportsAllDrives sets the optional parameter "supportsAllDrives": Whether
// the requesting application supports both My Drives and shared drives.
func (c *FilesUntrashCall) SupportsAllDrives(supportsAllDrives bool) *FilesUntrashCall {
	c.urlParams_.Set("supportsAllDrives", fmt.Sprint(supportsAllDrives))
	return c
}

// SupportsTeamDrives sets the optional parameter "supportsTeamDrives":
// Deprecated: Use `supportsAllDrives` instead.
func (c *FilesUntrashCall) SupportsTeamDrives(supportsTeamDrives bool) *FilesUntrashCall {
	c.urlParams_.Set("supportsTeamDrives", fmt.Sprint(supportsTeamDrives))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *FilesUntrashCall) Fields(s ...googleapi.Field) *FilesUntrashCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *FilesUntrashCall) Context(ctx context.Context) *FilesUntrashCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *FilesUntrashCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *FilesUntrashCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/untrash")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId": c.fileId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.files.untrash", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.files.untrash" call.
// Any non-2xx status code is an error. Response headers are in either
// *File.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *FilesUntrashCall) Do(opts ...googleapi.CallOption) (*File, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &File{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.files.untrash", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type FilesUpdateCall struct {
	s          *Service
	fileId     string
	file       *File
	urlParams_ gensupport.URLParams
	mediaInfo_ *gensupport.MediaInfo
	ctx_       context.Context
	header_    http.Header
}

// Update:  Updates a file's metadata and/or content. When calling this method,
// only populate fields in the request that you want to modify. When updating
// fields, some fields might be changed automatically, such as `modifiedDate`.
// This method supports patch semantics. This method supports an */upload* URI
// and accepts uploaded media with the following characteristics: - *Maximum
// file size:* 5,120 GB - *Accepted Media MIME types:*`*/*` Note: Specify a
// valid MIME type, rather than the literal `*/*` value. The literal `*/*` is
// only used to indicate that any valid MIME type can be uploaded. For more
// information on uploading files, see Upload file data
// (/workspace/drive/api/guides/manage-uploads).
//
// - fileId: The ID of the file to update.
func (r *FilesService) Update(fileId string, file *File) *FilesUpdateCall {
	c := &FilesUpdateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.file = file
	return c
}

// AddParents sets the optional parameter "addParents": Comma-separated list of
// parent IDs to add.
func (c *FilesUpdateCall) AddParents(addParents string) *FilesUpdateCall {
	c.urlParams_.Set("addParents", addParents)
	return c
}

// Convert sets the optional parameter "convert": Deprecated: This parameter
// has no function.
func (c *FilesUpdateCall) Convert(convert bool) *FilesUpdateCall {
	c.urlParams_.Set("convert", fmt.Sprint(convert))
	return c
}

// EnforceSingleParent sets the optional parameter "enforceSingleParent":
// Deprecated: Adding files to multiple folders is no longer supported. Use
// `shortcuts` instead.
func (c *FilesUpdateCall) EnforceSingleParent(enforceSingleParent bool) *FilesUpdateCall {
	c.urlParams_.Set("enforceSingleParent", fmt.Sprint(enforceSingleParent))
	return c
}

// IncludeLabels sets the optional parameter "includeLabels": A comma-separated
// list of IDs of labels to include in the `labelInfo` part of the response.
func (c *FilesUpdateCall) IncludeLabels(includeLabels string) *FilesUpdateCall {
	c.urlParams_.Set("includeLabels", includeLabels)
	return c
}

// IncludePermissionsForView sets the optional parameter
// "includePermissionsForView": Specifies which additional view's permissions
// to include in the response. Only `published` is supported.
func (c *FilesUpdateCall) IncludePermissionsForView(includePermissionsForView string) *FilesUpdateCall {
	c.urlParams_.Set("includePermissionsForView", includePermissionsForView)
	return c
}

// ModifiedDateBehavior sets the optional parameter "modifiedDateBehavior":
// Determines the behavior in which `modifiedDate` is updated. This overrides
// `setModifiedDate`.
//
// Possible values:
//
//	"fromBody" - Set `modifiedDate` to the value provided in the body of the
//
// request. No change if no value was provided.
//
//	"fromBodyIfNeeded" - Set `modifiedDate` to the value provided in the body
//
// of the request depending on other contents of the update.
//
//	"fromBodyOrNow" - Set modifiedDate to the value provided in the body of
//
// the request, or to the current time if no value was provided.
//
//	"noChange" - Maintain the previous value of `modifiedDate`.
//	"now" - Set `modifiedDate` to the current time.
//	"nowIfNeeded" - Set `modifiedDate` to the current time depending on
//
// contents of the update.
func (c *FilesUpdateCall) ModifiedDateBehavior(modifiedDateBehavior string) *FilesUpdateCall {
	c.urlParams_.Set("modifiedDateBehavior", modifiedDateBehavior)
	return c
}

// NewRevision sets the optional parameter "newRevision": Whether a blob upload
// should create a new revision. If false, the blob data in the current head
// revision is replaced. If true or not set, a new blob is created as head
// revision, and previous unpinned revisions are preserved for a short period
// of time. Pinned revisions are stored indefinitely, using additional storage
// quota, up to a maximum of 200 revisions. For details on how revisions are
// retained, see the Drive Help Center
// (https://support.google.com/drive/answer/2409045).
func (c *FilesUpdateCall) NewRevision(newRevision bool) *FilesUpdateCall {
	c.urlParams_.Set("newRevision", fmt.Sprint(newRevision))
	return c
}

// Ocr sets the optional parameter "ocr": Whether to attempt OCR on .jpg, .png,
// .gif, or .pdf uploads.
func (c *FilesUpdateCall) Ocr(ocr bool) *FilesUpdateCall {
	c.urlParams_.Set("ocr", fmt.Sprint(ocr))
	return c
}

// OcrLanguage sets the optional parameter "ocrLanguage": If ocr is true, hints
// at the language to use. Valid values are BCP 47 codes.
func (c *FilesUpdateCall) OcrLanguage(ocrLanguage string) *FilesUpdateCall {
	c.urlParams_.Set("ocrLanguage", ocrLanguage)
	return c
}

// Pinned sets the optional parameter "pinned": Whether to pin the new
// revision. A file can have a maximum of 200 pinned revisions.
func (c *FilesUpdateCall) Pinned(pinned bool) *FilesUpdateCall {
	c.urlParams_.Set("pinned", fmt.Sprint(pinned))
	return c
}

// RemoveParents sets the optional parameter "removeParents": Comma-separated
// list of parent IDs to remove.
func (c *FilesUpdateCall) RemoveParents(removeParents string) *FilesUpdateCall {
	c.urlParams_.Set("removeParents", removeParents)
	return c
}

// SetModifiedDate sets the optional parameter "setModifiedDate": Whether to
// set the modified date using the value supplied in the request body. Setting
// this field to `true` is equivalent to `modifiedDateBehavior=fromBodyOrNow`,
// and `false` is equivalent to `modifiedDateBehavior=now`. To prevent any
// changes to the modified date set `modifiedDateBehavior=noChange`.
func (c *FilesUpdateCall) SetModifiedDate(setModifiedDate bool) *FilesUpdateCall {
	c.urlParams_.Set("setModifiedDate", fmt.Sprint(setModifiedDate))
	return c
}

// SupportsAllDrives sets the optional parameter "supportsAllDrives": Whether
// the requesting application supports both My Drives and shared drives.
func (c *FilesUpdateCall) SupportsAllDrives(supportsAllDrives bool) *FilesUpdateCall {
	c.urlParams_.Set("supportsAllDrives", fmt.Sprint(supportsAllDrives))
	return c
}

// SupportsTeamDrives sets the optional parameter "supportsTeamDrives":
// Deprecated: Use `supportsAllDrives` instead.
func (c *FilesUpdateCall) SupportsTeamDrives(supportsTeamDrives bool) *FilesUpdateCall {
	c.urlParams_.Set("supportsTeamDrives", fmt.Sprint(supportsTeamDrives))
	return c
}

// TimedTextLanguage sets the optional parameter "timedTextLanguage": The
// language of the timed text.
func (c *FilesUpdateCall) TimedTextLanguage(timedTextLanguage string) *FilesUpdateCall {
	c.urlParams_.Set("timedTextLanguage", timedTextLanguage)
	return c
}

// TimedTextTrackName sets the optional parameter "timedTextTrackName": The
// timed text track name.
func (c *FilesUpdateCall) TimedTextTrackName(timedTextTrackName string) *FilesUpdateCall {
	c.urlParams_.Set("timedTextTrackName", timedTextTrackName)
	return c
}

// UpdateViewedDate sets the optional parameter "updateViewedDate": Whether to
// update the view date after successfully updating the file.
func (c *FilesUpdateCall) UpdateViewedDate(updateViewedDate bool) *FilesUpdateCall {
	c.urlParams_.Set("updateViewedDate", fmt.Sprint(updateViewedDate))
	return c
}

// UseContentAsIndexableText sets the optional parameter
// "useContentAsIndexableText": Whether to use the content as indexable text.
func (c *FilesUpdateCall) UseContentAsIndexableText(useContentAsIndexableText bool) *FilesUpdateCall {
	c.urlParams_.Set("useContentAsIndexableText", fmt.Sprint(useContentAsIndexableText))
	return c
}

// Media specifies the media to upload in one or more chunks. The chunk size
// may be controlled by supplying a MediaOption generated by
// googleapi.ChunkSize. The chunk size defaults to
// googleapi.DefaultUploadChunkSize.The Content-Type header used in the upload
// request will be determined by sniffing the contents of r, unless a
// MediaOption generated by googleapi.ContentType is supplied.
// At most one of Media and ResumableMedia may be set.
func (c *FilesUpdateCall) Media(r io.Reader, options ...googleapi.MediaOption) *FilesUpdateCall {
	c.mediaInfo_ = gensupport.NewInfoFromMedia(r, options)
	return c
}

// ResumableMedia specifies the media to upload in chunks and can be canceled
// with ctx.
//
// Deprecated: use Media instead.
//
// At most one of Media and ResumableMedia may be set. mediaType identifies the
// MIME media type of the upload, such as "image/png". If mediaType is "", it
// will be auto-detected. The provided ctx will supersede any context
// previously provided to the Context method.
func (c *FilesUpdateCall) ResumableMedia(ctx context.Context, r io.ReaderAt, size int64, mediaType string) *FilesUpdateCall {
	c.ctx_ = ctx
	c.mediaInfo_ = gensupport.NewInfoFromResumableMedia(r, size, mediaType)
	return c
}

// ProgressUpdater provides a callback function that will be called after every
// chunk. It should be a low-latency function in order to not slow down the
// upload operation. This should only be called when using ResumableMedia (as
// opposed to Media).
func (c *FilesUpdateCall) ProgressUpdater(pu googleapi.ProgressUpdater) *FilesUpdateCall {
	c.mediaInfo_.SetProgressUpdater(pu)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *FilesUpdateCall) Fields(s ...googleapi.Field) *FilesUpdateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
// This context will supersede any context previously provided to the
// ResumableMedia method.
func (c *FilesUpdateCall) Context(ctx context.Context) *FilesUpdateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *FilesUpdateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *FilesUpdateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.file)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}")
	if c.mediaInfo_ != nil {
		urls = googleapi.ResolveRelative(c.s.BasePath, "/upload/drive/v2/files/{fileId}")
		c.urlParams_.Set("uploadType", c.mediaInfo_.UploadType())
	}
	newBody, getBody, cleanup := c.mediaInfo_.UploadRequest(reqHeaders, body)
	defer cleanup()
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PUT", urls, newBody)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	req.GetBody = getBody
	googleapi.Expand(req.URL, map[string]string{
		"fileId": c.fileId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.files.update", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.files.update" call.
// Any non-2xx status code is an error. Response headers are in either
// *File.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *FilesUpdateCall) Do(opts ...googleapi.CallOption) (*File, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	rx := c.mediaInfo_.ResumableUpload(res.Header.Get("Location"))
	if rx != nil {
		rx.Client = c.s.client
		rx.UserAgent = c.s.userAgent()
		ctx := c.ctx_
		if ctx == nil {
			ctx = context.TODO()
		}
		res, err = rx.Upload(ctx)
		if err != nil {
			return nil, err
		}
		defer res.Body.Close()
		if err := googleapi.CheckResponse(res); err != nil {
			return nil, gensupport.WrapError(err)
		}
	}
	ret := &File{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.files.update", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type FilesWatchCall struct {
	s          *Service
	fileId     string
	channel    *Channel
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Watch: Subscribes to changes to a file.
//
// - fileId: The ID for the file in question.
func (r *FilesService) Watch(fileId string, channel *Channel) *FilesWatchCall {
	c := &FilesWatchCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.channel = channel
	return c
}

// AcknowledgeAbuse sets the optional parameter "acknowledgeAbuse": Whether the
// user is acknowledging the risk of downloading known malware or other abusive
// files. This is only applicable when the `alt` parameter is set to `media`
// and the user is the owner of the file or an organizer of the shared drive in
// which the file resides.
func (c *FilesWatchCall) AcknowledgeAbuse(acknowledgeAbuse bool) *FilesWatchCall {
	c.urlParams_.Set("acknowledgeAbuse", fmt.Sprint(acknowledgeAbuse))
	return c
}

// IncludeLabels sets the optional parameter "includeLabels": A comma-separated
// list of IDs of labels to include in the `labelInfo` part of the response.
func (c *FilesWatchCall) IncludeLabels(includeLabels string) *FilesWatchCall {
	c.urlParams_.Set("includeLabels", includeLabels)
	return c
}

// IncludePermissionsForView sets the optional parameter
// "includePermissionsForView": Specifies which additional view's permissions
// to include in the response. Only `published` is supported.
func (c *FilesWatchCall) IncludePermissionsForView(includePermissionsForView string) *FilesWatchCall {
	c.urlParams_.Set("includePermissionsForView", includePermissionsForView)
	return c
}

// Projection sets the optional parameter "projection": Deprecated: This
// parameter has no function.
//
// Possible values:
//
//	"BASIC" - Deprecated.
//	"FULL" - Deprecated.
func (c *FilesWatchCall) Projection(projection string) *FilesWatchCall {
	c.urlParams_.Set("projection", projection)
	return c
}

// RevisionId sets the optional parameter "revisionId": Specifies the Revision
// ID that should be downloaded. Ignored unless alt=media is specified.
func (c *FilesWatchCall) RevisionId(revisionId string) *FilesWatchCall {
	c.urlParams_.Set("revisionId", revisionId)
	return c
}

// SupportsAllDrives sets the optional parameter "supportsAllDrives": Whether
// the requesting application supports both My Drives and shared drives.
func (c *FilesWatchCall) SupportsAllDrives(supportsAllDrives bool) *FilesWatchCall {
	c.urlParams_.Set("supportsAllDrives", fmt.Sprint(supportsAllDrives))
	return c
}

// SupportsTeamDrives sets the optional parameter "supportsTeamDrives":
// Deprecated: Use `supportsAllDrives` instead.
func (c *FilesWatchCall) SupportsTeamDrives(supportsTeamDrives bool) *FilesWatchCall {
	c.urlParams_.Set("supportsTeamDrives", fmt.Sprint(supportsTeamDrives))
	return c
}

// UpdateViewedDate sets the optional parameter "updateViewedDate": Deprecated:
// Use files.update with modifiedDateBehavior=noChange, updateViewedDate=true
// and an empty request body.
func (c *FilesWatchCall) UpdateViewedDate(updateViewedDate bool) *FilesWatchCall {
	c.urlParams_.Set("updateViewedDate", fmt.Sprint(updateViewedDate))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *FilesWatchCall) Fields(s ...googleapi.Field) *FilesWatchCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *FilesWatchCall) Context(ctx context.Context) *FilesWatchCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *FilesWatchCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *FilesWatchCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.channel)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/watch")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId": c.fileId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.files.watch", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.files.watch" call.
// Any non-2xx status code is an error. Response headers are in either
// *Channel.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *FilesWatchCall) Do(opts ...googleapi.CallOption) (*Channel, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Channel{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.files.watch", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ParentsDeleteCall struct {
	s          *Service
	fileId     string
	parentId   string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Delete: Removes a parent from a file.
//
// - fileId: The ID of the file.
// - parentId: The ID of the parent.
func (r *ParentsService) Delete(fileId string, parentId string) *ParentsDeleteCall {
	c := &ParentsDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.parentId = parentId
	return c
}

// EnforceSingleParent sets the optional parameter "enforceSingleParent":
// Deprecated: If an item is not in a shared drive and its last parent is
// removed, the item is placed under its owner's root.
func (c *ParentsDeleteCall) EnforceSingleParent(enforceSingleParent bool) *ParentsDeleteCall {
	c.urlParams_.Set("enforceSingleParent", fmt.Sprint(enforceSingleParent))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ParentsDeleteCall) Fields(s ...googleapi.Field) *ParentsDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ParentsDeleteCall) Context(ctx context.Context) *ParentsDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ParentsDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ParentsDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/parents/{parentId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId":   c.fileId,
		"parentId": c.parentId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.parents.delete", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.parents.delete" call.
func (c *ParentsDeleteCall) Do(opts ...googleapi.CallOption) error {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if err != nil {
		return err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return gensupport.WrapError(err)
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.parents.delete", "response", internallog.HTTPResponse(res, nil))
	return nil
}

type ParentsGetCall struct {
	s            *Service
	fileId       string
	parentId     string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets a specific parent reference.
//
// - fileId: The ID of the file.
// - parentId: The ID of the parent.
func (r *ParentsService) Get(fileId string, parentId string) *ParentsGetCall {
	c := &ParentsGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.parentId = parentId
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ParentsGetCall) Fields(s ...googleapi.Field) *ParentsGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *ParentsGetCall) IfNoneMatch(entityTag string) *ParentsGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ParentsGetCall) Context(ctx context.Context) *ParentsGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ParentsGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ParentsGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/parents/{parentId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId":   c.fileId,
		"parentId": c.parentId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.parents.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.parents.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *ParentReference.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *ParentsGetCall) Do(opts ...googleapi.CallOption) (*ParentReference, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ParentReference{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.parents.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ParentsInsertCall struct {
	s               *Service
	fileId          string
	parentreference *ParentReference
	urlParams_      gensupport.URLParams
	ctx_            context.Context
	header_         http.Header
}

// Insert: Adds a parent folder for a file.
//
// - fileId: The ID of the file.
func (r *ParentsService) Insert(fileId string, parentreference *ParentReference) *ParentsInsertCall {
	c := &ParentsInsertCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.parentreference = parentreference
	return c
}

// EnforceSingleParent sets the optional parameter "enforceSingleParent":
// Deprecated: Adding files to multiple folders is no longer supported. Use
// `shortcuts` instead.
func (c *ParentsInsertCall) EnforceSingleParent(enforceSingleParent bool) *ParentsInsertCall {
	c.urlParams_.Set("enforceSingleParent", fmt.Sprint(enforceSingleParent))
	return c
}

// SupportsAllDrives sets the optional parameter "supportsAllDrives": Whether
// the requesting application supports both My Drives and shared drives.
func (c *ParentsInsertCall) SupportsAllDrives(supportsAllDrives bool) *ParentsInsertCall {
	c.urlParams_.Set("supportsAllDrives", fmt.Sprint(supportsAllDrives))
	return c
}

// SupportsTeamDrives sets the optional parameter "supportsTeamDrives":
// Deprecated: Use `supportsAllDrives` instead.
func (c *ParentsInsertCall) SupportsTeamDrives(supportsTeamDrives bool) *ParentsInsertCall {
	c.urlParams_.Set("supportsTeamDrives", fmt.Sprint(supportsTeamDrives))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ParentsInsertCall) Fields(s ...googleapi.Field) *ParentsInsertCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ParentsInsertCall) Context(ctx context.Context) *ParentsInsertCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ParentsInsertCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ParentsInsertCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.parentreference)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/parents")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId": c.fileId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.parents.insert", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.parents.insert" call.
// Any non-2xx status code is an error. Response headers are in either
// *ParentReference.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *ParentsInsertCall) Do(opts ...googleapi.CallOption) (*ParentReference, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ParentReference{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.parents.insert", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ParentsListCall struct {
	s            *Service
	fileId       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists a file's parents.
//
// - fileId: The ID of the file.
func (r *ParentsService) List(fileId string) *ParentsListCall {
	c := &ParentsListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ParentsListCall) Fields(s ...googleapi.Field) *ParentsListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *ParentsListCall) IfNoneMatch(entityTag string) *ParentsListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ParentsListCall) Context(ctx context.Context) *ParentsListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ParentsListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ParentsListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/parents")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId": c.fileId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.parents.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.parents.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ParentList.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *ParentsListCall) Do(opts ...googleapi.CallOption) (*ParentList, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ParentList{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.parents.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type PermissionsDeleteCall struct {
	s            *Service
	fileId       string
	permissionId string
	urlParams_   gensupport.URLParams
	ctx_         context.Context
	header_      http.Header
}

// Delete: Deletes a permission from a file or shared drive. **Warning:**
// Concurrent permissions operations on the same file are not supported; only
// the last update is applied.
//
// - fileId: The ID for the file or shared drive.
// - permissionId: The ID for the permission.
func (r *PermissionsService) Delete(fileId string, permissionId string) *PermissionsDeleteCall {
	c := &PermissionsDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.permissionId = permissionId
	return c
}

// EnforceExpansiveAccess sets the optional parameter "enforceExpansiveAccess":
// Whether the request should enforce expansive access rules.
func (c *PermissionsDeleteCall) EnforceExpansiveAccess(enforceExpansiveAccess bool) *PermissionsDeleteCall {
	c.urlParams_.Set("enforceExpansiveAccess", fmt.Sprint(enforceExpansiveAccess))
	return c
}

// SupportsAllDrives sets the optional parameter "supportsAllDrives": Whether
// the requesting application supports both My Drives and shared drives.
func (c *PermissionsDeleteCall) SupportsAllDrives(supportsAllDrives bool) *PermissionsDeleteCall {
	c.urlParams_.Set("supportsAllDrives", fmt.Sprint(supportsAllDrives))
	return c
}

// SupportsTeamDrives sets the optional parameter "supportsTeamDrives":
// Deprecated: Use `supportsAllDrives` instead.
func (c *PermissionsDeleteCall) SupportsTeamDrives(supportsTeamDrives bool) *PermissionsDeleteCall {
	c.urlParams_.Set("supportsTeamDrives", fmt.Sprint(supportsTeamDrives))
	return c
}

// UseDomainAdminAccess sets the optional parameter "useDomainAdminAccess":
// Issue the request as a domain administrator; if set to true, then the
// requester will be granted access if the file ID parameter refers to a shared
// drive and the requester is an administrator of the domain to which the
// shared drive belongs.
func (c *PermissionsDeleteCall) UseDomainAdminAccess(useDomainAdminAccess bool) *PermissionsDeleteCall {
	c.urlParams_.Set("useDomainAdminAccess", fmt.Sprint(useDomainAdminAccess))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *PermissionsDeleteCall) Fields(s ...googleapi.Field) *PermissionsDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *PermissionsDeleteCall) Context(ctx context.Context) *PermissionsDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *PermissionsDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *PermissionsDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/permissions/{permissionId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId":       c.fileId,
		"permissionId": c.permissionId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.permissions.delete", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.permissions.delete" call.
func (c *PermissionsDeleteCall) Do(opts ...googleapi.CallOption) error {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if err != nil {
		return err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return gensupport.WrapError(err)
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.permissions.delete", "response", internallog.HTTPResponse(res, nil))
	return nil
}

type PermissionsGetCall struct {
	s            *Service
	fileId       string
	permissionId string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets a permission by ID.
//
// - fileId: The ID for the file or shared drive.
// - permissionId: The ID for the permission.
func (r *PermissionsService) Get(fileId string, permissionId string) *PermissionsGetCall {
	c := &PermissionsGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.permissionId = permissionId
	return c
}

// SupportsAllDrives sets the optional parameter "supportsAllDrives": Whether
// the requesting application supports both My Drives and shared drives.
func (c *PermissionsGetCall) SupportsAllDrives(supportsAllDrives bool) *PermissionsGetCall {
	c.urlParams_.Set("supportsAllDrives", fmt.Sprint(supportsAllDrives))
	return c
}

// SupportsTeamDrives sets the optional parameter "supportsTeamDrives":
// Deprecated: Use `supportsAllDrives` instead.
func (c *PermissionsGetCall) SupportsTeamDrives(supportsTeamDrives bool) *PermissionsGetCall {
	c.urlParams_.Set("supportsTeamDrives", fmt.Sprint(supportsTeamDrives))
	return c
}

// UseDomainAdminAccess sets the optional parameter "useDomainAdminAccess":
// Issue the request as a domain administrator; if set to true, then the
// requester will be granted access if the file ID parameter refers to a shared
// drive and the requester is an administrator of the domain to which the
// shared drive belongs.
func (c *PermissionsGetCall) UseDomainAdminAccess(useDomainAdminAccess bool) *PermissionsGetCall {
	c.urlParams_.Set("useDomainAdminAccess", fmt.Sprint(useDomainAdminAccess))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *PermissionsGetCall) Fields(s ...googleapi.Field) *PermissionsGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *PermissionsGetCall) IfNoneMatch(entityTag string) *PermissionsGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *PermissionsGetCall) Context(ctx context.Context) *PermissionsGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *PermissionsGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *PermissionsGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/permissions/{permissionId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId":       c.fileId,
		"permissionId": c.permissionId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.permissions.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.permissions.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *Permission.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *PermissionsGetCall) Do(opts ...googleapi.CallOption) (*Permission, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Permission{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.permissions.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type PermissionsGetIdForEmailCall struct {
	s            *Service
	email        string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// GetIdForEmail: Returns the permission ID for an email address.
//
// - email: The email address for which to return a permission ID.
func (r *PermissionsService) GetIdForEmail(email string) *PermissionsGetIdForEmailCall {
	c := &PermissionsGetIdForEmailCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.email = email
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *PermissionsGetIdForEmailCall) Fields(s ...googleapi.Field) *PermissionsGetIdForEmailCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *PermissionsGetIdForEmailCall) IfNoneMatch(entityTag string) *PermissionsGetIdForEmailCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *PermissionsGetIdForEmailCall) Context(ctx context.Context) *PermissionsGetIdForEmailCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *PermissionsGetIdForEmailCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *PermissionsGetIdForEmailCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "permissionIds/{email}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"email": c.email,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.permissions.getIdForEmail", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.permissions.getIdForEmail" call.
// Any non-2xx status code is an error. Response headers are in either
// *PermissionId.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *PermissionsGetIdForEmailCall) Do(opts ...googleapi.CallOption) (*PermissionId, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &PermissionId{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.permissions.getIdForEmail", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type PermissionsInsertCall struct {
	s          *Service
	fileId     string
	permission *Permission
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Insert: Inserts a permission for a file or shared drive. **Warning:**
// Concurrent permissions operations on the same file are not supported; only
// the last update is applied.
//
// - fileId: The ID for the file or shared drive.
func (r *PermissionsService) Insert(fileId string, permission *Permission) *PermissionsInsertCall {
	c := &PermissionsInsertCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.permission = permission
	return c
}

// EmailMessage sets the optional parameter "emailMessage": A plain text custom
// message to include in notification emails.
func (c *PermissionsInsertCall) EmailMessage(emailMessage string) *PermissionsInsertCall {
	c.urlParams_.Set("emailMessage", emailMessage)
	return c
}

// EnforceSingleParent sets the optional parameter "enforceSingleParent":
// Deprecated: See `moveToNewOwnersRoot` for details.
func (c *PermissionsInsertCall) EnforceSingleParent(enforceSingleParent bool) *PermissionsInsertCall {
	c.urlParams_.Set("enforceSingleParent", fmt.Sprint(enforceSingleParent))
	return c
}

// MoveToNewOwnersRoot sets the optional parameter "moveToNewOwnersRoot": This
// parameter will only take effect if the item is not in a shared drive and the
// request is attempting to transfer the ownership of the item. If set to
// `true`, the item will be moved to the new owner's My Drive root folder and
// all prior parents removed. If set to `false`, parents are not changed.
func (c *PermissionsInsertCall) MoveToNewOwnersRoot(moveToNewOwnersRoot bool) *PermissionsInsertCall {
	c.urlParams_.Set("moveToNewOwnersRoot", fmt.Sprint(moveToNewOwnersRoot))
	return c
}

// SendNotificationEmails sets the optional parameter "sendNotificationEmails":
// Whether to send notification emails when sharing to users or groups. This
// parameter is ignored and an email is sent if the `role` is `owner`.
func (c *PermissionsInsertCall) SendNotificationEmails(sendNotificationEmails bool) *PermissionsInsertCall {
	c.urlParams_.Set("sendNotificationEmails", fmt.Sprint(sendNotificationEmails))
	return c
}

// SupportsAllDrives sets the optional parameter "supportsAllDrives": Whether
// the requesting application supports both My Drives and shared drives.
func (c *PermissionsInsertCall) SupportsAllDrives(supportsAllDrives bool) *PermissionsInsertCall {
	c.urlParams_.Set("supportsAllDrives", fmt.Sprint(supportsAllDrives))
	return c
}

// SupportsTeamDrives sets the optional parameter "supportsTeamDrives":
// Deprecated: Use `supportsAllDrives` instead.
func (c *PermissionsInsertCall) SupportsTeamDrives(supportsTeamDrives bool) *PermissionsInsertCall {
	c.urlParams_.Set("supportsTeamDrives", fmt.Sprint(supportsTeamDrives))
	return c
}

// UseDomainAdminAccess sets the optional parameter "useDomainAdminAccess":
// Issue the request as a domain administrator; if set to true, then the
// requester will be granted access if the file ID parameter refers to a shared
// drive and the requester is an administrator of the domain to which the
// shared drive belongs.
func (c *PermissionsInsertCall) UseDomainAdminAccess(useDomainAdminAccess bool) *PermissionsInsertCall {
	c.urlParams_.Set("useDomainAdminAccess", fmt.Sprint(useDomainAdminAccess))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *PermissionsInsertCall) Fields(s ...googleapi.Field) *PermissionsInsertCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *PermissionsInsertCall) Context(ctx context.Context) *PermissionsInsertCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *PermissionsInsertCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *PermissionsInsertCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.permission)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/permissions")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId": c.fileId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.permissions.insert", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.permissions.insert" call.
// Any non-2xx status code is an error. Response headers are in either
// *Permission.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *PermissionsInsertCall) Do(opts ...googleapi.CallOption) (*Permission, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Permission{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.permissions.insert", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type PermissionsListCall struct {
	s            *Service
	fileId       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists a file's or shared drive's permissions.
//
// - fileId: The ID for the file or shared drive.
func (r *PermissionsService) List(fileId string) *PermissionsListCall {
	c := &PermissionsListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	return c
}

// IncludePermissionsForView sets the optional parameter
// "includePermissionsForView": Specifies which additional view's permissions
// to include in the response. Only `published` is supported.
func (c *PermissionsListCall) IncludePermissionsForView(includePermissionsForView string) *PermissionsListCall {
	c.urlParams_.Set("includePermissionsForView", includePermissionsForView)
	return c
}

// MaxResults sets the optional parameter "maxResults": The maximum number of
// permissions to return per page. When not set for files in a shared drive, at
// most 100 results will be returned. When not set for files that are not in a
// shared drive, the entire list will be returned.
func (c *PermissionsListCall) MaxResults(maxResults int64) *PermissionsListCall {
	c.urlParams_.Set("maxResults", fmt.Sprint(maxResults))
	return c
}

// PageToken sets the optional parameter "pageToken": The token for continuing
// a previous list request on the next page. This should be set to the value of
// `nextPageToken` from the previous response.
func (c *PermissionsListCall) PageToken(pageToken string) *PermissionsListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// SupportsAllDrives sets the optional parameter "supportsAllDrives": Whether
// the requesting application supports both My Drives and shared drives.
func (c *PermissionsListCall) SupportsAllDrives(supportsAllDrives bool) *PermissionsListCall {
	c.urlParams_.Set("supportsAllDrives", fmt.Sprint(supportsAllDrives))
	return c
}

// SupportsTeamDrives sets the optional parameter "supportsTeamDrives":
// Deprecated: Use `supportsAllDrives` instead.
func (c *PermissionsListCall) SupportsTeamDrives(supportsTeamDrives bool) *PermissionsListCall {
	c.urlParams_.Set("supportsTeamDrives", fmt.Sprint(supportsTeamDrives))
	return c
}

// UseDomainAdminAccess sets the optional parameter "useDomainAdminAccess":
// Issue the request as a domain administrator; if set to true, then the
// requester will be granted access if the file ID parameter refers to a shared
// drive and the requester is an administrator of the domain to which the
// shared drive belongs.
func (c *PermissionsListCall) UseDomainAdminAccess(useDomainAdminAccess bool) *PermissionsListCall {
	c.urlParams_.Set("useDomainAdminAccess", fmt.Sprint(useDomainAdminAccess))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *PermissionsListCall) Fields(s ...googleapi.Field) *PermissionsListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *PermissionsListCall) IfNoneMatch(entityTag string) *PermissionsListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *PermissionsListCall) Context(ctx context.Context) *PermissionsListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *PermissionsListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *PermissionsListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/permissions")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId": c.fileId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.permissions.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.permissions.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *PermissionList.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *PermissionsListCall) Do(opts ...googleapi.CallOption) (*PermissionList, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &PermissionList{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.permissions.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *PermissionsListCall) Pages(ctx context.Context, f func(*PermissionList) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type PermissionsPatchCall struct {
	s            *Service
	fileId       string
	permissionId string
	permission   *Permission
	urlParams_   gensupport.URLParams
	ctx_         context.Context
	header_      http.Header
}

// Patch: Updates a permission using patch semantics. **Warning:** Concurrent
// permissions operations on the same file are not supported; only the last
// update is applied.
//
// - fileId: The ID for the file or shared drive.
// - permissionId: The ID for the permission.
func (r *PermissionsService) Patch(fileId string, permissionId string, permission *Permission) *PermissionsPatchCall {
	c := &PermissionsPatchCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.permissionId = permissionId
	c.permission = permission
	return c
}

// EnforceExpansiveAccess sets the optional parameter "enforceExpansiveAccess":
// Whether the request should enforce expansive access rules.
func (c *PermissionsPatchCall) EnforceExpansiveAccess(enforceExpansiveAccess bool) *PermissionsPatchCall {
	c.urlParams_.Set("enforceExpansiveAccess", fmt.Sprint(enforceExpansiveAccess))
	return c
}

// RemoveExpiration sets the optional parameter "removeExpiration": Whether to
// remove the expiration date.
func (c *PermissionsPatchCall) RemoveExpiration(removeExpiration bool) *PermissionsPatchCall {
	c.urlParams_.Set("removeExpiration", fmt.Sprint(removeExpiration))
	return c
}

// SupportsAllDrives sets the optional parameter "supportsAllDrives": Whether
// the requesting application supports both My Drives and shared drives.
func (c *PermissionsPatchCall) SupportsAllDrives(supportsAllDrives bool) *PermissionsPatchCall {
	c.urlParams_.Set("supportsAllDrives", fmt.Sprint(supportsAllDrives))
	return c
}

// SupportsTeamDrives sets the optional parameter "supportsTeamDrives":
// Deprecated: Use `supportsAllDrives` instead.
func (c *PermissionsPatchCall) SupportsTeamDrives(supportsTeamDrives bool) *PermissionsPatchCall {
	c.urlParams_.Set("supportsTeamDrives", fmt.Sprint(supportsTeamDrives))
	return c
}

// TransferOwnership sets the optional parameter "transferOwnership": Whether
// changing a role to `owner` downgrades the current owners to writers. Does
// nothing if the specified role is not `owner`.
func (c *PermissionsPatchCall) TransferOwnership(transferOwnership bool) *PermissionsPatchCall {
	c.urlParams_.Set("transferOwnership", fmt.Sprint(transferOwnership))
	return c
}

// UseDomainAdminAccess sets the optional parameter "useDomainAdminAccess":
// Issue the request as a domain administrator; if set to true, then the
// requester will be granted access if the file ID parameter refers to a shared
// drive and the requester is an administrator of the domain to which the
// shared drive belongs.
func (c *PermissionsPatchCall) UseDomainAdminAccess(useDomainAdminAccess bool) *PermissionsPatchCall {
	c.urlParams_.Set("useDomainAdminAccess", fmt.Sprint(useDomainAdminAccess))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *PermissionsPatchCall) Fields(s ...googleapi.Field) *PermissionsPatchCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *PermissionsPatchCall) Context(ctx context.Context) *PermissionsPatchCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *PermissionsPatchCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *PermissionsPatchCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.permission)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/permissions/{permissionId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PATCH", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId":       c.fileId,
		"permissionId": c.permissionId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.permissions.patch", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.permissions.patch" call.
// Any non-2xx status code is an error. Response headers are in either
// *Permission.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *PermissionsPatchCall) Do(opts ...googleapi.CallOption) (*Permission, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Permission{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.permissions.patch", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type PermissionsUpdateCall struct {
	s            *Service
	fileId       string
	permissionId string
	permission   *Permission
	urlParams_   gensupport.URLParams
	ctx_         context.Context
	header_      http.Header
}

// Update: Updates a permission. **Warning:** Concurrent permissions operations
// on the same file are not supported; only the last update is applied.
//
// - fileId: The ID for the file or shared drive.
// - permissionId: The ID for the permission.
func (r *PermissionsService) Update(fileId string, permissionId string, permission *Permission) *PermissionsUpdateCall {
	c := &PermissionsUpdateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.permissionId = permissionId
	c.permission = permission
	return c
}

// EnforceExpansiveAccess sets the optional parameter "enforceExpansiveAccess":
// Whether the request should enforce expansive access rules.
func (c *PermissionsUpdateCall) EnforceExpansiveAccess(enforceExpansiveAccess bool) *PermissionsUpdateCall {
	c.urlParams_.Set("enforceExpansiveAccess", fmt.Sprint(enforceExpansiveAccess))
	return c
}

// RemoveExpiration sets the optional parameter "removeExpiration": Whether to
// remove the expiration date.
func (c *PermissionsUpdateCall) RemoveExpiration(removeExpiration bool) *PermissionsUpdateCall {
	c.urlParams_.Set("removeExpiration", fmt.Sprint(removeExpiration))
	return c
}

// SupportsAllDrives sets the optional parameter "supportsAllDrives": Whether
// the requesting application supports both My Drives and shared drives.
func (c *PermissionsUpdateCall) SupportsAllDrives(supportsAllDrives bool) *PermissionsUpdateCall {
	c.urlParams_.Set("supportsAllDrives", fmt.Sprint(supportsAllDrives))
	return c
}

// SupportsTeamDrives sets the optional parameter "supportsTeamDrives":
// Deprecated: Use `supportsAllDrives` instead.
func (c *PermissionsUpdateCall) SupportsTeamDrives(supportsTeamDrives bool) *PermissionsUpdateCall {
	c.urlParams_.Set("supportsTeamDrives", fmt.Sprint(supportsTeamDrives))
	return c
}

// TransferOwnership sets the optional parameter "transferOwnership": Whether
// changing a role to `owner` downgrades the current owners to writers. Does
// nothing if the specified role is not `owner`.
func (c *PermissionsUpdateCall) TransferOwnership(transferOwnership bool) *PermissionsUpdateCall {
	c.urlParams_.Set("transferOwnership", fmt.Sprint(transferOwnership))
	return c
}

// UseDomainAdminAccess sets the optional parameter "useDomainAdminAccess":
// Issue the request as a domain administrator; if set to true, then the
// requester will be granted access if the file ID parameter refers to a shared
// drive and the requester is an administrator of the domain to which the
// shared drive belongs.
func (c *PermissionsUpdateCall) UseDomainAdminAccess(useDomainAdminAccess bool) *PermissionsUpdateCall {
	c.urlParams_.Set("useDomainAdminAccess", fmt.Sprint(useDomainAdminAccess))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *PermissionsUpdateCall) Fields(s ...googleapi.Field) *PermissionsUpdateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *PermissionsUpdateCall) Context(ctx context.Context) *PermissionsUpdateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *PermissionsUpdateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *PermissionsUpdateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.permission)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/permissions/{permissionId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PUT", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId":       c.fileId,
		"permissionId": c.permissionId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.permissions.update", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.permissions.update" call.
// Any non-2xx status code is an error. Response headers are in either
// *Permission.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *PermissionsUpdateCall) Do(opts ...googleapi.CallOption) (*Permission, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Permission{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.permissions.update", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type PropertiesDeleteCall struct {
	s           *Service
	fileId      string
	propertyKey string
	urlParams_  gensupport.URLParams
	ctx_        context.Context
	header_     http.Header
}

// Delete: Deletes a property.
//
// - fileId: The ID of the file.
// - propertyKey: The key of the property.
func (r *PropertiesService) Delete(fileId string, propertyKey string) *PropertiesDeleteCall {
	c := &PropertiesDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.propertyKey = propertyKey
	return c
}

// Visibility sets the optional parameter "visibility": The visibility of the
// property.
func (c *PropertiesDeleteCall) Visibility(visibility string) *PropertiesDeleteCall {
	c.urlParams_.Set("visibility", visibility)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *PropertiesDeleteCall) Fields(s ...googleapi.Field) *PropertiesDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *PropertiesDeleteCall) Context(ctx context.Context) *PropertiesDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *PropertiesDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *PropertiesDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/properties/{propertyKey}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId":      c.fileId,
		"propertyKey": c.propertyKey,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.properties.delete", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.properties.delete" call.
func (c *PropertiesDeleteCall) Do(opts ...googleapi.CallOption) error {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if err != nil {
		return err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return gensupport.WrapError(err)
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.properties.delete", "response", internallog.HTTPResponse(res, nil))
	return nil
}

type PropertiesGetCall struct {
	s            *Service
	fileId       string
	propertyKey  string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets a property by its key.
//
// - fileId: The ID of the file.
// - propertyKey: The key of the property.
func (r *PropertiesService) Get(fileId string, propertyKey string) *PropertiesGetCall {
	c := &PropertiesGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.propertyKey = propertyKey
	return c
}

// Visibility sets the optional parameter "visibility": The visibility of the
// property.
func (c *PropertiesGetCall) Visibility(visibility string) *PropertiesGetCall {
	c.urlParams_.Set("visibility", visibility)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *PropertiesGetCall) Fields(s ...googleapi.Field) *PropertiesGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *PropertiesGetCall) IfNoneMatch(entityTag string) *PropertiesGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *PropertiesGetCall) Context(ctx context.Context) *PropertiesGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *PropertiesGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *PropertiesGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/properties/{propertyKey}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId":      c.fileId,
		"propertyKey": c.propertyKey,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.properties.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.properties.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *Property.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *PropertiesGetCall) Do(opts ...googleapi.CallOption) (*Property, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Property{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.properties.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type PropertiesInsertCall struct {
	s          *Service
	fileId     string
	property   *Property
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Insert: Adds a property to a file, or updates it if it already exists.
//
// - fileId: The ID of the file.
func (r *PropertiesService) Insert(fileId string, property *Property) *PropertiesInsertCall {
	c := &PropertiesInsertCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.property = property
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *PropertiesInsertCall) Fields(s ...googleapi.Field) *PropertiesInsertCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *PropertiesInsertCall) Context(ctx context.Context) *PropertiesInsertCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *PropertiesInsertCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *PropertiesInsertCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.property)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/properties")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId": c.fileId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.properties.insert", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.properties.insert" call.
// Any non-2xx status code is an error. Response headers are in either
// *Property.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *PropertiesInsertCall) Do(opts ...googleapi.CallOption) (*Property, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Property{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.properties.insert", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type PropertiesListCall struct {
	s            *Service
	fileId       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists a file's properties.
//
// - fileId: The ID of the file.
func (r *PropertiesService) List(fileId string) *PropertiesListCall {
	c := &PropertiesListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *PropertiesListCall) Fields(s ...googleapi.Field) *PropertiesListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *PropertiesListCall) IfNoneMatch(entityTag string) *PropertiesListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *PropertiesListCall) Context(ctx context.Context) *PropertiesListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *PropertiesListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *PropertiesListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/properties")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId": c.fileId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.properties.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.properties.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *PropertyList.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *PropertiesListCall) Do(opts ...googleapi.CallOption) (*PropertyList, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &PropertyList{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.properties.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type PropertiesPatchCall struct {
	s           *Service
	fileId      string
	propertyKey string
	property    *Property
	urlParams_  gensupport.URLParams
	ctx_        context.Context
	header_     http.Header
}

// Patch: Updates a property.
//
// - fileId: The ID of the file.
// - propertyKey: The key of the property.
func (r *PropertiesService) Patch(fileId string, propertyKey string, property *Property) *PropertiesPatchCall {
	c := &PropertiesPatchCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.propertyKey = propertyKey
	c.property = property
	return c
}

// Visibility sets the optional parameter "visibility": The visibility of the
// property. Allowed values are PRIVATE and PUBLIC. (Default: PRIVATE)
func (c *PropertiesPatchCall) Visibility(visibility string) *PropertiesPatchCall {
	c.urlParams_.Set("visibility", visibility)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *PropertiesPatchCall) Fields(s ...googleapi.Field) *PropertiesPatchCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *PropertiesPatchCall) Context(ctx context.Context) *PropertiesPatchCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *PropertiesPatchCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *PropertiesPatchCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.property)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/properties/{propertyKey}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PATCH", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId":      c.fileId,
		"propertyKey": c.propertyKey,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.properties.patch", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.properties.patch" call.
// Any non-2xx status code is an error. Response headers are in either
// *Property.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *PropertiesPatchCall) Do(opts ...googleapi.CallOption) (*Property, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Property{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.properties.patch", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type PropertiesUpdateCall struct {
	s           *Service
	fileId      string
	propertyKey string
	property    *Property
	urlParams_  gensupport.URLParams
	ctx_        context.Context
	header_     http.Header
}

// Update: Updates a property.
//
// - fileId: The ID of the file.
// - propertyKey: The key of the property.
func (r *PropertiesService) Update(fileId string, propertyKey string, property *Property) *PropertiesUpdateCall {
	c := &PropertiesUpdateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.propertyKey = propertyKey
	c.property = property
	return c
}

// Visibility sets the optional parameter "visibility": The visibility of the
// property. Allowed values are PRIVATE and PUBLIC. (Default: PRIVATE)
func (c *PropertiesUpdateCall) Visibility(visibility string) *PropertiesUpdateCall {
	c.urlParams_.Set("visibility", visibility)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *PropertiesUpdateCall) Fields(s ...googleapi.Field) *PropertiesUpdateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *PropertiesUpdateCall) Context(ctx context.Context) *PropertiesUpdateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *PropertiesUpdateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *PropertiesUpdateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.property)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/properties/{propertyKey}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PUT", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId":      c.fileId,
		"propertyKey": c.propertyKey,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.properties.update", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.properties.update" call.
// Any non-2xx status code is an error. Response headers are in either
// *Property.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *PropertiesUpdateCall) Do(opts ...googleapi.CallOption) (*Property, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Property{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.properties.update", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type RepliesDeleteCall struct {
	s          *Service
	fileId     string
	commentId  string
	replyId    string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Delete: Deletes a reply.
//
// - commentId: The ID of the comment.
// - fileId: The ID of the file.
// - replyId: The ID of the reply.
func (r *RepliesService) Delete(fileId string, commentId string, replyId string) *RepliesDeleteCall {
	c := &RepliesDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.commentId = commentId
	c.replyId = replyId
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *RepliesDeleteCall) Fields(s ...googleapi.Field) *RepliesDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *RepliesDeleteCall) Context(ctx context.Context) *RepliesDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *RepliesDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *RepliesDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/comments/{commentId}/replies/{replyId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId":    c.fileId,
		"commentId": c.commentId,
		"replyId":   c.replyId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.replies.delete", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.replies.delete" call.
func (c *RepliesDeleteCall) Do(opts ...googleapi.CallOption) error {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if err != nil {
		return err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return gensupport.WrapError(err)
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.replies.delete", "response", internallog.HTTPResponse(res, nil))
	return nil
}

type RepliesGetCall struct {
	s            *Service
	fileId       string
	commentId    string
	replyId      string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets a reply.
//
// - commentId: The ID of the comment.
// - fileId: The ID of the file.
// - replyId: The ID of the reply.
func (r *RepliesService) Get(fileId string, commentId string, replyId string) *RepliesGetCall {
	c := &RepliesGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.commentId = commentId
	c.replyId = replyId
	return c
}

// IncludeDeleted sets the optional parameter "includeDeleted": If set, this
// will succeed when retrieving a deleted reply.
func (c *RepliesGetCall) IncludeDeleted(includeDeleted bool) *RepliesGetCall {
	c.urlParams_.Set("includeDeleted", fmt.Sprint(includeDeleted))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *RepliesGetCall) Fields(s ...googleapi.Field) *RepliesGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *RepliesGetCall) IfNoneMatch(entityTag string) *RepliesGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *RepliesGetCall) Context(ctx context.Context) *RepliesGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *RepliesGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *RepliesGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/comments/{commentId}/replies/{replyId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId":    c.fileId,
		"commentId": c.commentId,
		"replyId":   c.replyId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.replies.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.replies.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *CommentReply.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *RepliesGetCall) Do(opts ...googleapi.CallOption) (*CommentReply, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &CommentReply{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.replies.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type RepliesInsertCall struct {
	s            *Service
	fileId       string
	commentId    string
	commentreply *CommentReply
	urlParams_   gensupport.URLParams
	ctx_         context.Context
	header_      http.Header
}

// Insert: Creates a new reply to the given comment.
//
// - commentId: The ID of the comment.
// - fileId: The ID of the file.
func (r *RepliesService) Insert(fileId string, commentId string, commentreply *CommentReply) *RepliesInsertCall {
	c := &RepliesInsertCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.commentId = commentId
	c.commentreply = commentreply
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *RepliesInsertCall) Fields(s ...googleapi.Field) *RepliesInsertCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *RepliesInsertCall) Context(ctx context.Context) *RepliesInsertCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *RepliesInsertCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *RepliesInsertCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.commentreply)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/comments/{commentId}/replies")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId":    c.fileId,
		"commentId": c.commentId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.replies.insert", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.replies.insert" call.
// Any non-2xx status code is an error. Response headers are in either
// *CommentReply.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *RepliesInsertCall) Do(opts ...googleapi.CallOption) (*CommentReply, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &CommentReply{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.replies.insert", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type RepliesListCall struct {
	s            *Service
	fileId       string
	commentId    string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists all of the replies to a comment.
//
// - commentId: The ID of the comment.
// - fileId: The ID of the file.
func (r *RepliesService) List(fileId string, commentId string) *RepliesListCall {
	c := &RepliesListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.commentId = commentId
	return c
}

// IncludeDeleted sets the optional parameter "includeDeleted": If set, all
// replies, including deleted replies (with content stripped) will be returned.
func (c *RepliesListCall) IncludeDeleted(includeDeleted bool) *RepliesListCall {
	c.urlParams_.Set("includeDeleted", fmt.Sprint(includeDeleted))
	return c
}

// MaxResults sets the optional parameter "maxResults": The maximum number of
// replies to include in the response, used for paging.
func (c *RepliesListCall) MaxResults(maxResults int64) *RepliesListCall {
	c.urlParams_.Set("maxResults", fmt.Sprint(maxResults))
	return c
}

// PageToken sets the optional parameter "pageToken": The continuation token,
// used to page through large result sets. To get the next page of results, set
// this parameter to the value of "nextPageToken" from the previous response.
func (c *RepliesListCall) PageToken(pageToken string) *RepliesListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *RepliesListCall) Fields(s ...googleapi.Field) *RepliesListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *RepliesListCall) IfNoneMatch(entityTag string) *RepliesListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *RepliesListCall) Context(ctx context.Context) *RepliesListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *RepliesListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *RepliesListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/comments/{commentId}/replies")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId":    c.fileId,
		"commentId": c.commentId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.replies.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.replies.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *CommentReplyList.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *RepliesListCall) Do(opts ...googleapi.CallOption) (*CommentReplyList, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &CommentReplyList{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.replies.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *RepliesListCall) Pages(ctx context.Context, f func(*CommentReplyList) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type RepliesPatchCall struct {
	s            *Service
	fileId       string
	commentId    string
	replyId      string
	commentreply *CommentReply
	urlParams_   gensupport.URLParams
	ctx_         context.Context
	header_      http.Header
}

// Patch: Updates an existing reply.
//
// - commentId: The ID of the comment.
// - fileId: The ID of the file.
// - replyId: The ID of the reply.
func (r *RepliesService) Patch(fileId string, commentId string, replyId string, commentreply *CommentReply) *RepliesPatchCall {
	c := &RepliesPatchCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.commentId = commentId
	c.replyId = replyId
	c.commentreply = commentreply
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *RepliesPatchCall) Fields(s ...googleapi.Field) *RepliesPatchCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *RepliesPatchCall) Context(ctx context.Context) *RepliesPatchCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *RepliesPatchCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *RepliesPatchCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.commentreply)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/comments/{commentId}/replies/{replyId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PATCH", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId":    c.fileId,
		"commentId": c.commentId,
		"replyId":   c.replyId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.replies.patch", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.replies.patch" call.
// Any non-2xx status code is an error. Response headers are in either
// *CommentReply.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *RepliesPatchCall) Do(opts ...googleapi.CallOption) (*CommentReply, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &CommentReply{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.replies.patch", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type RepliesUpdateCall struct {
	s            *Service
	fileId       string
	commentId    string
	replyId      string
	commentreply *CommentReply
	urlParams_   gensupport.URLParams
	ctx_         context.Context
	header_      http.Header
}

// Update: Updates an existing reply.
//
// - commentId: The ID of the comment.
// - fileId: The ID of the file.
// - replyId: The ID of the reply.
func (r *RepliesService) Update(fileId string, commentId string, replyId string, commentreply *CommentReply) *RepliesUpdateCall {
	c := &RepliesUpdateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.commentId = commentId
	c.replyId = replyId
	c.commentreply = commentreply
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *RepliesUpdateCall) Fields(s ...googleapi.Field) *RepliesUpdateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *RepliesUpdateCall) Context(ctx context.Context) *RepliesUpdateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *RepliesUpdateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *RepliesUpdateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.commentreply)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/comments/{commentId}/replies/{replyId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PUT", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId":    c.fileId,
		"commentId": c.commentId,
		"replyId":   c.replyId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.replies.update", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.replies.update" call.
// Any non-2xx status code is an error. Response headers are in either
// *CommentReply.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *RepliesUpdateCall) Do(opts ...googleapi.CallOption) (*CommentReply, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &CommentReply{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.replies.update", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type RevisionsDeleteCall struct {
	s          *Service
	fileId     string
	revisionId string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Delete: Permanently deletes a file version. You can only delete revisions
// for files with binary content, like images or videos. Revisions for other
// files, like Google Docs or Sheets, and the last remaining file version can't
// be deleted.
//
// - fileId: The ID of the file.
// - revisionId: The ID of the revision.
func (r *RevisionsService) Delete(fileId string, revisionId string) *RevisionsDeleteCall {
	c := &RevisionsDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.revisionId = revisionId
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *RevisionsDeleteCall) Fields(s ...googleapi.Field) *RevisionsDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *RevisionsDeleteCall) Context(ctx context.Context) *RevisionsDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *RevisionsDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *RevisionsDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/revisions/{revisionId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId":     c.fileId,
		"revisionId": c.revisionId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.revisions.delete", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.revisions.delete" call.
func (c *RevisionsDeleteCall) Do(opts ...googleapi.CallOption) error {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if err != nil {
		return err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return gensupport.WrapError(err)
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.revisions.delete", "response", internallog.HTTPResponse(res, nil))
	return nil
}

type RevisionsGetCall struct {
	s            *Service
	fileId       string
	revisionId   string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets a specific revision.
//
// - fileId: The ID of the file.
// - revisionId: The ID of the revision.
func (r *RevisionsService) Get(fileId string, revisionId string) *RevisionsGetCall {
	c := &RevisionsGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.revisionId = revisionId
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *RevisionsGetCall) Fields(s ...googleapi.Field) *RevisionsGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *RevisionsGetCall) IfNoneMatch(entityTag string) *RevisionsGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *RevisionsGetCall) Context(ctx context.Context) *RevisionsGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *RevisionsGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *RevisionsGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/revisions/{revisionId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId":     c.fileId,
		"revisionId": c.revisionId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.revisions.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.revisions.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *Revision.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *RevisionsGetCall) Do(opts ...googleapi.CallOption) (*Revision, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Revision{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.revisions.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type RevisionsListCall struct {
	s            *Service
	fileId       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists a file's revisions.
//
// - fileId: The ID of the file.
func (r *RevisionsService) List(fileId string) *RevisionsListCall {
	c := &RevisionsListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	return c
}

// MaxResults sets the optional parameter "maxResults": Maximum number of
// revisions to return.
func (c *RevisionsListCall) MaxResults(maxResults int64) *RevisionsListCall {
	c.urlParams_.Set("maxResults", fmt.Sprint(maxResults))
	return c
}

// PageToken sets the optional parameter "pageToken": Page token for revisions.
// To get the next page of results, set this parameter to the value of
// "nextPageToken" from the previous response.
func (c *RevisionsListCall) PageToken(pageToken string) *RevisionsListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *RevisionsListCall) Fields(s ...googleapi.Field) *RevisionsListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *RevisionsListCall) IfNoneMatch(entityTag string) *RevisionsListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *RevisionsListCall) Context(ctx context.Context) *RevisionsListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *RevisionsListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *RevisionsListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/revisions")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId": c.fileId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.revisions.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.revisions.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *RevisionList.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *RevisionsListCall) Do(opts ...googleapi.CallOption) (*RevisionList, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &RevisionList{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.revisions.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *RevisionsListCall) Pages(ctx context.Context, f func(*RevisionList) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type RevisionsPatchCall struct {
	s          *Service
	fileId     string
	revisionId string
	revision   *Revision
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Patch: Updates a revision.
//
// - fileId: The ID for the file.
// - revisionId: The ID for the revision.
func (r *RevisionsService) Patch(fileId string, revisionId string, revision *Revision) *RevisionsPatchCall {
	c := &RevisionsPatchCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.revisionId = revisionId
	c.revision = revision
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *RevisionsPatchCall) Fields(s ...googleapi.Field) *RevisionsPatchCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *RevisionsPatchCall) Context(ctx context.Context) *RevisionsPatchCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *RevisionsPatchCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *RevisionsPatchCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.revision)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/revisions/{revisionId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PATCH", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId":     c.fileId,
		"revisionId": c.revisionId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.revisions.patch", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.revisions.patch" call.
// Any non-2xx status code is an error. Response headers are in either
// *Revision.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *RevisionsPatchCall) Do(opts ...googleapi.CallOption) (*Revision, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Revision{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.revisions.patch", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type RevisionsUpdateCall struct {
	s          *Service
	fileId     string
	revisionId string
	revision   *Revision
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Update: Updates a revision.
//
// - fileId: The ID for the file.
// - revisionId: The ID for the revision.
func (r *RevisionsService) Update(fileId string, revisionId string, revision *Revision) *RevisionsUpdateCall {
	c := &RevisionsUpdateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.fileId = fileId
	c.revisionId = revisionId
	c.revision = revision
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *RevisionsUpdateCall) Fields(s ...googleapi.Field) *RevisionsUpdateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *RevisionsUpdateCall) Context(ctx context.Context) *RevisionsUpdateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *RevisionsUpdateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *RevisionsUpdateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.revision)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "files/{fileId}/revisions/{revisionId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PUT", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"fileId":     c.fileId,
		"revisionId": c.revisionId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.revisions.update", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.revisions.update" call.
// Any non-2xx status code is an error. Response headers are in either
// *Revision.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *RevisionsUpdateCall) Do(opts ...googleapi.CallOption) (*Revision, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Revision{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.revisions.update", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type TeamdrivesDeleteCall struct {
	s           *Service
	teamDriveId string
	urlParams_  gensupport.URLParams
	ctx_        context.Context
	header_     http.Header
}

// Delete: Deprecated: Use `drives.delete` instead.
//
// - teamDriveId: The ID of the Team Drive.
func (r *TeamdrivesService) Delete(teamDriveId string) *TeamdrivesDeleteCall {
	c := &TeamdrivesDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.teamDriveId = teamDriveId
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *TeamdrivesDeleteCall) Fields(s ...googleapi.Field) *TeamdrivesDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *TeamdrivesDeleteCall) Context(ctx context.Context) *TeamdrivesDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *TeamdrivesDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *TeamdrivesDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "teamdrives/{teamDriveId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"teamDriveId": c.teamDriveId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.teamdrives.delete", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.teamdrives.delete" call.
func (c *TeamdrivesDeleteCall) Do(opts ...googleapi.CallOption) error {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if err != nil {
		return err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return gensupport.WrapError(err)
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.teamdrives.delete", "response", internallog.HTTPResponse(res, nil))
	return nil
}

type TeamdrivesGetCall struct {
	s            *Service
	teamDriveId  string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Deprecated: Use `drives.get` instead.
//
// - teamDriveId: The ID of the Team Drive.
func (r *TeamdrivesService) Get(teamDriveId string) *TeamdrivesGetCall {
	c := &TeamdrivesGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.teamDriveId = teamDriveId
	return c
}

// UseDomainAdminAccess sets the optional parameter "useDomainAdminAccess":
// Issue the request as a domain administrator; if set to true, then the
// requester will be granted access if they are an administrator of the domain
// to which the Team Drive belongs.
func (c *TeamdrivesGetCall) UseDomainAdminAccess(useDomainAdminAccess bool) *TeamdrivesGetCall {
	c.urlParams_.Set("useDomainAdminAccess", fmt.Sprint(useDomainAdminAccess))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *TeamdrivesGetCall) Fields(s ...googleapi.Field) *TeamdrivesGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *TeamdrivesGetCall) IfNoneMatch(entityTag string) *TeamdrivesGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *TeamdrivesGetCall) Context(ctx context.Context) *TeamdrivesGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *TeamdrivesGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *TeamdrivesGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "teamdrives/{teamDriveId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"teamDriveId": c.teamDriveId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.teamdrives.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.teamdrives.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *TeamDrive.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *TeamdrivesGetCall) Do(opts ...googleapi.CallOption) (*TeamDrive, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &TeamDrive{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.teamdrives.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type TeamdrivesInsertCall struct {
	s          *Service
	teamdrive  *TeamDrive
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Insert: Deprecated: Use `drives.insert` instead.
//
//   - requestId: An ID, such as a random UUID, which uniquely identifies this
//     user's request for idempotent creation of a Team Drive. A repeated request
//     by the same user and with the same request ID will avoid creating
//     duplicates by attempting to create the same Team Drive. If the Team Drive
//     already exists a 409 error will be returned.
func (r *TeamdrivesService) Insert(requestId string, teamdrive *TeamDrive) *TeamdrivesInsertCall {
	c := &TeamdrivesInsertCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.urlParams_.Set("requestId", requestId)
	c.teamdrive = teamdrive
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *TeamdrivesInsertCall) Fields(s ...googleapi.Field) *TeamdrivesInsertCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *TeamdrivesInsertCall) Context(ctx context.Context) *TeamdrivesInsertCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *TeamdrivesInsertCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *TeamdrivesInsertCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.teamdrive)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "teamdrives")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.teamdrives.insert", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.teamdrives.insert" call.
// Any non-2xx status code is an error. Response headers are in either
// *TeamDrive.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *TeamdrivesInsertCall) Do(opts ...googleapi.CallOption) (*TeamDrive, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &TeamDrive{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.teamdrives.insert", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type TeamdrivesListCall struct {
	s            *Service
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Deprecated: Use `drives.list` instead.
func (r *TeamdrivesService) List() *TeamdrivesListCall {
	c := &TeamdrivesListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	return c
}

// MaxResults sets the optional parameter "maxResults": Maximum number of Team
// Drives to return.
func (c *TeamdrivesListCall) MaxResults(maxResults int64) *TeamdrivesListCall {
	c.urlParams_.Set("maxResults", fmt.Sprint(maxResults))
	return c
}

// PageToken sets the optional parameter "pageToken": Page token for Team
// Drives.
func (c *TeamdrivesListCall) PageToken(pageToken string) *TeamdrivesListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Q sets the optional parameter "q": Query string for searching Team Drives.
func (c *TeamdrivesListCall) Q(q string) *TeamdrivesListCall {
	c.urlParams_.Set("q", q)
	return c
}

// UseDomainAdminAccess sets the optional parameter "useDomainAdminAccess":
// Issue the request as a domain administrator; if set to true, then all Team
// Drives of the domain in which the requester is an administrator are
// returned.
func (c *TeamdrivesListCall) UseDomainAdminAccess(useDomainAdminAccess bool) *TeamdrivesListCall {
	c.urlParams_.Set("useDomainAdminAccess", fmt.Sprint(useDomainAdminAccess))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *TeamdrivesListCall) Fields(s ...googleapi.Field) *TeamdrivesListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *TeamdrivesListCall) IfNoneMatch(entityTag string) *TeamdrivesListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *TeamdrivesListCall) Context(ctx context.Context) *TeamdrivesListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *TeamdrivesListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *TeamdrivesListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "teamdrives")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.teamdrives.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.teamdrives.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *TeamDriveList.ServerResponse.Header or (if a response was returned at all)
// in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *TeamdrivesListCall) Do(opts ...googleapi.CallOption) (*TeamDriveList, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &TeamDriveList{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.teamdrives.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *TeamdrivesListCall) Pages(ctx context.Context, f func(*TeamDriveList) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type TeamdrivesUpdateCall struct {
	s           *Service
	teamDriveId string
	teamdrive   *TeamDrive
	urlParams_  gensupport.URLParams
	ctx_        context.Context
	header_     http.Header
}

// Update: Deprecated: Use `drives.update` instead.
//
// - teamDriveId: The ID of the Team Drive.
func (r *TeamdrivesService) Update(teamDriveId string, teamdrive *TeamDrive) *TeamdrivesUpdateCall {
	c := &TeamdrivesUpdateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.teamDriveId = teamDriveId
	c.teamdrive = teamdrive
	return c
}

// UseDomainAdminAccess sets the optional parameter "useDomainAdminAccess":
// Issue the request as a domain administrator; if set to true, then the
// requester will be granted access if they are an administrator of the domain
// to which the Team Drive belongs.
func (c *TeamdrivesUpdateCall) UseDomainAdminAccess(useDomainAdminAccess bool) *TeamdrivesUpdateCall {
	c.urlParams_.Set("useDomainAdminAccess", fmt.Sprint(useDomainAdminAccess))
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *TeamdrivesUpdateCall) Fields(s ...googleapi.Field) *TeamdrivesUpdateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *TeamdrivesUpdateCall) Context(ctx context.Context) *TeamdrivesUpdateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *TeamdrivesUpdateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *TeamdrivesUpdateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.teamdrive)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "teamdrives/{teamDriveId}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PUT", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"teamDriveId": c.teamDriveId,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "drive.teamdrives.update", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "drive.teamdrives.update" call.
// Any non-2xx status code is an error. Response headers are in either
// *TeamDrive.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *TeamdrivesUpdateCall) Do(opts ...googleapi.CallOption) (*TeamDrive, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &TeamDrive{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "drive.teamdrives.update", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}
