// Copyright (c) 2020, 2021, Maxime Soulé
// All rights reserved.
//
// This source code is licensed under the BSD-style license found in the
// LICENSE file in the root directory of this source tree.

package tdhttp

import (
	"encoding/json"
	"encoding/xml"
	"fmt"
	"io"
	"net/http"
	"net/http/httptest"
	"net/url"
	"reflect"
	"runtime"
	"strings"
	"testing"
	"time"

	"github.com/maxatome/go-testdeep/helpers/tdhttp/internal"
	"github.com/maxatome/go-testdeep/helpers/tdutil"
	"github.com/maxatome/go-testdeep/internal/color"
	"github.com/maxatome/go-testdeep/internal/ctxerr"
	"github.com/maxatome/go-testdeep/internal/types"
	"github.com/maxatome/go-testdeep/td"
)

// TestAPI allows to test one HTTP API. See [NewTestAPI] function to
// create a new instance and get some examples of use.
type TestAPI struct {
	t       *td.T
	handler http.Handler
	name    string

	sentAt        time.Time
	response      *httptest.ResponseRecorder
	statusFailed  bool
	headerFailed  bool
	cookiesFailed bool
	bodyFailed    bool

	// autoDumpResponse dumps the received response when a test fails.
	autoDumpResponse bool
	responseDumped   bool
}

// NewTestAPI creates a [TestAPI] that can be used to test routes of the
// API behind handler.
//
//	tdhttp.NewTestAPI(t, mux).
//	  Get("/test").
//	  CmpStatus(200).
//	  CmpBody("OK!")
//
// Several routes can be tested with the same instance as in:
//
//	ta := tdhttp.NewTestAPI(t, mux)
//
//	ta.Get("/test").
//	  CmpStatus(200).
//	  CmpBody("OK!")
//
//	ta.Get("/ping").
//	  CmpStatus(200).
//	  CmpBody("pong")
//
// Note that tb can be a [*testing.T] as well as a [*td.T].
func NewTestAPI(tb testing.TB, handler http.Handler) *TestAPI {
	return &TestAPI{
		t:       td.NewT(tb),
		handler: handler,
	}
}

// With creates a new [*TestAPI] instance copied from t, but resetting
// the [testing.TB] instance the tests are based on to tb. The
// returned instance is independent from t, sharing only the same
// handler.
//
// It is typically used when the [TestAPI] instance is "reused" in
// sub-tests, as in:
//
//	func TestMyAPI(t *testing.T) {
//	  ta := tdhttp.NewTestAPI(t, MyAPIHandler())
//
//	  ta.Get("/test").CmpStatus(200)
//
//	  t.Run("errors", func (t *testing.T) {
//	    ta := ta.With(t)
//
//	    ta.Get("/test?bad=1").CmpStatus(400)
//	    ta.Get("/test?bad=buzz").CmpStatus(400)
//	  }
//
//	  ta.Get("/next").CmpStatus(200)
//	}
//
// Note that tb can be a [*testing.T] as well as a [*td.T].
//
// See [TestAPI.Run] for another way to handle subtests.
func (t *TestAPI) With(tb testing.TB) *TestAPI {
	return &TestAPI{
		t:                td.NewT(tb),
		handler:          t.handler,
		autoDumpResponse: t.autoDumpResponse,
	}
}

// T returns the internal instance of [*td.T].
func (t *TestAPI) T() *td.T {
	return t.t
}

// Run runs f as a subtest of t called name.
func (t *TestAPI) Run(name string, f func(t *TestAPI)) bool {
	return t.t.Run(name, func(tdt *td.T) {
		f(NewTestAPI(tdt, t.handler))
	})
}

// AutoDumpResponse allows to dump the HTTP response when the first
// error is encountered after a request.
func (t *TestAPI) AutoDumpResponse() *TestAPI {
	t.autoDumpResponse = true
	return t
}

// Name allows to name the series of tests that follow. This name is
// used as a prefix for all following tests, in case of failure to
// qualify each test. If len(args) > 1 and the first item of args is
// a string and contains a '%' rune then [fmt.Fprintf] is used to
// compose the name, else args are passed to [fmt.Fprint].
func (t *TestAPI) Name(args ...any) *TestAPI {
	t.name = tdutil.BuildTestName(args...)
	if t.name != "" {
		t.name += ": "
	}
	return t
}

// Request sends a new HTTP request to the tested API. Any Cmp* or
// [TestAPI.NoBody] methods can now be called.
//
// Note that [TestAPI.Failed] status is reset just after this call.
func (t *TestAPI) Request(req *http.Request) *TestAPI {
	t.response = httptest.NewRecorder()

	t.statusFailed = false
	t.headerFailed = false
	t.cookiesFailed = false
	t.bodyFailed = false
	t.sentAt = time.Now().Truncate(0)
	t.responseDumped = false

	t.handler.ServeHTTP(t.response, req)

	return t
}

func (t *TestAPI) checkRequestSent() bool {
	t.t.Helper()

	// If no request has been sent, display a nice error message
	return t.t.RootName("Request").
		Code(t.response != nil,
			func(sent bool) error {
				if sent {
					return nil
				}
				return &ctxerr.Error{
					Message: "%% not sent!",
					Summary: ctxerr.NewSummary("A request must be sent before testing status, header or body"),
				}
			},
			t.name+"request is sent")
}

// Failed returns true if any Cmp* or [TestAPI.NoBody] method failed since last
// request sending.
func (t *TestAPI) Failed() bool {
	return t.statusFailed || t.headerFailed || t.cookiesFailed || t.bodyFailed
}

// Get sends a HTTP GET to the tested API. Any Cmp* or [TestAPI.NoBody] methods
// can now be called.
//
// Note that [TestAPI.Failed] status is reset just after this call.
//
// See [NewRequest] for all possible formats accepted in headersQueryParams.
func (t *TestAPI) Get(target string, headersQueryParams ...any) *TestAPI {
	req, err := get(target, headersQueryParams...)
	if err != nil {
		t.t.Helper()
		t.t.Fatal(err)
	}
	return t.Request(req)
}

// Head sends a HTTP HEAD to the tested API. Any Cmp* or [TestAPI.NoBody] methods
// can now be called.
//
// Note that [TestAPI.Failed] status is reset just after this call.
//
// See [NewRequest] for all possible formats accepted in headersQueryParams.
func (t *TestAPI) Head(target string, headersQueryParams ...any) *TestAPI {
	req, err := head(target, headersQueryParams...)
	if err != nil {
		t.t.Helper()
		t.t.Fatal(err)
	}
	return t.Request(req)
}

// Options sends a HTTP OPTIONS to the tested API. Any Cmp* or
// [TestAPI.NoBody] methods can now be called.
//
// Note that [TestAPI.Failed] status is reset just after this call.
//
// See [NewRequest] for all possible formats accepted in headersQueryParams.
func (t *TestAPI) Options(target string, body io.Reader, headersQueryParams ...any) *TestAPI {
	req, err := options(target, body, headersQueryParams...)
	if err != nil {
		t.t.Helper()
		t.t.Fatal(err)
	}
	return t.Request(req)
}

// Post sends a HTTP POST to the tested API. Any Cmp* or
// [TestAPI.NoBody] methods can now be called.
//
// Note that [TestAPI.Failed] status is reset just after this call.
//
// See [NewRequest] for all possible formats accepted in headersQueryParams.
func (t *TestAPI) Post(target string, body io.Reader, headersQueryParams ...any) *TestAPI {
	req, err := post(target, body, headersQueryParams...)
	if err != nil {
		t.t.Helper()
		t.t.Fatal(err)
	}
	return t.Request(req)
}

// PostForm sends a HTTP POST with data's keys and values URL-encoded
// as the request body to the tested API. "Content-Type" header is
// automatically set to "application/x-www-form-urlencoded". Any Cmp*
// or [TestAPI.NoBody] methods can now be called.
//
// Note that [TestAPI.Failed] status is reset just after this call.
//
// See [NewRequest] for all possible formats accepted in headersQueryParams.
func (t *TestAPI) PostForm(target string, data url.Values, headersQueryParams ...any) *TestAPI {
	req, err := postForm(target, data, headersQueryParams...)
	if err != nil {
		t.t.Helper()
		t.t.Fatal(err)
	}
	return t.Request(req)
}

// PostMultipartFormData sends a HTTP POST multipart request, like
// multipart/form-data one for example. See [MultipartBody] type for
// details. "Content-Type" header is automatically set depending on
// data.MediaType (defaults to "multipart/form-data") and data.Boundary
// (defaults to "go-testdeep-42"). Any Cmp* or [TestAPI.NoBody] methods can now
// be called.
//
// Note that [TestAPI.Failed] status is reset just after this call.
//
//	ta.PostMultipartFormData("/data",
//	  &tdhttp.MultipartBody{
//	    // "multipart/form-data" by default
//	    Parts: []*tdhttp.MultipartPart{
//	      tdhttp.NewMultipartPartString("type", "Sales"),
//	      tdhttp.NewMultipartPartFile("report", "report.json", "application/json"),
//	    },
//	  },
//	  "X-Foo", "Foo-value",
//	  "X-Zip", "Zip-value",
//	)
//
// See [NewRequest] for all possible formats accepted in headersQueryParams.
func (t *TestAPI) PostMultipartFormData(target string, data *MultipartBody, headersQueryParams ...any) *TestAPI {
	req, err := postMultipartFormData(target, data, headersQueryParams...)
	if err != nil {
		t.t.Helper()
		t.t.Fatal(err)
	}
	return t.Request(req)
}

// Put sends a HTTP PUT to the tested API. Any Cmp* or [TestAPI.NoBody] methods
// can now be called.
//
// Note that [TestAPI.Failed] status is reset just after this call.
//
// See [NewRequest] for all possible formats accepted in headersQueryParams.
func (t *TestAPI) Put(target string, body io.Reader, headersQueryParams ...any) *TestAPI {
	req, err := put(target, body, headersQueryParams...)
	if err != nil {
		t.t.Helper()
		t.t.Fatal(err)
	}
	return t.Request(req)
}

// Patch sends a HTTP PATCH to the tested API. Any Cmp* or [TestAPI.NoBody] methods
// can now be called.
//
// Note that [TestAPI.Failed] status is reset just after this call.
//
// See [NewRequest] for all possible formats accepted in headersQueryParams.
func (t *TestAPI) Patch(target string, body io.Reader, headersQueryParams ...any) *TestAPI {
	req, err := patch(target, body, headersQueryParams...)
	if err != nil {
		t.t.Helper()
		t.t.Fatal(err)
	}
	return t.Request(req)
}

// Delete sends a HTTP DELETE to the tested API. Any Cmp* or [TestAPI.NoBody] methods
// can now be called.
//
// Note that [TestAPI.Failed] status is reset just after this call.
//
// See [NewRequest] for all possible formats accepted in headersQueryParams.
func (t *TestAPI) Delete(target string, body io.Reader, headersQueryParams ...any) *TestAPI {
	req, err := del(target, body, headersQueryParams...)
	if err != nil {
		t.t.Helper()
		t.t.Fatal(err)
	}
	return t.Request(req)
}

// NewJSONRequest sends a HTTP request with body marshaled to
// JSON. "Content-Type" header is automatically set to
// "application/json". Any Cmp* or [TestAPI.NoBody] methods can now be called.
//
// Note that [TestAPI.Failed] status is reset just after this call.
//
// See [NewRequest] for all possible formats accepted in headersQueryParams.
func (t *TestAPI) NewJSONRequest(method, target string, body any, headersQueryParams ...any) *TestAPI {
	req, err := newJSONRequest(method, target, body, headersQueryParams...)
	if err != nil {
		t.t.Helper()
		t.t.Fatal(err)
	}
	return t.Request(req)
}

// PostJSON sends a HTTP POST with body marshaled to
// JSON. "Content-Type" header is automatically set to
// "application/json". Any Cmp* or [TestAPI.NoBody] methods can now be called.
//
// Note that [TestAPI.Failed] status is reset just after this call.
//
// See [NewRequest] for all possible formats accepted in headersQueryParams.
func (t *TestAPI) PostJSON(target string, body any, headersQueryParams ...any) *TestAPI {
	req, err := newJSONRequest(http.MethodPost, target, body, headersQueryParams...)
	if err != nil {
		t.t.Helper()
		t.t.Fatal(err)
	}
	return t.Request(req)
}

// PutJSON sends a HTTP PUT with body marshaled to
// JSON. "Content-Type" header is automatically set to
// "application/json". Any Cmp* or [TestAPI.NoBody] methods can now be called.
//
// Note that [TestAPI.Failed] status is reset just after this call.
//
// See [NewRequest] for all possible formats accepted in headersQueryParams.
func (t *TestAPI) PutJSON(target string, body any, headersQueryParams ...any) *TestAPI {
	req, err := newJSONRequest(http.MethodPut, target, body, headersQueryParams...)
	if err != nil {
		t.t.Helper()
		t.t.Fatal(err)
	}
	return t.Request(req)
}

// PatchJSON sends a HTTP PATCH with body marshaled to
// JSON. "Content-Type" header is automatically set to
// "application/json". Any Cmp* or [TestAPI.NoBody] methods can now be called.
//
// Note that [TestAPI.Failed] status is reset just after this call.
//
// See [NewRequest] for all possible formats accepted in headersQueryParams.
func (t *TestAPI) PatchJSON(target string, body any, headersQueryParams ...any) *TestAPI {
	req, err := newJSONRequest(http.MethodPatch, target, body, headersQueryParams...)
	if err != nil {
		t.t.Helper()
		t.t.Fatal(err)
	}
	return t.Request(req)
}

// DeleteJSON sends a HTTP DELETE with body marshaled to
// JSON. "Content-Type" header is automatically set to
// "application/json". Any Cmp* or [TestAPI.NoBody] methods can now be called.
//
// Note that [TestAPI.Failed] status is reset just after this call.
//
// See [NewRequest] for all possible formats accepted in headersQueryParams.
func (t *TestAPI) DeleteJSON(target string, body any, headersQueryParams ...any) *TestAPI {
	req, err := newJSONRequest(http.MethodDelete, target, body, headersQueryParams...)
	if err != nil {
		t.t.Helper()
		t.t.Fatal(err)
	}
	return t.Request(req)
}

// NewXMLRequest sends a HTTP request with body marshaled to
// XML. "Content-Type" header is automatically set to
// "application/xml". Any Cmp* or [TestAPI.NoBody] methods can now be called.
//
// Note that [TestAPI.Failed] status is reset just after this call.
//
// See [NewRequest] for all possible formats accepted in headersQueryParams.
func (t *TestAPI) NewXMLRequest(method, target string, body any, headersQueryParams ...any) *TestAPI {
	req, err := newXMLRequest(method, target, body, headersQueryParams...)
	if err != nil {
		t.t.Helper()
		t.t.Fatal(err)
	}
	return t.Request(req)
}

// PostXML sends a HTTP POST with body marshaled to
// XML. "Content-Type" header is automatically set to
// "application/xml". Any Cmp* or [TestAPI.NoBody] methods can now be called.
//
// Note that [TestAPI.Failed] status is reset just after this call.
//
// See [NewRequest] for all possible formats accepted in headersQueryParams.
func (t *TestAPI) PostXML(target string, body any, headersQueryParams ...any) *TestAPI {
	req, err := newXMLRequest(http.MethodPost, target, body, headersQueryParams...)
	if err != nil {
		t.t.Helper()
		t.t.Fatal(err)
	}
	return t.Request(req)
}

// PutXML sends a HTTP PUT with body marshaled to
// XML. "Content-Type" header is automatically set to
// "application/xml". Any Cmp* or [TestAPI.NoBody] methods can now be called.
//
// Note that [TestAPI.Failed] status is reset just after this call.
//
// See [NewRequest] for all possible formats accepted in headersQueryParams.
func (t *TestAPI) PutXML(target string, body any, headersQueryParams ...any) *TestAPI {
	req, err := newXMLRequest(http.MethodPut, target, body, headersQueryParams...)
	if err != nil {
		t.t.Helper()
		t.t.Fatal(err)
	}
	return t.Request(req)
}

// PatchXML sends a HTTP PATCH with body marshaled to
// XML. "Content-Type" header is automatically set to
// "application/xml". Any Cmp* or [TestAPI.NoBody] methods can now be called.
//
// Note that [TestAPI.Failed] status is reset just after this call.
//
// See [NewRequest] for all possible formats accepted in headersQueryParams.
func (t *TestAPI) PatchXML(target string, body any, headersQueryParams ...any) *TestAPI {
	req, err := newXMLRequest(http.MethodPatch, target, body, headersQueryParams...)
	if err != nil {
		t.t.Helper()
		t.t.Fatal(err)
	}
	return t.Request(req)
}

// DeleteXML sends a HTTP DELETE with body marshaled to
// XML. "Content-Type" header is automatically set to
// "application/xml". Any Cmp* or [TestAPI.NoBody] methods can now be called.
//
// Note that [TestAPI.Failed] status is reset just after this call.
//
// See [NewRequest] for all possible formats accepted in headersQueryParams.
func (t *TestAPI) DeleteXML(target string, body any, headersQueryParams ...any) *TestAPI {
	req, err := newXMLRequest(http.MethodDelete, target, body, headersQueryParams...)
	if err != nil {
		t.t.Helper()
		t.t.Fatal(err)
	}
	return t.Request(req)
}

// CmpStatus tests the last request response status against
// expectedStatus. expectedStatus can be an int to match a fixed HTTP
// status code, or a [td.TestDeep] operator.
//
//	ta := tdhttp.NewTestAPI(t, mux)
//
//	ta.Get("/test").
//	  CmpStatus(http.StatusOK)
//
//	ta.PostJSON("/new", map[string]string{"name": "Bob"}).
//	  CmpStatus(td.Between(200, 202))
//
// It fails if no request has been sent yet.
func (t *TestAPI) CmpStatus(expectedStatus any) *TestAPI {
	defer t.t.AnchorsPersistTemporarily()()

	t.t.Helper()

	if !t.checkRequestSent() {
		t.statusFailed = true
		return t
	}

	if !t.t.RootName("Response.Status").
		CmpLax(t.response.Code, expectedStatus, t.name+"status code should match") {
		t.statusFailed = true

		if t.autoDumpResponse {
			t.dumpResponse()
		}
	}

	return t
}

// CmpHeader tests the last request response header against
// expectedHeader. expectedHeader can be a [http.Header] or a
// [td.TestDeep] operator. Keep in mind that if it is a [http.Header],
// it has to match exactly the response header. Often only the
// presence of a header key is needed:
//
//	ta := tdhttp.NewTestAPI(t, mux).
//	  PostJSON("/new", map[string]string{"name": "Bob"}).
//	  CmdStatus(201).
//	  CmpHeader(td.ContainsKey("X-Custom"))
//
// or some specific key, value pairs:
//
//	ta.CmpHeader(td.SuperMapOf(
//	  http.Header{
//	    "X-Account": []string{"Bob"},
//	  },
//	  td.MapEntries{
//	    "X-Token": td.Bag(td.Re(`^[a-z0-9-]{32}\z`)),
//	  }),
//	)
//
// Note that CmpHeader calls can be chained:
//
//	ta.CmpHeader(td.ContainsKey("X-Account")).
//	  CmpHeader(td.ContainsKey("X-Token"))
//
// instead of doing all tests in one call as [td.All] operator allows it:
//
//	ta.CmpHeader(td.All(
//	  td.ContainsKey("X-Account"),
//	  td.ContainsKey("X-Token")),
//	)
//
// It fails if no request has been sent yet.
func (t *TestAPI) CmpHeader(expectedHeader any) *TestAPI {
	defer t.t.AnchorsPersistTemporarily()()

	t.t.Helper()

	if !t.checkRequestSent() {
		t.headerFailed = true
		return t
	}

	if !t.t.RootName("Response.Header").
		CmpLax(t.response.Header(), expectedHeader, t.name+"header should match") {
		t.headerFailed = true

		if t.autoDumpResponse {
			t.dumpResponse()
		}
	}

	return t
}

// CmpCookies tests the last request response cookies against
// expectedCookies. expectedCookies can be a [][*http.Cookie] or a
// [td.TestDeep] operator. Keep in mind that if it is a
// [][*http.Cookie], it has to match exactly the response
// cookies. Often only the presence of a cookie key is needed:
//
//	ta := tdhttp.NewTestAPI(t, mux).
//	  PostJSON("/login", map[string]string{"name": "Bob", "password": "Sponge"}).
//	  CmdStatus(200).
//	  CmpCookies(td.SuperBagOf(td.Struct(&http.Cookie{Name: "cookie_session"}, nil))).
//	  CmpCookies(td.SuperBagOf(td.Smuggle("Name", "cookie_session"))) // shorter
//
// To make tests easier, [http.Cookie.Raw] and [http.Cookie.RawExpires] fields
// of each [*http.Cookie] are zeroed before doing the comparison. So no need
// to fill them when comparing against a simple literal as in:
//
//	ta := tdhttp.NewTestAPI(t, mux).
//	  PostJSON("/login", map[string]string{"name": "Bob", "password": "Sponge"}).
//	  CmdStatus(200).
//	  CmpCookies([]*http.Cookies{
//	    {Name: "cookieName1", Value: "cookieValue1"},
//	    {Name: "cookieName2", Value: "cookieValue2"},
//	  })
//
// It fails if no request has been sent yet.
func (t *TestAPI) CmpCookies(expectedCookies any) *TestAPI {
	defer t.t.AnchorsPersistTemporarily()()

	t.t.Helper()

	if !t.checkRequestSent() {
		t.cookiesFailed = true
		return t
	}

	// Empty Raw* fields to make comparisons easier
	cookies := t.response.Result().Cookies()
	for _, c := range cookies {
		c.RawExpires, c.Raw = "", ""
	}

	if !t.t.RootName("Response.Cookie").
		CmpLax(cookies, expectedCookies, t.name+"cookies should match") {
		t.cookiesFailed = true

		if t.autoDumpResponse {
			t.dumpResponse()
		}
	}

	return t
}

// findCmpXBodyCaller finds the oldest Cmp* method called.
func findCmpXBodyCaller() string {
	var (
		fn    string
		pc    [20]uintptr
		found bool
	)
	if num := runtime.Callers(5, pc[:]); num > 0 {
		frames := runtime.CallersFrames(pc[:num])
		for {
			frame, more := frames.Next()
			if pos := strings.Index(frame.Function, "tdhttp.(*TestAPI).Cmp"); pos > 0 {
				fn = frame.Function[pos+18:]
				found = true
			} else if found {
				more = false
			}
			if !more {
				break
			}
		}
	}
	return fn
}

func (t *TestAPI) cmpMarshaledBody(
	acceptEmptyBody bool,
	unmarshal func([]byte, any) error,
	expectedBody any,
) *TestAPI {
	defer t.t.AnchorsPersistTemporarily()()

	t.t.Helper()

	if !t.checkRequestSent() {
		t.bodyFailed = true
		return t
	}

	if !acceptEmptyBody &&
		!t.t.RootName("Response body").Code(t.response.Body.Bytes(),
			func(b []byte) error {
				if len(b) > 0 {
					return nil
				}
				return &ctxerr.Error{
					Message: "%% is empty!",
					Summary: ctxerr.NewSummary(
						"Body cannot be empty when using " + findCmpXBodyCaller()),
				}
			},
			t.name+"body should not be empty") {
		t.bodyFailed = true
		if t.autoDumpResponse {
			t.dumpResponse()
		}
		return t
	}

	tt := t.t.RootName("Response.Body")

	var bodyType reflect.Type

	// If expectedBody is a TestDeep operator, try to ask it the type
	// behind it. It should work in most cases (typically Struct(),
	// Map() & Slice()).
	var unknownExpectedType, showRawBody bool
	op, ok := expectedBody.(td.TestDeep)
	if ok {
		bodyType = op.TypeBehind()
		if bodyType == nil {
			// As the expected body type cannot be guessed, try to
			// unmarshal in an any
			bodyType = types.Interface
			unknownExpectedType = true

			// Special case for Ignore & NotEmpty operators
			switch op.GetLocation().Func {
			case "Ignore", "NotEmpty":
				showRawBody = t.statusFailed // Show real body if status failed
			}
		}
	} else {
		bodyType = reflect.TypeOf(expectedBody)
		if bodyType == nil {
			bodyType = types.Interface
		}
	}

	// For unmarshaling below, body must be a pointer
	bodyPtr := reflect.New(bodyType)

	// Try to unmarshal body
	if !tt.RootName("unmarshal(Response.Body)").
		CmpNoError(unmarshal(t.response.Body.Bytes(), bodyPtr.Interface()), t.name+"body unmarshaling") {
		// If unmarshal failed, perhaps it's coz the expected body type
		// is unknown?
		if unknownExpectedType {
			tt.Logf("Cannot guess the body expected type as %[1]s TestDeep\n"+
				"operator does not know the type behind it.\n"+
				"You can try All(Isa(EXPECTED_TYPE), %[1]s(…)) to disambiguate…",
				op.GetLocation().Func)
		}
		showRawBody = true // let's show its real body contents
		t.bodyFailed = true
	} else if !tt.Cmp(bodyPtr.Elem().Interface(), expectedBody, t.name+"body contents is OK") {
		// Try to catch bad body expected type when nothing has been set
		// to non-zero during unmarshaling body. In this case, require
		// to show raw body contents.
		if len(t.response.Body.Bytes()) > 0 &&
			td.EqDeeply(bodyPtr.Interface(), reflect.New(bodyType).Interface()) {
			showRawBody = true
			tt.Log("Hmm… It seems nothing has been set during unmarshaling…")
		}
		t.bodyFailed = true
	}

	if showRawBody || (t.bodyFailed && t.autoDumpResponse) {
		t.dumpResponse()
	}

	return t
}

// CmpMarshaledBody tests that the last request response body can be
// unmarshaled using unmarshal function and then, that it matches
// expectedBody. expectedBody can be any type unmarshal function can
// handle, or a [td.TestDeep] operator.
//
// See [TestAPI.CmpJSONBody] and [TestAPI.CmpXMLBody] sources for
// examples of use.
//
// It fails if no request has been sent yet.
func (t *TestAPI) CmpMarshaledBody(unmarshal func([]byte, any) error, expectedBody any) *TestAPI {
	t.t.Helper()
	return t.cmpMarshaledBody(false, unmarshal, expectedBody)
}

// CmpBody tests the last request response body against
// expectedBody. expectedBody can be a []byte, a string or a [td.TestDeep]
// operator.
//
//	ta := tdhttp.NewTestAPI(t, mux)
//
//	ta.Get("/test").
//	  CmpStatus(http.StatusOK).
//	  CmpBody("OK!\n")
//
//	ta.Get("/test").
//	  CmpStatus(http.StatusOK).
//	  CmpBody(td.Contains("OK"))
//
// It fails if no request has been sent yet.
func (t *TestAPI) CmpBody(expectedBody any) *TestAPI {
	t.t.Helper()

	if expectedBody == nil {
		return t.NoBody()
	}

	return t.cmpMarshaledBody(
		true, // accept empty body
		func(body []byte, target any) error {
			switch target := target.(type) {
			case *string:
				*target = string(body)
			case *[]byte:
				*target = body
			case *any:
				*target = body
			default:
				// cmpMarshaledBody always calls us with target as a pointer
				return fmt.Errorf(
					"CmpBody only accepts expectedBody be a []byte, a string or a TestDeep operator allowing to match these types, but not type %s",
					reflect.TypeOf(target).Elem())
			}
			return nil
		},
		expectedBody)
}

// CmpJSONBody tests that the last request response body can be
// [json.Unmarshal]'ed and that it matches expectedBody. expectedBody
// can be any type one can [json.Unmarshal] into, or a [td.TestDeep]
// operator.
//
//	ta := tdhttp.NewTestAPI(t, mux)
//
//	ta.Get("/person/42").
//	  CmpStatus(http.StatusOK).
//	  CmpJSONBody(Person{
//	    ID:   42,
//	    Name: "Bob",
//	    Age:  26,
//	  })
//
//	ta.PostJSON("/person", Person{Name: "Bob", Age: 23}).
//	  CmpStatus(http.StatusCreated).
//	  CmpJSONBody(td.SStruct(
//	    Person{
//	      Name: "Bob",
//	      Age:  26,
//	    },
//	    td.StructFields{
//	      "ID": td.NotZero(),
//	    }))
//
// The same with anchoring, and so without [td.SStruct]:
//
//	ta := tdhttp.NewTestAPI(tt, mux)
//
//	ta.PostJSON("/person", Person{Name: "Bob", Age: 23}).
//	  CmpStatus(http.StatusCreated).
//	  CmpJSONBody(Person{
//	    ID:   ta.Anchor(td.NotZero(), uint64(0)).(uint64),
//	    Name: "Bob",
//	    Age:  26,
//	  })
//
// The same using [td.JSON]:
//
//	ta.PostJSON("/person", Person{Name: "Bob", Age: 23}).
//	  CmpStatus(http.StatusCreated).
//	  CmpJSONBody(td.JSON(`
//	{
//	  "id":   NotZero(),
//	  "name": "Bob",
//	  "age":  26
//	}`))
//
// It fails if no request has been sent yet.
func (t *TestAPI) CmpJSONBody(expectedBody any) *TestAPI {
	t.t.Helper()
	return t.CmpMarshaledBody(json.Unmarshal, expectedBody)
}

// CmpXMLBody tests that the last request response body can be
// [xml.Unmarshal]'ed and that it matches expectedBody. expectedBody
// can be any type one can [xml.Unmarshal] into, or a [td.TestDeep]
// operator.
//
//	ta := tdhttp.NewTestAPI(t, mux)
//
//	ta.Get("/person/42").
//	  CmpStatus(http.StatusOK).
//	  CmpXMLBody(Person{
//	    ID:   42,
//	    Name: "Bob",
//	    Age:  26,
//	  })
//
//	ta.Get("/person/43").
//	  CmpStatus(http.StatusOK).
//	  CmpXMLBody(td.SStruct(
//	    Person{
//	      Name: "Bob",
//	      Age:  26,
//	    },
//	    td.StructFields{
//	      "ID": td.NotZero(),
//	    }))
//
// The same with anchoring:
//
//	ta := tdhttp.NewTestAPI(tt, mux)
//
//	ta.Get("/person/42").
//	  CmpStatus(http.StatusOK).
//	  CmpXMLBody(Person{
//	    ID:   ta.Anchor(td.NotZero(), uint64(0)).(uint64),
//	    Name: "Bob",
//	    Age:  26,
//	  })
//
// It fails if no request has been sent yet.
func (t *TestAPI) CmpXMLBody(expectedBody any) *TestAPI {
	t.t.Helper()
	return t.CmpMarshaledBody(xml.Unmarshal, expectedBody)
}

// NoBody tests that the last request response body is empty.
//
// It fails if no request has been sent yet.
func (t *TestAPI) NoBody() *TestAPI {
	defer t.t.AnchorsPersistTemporarily()()

	t.t.Helper()

	if !t.checkRequestSent() {
		t.bodyFailed = true
		return t
	}

	ok := t.t.RootName("Response.Body").
		Code(len(t.response.Body.Bytes()) == 0,
			func(empty bool) error {
				if empty {
					return nil
				}
				return &ctxerr.Error{
					Message:  "%% is not empty",
					Got:      types.RawString("not empty"),
					Expected: types.RawString("empty"),
				}
			},
			"body should be empty")
	if !ok {
		t.bodyFailed = true

		// Systematically dump response, no AutoDumpResponse needed
		t.dumpResponse()
	}

	return t
}

// Or executes function fn if t.Failed() is true at the moment it is called.
//
// fn can have several types:
//   - func(body string) or func(t *td.T, body string)
//     → fn is called with response body as a string.
//     If no response has been received yet, body is "";
//   - func(body []byte) or func(t *td.T, body []byte)
//     → fn is called with response body as a []byte.
//     If no response has been received yet, body is nil;
//   - func(t *td.T, resp *httptest.ResponseRecorder)
//     → fn is called with the internal object containing the response.
//     See net/http/httptest for details.
//     If no response has been received yet, resp is nil.
//
// If fn type is not one of these types, it calls t.T().Fatal().
func (t *TestAPI) Or(fn any) *TestAPI {
	t.t.Helper()
	switch fn := fn.(type) {
	case func(string):
		if t.Failed() {
			var body string
			if t.response != nil && t.response.Body != nil {
				body = t.response.Body.String()
			}
			fn(body)
		}

	case func(*td.T, string):
		if t.Failed() {
			var body string
			if t.response != nil && t.response.Body != nil {
				body = t.response.Body.String()
			}
			fn(t.t, body)
		}

	case func([]byte):
		if t.Failed() {
			var body []byte
			if t.response != nil && t.response.Body != nil {
				body = t.response.Body.Bytes()
			}
			fn(body)
		}

	case func(*td.T, []byte):
		if t.Failed() {
			var body []byte
			if t.response != nil && t.response.Body != nil {
				body = t.response.Body.Bytes()
			}
			fn(t.t, body)
		}

	case func(*td.T, *httptest.ResponseRecorder):
		if t.Failed() {
			fn(t.t, t.response)
		}

	default:
		t.t.Fatal(color.BadUsage(
			"Or(func([*td.T,]string) | func([*td.T,][]byte) | func(*td.T,*httptest.ResponseRecorder))",
			fn, 1, true))
	}

	return t
}

// OrDumpResponse dumps the response if at least one previous test failed.
//
//	ta := tdhttp.NewTestAPI(t, handler)
//
//	ta.Get("/foo").
//	  CmpStatus(200).
//	  OrDumpResponse(). // if status check failed, dumps the response
//	  CmpBody("bar")    // if it fails, the response is not dumped
//
//	ta.Get("/foo").
//	  CmpStatus(200).
//	  CmpBody("bar").
//	  OrDumpResponse() // dumps the response if status and/or body checks fail
//
// See [TestAPI.AutoDumpResponse] method to automatize this dump.
func (t *TestAPI) OrDumpResponse() *TestAPI {
	if t.Failed() {
		t.dumpResponse()
	}
	return t
}

func (t *TestAPI) dumpResponse() {
	if t.responseDumped {
		return
	}

	t.t.Helper()
	if t.response != nil {
		t.responseDumped = true
		internal.DumpResponse(t.t, t.response.Result())
		return
	}

	t.t.Logf("No response received yet")
}

// Anchor returns a typed value allowing to anchor the [td.TestDeep]
// operator operator in a go classic literal like a struct, slice,
// array or map value.
//
//	ta := tdhttp.NewTestAPI(tt, mux)
//
//	ta.Get("/person/42").
//	  CmpStatus(http.StatusOK).
//	  CmpJSONBody(Person{
//	    ID:   ta.Anchor(td.NotZero(), uint64(0)).(uint64),
//	    Name: "Bob",
//	    Age:  26,
//	  })
//
// See [td.T.Anchor] for details.
//
// See [TestAPI.A] method for a shorter synonym of Anchor.
func (t *TestAPI) Anchor(operator td.TestDeep, model ...any) any {
	return t.t.Anchor(operator, model...)
}

// A is a synonym for [TestAPI.Anchor]. It returns a typed value allowing to
// anchor the [td.TestDeep] operator in a go classic literal
// like a struct, slice, array or map value.
//
//	ta := tdhttp.NewTestAPI(tt, mux)
//
//	ta.Get("/person/42").
//	  CmpStatus(http.StatusOK).
//	  CmpJSONBody(Person{
//	    ID:   ta.A(td.NotZero(), uint64(0)).(uint64),
//	    Name: "Bob",
//	    Age:  26,
//	  })
//
// See [td.T.Anchor] for details.
func (t *TestAPI) A(operator td.TestDeep, model ...any) any {
	return t.Anchor(operator, model...)
}

// SentAt returns the time just before the last request is handled. It
// can be used to check the time a route sets and returns, as in:
//
//	ta.PostJSON("/person/42", Person{Name: "Bob", Age: 23}).
//	  CmpStatus(http.StatusCreated).
//	  CmpJSONBody(Person{
//	    ID:        ta.A(td.NotZero(), uint64(0)).(uint64),
//	    Name:      "Bob",
//	    Age:       23,
//	    CreatedAt: ta.A(td.Between(ta.SentAt(), time.Now())).(time.Time),
//	  })
//
// checks that CreatedAt field is included between the time when the
// request has been sent, and the time when the comparison occurs.
func (t *TestAPI) SentAt() time.Time {
	return t.sentAt
}
