<?php

declare(strict_types=1);

/*
 * Copyright (c) 2023-2024 François Kooman <fkooman@tuxed.net>
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

namespace fkooman\Radius\Tests;

use fkooman\Radius\MessageAuthenticator;
use fkooman\Radius\Password;
use fkooman\Radius\RadiusPacket;
use PHPUnit\Framework\TestCase;

/**
 * @covers \fkooman\Radius\MessageAuthenticator
 *
 * @uses \fkooman\Radius\RadiusPacket
 * @uses \fkooman\Radius\Utils
 * @uses \fkooman\Radius\AttributeCollection
 * @uses \fkooman\Radius\Password
 */
class MessageAuthenticatorTest extends TestCase
{
    public function testCalculate(): void
    {
        $radiusPacket = new RadiusPacket(RadiusPacket::ACCESS_REQUEST, 0, hex2bin('6c91db38edcfa26a53c2c99891d49a43'));
        $radiusPacket->attributeCollection()->set('User-Name', 'foo');
        $radiusPacket->attributeCollection()->set('User-Password', Password::encrypt('bar', hex2bin('6c91db38edcfa26a53c2c99891d49a43'), 's3cr3t'));
        $radiusPacket->attributeCollection()->set('NAS-Identifier', 'my-nas-id');
        $this->assertSame('83c6328679671c385e16362238ef7f0d', bin2hex(MessageAuthenticator::calculate($radiusPacket, 's3cr3t')));
    }

    public function testVerify(): void
    {
        //(LOG) I Access-Request for User-Name "foo"
        //(LOG) D Access-Request for User-Password "bar"
        //(LOG) D Server: udp://10.43.43.5:1812
        //(LOG) D --> 01000048445ddd60b2a31e4e6b1b6d0d17b7b0c30105666f6f021257fb55787ea1780b3f1c46b3dd6cb1f6200b6d792d6e61732d69645012668183f84ab1f5b6e9f9f543e6fce0d3
        //(LOG) D <-- 02000040aace592a1f353c07774022ad02c6d83e121a41757468656e7469636174696f6e2073756363656564656450126c6bfa01eadd064c3076bbb7ce8850a1
        $responsePacket = RadiusPacket::fromBytes(
            hex2bin(
                '02000040aace592a1f353c07774022ad02c6d83e121a41757468656e7469636174696f6e2073756363656564656450126c6bfa01eadd064c3076bbb7ce8850a1'
            )
        );
        $this->assertTrue(MessageAuthenticator::verify($responsePacket, 's3cr3t', hex2bin('445ddd60b2a31e4e6b1b6d0d17b7b0c3')));
    }
}
