/*
 * Copyright (c) 2022 Andrea Biscuola <a@abiscuola.com>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

// rssgoemail: read your favourite RSS/Atom and gemini feeds from your
// e-mail client.
//
// Rssgoemail is a tool, inspired by rss2email to fetch your favourite feeds
// and send them in your local mailbox. rssgoemail can fetch the following
// feed formats:
//
//   - RSS/Atom from an HTTP/HTTPS URL.
//   - RSS/Atom from a gemini connection.
//   - Gemfeed from a gemini connection
//
// Already sent items are kept track of, to avoid sending them again with
// following fetches of the same feed.
package main

import (
	"bufio"
	"crypto/md5"
	"flag"
	"fmt"
	"os"
	"os/exec"
	"strings"
	"text/template"
	"time"
)

// Shared feed structure. All the supported feeds types uses this data
// structure to store the items.
type Feed struct {
	Author          string
	Email           string
	Title           string
	PublishedParsed time.Time
	Published       string
	Link            string
	Description     string
}

func (f *Feed) FormatPublished() string {
	return f.PublishedParsed.Format(time.RFC1123Z)
}

var mailTemplate = template.Must(template.New("mail").
	Parse(`{{/* eat the newline */ -}}
From: {{ printf "%q" .Feed.Author }} <{{ .Feed.Email }}>
To: {{ .Recipient }}
Date: {{ .Feed.FormatPublished }}
X-RSSGo-Email: {{ .Feed.Link }}
Subject: {{ .Feed.Title }}

URL: {{ .Feed.Link }}

{{ .Feed.Description }}
`))

// Function Feed is used to calculate a feed hash.
//
// The hash is the concatenation of the feed title + it's publishing date.
// Once calculated, the hash is used to check if the feed item was already
// fetched, storing the hash along it's published date in the cache file
// if not.
func (f *Feed) Hash() string {
	return fmt.Sprintf("%x", md5.Sum([]byte(fmt.Sprintf("%s%s", f.Title, f.Published))))
}

var centries = make(map[string]interface{})

func main() {
	cfg := flag.String("c", fmt.Sprintf("%s/.rssgoemail.conf", os.Getenv("HOME")), "alternative configuration file")
	flag.Parse()

	cachedir := fmt.Sprintf("%s/.cache/rssgoemail", os.Getenv("HOME"))

	file, e := os.Open(*cfg)
	if e != nil {
		fmt.Fprintln(os.Stderr, e)
		os.Exit(1)
	}

	/*
	 * We load the cache in memory in a map. It's going to use a bit of
	 * additional RAM, but searching in a map is much faster.
	 */

	e = os.MkdirAll(cachedir, os.ModeDir+0750)
	if e != nil {
		fmt.Fprintln(os.Stderr, e)
		os.Exit(1)
	}

	cachefile := fmt.Sprintf("%s/hashes", cachedir)
	cfile, e := os.OpenFile(cachefile, os.O_RDONLY|os.O_CREATE, 0640)
	if e != nil {
		fmt.Fprintln(os.Stderr, e)
		os.Exit(1)
	}

	/*
	 * Try to fetch and process every URL in the configurations file
	 */

	fscan := bufio.NewScanner(cfile)
	for fscan.Scan() {
		/* We will remove the split in a future version */
		split := strings.Split(fscan.Text(), "\t")
		centries[split[0]] = nil
	}
	cfile.Close()

	scanner := bufio.NewScanner(file)
	for scanner.Scan() {
		text := scanner.Text()
		if text == "" || strings.HasPrefix(text, "#") {
			continue
		}
		if e := fetchFeed(text); e != nil {
			fmt.Fprintln(os.Stderr, e)
		}
	}
	file.Close()

	/*
	 * We save the cache back in the file, truncating it
	 */

	file, e = os.Create(cachefile)
	if e != nil {
		fmt.Fprintln(os.Stderr, e)
		os.Exit(1)
	}
	defer file.Close()

	for k, _ := range centries {
		if _, e := file.WriteString(k + "\n"); e != nil {
			fmt.Fprintln(os.Stderr, e)
			os.Exit(1)
		}
	}
}

// Just pipe a basic e-mail to sendmail.
func sendFeed(item Feed) error {
	user := os.Getenv("USER")
	if user == "" {
		return fmt.Errorf("missing USER environment variable")
	}

	cmd := exec.Command("sendmail", "-t")
	pipe, e := cmd.StdinPipe()
	if e != nil {
		return e
	}
	cmd.Stdout = os.Stdout
	cmd.Stderr = os.Stderr

	e = cmd.Start()
	if e != nil {
		pipe.Close()
		return e
	}

	e = mailTemplate.Execute(pipe, struct {
		Feed      *Feed
		Recipient string
	}{&item, user})
	if e != nil {
		pipe.Close()
		cmd.Wait()
		return e
	}

	pipe.Close()
	return cmd.Wait()
}

// Fetch the feeds.
//
// For every feed, we fetch the data and store all of the non cached items in
// an array of pointers to Feed structures.
//
// All of the fetched items that are not in cache, are sent to the user actually
// running the program.
func fetchFeed(url string) error {
	var feeds []Feed
	var err error

	switch {
	case strings.HasPrefix(url, "https://") || strings.HasPrefix(url, "http://"):
		feeds, err = FetchRss(url)
	case strings.HasPrefix(url, "gemini://"):
		feeds, err = FetchGemfeed(url)
	default:
		err = fmt.Errorf("unknown url type")
	}

	if err != nil {
		return fmt.Errorf("failed to fetch %s: %w", url, err)
	}

	for _, feed := range feeds {
		if _, ok := centries[feed.Hash()]; ok {
			continue
		}
		if err := sendFeed(feed); err != nil {
			fmt.Fprintln(os.Stderr, err)
			continue
		}
		centries[feed.Hash()] = nil
	}

	return nil
}
