import * as React from 'react'
import * as Constants from '../../constants/teams'
import * as Types from '../../constants/types/teams'
import * as FsConstants from '../../constants/fs'
import * as FsTypes from '../../constants/types/fs'
import * as Container from '../../util/container'
import * as RouteTreeGen from '../../actions/route-tree-gen'
import * as Kb from '../../common-adapters'

type OwnProps = {
  attachTo?: () => React.Component<any> | null
  onHidden: () => void
  teamID: Types.TeamID
  visible: boolean
}

const mapStateToProps = (state: Container.TypedState, {teamID}: OwnProps) => {
  const teamDetails = Constants.getTeamDetails(state, teamID)
  const yourOperations = Constants.getCanPerformByID(state, teamID)
  const isBigTeam = Constants.isBigTeam(state, teamDetails.teamname)
  return {
    canCreateSubteam: yourOperations.manageSubteams,
    canDeleteTeam: yourOperations.deleteTeam && teamDetails.subteams?.size === 0,
    canManageChat: yourOperations.renameChannel,
    canViewFolder: !yourOperations.joinTeam,
    isBigTeam,
    teamname: teamDetails.teamname,
  }
}

const mapDispatchToProps = (dispatch: Container.TypedDispatch, {teamID}: OwnProps) => ({
  onCreateSubteam: () =>
    dispatch(
      RouteTreeGen.createNavigateAppend({
        path: [{props: {subteamOf: teamID}, selected: 'teamNewTeamDialog'}],
      })
    ),
  onDeleteTeam: () =>
    dispatch(
      RouteTreeGen.createNavigateAppend({
        path: [{props: {teamID}, selected: 'teamDeleteTeam'}],
      })
    ),
  onLeaveTeam: () =>
    dispatch(
      RouteTreeGen.createNavigateAppend({
        path: [{props: {teamID}, selected: 'teamReallyLeaveTeam'}],
      })
    ),
  onManageChat: () =>
    dispatch(
      RouteTreeGen.createNavigateAppend({
        path: [{props: {teamID}, selected: 'chatManageChannels'}],
      })
    ),
  onOpenFolder: (teamname: string) =>
    dispatch(FsConstants.makeActionForOpenPathInFilesTab(FsTypes.stringToPath(`/keybase/team/${teamname}`))),
})

type Props = {
  attachTo?: () => React.Component<any> | null
  items: Kb.MenuItems
  onHidden: () => void
  visible: boolean
}

const TeamMenu = ({attachTo, items, onHidden, visible}: Props) => {
  if (visible && items.length === 0) {
    onHidden()
    return null
  }
  return (
    <Kb.FloatingMenu
      attachTo={attachTo}
      closeOnSelect={true}
      items={items}
      onHidden={onHidden}
      visible={visible}
    />
  )
}

export default Container.connect(
  mapStateToProps,
  mapDispatchToProps,
  (stateProps, dispatchProps, ownProps: OwnProps) => {
    const items: Kb.MenuItems = []
    if (stateProps.canManageChat) {
      items.push({
        icon: 'iconfont-hash',
        onClick: dispatchProps.onManageChat,
        subTitle: stateProps.isBigTeam ? undefined : 'Turns this into a big team',
        title: stateProps.isBigTeam ? 'Manage chat channels' : 'Make chat channels...',
      })
    }
    if (stateProps.canCreateSubteam) {
      items.push({icon: 'iconfont-people', onClick: dispatchProps.onCreateSubteam, title: 'Create subteam'})
    }
    if (stateProps.canViewFolder) {
      items.push({
        icon: 'iconfont-folder-open',
        onClick: () => dispatchProps.onOpenFolder(stateProps.teamname),
        title: 'Open folder',
      })
    }
    items.push({
      danger: true,
      icon: 'iconfont-leave',
      onClick: dispatchProps.onLeaveTeam,
      title: 'Leave team',
    })
    if (stateProps.canDeleteTeam) {
      items.push({
        danger: true,
        icon: 'iconfont-remove',
        onClick: dispatchProps.onDeleteTeam,
        title: 'Delete team',
      })
    }

    return {
      attachTo: ownProps.attachTo,
      items,
      onHidden: ownProps.onHidden,
      visible: ownProps.visible,
    }
  }
)(TeamMenu) as any
