// Example: Set the session parameter in DSN and verify it
//
package main

import (
	"database/sql"
	"flag"
	"fmt"
	"log"
	"os"
	"strconv"

	sf "github.com/influxdata/gosnowflake"
)

// getDSN constructs a DSN based on the test connection parameters
func getDSN(params map[string]*string) (string, *sf.Config, error) {
	env := func(k string, failOnMissing bool) string {
		if value := os.Getenv(k); value != "" {
			return value
		}
		if failOnMissing {
			log.Fatalf("%v environment variable is not set.", k)
		}
		return ""
	}

	account := env("SNOWFLAKE_TEST_ACCOUNT", true)
	user := env("SNOWFLAKE_TEST_USER", true)
	password := env("SNOWFLAKE_TEST_PASSWORD", true)
	host := env("SNOWFLAKE_TEST_HOST", false)
	port := env("SNOWFLAKE_TEST_PORT", false)
	protocol := env("SNOWFLAKE_TEST_PROTOCOL", false)

	portStr, err := strconv.Atoi(port)
	if err != nil {
		return "", nil, err
	}
	cfg := &sf.Config{
		Account:  account,
		User:     user,
		Password: password,
		Host:     host,
		Port:     portStr,
		Protocol: protocol,
		Params:   params,
	}

	dsn, err := sf.DSN(cfg)
	return dsn, cfg, err
}

func main() {
	if !flag.Parsed() {
		flag.Parse()
	}

	tmfmt := "MM-DD-YYYY"
	dsn, cfg, err := getDSN(
		map[string]*string{
			"TIMESTAMP_OUTPUT_FORMAT": &tmfmt, // session parameter
		})
	if err != nil {
		log.Fatalf("failed to create DSN from Config: %v, err: %v", cfg, err)
	}
	if err != nil {
		log.Fatalf("failed to create DSN from Config: %v, err: %v", cfg, err)
	}
	db, err := sql.Open("snowflake", dsn)
	if err != nil {
		log.Fatalf("failed to connect. err: %v", err)
	}
	defer db.Close()
	query := "SHOW PARAMETERS LIKE 'TIMESTAMP_OUTPUT_FORMAT'"
	rows, err := db.Query(query) // no cancel is allowed
	if err != nil {
		log.Fatalf("failed to run a query. %v, err: %v", query, err)
	}
	defer rows.Close()

	for rows.Next() {
		p, err := sf.ScanSnowflakeParameter(rows)
		if err != nil {
			log.Fatalf("failed to get result. err: %v", err)
		}
		if p.Key != "TIMESTAMP_OUTPUT_FORMAT" {
			log.Fatalf("failed to get TIMESTAMP_. got: %v", p.Value)
		}
		fmt.Printf("fmt: %v\n", p.Value)
	}
	if rows.Err() != nil {
		fmt.Printf("ERROR: %v\n", rows.Err())
		return
	}
	fmt.Printf("Congrats! You have successfully run %v with Snowflake DB!\n", query)
}
