// Most of this is from gofmt:
// Copyright 2009 The Go Authors. All rights reserved.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

package main

import (
	"bytes"
	"errors"
	"flag"
	"fmt"
	"io"
	"io/ioutil"
	"log"
	"os"
	"os/exec"
	"path/filepath"
	"runtime"
	"strings"

	"github.com/stuart-warren/yamlfmt"
)

var (
	list   bool
	write  bool
	doDiff bool
	doFail bool

	errRequiresFmt = errors.New("RequiresFmt")
)

func main() {
	err := run(os.Stdin, os.Stdout, os.Args)
	if err == errRequiresFmt {
		os.Exit(1)
	}
	if err != nil {
		log.Fatalln(err)
	}
}

func run(in io.Reader, out io.Writer, args []string) error {
	flags := flag.NewFlagSet(args[0], flag.ExitOnError)

	flags.BoolVar(&list, "l", false, "list files whose formatting differs from yamlfmt's")
	flags.BoolVar(&write, "w", false, "write result to (source) file instead of stdout")
	flags.BoolVar(&doDiff, "d", false, "display diffs instead of rewriting files")
	flags.BoolVar(&doFail, "f", false, "exit non zero if changes detected")
	flags.Usage = func() {
		fmt.Fprintf(os.Stderr, "formats yaml files with 2 space indent, sorted dicts and non-indented lists\n")
		fmt.Fprintf(os.Stderr, "usage: yamlfmt [flags] [path ...]\n")
		flags.PrintDefaults()
	}
	flags.Parse(args[1:])

	if flags.NArg() == 0 {
		if write {
			return fmt.Errorf("error: cannot use -w with standard input")
		}
		if err := processFile("<standard input>", in, out, true); err != nil {
			return err
		}
	}

	for i := 0; i < flags.NArg(); i++ {
		path := flags.Arg(i)
		switch dir, err := os.Stat(path); {
		case err != nil:
			return err
		case dir.IsDir():
			return walkDir(path)
		default:
			if err := processFile(path, nil, os.Stdout, false); err != nil {
				return err
			}
		}
	}
	return nil
}

// If in == nil, the source is the contents of the file with the given filename.
func processFile(filename string, in io.Reader, out io.Writer, stdin bool) error {
	var perm os.FileMode = 0644
	if in == nil {
		f, err := os.Open(filename)
		if err != nil {
			return err
		}
		defer f.Close()
		fi, err := f.Stat()
		if err != nil {
			return err
		}
		in = f
		perm = fi.Mode().Perm()
	}

	src, err := ioutil.ReadAll(in)
	if err != nil {
		return err
	}

	res, err := yamlfmt.Format(bytes.NewBuffer(src))
	if err != nil {
		return err
	}

	if !list && !write && !doDiff {
		_, err = out.Write(res)
	}

	if !bytes.Equal(src, res) {
		// formatting has changed
		if list {
			fmt.Fprintln(out, filename)
		}
		if write {
			// make a temporary backup before overwriting original
			bakname, err := backupFile(filename+".", src, perm)
			if err != nil {
				return err
			}
			err = ioutil.WriteFile(filename, res, perm)
			if err != nil {
				os.Rename(bakname, filename)
				return err
			}
			err = os.Remove(bakname)
			if err != nil {
				return err
			}
		}
		if doDiff {
			data, err := diff(src, res, filename)
			if err != nil {
				return fmt.Errorf("computing diff: %s", err)
			}
			fmt.Printf("diff -u %s %s\n", filepath.ToSlash(filename+".orig"), filepath.ToSlash(filename))
			out.Write(data)
		}
		if doFail {
			return errRequiresFmt
		}
	}

	return err
}

type fileVisitor struct {
	changesDetected bool
}

func (fv *fileVisitor) visitFile(path string, f os.FileInfo, err error) error {
	if err == nil && isYamlFile(f) {
		err = processFile(path, nil, os.Stdout, false)
	}
	// Don't complain if a file was deleted in the meantime (i.e.
	// the directory changed concurrently while running gofmt).
	if err != nil && !os.IsNotExist(err) && err != errRequiresFmt {
		return err
	}
	if err == errRequiresFmt {
		fv.changesDetected = true
	}
	return nil
}

func walkDir(path string) error {
	fv := fileVisitor{}
	filepath.Walk(path, fv.visitFile)
	var err error
	if fv.changesDetected {
		err = errRequiresFmt
	}
	return err
}

const chmodSupported = runtime.GOOS != "windows"

// backupFile writes data to a new file named filename<number> with permissions perm,
// with <number randomly chosen such that the file name is unique. backupFile returns
// the chosen file name.
func backupFile(filename string, data []byte, perm os.FileMode) (string, error) {
	// create backup file
	f, err := ioutil.TempFile(filepath.Dir(filename), filepath.Base(filename))
	if err != nil {
		return "", err
	}
	bakname := f.Name()
	if chmodSupported {
		err = f.Chmod(perm)
		if err != nil {
			f.Close()
			os.Remove(bakname)
			return bakname, err
		}
	}

	// write data to backup file
	_, err = f.Write(data)
	if err1 := f.Close(); err == nil {
		err = err1
	}

	return bakname, err
}

func isYamlFile(f os.FileInfo) bool {
	// ignore non-Yaml files
	name := f.Name()
	return !f.IsDir() && (strings.HasSuffix(name, ".yaml") || strings.HasSuffix(name, ".yml"))
}

func writeTempFile(dir, prefix string, data []byte) (string, error) {
	file, err := ioutil.TempFile(dir, prefix)
	if err != nil {
		return "", err
	}
	_, err = file.Write(data)
	if err1 := file.Close(); err == nil {
		err = err1
	}
	if err != nil {
		os.Remove(file.Name())
		return "", err
	}
	return file.Name(), nil
}

// replaceTempFilename replaces temporary filenames in diff with actual one.
//
// --- /tmp/yamlfmt316145376	2017-02-03 19:13:00.280468375 -0500
// +++ /tmp/yamlfmt617882815	2017-02-03 19:13:00.280468375 -0500
// ...
// ->
// --- path/to/file.yaml.orig	2017-02-03 19:13:00.280468375 -0500
// +++ path/to/file.yaml	2017-02-03 19:13:00.280468375 -0500
// ...
func replaceTempFilename(diff []byte, filename string) ([]byte, error) {
	bs := bytes.SplitN(diff, []byte{'\n'}, 3)
	if len(bs) < 3 {
		return nil, fmt.Errorf("got unexpected diff for %s", filename)
	}
	// Preserve timestamps.
	var t0, t1 []byte
	if i := bytes.LastIndexByte(bs[0], '\t'); i != -1 {
		t0 = bs[0][i:]
	}
	if i := bytes.LastIndexByte(bs[1], '\t'); i != -1 {
		t1 = bs[1][i:]
	}
	// Always print filepath with slash separator.
	f := filepath.ToSlash(filename)
	bs[0] = []byte(fmt.Sprintf("--- %s%s", f+".orig", t0))
	bs[1] = []byte(fmt.Sprintf("+++ %s%s", f, t1))
	return bytes.Join(bs, []byte{'\n'}), nil
}

func diff(b1, b2 []byte, filename string) (data []byte, err error) {
	f1, err := writeTempFile("", "yamlfmt", b1)
	if err != nil {
		return
	}
	defer os.Remove(f1)

	f2, err := writeTempFile("", "yamlfmt", b2)
	if err != nil {
		return
	}
	defer os.Remove(f2)

	cmd := "diff"
	data, err = exec.Command(cmd, "-u", f1, f2).CombinedOutput()
	if len(data) > 0 {
		// diff exits with a non-zero status when the files don't match.
		// Ignore that failure as long as we get output.
		return replaceTempFilename(data, filename)
	}
	return
}
