// SPDX-FileCopyrightText: 2023 The Pion community <https://pion.ly>
// SPDX-License-Identifier: MIT

package srtp

import (
	"testing"

	"github.com/stretchr/testify/assert"
)

type testCipher struct {
	profile    ProtectionProfile // Protection profile
	masterKey  []byte            // Master key
	masterSalt []byte            // Master salt
	mki        []byte            // Master key identifier

	decryptedRTPPacket            []byte
	encryptedRTPPacket            []byte
	encryptedRTPPacketWithMKI     []byte
	authenticatedRTPPacket        []byte
	authenticatedRTPPacketWithMKI []byte

	decryptedRTCPPacket            []byte
	encryptedRTCPPacket            []byte
	encryptedRTCPPacketWithMKI     []byte
	authenticatedRTCPPacket        []byte
	authenticatedRTCPPacketWithMKI []byte
}

// create array of testCiphers for each supported profile.
func createTestCiphers(t *testing.T) []testCipher { //nolint:maintidx
	t.Helper()

	tests := []testCipher{
		{ //nolint:dupl
			profile: ProtectionProfileAes128CmHmacSha1_32,
			encryptedRTPPacket: []byte{
				0x80, 0x0f, 0x12, 0x34, 0xde, 0xca, 0xfb, 0xad,
				0xca, 0xfe, 0xba, 0xbe, 0xe2, 0xd8, 0xdf, 0x8f,
				0x7a, 0x75, 0xd6, 0x88, 0xc3, 0x50, 0x2e, 0xee,
				0xc2, 0xa9, 0x80, 0x66, 0xcd, 0x7c, 0x0d, 0x09,
			},
			encryptedRTCPPacket: []byte{
				0x81, 0xc8, 0x00, 0x0b, 0xca, 0xfe, 0xba, 0xbe,
				0x56, 0x74, 0xbf, 0x01, 0x81, 0x3d, 0xc0, 0x62,
				0xac, 0x1d, 0xf6, 0xf7, 0x5f, 0x77, 0xc6, 0x88,
				0x80, 0x00, 0x00, 0x01, 0x3d, 0xb7, 0xa1, 0x98,
				0x37, 0xff, 0x64, 0xe5, 0xcb, 0xd2,
			},
			encryptedRTPPacketWithMKI: []byte{
				0x80, 0x0f, 0x12, 0x34, 0xde, 0xca, 0xfb, 0xad,
				0xca, 0xfe, 0xba, 0xbe, 0xe2, 0xd8, 0xdf, 0x8f,
				0x7a, 0x75, 0xd6, 0x88, 0xc3, 0x50, 0x2e, 0xee,
				0xc2, 0xa9, 0x80, 0x66, 0x01, 0x02, 0x00, 0x04,
				0xcd, 0x7c, 0x0d, 0x09,
			},
			encryptedRTCPPacketWithMKI: []byte{
				0x81, 0xc8, 0x00, 0x0b, 0xca, 0xfe, 0xba, 0xbe,
				0x56, 0x74, 0xbf, 0x01, 0x81, 0x3d, 0xc0, 0x62,
				0xac, 0x1d, 0xf6, 0xf7, 0x5f, 0x77, 0xc6, 0x88,
				0x80, 0x00, 0x00, 0x01, 0x01, 0x02, 0x00, 0x04,
				0x3d, 0xb7, 0xa1, 0x98, 0x37, 0xff, 0x64, 0xe5,
				0xcb, 0xd2,
			},
			authenticatedRTPPacket: []byte{
				0x80, 0x0f, 0x12, 0x34, 0xde, 0xca, 0xfb, 0xad,
				0xca, 0xfe, 0xba, 0xbe, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0xda, 0x9a, 0x3c, 0xa1,
			},
			authenticatedRTCPPacket: []byte{
				0x81, 0xc8, 0x00, 0x0b, 0xca, 0xfe, 0xba, 0xbe,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0x00, 0x00, 0x00, 0x01, 0xd2, 0xa2, 0x36, 0x2d,
				0x01, 0x1b, 0x8c, 0xfc, 0x0a, 0xc9,
			},
			authenticatedRTPPacketWithMKI: []byte{
				0x80, 0x0f, 0x12, 0x34, 0xde, 0xca, 0xfb, 0xad,
				0xca, 0xfe, 0xba, 0xbe, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0x01, 0x02, 0x00, 0x04,
				0xda, 0x9a, 0x3c, 0xa1,
			},
			authenticatedRTCPPacketWithMKI: []byte{
				0x81, 0xc8, 0x00, 0x0b, 0xca, 0xfe, 0xba, 0xbe,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0x00, 0x00, 0x00, 0x01, 0x01, 0x02, 0x00, 0x04,
				0xd2, 0xa2, 0x36, 0x2d, 0x01, 0x1b, 0x8c, 0xfc,
				0x0a, 0xc9,
			},
		},
		{ //nolint:dupl
			profile: ProtectionProfileAes128CmHmacSha1_80,
			encryptedRTPPacket: []byte{
				0x80, 0x0f, 0x12, 0x34, 0xde, 0xca, 0xfb, 0xad,
				0xca, 0xfe, 0xba, 0xbe, 0xe2, 0xd8, 0xdf, 0x8f,
				0x7a, 0x75, 0xd6, 0x88, 0xc3, 0x50, 0x2e, 0xee,
				0xc2, 0xa9, 0x80, 0x66, 0xcd, 0x7c, 0x0d, 0x09,
				0xca, 0x44, 0x32, 0xa5, 0x6e, 0x3d,
			},
			encryptedRTCPPacket: []byte{
				0x81, 0xc8, 0x00, 0x0b, 0xca, 0xfe, 0xba, 0xbe,
				0x56, 0x74, 0xbf, 0x01, 0x81, 0x3d, 0xc0, 0x62,
				0xac, 0x1d, 0xf6, 0xf7, 0x5f, 0x77, 0xc6, 0x88,
				0x80, 0x00, 0x00, 0x01, 0x3d, 0xb7, 0xa1, 0x98,
				0x37, 0xff, 0x64, 0xe5, 0xcb, 0xd2,
			},
			encryptedRTPPacketWithMKI: []byte{
				0x80, 0x0f, 0x12, 0x34, 0xde, 0xca, 0xfb, 0xad,
				0xca, 0xfe, 0xba, 0xbe, 0xe2, 0xd8, 0xdf, 0x8f,
				0x7a, 0x75, 0xd6, 0x88, 0xc3, 0x50, 0x2e, 0xee,
				0xc2, 0xa9, 0x80, 0x66, 0x01, 0x02, 0x00, 0x04,
				0xcd, 0x7c, 0x0d, 0x09, 0xca, 0x44, 0x32, 0xa5,
				0x6e, 0x3d,
			},
			encryptedRTCPPacketWithMKI: []byte{
				0x81, 0xc8, 0x00, 0x0b, 0xca, 0xfe, 0xba, 0xbe,
				0x56, 0x74, 0xbf, 0x01, 0x81, 0x3d, 0xc0, 0x62,
				0xac, 0x1d, 0xf6, 0xf7, 0x5f, 0x77, 0xc6, 0x88,
				0x80, 0x00, 0x00, 0x01, 0x01, 0x02, 0x00, 0x04,
				0x3d, 0xb7, 0xa1, 0x98, 0x37, 0xff, 0x64, 0xe5,
				0xcb, 0xd2,
			},
			authenticatedRTPPacket: []byte{
				0x80, 0x0f, 0x12, 0x34, 0xde, 0xca, 0xfb, 0xad,
				0xca, 0xfe, 0xba, 0xbe, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0xda, 0x9a, 0x3c, 0xa1,
				0xba, 0x8e, 0xfd, 0xd7, 0x07, 0xdc,
			},
			authenticatedRTCPPacket: []byte{
				0x81, 0xc8, 0x00, 0x0b, 0xca, 0xfe, 0xba, 0xbe,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0x00, 0x00, 0x00, 0x01, 0xd2, 0xa2, 0x36, 0x2d,
				0x01, 0x1b, 0x8c, 0xfc, 0x0a, 0xc9,
			},
			authenticatedRTPPacketWithMKI: []byte{
				0x80, 0x0f, 0x12, 0x34, 0xde, 0xca, 0xfb, 0xad,
				0xca, 0xfe, 0xba, 0xbe, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0x01, 0x02, 0x00, 0x04,
				0xda, 0x9a, 0x3c, 0xa1, 0xba, 0x8e, 0xfd, 0xd7,
				0x07, 0xdc,
			},
			authenticatedRTCPPacketWithMKI: []byte{
				0x81, 0xc8, 0x00, 0x0b, 0xca, 0xfe, 0xba, 0xbe,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0x00, 0x00, 0x00, 0x01, 0x01, 0x02, 0x00, 0x04,
				0xd2, 0xa2, 0x36, 0x2d, 0x01, 0x1b, 0x8c, 0xfc,
				0x0a, 0xc9,
			},
		},
		{ //nolint:dupl
			profile: ProtectionProfileAes256CmHmacSha1_32,
			encryptedRTPPacket: []byte{
				0x80, 0x0f, 0x12, 0x34, 0xde, 0xca, 0xfb, 0xad,
				0xca, 0xfe, 0xba, 0xbe, 0xac, 0x3b, 0xca, 0x88,
				0x14, 0x37, 0x57, 0x83, 0x35, 0xc6, 0xd4, 0x57,
				0xf1, 0xc3, 0x6b, 0xa7, 0x3d, 0x71, 0x48, 0x63,
			},
			encryptedRTCPPacket: []byte{
				0x81, 0xc8, 0x00, 0x0b, 0xca, 0xfe, 0xba, 0xbe,
				0x97, 0x04, 0x31, 0xdc, 0x4a, 0xe6, 0xd2, 0xaf,
				0xd6, 0x54, 0xbf, 0x90, 0xf4, 0x35, 0x44, 0x9e,
				0x80, 0x00, 0x00, 0x01, 0xbf, 0x18, 0x18, 0x2d,
				0xd1, 0x18, 0x81, 0x28, 0x78, 0xb1,
			},
			encryptedRTPPacketWithMKI: []byte{
				0x80, 0x0f, 0x12, 0x34, 0xde, 0xca, 0xfb, 0xad,
				0xca, 0xfe, 0xba, 0xbe, 0xac, 0x3b, 0xca, 0x88,
				0x14, 0x37, 0x57, 0x83, 0x35, 0xc6, 0xd4, 0x57,
				0xf1, 0xc3, 0x6b, 0xa7, 0x01, 0x02, 0x00, 0x04,
				0x3d, 0x71, 0x48, 0x63,
			},
			encryptedRTCPPacketWithMKI: []byte{
				0x81, 0xc8, 0x00, 0x0b, 0xca, 0xfe, 0xba, 0xbe,
				0x97, 0x04, 0x31, 0xdc, 0x4a, 0xe6, 0xd2, 0xaf,
				0xd6, 0x54, 0xbf, 0x90, 0xf4, 0x35, 0x44, 0x9e,
				0x80, 0x00, 0x00, 0x01, 0x01, 0x02, 0x00, 0x04,
				0xbf, 0x18, 0x18, 0x2d, 0xd1, 0x18, 0x81, 0x28,
				0x78, 0xb1,
			},
			authenticatedRTPPacket: []byte{
				0x80, 0x0f, 0x12, 0x34, 0xde, 0xca, 0xfb, 0xad,
				0xca, 0xfe, 0xba, 0xbe, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0x3d, 0x03, 0x2a, 0x52,
			},
			authenticatedRTCPPacket: []byte{
				0x81, 0xc8, 0x00, 0x0b, 0xca, 0xfe, 0xba, 0xbe,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0x00, 0x00, 0x00, 0x01, 0xf6, 0xd9, 0xd0, 0xc1,
				0x44, 0xf6, 0x6a, 0xb5, 0x25, 0x43,
			},
			authenticatedRTPPacketWithMKI: []byte{
				0x80, 0x0f, 0x12, 0x34, 0xde, 0xca, 0xfb, 0xad,
				0xca, 0xfe, 0xba, 0xbe, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0x01, 0x02, 0x00, 0x04,
				0x3d, 0x03, 0x2a, 0x52,
			},
			authenticatedRTCPPacketWithMKI: []byte{
				0x81, 0xc8, 0x00, 0x0b, 0xca, 0xfe, 0xba, 0xbe,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0x00, 0x00, 0x00, 0x01, 0x01, 0x02, 0x00, 0x04,
				0xf6, 0xd9, 0xd0, 0xc1, 0x44, 0xf6, 0x6a, 0xb5,
				0x25, 0x43,
			},
		},
		{ //nolint:dupl
			profile: ProtectionProfileAes256CmHmacSha1_80,
			encryptedRTPPacket: []byte{
				0x80, 0x0f, 0x12, 0x34, 0xde, 0xca, 0xfb, 0xad,
				0xca, 0xfe, 0xba, 0xbe, 0xac, 0x3b, 0xca, 0x88,
				0x14, 0x37, 0x57, 0x83, 0x35, 0xc6, 0xd4, 0x57,
				0xf1, 0xc3, 0x6b, 0xa7, 0x3d, 0x71, 0x48, 0x63,
				0x90, 0x9b, 0xbf, 0x15, 0xac, 0xec,
			},
			encryptedRTCPPacket: []byte{
				0x81, 0xc8, 0x00, 0x0b, 0xca, 0xfe, 0xba, 0xbe,
				0x97, 0x04, 0x31, 0xdc, 0x4a, 0xe6, 0xd2, 0xaf,
				0xd6, 0x54, 0xbf, 0x90, 0xf4, 0x35, 0x44, 0x9e,
				0x80, 0x00, 0x00, 0x01, 0xbf, 0x18, 0x18, 0x2d,
				0xd1, 0x18, 0x81, 0x28, 0x78, 0xb1,
			},
			encryptedRTPPacketWithMKI: []byte{
				0x80, 0x0f, 0x12, 0x34, 0xde, 0xca, 0xfb, 0xad,
				0xca, 0xfe, 0xba, 0xbe, 0xac, 0x3b, 0xca, 0x88,
				0x14, 0x37, 0x57, 0x83, 0x35, 0xc6, 0xd4, 0x57,
				0xf1, 0xc3, 0x6b, 0xa7, 0x01, 0x02, 0x00, 0x04,
				0x3d, 0x71, 0x48, 0x63, 0x90, 0x9b, 0xbf, 0x15,
				0xac, 0xec,
			},
			encryptedRTCPPacketWithMKI: []byte{
				0x81, 0xc8, 0x00, 0x0b, 0xca, 0xfe, 0xba, 0xbe,
				0x97, 0x04, 0x31, 0xdc, 0x4a, 0xe6, 0xd2, 0xaf,
				0xd6, 0x54, 0xbf, 0x90, 0xf4, 0x35, 0x44, 0x9e,
				0x80, 0x00, 0x00, 0x01, 0x01, 0x02, 0x00, 0x04,
				0xbf, 0x18, 0x18, 0x2d, 0xd1, 0x18, 0x81, 0x28,
				0x78, 0xb1,
			},
			authenticatedRTPPacket: []byte{
				0x80, 0x0f, 0x12, 0x34, 0xde, 0xca, 0xfb, 0xad,
				0xca, 0xfe, 0xba, 0xbe, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0x3d, 0x03, 0x2a, 0x52,
				0x72, 0x97, 0x99, 0x48, 0x5c, 0x39,
			},
			authenticatedRTCPPacket: []byte{
				0x81, 0xc8, 0x00, 0x0b, 0xca, 0xfe, 0xba, 0xbe,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0x00, 0x00, 0x00, 0x01, 0xf6, 0xd9, 0xd0, 0xc1,
				0x44, 0xf6, 0x6a, 0xb5, 0x25, 0x43,
			},
			authenticatedRTPPacketWithMKI: []byte{
				0x80, 0x0f, 0x12, 0x34, 0xde, 0xca, 0xfb, 0xad,
				0xca, 0xfe, 0xba, 0xbe, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0x01, 0x02, 0x00, 0x04,
				0x3d, 0x03, 0x2a, 0x52, 0x72, 0x97, 0x99, 0x48,
				0x5c, 0x39,
			},
			authenticatedRTCPPacketWithMKI: []byte{
				0x81, 0xc8, 0x00, 0x0b, 0xca, 0xfe, 0xba, 0xbe,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0x00, 0x00, 0x00, 0x01, 0x01, 0x02, 0x00, 0x04,
				0xf6, 0xd9, 0xd0, 0xc1, 0x44, 0xf6, 0x6a, 0xb5,
				0x25, 0x43,
			},
		},
		{ //nolint:dupl
			profile: ProtectionProfileAeadAes128Gcm,
			encryptedRTPPacket: []byte{
				0x80, 0x0f, 0x12, 0x34, 0xde, 0xca, 0xfb, 0xad,
				0xca, 0xfe, 0xba, 0xbe, 0xc5, 0x00, 0x2e, 0xde,
				0x04, 0xcf, 0xdd, 0x2e, 0xb9, 0x11, 0x59, 0xe0,
				0x88, 0x0a, 0xa0, 0x6e, 0xd2, 0x97, 0x68, 0x26,
				0xf7, 0x96, 0xb2, 0x01, 0xdf, 0x31, 0x31, 0xa1,
				0x27, 0xe8, 0xa3, 0x92,
			},
			encryptedRTCPPacket: []byte{
				0x81, 0xc8, 0x00, 0x0b, 0xca, 0xfe, 0xba, 0xbe,
				0xc9, 0x8b, 0x8b, 0x5d, 0xf0, 0x39, 0x2a, 0x55,
				0x85, 0x2b, 0x6c, 0x21, 0xac, 0x8e, 0x70, 0x25,
				0xc5, 0x2c, 0x6f, 0xbe, 0xa2, 0xb3, 0xb4, 0x46,
				0xea, 0x31, 0x12, 0x3b, 0xa8, 0x8c, 0xe6, 0x1e,
				0x80, 0x00, 0x00, 0x01,
			},
			encryptedRTPPacketWithMKI: []byte{
				0x80, 0x0f, 0x12, 0x34, 0xde, 0xca, 0xfb, 0xad,
				0xca, 0xfe, 0xba, 0xbe, 0xc5, 0x00, 0x2e, 0xde,
				0x04, 0xcf, 0xdd, 0x2e, 0xb9, 0x11, 0x59, 0xe0,
				0x88, 0x0a, 0xa0, 0x6e, 0xd2, 0x97, 0x68, 0x26,
				0xf7, 0x96, 0xb2, 0x01, 0xdf, 0x31, 0x31, 0xa1,
				0x27, 0xe8, 0xa3, 0x92, 0x01, 0x02, 0x00, 0x04,
			},
			encryptedRTCPPacketWithMKI: []byte{
				0x81, 0xc8, 0x00, 0x0b, 0xca, 0xfe, 0xba, 0xbe,
				0xc9, 0x8b, 0x8b, 0x5d, 0xf0, 0x39, 0x2a, 0x55,
				0x85, 0x2b, 0x6c, 0x21, 0xac, 0x8e, 0x70, 0x25,
				0xc5, 0x2c, 0x6f, 0xbe, 0xa2, 0xb3, 0xb4, 0x46,
				0xea, 0x31, 0x12, 0x3b, 0xa8, 0x8c, 0xe6, 0x1e,
				0x80, 0x00, 0x00, 0x01, 0x01, 0x02, 0x00, 0x04,
			},
			authenticatedRTPPacket: []byte{
				0x80, 0x0f, 0x12, 0x34, 0xde, 0xca, 0xfb, 0xad,
				0xca, 0xfe, 0xba, 0xbe, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0x8b, 0xdd, 0xb6, 0x20,
				0xb1, 0x0d, 0x2f, 0xe2, 0x76, 0xf7, 0xbd, 0xcf,
				0xc5, 0xc3, 0x8a, 0xe5,
			},
			authenticatedRTCPPacket: []byte{
				0x81, 0xc8, 0x00, 0x0b, 0xca, 0xfe, 0xba, 0xbe,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0x0c, 0xf6, 0x35, 0x16, 0x8f, 0x82, 0x42, 0xa2,
				0x1b, 0x12, 0xd6, 0x64, 0xec, 0xd8, 0x62, 0xe8,
				0x00, 0x00, 0x00, 0x01,
			},
			authenticatedRTPPacketWithMKI: []byte{
				0x80, 0x0f, 0x12, 0x34, 0xde, 0xca, 0xfb, 0xad,
				0xca, 0xfe, 0xba, 0xbe, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0x8b, 0xdd, 0xb6, 0x20,
				0xb1, 0x0d, 0x2f, 0xe2, 0x76, 0xf7, 0xbd, 0xcf,
				0xc5, 0xc3, 0x8a, 0xe5, 0x01, 0x02, 0x00, 0x04,
			},
			authenticatedRTCPPacketWithMKI: []byte{
				0x81, 0xc8, 0x00, 0x0b, 0xca, 0xfe, 0xba, 0xbe,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0x0c, 0xf6, 0x35, 0x16, 0x8f, 0x82, 0x42, 0xa2,
				0x1b, 0x12, 0xd6, 0x64, 0xec, 0xd8, 0x62, 0xe8,
				0x00, 0x00, 0x00, 0x01, 0x01, 0x02, 0x00, 0x04,
			},
		},
		{ //nolint:dupl
			profile: ProtectionProfileAeadAes256Gcm,
			encryptedRTPPacket: []byte{
				0x80, 0x0f, 0x12, 0x34, 0xde, 0xca, 0xfb, 0xad,
				0xca, 0xfe, 0xba, 0xbe, 0xaf, 0x49, 0x96, 0x8f,
				0x7e, 0x9c, 0x43, 0xf8, 0x01, 0xdd, 0x0c, 0x84,
				0x8b, 0x1e, 0xc9, 0xb0, 0x29, 0xcd, 0xf8, 0x5c,
				0xb7, 0x9a, 0x2f, 0x95, 0x60, 0xd4, 0x69, 0x75,
				0x98, 0x50, 0x77, 0x25,
			},
			encryptedRTCPPacket: []byte{
				0x81, 0xc8, 0x00, 0x0b, 0xca, 0xfe, 0xba, 0xbe,
				0x98, 0x22, 0xba, 0x22, 0x96, 0x1c, 0x31, 0x48,
				0xe7, 0xb7, 0xec, 0x4f, 0x09, 0xf4, 0x26, 0xdc,
				0xf6, 0xb5, 0x9a, 0x75, 0xad, 0xec, 0x74, 0xfd,
				0xb9, 0x51, 0xb6, 0x66, 0x84, 0x24, 0xd4, 0xe2,
				0x80, 0x00, 0x00, 0x01,
			},
			encryptedRTPPacketWithMKI: []byte{
				0x80, 0x0f, 0x12, 0x34, 0xde, 0xca, 0xfb, 0xad,
				0xca, 0xfe, 0xba, 0xbe, 0xaf, 0x49, 0x96, 0x8f,
				0x7e, 0x9c, 0x43, 0xf8, 0x01, 0xdd, 0x0c, 0x84,
				0x8b, 0x1e, 0xc9, 0xb0, 0x29, 0xcd, 0xf8, 0x5c,
				0xb7, 0x9a, 0x2f, 0x95, 0x60, 0xd4, 0x69, 0x75,
				0x98, 0x50, 0x77, 0x25, 0x01, 0x02, 0x00, 0x04,
			},
			encryptedRTCPPacketWithMKI: []byte{
				0x81, 0xc8, 0x00, 0x0b, 0xca, 0xfe, 0xba, 0xbe,
				0x98, 0x22, 0xba, 0x22, 0x96, 0x1c, 0x31, 0x48,
				0xe7, 0xb7, 0xec, 0x4f, 0x09, 0xf4, 0x26, 0xdc,
				0xf6, 0xb5, 0x9a, 0x75, 0xad, 0xec, 0x74, 0xfd,
				0xb9, 0x51, 0xb6, 0x66, 0x84, 0x24, 0xd4, 0xe2,
				0x80, 0x00, 0x00, 0x01, 0x01, 0x02, 0x00, 0x04,
			},
			authenticatedRTPPacket: []byte{
				0x80, 0x0f, 0x12, 0x34, 0xde, 0xca, 0xfb, 0xad,
				0xca, 0xfe, 0xba, 0xbe, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0x9c, 0x27, 0x45, 0xcc,
				0xde, 0x31, 0xda, 0x1f, 0x03, 0xa5, 0x4c, 0xfd,
				0xfa, 0xa2, 0x62, 0x8d,
			},
			authenticatedRTCPPacket: []byte{
				0x81, 0xc8, 0x00, 0x0b, 0xca, 0xfe, 0xba, 0xbe,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0x22, 0x55, 0xdc, 0xaf, 0x86, 0x9a, 0xbb, 0x1c,
				0xd0, 0x1a, 0xe8, 0x35, 0x4c, 0x94, 0x11, 0xee,
				0x00, 0x00, 0x00, 0x01,
			},
			authenticatedRTPPacketWithMKI: []byte{
				0x80, 0x0f, 0x12, 0x34, 0xde, 0xca, 0xfb, 0xad,
				0xca, 0xfe, 0xba, 0xbe, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0x9c, 0x27, 0x45, 0xcc,
				0xde, 0x31, 0xda, 0x1f, 0x03, 0xa5, 0x4c, 0xfd,
				0xfa, 0xa2, 0x62, 0x8d, 0x01, 0x02, 0x00, 0x04,
			},
			authenticatedRTCPPacketWithMKI: []byte{
				0x81, 0xc8, 0x00, 0x0b, 0xca, 0xfe, 0xba, 0xbe,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0x22, 0x55, 0xdc, 0xaf, 0x86, 0x9a, 0xbb, 0x1c,
				0xd0, 0x1a, 0xe8, 0x35, 0x4c, 0x94, 0x11, 0xee,
				0x00, 0x00, 0x00, 0x01, 0x01, 0x02, 0x00, 0x04,
			},
		},
		{ //nolint:dupl
			profile: ProtectionProfileNullHmacSha1_32,
			encryptedRTPPacket: []byte{
				0x80, 0x0f, 0x12, 0x34, 0xde, 0xca, 0xfb, 0xad,
				0xca, 0xfe, 0xba, 0xbe, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0xda, 0x9a, 0x3c, 0xa1,
			},
			encryptedRTCPPacket: []byte{
				0x81, 0xc8, 0x00, 0x0b, 0xca, 0xfe, 0xba, 0xbe,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0x00, 0x00, 0x00, 0x01, 0xd2, 0xa2, 0x36, 0x2d,
				0x01, 0x1b, 0x8c, 0xfc, 0x0a, 0xc9,
			},
			encryptedRTPPacketWithMKI: []byte{
				0x80, 0x0f, 0x12, 0x34, 0xde, 0xca, 0xfb, 0xad,
				0xca, 0xfe, 0xba, 0xbe, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0x01, 0x02, 0x00, 0x04,
				0xda, 0x9a, 0x3c, 0xa1,
			},
			encryptedRTCPPacketWithMKI: []byte{
				0x81, 0xc8, 0x00, 0x0b, 0xca, 0xfe, 0xba, 0xbe,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0x00, 0x00, 0x00, 0x01, 0x01, 0x02, 0x00, 0x04,
				0xd2, 0xa2, 0x36, 0x2d, 0x01, 0x1b, 0x8c, 0xfc,
				0x0a, 0xc9,
			},
			authenticatedRTPPacket: []byte{
				0x80, 0x0f, 0x12, 0x34, 0xde, 0xca, 0xfb, 0xad,
				0xca, 0xfe, 0xba, 0xbe, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0xda, 0x9a, 0x3c, 0xa1,
			},
			authenticatedRTCPPacket: []byte{
				0x81, 0xc8, 0x00, 0x0b, 0xca, 0xfe, 0xba, 0xbe,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0x00, 0x00, 0x00, 0x01, 0xd2, 0xa2, 0x36, 0x2d,
				0x01, 0x1b, 0x8c, 0xfc, 0x0a, 0xc9,
			},
			authenticatedRTPPacketWithMKI: []byte{
				0x80, 0x0f, 0x12, 0x34, 0xde, 0xca, 0xfb, 0xad,
				0xca, 0xfe, 0xba, 0xbe, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0x01, 0x02, 0x00, 0x04,
				0xda, 0x9a, 0x3c, 0xa1,
			},
			authenticatedRTCPPacketWithMKI: []byte{
				0x81, 0xc8, 0x00, 0x0b, 0xca, 0xfe, 0xba, 0xbe,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0x00, 0x00, 0x00, 0x01, 0x01, 0x02, 0x00, 0x04,
				0xd2, 0xa2, 0x36, 0x2d, 0x01, 0x1b, 0x8c, 0xfc,
				0x0a, 0xc9,
			},
		},
		{ //nolint:dupl
			profile: ProtectionProfileNullHmacSha1_80,
			encryptedRTPPacket: []byte{
				0x80, 0x0f, 0x12, 0x34, 0xde, 0xca, 0xfb, 0xad,
				0xca, 0xfe, 0xba, 0xbe, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0xda, 0x9a, 0x3c, 0xa1,
				0xba, 0x8e, 0xfd, 0xd7, 0x07, 0xdc,
			},
			encryptedRTCPPacket: []byte{
				0x81, 0xc8, 0x00, 0x0b, 0xca, 0xfe, 0xba, 0xbe,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0x00, 0x00, 0x00, 0x01, 0xd2, 0xa2, 0x36, 0x2d,
				0x01, 0x1b, 0x8c, 0xfc, 0x0a, 0xc9,
			},
			encryptedRTPPacketWithMKI: []byte{
				0x80, 0x0f, 0x12, 0x34, 0xde, 0xca, 0xfb, 0xad,
				0xca, 0xfe, 0xba, 0xbe, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0x01, 0x02, 0x00, 0x04,
				0xda, 0x9a, 0x3c, 0xa1, 0xba, 0x8e, 0xfd, 0xd7,
				0x07, 0xdc,
			},
			encryptedRTCPPacketWithMKI: []byte{
				0x81, 0xc8, 0x00, 0x0b, 0xca, 0xfe, 0xba, 0xbe,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0x00, 0x00, 0x00, 0x01, 0x01, 0x02, 0x00, 0x04,
				0xd2, 0xa2, 0x36, 0x2d, 0x01, 0x1b, 0x8c, 0xfc,
				0x0a, 0xc9,
			},
			authenticatedRTPPacket: []byte{
				0x80, 0x0f, 0x12, 0x34, 0xde, 0xca, 0xfb, 0xad,
				0xca, 0xfe, 0xba, 0xbe, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0xda, 0x9a, 0x3c, 0xa1,
				0xba, 0x8e, 0xfd, 0xd7, 0x07, 0xdc,
			},
			authenticatedRTCPPacket: []byte{
				0x81, 0xc8, 0x00, 0x0b, 0xca, 0xfe, 0xba, 0xbe,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0x00, 0x00, 0x00, 0x01, 0xd2, 0xa2, 0x36, 0x2d,
				0x01, 0x1b, 0x8c, 0xfc, 0x0a, 0xc9,
			},
			authenticatedRTPPacketWithMKI: []byte{
				0x80, 0x0f, 0x12, 0x34, 0xde, 0xca, 0xfb, 0xad,
				0xca, 0xfe, 0xba, 0xbe, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0x01, 0x02, 0x00, 0x04,
				0xda, 0x9a, 0x3c, 0xa1, 0xba, 0x8e, 0xfd, 0xd7,
				0x07, 0xdc,
			},
			authenticatedRTCPPacketWithMKI: []byte{
				0x81, 0xc8, 0x00, 0x0b, 0xca, 0xfe, 0xba, 0xbe,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
				0x00, 0x00, 0x00, 0x01, 0x01, 0x02, 0x00, 0x04,
				0xd2, 0xa2, 0x36, 0x2d, 0x01, 0x1b, 0x8c, 0xfc,
				0x0a, 0xc9,
			},
		},
	}

	masterKey := []byte{
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
		0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
		0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
	}
	masterSalt := []byte{
		0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7,
		0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad,
	}
	mki := []byte{0x01, 0x02, 0x00, 0x04}
	decryptedRTPPacket := []byte{
		0x80, 0x0f, 0x12, 0x34, 0xde, 0xca, 0xfb, 0xad,
		0xca, 0xfe, 0xba, 0xbe, 0xab, 0xab, 0xab, 0xab,
		0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
		0xab, 0xab, 0xab, 0xab,
	}
	decryptedRTCPPacket := []byte{
		0x81, 0xc8, 0x00, 0x0b, 0xca, 0xfe, 0xba, 0xbe,
		0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
		0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab, 0xab,
	}

	for key, v := range tests {
		keyLen, err := v.profile.KeyLen()
		assert.NoError(t, err)

		saltLen, err := v.profile.SaltLen()
		assert.NoError(t, err)

		tests[key].masterKey = masterKey[:keyLen]
		tests[key].masterSalt = masterSalt[:saltLen]
		tests[key].mki = mki
		tests[key].decryptedRTPPacket = decryptedRTPPacket
		tests[key].decryptedRTCPPacket = decryptedRTCPPacket
	}

	return tests
}

// nolint:maintidx,dupl
func TestSrtpCipher(t *testing.T) {
	const testSSRC = 0xcafebabe

	for _, testCase := range createTestCiphers(t) {
		t.Run(testCase.profile.String(), func(t *testing.T) {
			assert.Equal(t, testCase.decryptedRTPPacket, testCase.authenticatedRTPPacket[:len(testCase.decryptedRTPPacket)])
			assert.Equal(t, testCase.decryptedRTCPPacket, testCase.authenticatedRTCPPacket[:len(testCase.decryptedRTCPPacket)])

			t.Run("Encrypt RTP", func(t *testing.T) {
				ctx, err := CreateContext(testCase.masterKey, testCase.masterSalt, testCase.profile)
				assert.NoError(t, err)

				t.Run("New Allocation", func(t *testing.T) {
					actualEncrypted, err := ctx.EncryptRTP(nil, testCase.decryptedRTPPacket, nil)
					assert.NoError(t, err)
					assert.Equal(t, testCase.encryptedRTPPacket, actualEncrypted)
				})

				t.Run("Same buffer", func(t *testing.T) {
					buffer := make([]byte, 0, 1000)
					src, dst := buffer, buffer
					src = append(src, testCase.decryptedRTPPacket...)
					assert.True(t, isSameBuffer(dst, src))

					actualEncrypted, err := ctx.EncryptRTP(dst, src, nil)
					assert.NoError(t, err)
					assert.Equal(t, testCase.encryptedRTPPacket, actualEncrypted)
					assert.True(t, isSameBuffer(actualEncrypted, src))
				})
			})

			t.Run("Decrypt RTP", func(t *testing.T) {
				ctx, err := CreateContext(testCase.masterKey, testCase.masterSalt, testCase.profile)
				assert.NoError(t, err)

				t.Run("New Allocation", func(t *testing.T) {
					actualDecrypted, err := ctx.DecryptRTP(nil, testCase.encryptedRTPPacket, nil)
					assert.NoError(t, err)
					assert.Equal(t, testCase.decryptedRTPPacket, actualDecrypted)
				})

				t.Run("Same buffer", func(t *testing.T) {
					buffer := make([]byte, 0, 1000)
					src, dst := buffer, buffer
					src = append(src, testCase.encryptedRTPPacket...)
					assert.True(t, isSameBuffer(dst, src))

					actualDecrypted, err := ctx.DecryptRTP(dst, src, nil)
					assert.NoError(t, err)
					assert.Equal(t, testCase.decryptedRTPPacket, actualDecrypted)
					assert.True(t, isSameBuffer(actualDecrypted, src))
				})
			})

			t.Run("Encrypt RTCP", func(t *testing.T) {
				ctx, err := CreateContext(testCase.masterKey, testCase.masterSalt, testCase.profile)
				assert.NoError(t, err)

				t.Run("New Allocation", func(t *testing.T) {
					ctx.SetIndex(testSSRC, 0)
					actualEncrypted, err := ctx.EncryptRTCP(nil, testCase.decryptedRTCPPacket, nil)
					assert.NoError(t, err)
					assert.Equal(t, testCase.encryptedRTCPPacket, actualEncrypted)
				})

				t.Run("Same buffer", func(t *testing.T) {
					ctx.SetIndex(testSSRC, 0)

					buffer := make([]byte, 0, 1000)
					src, dst := buffer, buffer
					src = append(src, testCase.decryptedRTCPPacket...)
					assert.True(t, isSameBuffer(dst, src))

					actualEncrypted, err := ctx.EncryptRTCP(dst, src, nil)
					assert.NoError(t, err)
					assert.Equal(t, testCase.encryptedRTCPPacket, actualEncrypted)
					assert.True(t, isSameBuffer(actualEncrypted, src))
				})
			})

			t.Run("Decrypt RTCP", func(t *testing.T) {
				ctx, err := CreateContext(testCase.masterKey, testCase.masterSalt, testCase.profile)
				assert.NoError(t, err)

				t.Run("New Allocation", func(t *testing.T) {
					ctx.SetIndex(testSSRC, 0)
					actualDecrypted, err := ctx.DecryptRTCP(nil, testCase.encryptedRTCPPacket, nil)
					assert.NoError(t, err)
					assert.Equal(t, testCase.decryptedRTCPPacket, actualDecrypted)
				})

				t.Run("Same buffer", func(t *testing.T) {
					ctx.SetIndex(testSSRC, 0)

					buffer := make([]byte, 0, 1000)
					src, dst := buffer, buffer
					src = append(src, testCase.encryptedRTCPPacket...)
					assert.True(t, isSameBuffer(dst, src))

					actualDecrypted, err := ctx.DecryptRTCP(dst, src, nil)
					assert.NoError(t, err)
					assert.Equal(t, testCase.decryptedRTCPPacket, actualDecrypted)
					assert.True(t, isSameBuffer(actualDecrypted, src))
				})
			})

			t.Run("Encrypt RTP with MKI", func(t *testing.T) {
				ctx, err := CreateContext(
					testCase.masterKey, testCase.masterSalt, testCase.profile, MasterKeyIndicator(testCase.mki),
				)
				assert.NoError(t, err)

				t.Run("New Allocation", func(t *testing.T) {
					actualEncrypted, err := ctx.EncryptRTP(nil, testCase.decryptedRTPPacket, nil)
					assert.NoError(t, err)
					assert.Equal(t, testCase.encryptedRTPPacketWithMKI, actualEncrypted)
				})
			})

			t.Run("Decrypt RTP with MKI", func(t *testing.T) {
				ctx, err := CreateContext(
					testCase.masterKey, testCase.masterSalt, testCase.profile, MasterKeyIndicator(testCase.mki),
				)
				assert.NoError(t, err)

				t.Run("New Allocation", func(t *testing.T) {
					actualDecrypted, err := ctx.DecryptRTP(nil, testCase.encryptedRTPPacketWithMKI, nil)
					assert.NoError(t, err)
					assert.Equal(t, testCase.decryptedRTPPacket, actualDecrypted)
				})
			})

			t.Run("Encrypt RTCP with MKI", func(t *testing.T) {
				ctx, err := CreateContext(
					testCase.masterKey, testCase.masterSalt, testCase.profile, MasterKeyIndicator(testCase.mki),
				)
				assert.NoError(t, err)

				t.Run("New Allocation", func(t *testing.T) {
					actualEncrypted, err := ctx.EncryptRTCP(nil, testCase.decryptedRTCPPacket, nil)
					assert.NoError(t, err)
					assert.Equal(t, testCase.encryptedRTCPPacketWithMKI, actualEncrypted)
				})
			})

			t.Run("Decrypt RTCP with MKI", func(t *testing.T) {
				ctx, err := CreateContext(
					testCase.masterKey, testCase.masterSalt, testCase.profile, MasterKeyIndicator(testCase.mki),
				)
				assert.NoError(t, err)

				t.Run("New Allocation", func(t *testing.T) {
					actualDecrypted, err := ctx.DecryptRTCP(nil, testCase.encryptedRTCPPacketWithMKI, nil)
					assert.NoError(t, err)
					assert.Equal(t, testCase.decryptedRTCPPacket, actualDecrypted)
				})
			})

			t.Run("Encrypt RTP with NULL cipher", func(t *testing.T) {
				ctx, err := CreateContext(
					testCase.masterKey, testCase.masterSalt, testCase.profile, SRTPNoEncryption(), SRTCPNoEncryption(),
				)
				assert.NoError(t, err)

				t.Run("New Allocation", func(t *testing.T) {
					actualEncrypted, err := ctx.EncryptRTP(nil, testCase.decryptedRTPPacket, nil)
					assert.NoError(t, err)
					assert.Equal(t, testCase.decryptedRTPPacket, actualEncrypted[:len(testCase.decryptedRTPPacket)])
					assert.Equal(t, testCase.authenticatedRTPPacket, actualEncrypted)
				})
			})

			t.Run("Decrypt RTP with NULL cipher", func(t *testing.T) {
				ctx, err := CreateContext(
					testCase.masterKey, testCase.masterSalt, testCase.profile, SRTPNoEncryption(), SRTCPNoEncryption(),
				)
				assert.NoError(t, err)

				t.Run("New Allocation", func(t *testing.T) {
					actualDecrypted, err := ctx.DecryptRTP(nil, testCase.authenticatedRTPPacket, nil)
					assert.NoError(t, err)
					assert.Equal(t, testCase.decryptedRTPPacket, actualDecrypted)
				})
			})

			t.Run("Encrypt RTCP with NULL cipher", func(t *testing.T) {
				ctx, err := CreateContext(
					testCase.masterKey, testCase.masterSalt, testCase.profile, SRTPNoEncryption(), SRTCPNoEncryption(),
				)
				assert.NoError(t, err)

				t.Run("New Allocation", func(t *testing.T) {
					actualEncrypted, err := ctx.EncryptRTCP(nil, testCase.decryptedRTCPPacket, nil)
					assert.NoError(t, err)
					assert.Equal(t, testCase.decryptedRTCPPacket, actualEncrypted[:len(testCase.decryptedRTCPPacket)])
					assert.Equal(t, testCase.authenticatedRTCPPacket, actualEncrypted)
				})
			})

			t.Run("Decrypt RTCP with NULL cipher", func(t *testing.T) {
				ctx, err := CreateContext(
					testCase.masterKey, testCase.masterSalt, testCase.profile, SRTPNoEncryption(), SRTCPNoEncryption(),
				)
				assert.NoError(t, err)

				t.Run("New Allocation", func(t *testing.T) {
					actualDecrypted, err := ctx.DecryptRTCP(nil, testCase.authenticatedRTCPPacket, nil)
					assert.NoError(t, err)
					assert.Equal(t, testCase.decryptedRTCPPacket, actualDecrypted)
				})
			})

			t.Run("Encrypt RTP with NULL cipher and MKI", func(t *testing.T) {
				ctx, err := CreateContext(
					testCase.masterKey,
					testCase.masterSalt,
					testCase.profile,
					SRTPNoEncryption(),
					SRTCPNoEncryption(),
					MasterKeyIndicator(testCase.mki),
				)
				assert.NoError(t, err)

				t.Run("New Allocation", func(t *testing.T) {
					actualEncrypted, err := ctx.EncryptRTP(nil, testCase.decryptedRTPPacket, nil)
					assert.NoError(t, err)
					assert.Equal(t, testCase.decryptedRTPPacket, actualEncrypted[:len(testCase.decryptedRTPPacket)])
					assert.Equal(t, testCase.authenticatedRTPPacketWithMKI, actualEncrypted)
				})
			})

			t.Run("Decrypt RTP with NULL cipher and MKI", func(t *testing.T) {
				ctx, err := CreateContext(
					testCase.masterKey,
					testCase.masterSalt,
					testCase.profile,
					SRTPNoEncryption(),
					SRTCPNoEncryption(),
					MasterKeyIndicator(testCase.mki),
				)
				assert.NoError(t, err)

				t.Run("New Allocation", func(t *testing.T) {
					actualDecrypted, err := ctx.DecryptRTP(nil, testCase.authenticatedRTPPacketWithMKI, nil)
					assert.NoError(t, err)
					assert.Equal(t, testCase.decryptedRTPPacket, actualDecrypted)
				})
			})

			t.Run("Encrypt RTCP with NULL cipher and MKI", func(t *testing.T) {
				ctx, err := CreateContext(
					testCase.masterKey,
					testCase.masterSalt,
					testCase.profile,
					SRTPNoEncryption(),
					SRTCPNoEncryption(),
					MasterKeyIndicator(testCase.mki),
				)
				assert.NoError(t, err)

				t.Run("New Allocation", func(t *testing.T) {
					actualEncrypted, err := ctx.EncryptRTCP(nil, testCase.decryptedRTCPPacket, nil)
					assert.NoError(t, err)
					assert.Equal(t, testCase.decryptedRTCPPacket, actualEncrypted[:len(testCase.decryptedRTCPPacket)])
					assert.Equal(t, testCase.authenticatedRTCPPacketWithMKI, actualEncrypted)
				})
			})

			t.Run("Decrypt RTCP with NULL cipher and MKI", func(t *testing.T) {
				ctx, err := CreateContext(
					testCase.masterKey,
					testCase.masterSalt,
					testCase.profile,
					SRTPNoEncryption(),
					SRTCPNoEncryption(),
					MasterKeyIndicator(testCase.mki),
				)
				assert.NoError(t, err)

				t.Run("New Allocation", func(t *testing.T) {
					actualDecrypted, err := ctx.DecryptRTCP(nil, testCase.authenticatedRTCPPacketWithMKI, nil)
					assert.NoError(t, err)
					assert.Equal(t, testCase.decryptedRTCPPacket, actualDecrypted)
				})
			})

			srtpAuthTagLen, err := testCase.profile.AuthTagRTPLen()
			assert.NoError(t, err)
			if srtpAuthTagLen != 0 {
				t.Run("Encrypt RTP with changed RTP auth tag len", func(t *testing.T) {
					ctx, err := CreateContext(testCase.masterKey, testCase.masterSalt, testCase.profile,
						SRTPAuthenticationTagLength(srtpAuthTagLen-2))
					assert.NoError(t, err)

					t.Run("New Allocation", func(t *testing.T) {
						actualEncrypted, err := ctx.EncryptRTP(nil, testCase.decryptedRTPPacket, nil)
						assert.NoError(t, err)
						assert.Equal(t, testCase.encryptedRTPPacket[:len(testCase.encryptedRTPPacket)-2], actualEncrypted)
					})
				})

				t.Run("Decrypt RTP with changed RTP auth tag len", func(t *testing.T) {
					ctx, err := CreateContext(testCase.masterKey, testCase.masterSalt, testCase.profile,
						SRTPAuthenticationTagLength(srtpAuthTagLen-2))
					assert.NoError(t, err)

					t.Run("New Allocation", func(t *testing.T) {
						actualDecrypted, err := ctx.DecryptRTP(nil, testCase.encryptedRTPPacket[:len(testCase.encryptedRTPPacket)-2], nil)
						assert.NoError(t, err)
						assert.Equal(t, testCase.decryptedRTPPacket, actualDecrypted)
					})
				})
			}
		})
	}
}
