// +build ignore

package main

/*
 * This program tests the 'memimagedraw' primitive stochastically.
 * It tests the combination aspects of it thoroughly, but since the
 * three images it uses are disjoint, it makes no check of the
 * correct behavior when images overlap.  That is, however, much
 * easier to get right and to test.
 */

import (
	"bufio"
	"bytes"
	"flag"
	"fmt"
	"io"
	"log"
	"math/rand"
	"os"
	"path/filepath"
	"time"

	"9fans.net/go/draw"
	"9fans.net/go/draw/memdraw"
)

const DBG = 0

func RGB2K(r, g, b uint8) uint8 {
	// fmt.Printf("RGB2K %#x %#x %#x -> %#x\n%s", r, g, b, uint8((299*int(r)+587*int(g)+114*int(b))/1000), string(debug.Stack()))
	return uint8((299*int(r) + 587*int(g) + 114*int(b)) / 1000)
}

var dchan string
var schan string
var mchan string

var drawdebug int = 0
var seed int64
var niters int = 100
var dbpp int   /* bits per pixel in destination */
var sbpp int   /* bits per pixel in src */
var mbpp int   /* bits per pixel in mask */
var dpm uint8  /* pixel mask at high part of byte, in destination */
var nbytes int /* in destination */

var Xrange int = 64
var Yrange int = 16

var dst *memdraw.Image
var src *memdraw.Image
var mask *memdraw.Image
var stmp *memdraw.Image
var mtmp *memdraw.Image
var ones *memdraw.Image
var dstbits []uint8
var srcbits []uint8
var maskbits []uint8
var savedstbits []uint8

func rdb() {
}

func iprint(format string, args ...interface{}) {
	fmt.Fprintf(os.Stderr, format, args...)
}

func main() {
	memdraw.Init()
	seed = time.Now().UnixNano()

	flag.IntVar(&Xrange, "x", Xrange, "")
	flag.IntVar(&Yrange, "y", Yrange, "")
	flag.IntVar(&niters, "n", niters, "")
	flag.Int64Var(&seed, "s", seed, "")
	flag.Usage = func() {
		fmt.Fprintf(os.Stderr, "usage: dtest [dchan [schan [mchan]]]\n")
		os.Exit(2)
	}
	flag.Parse()

	dchan = "r8g8b8"
	schan = "r8g8b8"
	mchan = "r8g8b8"
	switch flag.NArg() {
	default:
		flag.Usage()
	case 3:
		mchan = flag.Arg(2)
		fallthrough
	case 2:
		schan = flag.Arg(1)
		fallthrough
	case 1:
		dchan = flag.Arg(0)
		fallthrough
	case 0:
		// ok
	}

	fmt.Fprintf(os.Stderr, "%s -x %d -y %d -s 0x%x %s %s %s\n", filepath.Base(os.Args[0]), Xrange, Yrange, seed, dchan, schan, mchan)
	rand.Seed(seed)

	alloc := func(s string) *memdraw.Image {
		pix, err := draw.ParsePix(s)
		if err != nil {
			log.Fatal(err)
		}
		i, err := memdraw.AllocImage(draw.Rect(0, 0, Xrange, Yrange), pix)
		if err != nil {
			log.Fatal(err)
		}
		return i
	}

	dst = alloc(dchan)
	src = alloc(schan)
	mask = alloc(mchan)
	stmp = alloc(schan)
	mtmp = alloc(mchan)
	ones = alloc(mchan)

	nbytes = (4*Xrange+4)*Yrange + 4
	srcbits = make([]byte, nbytes)
	dstbits = make([]byte, nbytes)
	maskbits = make([]byte, nbytes)
	savedstbits = make([]byte, nbytes)

	dbpp = dst.Depth
	sbpp = src.Depth
	mbpp = mask.Depth
	dpm = uint8(0xFF) ^ (0xFF >> dbpp)
	b := ones.Data.Bdata[:int(ones.Width)*4*Yrange]
	for i := range b {
		b[i] = 0xFF
	}

	fmt.Fprintf(os.Stderr, "dtest: verify single pixel operation\n")
	verifyone()

	fmt.Fprintf(os.Stderr, "dtest: verify full line non-replicated\n")
	verifyline()

	fmt.Fprintf(os.Stderr, "dtest: verify full rectangle non-replicated\n")
	verifyrect()

	fmt.Fprintf(os.Stderr, "dtest: verify full rectangle source replicated\n")
	verifyrectrepl(1, 0)

	fmt.Fprintf(os.Stderr, "dtest: verify full rectangle mask replicated\n")
	verifyrectrepl(0, 1)

	fmt.Fprintf(os.Stderr, "dtest: verify full rectangle source and mask replicated\n")
	verifyrectrepl(1, 1)
}

/*
 * Dump out an ASCII representation of an image.  The label specifies
 * a list of characters to put at various points in the picture.
 */
func Bprintr5g6b5(bio io.Writer, _ string, v uint32) {
	r := (v >> 11) & 31
	g := (v >> 5) & 63
	b := v & 31
	fmt.Fprintf(bio, "%.2x%.2x%.2x", r, g, b)
}

func Bprintr5g5b5a1(bio io.Writer, _ string, v uint32) {
	r := (v >> 11) & 31
	g := (v >> 6) & 31
	b := (v >> 1) & 31
	a := v & 1
	fmt.Fprintf(bio, "%.2x%.2x%.2x%.2x", r, g, b, a)
}

func Bprint(bio io.Writer, format string, arg uint32) {
	fmt.Fprintf(bio, format, arg)
}

func dumpimage(name string, img *memdraw.Image, vdata []byte, labelpt draw.Point) {
	var fmt_ func(io.Writer, string, uint32)
	var format string
	switch img.Depth {
	case 1, 2, 4:
		fmt_ = Bprint
		format = "%.1x"
	case 8:
		fmt_ = Bprint
		format = "%.2x"
	case 16:
		if img.Pix == draw.RGB16 {
			fmt_ = Bprintr5g6b5
		} else {
			fmt_ = Bprint
			format = "%.4x"
		}
	case 24:
		fmt_ = Bprint
		format = "%.6x"
	case 32:
		fmt_ = Bprint
		format = "%.8x"
	}
	if fmt_ == nil {
		panic("bad format")
	}

	r := img.R
	b := bufio.NewWriter(os.Stderr)
	data := vdata
	bpp := img.Depth
	repl := 0
	if img.Flags&memdraw.Frepl != 0 {
		repl = 1
	}
	fmt.Fprintf(b, "%s\t%d\tr %v clipr %v repl %d data %p *%v\n", name, r.Min.X, r, img.Clipr, repl, vdata, labelpt)
	mask := (uint32(1) << bpp) - 1
	/*	for(y=r.Min.Y; y<r.Max.Y; y++){ */
	for y := 0; y < Yrange; y++ {
		nb := 0
		v := uint32(0)
		p := data[(-len(img.BytesAt(draw.Pt(0, y))) + len(img.Data.Bdata)):]
		fmt.Fprintf(b, "%-4d\t", y)
		/*		for(x=r.Min.X; x<r.Max.X; x++){ */
		for x := 0; x < Xrange; x++ {
			if x == 0 {
				fmt.Fprintf(b, "\t")
			}

			if x != 0 && (x%8) == 0 {
				fmt.Fprintf(b, " ")
			}

			npr := 0
			if x == labelpt.X && y == labelpt.Y {
				fmt.Fprintf(b, "*")
				npr++
			}
			if npr == 0 {
				fmt.Fprintf(b, " ")
			}

			for nb < bpp {
				v &= (1 << nb) - 1
				v |= uint32(p[0]) << nb
				p = p[1:]
				nb += 8
			}
			nb -= bpp
			/*			fmt.Fprintf(os.Stderr, "bpp %d v %.8x mask %.8x nb %d\n", bpp, v, mask, nb); */
			fmt_(b, format, (v>>nb)&mask)
		}
		fmt.Fprintf(b, "\n")
	}
	b.Flush()
}

/*
 * Verify that the destination pixel has the specified value.
 * The value is in the high bits of v, suitably masked, but must
 * be extracted from the destination memdraw.Image.
 */
func checkone(p draw.Point, sp draw.Point, mp draw.Point) {
	delta := -len(dst.BytesAt(p)) - -len(dst.Data.Bdata)
	dp := dst.Data.Bdata[delta:]
	sdp := savedstbits[delta:]
	w := (dst.Depth + 7) / 8

	if !bytes.Equal(dp[:w], sdp[:w]) {
		fmt.Fprintf(os.Stderr, "dtest: one bad pixel drawing at dst %v from source %v mask %v\n", p, sp, mp)
		fmt.Fprintf(os.Stderr, " %x should be %x\n", dp[:w], sdp[:w])
		fmt.Fprintf(os.Stderr, "addresses dst %p src %p mask %p\n", dp, src.BytesAt(sp), mask.BytesAt(mp))
		dumpimage("src", src, src.Data.Bdata, sp)
		dumpimage("mask", mask, mask.Data.Bdata, mp)
		dumpimage("origdst", dst, dstbits, p)
		dumpimage("dst", dst, dst.Data.Bdata, p)
		dumpimage("gooddst", dst, savedstbits, p)
		panic("fail")
	}
}

/*
 * Verify that the destination line has the same value as the saved line.
 */
// #define RECTPTS(r) (r).Min.X, (r).Min.Y, (r).Max.X, (r).Max.Y
func checkline(r draw.Rectangle, sp draw.Point, mp draw.Point, y int, stmp *memdraw.Image, mtmp *memdraw.Image) {
	dp := dst.BytesAt(draw.Pt(0, y))
	saved := savedstbits[y*int(dst.Width)*4:]
	var nb int
	if dst.Depth < 8 {
		nb = Xrange / (8 / dst.Depth)
	} else {
		nb = Xrange * (dst.Depth / 8)
	}
	if !bytes.Equal(dp[:nb], saved[:nb]) {
		fmt.Fprintf(os.Stderr, "dtest: bad line at y=%d; saved %p dp %p\n", y, saved, dp)
		fmt.Fprintf(os.Stderr, "draw dst %v src %v mask %v\n", r, sp, mp)
		dumpimage("src", src, src.Data.Bdata, sp)
		if stmp != nil {
			dumpimage("stmp", stmp, stmp.Data.Bdata, sp)
		}
		dumpimage("mask", mask, mask.Data.Bdata, mp)
		if mtmp != nil {
			dumpimage("mtmp", mtmp, mtmp.Data.Bdata, mp)
		}
		dumpimage("origdst", dst, dstbits, r.Min)
		dumpimage("dst", dst, dst.Data.Bdata, r.Min)
		dumpimage("gooddst", dst, savedstbits, r.Min)
		panic("fail")
	}
}

/*
 * Fill the bits of an image with random data.
 * The memdraw.Image parameter is used only to make sure
 * the data is well formatted: only ucbits is written.
 */
func fill(img *memdraw.Image, ucbits []uint8) {
	if img.Flags&memdraw.Falpha == 0 {
		up := ucbits
		var i int
		for i = 0; i < nbytes/2; i++ {
			up[0] = byte(rand.Int() >> 7)
			up = up[1:]
		}
		if i+i != nbytes {
			up[0] = byte(rand.Int() >> 7)
		}
	} else {
		data := img.Data.Bdata
		img.Data.Bdata = ucbits

		for x := img.R.Min.X; x < img.R.Max.X; x++ {
			for y := img.R.Min.Y; y < img.R.Max.Y; y++ {
				alpha := rand.Int() >> 4
				r := uint8(rand.Int() % (alpha + 1))
				g := uint8(rand.Int() % (alpha + 1))
				b := uint8(rand.Int() % (alpha + 1))
				putpixel(img, draw.Pt(x, y), rgbatopix(r, g, b, uint8(alpha)))
			}
		}
		img.Data.Bdata = data
	}

}

func memmove(x, y []byte, n int) {
	copy(x[:n], y[:n])
}
func memset(x []byte, v byte, n int) {
	x = x[:n]
	for i := range x {
		x[i] = v
	}
}

func assert(b bool) {
	if !b {
		panic("assert")
	}
}

/*
 * Mask is preset; do the rest
 */
func verifyonemask() {
	fill(dst, dstbits)
	fill(src, srcbits)
	memmove(dst.Data.Bdata, dstbits, int(dst.Width)*4*Yrange)
	memmove(src.Data.Bdata, srcbits, int(src.Width)*4*Yrange)
	memmove(mask.Data.Bdata, maskbits, int(mask.Width)*4*Yrange)
	var dp draw.Point

	dp.X = rand.Intn(Xrange)
	dp.Y = rand.Intn(Yrange)
	var sp draw.Point

	sp.X = rand.Intn(Xrange)
	sp.Y = rand.Intn(Yrange)
	var mp draw.Point

	mp.X = rand.Intn(Xrange)
	mp.Y = rand.Intn(Yrange)

	drawonepixel(dst, dp, src, sp, mask, mp)
	memmove(mask.Data.Bdata, maskbits, int(mask.Width)*4*Yrange)
	memmove(savedstbits, dst.Data.Bdata, int(dst.Width)*4*Yrange)

	memmove(dst.Data.Bdata, dstbits, int(dst.Width)*4*Yrange)
	dst.Draw(draw.Rect(dp.X, dp.Y, dp.X+1, dp.Y+1), src, sp, mask, mp, draw.SoverD)
	memmove(mask.Data.Bdata, maskbits, int(mask.Width)*4*Yrange)

	checkone(dp, sp, mp)
}

func verifyone() {
	/* mask all zeros */
	memset(maskbits, 0, nbytes)
	var i int
	for i = 0; i < niters; i++ {
		verifyonemask()
	}

	/* mask all ones */
	memset(maskbits, 0xFF, nbytes)
	for i = 0; i < niters; i++ {
		verifyonemask()
	}

	/* random mask */
	for i = 0; i < niters; i++ {
		fill(mask, maskbits)
		verifyonemask()
	}
}

/*
 * Mask is preset; do the rest
 */
func verifylinemask() {
	if Xrange <= 1 || Yrange <= 1 {
		return
	}
	fill(dst, dstbits)
	fill(src, srcbits)
	memmove(dst.Data.Bdata, dstbits, int(dst.Width)*4*Yrange)
	memmove(src.Data.Bdata, srcbits, int(src.Width)*4*Yrange)
	memmove(mask.Data.Bdata, maskbits, int(mask.Width)*4*Yrange)
	var dr draw.Rectangle

	dr.Min.X = rand.Intn(Xrange - 1)
	dr.Min.Y = rand.Intn(Yrange - 1)
	dr.Max.X = dr.Min.X + 1 + rand.Intn(Xrange-1-dr.Min.X)
	dr.Max.Y = dr.Min.Y + 1
	var sp draw.Point

	sp.X = rand.Intn(Xrange)
	sp.Y = rand.Intn(Yrange)
	var mp draw.Point

	mp.X = rand.Intn(Xrange)
	mp.Y = rand.Intn(Yrange)

	tp := sp
	up := mp
	for x := dr.Min.X; x < dr.Max.X && tp.X < Xrange && up.X < Xrange; func() { x++; tp.X++; up.X++ }() {
		dst.Draw(draw.Rect(x, dr.Min.Y, x+1, dr.Min.Y+1), src, tp, mask, up, draw.SoverD)
	}
	memmove(savedstbits, dst.Data.Bdata, int(dst.Width)*4*Yrange)

	memmove(dst.Data.Bdata, dstbits, int(dst.Width)*4*Yrange)

	dst.Draw(dr, src, sp, mask, mp, draw.SoverD)
	checkline(dr, draw.Repl(src.R, sp), draw.Repl(mask.R, mp), dr.Min.Y, nil, nil)
}

func verifyline() {
	/* mask all ones */
	memset(maskbits, 0xFF, nbytes)
	var i int
	for i = 0; i < niters; i++ {
		verifylinemask()
	}

	/* mask all zeros */
	memset(maskbits, 0, nbytes)
	for i = 0; i < niters; i++ {
		verifylinemask()
	}

	/* random mask */
	for i = 0; i < niters; i++ {
		fill(mask, maskbits)
		verifylinemask()
	}
}

/*
 * Mask is preset; do the rest
 */
func verifyrectmask() {
	if Xrange <= 1 || Yrange <= 1 {
		return
	}
	fill(dst, dstbits)
	fill(src, srcbits)
	memmove(dst.Data.Bdata, dstbits, int(dst.Width)*4*Yrange)
	memmove(src.Data.Bdata, srcbits, int(src.Width)*4*Yrange)
	memmove(mask.Data.Bdata, maskbits, int(mask.Width)*4*Yrange)
	var dr draw.Rectangle

	dr.Min.X = rand.Intn(Xrange - 1)
	dr.Min.Y = rand.Intn(Yrange - 1)
	dr.Max.X = dr.Min.X + 1 + rand.Intn(Xrange-1-dr.Min.X)
	dr.Max.Y = dr.Min.Y + 1 + rand.Intn(Yrange-1-dr.Min.Y)
	var sp draw.Point

	sp.X = rand.Intn(Xrange)
	sp.Y = rand.Intn(Yrange)
	var mp draw.Point

	mp.X = rand.Intn(Xrange)
	mp.Y = rand.Intn(Yrange)

	tp := sp
	up := mp
	var y int
	for y = dr.Min.Y; y < dr.Max.Y && tp.Y < Yrange && up.Y < Yrange; func() { y++; tp.Y++; up.Y++ }() {
		for x := dr.Min.X; x < dr.Max.X && tp.X < Xrange && up.X < Xrange; func() { x++; tp.X++; up.X++ }() {
			dst.Draw(draw.Rect(x, y, x+1, y+1), src, tp, mask, up, draw.SoverD)
		}
		tp.X = sp.X
		up.X = mp.X
	}
	memmove(savedstbits, dst.Data.Bdata, int(dst.Width)*4*Yrange)

	memmove(dst.Data.Bdata, dstbits, int(dst.Width)*4*Yrange)

	dst.Draw(dr, src, sp, mask, mp, draw.SoverD)
	for y = 0; y < Yrange; y++ {
		checkline(dr, draw.Repl(src.R, sp), draw.Repl(mask.R, mp), y, nil, nil)
	}
}

func verifyrect() {
	/* mask all zeros */
	memset(maskbits, 0, nbytes)
	var i int
	for i = 0; i < niters; i++ {
		verifyrectmask()
	}

	/* mask all ones */
	memset(maskbits, 0xFF, nbytes)
	for i = 0; i < niters; i++ {
		verifyrectmask()
	}

	/* random mask */
	for i = 0; i < niters; i++ {
		fill(mask, maskbits)
		verifyrectmask()
	}
}

func randrect() draw.Rectangle {
	var r draw.Rectangle
	r.Min.X = rand.Intn(Xrange - 1)
	r.Min.Y = rand.Intn(Yrange - 1)
	r.Max.X = r.Min.X + 1 + rand.Intn(Xrange-1-r.Min.X)
	r.Max.Y = r.Min.Y + 1 + rand.Intn(Yrange-1-r.Min.Y)
	return r
}

/*
 * Return coordinate corresponding to x withing range [minx, maxx)
 */
func tilexy(minx int, maxx int, x int) int {
	sx := (x - minx) % (maxx - minx)
	if sx < 0 {
		sx += maxx - minx
	}
	return sx + minx
}

func replicate(i *memdraw.Image, tmp *memdraw.Image) {
	var r draw.Rectangle
	/* choose the replication window (i->r) */
	r.Min.X = rand.Intn(Xrange - 1)
	r.Min.Y = rand.Intn(Yrange - 1)
	/* make it trivial more often than pure chance allows */
	switch rand.Int() & 0 {
	case 1:
		r.Max.X = r.Min.X + 2
		r.Max.Y = r.Min.Y + 2
		if r.Max.X < Xrange && r.Max.Y < Yrange {
			break
		}
		fallthrough
	/* fall through */
	case 0:
		r.Max.X = r.Min.X + 1
		r.Max.Y = r.Min.Y + 1
	default:
		if r.Min.X+3 >= Xrange {
			r.Max.X = Xrange
		} else {
			r.Max.X = r.Min.X + 3 + rand.Intn(Xrange-(r.Min.X+3))
		}

		if r.Min.Y+3 >= Yrange {
			r.Max.Y = Yrange
		} else {
			r.Max.Y = r.Min.Y + 3 + rand.Intn(Yrange-(r.Min.Y+3))
		}
	}
	assert(r.Min.X >= 0)
	assert(r.Max.X <= Xrange)
	assert(r.Min.Y >= 0)
	assert(r.Max.Y <= Yrange)
	/* copy from i to tmp so we have just the replicated bits */
	nb := int(tmp.Width) * 4 * Yrange
	memset(tmp.Data.Bdata, 0, nb)
	tmp.Draw(r, i, r.Min, ones, r.Min, draw.SoverD)
	memmove(i.Data.Bdata, tmp.Data.Bdata, nb)
	/* i is now a non-replicated instance of the replication */
	/* replicate it by hand through tmp */
	memset(tmp.Data.Bdata, 0, nb)
	x := -(tilexy(r.Min.X, r.Max.X, 0) - r.Min.X)
	for ; x < Xrange; x += r.Dx() {
		y := -(tilexy(r.Min.Y, r.Max.Y, 0) - r.Min.Y)
		for ; y < Yrange; y += r.Dy() {
			var r1 draw.Rectangle
			/* set r1 to instance of tile by translation */
			r1.Min.X = x
			r1.Min.Y = y
			r1.Max.X = r1.Min.X + r.Dx()
			r1.Max.Y = r1.Min.Y + r.Dy()
			tmp.Draw(r1, i, r.Min, ones, r.Min, draw.SoverD)
		}
	}
	i.Flags |= memdraw.Frepl
	i.R = r
	i.Clipr = randrect()
	/*	fprint(2, "replicate [[%d %d] [%d %d]] [[%d %d][%d %d]]\n", r.Min.X, r.Min.Y, r.Max.X, r.Max.Y, */
	/*		i->clipr.Min.X, i->clipr.Min.Y, i->clipr.Max.X, i->clipr.Max.Y); */
	tmp.Clipr = i.Clipr
}

/*
 * Mask is preset; do the rest
 */
func verifyrectmaskrepl(srcrepl int, maskrepl int) {
	/*	fmt.Fprintf(os.Stderr, "verfrect %d %d\n", srcrepl, maskrepl); */
	src.Flags &^= memdraw.Frepl
	src.R = draw.Rect(0, 0, Xrange, Yrange)
	src.Clipr = src.R
	stmp.Flags &^= memdraw.Frepl
	stmp.R = draw.Rect(0, 0, Xrange, Yrange)
	stmp.Clipr = src.R
	mask.Flags &^= memdraw.Frepl
	mask.R = draw.Rect(0, 0, Xrange, Yrange)
	mask.Clipr = mask.R
	mtmp.Flags &^= memdraw.Frepl
	mtmp.R = draw.Rect(0, 0, Xrange, Yrange)
	mtmp.Clipr = mask.R

	fill(dst, dstbits)
	fill(src, srcbits)

	memmove(dst.Data.Bdata, dstbits, int(dst.Width)*4*Yrange)
	memmove(src.Data.Bdata, srcbits, int(src.Width)*4*Yrange)
	memmove(mask.Data.Bdata, maskbits, int(mask.Width)*4*Yrange)
	var s *memdraw.Image

	if srcrepl != 0 {
		replicate(src, stmp)
		s = stmp
	} else {
		s = src
	}
	var m *memdraw.Image
	if maskrepl != 0 {
		replicate(mask, mtmp)
		m = mtmp
	} else {
		m = mask
	}

	dr := randrect()
	var sp draw.Point

	sp.X = rand.Intn(Xrange)
	sp.Y = rand.Intn(Yrange)
	var mp draw.Point

	mp.X = rand.Intn(Xrange)
	mp.Y = rand.Intn(Yrange)

	if DBG != 0 {
		fmt.Fprintf(os.Stderr, "smalldraws\n")
	}
	var tp draw.Point
	tp.Y = sp.Y
	var up draw.Point
	up.Y = mp.Y
	y := dr.Min.Y
	for ; y < dr.Max.Y && tp.Y < Yrange && up.Y < Yrange; func() { y++; tp.Y++; up.Y++ }() {
		tp.X = sp.X
		up.X = mp.X
		x := dr.Min.X
		for ; x < dr.Max.X && tp.X < Xrange && up.X < Xrange; func() { x++; tp.X++; up.X++ }() {
			dst.Draw(draw.Rect(x, y, x+1, y+1), s, tp, m, up, draw.SoverD)
		}
	}
	memmove(savedstbits, dst.Data.Bdata, int(dst.Width)*4*Yrange)

	memmove(dst.Data.Bdata, dstbits, int(dst.Width)*4*Yrange)

	if DBG != 0 {
		fmt.Fprintf(os.Stderr, "bigdraw\n")
	}
	dst.Draw(dr, src, sp, mask, mp, draw.SoverD)
	for y = 0; y < Yrange; y++ {
		stmp := stmp
		if srcrepl == 0 {
			stmp = nil
		}
		mtmp := mtmp
		if maskrepl == 0 {
			mtmp = nil
		}
		checkline(dr, draw.Repl(src.R, sp), draw.Repl(mask.R, mp), y, stmp, mtmp)
	}
}

func verifyrectrepl(srcrepl int, maskrepl int) {
	if Xrange <= 1 || Yrange <= 1 {
		return
	}
	/* mask all ones */
	memset(maskbits, 0xFF, nbytes)
	var i int
	for i = 0; i < niters; i++ {
		verifyrectmaskrepl(srcrepl, maskrepl)
	}

	/* mask all zeros */
	memset(maskbits, 0, nbytes)
	for i = 0; i < niters; i++ {
		verifyrectmaskrepl(srcrepl, maskrepl)
	}

	/* random mask */
	for i = 0; i < niters; i++ {
		fill(mask, maskbits)
		verifyrectmaskrepl(srcrepl, maskrepl)
	}
}

/*
 * Trivial draw implementation.
 * Color values are passed around as u32ints containing ααRRGGBB
 */

/*
 * Convert v, which is nhave bits wide, into its nwant bits wide equivalent.
 * Replicates to widen the value, truncates to narrow it.
 */
func replbits(v uint32, nhave, nwant uint) uint32 {
	v &= (1 << nhave) - 1
	for ; nhave < nwant; nhave *= 2 {
		v |= v << nhave
	}
	v >>= (nhave - nwant)
	return v & ((1 << nwant) - 1)
}

/*
 * Decode a pixel into the uchar* values.
 */
func pixtorgba(v uint32) (r, g, b, a uint8) {
	a = uint8(v >> 24)
	r = uint8(v >> 16)
	g = uint8(v >> 8)
	b = uint8(v)
	return
}

/*
 * Convert uchar channels into u32int pixel.
 */
func rgbatopix(r uint8, g uint8, b uint8, a uint8) uint32 {
	return uint32(a)<<24 | uint32(r)<<16 | uint32(g)<<8 | uint32(b)
}

/*
 * Retrieve the pixel value at pt in the image.
 */
func getpixel(img *memdraw.Image, pt draw.Point) uint32 {
	var r, g, b, a uint8
	a = ^uint8(0) /* default alpha is full */

	p := img.BytesAt(pt)
	v := uint32(p[0]) | uint32(p[1])<<8 | uint32(p[2])<<16 | uint32(p[3])<<24
	bpp := uint(img.Depth)
	if bpp < 8 {
		/*
		 * Sub-byte greyscale pixels.
		 *
		 * We want to throw away the top pt.X%npack pixels and then use the next bpp bits
		 * in the bottom byte of v.  This madness is due to having big endian bits
		 * but little endian bytes.
		 */
		npack := 8 / bpp
		v >>= 8 - bpp*(uint(pt.X)%npack+1)
		v &= (1 << bpp) - 1
		b = uint8(replbits(uint32(v), bpp, 8))
		g = b
		r = g
	} else {
		/*
		 * General case.  We need to parse the channel descriptor and do what it says.
		 * In all channels but the color map, we replicate to 8 bits because that's the
		 * precision that all calculations are done at.
		 *
		 * In the case of the color map, we leave the bits alone, in case a color map
		 * with less than 8 bits of index is used.  This is currently disallowed, so it's
		 * sort of silly.
		 */

		for c := img.Pix; c != 0; c >>= 8 {
			nbits := uint(c & 15)
			typ := int(c>>4) & 15
			bits := v & ((1 << nbits) - 1)
			rbits := uint8(replbits(uint32(bits), nbits, 8))
			v >>= nbits
			switch typ {
			case draw.CRed:
				r = rbits
			case draw.CGreen:
				g = rbits
			case draw.CBlue:
				b = rbits
			case draw.CGrey:
				b = rbits
				g = b
				r = g
			case draw.CAlpha:
				a = rbits
			case draw.CMap:
				p = cmap2rgb[3*int(bits):]
				r = p[0]
				g = p[1]
				b = p[2]
				if DBG != 0 {
					fmt.Fprintf(os.Stderr, "%x -> %x %x %x\n", bits, r, g, b)
				}
			case draw.CIgnore:
				break
			default:
				fmt.Fprintf(os.Stderr, "unknown channel type %d\n", typ)
				panic("channel")
			}
		}
	}
	return rgbatopix(r, g, b, a)
}

/*
 * Return the greyscale equivalent of a pixel.
 */
func getgrey(img *memdraw.Image, pt draw.Point) uint8 {
	rgba := getpixel(img, pt)
	r, g, b, _ := pixtorgba(rgba)
	return RGB2K(r, g, b)
}

/*
 * Return the value at pt in image, if image is interpreted
 * as a mask.  This means the alpha channel if present, else
 * the greyscale or its computed equivalent.
 */
func getmask(img *memdraw.Image, pt draw.Point) uint8 {
	if img.Flags&memdraw.Falpha != 0 {
		return uint8(getpixel(img, pt) >> 24)
	} else {
		return getgrey(img, pt)
	}
}

// #undef DBG

// #define DBG 0
/*
 * Write a pixel to img at point pt.
 *
 * We do this by reading a 32-bit little endian
 * value from p and then writing it back
 * after tweaking the appropriate bits.  Because
 * the data is little endian, we don't have to worry
 * about what the actual depth is, as long as it is
 * less than 32 bits.
 */
func putpixel(img *memdraw.Image, pt draw.Point, nv uint32) {
	r, g, b, a := pixtorgba(nv)

	p := img.BytesAt(pt)
	v := uint32(p[0]) | uint32(p[1])<<8 | uint32(p[2])<<16 | uint32(p[3])<<24
	bpp := uint(img.Depth)
	if DBG != 0 {
		fmt.Fprintf(os.Stderr, "v %.8x...", v)
	}
	if bpp < 8 {
		/*
		 * Sub-byte greyscale pixels.  We need to skip the leftmost pt.X%npack pixels,
		 * which is equivalent to skipping the rightmost npack - pt.X%npack - 1 pixels.
		 */
		npack := 8 / bpp
		sh := bpp * (npack - uint(pt.X)%npack - 1)
		bits := RGB2K(r, g, b)
		if DBG != 0 {
			fmt.Fprintf(os.Stderr, "repl %x 8 %d = %x...", bits, bpp, replbits(uint32(bits), 8, bpp))
		}
		bits = uint8(replbits(uint32(bits), 8, bpp))
		mask := uint32(1<<bpp) - 1
		if DBG != 0 {
			fmt.Fprintf(os.Stderr, "bits %x mask %x sh %d...", bits, mask, sh)
		}
		mask <<= sh
		bits <<= sh
		if DBG != 0 {
			fmt.Fprintf(os.Stderr, "(%x & %x) | (%x & %x)", v, ^mask, bits, mask)
		}
		v = v&^mask | uint32(bits)&mask
	} else {
		/*
		 * General case.  We need to parse the channel descriptor again.
		 */
		sh := uint(0)
		for c := img.Pix; c != 0; c >>= 8 {
			nbits := uint(c & 15)
			typ := int(c>>4) & 15
			var q []uint8
			var bits uint32
			switch typ {
			case draw.CRed:
				bits = uint32(r)
			case draw.CGreen:
				bits = uint32(g)
			case draw.CBlue:
				bits = uint32(b)
			case draw.CGrey:
				bits = uint32(RGB2K(r, g, b))
			case draw.CAlpha:
				bits = uint32(a)
			case draw.CIgnore:
				bits = 0
			case draw.CMap:
				q = rgb2cmap[:]
				bits = uint32(q[(uint32(r)>>4)*16*16+(uint32(g)>>4)*16+(uint32(b)>>4)])
				if DBG != 0 {
					fmt.Fprintf(os.Stderr, "%x %x %x -> %x\n", r, g, b, bits)
				}
			default:
				fmt.Fprintf(os.Stderr, "unknown channel type %d\n", typ)
				panic("channel")
			}

			if DBG != 0 {
				fmt.Fprintf(os.Stderr, "repl %x 8 %d = %x...", bits, nbits, replbits(uint32(bits), 8, nbits))
			}
			if typ != draw.CMap {
				bits = replbits(uint32(bits), 8, nbits)
			}
			mask := uint32(1<<nbits) - 1
			if DBG != 0 {
				fmt.Fprintf(os.Stderr, "bits %x mask %x sh %d...", bits, mask, sh)
			}
			bits <<= sh
			mask <<= sh
			v = v&^mask | uint32(bits)&mask
			sh += nbits
		}
	}
	if DBG != 0 {
		fmt.Fprintf(os.Stderr, "v %.8x\n", v)
	}
	p[0] = uint8(v)
	p[1] = uint8(v >> 8)
	p[2] = uint8(v >> 16)
	p[3] = uint8(v >> 24)
}

func drawonepixel(dst *memdraw.Image, dp draw.Point, src *memdraw.Image, sp draw.Point, mask *memdraw.Image, mp draw.Point) {
	dr, dg, db, da := pixtorgba(getpixel(dst, dp))
	sr, sg, sb, sa := pixtorgba(getpixel(src, sp))
	m := uint32(getmask(mask, mp))
	M := 255 - (uint32(sa)*m+127)/255

	if DBG != 0 {
		fmt.Fprintf(os.Stderr, "dst %x %x %x %x src %x %x %x %x m %x M %x = ", dr, dg, db, da, sr, sg, sb, sa, m, M)
	}
	if dst.Flags&memdraw.Fgrey != 0 {
		/*
		 * We need to do the conversion to grey before the alpha calculation
		 * because the draw operator does this, and we need to be operating
		 * at the same precision so we get exactly the same answers.
		 */
		sk := RGB2K(sr, sg, sb)
		dk := RGB2K(dr, dg, db)
		dk = uint8((uint32(sk)*m + uint32(dk)*M + 127) / 255)
		db = dk
		dg = db
		dr = dg
		da = uint8((uint32(sa)*m + uint32(da)*M + 127) / 255)
	} else {
		/*
		 * True color alpha calculation treats all channels (including alpha)
		 * the same.  It might have been nice to use an array, but oh well.
		 */
		dr = uint8((uint32(sr)*m + uint32(dr)*M + 127) / 255)
		dg = uint8((uint32(sg)*m + uint32(dg)*M + 127) / 255)
		db = uint8((uint32(sb)*m + uint32(db)*M + 127) / 255)
		da = uint8((uint32(sa)*m + uint32(da)*M + 127) / 255)
	}

	if DBG != 0 {
		fmt.Fprintf(os.Stderr, "%x %x %x %x %x\n", dr, dg, db, da, rgbatopix(dr, dg, db, da))
	}
	putpixel(dst, dp, rgbatopix(dr, dg, db, da))
}

var cmap2rgb = [...]uint8{
	0x00, 0x00, 0x00, 0x00, 0x00, 0x44, 0x00, 0x00, 0x88, 0x00, 0x00, 0xcc, 0x00, 0x44, 0x00, 0x00,
	0x44, 0x44, 0x00, 0x44, 0x88, 0x00, 0x44, 0xcc, 0x00, 0x88, 0x00, 0x00, 0x88, 0x44, 0x00, 0x88,
	0x88, 0x00, 0x88, 0xcc, 0x00, 0xcc, 0x00, 0x00, 0xcc, 0x44, 0x00, 0xcc, 0x88, 0x00, 0xcc, 0xcc,
	0x00, 0xdd, 0xdd, 0x11, 0x11, 0x11, 0x00, 0x00, 0x55, 0x00, 0x00, 0x99, 0x00, 0x00, 0xdd, 0x00,
	0x55, 0x00, 0x00, 0x55, 0x55, 0x00, 0x4c, 0x99, 0x00, 0x49, 0xdd, 0x00, 0x99, 0x00, 0x00, 0x99,
	0x4c, 0x00, 0x99, 0x99, 0x00, 0x93, 0xdd, 0x00, 0xdd, 0x00, 0x00, 0xdd, 0x49, 0x00, 0xdd, 0x93,
	0x00, 0xee, 0x9e, 0x00, 0xee, 0xee, 0x22, 0x22, 0x22, 0x00, 0x00, 0x66, 0x00, 0x00, 0xaa, 0x00,
	0x00, 0xee, 0x00, 0x66, 0x00, 0x00, 0x66, 0x66, 0x00, 0x55, 0xaa, 0x00, 0x4f, 0xee, 0x00, 0xaa,
	0x00, 0x00, 0xaa, 0x55, 0x00, 0xaa, 0xaa, 0x00, 0x9e, 0xee, 0x00, 0xee, 0x00, 0x00, 0xee, 0x4f,
	0x00, 0xff, 0x55, 0x00, 0xff, 0xaa, 0x00, 0xff, 0xff, 0x33, 0x33, 0x33, 0x00, 0x00, 0x77, 0x00,
	0x00, 0xbb, 0x00, 0x00, 0xff, 0x00, 0x77, 0x00, 0x00, 0x77, 0x77, 0x00, 0x5d, 0xbb, 0x00, 0x55,
	0xff, 0x00, 0xbb, 0x00, 0x00, 0xbb, 0x5d, 0x00, 0xbb, 0xbb, 0x00, 0xaa, 0xff, 0x00, 0xff, 0x00,
	0x44, 0x00, 0x44, 0x44, 0x00, 0x88, 0x44, 0x00, 0xcc, 0x44, 0x44, 0x00, 0x44, 0x44, 0x44, 0x44,
	0x44, 0x88, 0x44, 0x44, 0xcc, 0x44, 0x88, 0x00, 0x44, 0x88, 0x44, 0x44, 0x88, 0x88, 0x44, 0x88,
	0xcc, 0x44, 0xcc, 0x00, 0x44, 0xcc, 0x44, 0x44, 0xcc, 0x88, 0x44, 0xcc, 0xcc, 0x44, 0x00, 0x00,
	0x55, 0x00, 0x00, 0x55, 0x00, 0x55, 0x4c, 0x00, 0x99, 0x49, 0x00, 0xdd, 0x55, 0x55, 0x00, 0x55,
	0x55, 0x55, 0x4c, 0x4c, 0x99, 0x49, 0x49, 0xdd, 0x4c, 0x99, 0x00, 0x4c, 0x99, 0x4c, 0x4c, 0x99,
	0x99, 0x49, 0x93, 0xdd, 0x49, 0xdd, 0x00, 0x49, 0xdd, 0x49, 0x49, 0xdd, 0x93, 0x49, 0xdd, 0xdd,
	0x4f, 0xee, 0xee, 0x66, 0x00, 0x00, 0x66, 0x00, 0x66, 0x55, 0x00, 0xaa, 0x4f, 0x00, 0xee, 0x66,
	0x66, 0x00, 0x66, 0x66, 0x66, 0x55, 0x55, 0xaa, 0x4f, 0x4f, 0xee, 0x55, 0xaa, 0x00, 0x55, 0xaa,
	0x55, 0x55, 0xaa, 0xaa, 0x4f, 0x9e, 0xee, 0x4f, 0xee, 0x00, 0x4f, 0xee, 0x4f, 0x4f, 0xee, 0x9e,
	0x55, 0xff, 0xaa, 0x55, 0xff, 0xff, 0x77, 0x00, 0x00, 0x77, 0x00, 0x77, 0x5d, 0x00, 0xbb, 0x55,
	0x00, 0xff, 0x77, 0x77, 0x00, 0x77, 0x77, 0x77, 0x5d, 0x5d, 0xbb, 0x55, 0x55, 0xff, 0x5d, 0xbb,
	0x00, 0x5d, 0xbb, 0x5d, 0x5d, 0xbb, 0xbb, 0x55, 0xaa, 0xff, 0x55, 0xff, 0x00, 0x55, 0xff, 0x55,
	0x88, 0x00, 0x88, 0x88, 0x00, 0xcc, 0x88, 0x44, 0x00, 0x88, 0x44, 0x44, 0x88, 0x44, 0x88, 0x88,
	0x44, 0xcc, 0x88, 0x88, 0x00, 0x88, 0x88, 0x44, 0x88, 0x88, 0x88, 0x88, 0x88, 0xcc, 0x88, 0xcc,
	0x00, 0x88, 0xcc, 0x44, 0x88, 0xcc, 0x88, 0x88, 0xcc, 0xcc, 0x88, 0x00, 0x00, 0x88, 0x00, 0x44,
	0x99, 0x00, 0x4c, 0x99, 0x00, 0x99, 0x93, 0x00, 0xdd, 0x99, 0x4c, 0x00, 0x99, 0x4c, 0x4c, 0x99,
	0x4c, 0x99, 0x93, 0x49, 0xdd, 0x99, 0x99, 0x00, 0x99, 0x99, 0x4c, 0x99, 0x99, 0x99, 0x93, 0x93,
	0xdd, 0x93, 0xdd, 0x00, 0x93, 0xdd, 0x49, 0x93, 0xdd, 0x93, 0x93, 0xdd, 0xdd, 0x99, 0x00, 0x00,
	0xaa, 0x00, 0x00, 0xaa, 0x00, 0x55, 0xaa, 0x00, 0xaa, 0x9e, 0x00, 0xee, 0xaa, 0x55, 0x00, 0xaa,
	0x55, 0x55, 0xaa, 0x55, 0xaa, 0x9e, 0x4f, 0xee, 0xaa, 0xaa, 0x00, 0xaa, 0xaa, 0x55, 0xaa, 0xaa,
	0xaa, 0x9e, 0x9e, 0xee, 0x9e, 0xee, 0x00, 0x9e, 0xee, 0x4f, 0x9e, 0xee, 0x9e, 0x9e, 0xee, 0xee,
	0xaa, 0xff, 0xff, 0xbb, 0x00, 0x00, 0xbb, 0x00, 0x5d, 0xbb, 0x00, 0xbb, 0xaa, 0x00, 0xff, 0xbb,
	0x5d, 0x00, 0xbb, 0x5d, 0x5d, 0xbb, 0x5d, 0xbb, 0xaa, 0x55, 0xff, 0xbb, 0xbb, 0x00, 0xbb, 0xbb,
	0x5d, 0xbb, 0xbb, 0xbb, 0xaa, 0xaa, 0xff, 0xaa, 0xff, 0x00, 0xaa, 0xff, 0x55, 0xaa, 0xff, 0xaa,
	0xcc, 0x00, 0xcc, 0xcc, 0x44, 0x00, 0xcc, 0x44, 0x44, 0xcc, 0x44, 0x88, 0xcc, 0x44, 0xcc, 0xcc,
	0x88, 0x00, 0xcc, 0x88, 0x44, 0xcc, 0x88, 0x88, 0xcc, 0x88, 0xcc, 0xcc, 0xcc, 0x00, 0xcc, 0xcc,
	0x44, 0xcc, 0xcc, 0x88, 0xcc, 0xcc, 0xcc, 0xcc, 0x00, 0x00, 0xcc, 0x00, 0x44, 0xcc, 0x00, 0x88,
	0xdd, 0x00, 0x93, 0xdd, 0x00, 0xdd, 0xdd, 0x49, 0x00, 0xdd, 0x49, 0x49, 0xdd, 0x49, 0x93, 0xdd,
	0x49, 0xdd, 0xdd, 0x93, 0x00, 0xdd, 0x93, 0x49, 0xdd, 0x93, 0x93, 0xdd, 0x93, 0xdd, 0xdd, 0xdd,
	0x00, 0xdd, 0xdd, 0x49, 0xdd, 0xdd, 0x93, 0xdd, 0xdd, 0xdd, 0xdd, 0x00, 0x00, 0xdd, 0x00, 0x49,
	0xee, 0x00, 0x4f, 0xee, 0x00, 0x9e, 0xee, 0x00, 0xee, 0xee, 0x4f, 0x00, 0xee, 0x4f, 0x4f, 0xee,
	0x4f, 0x9e, 0xee, 0x4f, 0xee, 0xee, 0x9e, 0x00, 0xee, 0x9e, 0x4f, 0xee, 0x9e, 0x9e, 0xee, 0x9e,
	0xee, 0xee, 0xee, 0x00, 0xee, 0xee, 0x4f, 0xee, 0xee, 0x9e, 0xee, 0xee, 0xee, 0xee, 0x00, 0x00,
	0xff, 0x00, 0x00, 0xff, 0x00, 0x55, 0xff, 0x00, 0xaa, 0xff, 0x00, 0xff, 0xff, 0x55, 0x00, 0xff,
	0x55, 0x55, 0xff, 0x55, 0xaa, 0xff, 0x55, 0xff, 0xff, 0xaa, 0x00, 0xff, 0xaa, 0x55, 0xff, 0xaa,
	0xaa, 0xff, 0xaa, 0xff, 0xff, 0xff, 0x00, 0xff, 0xff, 0x55, 0xff, 0xff, 0xaa, 0xff, 0xff, 0xff,
}

var rgb2cmap = [...]uint8{
	0x00, 0x00, 0x11, 0x01, 0x01, 0x12, 0x23, 0x34, 0x02, 0x13, 0x24, 0x35, 0x03, 0x14, 0x25, 0x36,
	0x00, 0x11, 0x11, 0x01, 0x01, 0x12, 0x23, 0x34, 0x02, 0x13, 0x24, 0x35, 0x03, 0x14, 0x25, 0x36,
	0x11, 0x11, 0x11, 0x01, 0x01, 0x12, 0x23, 0x34, 0x02, 0x13, 0x24, 0x35, 0x03, 0x14, 0x25, 0x36,
	0x04, 0x04, 0x04, 0x05, 0x05, 0x05, 0x05, 0x06, 0x06, 0x06, 0x17, 0x07, 0x07, 0x18, 0x18, 0x29,
	0x04, 0x04, 0x04, 0x05, 0x05, 0x05, 0x16, 0x06, 0x06, 0x17, 0x28, 0x07, 0x07, 0x18, 0x29, 0x3a,
	0x15, 0x15, 0x15, 0x05, 0x05, 0x16, 0x16, 0x06, 0x06, 0x17, 0x28, 0x39, 0x07, 0x18, 0x29, 0x3a,
	0x26, 0x26, 0x26, 0x05, 0x16, 0x16, 0x27, 0x27, 0x38, 0x28, 0x28, 0x39, 0x39, 0x29, 0x29, 0x3a,
	0x37, 0x37, 0x37, 0x09, 0x09, 0x09, 0x27, 0x38, 0x0a, 0x0a, 0x39, 0x0b, 0x0b, 0x0b, 0x1c, 0x3a,
	0x08, 0x08, 0x08, 0x09, 0x09, 0x09, 0x38, 0x0a, 0x0a, 0x0a, 0x1b, 0x0b, 0x0b, 0x1c, 0x1c, 0x2d,
	0x19, 0x19, 0x19, 0x09, 0x1a, 0x1a, 0x2b, 0x0a, 0x0a, 0x1b, 0x1b, 0x0b, 0x0b, 0x1c, 0x2d, 0x3e,
	0x2a, 0x2a, 0x2a, 0x1a, 0x2b, 0x2b, 0x2b, 0x3c, 0x1b, 0x1b, 0x2c, 0x2c, 0x3d, 0x2d, 0x2d, 0x3e,
	0x3b, 0x3b, 0x3b, 0x0d, 0x0d, 0x3c, 0x3c, 0x0e, 0x0e, 0x0e, 0x2c, 0x3d, 0x0f, 0x0f, 0x3e, 0x3e,
	0x0c, 0x0c, 0x0c, 0x0d, 0x0d, 0x0d, 0x3c, 0x0e, 0x0e, 0x0e, 0x3d, 0x0f, 0x0f, 0x0f, 0x10, 0x3e,
	0x1d, 0x1d, 0x1d, 0x1e, 0x1e, 0x1e, 0x2f, 0x0e, 0x1f, 0x1f, 0x20, 0x0f, 0x0f, 0x10, 0x10, 0x21,
	0x2e, 0x2e, 0x2e, 0x1e, 0x2f, 0x2f, 0x2f, 0x1f, 0x1f, 0x20, 0x20, 0x31, 0x10, 0x10, 0x21, 0x21,
	0x3f, 0x3f, 0x3f, 0x2f, 0x30, 0x30, 0x30, 0x30, 0x20, 0x31, 0x31, 0x31, 0x31, 0x21, 0x21, 0x32,
	0x00, 0x11, 0x11, 0x01, 0x01, 0x12, 0x23, 0x34, 0x02, 0x13, 0x24, 0x35, 0x03, 0x14, 0x25, 0x36,
	0x11, 0x11, 0x11, 0x01, 0x01, 0x12, 0x23, 0x34, 0x02, 0x13, 0x24, 0x35, 0x03, 0x14, 0x25, 0x36,
	0x11, 0x11, 0x22, 0x22, 0x01, 0x12, 0x23, 0x34, 0x02, 0x13, 0x24, 0x35, 0x03, 0x14, 0x25, 0x36,
	0x04, 0x04, 0x22, 0x05, 0x05, 0x05, 0x05, 0x06, 0x06, 0x06, 0x17, 0x07, 0x07, 0x18, 0x18, 0x29,
	0x04, 0x04, 0x04, 0x05, 0x05, 0x05, 0x16, 0x06, 0x06, 0x17, 0x28, 0x07, 0x07, 0x18, 0x29, 0x3a,
	0x15, 0x15, 0x15, 0x05, 0x05, 0x16, 0x16, 0x06, 0x06, 0x17, 0x28, 0x39, 0x07, 0x18, 0x29, 0x3a,
	0x26, 0x26, 0x26, 0x05, 0x16, 0x16, 0x27, 0x27, 0x38, 0x28, 0x28, 0x39, 0x39, 0x29, 0x29, 0x3a,
	0x37, 0x37, 0x37, 0x09, 0x09, 0x09, 0x27, 0x38, 0x0a, 0x0a, 0x39, 0x0b, 0x0b, 0x0b, 0x1c, 0x3a,
	0x08, 0x08, 0x08, 0x09, 0x09, 0x09, 0x38, 0x0a, 0x0a, 0x0a, 0x1b, 0x0b, 0x0b, 0x1c, 0x1c, 0x2d,
	0x19, 0x19, 0x19, 0x09, 0x1a, 0x1a, 0x2b, 0x0a, 0x0a, 0x1b, 0x1b, 0x0b, 0x0b, 0x1c, 0x2d, 0x3e,
	0x2a, 0x2a, 0x2a, 0x1a, 0x2b, 0x2b, 0x2b, 0x3c, 0x1b, 0x1b, 0x2c, 0x2c, 0x3d, 0x2d, 0x2d, 0x3e,
	0x3b, 0x3b, 0x3b, 0x0d, 0x0d, 0x3c, 0x3c, 0x0e, 0x0e, 0x0e, 0x2c, 0x3d, 0x0f, 0x0f, 0x3e, 0x3e,
	0x0c, 0x0c, 0x0c, 0x0d, 0x0d, 0x0d, 0x3c, 0x0e, 0x0e, 0x0e, 0x3d, 0x0f, 0x0f, 0x0f, 0x10, 0x3e,
	0x1d, 0x1d, 0x1d, 0x1e, 0x1e, 0x1e, 0x2f, 0x0e, 0x1f, 0x1f, 0x20, 0x0f, 0x0f, 0x10, 0x10, 0x21,
	0x2e, 0x2e, 0x2e, 0x1e, 0x2f, 0x2f, 0x2f, 0x1f, 0x1f, 0x20, 0x20, 0x31, 0x10, 0x10, 0x21, 0x21,
	0x3f, 0x3f, 0x3f, 0x2f, 0x30, 0x30, 0x30, 0x30, 0x20, 0x31, 0x31, 0x31, 0x31, 0x21, 0x21, 0x32,
	0x11, 0x11, 0x11, 0x01, 0x01, 0x12, 0x23, 0x34, 0x02, 0x13, 0x24, 0x35, 0x03, 0x14, 0x25, 0x36,
	0x11, 0x11, 0x22, 0x22, 0x01, 0x12, 0x23, 0x34, 0x02, 0x13, 0x24, 0x35, 0x03, 0x14, 0x25, 0x36,
	0x11, 0x22, 0x22, 0x22, 0x33, 0x33, 0x23, 0x34, 0x02, 0x13, 0x24, 0x35, 0x03, 0x14, 0x25, 0x36,
	0x04, 0x22, 0x22, 0x33, 0x33, 0x33, 0x05, 0x06, 0x06, 0x06, 0x17, 0x07, 0x07, 0x18, 0x18, 0x29,
	0x04, 0x04, 0x33, 0x33, 0x33, 0x05, 0x16, 0x06, 0x06, 0x17, 0x28, 0x07, 0x07, 0x18, 0x29, 0x3a,
	0x15, 0x15, 0x33, 0x33, 0x05, 0x16, 0x16, 0x06, 0x06, 0x17, 0x28, 0x39, 0x07, 0x18, 0x29, 0x3a,
	0x26, 0x26, 0x26, 0x05, 0x16, 0x16, 0x27, 0x27, 0x38, 0x28, 0x28, 0x39, 0x39, 0x29, 0x29, 0x3a,
	0x37, 0x37, 0x37, 0x09, 0x09, 0x09, 0x27, 0x38, 0x0a, 0x0a, 0x39, 0x0b, 0x0b, 0x0b, 0x1c, 0x3a,
	0x08, 0x08, 0x08, 0x09, 0x09, 0x09, 0x38, 0x0a, 0x0a, 0x0a, 0x1b, 0x0b, 0x0b, 0x1c, 0x1c, 0x2d,
	0x19, 0x19, 0x19, 0x09, 0x1a, 0x1a, 0x2b, 0x0a, 0x0a, 0x1b, 0x1b, 0x0b, 0x0b, 0x1c, 0x2d, 0x3e,
	0x2a, 0x2a, 0x2a, 0x1a, 0x2b, 0x2b, 0x2b, 0x3c, 0x1b, 0x1b, 0x2c, 0x2c, 0x3d, 0x2d, 0x2d, 0x3e,
	0x3b, 0x3b, 0x3b, 0x0d, 0x0d, 0x3c, 0x3c, 0x0e, 0x0e, 0x0e, 0x2c, 0x3d, 0x0f, 0x0f, 0x3e, 0x3e,
	0x0c, 0x0c, 0x0c, 0x0d, 0x0d, 0x0d, 0x3c, 0x0e, 0x0e, 0x0e, 0x3d, 0x0f, 0x0f, 0x0f, 0x10, 0x3e,
	0x1d, 0x1d, 0x1d, 0x1e, 0x1e, 0x1e, 0x2f, 0x0e, 0x1f, 0x1f, 0x20, 0x0f, 0x0f, 0x10, 0x10, 0x21,
	0x2e, 0x2e, 0x2e, 0x1e, 0x2f, 0x2f, 0x2f, 0x1f, 0x1f, 0x20, 0x20, 0x31, 0x10, 0x10, 0x21, 0x21,
	0x3f, 0x3f, 0x3f, 0x2f, 0x30, 0x30, 0x30, 0x30, 0x20, 0x31, 0x31, 0x31, 0x31, 0x21, 0x21, 0x32,
	0x4f, 0x4f, 0x22, 0x40, 0x40, 0x40, 0x40, 0x41, 0x41, 0x41, 0x52, 0x42, 0x42, 0x53, 0x53, 0x64,
	0x4f, 0x22, 0x22, 0x22, 0x40, 0x40, 0x40, 0x41, 0x41, 0x41, 0x52, 0x42, 0x42, 0x53, 0x53, 0x64,
	0x22, 0x22, 0x22, 0x33, 0x33, 0x33, 0x40, 0x41, 0x41, 0x41, 0x52, 0x42, 0x42, 0x53, 0x53, 0x64,
	0x43, 0x22, 0x33, 0x33, 0x33, 0x44, 0x44, 0x45, 0x45, 0x45, 0x56, 0x46, 0x46, 0x46, 0x57, 0x68,
	0x43, 0x43, 0x33, 0x33, 0x44, 0x44, 0x44, 0x45, 0x45, 0x45, 0x56, 0x46, 0x46, 0x57, 0x57, 0x68,
	0x43, 0x43, 0x33, 0x44, 0x44, 0x44, 0x55, 0x45, 0x45, 0x56, 0x56, 0x46, 0x46, 0x57, 0x68, 0x68,
	0x43, 0x43, 0x43, 0x44, 0x44, 0x55, 0x55, 0x45, 0x45, 0x56, 0x67, 0x46, 0x46, 0x57, 0x68, 0x79,
	0x47, 0x47, 0x47, 0x48, 0x48, 0x48, 0x48, 0x49, 0x49, 0x49, 0x49, 0x4a, 0x4a, 0x4a, 0x5b, 0x79,
	0x47, 0x47, 0x47, 0x48, 0x48, 0x48, 0x48, 0x49, 0x49, 0x49, 0x5a, 0x4a, 0x4a, 0x4a, 0x5b, 0x6c,
	0x47, 0x47, 0x47, 0x48, 0x48, 0x59, 0x59, 0x49, 0x49, 0x5a, 0x5a, 0x4a, 0x4a, 0x5b, 0x5b, 0x6c,
	0x58, 0x58, 0x58, 0x59, 0x59, 0x59, 0x6a, 0x49, 0x5a, 0x5a, 0x6b, 0x6b, 0x5b, 0x5b, 0x6c, 0x7d,
	0x4b, 0x4b, 0x4b, 0x4c, 0x4c, 0x4c, 0x4c, 0x4d, 0x4d, 0x4d, 0x6b, 0x4e, 0x4e, 0x4e, 0x6c, 0x7d,
	0x4b, 0x4b, 0x4b, 0x4c, 0x4c, 0x4c, 0x4c, 0x4d, 0x4d, 0x4d, 0x5e, 0x4e, 0x4e, 0x4e, 0x5f, 0x5f,
	0x5c, 0x5c, 0x5c, 0x4c, 0x5d, 0x5d, 0x5d, 0x4d, 0x4d, 0x5e, 0x5e, 0x4e, 0x4e, 0x5f, 0x5f, 0x60,
	0x5c, 0x5c, 0x5c, 0x5d, 0x5d, 0x6e, 0x6e, 0x5e, 0x5e, 0x5e, 0x6f, 0x6f, 0x5f, 0x5f, 0x60, 0x60,
	0x6d, 0x6d, 0x6d, 0x6e, 0x6e, 0x6e, 0x7f, 0x7f, 0x6f, 0x6f, 0x70, 0x70, 0x5f, 0x60, 0x60, 0x71,
	0x4f, 0x4f, 0x40, 0x40, 0x40, 0x40, 0x51, 0x41, 0x41, 0x52, 0x63, 0x42, 0x42, 0x53, 0x64, 0x75,
	0x4f, 0x4f, 0x22, 0x40, 0x40, 0x40, 0x51, 0x41, 0x41, 0x52, 0x63, 0x42, 0x42, 0x53, 0x64, 0x75,
	0x43, 0x22, 0x33, 0x33, 0x33, 0x40, 0x51, 0x41, 0x41, 0x52, 0x63, 0x42, 0x42, 0x53, 0x64, 0x75,
	0x43, 0x43, 0x33, 0x33, 0x44, 0x44, 0x44, 0x45, 0x45, 0x45, 0x56, 0x46, 0x46, 0x57, 0x57, 0x68,
	0x43, 0x43, 0x33, 0x44, 0x44, 0x44, 0x55, 0x45, 0x45, 0x56, 0x56, 0x46, 0x46, 0x57, 0x68, 0x68,
	0x43, 0x43, 0x43, 0x44, 0x44, 0x55, 0x55, 0x45, 0x45, 0x56, 0x67, 0x46, 0x46, 0x57, 0x68, 0x79,
	0x54, 0x54, 0x54, 0x44, 0x55, 0x55, 0x55, 0x45, 0x56, 0x56, 0x67, 0x78, 0x78, 0x57, 0x68, 0x79,
	0x47, 0x47, 0x47, 0x48, 0x48, 0x48, 0x48, 0x49, 0x49, 0x49, 0x49, 0x4a, 0x4a, 0x4a, 0x5b, 0x79,
	0x47, 0x47, 0x47, 0x48, 0x48, 0x48, 0x59, 0x49, 0x49, 0x49, 0x5a, 0x4a, 0x4a, 0x5b, 0x5b, 0x6c,
	0x58, 0x58, 0x58, 0x48, 0x59, 0x59, 0x59, 0x49, 0x49, 0x5a, 0x5a, 0x4a, 0x4a, 0x5b, 0x6c, 0x6c,
	0x69, 0x69, 0x69, 0x59, 0x59, 0x6a, 0x6a, 0x49, 0x5a, 0x5a, 0x6b, 0x6b, 0x5b, 0x5b, 0x6c, 0x7d,
	0x4b, 0x4b, 0x4b, 0x4c, 0x4c, 0x4c, 0x7b, 0x4d, 0x4d, 0x4d, 0x6b, 0x4e, 0x4e, 0x4e, 0x7d, 0x7d,
	0x4b, 0x4b, 0x4b, 0x4c, 0x4c, 0x4c, 0x7b, 0x4d, 0x4d, 0x4d, 0x5e, 0x4e, 0x4e, 0x4e, 0x5f, 0x7d,
	0x5c, 0x5c, 0x5c, 0x5d, 0x5d, 0x5d, 0x5d, 0x4d, 0x5e, 0x5e, 0x5e, 0x4e, 0x4e, 0x5f, 0x5f, 0x60,
	0x6d, 0x6d, 0x6d, 0x5d, 0x6e, 0x6e, 0x6e, 0x5e, 0x5e, 0x6f, 0x6f, 0x70, 0x5f, 0x5f, 0x60, 0x60,
	0x7e, 0x7e, 0x7e, 0x6e, 0x6e, 0x7f, 0x7f, 0x7f, 0x6f, 0x6f, 0x70, 0x70, 0x70, 0x60, 0x60, 0x71,
	0x50, 0x50, 0x50, 0x40, 0x40, 0x51, 0x51, 0x41, 0x41, 0x52, 0x63, 0x74, 0x42, 0x53, 0x64, 0x75,
	0x50, 0x50, 0x50, 0x40, 0x40, 0x51, 0x51, 0x41, 0x41, 0x52, 0x63, 0x74, 0x42, 0x53, 0x64, 0x75,
	0x50, 0x50, 0x33, 0x33, 0x40, 0x51, 0x51, 0x41, 0x41, 0x52, 0x63, 0x74, 0x42, 0x53, 0x64, 0x75,
	0x43, 0x43, 0x33, 0x44, 0x44, 0x44, 0x55, 0x45, 0x45, 0x56, 0x56, 0x46, 0x46, 0x57, 0x68, 0x68,
	0x43, 0x43, 0x43, 0x44, 0x44, 0x55, 0x55, 0x45, 0x45, 0x56, 0x67, 0x46, 0x46, 0x57, 0x68, 0x79,
	0x54, 0x54, 0x54, 0x44, 0x55, 0x55, 0x55, 0x45, 0x56, 0x56, 0x67, 0x78, 0x78, 0x57, 0x68, 0x79,
	0x54, 0x54, 0x54, 0x55, 0x55, 0x55, 0x66, 0x66, 0x56, 0x67, 0x67, 0x78, 0x78, 0x68, 0x68, 0x79,
	0x47, 0x47, 0x47, 0x48, 0x48, 0x48, 0x66, 0x49, 0x49, 0x49, 0x78, 0x78, 0x4a, 0x4a, 0x5b, 0x79,
	0x47, 0x47, 0x47, 0x48, 0x48, 0x59, 0x59, 0x49, 0x49, 0x5a, 0x5a, 0x4a, 0x4a, 0x5b, 0x6c, 0x6c,
	0x58, 0x58, 0x58, 0x59, 0x59, 0x59, 0x6a, 0x49, 0x5a, 0x5a, 0x6b, 0x6b, 0x5b, 0x5b, 0x6c, 0x7d,
	0x69, 0x69, 0x69, 0x59, 0x6a, 0x6a, 0x6a, 0x7b, 0x5a, 0x6b, 0x6b, 0x6b, 0x7c, 0x6c, 0x6c, 0x7d,
	0x7a, 0x7a, 0x7a, 0x4c, 0x4c, 0x7b, 0x7b, 0x7b, 0x4d, 0x6b, 0x6b, 0x7c, 0x7c, 0x4e, 0x7d, 0x7d,
	0x4b, 0x4b, 0x4b, 0x4c, 0x4c, 0x7b, 0x7b, 0x4d, 0x4d, 0x5e, 0x7c, 0x7c, 0x4e, 0x5f, 0x5f, 0x7d,
	0x5c, 0x5c, 0x5c, 0x5d, 0x5d, 0x5d, 0x6e, 0x4d, 0x5e, 0x5e, 0x6f, 0x4e, 0x5f, 0x5f, 0x60, 0x60,
	0x6d, 0x6d, 0x6d, 0x6e, 0x6e, 0x6e, 0x6e, 0x5e, 0x6f, 0x6f, 0x6f, 0x70, 0x5f, 0x60, 0x60, 0x71,
	0x7e, 0x7e, 0x7e, 0x6e, 0x7f, 0x7f, 0x7f, 0x7f, 0x6f, 0x70, 0x70, 0x70, 0x70, 0x60, 0x71, 0x71,
	0x61, 0x61, 0x61, 0x40, 0x51, 0x51, 0x62, 0x62, 0x73, 0x63, 0x63, 0x74, 0x74, 0x64, 0x64, 0x75,
	0x61, 0x61, 0x61, 0x40, 0x51, 0x51, 0x62, 0x62, 0x73, 0x63, 0x63, 0x74, 0x74, 0x64, 0x64, 0x75,
	0x61, 0x61, 0x61, 0x40, 0x51, 0x51, 0x62, 0x62, 0x73, 0x63, 0x63, 0x74, 0x74, 0x64, 0x64, 0x75,
	0x43, 0x43, 0x43, 0x44, 0x44, 0x55, 0x55, 0x45, 0x45, 0x56, 0x67, 0x46, 0x46, 0x57, 0x68, 0x79,
	0x54, 0x54, 0x54, 0x44, 0x55, 0x55, 0x55, 0x45, 0x56, 0x56, 0x67, 0x78, 0x78, 0x57, 0x68, 0x79,
	0x54, 0x54, 0x54, 0x55, 0x55, 0x55, 0x66, 0x66, 0x56, 0x67, 0x67, 0x78, 0x78, 0x68, 0x68, 0x79,
	0x65, 0x65, 0x65, 0x55, 0x55, 0x66, 0x66, 0x66, 0x77, 0x67, 0x78, 0x78, 0x78, 0x78, 0x79, 0x79,
	0x65, 0x65, 0x65, 0x48, 0x48, 0x66, 0x66, 0x77, 0x77, 0x77, 0x78, 0x78, 0x78, 0x5b, 0x79, 0x79,
	0x76, 0x76, 0x76, 0x48, 0x59, 0x59, 0x77, 0x77, 0x77, 0x5a, 0x5a, 0x4a, 0x4a, 0x5b, 0x6c, 0x6c,
	0x69, 0x69, 0x69, 0x59, 0x59, 0x6a, 0x6a, 0x77, 0x5a, 0x5a, 0x6b, 0x6b, 0x5b, 0x6c, 0x6c, 0x7d,
	0x69, 0x69, 0x69, 0x6a, 0x6a, 0x6a, 0x7b, 0x7b, 0x5a, 0x6b, 0x6b, 0x7c, 0x7c, 0x6c, 0x7d, 0x7d,
	0x7a, 0x7a, 0x7a, 0x4c, 0x7b, 0x7b, 0x7b, 0x7b, 0x4d, 0x6b, 0x7c, 0x7c, 0x7c, 0x7c, 0x7d, 0x7d,
	0x7a, 0x7a, 0x7a, 0x4c, 0x7b, 0x7b, 0x7b, 0x7b, 0x4d, 0x5e, 0x7c, 0x7c, 0x7c, 0x5f, 0x5f, 0x7d,
	0x6d, 0x6d, 0x6d, 0x5d, 0x5d, 0x6e, 0x7b, 0x5e, 0x5e, 0x6f, 0x6f, 0x7c, 0x5f, 0x5f, 0x60, 0x60,
	0x6d, 0x6d, 0x6d, 0x6e, 0x6e, 0x6e, 0x7f, 0x7f, 0x6f, 0x6f, 0x70, 0x70, 0x5f, 0x60, 0x60, 0x71,
	0x7e, 0x7e, 0x7e, 0x7f, 0x7f, 0x7f, 0x7f, 0x7f, 0x6f, 0x70, 0x70, 0x70, 0x70, 0x60, 0x71, 0x71,
	0x72, 0x72, 0x72, 0x8f, 0x8f, 0x62, 0x62, 0x73, 0x73, 0x80, 0x74, 0x81, 0x81, 0x81, 0x92, 0x75,
	0x72, 0x72, 0x72, 0x8f, 0x8f, 0x62, 0x62, 0x73, 0x73, 0x80, 0x74, 0x81, 0x81, 0x81, 0x92, 0x75,
	0x72, 0x72, 0x72, 0x83, 0x83, 0x62, 0x62, 0x73, 0x73, 0x80, 0x74, 0x81, 0x81, 0x81, 0x92, 0x75,
	0x82, 0x82, 0x82, 0x83, 0x83, 0x83, 0x83, 0x84, 0x84, 0x84, 0x84, 0x85, 0x85, 0x85, 0x96, 0x79,
	0x82, 0x82, 0x82, 0x83, 0x83, 0x83, 0x66, 0x84, 0x84, 0x84, 0x67, 0x85, 0x85, 0x85, 0x96, 0x79,
	0x65, 0x65, 0x65, 0x83, 0x83, 0x66, 0x66, 0x66, 0x84, 0x84, 0x78, 0x78, 0x85, 0x85, 0x96, 0x79,
	0x65, 0x65, 0x65, 0x83, 0x66, 0x66, 0x66, 0x77, 0x77, 0x77, 0x78, 0x78, 0x78, 0x96, 0x79, 0x79,
	0x76, 0x76, 0x76, 0x87, 0x87, 0x66, 0x77, 0x77, 0x77, 0x88, 0x78, 0x89, 0x89, 0x89, 0x89, 0x79,
	0x76, 0x76, 0x76, 0x87, 0x87, 0x87, 0x77, 0x77, 0x88, 0x88, 0x88, 0x89, 0x89, 0x89, 0x9a, 0x9a,
	0x86, 0x86, 0x86, 0x87, 0x87, 0x87, 0x77, 0x88, 0x88, 0x88, 0x6b, 0x89, 0x89, 0x9a, 0x9a, 0x7d,
	0x7a, 0x7a, 0x7a, 0x87, 0x6a, 0x7b, 0x7b, 0x7b, 0x88, 0x6b, 0x6b, 0x7c, 0x7c, 0x9a, 0x7d, 0x7d,
	0x8a, 0x8a, 0x8a, 0x8b, 0x8b, 0x7b, 0x7b, 0x8c, 0x8c, 0x8c, 0x7c, 0x7c, 0x8d, 0x8d, 0x7d, 0x7d,
	0x8a, 0x8a, 0x8a, 0x8b, 0x8b, 0x8b, 0x7b, 0x8c, 0x8c, 0x8c, 0x7c, 0x8d, 0x8d, 0x8d, 0x9e, 0x9e,
	0x8a, 0x8a, 0x8a, 0x8b, 0x8b, 0x8b, 0x9c, 0x8c, 0x8c, 0x9d, 0x9d, 0x8d, 0x8d, 0x9e, 0x9e, 0x9e,
	0x9b, 0x9b, 0x9b, 0x9c, 0x9c, 0x9c, 0x7f, 0x8c, 0x9d, 0x9d, 0x70, 0x70, 0x9e, 0x9e, 0x9e, 0x71,
	0x7e, 0x7e, 0x7e, 0x7f, 0x7f, 0x7f, 0x7f, 0x7f, 0x9d, 0x70, 0x70, 0x70, 0x9e, 0x9e, 0x71, 0x71,
	0x8e, 0x8e, 0x8e, 0x8f, 0x8f, 0x8f, 0x73, 0x73, 0x80, 0x80, 0x91, 0x81, 0x81, 0x92, 0x92, 0xa3,
	0x8e, 0x8e, 0x8e, 0x8f, 0x8f, 0x8f, 0x73, 0x73, 0x80, 0x80, 0x91, 0x81, 0x81, 0x92, 0x92, 0xa3,
	0x82, 0x82, 0x82, 0x83, 0x83, 0x83, 0x73, 0x73, 0x80, 0x80, 0x91, 0x81, 0x81, 0x92, 0x92, 0xa3,
	0x82, 0x82, 0x82, 0x83, 0x83, 0x83, 0x83, 0x84, 0x84, 0x84, 0x95, 0x85, 0x85, 0x85, 0x96, 0xa7,
	0x82, 0x82, 0x82, 0x83, 0x83, 0x83, 0x94, 0x84, 0x84, 0x84, 0x95, 0x85, 0x85, 0x96, 0x96, 0xa7,
	0x82, 0x82, 0x82, 0x83, 0x83, 0x94, 0x94, 0x84, 0x84, 0x95, 0x95, 0x85, 0x85, 0x96, 0xa7, 0xa7,
	0x76, 0x76, 0x76, 0x83, 0x94, 0x94, 0x77, 0x77, 0x77, 0x95, 0x95, 0x85, 0x85, 0x96, 0xa7, 0xa7,
	0x76, 0x76, 0x76, 0x87, 0x87, 0x87, 0x77, 0x77, 0x88, 0x88, 0x88, 0x89, 0x89, 0x89, 0x9a, 0x9a,
	0x86, 0x86, 0x86, 0x87, 0x87, 0x87, 0x77, 0x88, 0x88, 0x88, 0x99, 0x89, 0x89, 0x9a, 0x9a, 0xab,
	0x86, 0x86, 0x86, 0x87, 0x87, 0x98, 0x98, 0x88, 0x88, 0x99, 0x99, 0x89, 0x89, 0x9a, 0x9a, 0xab,
	0x97, 0x97, 0x97, 0x98, 0x98, 0x98, 0x98, 0x88, 0x99, 0x99, 0x99, 0x89, 0x9a, 0x9a, 0xab, 0xab,
	0x8a, 0x8a, 0x8a, 0x8b, 0x8b, 0x8b, 0x8b, 0x8c, 0x8c, 0x8c, 0x8c, 0x8d, 0x8d, 0x8d, 0xab, 0xbc,
	0x8a, 0x8a, 0x8a, 0x8b, 0x8b, 0x8b, 0x8b, 0x8c, 0x8c, 0x8c, 0x9d, 0x8d, 0x8d, 0x8d, 0x9e, 0x9e,
	0x9b, 0x9b, 0x9b, 0x8b, 0x9c, 0x9c, 0x9c, 0x8c, 0x9d, 0x9d, 0x9d, 0x8d, 0x8d, 0x9e, 0x9e, 0xaf,
	0x9b, 0x9b, 0x9b, 0x9c, 0x9c, 0xad, 0xad, 0x9d, 0x9d, 0x9d, 0xae, 0xae, 0x9e, 0x9e, 0xaf, 0xaf,
	0xac, 0xac, 0xac, 0xad, 0xad, 0xad, 0xad, 0x9d, 0xae, 0xae, 0xae, 0xbf, 0x9e, 0xaf, 0xaf, 0xaf,
	0x9f, 0x9f, 0x9f, 0x8f, 0x90, 0x90, 0xa1, 0x80, 0x80, 0x91, 0x91, 0x81, 0x81, 0x92, 0xa3, 0xb4,
	0x9f, 0x9f, 0x9f, 0x8f, 0x90, 0x90, 0xa1, 0x80, 0x80, 0x91, 0x91, 0x81, 0x81, 0x92, 0xa3, 0xb4,
	0x9f, 0x9f, 0x9f, 0x83, 0x90, 0x90, 0xa1, 0x80, 0x80, 0x91, 0x91, 0x81, 0x81, 0x92, 0xa3, 0xb4,
	0x82, 0x82, 0x82, 0x83, 0x83, 0x94, 0x94, 0x84, 0x84, 0x95, 0x95, 0x85, 0x85, 0x96, 0x96, 0xa7,
	0x93, 0x93, 0x93, 0x83, 0x94, 0x94, 0x94, 0x84, 0x84, 0x95, 0x95, 0x85, 0x85, 0x96, 0xa7, 0xa7,
	0x93, 0x93, 0x93, 0x94, 0x94, 0x94, 0xa5, 0x84, 0x95, 0x95, 0xa6, 0xa6, 0x96, 0x96, 0xa7, 0xb8,
	0xa4, 0xa4, 0xa4, 0x94, 0x94, 0xa5, 0xa5, 0x77, 0x95, 0x95, 0xa6, 0xa6, 0x96, 0xa7, 0xa7, 0xb8,
	0x86, 0x86, 0x86, 0x87, 0x87, 0x87, 0x77, 0x88, 0x88, 0x88, 0x99, 0x89, 0x89, 0x9a, 0x9a, 0xb8,
	0x86, 0x86, 0x86, 0x87, 0x87, 0x98, 0x98, 0x88, 0x88, 0x99, 0x99, 0x89, 0x89, 0x9a, 0x9a, 0xab,
	0x97, 0x97, 0x97, 0x98, 0x98, 0x98, 0x98, 0x88, 0x99, 0x99, 0x99, 0x89, 0x9a, 0x9a, 0xab, 0xab,
	0x97, 0x97, 0x97, 0x98, 0x98, 0xa9, 0xa9, 0x99, 0x99, 0x99, 0xaa, 0xaa, 0x9a, 0xab, 0xab, 0xbc,
	0x8a, 0x8a, 0x8a, 0x8b, 0x8b, 0xa9, 0xa9, 0x8c, 0x8c, 0x8c, 0xaa, 0x8d, 0x8d, 0x8d, 0xab, 0xbc,
	0x8a, 0x8a, 0x8a, 0x8b, 0x8b, 0x9c, 0x9c, 0x8c, 0x8c, 0x9d, 0x9d, 0x8d, 0x8d, 0x9e, 0x9e, 0xbc,
	0x9b, 0x9b, 0x9b, 0x9c, 0x9c, 0x9c, 0xad, 0x9d, 0x9d, 0x9d, 0xae, 0x8d, 0x9e, 0x9e, 0xaf, 0xaf,
	0xac, 0xac, 0xac, 0x9c, 0xad, 0xad, 0xad, 0x9d, 0x9d, 0xae, 0xae, 0xae, 0x9e, 0xaf, 0xaf, 0xaf,
	0xbd, 0xbd, 0xbd, 0xad, 0xad, 0xbe, 0xbe, 0xbe, 0xae, 0xae, 0xbf, 0xbf, 0xbf, 0xaf, 0xaf, 0xb0,
	0xa0, 0xa0, 0xa0, 0x90, 0xa1, 0xa1, 0xa1, 0xb2, 0x91, 0x91, 0xa2, 0xa2, 0xb3, 0xa3, 0xa3, 0xb4,
	0xa0, 0xa0, 0xa0, 0x90, 0xa1, 0xa1, 0xa1, 0xb2, 0x91, 0x91, 0xa2, 0xa2, 0xb3, 0xa3, 0xa3, 0xb4,
	0xa0, 0xa0, 0xa0, 0x90, 0xa1, 0xa1, 0xa1, 0xb2, 0x91, 0x91, 0xa2, 0xa2, 0xb3, 0xa3, 0xa3, 0xb4,
	0x93, 0x93, 0x93, 0x94, 0x94, 0x94, 0xa5, 0x84, 0x95, 0x95, 0xa6, 0xa6, 0x96, 0x96, 0xa7, 0xb8,
	0xa4, 0xa4, 0xa4, 0x94, 0x94, 0xa5, 0xa5, 0x84, 0x95, 0x95, 0xa6, 0xa6, 0x96, 0x96, 0xa7, 0xb8,
	0xa4, 0xa4, 0xa4, 0x94, 0xa5, 0xa5, 0xa5, 0xb6, 0x95, 0xa6, 0xa6, 0xa6, 0xb7, 0xa7, 0xa7, 0xb8,
	0xa4, 0xa4, 0xa4, 0xa5, 0xa5, 0xa5, 0xb6, 0xb6, 0x95, 0xa6, 0xa6, 0xb7, 0xb7, 0xa7, 0xb8, 0xb8,
	0xb5, 0xb5, 0xb5, 0x87, 0x87, 0xb6, 0xb6, 0xb6, 0x88, 0x99, 0xa6, 0xb7, 0xb7, 0x9a, 0xb8, 0xb8,
	0x97, 0x97, 0x97, 0x98, 0x98, 0x98, 0x98, 0x88, 0x99, 0x99, 0x99, 0x89, 0x9a, 0x9a, 0xab, 0xab,
	0x97, 0x97, 0x97, 0x98, 0x98, 0xa9, 0xa9, 0x99, 0x99, 0x99, 0xaa, 0xaa, 0x9a, 0xab, 0xab, 0xbc,
	0xa8, 0xa8, 0xa8, 0xa9, 0xa9, 0xa9, 0xa9, 0xa9, 0x99, 0xaa, 0xaa, 0xaa, 0xbb, 0xab, 0xab, 0xbc,
	0xa8, 0xa8, 0xa8, 0xa9, 0xa9, 0xa9, 0xba, 0xba, 0x8c, 0xaa, 0xaa, 0xbb, 0xbb, 0xab, 0xbc, 0xbc,
	0xb9, 0xb9, 0xb9, 0x9c, 0x9c, 0xba, 0xba, 0xba, 0x9d, 0x9d, 0xbb, 0xbb, 0xbb, 0x9e, 0x9e, 0xbc,
	0xac, 0xac, 0xac, 0x9c, 0x9c, 0xad, 0xad, 0x9d, 0x9d, 0xae, 0xae, 0xae, 0x9e, 0x9e, 0xaf, 0xaf,
	0xac, 0xac, 0xac, 0xad, 0xad, 0xad, 0xbe, 0xbe, 0xae, 0xae, 0xae, 0xbf, 0x9e, 0xaf, 0xaf, 0xb0,
	0xbd, 0xbd, 0xbd, 0xbe, 0xbe, 0xbe, 0xbe, 0xbe, 0xae, 0xbf, 0xbf, 0xbf, 0xbf, 0xaf, 0xb0, 0xb0,
	0xb1, 0xb1, 0xb1, 0xce, 0xce, 0xb2, 0xb2, 0xcf, 0xcf, 0xa2, 0xa2, 0xb3, 0xb3, 0xc0, 0xb4, 0xb4,
	0xb1, 0xb1, 0xb1, 0xce, 0xce, 0xb2, 0xb2, 0xcf, 0xcf, 0xa2, 0xa2, 0xb3, 0xb3, 0xc0, 0xb4, 0xb4,
	0xb1, 0xb1, 0xb1, 0xc2, 0xc2, 0xb2, 0xb2, 0xc3, 0xc3, 0xa2, 0xa2, 0xb3, 0xb3, 0xc0, 0xb4, 0xb4,
	0xc1, 0xc1, 0xc1, 0xc2, 0xc2, 0xc2, 0xa5, 0xc3, 0xc3, 0xc3, 0xa6, 0xc4, 0xc4, 0xc4, 0xa7, 0xb8,
	0xc1, 0xc1, 0xc1, 0xc2, 0xc2, 0xa5, 0xb6, 0xc3, 0xc3, 0xc3, 0xa6, 0xc4, 0xc4, 0xc4, 0xb8, 0xb8,
	0xb5, 0xb5, 0xb5, 0xc2, 0xa5, 0xb6, 0xb6, 0xb6, 0xc3, 0xa6, 0xa6, 0xb7, 0xb7, 0xc4, 0xb8, 0xb8,
	0xb5, 0xb5, 0xb5, 0xa5, 0xb6, 0xb6, 0xb6, 0xb6, 0xc3, 0xa6, 0xb7, 0xb7, 0xb7, 0xb7, 0xb8, 0xb8,
	0xc5, 0xc5, 0xc5, 0xc6, 0xc6, 0xb6, 0xb6, 0xc7, 0xc7, 0xc7, 0xb7, 0xb7, 0xc8, 0xc8, 0xb8, 0xb8,
	0xc5, 0xc5, 0xc5, 0xc6, 0xc6, 0xc6, 0xc6, 0xc7, 0xc7, 0xc7, 0xaa, 0xc8, 0xc8, 0xc8, 0xab, 0xbc,
	0xa8, 0xa8, 0xa8, 0xc6, 0xc6, 0xa9, 0xa9, 0xc7, 0xc7, 0xaa, 0xaa, 0xaa, 0xc8, 0xc8, 0xab, 0xbc,
	0xa8, 0xa8, 0xa8, 0xa9, 0xa9, 0xa9, 0xba, 0xba, 0xaa, 0xaa, 0xaa, 0xbb, 0xbb, 0xab, 0xbc, 0xbc,
	0xb9, 0xb9, 0xb9, 0xca, 0xca, 0xba, 0xba, 0xba, 0xcb, 0xaa, 0xbb, 0xbb, 0xbb, 0xcc, 0xbc, 0xbc,
	0xb9, 0xb9, 0xb9, 0xca, 0xca, 0xba, 0xba, 0xcb, 0xcb, 0xcb, 0xbb, 0xbb, 0xcc, 0xcc, 0xcc, 0xbc,
	0xc9, 0xc9, 0xc9, 0xca, 0xca, 0xca, 0xba, 0xcb, 0xcb, 0xcb, 0xae, 0xcc, 0xcc, 0xcc, 0xaf, 0xaf,
	0xbd, 0xbd, 0xbd, 0xad, 0xbe, 0xbe, 0xbe, 0xbe, 0xae, 0xae, 0xbf, 0xbf, 0xcc, 0xaf, 0xaf, 0xb0,
	0xbd, 0xbd, 0xbd, 0xbe, 0xbe, 0xbe, 0xbe, 0xbe, 0xbf, 0xbf, 0xbf, 0xbf, 0xbf, 0xaf, 0xb0, 0xb0,
	0xcd, 0xcd, 0xcd, 0xce, 0xce, 0xce, 0xb2, 0xcf, 0xcf, 0xcf, 0xb3, 0xb3, 0xc0, 0xc0, 0xd1, 0xb4,
	0xcd, 0xcd, 0xcd, 0xce, 0xce, 0xce, 0xb2, 0xcf, 0xcf, 0xcf, 0xb3, 0xb3, 0xc0, 0xc0, 0xd1, 0xb4,
	0xc1, 0xc1, 0xc1, 0xc2, 0xc2, 0xc2, 0xb2, 0xc3, 0xc3, 0xc3, 0xb3, 0xb3, 0xc0, 0xc0, 0xd1, 0xb4,
	0xc1, 0xc1, 0xc1, 0xc2, 0xc2, 0xc2, 0xc2, 0xc3, 0xc3, 0xc3, 0xd4, 0xc4, 0xc4, 0xc4, 0xd5, 0xd5,
	0xc1, 0xc1, 0xc1, 0xc2, 0xc2, 0xc2, 0xb6, 0xc3, 0xc3, 0xc3, 0xd4, 0xc4, 0xc4, 0xc4, 0xd5, 0xb8,
	0xc1, 0xc1, 0xc1, 0xc2, 0xc2, 0xb6, 0xb6, 0xc3, 0xc3, 0xd4, 0xb7, 0xb7, 0xc4, 0xd5, 0xd5, 0xb8,
	0xb5, 0xb5, 0xb5, 0xc2, 0xb6, 0xb6, 0xb6, 0xb6, 0xc3, 0xd4, 0xb7, 0xb7, 0xb7, 0xd5, 0xd5, 0xb8,
	0xc5, 0xc5, 0xc5, 0xc6, 0xc6, 0xc6, 0xb6, 0xc7, 0xc7, 0xc7, 0xb7, 0xc8, 0xc8, 0xc8, 0xd9, 0xd9,
	0xc5, 0xc5, 0xc5, 0xc6, 0xc6, 0xc6, 0xc6, 0xc7, 0xc7, 0xc7, 0xd8, 0xc8, 0xc8, 0xc8, 0xd9, 0xd9,
	0xc5, 0xc5, 0xc5, 0xc6, 0xc6, 0xd7, 0xd7, 0xc7, 0xc7, 0xd8, 0xd8, 0xc8, 0xc8, 0xd9, 0xd9, 0xbc,
	0xb9, 0xb9, 0xb9, 0xd7, 0xd7, 0xba, 0xba, 0xba, 0xd8, 0xd8, 0xbb, 0xbb, 0xbb, 0xd9, 0xd9, 0xbc,
	0xb9, 0xb9, 0xb9, 0xca, 0xca, 0xba, 0xba, 0xcb, 0xcb, 0xcb, 0xbb, 0xbb, 0xcc, 0xcc, 0xcc, 0xbc,
	0xc9, 0xc9, 0xc9, 0xca, 0xca, 0xca, 0xba, 0xcb, 0xcb, 0xcb, 0xbb, 0xcc, 0xcc, 0xcc, 0xdd, 0xdd,
	0xc9, 0xc9, 0xc9, 0xca, 0xca, 0xdb, 0xdb, 0xcb, 0xcb, 0xdc, 0xdc, 0xcc, 0xcc, 0xdd, 0xdd, 0xdd,
	0xda, 0xda, 0xda, 0xdb, 0xdb, 0xdb, 0xdb, 0xdc, 0xdc, 0xdc, 0xdc, 0xcc, 0xdd, 0xdd, 0xdd, 0xb0,
	0xbd, 0xbd, 0xbd, 0xdb, 0xbe, 0xbe, 0xbe, 0xdc, 0xdc, 0xbf, 0xbf, 0xbf, 0xdd, 0xdd, 0xb0, 0xb0,
	0xde, 0xde, 0xde, 0xdf, 0xdf, 0xdf, 0xe0, 0xcf, 0xd0, 0xd0, 0xe1, 0xc0, 0xc0, 0xd1, 0xd1, 0xe2,
	0xde, 0xde, 0xde, 0xdf, 0xdf, 0xdf, 0xe0, 0xcf, 0xd0, 0xd0, 0xe1, 0xc0, 0xc0, 0xd1, 0xd1, 0xe2,
	0xde, 0xde, 0xde, 0xdf, 0xdf, 0xdf, 0xe0, 0xc3, 0xd0, 0xd0, 0xe1, 0xc0, 0xc0, 0xd1, 0xd1, 0xe2,
	0xd2, 0xd2, 0xd2, 0xc2, 0xd3, 0xd3, 0xd3, 0xc3, 0xc3, 0xd4, 0xd4, 0xc4, 0xc4, 0xd5, 0xd5, 0xe6,
	0xd2, 0xd2, 0xd2, 0xd3, 0xd3, 0xd3, 0xd3, 0xc3, 0xd4, 0xd4, 0xd4, 0xc4, 0xc4, 0xd5, 0xd5, 0xe6,
	0xd2, 0xd2, 0xd2, 0xd3, 0xd3, 0xd3, 0xe4, 0xc3, 0xd4, 0xd4, 0xe5, 0xc4, 0xd5, 0xd5, 0xe6, 0xe6,
	0xe3, 0xe3, 0xe3, 0xd3, 0xd3, 0xe4, 0xb6, 0xd4, 0xd4, 0xe5, 0xe5, 0xb7, 0xd5, 0xd5, 0xe6, 0xe6,
	0xc5, 0xc5, 0xc5, 0xc6, 0xc6, 0xc6, 0xd7, 0xc7, 0xc7, 0xd8, 0xd8, 0xc8, 0xc8, 0xd9, 0xd9, 0xd9,
	0xd6, 0xd6, 0xd6, 0xc6, 0xd7, 0xd7, 0xd7, 0xc7, 0xd8, 0xd8, 0xd8, 0xc8, 0xc8, 0xd9, 0xd9, 0xea,
	0xd6, 0xd6, 0xd6, 0xd7, 0xd7, 0xd7, 0xe8, 0xd8, 0xd8, 0xd8, 0xe9, 0xc8, 0xd9, 0xd9, 0xea, 0xea,
	0xe7, 0xe7, 0xe7, 0xd7, 0xd7, 0xe8, 0xe8, 0xd8, 0xd8, 0xe9, 0xe9, 0xe9, 0xd9, 0xd9, 0xea, 0xea,
	0xc9, 0xc9, 0xc9, 0xca, 0xca, 0xca, 0xba, 0xcb, 0xcb, 0xcb, 0xe9, 0xcc, 0xcc, 0xcc, 0xea, 0xea,
	0xc9, 0xc9, 0xc9, 0xca, 0xca, 0xdb, 0xdb, 0xcb, 0xcb, 0xdc, 0xdc, 0xcc, 0xcc, 0xdd, 0xdd, 0xdd,
	0xda, 0xda, 0xda, 0xdb, 0xdb, 0xdb, 0xdb, 0xdc, 0xdc, 0xdc, 0xdc, 0xcc, 0xdd, 0xdd, 0xdd, 0xee,
	0xda, 0xda, 0xda, 0xdb, 0xdb, 0xec, 0xec, 0xdc, 0xdc, 0xed, 0xed, 0xed, 0xdd, 0xdd, 0xee, 0xee,
	0xeb, 0xeb, 0xeb, 0xec, 0xec, 0xec, 0xec, 0xdc, 0xed, 0xed, 0xed, 0xed, 0xdd, 0xee, 0xee, 0xee,
	0xef, 0xef, 0xef, 0xdf, 0xe0, 0xe0, 0xe0, 0xd0, 0xd0, 0xe1, 0xe1, 0xf2, 0xd1, 0xd1, 0xe2, 0xe2,
	0xef, 0xef, 0xef, 0xdf, 0xe0, 0xe0, 0xe0, 0xd0, 0xd0, 0xe1, 0xe1, 0xf2, 0xd1, 0xd1, 0xe2, 0xe2,
	0xef, 0xef, 0xef, 0xdf, 0xe0, 0xe0, 0xe0, 0xd0, 0xd0, 0xe1, 0xe1, 0xf2, 0xd1, 0xd1, 0xe2, 0xe2,
	0xd2, 0xd2, 0xd2, 0xd3, 0xd3, 0xe4, 0xe4, 0xd4, 0xd4, 0xd4, 0xe5, 0xe5, 0xd5, 0xd5, 0xe6, 0xe6,
	0xe3, 0xe3, 0xe3, 0xd3, 0xe4, 0xe4, 0xe4, 0xd4, 0xd4, 0xe5, 0xe5, 0xf6, 0xd5, 0xd5, 0xe6, 0xe6,
	0xe3, 0xe3, 0xe3, 0xe4, 0xe4, 0xe4, 0xe4, 0xd4, 0xe5, 0xe5, 0xe5, 0xf6, 0xd5, 0xe6, 0xe6, 0xf7,
	0xe3, 0xe3, 0xe3, 0xe4, 0xe4, 0xe4, 0xf5, 0xf5, 0xe5, 0xe5, 0xf6, 0xf6, 0xd5, 0xe6, 0xe6, 0xf7,
	0xd6, 0xd6, 0xd6, 0xd7, 0xd7, 0xd7, 0xf5, 0xc7, 0xd8, 0xd8, 0xf6, 0xc8, 0xd9, 0xd9, 0xd9, 0xf7,
	0xd6, 0xd6, 0xd6, 0xd7, 0xd7, 0xe8, 0xe8, 0xd8, 0xd8, 0xd8, 0xe9, 0xe9, 0xd9, 0xd9, 0xea, 0xea,
	0xe7, 0xe7, 0xe7, 0xd7, 0xe8, 0xe8, 0xe8, 0xd8, 0xd8, 0xe9, 0xe9, 0xe9, 0xd9, 0xea, 0xea, 0xea,
	0xe7, 0xe7, 0xe7, 0xe8, 0xe8, 0xe8, 0xf9, 0xf9, 0xe9, 0xe9, 0xe9, 0xfa, 0xd9, 0xea, 0xea, 0xfb,
	0xf8, 0xf8, 0xf8, 0xe8, 0xf9, 0xf9, 0xf9, 0xcb, 0xe9, 0xe9, 0xfa, 0xfa, 0xcc, 0xea, 0xea, 0xfb,
	0xda, 0xda, 0xda, 0xdb, 0xdb, 0xdb, 0xdb, 0xdc, 0xdc, 0xdc, 0xdc, 0xcc, 0xdd, 0xdd, 0xdd, 0xee,
	0xda, 0xda, 0xda, 0xdb, 0xdb, 0xec, 0xec, 0xdc, 0xdc, 0xed, 0xed, 0xed, 0xdd, 0xdd, 0xee, 0xee,
	0xeb, 0xeb, 0xeb, 0xec, 0xec, 0xec, 0xec, 0xdc, 0xed, 0xed, 0xed, 0xed, 0xdd, 0xee, 0xee, 0xee,
	0xeb, 0xeb, 0xeb, 0xec, 0xec, 0xfd, 0xfd, 0xfd, 0xed, 0xed, 0xfe, 0xfe, 0xee, 0xee, 0xee, 0xff,
	0xf0, 0xf0, 0xf0, 0xe0, 0xf1, 0xf1, 0xf1, 0xf1, 0xe1, 0xf2, 0xf2, 0xf2, 0xf2, 0xe2, 0xe2, 0xf3,
	0xf0, 0xf0, 0xf0, 0xe0, 0xf1, 0xf1, 0xf1, 0xf1, 0xe1, 0xf2, 0xf2, 0xf2, 0xf2, 0xe2, 0xe2, 0xf3,
	0xf0, 0xf0, 0xf0, 0xe0, 0xf1, 0xf1, 0xf1, 0xf1, 0xe1, 0xf2, 0xf2, 0xf2, 0xf2, 0xe2, 0xe2, 0xf3,
	0xe3, 0xe3, 0xe3, 0xe4, 0xe4, 0xe4, 0xf5, 0xf5, 0xe5, 0xe5, 0xf6, 0xf6, 0xd5, 0xe6, 0xe6, 0xf7,
	0xf4, 0xf4, 0xf4, 0xe4, 0xe4, 0xf5, 0xf5, 0xf5, 0xe5, 0xe5, 0xf6, 0xf6, 0xf6, 0xe6, 0xe6, 0xf7,
	0xf4, 0xf4, 0xf4, 0xe4, 0xf5, 0xf5, 0xf5, 0xf5, 0xe5, 0xf6, 0xf6, 0xf6, 0xf6, 0xe6, 0xf7, 0xf7,
	0xf4, 0xf4, 0xf4, 0xf5, 0xf5, 0xf5, 0xf5, 0xf5, 0xe5, 0xf6, 0xf6, 0xf6, 0xf6, 0xe6, 0xf7, 0xf7,
	0xf4, 0xf4, 0xf4, 0xf5, 0xf5, 0xf5, 0xf5, 0xf5, 0xd8, 0xf6, 0xf6, 0xf6, 0xd9, 0xd9, 0xf7, 0xf7,
	0xe7, 0xe7, 0xe7, 0xe8, 0xe8, 0xe8, 0xe8, 0xd8, 0xe9, 0xe9, 0xe9, 0xfa, 0xd9, 0xea, 0xea, 0xea,
	0xf8, 0xf8, 0xf8, 0xe8, 0xe8, 0xf9, 0xf9, 0xf9, 0xe9, 0xe9, 0xfa, 0xfa, 0xfa, 0xea, 0xea, 0xfb,
	0xf8, 0xf8, 0xf8, 0xf9, 0xf9, 0xf9, 0xf9, 0xf9, 0xe9, 0xfa, 0xfa, 0xfa, 0xfa, 0xea, 0xfb, 0xfb,
	0xf8, 0xf8, 0xf8, 0xf9, 0xf9, 0xf9, 0xf9, 0xf9, 0xfa, 0xfa, 0xfa, 0xfa, 0xfa, 0xea, 0xfb, 0xfb,
	0xf8, 0xf8, 0xf8, 0xdb, 0xf9, 0xf9, 0xf9, 0xdc, 0xdc, 0xfa, 0xfa, 0xfa, 0xdd, 0xdd, 0xee, 0xfb,
	0xeb, 0xeb, 0xeb, 0xec, 0xec, 0xec, 0xec, 0xdc, 0xed, 0xed, 0xed, 0xed, 0xdd, 0xee, 0xee, 0xee,
	0xeb, 0xeb, 0xeb, 0xec, 0xec, 0xfd, 0xfd, 0xfd, 0xed, 0xed, 0xfe, 0xfe, 0xee, 0xee, 0xee, 0xff,
	0xfc, 0xfc, 0xfc, 0xfd, 0xfd, 0xfd, 0xfd, 0xfd, 0xed, 0xfe, 0xfe, 0xfe, 0xfe, 0xee, 0xff, 0xff,
}
