// Unless explicitly stated otherwise all files in this repository are licensed
// under the Apache License Version 2.0.
// This product includes software developed at Datadog (https://www.datadoghq.com/).
// Copyright 2023-present Datadog, Inc.

package apisec

import (
	"context"
	"fmt"
	"math/rand"
	"net/http"
	"runtime"
	"sync"
	"sync/atomic"
	"testing"
	"time"

	"github.com/DataDog/appsec-internal-go/apisec/internal/config"
	"github.com/DataDog/appsec-internal-go/apisec/internal/timed/testutil"
	"github.com/stretchr/testify/assert"
)

func TestSampler(t *testing.T) {
	initTestVector()

	ctx := context.Background()
	if deadline, ok := t.Deadline(); ok {
		var cancel context.CancelFunc
		ctx, cancel = context.WithDeadline(ctx, deadline)
		defer cancel()
	}

	const samplesToTake = config.MaxItemCount << 10

	type testCase struct {
		// KeySpace is the set of keys to randomly draw from when sampling.
		KeySpace []SamplingKey
		// SimulatedTPS is the number of samples per second to simulate.
		SimulatedTPS int
		// ExpectedKeepRate is the expected rate of samples being kept [0;1]
		ExpectedKeepRate float64
		// AllowedDelta is the allowed delta for the keep rate.
		AllowedDelta float64
	}
	testCases := map[string]testCase{
		"small": {
			KeySpace:         testVector[:config.MaxItemCount/32],
			SimulatedTPS:     10,
			ExpectedKeepRate: .0415,
			AllowedDelta:     .0001,
		},
		"medium": {
			KeySpace:         testVector[:config.MaxItemCount/16],
			SimulatedTPS:     20,
			ExpectedKeepRate: .0415,
			AllowedDelta:     .0001,
		},
		"high": {
			KeySpace:         testVector[:config.MaxItemCount*2/3],
			SimulatedTPS:     1000,
			ExpectedKeepRate: .0091,
			AllowedDelta:     .0001,
		},
		"large": {
			KeySpace:         testVector[:config.MaxItemCount],
			SimulatedTPS:     1000,
			ExpectedKeepRate: .01,
			AllowedDelta:     .005, // Small chance of collision here... so a bit more wiggle room...
		},
		"extreme": { // Not actually realistic usage... Evictions galore!
			KeySpace:         testVector[:2*config.MaxItemCount],
			SimulatedTPS:     10000,
			ExpectedKeepRate: .21,
			AllowedDelta:     .05, // Very random evictions, so keep rate can be pretty off...
		},
	}

	for name, tc := range testCases {
		t.Run(name, func(t *testing.T) {
			assert.Positive(t, tc.SimulatedTPS)
			assert.GreaterOrEqual(t, tc.ExpectedKeepRate, 0.0)
			assert.GreaterOrEqual(t, 1.0, tc.ExpectedKeepRate)

			const goroutineCount = 10 // Fixed to ensure consistent keep rate...
			ctx, cancel := context.WithCancel(ctx)
			clock := testutil.NewFakeClock(ctx, t, goroutineCount)
			defer func() {
				cancel()
				clock.WaitForTick()
			}()

			var (
				subject = newSampler(30*time.Second, clock.Unix)
				sb      sync.WaitGroup // Start barrier
				wg      sync.WaitGroup // Completion barrier
				kept    atomic.Uint64
				dropped atomic.Uint64
			)
			sb.Add(1 + goroutineCount) // All child goroutines + this one...
			wg.Add(goroutineCount)
			for range goroutineCount {
				go func() {
					defer wg.Done()

					sb.Done() // We're ready for business, signal to the start barrier...
					sb.Wait() // Wait for all the goroutines to have started...

					for i := range samplesToTake {
						if subject.DecisionFor(randomOne(tc.KeySpace)) {
							kept.Add(1)
						} else {
							dropped.Add(1)
						}
						// The clock ticks every 10 samples (model a server processing 10 TPS)
						if i%tc.SimulatedTPS == 0 {
							clock.WaitForTick()
						}
					}
				}()
			}

			sb.Done()
			wg.Wait()

			samplesTaken := kept.Load() + dropped.Load()

			keepRate := float64(kept.Load()) / float64(samplesTaken)
			assert.InDelta(t, tc.ExpectedKeepRate, keepRate, tc.AllowedDelta, "expected keep rate around %.2f±%.2f%%, was %.2f%%", 100*tc.ExpectedKeepRate, 100*tc.AllowedDelta, 100*keepRate)
		})
	}
}

func TestSamplingKeyHash(t *testing.T) {
	initTestVector()

	t.Run("collisions-are-infrequent", func(t *testing.T) {
		hashes := make(map[uint64]struct{}, len(testVector))
		for _, key := range testVector {
			hashes[key.hash()] = struct{}{}
		}

		// Validates that the hash function results in less than 1% collisions. This
		// is a little arbitrary, but should ensure the function as at least
		// somewhat fit for purpose here.
		assert.GreaterOrEqual(t, float64(len(hashes))/float64(len(testVector)), .99,
			"the hash function should have less than 10% collisions")
	})

	t.Run("distribution-is-uniform-on-buckets", func(t *testing.T) {
		buckets := make(map[int]struct{}, len(testVector))
		for _, key := range testVector {
			buckets[int(key.hash()%config.MaxItemCount)] = struct{}{}
		}

		// Validates that the hash function results in hitting at least 80% of the
		// buckets. This is arbitrary, and may feel a little low, but our sample
		// count is quite close to the bucket count, so this is actually solid.
		assert.GreaterOrEqual(t, float64(len(buckets))/float64(config.MaxItemCount), .80,
			"the hash function should slot into 50% of the buckets at least")
	})
}

func BenchmarkSampler(b *testing.B) {
	initTestVector()

	for _, keySpaceSize := range []int{config.MaxItemCount / 2, config.MaxItemCount * 2} {
		keySpace := testVector[:keySpaceSize]
		b.Run(fmt.Sprintf("keySpaceSize=%d", keySpaceSize), func(b *testing.B) {
			for _, parallelism := range []int{1, 1000} {
				b.Run(fmt.Sprintf("parallelism=%d/interval=1s", parallelism), func(b *testing.B) {
					var (
						subject = NewSamplerWithInterval(time.Second)
						kept    atomic.Uint64
						dropped atomic.Uint64
					)

					// We'll synchronize the start of all of our goroutines to ensure contention happens early.
					b.StopTimer()
					b.SetParallelism(parallelism)
					var (
						wg      sync.WaitGroup
						barrier = make(chan struct{})
					)
					wg.Add(parallelism * runtime.GOMAXPROCS(0))

					go func() {
						wg.Wait()
						b.StartTimer()
						close(barrier) // Un-block all the goroutines; we're in business!
					}()

					b.RunParallel(func(pb *testing.PB) {
						wg.Done()
						<-barrier

						for i := 0; pb.Next(); i = (i + 1) % keySpaceSize {
							if subject.DecisionFor(keySpace[i]) {
								kept.Add(1)
							} else {
								dropped.Add(1)
							}
						}
					})

					fkept := float64(kept.Load())
					b.ReportMetric(100*fkept/(fkept+float64(dropped.Load())), "keep%")
				})
			}
		})
	}
}

func randomOne[T any](list []T) T {
	return list[rand.Intn(len(list))]
}

var (
	httpMethods = []string{
		http.MethodGet,
		http.MethodHead,
		http.MethodPost,
		http.MethodPut,
		http.MethodPatch,
		http.MethodDelete,
		http.MethodConnect,
		http.MethodOptions,
		http.MethodTrace,
	}
	httpStatusCodes = []int{
		http.StatusContinue,
		http.StatusSwitchingProtocols,
		http.StatusProcessing,
		http.StatusEarlyHints,
		http.StatusOK,
		http.StatusCreated,
		http.StatusAccepted,
		http.StatusNonAuthoritativeInfo,
		http.StatusNoContent,
		http.StatusResetContent,
		http.StatusPartialContent,
		http.StatusMultiStatus,
		http.StatusAlreadyReported,
		http.StatusIMUsed,
		http.StatusMultipleChoices,
		http.StatusMovedPermanently,
		http.StatusFound,
		http.StatusSeeOther,
		http.StatusNotModified,
		http.StatusUseProxy,
		http.StatusTemporaryRedirect,
		http.StatusPermanentRedirect,
		http.StatusBadRequest,
		http.StatusUnauthorized,
		http.StatusPaymentRequired,
		http.StatusForbidden,
		http.StatusNotFound,
		http.StatusMethodNotAllowed,
		http.StatusNotAcceptable,
		http.StatusProxyAuthRequired,
		http.StatusRequestTimeout,
		http.StatusConflict,
		http.StatusGone,
		http.StatusLengthRequired,
		http.StatusPreconditionFailed,
		http.StatusRequestEntityTooLarge,
		http.StatusRequestURITooLong,
		http.StatusUnsupportedMediaType,
		http.StatusRequestedRangeNotSatisfiable,
		http.StatusExpectationFailed,
		http.StatusTeapot,
		http.StatusMisdirectedRequest,
		http.StatusUnprocessableEntity,
		http.StatusLocked,
		http.StatusFailedDependency,
		http.StatusTooEarly,
		http.StatusUpgradeRequired,
		http.StatusPreconditionRequired,
		http.StatusTooManyRequests,
		http.StatusRequestHeaderFieldsTooLarge,
		http.StatusUnavailableForLegalReasons,
		http.StatusInternalServerError,
		http.StatusNotImplemented,
		http.StatusBadGateway,
		http.StatusServiceUnavailable,
		http.StatusGatewayTimeout,
		http.StatusHTTPVersionNotSupported,
		http.StatusVariantAlsoNegotiates,
		http.StatusInsufficientStorage,
		http.StatusLoopDetected,
		http.StatusNotExtended,
		http.StatusNetworkAuthenticationRequired,
	}
	httpRoute = []string{
		"/{namespace}/{repository}/@{snapshot}/*",
		"/api/ui/query/timeseries",
		"/api/v1/synthetics/private-locations/tests",
		"/api/ui/frontend_telemetry/metrics",
		"/api/ui/query/rapid-scalar",
		"/api/v1/validate",
		"/config",
		"/v8/artifacts/{id}",
		"/api/ui/feature-flags/get-active-feature-flags",
		"/api/v1/synthetics/private-locations/tests/{publicID}/results",
		"/api/v1/distribution_points",
		"/api/<version>/logs-analytics/<event_platform_type:type>",
		"/internal/snapshot/pull",
		"/*",
		"/api/unstable/apm/entities",
		"/v2/",
		"/dashboard/shared_widget_update/{id}/{widget_id}",
		"/authn",
		"/api/v1/synthetics/tests/updates/<polling>",
		"/authz",
		"/v3/elements/search",
		"/api/ui/event-platform/logs/facets",
		"/vault/agent/",
		"/internal/snapshot/push",
		"/api/v2/dashboard/presence/{resourceId}",
		"/api/v1/snapshot",
		"/api/v1/query",
		"/api/v1/series",
		"/v8/artifacts/events",
		"/api/v1/monitor/<monitor_id>",
		"/series/batch_query",
		"/orgs/{orgId}/analytics/requests",
		"/api/v0.1/status",
		"/api/v1/synthetics/private-locations/tests/<public_id>/results",
		"/api/ui/error-tracking/issues/searches",
		"/api/intake/ci/custom_spans",
		"/internal/error-tracking/search",
		"/v2/{name:(?:(?:(?:[a-zA-Z0-9]",
		"/api/v0.1/configurations",
		"/api/v2/apm/service_inference",
		"/api/v2/git/repository/search_commits",
		"/jobs/<job_id>",
		"/api/v2/libraries/tests/services/setting",
		"/api/ui/data_streams/map",
		"/readiness",
		"/api/ui/apm/latency_breakdowns",
		"/api/v2/discussions/dashboard/{dashboardID}",
		"^dump-request$",
		"/api/v2/watchdog/insights/search/<request_id>",
		"/api/ui/apm/versions",
		"/probe",
		"/api/v2/watchdog/insights/search",
		"/api/ui/orchestration/facets/v2",
		"/account/login",
		"/-/readiness",
		"/api/ui/apm/resources",
		"/-/liveness",
		"/monitor/evaluation_graph",
		"/{namespace}/{repository}/@{snapshot}/simple/{project}",
		"/watchdoginsights/outliers_query",
		"/health/readiness",
		"/health/liveness",
		"/api/v2/on-prem-management-service/workflow-tasks/dequeue",
		"/api/v2/users",
		"/api/ui/slack-app/event",
		"/token",
		"/v2/{name:(?:(?:(?:[a-zA-Z0-9]",
		"/watchdoginsights/outliers_resume",
		"/_health",
		"/api/unstable/coscreen",
		"/internal/v0/3rd-party-software-info/query",
		"/api/v2/org_configs/<org_config_name>",
		"/health",
		"/watchdoginsights/watchdog_stories_query",
		"/api/<version>/logs-analytics/<event_platform_type:type>/<requestId>",
		"/api/intake/llm-obs/v1/trace/spans",
		"/internal/ciem/enablementRequirements/{orgId}",
		"/api/v2/ddsql/table",
		"/monitor/search",
		"/api/v2/cases/count",
		"/api/ui/apm/live-search",
		"/api/ui/orchestration/entities",
		"/api/ui/experiments/get-all-experiments",
		"/api/ui/event-platform/trace/facets",
		"/api/v1/settings/favorite/list",
		"/api/ui/team",
		"/api/v1/dashboard/<dd_resource:dashboard_id>",
		"/api/v2/query/timeseries",
		"/api/<version>/watchdog/stories",
		"/orgs/{orgId}/analytics/requests/{requestId}",
		"/api/v2/team",
		"/api/unstable/apm/entities/graph",
		"/api/v2/incidents/search",
		"/api/ui/support/authenticate",
		"/dequeue_job",
		"/api/ui/apm/breakdown",
		"/static/*",
		"/api/unstable/change-stories",
		"/v2/{name:(?:(?:(?:[a-zA-Z0-9]",
		"/api/v1/logs/views",
		"/api/v2/ci/libraries/tests",
		"/api/v2/notebook/presence/{resourceId}",
		"/api/unstable/restriction_policy/relations/bulk",
		"/orgs/{orgId}/scopes",
		"/api/v1/security_analytics/activation",
		"/api/v2/cases",
		"/api/ui/api-monitoring/endpoint",
		"/watchdoginsights/log_anomalies_query",
		"/api/v1/slo/<slo_id>/history",
		"/api/ui/api-monitoring/endpoints",
		"/api/v1/logs/indexes",
		"/orgs/{orgId}/security_analytics/activation",
		"/lock",
		"/v2/{repo:(?:(?:(?:[a-zA-Z0-9]",
		"/api/ui/apm/web/metadata",
		"/metric/hosts_and_tags",
		"/api/v1/monitor",
		"/api/ui/event-platform/rum/facets",
		"/favicon.ico",
		"/api/ui/event-platform/feed/facets",
		"/api/v2/workflows/{workflowId}/instances",
		"/watchdoginsights/profiling_query",
		"/api/v2/case/presence/{resourceId}",
		"/api/v1/metrics/<metric_name>",
		"/api/<version>/watchdog/story_facets",
		"/v1/applications/{appId}",
		"/snapshot/view/<bucket>/<org>/<date>/<id>",
		"/fabric-proxy-healthcheck",
		"/api/v2/users/<user_id>",
		"/api/v2/current_user/user_team_fave",
		"/dashboard/*",
		"/api/v1/monitor/<monitor_id>/validate",
		"/api/ui/query/scalar",
		"/api/v2/static-analysis/rulesets/{RulesetName}",
		"/api/ui/data_streams/info",
		"/api/v2/event-platform/profile/activity",
		"/metric/flat_tags_for_metric",
		"/api/v2/services/definitions/<path:path>",
		"/api/unstable/databases/hosts",
		"^$",
		"api/usage/v1/hourly-usage.{ext}",
		"/internal/api/v2/devflow/run-workflow",
		"/api/ui/apm/host_groups",
		"/api/v2/ci/tests/skippable",
		"/api/v2/restriction_policy/<resource>",
		"/status/index",
		"/api/v1/logs/default-views",
		"/api/v2/cases/aggregate",
		"/dash/set_preference",
		"/internal/catalog/v1/namespaces/*",
		"/computeMetadata/v1/instance/service-accounts/{serviceAccount:[^/]+}/token",
		"/metric/metric_metadata",
		"/orgs/{orgId}/security_analytics/activation/{product}/pricing_model",
		"/api/unstable/databases/recommendations/aggregated",
		"/api/ui/event-platform/audit/facets",
		"/api/v1/restriction_policy/{resourceID}",
		"/api/ui/event-platform/query/field",
		"/api/v1/monitor/validate",
		"/metric/contexts_for_metrics",
		"/graphql",
		"/{version}/meta-data/iam/security-credentials/{role:.+}",
		"/orgs/{orgId}/indexes",
		"/api/v2/workflows/<workflow_id>/instances/<instance_id>",
		"/api/v1/monitor/<monitor_id>/search_groups",
		"/v1/applications/{appId}/streaming",
		"/check/<check_name>/contexts",
		"/api/v2/process/summary",
		"/{version}/meta-data/iam/security-credentials/",
		"/api/ui/flows/scalar",
		"/api/v2/catalog/entity",
		"/api/ui/integration/jira/webhook/{org_uuid}",
		"/api/v1/apps/:app/inputs",
		"/internal/dashboards/search/_index",
		"/api/v1/apps/:app/inputs/:inputId",
		"/api/v2/services/definitions",
		"/api/v2/embed_query/timeseries",
		"/api/ui/orchestration/manifest_history",
		"/api/ui/monitor/search_all_counts",
		"/api/ui/apm/latest_deployments",
		"/",
		"/internal/compliance/frameworks/{orgId}",
		"/api/v2/ci/pipeline/tags",
		"/api/unstable/public_dashboard_widget_query",
		"/api/v2/logs/events/search",
		"/api/unstable/datasets",
		"/internal/event-platform/query/composite",
		"/api/ui/trace/{traceID}",
		"/api/unstable/accupath/entities/graph/{entityType}/*",
		"^click$",
		"/api/v1/monitor/groups/search",
		"/api/v2/static-analysis/config/client",
		"/api/v2/incidents-incident/presence/{resourceId}",
		"/api/v2/metrics/<metric_name>/volumes",
		"/api/v1/dashboard_search",
		"/api/<version>/hosts",
		"/v2/{repo:(?:(?:(?:[a-zA-Z0-9]",
		"/api/v2/roles",
		"/api/ui/event-platform/{track}/standard-attributes/rules",
		"/api/v2/ci/pipeline/metrics",
		"/api/v2/host_map/nodes_by_group",
		"/api/<version>/databases/hosts",
		"/api/ui/apm/widget_stats_histogram",
		"/api/<version>/logs-queries/list",
		"/api/ui/dashboards/<path:dashboard_id>/payload_version_history",
		"/api/unstable/on-call/teams",
		"/api/v1/tags/hosts/<host_id>",
		"/api/ui/security/asm/services",
		"/api/v2/synthetics/tests/<public_id>/results",
		"/api/ui/data_streams/alerts",
		"/api/v2/workflows/{workflowId}/instances/{instanceId}",
		"/api/v2/incident/presence/{resourceId}",
		"/mail/sendgrid_incoming",
		"/v2/{name:(?:(?:(?:[a-zA-Z0-9]",
		"/api/v1/legacy_current_user",
		"/unauthenticated_probe",
		"/api/ui/services/catalog",
		"/v3/elements/update",
		"/api/unstable/obs-pipelines/quotas/{pipelineId}/buckets",
		"/api/v2/services/catalog/product_areas",
		"/computeMetadata/v1/instance/service-accounts/{serviceAccount:[^/]+}/",
		"/api/v2/apm/services/stats",
		"/api/v1/intake_monitoring",
		"/series/proc_key_query",
		"/api/v2/incidents-timeline/presence/{resourceId}",
		"/lock/<lock_id>",
		"/api/ui/apm/service/graph_real_time",
		"/api/ui/error-tracking/issues/field-translations",
		"/internal/security-graph/shards/{orgID}",
		"/api/<version>/events/<event_id>",
		"/autoscaling/recommendation",
		"/api/ui/event-platform/databasequery/facets",
		"/api/unstable/databases/monitors",
		"/api/unstable/apm/entities/ranked_entity_id_tags",
		"/api/unstable/cost/tags",
		"/api/intake/aws/securitylake",
		"/tile/validate",
		"/api/unstable/datasets/<dataset_uuid>",
		"/api/ui/monitor/search_all",
		"/api/v1/synthetics/tests/uptimes",
		"/api/ui/metric/search",
		"/api/v2/notebook/multiplayer/{resourceId}",
		"/api/v2/ci/pipeline",
		"/api/v1/notebooks/<notebook_id>",
		"/api/v2/host_map/filters_and_groups",
		"/monitor/validate_text",
		"/api/ui/apm/has_traces",
		"/api/ui/apm/dependency_stats_tree",
		"/api/v1/rum/projects",
		"/api/unstable/databases/recommendations",
		"/api/v1/check_run",
		"/api/v2/scorecard/grouped-outcomes",
		"/integration/slack/inbound-event",
		"/api/v2/intake_monitoring",
		"/api/unstable/on-call/pages",
		"/api/ui/event-platform/{track}/livetail/token",
		"/api/ui/app_installations",
		"/internal/event-platform/query/composite/resume",
		"/api/v2/services/integrations/pagerduty",
		"/api/v1/monitor/search",
		"/v1/applications",
		"/api/ui/api-monitoring/endpoints/count",
		"/liveness",
		"/dashboard/template-variables/values",
		"/v1/version",
		"/api/v2/workflows",
		"/api/v1/notebooks",
		"/metric/tag_keys_for_metric",
		"/api/unstable/security/appsec/vm/activation",
		"/computeMetadata/v1/project/project-id",
		"/api/v1/synthetics/tests/<dd_resource:public_id>/locations",
		"/api/ui/slo/correction_windows",
		"/files/{ref}/{filename}",
		"/api/unstable/datasets/{datasetUUID}",
		"/api/v1/integration/webhooks/configuration/webhooks/<webhook_name>",
		"/process/tags/v2",
		"/api/v2/monitor/<monitor_id>/downtime_matches",
		"/apm/*",
		"/api/v2/security_monitoring/rules",
		`/sb/{string_1:[^/?()[\]]+}`,
		"^basic-auth$",
		"/v3/elements",
		"/utils/healthcheck",
		"/api/v2/scorecard/outcomes/batch",
		"/metadata/identity/oauth2/token",
		"/api/ui/apm/web/metadata/{serviceName}",
		"/api/v1/synthetics/tests/search",
		"/api/v1/logs/rate_limited_indexes",
		"/api/<version>/node_map/nodes_by_group",
		"/api/unstable/symdb-api/scopes/search",
		"/api/v2/catalog/relation",
		"/monitor/search_counts",
		"/api/unstable/stegadography/cache",
		"/query",
		"/api/ui/event-platform/signal/facets",
		"/api/beta/sketches",
		"^simple-variable$",
		"/internal/api/repository/{owner}/{repository}/history",
		"/api/v2/current_user",
		"/connectors/{connector}/status",
		"/v1/health",
		"/api/ui/error-tracking/issues/field-values",
		"/api/unstable/symdb-api/services/metadata",
		"/api/ui/span_links/{traceID}",
		"/internal/domains/{DomainName}/routegroups/{RouteGroupName}",
		"/v1/orgs/{orgId}/logs-queries/list",
		"/api/v2/permissions",
		"/api/v2/workflows/<workflow_id>/instances",
		"/api/ui/watchdog/explains/influential-tags/ff",
		"/apm/home",
		"/cost/tags/pod/{namespace}/{pod_name}",
		"/api/v2/integration/salesforce_marketing_cloud/event",
		"/api/ui/cost/settings",
		"^click-trap$",
		"/api/v2/watchdog-story/presence/{resourceId}",
		"/api/v2/reporting/schedule/<string:resource_type_str>/<dd_resource_or_int:resource_id_str>",
		"/api/v2/static-analysis/default-rulesets/{Language}",
		"/indicators",
		"/api/ui/cost/multisource",
		"/internal/api/v2/spark_jobs/{spark_job_id}",
		"/api/v2/scorecard/rules",
		"/api/v2/roles/{roleUuid}",
		"/internal/ci/gitlab/pipeline/*",
		"/api/v2/security_monitoring/configuration/notification_rules",
		"/api/ui/event-platform/query/field-value",
		"/api/v2/synthetics/private-locations/config",
		"/api/v2/test/libraries/test-management/tests",
		"/api/v1/slo",
		"/cluster_configuration/<config_name>",
		"/api/v1/slo/search",
		"/api/ui/error-tracking/config",
		"/api/ui/event-platform/profile/facets",
		"/internal/api/v2/concurrency_locks",
		"/api/v2/rum/funnel",
		"/api/v0/synthetics/tests/<dd_resource:public_id>/status",
		"/catalog/v1/namespaces",
		"/api/unstable/restriction_policy/principal/<principal>",
		"/api/v2/services/integrations/pagerduty/oncalls",
		"/api/v1/synthetics/tests",
		"/api/v1/monitor/tags",
		"/internal/notebooks/search/_index",
		"/orgs/{orgId}/cloud_security_management/resource_filters",
		"/api/v1/integration",
		"^cors-protected-resource$",
		"/connectors/{connector}",
		"/api/unstable/ui/trace/{traceID}",
		"/alert/query_value/<alert_id>",
		"/api/v2/query/scalar",
		"/metric/histogram",
		"/api/ui/resource_catalog/list",
		"/v2/orgs/{orgId}/{trackType}/events/search",
		"/api/ui/remote_config/*",
		"/api/v2/security_monitoring/rules/{ruleId}",
		"/api/v2/api_keys/<api_key_id>",
		"^status$",
		"/api/v2/source-code-integration/enrich-stack-trace",
		"/monitor/validate_param",
		"/orgs/{orgId}/v2/security_analytics/rules",
		"/api/v2/on-prem-management-service/runner/health-check",
		"/api/v2/security_monitoring/signals/<signal_id>",
		"/event/distrib",
		"/internal/ciem/enabledIntegrations/{orgId}",
		"/jobs",
		"/internal/core/v2/prefixes/{id}",
		"/api/v2/apicatalog/endpoints",
		"/v2/orgs/{orgId}/vm",
		"^sigv4$",
		"/oauth2/v1/token",
		"/api/v1/synthetics/tests/browser/<public_id>",
		"/api/v1/dashboard/public/<string:token>",
		"^simple-iframe$",
		"/api/<version>/host_map/node_values",
		"/api/v2/services/integrations/pagerduty/incidents",
		"/v2/orgs/{orgId}/security_monitoring/configuration/notification_rules",
		"/internal/intake/developer-telemetry/count",
		"/api/v2/org_connections",
		"/api/v1/synthetics/browser/tests/<public_id>/results/<result_id>/steps/<step_index>/screenshot",
		"/api/ui/security_monitoring/content_packs",
		"/api/unstable/on-call/escalation-policies/{policyUUID}/steps/{policyStepUUID}/escalations",
		"/orgs/{orgId}/security_analytics/signals/get_metadata",
		"/api/ui/container_images/facets",
		"/api/ui/on-call/shifts",
		"/api/v2/services/integrations/opsgenie",
		"/monitors/*",
		"/api/v2/services/integrations/opsgenie/incidents",
		"/api/<version>/logs/analytics/aggregate",
		"/orgs/{orgId}/analytics/requests/validate-search",
		"/.well-known/jwks.json",
		"/api/v2/events",
		"/api/v1/slo/<slo_id>",
		"/api/v1/monitor/<monitor_id>/unmute",
		"/*id",
		"/api/v1/rum/replay/sessions/<session_id>/views/<view_id>/segments",
		"/api/v2/infrastructure/devices",
		"/probe_smart_edge",
		"/api/ui/event-platform/profilemetrics/facets",
		"/api/ui/data_streams/apm_streaming_services",
		"/api/v2/ucap/annotations/search-by-file",
		"/api/v2/processes",
		"/healthz/liveness",
		"/api/v2/reporting/schedule/list",
		"/catalog/v1/namespaces/{namespace}/tables/{table}/metrics",
		"/watchdoginsights/synthetics_search_tests",
		"/v2/models/{model_name}/versions/{model_version}/ready",
		"/security/*",
		"/v1/task/{taskId}",
		"/api/v0.1/org",
		"/clusters",
		"/watchdoginsights/outliers_cancel",
		"/logs",
		"/api/v1/dashboard/lists",
		"/v3/elements/aggregate",
		"/api/v1/notebook/<notebook_id>/comments",
		"/api/unstable/apm/tracer_agent_latest_versions",
		"/api/ui/event-platform/{track}/org/state",
		"/orgs/{orgId}/facets-with-reserved-fields",
		"/event/icon/<filename>",
		"/profiling/api/v1/aggregate",
		"/api/ui/event-platform/netflow/facets",
		"/api/v2/ci/my-commits",
		"/v2/orgs/{orgId}/{trackType}/analytics/aggregate",
		"/api/v1/synthetics/tests/<public_id>",
		"/api/v1/synthetics/ci/batch/<batch_id>",
		"/api/v1/security_analytics/rules/facet_info",
		"/orgs/{orgId}/state",
		"/api/ui/apm/live-search/keys",
		"/api/v1/events",
		"/api/v1/synthetics/tests/poll_results",
		"/api/v2/monitor/recommended",
		"/api/v2/ucap/annotations/search",
		"/api/v2/scorecard/outcomes",
		"/internal/csm/onboarding/steps",
		"/api/unstable/obs-pipelines/quotas/batch",
		"/api/v2/incidents/<incident_uuid>/relationships/integrations",
		"/api/ui/notebooks/<path:notebook_id>/payload_version_history",
		"/api/v2/team/<team_id>/memberships",
		"/api/v2/security_monitoring/configuration/security_filters",
		"/api/v2/internal/*",
		"/api/ui/remote_config/products/live_debugging/probes",
		"/series/embed_batch_query",
		"/api/ui/error-tracking/issues/facets",
		"/api/v2/services/definitions/{service:.*}",
		"/api/v1/integrations/tiles/list",
		"/api/v2/services/catalog",
		"/healthz",
		"/api/v1/metadata",
		"/api/v2/logs/events",
		"/api/v2/bits-ai-investigation/presence/{resourceId}",
		"/status",
		"/-/health",
		"/api/v2/api_keys",
		"/api/v2/metrics",
		"/api/v1/synthetics/settings",
		"/internal/api/ui/prr-survey-relay/surveys/{ServiceName}",
		"/v1/info/state",
		"/internal/disruptioncrons",
		"/internal/hypothesis",
		"/api/ui/error-tracking/issues/{issueID}",
		"/series/proc_container_query",
		"/api/v2/slo/{sloID}/status",
		"/v3/entities",
		"/account/change_subdomain/<token>",
		"^headers$",
		"/rum/*",
		"/api/:version/jobs/request",
		"/api/<version>/apm/config/retention-filters",
		"/api/v2/monitor/policy",
		"/api/v1/role/{roleUuid}/user/{userHandle}",
		"/api/ui/billing/vat_required",
		"/api/ui/apm/resource",
		"/api/v0/synthetics/tests/<public_id>",
		"/api/v2/product_trial_summary",
		"/account/oauth2",
		"/api/v1/restriction_policy/principal/{principalID}",
		"/proc/query",
		"/api/v1/insights",
		"/intake",
		"/profiling/api/v1/profiles/<profile_id>/breakdown",
		"/api/ui/error-tracking/issues/searches/{searchID}",
		"/eee-intake/prod/gitlab",
		"/internal/reporting/job/dequeue",
		"/api/v2/incidents/<incident_id>",
		"/api/v2/users/<user_id>/permissions",
		"/api/unstable/security_monitoring/csm_features",
		"/api/v2/bits-ai/config/global",
		"/api/v2/powerpacks",
		"/api/v1/lineage",
		"/api/ui/public_dashboard/<share_token>",
		"/account/plan/plan_details",
		"/v1/memory",
		"/api/v2/incidents/<incident_uuid>/timeline",
		"/api/v2/metrics/<metric_name>/all-tags",
		"/api/unstable/on-call/pages/{pageIdentifier}",
		"/v1/task/{taskId}/status",
		"/monitor/notifications",
		"/api/v2/workflow_instance_sources/<source_id>/instances",
		"/account/integration",
		"/api/ui/usage/billable/overall",
		"/catalog/v1/config",
		"/api/v2/security_monitoring/livetail",
		"/api/<version>/events",
		"/api/<version>/node_map/node_values",
		"/api/v2/application_keys",
		"/api/unstable/slo/search/group_by_tag",
		"/api/ui/event-platform/appsecspan/facets",
		"/api/ui/monitor/<monitor_id>/evaluation_graph",
		"/api/v2/rum/analytics/aggregate",
		"/api/v2/rum/applications",
		"/api/<version>/node_map/filters_and_groups",
		"/api/v2/static-analysis-sca/dependencies",
		"/api/v1/hosts/totals",
		"/api/ui/notebooks/lfg",
		"/metrics",
		"api/usage/v1/status",
		"/api/v2/workflow_instance_sources/{id}/instances",
		"/api/ui/ms-teams-app/bot/activity",
		"/api/v0.2/stats",
		"/internal/api/v2/spark_jobs",
		"^scroll$",
		"/api/v1/audit/config",
		"/api/ui/flows/timeseries",
		"/v1/status",
		"/healthcheck",
		"/monitors/manage",
		"/api/unstable/catalog/definition",
		"/orgs/{orgId}/security_analytics/rules/{ruleId}",
		"/internal/dashboards/search/_bulkindex",
		"/api/ui/trace/related_dashboards",
		"/api/v2/source-code-integration/code-snippet",
		"/synthetics/*",
		"/agent_stats/report_failure",
		"/api/v1/synthetics/_multilocator/updates",
		"/account/saml/assertion",
		"/api/v2/notebooks/search",
		"/v1/task/{taskId}/results/{bufferId}/{token}",
		"/api/v2/rum/resource",
		"/api/v2/rum/events",
		"/api/v2/events/search",
		"/api/ui/event-platform/citest/facets",
		"/api/ui/cost/aws_accounts",
		"/api/v2/incidents",
		"/api/ui/debugger/probe-statuses",
		"/api/ui/event-platform/cipipeline/facets",
		"/api/v2/synthetics/tests/<public_id>/results/<result_id>",
		"/api/v1/security_analytics/rules/facets",
		"/screen/setup_notify_change",
		"/api/v2/metrics/{metricName}/assets",
		"/orgs/{orgId}/security_analytics/rules/facet_info",
		"^assert-element-content$",
		"^user-agent$",
		"/api/v2/saml_configurations/<saml_config_uuid>",
		"/signup/process",
		"/api/ui/metrics/summary/list",
		"/api/v2/integration/gcp/accounts",
		"/api/v2/logs/config/custom-destinations/<custom_destination_id>",
		"/api/v2/embed_query/scalar",
		"/api/v2/dora/settings",
		"/api/v2/apicatalog/tags",
		"/api/v1/logs/views/<view_id>",
		"/api/v2/host_map/node_names",
		"/v1/announcement/{node_id}",
		"/api/unstable/llm-obs/v2/clusters",
		"/v1/service/{type}/{pool}",
		"^input$",
		"/internal/intake/developer-telemetry/distribution",
		"/api/v2/dora/deployment",
		"/api/unstable/monitor/on_missing_data_default_details",
		"/api/v2/rum/replay/sessions/{sessionID}/watchers",
		"^authenticated$",
		"/functions",
		"^assert-text-nowhere$",
		"^iframeception$",
		"^small-device$",
		"/api/v2/{trackType}",
		"/v2/_scanning/",
		"^assert-text-present$",
		"^iframe-in-popup$",
		"/api/v1/synthetics/config",
		"^prototype$",
		"^cors$",
		"/api/v1/security_analytics/rules/{ruleId}",
		"/api/v1/synthetics/tests/api/<public_id>",
		"/gomod/{ref}/*",
		"^hover$",
		"^redirect-auth$",
		"/api/v1/synthetics/variables",
		"/v2/_scanning/{name:(?:(?:(?:[a-zA-Z0-9]",
		"/v2/orgs/{orgId}/security_monitoring/configuration/security_filters",
		"/process/get_token",
		"/v1/statement/executing/{queryId}/{slug}/{token}",
		"^select-option$",
		"/api/v2/pets/{petID}",
		"/monitors/{int_1:[0-9]+}/edit",
		"/api/unstable/ddsql/custom_output",
		"/api/v2/spreadsheets/{spreadsheetID}",
		"/*repository_path/git-upload-pack",
		"/notebook/*",
		"/api/unstable/trace/{traceID}",
		"/api/ui/app_listings",
		"/api/ui/remote_config/management/org_config",
		"/api/v1/security_analytics/rules/tags",
		"/api/v1/rum/projects/<id>",
		"/account/saml/assertion/id/<public_id>",
		"/api/unstable/on-call/teams/{teamUUID}/on-call",
		"/api/v2/apm/services",
		"/internal/core/v2/routetables/{id}",
		"/api/ui/container_images",
		"/api/ui/security/appsec/flaggedips/{from}/{to}",
		"/api/v2/incidents/config/rules",
		"/orgs/{orgId}/facets",
		"/api/v2/flamegraph",
		"/api/v2/oauth2/.well-known/sites",
		"/orgs/{orgId}/v2/security_analytics/livetail",
		"/api/<version>/logs/config/archives/<archive_id>",
		"/internal/security-graph/shards/mappings",
		"/api/ui/team-list",
		"fakecdnoutage",
		"/api/:version/jobs/:id",
		"/api/ui/cost/lite",
		"/api/v2/internal/admin/org_access_request_meta/<org_id>",
		"/api/unstable/integration/aws/accounts",
		"/internal/core/v2/pools/{id}/summary",
		"/api/v2/metrics/<metric_name>/tags",
		"/monitor/bulk_delete",
		"/orgs/{orgId}/security_analytics/rules",
		"/v1/task/{taskId}/results/{bufferId}/{token}/acknowledge",
		"^upload-file$",
		"/api/ui/event-platform/cpfinding/facets",
		"/account/forgot_password",
		"/api/v2/rum/replay/playlists/{playlistID}",
		"/internal/catalog/v1/namespaces",
		"/integration/github/private-apps/hostnames",
		"/api/v2/rum/resource/token",
		"/api/v1/synthetics/tests/browser/<public_id>/results",
		"/api/v1/synthetics/tests/<public_id>/results",
		`/orchestration/{string_1:[^/?()[\]]+}/{string_2:[^/?()[\]]+}`,
		"/api/<version>/observability_pipelines/v2/graphql",
		"/api/v1/monitor/<monitor_id>/related_dashboards",
		`/monitors/create/{string_1:[^/?()[\]]+}`,
		"/xds/validate-config",
		"/api/ui/data_streams/clusters",
		"/v2/orgs/{orgId}/logs/config/custom-destination/{customDestinationId}",
		"/api/ui/remote_config/products/asm/waf/exclusion_filters",
		"/api/v2/py/pets/{pet_id}",
		"/api/v2/security_monitoring/configuration/suppressions/{suppressionId}",
		"/api/v2/rum/events/search",
		"/api/unstable/smoke-test-go-classic-http/five",
		"/api/unstable/smoke-test-go-classic-http/gzip",
		"/api/unstable/smoke-test-go-classic-http/vault/secret",
		"/api/unstable/smoke-test-go-classic-http/consul/value",
		"/api/unstable/smoke-test-go-classic-http/experiment/status",
		"/api/unstable/smoke-test-go-classic-http/panic",
		"/metric/explorer",
		"/api/unstable/smoke-test-go-classic-http/four",
		"/api/unstable/smoke-test-go-classic-http/tap",
		"/api/v1/security_analytics/resource_type_mapping",
		"/api/ui/flows/list",
		"/orgs/{orgId}/security_analytics/activation/{product}/enabled_log_sources",
		"/api/v2/rum/replay/sessions/<session_id>/retention",
		"/api/ui/resource_catalog/facet_resources",
		"/v2/orgs/{orgId}/security_monitoring/cloud_workload_security/agent_rules/csm_features",
		"/api/v1/notebooks/<notebook_id>/cells/<cell_id>",
		"/api/<version>/tag_keys",
		"/api/v2/apm/primary_tags/list",
		"/api/ui/billing/accessible_features",
		"/api/ui/ip-lookup/get-info",
		"/api/v2/rum/replay/sessions/{sessionID}/watches",
		"/api/v2/spans/events",
		"/api/v2/app-builder/apps/{appId}",
		"/error-tracking",
		"/api/v2/team/{team_id}/memberships",
		"/api/v2/ci/pipelines/events/search",
		"/locks",
		"/api/v1/org",
		"/api/v2/connection/connections",
		"/orgs/{orgId}/security_analytics/rules/tags",
		"/api/v2/ddsql/user/table",
		"/api/v1/integration/azure",
		"/api/ui/event-platform/llmobs/facets",
		"/logs/<path:path>",
		"/api/ui/remote_config/products/live_debugging/probes/log/{probeID}",
		"/api/v1/check_github",
		"/security/appsec/*",
		"/api/ui/metrics/summary/facet_info",
		"/api/v2/internal/admin/get_support_org_list",
		"/api/ui/event-platform/ccm/facets",
		"/api/v1/org/<org_pid>",
		"/api/v1/widget/share",
		"/v1/statement/queued/{queryId}/{slug}/{token}",
		"/api/ui/remote_config/products/fleet/agents/count",
		"/api/ui/resource-catalog-api/resources",
		"/api/v2/reference-tables/{TableName}/rows",
		"/api/v1/monitor/can_delete",
		"/api/v2/spreadsheets",
		"/debug/vars",
		"/api/:version/jobs/:id/artifacts",
		"/api/v2/infrastructure/facets",
		"/api/<version>/event-platform/<track>/processors/samples",
		"/api/v1/downtime",
		"/api/v1/integration/gcp",
		"/api/v1/slo/correction/<public_id>",
		"/profiling/api/v1/profiles/<profile_id>/timeline",
		"/api/ui/flows/map",
		"/api/v1/service_dependencies/*",
		"/api/v1/timeline",
		"/api/unstable/error-tracking-codegen/*",
		"/api/v2/git/repository/packfile",
		"/api/v2/actions-datastores/{datastoreId}/items",
		"/api/v2/integration/aws/accounts/{aws_account_config_id}",
		"/oauth2/v1/keys",
		"/api/ui/error-tracking/issues/field-values/{fieldValuesID}",
		"/api/ui/remote_config/products/asm/waf/policies",
		"/api/v2/agentless_scanning/accounts/azure",
		"/api/ui/security_monitoring/rules/dependents",
		"/api/<version>/node_map/node_names",
		"/account/saml/metadata.xml",
		"/metric/distribution/summary_aggr",
		"/api/v2/current_user/application_keys",
		"/api/v1/synthetics/private-locations/tests/files",
		"/api/ui/usage/relevant-summary",
		"/api/ui/error-tracking/issues/{issueID}/suspect-commit",
		"/api/v2/incidents/config/user-defined-roles",
		"/api/v2/incidents/{IncidentUUID}/rule-execution-states",
		"/java-bazel/{ref}/{name}/{filename}",
		"/api/v2/saml_configurations",
		"/intake/",
		"/account/saml/config/<saml_config_uuid>/sp_metadata.xml",
		"/api/ui/usage/ddm-summary",
		"/api/v2/ide/product-availability",
		"/api/v2/logs/historical_indexes",
		"/api/unstable/on-call/schedules/{scheduleUUID}/on-call",
		"/api/v1/integration/pagerduty/configuration/services/<path:service_name>",
		"api/usage/v1/relevant-usage.json",
		"/api/ui/usage/relevant-trends",
		"/api/ui/usage/summary",
		"/api/ui/usage/trends",
		"/api/v2/users/<user_uuid>/memberships",
		"/orgs/{orgId}/security_analytics/rules/dependents",
		"/api/ui/error-tracking/issues/{issueID}/activity-timeline",
		"/oauth2/v1/.well-known/openid-configuration",
		"/api/v2/org",
		"/api/:version/jobs/:id/trace",
		"/api/unstable/sensitive-data-scanner/groups",
		"/api/v2/saml_configurations/idp_metadata",
		"/api/unstable/cost/tag_keys",
		"/v2/orgs/{orgId}/external-archives",
		"/api/unstable/metric/<metric_name>/tag_cardinalities",
		"/internal/api/v2/concurrency_locks/{concurrency_lock_definition_id}/{workflow_id}",
		"/api/ui/monitor/evaluation_preview",
		"/api/v1/synthetics/private-locations",
		"/api/v2/static-analysis/analysis/diff-aware",
		"/api/unstable/error-tracking-codegen/ai-suggestions",
		"/api/ui/app_listings/<app_short_name>",
		"/api/v1/breakdown",
		"/api/ui/remote_config/products/live_debugging/probes/log",
		"/api/v1/dashboard",
		"/api/v2/incidents/config/types",
		"/graph/embed",
		"/orgs/{orgId}/security_analytics/rules/facets",
		"/api/ui/orchestration/manifest_content",
		"/api/v2/consul/value",
		"/api/v2/ide/services/search",
		"/api/v2/vault/secret",
		"/account/login/by_email",
		"/api/ui/usage/ddm-trends",
		"/api/v2/exception",
		"/api/unstable/remote_config/products/asm/activation",
		"/api/v2/experiment/status",
		"/v2/_dd/operations/replicate",
		"/api/v1/correlations/<request_id>",
		"/internal/trino-ui/{cluster}/cluster_details_json",
		"/api/unstable/profiles/list",
		"/api/v2/app-builder/apps",
		"/api/v2/agentless_scanning/setup_feedback/accounts/aws",
		"/api/v2/identity_providers",
		"/api/v2/cloud_security_management/resource_filters",
		"/api/ui/event-platform/synthetics/facets",
		"/api/v2/hamr",
		"/api/v2/agentless_scanning/accounts/aws",
		"/api/ui/event-platform/query/composite-list",
		"/api/v2/users/<user_id>/orgs",
		"/api/ui/security_monitoring/configuration/suppressions/rules/{ruleId}",
		"/internal/api/v2/spark_jobs/{spark_job_id}/{retry_attempt}/state",
		"/api/v2/quality-gates/evaluate",
		"/source/list",
		"/api/ui/siem-historical-detections/jobs/facet_info",
		"/api/v1/trial",
		"/overview/system_tiles",
		"/api/v1/screen/integration/<dashboard_id>",
		"/api/v1/downtime/search",
		"/security",
		"/internal/error-tracking/incoming",
		"/v2/orgs/{orgId}/logs/config/archives/{externalId}",
		"/api/unstable/team",
		"/api/v2/internal/admin/switch_handle_post",
		"/notebook",
		"/v3/orgs/{orgId}/{trackType}/pipelines",
		"/api/ui/data_streams/schemas",
		"/integration/slack/configuration",
		"/api/ui/security/appsec/responses/entities",
		"/metric/distribution/list_summaries",
		"/api/v1/downtime/cancel/by_scope",
		"/api/unstable/stegadography/cache/{id}",
		"/account/login/id/<public_id>",
		"/api/v1/health",
		"/api/v2/spreadsheets/{spreadsheetID}/table-sheets/{sheetID}",
		"/api/v2/static-analysis/repositories/counts/analysis",
		"/api/v2/reference-tables",
		"/api/v2/roles/{roleUuid}/permissions",
		"/api/v1/synthetics/browser/tests/<dd_resource:public_id>",
		"/api/v2/incidents/config/org/settings/<incident_type_uuid>",
		"/api/v2/roles/{roleUuid}/users",
		"/api/v2/downtime",
		"/api/v2/rum/replay/playlists",
		"/api/v2/incidents/config/user-defined-fields",
		"/api/v2/incidents/<incident_identifier>/responders",
		"/api/ui/twilio/voice/events",
		"api/usage/v1/active-usage-types.json",
		"/api/v2/audit/events",
		"/api/v2/identity_providers/<idp_uuid>/users",
		"/api/v1/graph/embed",
		"/api/v2/incidents/config/integrations/<incident_type_uuid>/workflows",
		"/api/v2/logs/views/mark-as-viewed",
		"/api/ui/billing/last-updated",
		"/signup_corp",
		"/integration/github/private-apps/enrich-link",
		"None",
		"/api/v2/connection/integrations",
		"/internal/doggo/search",
		"/api/v2/source-code-integration/repositories-v2",
		"/api/ui/billing/org/is-super",
		"/api/v2/cases/projects",
		"/api/v2/source-code-integration/code-owners",
		"/api/v2/spans/events/search",
		"/.well-known/apollo/server-health",
		"/api/v2/on_prem_runners",
		"/get_org_billing_data",
		"/api/ui/product-catalog/be/ddm/product",
		"/api/ui/product-catalog/bp/billing-dimension",
		"/api/ui/product-catalog/rq/usage_attribution",
		"/api/ui/product-catalog/be/billing-dimension",
		"/api/ui/product-catalog/be/category",
		"/api/ui/product-catalog/bp/percent-based",
		"/api/v1/synthetics/browser/subtests/<public_id>/parents",
		"/api/v2/authn_mappings/<authn_mapping_uuid>",
		"/api/v2/on-call/presence/{resourceId}",
		"/locks/delete",
		"/api/v2/metrics/<metric_name>/estimate",
		"/api/v1/dashboard/recommendations_load_event/<string:dash_type>/<string:dash_id>",
		"/api/:version/runners/verify",
		"/api/ui/integration/aws/integrationissues/account",
		"/api/unstable/csm/onboarding/agents/count",
		"/api/unstable/csm/onboarding/agents",
		"/api/unstable/csm/onboarding/agents/facets",
		"/api/v1/synthetics/browser/tests/<public_id>/results/<result_id>/artifacts",
		"/api/v2/workflow_actions",
		"/api/v2/apicatalog/facet/{facet}",
		"/api/ui/csm/onboarding/agents/count",
		"/api/ui/resource-catalog-api/distinct",
		"/organization-settings/*",
		"/api/v2/rum/replay/sessions/<session_id>/metadata",
		"/api/ui/spans/custom-metrics",
		"/services",
		"/api/ui/obs-pipelines/pipelines/{pipelineId}",
		"/api/v1/compliance_monitoring/findings/rule_based_view",
		"/api/v2/static-analysis/repositories/products-enabled",
		"/internal/cit/pr-comment",
		"/v1/statement",
		"/api/ui/integration/aws/integrationissues/status",
		"/api/v2/py/pets",
		"/process/agent_hosts",
		"/api/unstable/smoke-test-go-classic-http/smoketest/grpc",
		"/-/healthy",
		"/api/unstable/smoke-test-go-classic-http-dest/smoketest/grpc",
		"/api/unstable/smoke-test-go-classic-http/paginated/number-size",
		"/api/unstable/smoke-test-go-classic-http/paginated/cursor",
		"/internal/slack/event",
		"/startup",
		"/api/ui/data_streams/lag_metrics",
		"/api/unstable/smoke-test-go-classic-http/paginated/offset-limit",
		"/api/v1/synthetics/settings/default_locations",
		"/api/v2/pets",
		"/v2/orgs/{orgId}/security_monitoring/configuration/suppressions/rules/{ruleId}",
		"/api/v2/actions-datastores/{datastoreId}",
		"/api/unstable/on-prem-management-service/workflow-tasks/dequeue",
		"/api/v1/search",
		"/api/:version/job",
		"/reports/v2/agents",
		"/api/ui/ide/profiles/analytics",
		"/api/ui/metrics/summary/export",
		"/api/v2/paginated/cursor_limit",
		"/api/v2/paginated/offset_limit",
		"/api/ui/obs-pipelines/pipelines/{pipelineId}/relationships/draft",
		"/api/v2/cloud_security_management/iac_remediation/code_location/finding",
		"/api/v2/response_test/five",
		"/api/v1/metrics",
		"/api/v2/response_test/four",
		"/api/v1/slo/<slo_id>/corrections",
		"/api/v2/paginated/number_size",
		"/api/unstable/databases/recommendations/tags",
		"/api/v2/services/integrations/opsgenie/oncalls",
		"/api/v2/integration/aws/accounts",
		"/u",
		"/api/ui/security/appsec/responses/integrations",
		"/api/v1/synthetics/tests/trigger/ci",
		"/internal/reporting/job/heartbeat",
		"/api/v1/integration/aws",
		"/api/v2/incidents/incident_roles/reserved_roles",
		"/downtime/monitors/match",
		"/-/ready",
		"/api/v1/security_analytics/rules",
		"/v1/bundles/general",
		"/api/ui/integrations/<app_short_name>/accounts",
		"/api/ui/slo/summary_widget_query",
		"/api/ui/workflows/<workflow_id>",
		"/api/unstable/data_security/summary",
		"/api/v2/change-tracking/service_edges",
		"/internal/v1/events",
		"/api/ui/event-platform/secruntime/facets",
		"/ticino/agent/",
		"/api/unstable/cost/tag_keys/<path:tag_key>",
		"/api/ui/apm/default_env_setting",
		"/api/ui/security/appsec/responses/safeguards",
		"/internal/gitlab/*",
		"/api/intake/llm-obs/v1/eval-metric",
		"/api/ui/remote_config/products/asm/waf/rules",
		"/api/v2/workflow_actions/<bundle_name>",
		"^email_ses$",
		"/api/v1/notebooks/<notebook_id>/cells",
		"/api/v1/slo/saved_view/search",
		"/jobs/spark_config",
		"/api/ui/csm/onboarding/agents/heartbeats",
		"/api/ui/remote_config/products/fleet/agents",
		"/gateway/backend/all",
		"/v1/task/{taskId}/results/{bufferId}",
		"/api/<version>/apm/pipelines",
		"/api/v2/rum/sankey",
		"/api/unstable/cost/tag_groups",
		"/api/v2/users/{user_id}/memberships",
		"/api/v2/logs/config/custom-destinations",
		"/api/v1/settings/favorite",
		"/api/v2/event-correlation/rules",
		"/api/v1/graph/snapshot",
		"/api/v2/ddsql/user/csv",
		"/api/v2/source-code-integration/get-file-link",
		"/api/unstable/on-call/pages/facets",
		"/api/ui/ide/profiles/flamegraph",
		"/api/v2/metrics/<metric_name>/active-configurations",
		"/api/ui/data_streams/service_summary",
		"/api/v1/inventories/taxonomy",
		"/_internal/health",
		"/api/v2/containers",
		"/ready",
		"/api/ui/apm/operation_names",
		"/api/v1/logs/configuration/indexes",
		"/orgs/{orgId}/integrations",
		"/v1/issuer/sycamore/token",
		"/api/v2/notifications/handles",
		"/api/v2/workflow_templates",
		"/api/v1/slo/preview_history",
		"/event/explorer",
		"/api/v1/service_dependencies",
		"/infrastructure",
		"/api/v2/static-analysis/secrets/rules",
		"/v2/_scanning/{name:(?:(?:(?:[a-zA-Z0-9]",
		"/orgs/{orgId}/security_analytics/default_rules",
		"/api/ui/apm/host_group_keys",
		"/api/ui/resource_catalog/list/group",
		"/api/v1/downtime/<downtime_id>",
		"/api/v2/audit/events/search",
		"/internal/v2/telemetry-archive/api/orgs/{orgID}/analytics/requests",
		"/api/v1/dash/integration/<dashboard_id>",
		"/gateway",
		"/account/not_my_account",
		"/api/v2/source-code-integration/repo-ids/search",
		"/api/v2/tags/enrichment/{rulesetId}",
		"/api/<version>/watchdog/stories/<string:story_key>",
		"/api/ui/integration/aws/metric_streams_config/{aws_account_config_id}",
		"/api/ui/apps/categories",
		"/api/ui/slo/search/tag_keys",
		"/api/ui/workflows/{workflowId}",
		"/api/unstable/data_security/resources/csv",
		"/account/login/recaptcha",
		"/api/ui/signup/event",
		"/api/unstable/data_security/resources",
		"/api/unstable/query-prediction/predict",
		"/api/v1/synthetics/tests/fast",
		"/api/v2/series",
		"/process",
		"/api/<version>/usage/hourly-attribution",
		"/api/unstable/databases/datamodels/query",
		"/kubernetes_jobs/<kubernetes_job_id>",
		"/v2/orgs/{orgId}/historical-indexes",
		"/api/:version/projects/:id",
		"/api/v2/security_monitoring/signals/search",
		"/api/v1/logs/filters",
		"/internal/csm/onboarding/capabilities",
		"^create-element-with-user-provided-id$",
		"/api/ui/reporting/recurrence/list",
		"/specs/<spec_name>/trigger",
	}
)

var (
	once       sync.Once
	testVector []SamplingKey
)

func initTestVector() {
	once.Do(func() {
		testVector = make([]SamplingKey, len(httpMethods)*len(httpRoute)*len(httpStatusCodes))
		var mi, ri, si int
		for i := range testVector {
			testVector[i] = SamplingKey{
				Method:     httpMethods[mi],
				Route:      httpRoute[ri],
				StatusCode: httpStatusCodes[si],
			}
			if si = (si + 1) % len(httpStatusCodes); si == 0 {
				if ri = (ri + 1) % len(httpRoute); ri == 0 {
					mi++
				}
			}
		}

		// Shuffle the order of the key space so it's random, but use a constant-seed RNG so that all
		// runs are equal (useful in benchmarks).
		rand := rand.New(rand.NewSource(1337))
		rand.Shuffle(len(testVector), func(i int, j int) { testVector[i], testVector[j] = testVector[j], testVector[i] })
	})
}
