// Copyright (c) 2024 Karl Gaissmaier
// SPDX-License-Identifier: MIT

// Please read the ART paper ./doc/artlookup.pdf
// to understand the baseIndex algorithm.

package bart

// netmask for bits
//
//	0b0000_0000, // bits == 0
//	0b1000_0000, // bits == 1
//	0b1100_0000, // bits == 2
//	0b1110_0000, // bits == 3
//	0b1111_0000, // bits == 4
//	0b1111_1000, // bits == 5
//	0b1111_1100, // bits == 6
//	0b1111_1110, // bits == 7
//	0b1111_1111, // bits == 8
func netMask(bits int) uint8 {
	return 0b1111_1111 << (8 - bits)
}

// baseIndex of the first host route 0/8: pfxToIdx(0,8)
const firstHostIdx = 256

// pfxToIdx, maps a prefix table as a 'complete binary tree'.
func pfxToIdx(octet byte, prefixLen int) uint {
	return uint(octet>>(strideLen-prefixLen)) + (1 << prefixLen)
}

// hostIndex, just pfxToIdx(octet, 8) but faster.
func hostIndex(octet uint) uint {
	return octet + firstHostIdx
}

// cmpIndexRank, sort indexes in prefix sort order.
func cmpIndexRank(aIdx, bIdx uint) int {
	// convert idx to prefix
	aOctet, aBits := idxToPfx(aIdx)
	bOctet, bBits := idxToPfx(bIdx)

	// cmp the prefixes, first by address and then by bits
	if aOctet == bOctet {
		if aBits <= bBits {
			return -1
		}

		return 1
	}

	if aOctet < bOctet {
		return -1
	}

	return 1
}

// idxToPfx returns the octet and prefix len of baseIdx.
// It's the inverse to pfxToIdx.
func idxToPfx(idx uint) (octet byte, pfxLen int) {
	return baseIdxLookupTbl[idx].octet, int(baseIdxLookupTbl[idx].bits)
}

// baseIdxLookupTbl, maps idx => octet/bits
//
// Use the pre computed lookup table, the func baseIndexToPrefix is too slow
// in comparison to an array access.
//
//	func baseIndexToPrefix(idx uint) (octet byte, pfxLen int) {
//		nlz := bits.LeadingZeros(idx)
//		pfxLen = strconv.IntSize - nlz - 1
//		octet = byte((idx & (0xFF >> (8 - pfxLen))) << (8 - pfxLen))
//		return octet, pfxLen
//	}
var baseIdxLookupTbl = [512]struct {
	octet byte
	bits  int8
}{
	/* idx == 0   */ {0, -1}, // invalid
	/* idx == 1   */ {0, 0},
	/* idx == 2   */ {0, 1},
	/* idx == 3   */ {128, 1},
	/* idx == 4   */ {0, 2},
	/* idx == 5   */ {64, 2},
	/* idx == 6   */ {128, 2},
	/* idx == 7   */ {192, 2},
	/* idx == 8   */ {0, 3},
	/* idx == 9   */ {32, 3},
	/* idx == 10  */ {64, 3},
	/* idx == 11  */ {96, 3},
	/* idx == 12  */ {128, 3},
	/* idx == 13  */ {160, 3},
	/* idx == 14  */ {192, 3},
	/* idx == 15  */ {224, 3},
	/* idx == 16  */ {0, 4},
	/* idx == 17  */ {16, 4},
	/* idx == 18  */ {32, 4},
	/* idx == 19  */ {48, 4},
	/* idx == 20  */ {64, 4},
	/* idx == 21  */ {80, 4},
	/* idx == 22  */ {96, 4},
	/* idx == 23  */ {112, 4},
	/* idx == 24  */ {128, 4},
	/* idx == 25  */ {144, 4},
	/* idx == 26  */ {160, 4},
	/* idx == 27  */ {176, 4},
	/* idx == 28  */ {192, 4},
	/* idx == 29  */ {208, 4},
	/* idx == 30  */ {224, 4},
	/* idx == 31  */ {240, 4},
	/* idx == 32  */ {0, 5},
	/* idx == 33  */ {8, 5},
	/* idx == 34  */ {16, 5},
	/* idx == 35  */ {24, 5},
	/* idx == 36  */ {32, 5},
	/* idx == 37  */ {40, 5},
	/* idx == 38  */ {48, 5},
	/* idx == 39  */ {56, 5},
	/* idx == 40  */ {64, 5},
	/* idx == 41  */ {72, 5},
	/* idx == 42  */ {80, 5},
	/* idx == 43  */ {88, 5},
	/* idx == 44  */ {96, 5},
	/* idx == 45  */ {104, 5},
	/* idx == 46  */ {112, 5},
	/* idx == 47  */ {120, 5},
	/* idx == 48  */ {128, 5},
	/* idx == 49  */ {136, 5},
	/* idx == 50  */ {144, 5},
	/* idx == 51  */ {152, 5},
	/* idx == 52  */ {160, 5},
	/* idx == 53  */ {168, 5},
	/* idx == 54  */ {176, 5},
	/* idx == 55  */ {184, 5},
	/* idx == 56  */ {192, 5},
	/* idx == 57  */ {200, 5},
	/* idx == 58  */ {208, 5},
	/* idx == 59  */ {216, 5},
	/* idx == 60  */ {224, 5},
	/* idx == 61  */ {232, 5},
	/* idx == 62  */ {240, 5},
	/* idx == 63  */ {248, 5},
	/* idx == 64  */ {0, 6},
	/* idx == 65  */ {4, 6},
	/* idx == 66  */ {8, 6},
	/* idx == 67  */ {12, 6},
	/* idx == 68  */ {16, 6},
	/* idx == 69  */ {20, 6},
	/* idx == 70  */ {24, 6},
	/* idx == 71  */ {28, 6},
	/* idx == 72  */ {32, 6},
	/* idx == 73  */ {36, 6},
	/* idx == 74  */ {40, 6},
	/* idx == 75  */ {44, 6},
	/* idx == 76  */ {48, 6},
	/* idx == 77  */ {52, 6},
	/* idx == 78  */ {56, 6},
	/* idx == 79  */ {60, 6},
	/* idx == 80  */ {64, 6},
	/* idx == 81  */ {68, 6},
	/* idx == 82  */ {72, 6},
	/* idx == 83  */ {76, 6},
	/* idx == 84  */ {80, 6},
	/* idx == 85  */ {84, 6},
	/* idx == 86  */ {88, 6},
	/* idx == 87  */ {92, 6},
	/* idx == 88  */ {96, 6},
	/* idx == 89  */ {100, 6},
	/* idx == 90  */ {104, 6},
	/* idx == 91  */ {108, 6},
	/* idx == 92  */ {112, 6},
	/* idx == 93  */ {116, 6},
	/* idx == 94  */ {120, 6},
	/* idx == 95  */ {124, 6},
	/* idx == 96  */ {128, 6},
	/* idx == 97  */ {132, 6},
	/* idx == 98  */ {136, 6},
	/* idx == 99  */ {140, 6},
	/* idx == 100 */ {144, 6},
	/* idx == 101 */ {148, 6},
	/* idx == 102 */ {152, 6},
	/* idx == 103 */ {156, 6},
	/* idx == 104 */ {160, 6},
	/* idx == 105 */ {164, 6},
	/* idx == 106 */ {168, 6},
	/* idx == 107 */ {172, 6},
	/* idx == 108 */ {176, 6},
	/* idx == 109 */ {180, 6},
	/* idx == 110 */ {184, 6},
	/* idx == 111 */ {188, 6},
	/* idx == 112 */ {192, 6},
	/* idx == 113 */ {196, 6},
	/* idx == 114 */ {200, 6},
	/* idx == 115 */ {204, 6},
	/* idx == 116 */ {208, 6},
	/* idx == 117 */ {212, 6},
	/* idx == 118 */ {216, 6},
	/* idx == 119 */ {220, 6},
	/* idx == 120 */ {224, 6},
	/* idx == 121 */ {228, 6},
	/* idx == 122 */ {232, 6},
	/* idx == 123 */ {236, 6},
	/* idx == 124 */ {240, 6},
	/* idx == 125 */ {244, 6},
	/* idx == 126 */ {248, 6},
	/* idx == 127 */ {252, 6},
	/* idx == 128 */ {0, 7},
	/* idx == 129 */ {2, 7},
	/* idx == 130 */ {4, 7},
	/* idx == 131 */ {6, 7},
	/* idx == 132 */ {8, 7},
	/* idx == 133 */ {10, 7},
	/* idx == 134 */ {12, 7},
	/* idx == 135 */ {14, 7},
	/* idx == 136 */ {16, 7},
	/* idx == 137 */ {18, 7},
	/* idx == 138 */ {20, 7},
	/* idx == 139 */ {22, 7},
	/* idx == 140 */ {24, 7},
	/* idx == 141 */ {26, 7},
	/* idx == 142 */ {28, 7},
	/* idx == 143 */ {30, 7},
	/* idx == 144 */ {32, 7},
	/* idx == 145 */ {34, 7},
	/* idx == 146 */ {36, 7},
	/* idx == 147 */ {38, 7},
	/* idx == 148 */ {40, 7},
	/* idx == 149 */ {42, 7},
	/* idx == 150 */ {44, 7},
	/* idx == 151 */ {46, 7},
	/* idx == 152 */ {48, 7},
	/* idx == 153 */ {50, 7},
	/* idx == 154 */ {52, 7},
	/* idx == 155 */ {54, 7},
	/* idx == 156 */ {56, 7},
	/* idx == 157 */ {58, 7},
	/* idx == 158 */ {60, 7},
	/* idx == 159 */ {62, 7},
	/* idx == 160 */ {64, 7},
	/* idx == 161 */ {66, 7},
	/* idx == 162 */ {68, 7},
	/* idx == 163 */ {70, 7},
	/* idx == 164 */ {72, 7},
	/* idx == 165 */ {74, 7},
	/* idx == 166 */ {76, 7},
	/* idx == 167 */ {78, 7},
	/* idx == 168 */ {80, 7},
	/* idx == 169 */ {82, 7},
	/* idx == 170 */ {84, 7},
	/* idx == 171 */ {86, 7},
	/* idx == 172 */ {88, 7},
	/* idx == 173 */ {90, 7},
	/* idx == 174 */ {92, 7},
	/* idx == 175 */ {94, 7},
	/* idx == 176 */ {96, 7},
	/* idx == 177 */ {98, 7},
	/* idx == 178 */ {100, 7},
	/* idx == 179 */ {102, 7},
	/* idx == 180 */ {104, 7},
	/* idx == 181 */ {106, 7},
	/* idx == 182 */ {108, 7},
	/* idx == 183 */ {110, 7},
	/* idx == 184 */ {112, 7},
	/* idx == 185 */ {114, 7},
	/* idx == 186 */ {116, 7},
	/* idx == 187 */ {118, 7},
	/* idx == 188 */ {120, 7},
	/* idx == 189 */ {122, 7},
	/* idx == 190 */ {124, 7},
	/* idx == 191 */ {126, 7},
	/* idx == 192 */ {128, 7},
	/* idx == 193 */ {130, 7},
	/* idx == 194 */ {132, 7},
	/* idx == 195 */ {134, 7},
	/* idx == 196 */ {136, 7},
	/* idx == 197 */ {138, 7},
	/* idx == 198 */ {140, 7},
	/* idx == 199 */ {142, 7},
	/* idx == 200 */ {144, 7},
	/* idx == 201 */ {146, 7},
	/* idx == 202 */ {148, 7},
	/* idx == 203 */ {150, 7},
	/* idx == 204 */ {152, 7},
	/* idx == 205 */ {154, 7},
	/* idx == 206 */ {156, 7},
	/* idx == 207 */ {158, 7},
	/* idx == 208 */ {160, 7},
	/* idx == 209 */ {162, 7},
	/* idx == 210 */ {164, 7},
	/* idx == 211 */ {166, 7},
	/* idx == 212 */ {168, 7},
	/* idx == 213 */ {170, 7},
	/* idx == 214 */ {172, 7},
	/* idx == 215 */ {174, 7},
	/* idx == 216 */ {176, 7},
	/* idx == 217 */ {178, 7},
	/* idx == 218 */ {180, 7},
	/* idx == 219 */ {182, 7},
	/* idx == 220 */ {184, 7},
	/* idx == 221 */ {186, 7},
	/* idx == 222 */ {188, 7},
	/* idx == 223 */ {190, 7},
	/* idx == 224 */ {192, 7},
	/* idx == 225 */ {194, 7},
	/* idx == 226 */ {196, 7},
	/* idx == 227 */ {198, 7},
	/* idx == 228 */ {200, 7},
	/* idx == 229 */ {202, 7},
	/* idx == 230 */ {204, 7},
	/* idx == 231 */ {206, 7},
	/* idx == 232 */ {208, 7},
	/* idx == 233 */ {210, 7},
	/* idx == 234 */ {212, 7},
	/* idx == 235 */ {214, 7},
	/* idx == 236 */ {216, 7},
	/* idx == 237 */ {218, 7},
	/* idx == 238 */ {220, 7},
	/* idx == 239 */ {222, 7},
	/* idx == 240 */ {224, 7},
	/* idx == 241 */ {226, 7},
	/* idx == 242 */ {228, 7},
	/* idx == 243 */ {230, 7},
	/* idx == 244 */ {232, 7},
	/* idx == 245 */ {234, 7},
	/* idx == 246 */ {236, 7},
	/* idx == 247 */ {238, 7},
	/* idx == 248 */ {240, 7},
	/* idx == 249 */ {242, 7},
	/* idx == 250 */ {244, 7},
	/* idx == 251 */ {246, 7},
	/* idx == 252 */ {248, 7},
	/* idx == 253 */ {250, 7},
	/* idx == 254 */ {252, 7},
	/* idx == 255 */ {254, 7},
	//
	// START of HOST ROUTES, pfxLen == 8
	//
	/* idx == 256 */ {0, 8},
	/* idx == 257 */ {1, 8},
	/* idx == 258 */ {2, 8},
	/* idx == 259 */ {3, 8},
	/* idx == 260 */ {4, 8},
	/* idx == 261 */ {5, 8},
	/* idx == 262 */ {6, 8},
	/* idx == 263 */ {7, 8},
	/* idx == 264 */ {8, 8},
	/* idx == 265 */ {9, 8},
	/* idx == 266 */ {10, 8},
	/* idx == 267 */ {11, 8},
	/* idx == 268 */ {12, 8},
	/* idx == 269 */ {13, 8},
	/* idx == 270 */ {14, 8},
	/* idx == 271 */ {15, 8},
	/* idx == 272 */ {16, 8},
	/* idx == 273 */ {17, 8},
	/* idx == 274 */ {18, 8},
	/* idx == 275 */ {19, 8},
	/* idx == 276 */ {20, 8},
	/* idx == 277 */ {21, 8},
	/* idx == 278 */ {22, 8},
	/* idx == 279 */ {23, 8},
	/* idx == 280 */ {24, 8},
	/* idx == 281 */ {25, 8},
	/* idx == 282 */ {26, 8},
	/* idx == 283 */ {27, 8},
	/* idx == 284 */ {28, 8},
	/* idx == 285 */ {29, 8},
	/* idx == 286 */ {30, 8},
	/* idx == 287 */ {31, 8},
	/* idx == 288 */ {32, 8},
	/* idx == 289 */ {33, 8},
	/* idx == 290 */ {34, 8},
	/* idx == 291 */ {35, 8},
	/* idx == 292 */ {36, 8},
	/* idx == 293 */ {37, 8},
	/* idx == 294 */ {38, 8},
	/* idx == 295 */ {39, 8},
	/* idx == 296 */ {40, 8},
	/* idx == 297 */ {41, 8},
	/* idx == 298 */ {42, 8},
	/* idx == 299 */ {43, 8},
	/* idx == 300 */ {44, 8},
	/* idx == 301 */ {45, 8},
	/* idx == 302 */ {46, 8},
	/* idx == 303 */ {47, 8},
	/* idx == 304 */ {48, 8},
	/* idx == 305 */ {49, 8},
	/* idx == 306 */ {50, 8},
	/* idx == 307 */ {51, 8},
	/* idx == 308 */ {52, 8},
	/* idx == 309 */ {53, 8},
	/* idx == 310 */ {54, 8},
	/* idx == 311 */ {55, 8},
	/* idx == 312 */ {56, 8},
	/* idx == 313 */ {57, 8},
	/* idx == 314 */ {58, 8},
	/* idx == 315 */ {59, 8},
	/* idx == 316 */ {60, 8},
	/* idx == 317 */ {61, 8},
	/* idx == 318 */ {62, 8},
	/* idx == 319 */ {63, 8},
	/* idx == 320 */ {64, 8},
	/* idx == 321 */ {65, 8},
	/* idx == 322 */ {66, 8},
	/* idx == 323 */ {67, 8},
	/* idx == 324 */ {68, 8},
	/* idx == 325 */ {69, 8},
	/* idx == 326 */ {70, 8},
	/* idx == 327 */ {71, 8},
	/* idx == 328 */ {72, 8},
	/* idx == 329 */ {73, 8},
	/* idx == 330 */ {74, 8},
	/* idx == 331 */ {75, 8},
	/* idx == 332 */ {76, 8},
	/* idx == 333 */ {77, 8},
	/* idx == 334 */ {78, 8},
	/* idx == 335 */ {79, 8},
	/* idx == 336 */ {80, 8},
	/* idx == 337 */ {81, 8},
	/* idx == 338 */ {82, 8},
	/* idx == 339 */ {83, 8},
	/* idx == 340 */ {84, 8},
	/* idx == 341 */ {85, 8},
	/* idx == 342 */ {86, 8},
	/* idx == 343 */ {87, 8},
	/* idx == 344 */ {88, 8},
	/* idx == 345 */ {89, 8},
	/* idx == 346 */ {90, 8},
	/* idx == 347 */ {91, 8},
	/* idx == 348 */ {92, 8},
	/* idx == 349 */ {93, 8},
	/* idx == 350 */ {94, 8},
	/* idx == 351 */ {95, 8},
	/* idx == 352 */ {96, 8},
	/* idx == 353 */ {97, 8},
	/* idx == 354 */ {98, 8},
	/* idx == 355 */ {99, 8},
	/* idx == 356 */ {100, 8},
	/* idx == 357 */ {101, 8},
	/* idx == 358 */ {102, 8},
	/* idx == 359 */ {103, 8},
	/* idx == 360 */ {104, 8},
	/* idx == 361 */ {105, 8},
	/* idx == 362 */ {106, 8},
	/* idx == 363 */ {107, 8},
	/* idx == 364 */ {108, 8},
	/* idx == 365 */ {109, 8},
	/* idx == 366 */ {110, 8},
	/* idx == 367 */ {111, 8},
	/* idx == 368 */ {112, 8},
	/* idx == 369 */ {113, 8},
	/* idx == 370 */ {114, 8},
	/* idx == 371 */ {115, 8},
	/* idx == 372 */ {116, 8},
	/* idx == 373 */ {117, 8},
	/* idx == 374 */ {118, 8},
	/* idx == 375 */ {119, 8},
	/* idx == 376 */ {120, 8},
	/* idx == 377 */ {121, 8},
	/* idx == 378 */ {122, 8},
	/* idx == 379 */ {123, 8},
	/* idx == 380 */ {124, 8},
	/* idx == 381 */ {125, 8},
	/* idx == 382 */ {126, 8},
	/* idx == 383 */ {127, 8},
	/* idx == 384 */ {128, 8},
	/* idx == 385 */ {129, 8},
	/* idx == 386 */ {130, 8},
	/* idx == 387 */ {131, 8},
	/* idx == 388 */ {132, 8},
	/* idx == 389 */ {133, 8},
	/* idx == 390 */ {134, 8},
	/* idx == 391 */ {135, 8},
	/* idx == 392 */ {136, 8},
	/* idx == 393 */ {137, 8},
	/* idx == 394 */ {138, 8},
	/* idx == 395 */ {139, 8},
	/* idx == 396 */ {140, 8},
	/* idx == 397 */ {141, 8},
	/* idx == 398 */ {142, 8},
	/* idx == 399 */ {143, 8},
	/* idx == 400 */ {144, 8},
	/* idx == 401 */ {145, 8},
	/* idx == 402 */ {146, 8},
	/* idx == 403 */ {147, 8},
	/* idx == 404 */ {148, 8},
	/* idx == 405 */ {149, 8},
	/* idx == 406 */ {150, 8},
	/* idx == 407 */ {151, 8},
	/* idx == 408 */ {152, 8},
	/* idx == 409 */ {153, 8},
	/* idx == 410 */ {154, 8},
	/* idx == 411 */ {155, 8},
	/* idx == 412 */ {156, 8},
	/* idx == 413 */ {157, 8},
	/* idx == 414 */ {158, 8},
	/* idx == 415 */ {159, 8},
	/* idx == 416 */ {160, 8},
	/* idx == 417 */ {161, 8},
	/* idx == 418 */ {162, 8},
	/* idx == 419 */ {163, 8},
	/* idx == 420 */ {164, 8},
	/* idx == 421 */ {165, 8},
	/* idx == 422 */ {166, 8},
	/* idx == 423 */ {167, 8},
	/* idx == 424 */ {168, 8},
	/* idx == 425 */ {169, 8},
	/* idx == 426 */ {170, 8},
	/* idx == 427 */ {171, 8},
	/* idx == 428 */ {172, 8},
	/* idx == 429 */ {173, 8},
	/* idx == 430 */ {174, 8},
	/* idx == 431 */ {175, 8},
	/* idx == 432 */ {176, 8},
	/* idx == 433 */ {177, 8},
	/* idx == 434 */ {178, 8},
	/* idx == 435 */ {179, 8},
	/* idx == 436 */ {180, 8},
	/* idx == 437 */ {181, 8},
	/* idx == 438 */ {182, 8},
	/* idx == 439 */ {183, 8},
	/* idx == 440 */ {184, 8},
	/* idx == 441 */ {185, 8},
	/* idx == 442 */ {186, 8},
	/* idx == 443 */ {187, 8},
	/* idx == 444 */ {188, 8},
	/* idx == 445 */ {189, 8},
	/* idx == 446 */ {190, 8},
	/* idx == 447 */ {191, 8},
	/* idx == 448 */ {192, 8},
	/* idx == 449 */ {193, 8},
	/* idx == 450 */ {194, 8},
	/* idx == 451 */ {195, 8},
	/* idx == 452 */ {196, 8},
	/* idx == 453 */ {197, 8},
	/* idx == 454 */ {198, 8},
	/* idx == 455 */ {199, 8},
	/* idx == 456 */ {200, 8},
	/* idx == 457 */ {201, 8},
	/* idx == 458 */ {202, 8},
	/* idx == 459 */ {203, 8},
	/* idx == 460 */ {204, 8},
	/* idx == 461 */ {205, 8},
	/* idx == 462 */ {206, 8},
	/* idx == 463 */ {207, 8},
	/* idx == 464 */ {208, 8},
	/* idx == 465 */ {209, 8},
	/* idx == 466 */ {210, 8},
	/* idx == 467 */ {211, 8},
	/* idx == 468 */ {212, 8},
	/* idx == 469 */ {213, 8},
	/* idx == 470 */ {214, 8},
	/* idx == 471 */ {215, 8},
	/* idx == 472 */ {216, 8},
	/* idx == 473 */ {217, 8},
	/* idx == 474 */ {218, 8},
	/* idx == 475 */ {219, 8},
	/* idx == 476 */ {220, 8},
	/* idx == 477 */ {221, 8},
	/* idx == 478 */ {222, 8},
	/* idx == 479 */ {223, 8},
	/* idx == 480 */ {224, 8},
	/* idx == 481 */ {225, 8},
	/* idx == 482 */ {226, 8},
	/* idx == 483 */ {227, 8},
	/* idx == 484 */ {228, 8},
	/* idx == 485 */ {229, 8},
	/* idx == 486 */ {230, 8},
	/* idx == 487 */ {231, 8},
	/* idx == 488 */ {232, 8},
	/* idx == 489 */ {233, 8},
	/* idx == 490 */ {234, 8},
	/* idx == 491 */ {235, 8},
	/* idx == 492 */ {236, 8},
	/* idx == 493 */ {237, 8},
	/* idx == 494 */ {238, 8},
	/* idx == 495 */ {239, 8},
	/* idx == 496 */ {240, 8},
	/* idx == 497 */ {241, 8},
	/* idx == 498 */ {242, 8},
	/* idx == 499 */ {243, 8},
	/* idx == 500 */ {244, 8},
	/* idx == 501 */ {245, 8},
	/* idx == 502 */ {246, 8},
	/* idx == 503 */ {247, 8},
	/* idx == 504 */ {248, 8},
	/* idx == 505 */ {249, 8},
	/* idx == 506 */ {250, 8},
	/* idx == 507 */ {251, 8},
	/* idx == 508 */ {252, 8},
	/* idx == 509 */ {253, 8},
	/* idx == 510 */ {254, 8},
	/* idx == 511 */ {255, 8},
}
