package test

import (
	"fmt"
	"testing"

	"github.com/seancfoley/ipaddress-go/ipaddr"
)

//
// trie ElementContains
//

func BenchmarkIPv4ElementContains(b *testing.B) {
	benchIPv4AddressContains(b, ipaddr.NewIPAddressString("52.95.110.1").GetAddress().ToIPv4())
}

func BenchmarkIPv6ElementContains(b *testing.B) {
	benchIPv6AddressContains(b, ipaddr.NewIPAddressString("2620:107:300f::36b7:ff81").GetAddress().ToIPv6())
}

func BenchmarkIPv4ElementNotContains(b *testing.B) {
	benchIPv4AddressContains(b, ipaddr.NewIPAddressString("123.123.123.123").GetAddress().ToIPv4())
}

func BenchmarkIPv6ElementNotContains(b *testing.B) {
	benchIPv6AddressContains(b, ipaddr.NewIPAddressString("2620::ffff").GetAddress().ToIPv6())
}

func benchIPv4AddressContains(b *testing.B, addr *ipaddr.IPv4Address) {
	printOp("ElementContains %v %v\n", addr, ipv4Trie.ElementContains(addr))
	for n := 0; n < b.N; n++ {
		ipv4Trie.ElementContains(addr)
	}
}

func benchIPv6AddressContains(b *testing.B, addr *ipaddr.IPv6Address) {
	printOp("ElementContains %v %v\n", addr, ipv6Trie.ElementContains(addr))
	for n := 0; n < b.N; n++ {
		ipv6Trie.ElementContains(addr)
	}
}

//
// trie LongestPrefixMatch
//

func BenchmarkIPv4LongestPrefixMatch(b *testing.B) {
	benchIPv4LongestPrefixMatch(b, ipaddr.NewIPAddressString("52.95.110.1").GetAddress().ToIPv4())
}

func BenchmarkIPv6LongestPrefixMatch(b *testing.B) {
	benchIPv6LongestPrefixMatch(b, ipaddr.NewIPAddressString("2620:107:300f::36b7:ff81").GetAddress().ToIPv6())
}

func BenchmarkIPv4LongestPrefixNoMatch(b *testing.B) {
	benchIPv4LongestPrefixMatch(b, ipaddr.NewIPAddressString("123.123.123.123").GetAddress().ToIPv4())
}

func BenchmarkIPv6LongestPrefixNoMatch(b *testing.B) {
	benchIPv6LongestPrefixMatch(b, ipaddr.NewIPAddressString("2620::ffff").GetAddress().ToIPv6())
}

func benchIPv4LongestPrefixMatch(b *testing.B, addr *ipaddr.IPv4Address) {
	printOp("LongestPrefixMatch %v %v\n", addr, ipv4Trie.LongestPrefixMatch(addr))
	for n := 0; n < b.N; n++ {
		ipv4Trie.LongestPrefixMatch(addr)
	}
}

func benchIPv6LongestPrefixMatch(b *testing.B, addr *ipaddr.IPv6Address) {
	printOp("LongestPrefixMatch %v %v\n", addr, ipv6Trie.LongestPrefixMatch(addr))
	for n := 0; n < b.N; n++ {
		ipv6Trie.LongestPrefixMatch(addr)
	}
}

//
// trie ShortestPrefixMatch
//

func BenchmarkIPv4ShortestPrefixMatch(b *testing.B) {
	benchIPv4ShortestPrefixMatch(b, ipaddr.NewIPAddressString("52.95.110.1").GetAddress().ToIPv4())
}

func BenchmarkIPv6ShortestPrefixMatch(b *testing.B) {
	benchIPv6ShortestPrefixMatch(b, ipaddr.NewIPAddressString("2620:107:300f::36b7:ff81").GetAddress().ToIPv6())
}

func BenchmarkIPv4ShortestPrefixNoMatch(b *testing.B) {
	benchIPv4ShortestPrefixMatch(b, ipaddr.NewIPAddressString("123.123.123.123").GetAddress().ToIPv4())
}

func BenchmarkIPv6ShortestPrefixNoMatch(b *testing.B) {
	benchIPv6ShortestPrefixMatch(b, ipaddr.NewIPAddressString("2620::ffff").GetAddress().ToIPv6())
}

func benchIPv4ShortestPrefixMatch(b *testing.B, addr *ipaddr.IPv4Address) {
	printOp("ShortestPrefixMatch %v %v\n", addr, ipv4Trie.ShortestPrefixMatch(addr))
	for n := 0; n < b.N; n++ {
		ipv4Trie.ShortestPrefixMatch(addr)
	}
}

func benchIPv6ShortestPrefixMatch(b *testing.B, addr *ipaddr.IPv6Address) {
	printOp("ShortestPrefixMatch %v %v\n", addr, ipv6Trie.ShortestPrefixMatch(addr))
	for n := 0; n < b.N; n++ {
		ipv6Trie.ShortestPrefixMatch(addr)
	}
}

//
// trie ElementsContaining
//

func BenchmarkIPv4ElementsContaining(b *testing.B) {
	benchIPv4ElementsContaining(b, ipaddr.NewIPAddressString("52.95.110.1").GetAddress().ToIPv4())
}

func BenchmarkIPv6ElementsContaining(b *testing.B) {
	benchIPv6ElementsContaining(b, ipaddr.NewIPAddressString("2620:107:300f::36b7:ff81").GetAddress().ToIPv6())
}

func benchIPv4ElementsContaining(b *testing.B, addr *ipaddr.IPv4Address) {
	printOp("ElementsContaining %v %v\n", addr, ipv4Trie.ElementsContaining(addr))
	for n := 0; n < b.N; n++ {
		ipv4Trie.ElementsContaining(addr)
	}
}

func benchIPv6ElementsContaining(b *testing.B, addr *ipaddr.IPv6Address) {
	printOp("ElementsContaining %v %v\n", addr, ipv6Trie.ElementsContaining(addr))
	for n := 0; n < b.N; n++ {
		ipv6Trie.ElementsContaining(addr)
	}
}

//
// trie ElementsContainedBy
//

func BenchmarkIPv4ElementsContainedBy(b *testing.B) {
	benchIPv4ElementsContainedBy(b, ipaddr.NewIPAddressString("52.95.104.0/21").GetAddress().ToIPv4())
}

func BenchmarkIPv6ElementsContainedBy(b *testing.B) {
	benchIPv6ElementsContainedBy(b, ipaddr.NewIPAddressString("2a01:578:3::/80").GetAddress().ToIPv6())
}

func benchIPv4ElementsContainedBy(b *testing.B, addr *ipaddr.IPv4Address) {
	printOp("ElementsContainedBy %v %v\n", addr, ipv4Trie.ElementsContainedBy(addr))
	for n := 0; n < b.N; n++ {
		ipv4Trie.ElementsContainedBy(addr)
	}
}

func benchIPv6ElementsContainedBy(b *testing.B, addr *ipaddr.IPv6Address) {
	printOp("ElementsContainedBy %v %v\n", addr, ipv6Trie.ElementsContainedBy(addr))
	for n := 0; n < b.N; n++ {
		ipv6Trie.ElementsContainedBy(addr)
	}
}

//
// trie Contains
//

func BenchmarkIPv4Contains(b *testing.B) {
	benchIPv4Contains(b, ipaddr.NewIPAddressString("52.95.104.0/22").GetAddress().ToIPv4())
}

func BenchmarkIPv6Contains(b *testing.B) {
	benchIPv6Contains(b, ipaddr.NewIPAddressString("2a01:578:3::36e4:1000/122").GetAddress().ToIPv6())
}

func BenchmarkIPv4NotContains(b *testing.B) {
	benchIPv4Contains(b, ipaddr.NewIPAddressString("52.95.112.0/22").GetAddress().ToIPv4())
}

func BenchmarkIPv6NotContains(b *testing.B) {
	benchIPv6Contains(b, ipaddr.NewIPAddressString("2a01:578:3::36e4:1040/122").GetAddress().ToIPv6())
}

func benchIPv4Contains(b *testing.B, addr *ipaddr.IPv4Address) {
	printOp("Contains %v: %v\n", addr, ipv4Trie.Contains(addr))
	for n := 0; n < b.N; n++ {
		ipv4Trie.Contains(addr)
	}
}

func benchIPv6Contains(b *testing.B, addr *ipaddr.IPv6Address) {
	printOp("Contains %v: %v\n", addr, ipv6Trie.Contains(addr))
	for n := 0; n < b.N; n++ {
		ipv6Trie.Contains(addr)
	}
}

//
// trie Add and Remove
//

func BenchmarkIPv4AddAndRemove(b *testing.B) {
	benchIPv4AddAndRemove(b, ipaddr.NewIPAddressString("52.95.98.0/23").GetAddress().ToIPv4())
}

func BenchmarkIPv6AddAndRemove(b *testing.B) {
	benchIPv6AddAndRemove(b, ipaddr.NewIPAddressString("2a01:578:3::b022:9fe0/123").GetAddress().ToIPv6())
}

func benchIPv4AddAndRemove(b *testing.B, addr *ipaddr.IPv4Address) {
	printOp("Add %v %v\n", addr, ipv4Trie.Add(addr))
	printOp("Remove %v %v\n", addr, ipv4Trie.Remove(addr))
	for n := 0; n < b.N; n++ {
		ipv4Trie.Add(addr)
		ipv4Trie.Remove(addr)
	}
}

func benchIPv6AddAndRemove(b *testing.B, addr *ipaddr.IPv6Address) {
	printOp("Add %v %v\n", addr, ipv6Trie.Add(addr))
	printOp("Remove %v %v\n", addr, ipv6Trie.Remove(addr))
	for n := 0; n < b.N; n++ {
		ipv6Trie.Add(addr)
		ipv6Trie.Remove(addr)
	}
}

//
// trie Floor
//

func BenchmarkIPv4Floor(b *testing.B) {
	benchIPv4Floor(b, ipaddr.NewIPAddressString("52.95.112.0/22").GetAddress().ToIPv4())
}

func BenchmarkIPv6Floor(b *testing.B) {
	benchIPv6Floor(b, ipaddr.NewIPAddressString("2a01:578:3::36e4:1040/122").GetAddress().ToIPv6())
}

func benchIPv4Floor(b *testing.B, addr *ipaddr.IPv4Address) {
	printOp("Floor %v %v\n", addr, ipv4Trie.FloorAddedNode(addr).GetKey())
	for n := 0; n < b.N; n++ {
		ipv4Trie.FloorAddedNode(addr)
	}
}

func benchIPv6Floor(b *testing.B, addr *ipaddr.IPv6Address) {
	printOp("Floor %v %v\n", addr, ipv6Trie.FloorAddedNode(addr).GetKey())
	for n := 0; n < b.N; n++ {
		ipv6Trie.FloorAddedNode(addr)
	}
}

//
// trie Ceiling
//

func BenchmarkIPv4Ceiling(b *testing.B) {
	benchIPv4Ceiling(b, ipaddr.NewIPAddressString("52.95.112.0/22").GetAddress().ToIPv4())
}

func BenchmarkIPv6Ceiling(b *testing.B) {
	benchIPv6Ceiling(b, ipaddr.NewIPAddressString("2a01:578:3::36e4:1040/122").GetAddress().ToIPv6())
}

func benchIPv4Ceiling(b *testing.B, addr *ipaddr.IPv4Address) {
	printOp("Ceiling %v %v\n", addr, ipv4Trie.CeilingAddedNode(addr).GetKey())
	for n := 0; n < b.N; n++ {
		ipv4Trie.CeilingAddedNode(addr)
	}
}

func benchIPv6Ceiling(b *testing.B, addr *ipaddr.IPv6Address) {
	printOp("Ceiling %v %v\n", addr, ipv6Trie.CeilingAddedNode(addr).GetKey())
	for n := 0; n < b.N; n++ {
		ipv6Trie.CeilingAddedNode(addr)
	}
}

func printOp(format string, a ...any) {
	//fmt.Printf(format, a...)
}

func TestTrieCreation(t *testing.T) {
	fmt.Println(ipv4Trie.String())
	fmt.Println(ipv6Trie.String())
}

var ipv4Trie = constructIPv4Trie()
var ipv6Trie = constructIPv6Trie()

func constructIPv4Trie() *ipaddr.IPv4AddressTrie {
	trie := &ipaddr.Trie[*ipaddr.IPv4Address]{}
	for _, addrStr := range ipv4Prefs {
		addr, err := ipaddr.NewIPAddressString(addrStr).ToAddress()
		if err != nil {
			panic("failed to parse " + err.Error())
		}
		trie.Add(addr.ToIPv4())
	}
	return trie
}

func constructIPv6Trie() *ipaddr.IPv6AddressTrie {
	trie := &ipaddr.Trie[*ipaddr.IPv6Address]{}
	for _, addrStr := range ipv6Prefs {
		addr, err := ipaddr.NewIPAddressString(addrStr).ToAddress()
		if err != nil {
			panic("failed to parse " + err.Error())
		}
		trie.Add(addr.ToIPv6())
	}
	return trie
}

var ipv4Prefs = []string{
	"13.32.0.0/15", "13.54.0.0/15", "13.56.0.0/16", "13.57.0.0/16",
	"13.58.0.0/15", "13.112.0.0/14", "13.124.0.0/16", "13.125.0.0/16",
	"13.126.0.0/15", "13.209.0.0/16", "13.210.0.0/15", "13.228.0.0/15",
	"13.230.0.0/15", "13.232.0.0/14", "13.236.0.0/14", "13.250.0.0/15",
	"18.194.0.0/15", "18.196.0.0/15", "18.216.0.0/14", "18.220.0.0/14",
	"18.231.0.0/16", "23.20.0.0/14", "27.0.0.0/22", "34.192.0.0/12",
	"34.208.0.0/12", "34.224.0.0/12", "34.240.0.0/13", "34.248.0.0/13",
	"35.154.0.0/16", "35.155.0.0/16", "35.156.0.0/14", "35.160.0.0/13",
	"35.168.0.0/13", "35.176.0.0/15", "35.178.0.0/15", "35.180.0.0/15",
	"35.182.0.0/15", "43.250.192.0/24", "43.250.193.0/24", "46.51.128.0/18",
	"46.51.192.0/20", "46.51.216.0/21", "46.51.224.0/19", "46.137.0.0/17",
	"46.137.128.0/18", "46.137.192.0/19", "46.137.224.0/19", "50.16.0.0/15",
	"50.18.0.0/16", "50.19.0.0/16", "50.112.0.0/16", "52.0.0.0/15",
	"52.2.0.0/15", "52.4.0.0/14", "52.8.0.0/16", "52.9.0.0/16",
	"52.10.0.0/15", "52.12.0.0/15", "52.14.0.0/16", "52.15.0.0/16",
	"52.16.0.0/15", "52.18.0.0/15", "52.20.0.0/14", "52.24.0.0/14",
	"52.28.0.0/16", "52.29.0.0/16", "52.30.0.0/15", "52.32.0.0/14",
	"52.36.0.0/14", "52.40.0.0/14", "52.44.0.0/15", "52.46.0.0/18",
	"52.46.64.0/20", "52.46.80.0/21", "52.46.88.0/22", "52.46.92.0/22",
	"52.47.0.0/16", "52.48.0.0/14", "52.52.0.0/15", "52.54.0.0/15",
	"52.56.0.0/16", "52.57.0.0/16", "52.58.0.0/15", "52.60.0.0/16",
	"52.61.0.0/16", "52.62.0.0/15", "52.64.0.0/17", "52.64.128.0/17",
	"52.65.0.0/16", "52.66.0.0/16", "52.67.0.0/16", "52.68.0.0/15",
	"52.70.0.0/15", "52.72.0.0/15", "52.74.0.0/16", "52.76.0.0/17",
	"52.76.128.0/17", "52.77.0.0/16", "52.78.0.0/16", "52.79.0.0/16",
	"52.80.0.0/16", "52.82.187.0/24", "52.82.188.0/22", "52.82.196.0/22",
	"52.82.200.0/25", "52.82.204.0/23", "52.82.208.0/20", "52.83.0.0/16",
	"52.84.0.0/15", "52.86.0.0/15", "52.88.0.0/15", "52.90.0.0/15",
	"52.92.0.0/20", "52.92.16.0/20", "52.92.32.0/22", "52.92.39.0/24",
	"52.92.40.0/21", "52.92.48.0/22", "52.92.52.0/22", "52.92.56.0/22",
	"52.92.60.0/22", "52.92.64.0/22", "52.92.68.0/22", "52.92.72.0/22",
	"52.92.76.0/22", "52.92.80.0/22", "52.92.84.0/22", "52.92.88.0/22",
	"52.92.248.0/22", "52.92.252.0/22", "52.93.0.0/24", "52.93.1.0/24",
	"52.93.2.0/24", "52.93.3.0/24", "52.93.4.0/24", "52.93.5.0/24",
	"52.93.8.0/22", "52.93.16.0/24", "52.94.0.0/22", "52.94.4.0/24",
	"52.94.5.0/24", "52.94.6.0/24", "52.94.7.0/24", "52.94.8.0/24",
	"52.94.9.0/24", "52.94.10.0/24", "52.94.11.0/24", "52.94.12.0/24",
	"52.94.13.0/24", "52.94.14.0/24", "52.94.15.0/24", "52.94.16.0/24",
	"52.94.17.0/24", "52.94.20.0/24", "52.94.24.0/23", "52.94.26.0/23",
	"52.94.28.0/23", "52.94.30.0/23", "52.94.32.0/20", "52.94.48.0/20",
	"52.94.64.0/22", "52.94.72.0/22", "52.94.80.0/20", "52.94.96.0/20",
	"52.94.112.0/22", "52.94.192.0/22", "52.94.196.0/24", "52.94.197.0/24",
	"52.94.198.0/28", "52.94.198.16/28", "52.94.198.32/28", "52.94.198.48/28",
	"52.94.198.64/28", "52.94.198.80/28", "52.94.198.96/28", "52.94.198.112/28",
	"52.94.198.128/28", "52.94.198.144/28", "52.94.199.0/24", "52.94.200.0/24",
	"52.94.204.0/23", "52.94.206.0/23", "52.94.208.0/21", "52.94.216.0/21",
	"52.94.224.0/20", "52.94.240.0/22", "52.94.244.0/22", "52.94.248.0/28",
	"52.94.248.16/28", "52.94.248.32/28", "52.94.248.48/28", "52.94.248.64/28",
	"52.94.248.80/28", "52.94.248.96/28", "52.94.248.112/28", "52.94.248.128/28",
	"52.94.248.144/28", "52.94.248.160/28", "52.94.248.176/28", "52.94.248.192/28",
	"52.94.248.208/28", "52.94.248.224/28", "52.94.249.0/28", "52.94.249.16/28",
	"52.94.249.32/28", "52.94.249.64/28", "52.94.249.80/28", "52.94.252.0/23",
	"52.94.254.0/23", "52.95.0.0/20", "52.95.16.0/21", "52.95.24.0/22",
	"52.95.28.0/24", "52.95.30.0/23", "52.95.34.0/24", "52.95.35.0/24",
	"52.95.36.0/22", "52.95.40.0/24", "52.95.48.0/22", "52.95.56.0/22",
	"52.95.60.0/24", "52.95.61.0/24", "52.95.62.0/24", "52.95.63.0/24",
	"52.95.64.0/20", "52.95.80.0/20", "52.95.96.0/22", "52.95.100.0/22",
	"52.95.104.0/22", "52.95.108.0/23", "52.95.110.0/24", "52.95.111.0/24",
	"52.95.112.0/20", "52.95.128.0/21", "52.95.136.0/23", "52.95.138.0/24",
	"52.95.142.0/23", "52.95.144.0/24", "52.95.145.0/24", "52.95.146.0/23",
	"52.95.148.0/23", "52.95.150.0/24", "52.95.154.0/23", "52.95.156.0/24",
	"52.95.192.0/20", "52.95.212.0/22", "52.95.240.0/24", "52.95.241.0/24",
	"52.95.242.0/24", "52.95.243.0/24", "52.95.244.0/24", "52.95.245.0/24",
	"52.95.246.0/24", "52.95.247.0/24", "52.95.248.0/24", "52.95.249.0/24",
	"52.95.250.0/24", "52.95.251.0/24", "52.95.252.0/24", "52.95.253.0/24",
	"52.95.254.0/24", "52.95.255.0/28", "52.95.255.16/28", "52.95.255.32/28",
	"52.95.255.48/28", "52.95.255.64/28", "52.95.255.80/28", "52.95.255.96/28",
	"52.95.255.112/28", "52.95.255.128/28", "52.95.255.144/28", "52.119.160.0/20",
	"52.119.176.0/21", "52.119.184.0/22", "52.119.188.0/22", "52.119.192.0/22",
	"52.119.196.0/22", "52.119.206.0/23", "52.119.208.0/23", "52.119.212.0/23",
	"52.119.214.0/23", "52.119.216.0/21", "52.119.224.0/21", "52.119.232.0/21",
	"52.192.0.0/15", "52.196.0.0/14", "52.200.0.0/13", "52.208.0.0/13",
	"52.216.0.0/15", "52.218.0.0/17", "52.218.128.0/17", "52.219.0.0/20",
	"52.219.16.0/22", "52.219.20.0/22", "52.219.24.0/21", "52.219.32.0/21",
	"52.219.40.0/22", "52.219.44.0/22", "52.219.56.0/22", "52.219.60.0/23",
	"52.219.62.0/23", "52.219.64.0/22", "52.219.68.0/22", "52.219.72.0/22",
	"52.219.76.0/22", "52.219.80.0/20", "52.220.0.0/15", "52.222.0.0/17",
	"52.222.128.0/17", "54.64.0.0/15", "54.66.0.0/16", "54.67.0.0/16",
	"54.68.0.0/14", "54.72.0.0/15", "54.74.0.0/15", "54.76.0.0/15",
	"54.78.0.0/16", "54.79.0.0/16", "54.80.0.0/13", "54.88.0.0/14",
	"54.92.0.0/17", "54.92.128.0/17", "54.93.0.0/16", "54.94.0.0/16",
	"54.95.0.0/16", "54.144.0.0/14", "54.148.0.0/15", "54.150.0.0/16",
	"54.151.0.0/17", "54.151.128.0/17", "54.152.0.0/16", "54.153.0.0/17",
	"54.153.128.0/17", "54.154.0.0/16", "54.155.0.0/16", "54.156.0.0/14",
	"54.160.0.0/13", "54.168.0.0/16", "54.169.0.0/16", "54.170.0.0/15",
	"54.172.0.0/15", "54.174.0.0/15", "54.176.0.0/15", "54.178.0.0/16",
	"54.179.0.0/16", "54.182.0.0/16", "54.183.0.0/16", "54.184.0.0/13",
	"54.192.0.0/16", "54.193.0.0/16", "54.194.0.0/15", "54.196.0.0/15",
	"54.198.0.0/16", "54.199.0.0/16", "54.200.0.0/15", "54.202.0.0/15",
	"54.204.0.0/15", "54.206.0.0/16", "54.207.0.0/16", "54.208.0.0/15",
	"54.210.0.0/15", "54.212.0.0/15", "54.214.0.0/16", "54.215.0.0/16",
	"54.216.0.0/15", "54.218.0.0/16", "54.219.0.0/16", "54.220.0.0/16",
	"54.221.0.0/16", "54.222.0.0/19", "54.222.48.0/22", "54.222.57.0/24",
	"54.222.58.0/28", "54.222.128.0/17", "54.223.0.0/16", "54.224.0.0/15",
	"54.226.0.0/15", "54.228.0.0/16", "54.229.0.0/16", "54.230.0.0/16",
	"54.231.0.0/17", "54.231.128.0/19", "54.231.160.0/19", "54.231.192.0/20",
	"54.231.224.0/21", "54.231.232.0/21", "54.231.240.0/22", "54.231.244.0/22",
	"54.231.248.0/22", "54.231.252.0/24", "54.231.253.0/24", "54.232.0.0/16",
	"54.233.0.0/18", "54.233.64.0/18", "54.233.128.0/17", "54.234.0.0/15",
	"54.236.0.0/15", "54.238.0.0/16", "54.239.0.0/28", "54.239.0.16/28",
	"54.239.0.32/28", "54.239.0.48/28", "54.239.0.64/28", "54.239.0.80/28",
	"54.239.0.96/28", "54.239.0.112/28", "54.239.0.128/28", "54.239.0.144/28",
	"54.239.0.160/28", "54.239.0.176/28", "54.239.0.192/28", "54.239.0.208/28",
	"54.239.0.224/28", "54.239.0.240/28", "54.239.1.0/28", "54.239.1.16/28",
	"54.239.2.0/23", "54.239.4.0/22", "54.239.8.0/21", "54.239.16.0/20",
	"54.239.32.0/21", "54.239.48.0/22", "54.239.52.0/23", "54.239.54.0/23",
	"54.239.56.0/21", "54.239.96.0/24", "54.239.98.0/24", "54.239.99.0/24",
	"54.239.100.0/23", "54.239.104.0/23", "54.239.108.0/22", "54.239.116.0/22",
	"54.239.120.0/21", "54.239.128.0/18", "54.239.192.0/19", "54.240.128.0/18",
	"54.240.192.0/22", "54.240.196.0/24", "54.240.197.0/24", "54.240.198.0/24",
	"54.240.199.0/24", "54.240.200.0/24", "54.240.202.0/24", "54.240.203.0/24",
	"54.240.204.0/22", "54.240.208.0/22", "54.240.212.0/22", "54.240.216.0/22",
	"54.240.220.0/22", "54.240.225.0/24", "54.240.226.0/24", "54.240.227.0/24",
	"54.240.228.0/23", "54.240.230.0/23", "54.240.232.0/22", "54.240.244.0/22",
	"54.240.248.0/21", "54.241.0.0/16", "54.242.0.0/15", "54.244.0.0/16",
	"54.245.0.0/16", "54.246.0.0/16", "54.247.0.0/16", "54.248.0.0/15",
	"54.250.0.0/16", "54.251.0.0/16", "54.252.0.0/16", "54.253.0.0/16",
	"54.254.0.0/16", "54.255.0.0/16", "67.202.0.0/18", "72.21.192.0/19",
	"72.44.32.0/19", "75.101.128.0/17", "79.125.0.0/17", "87.238.80.0/21",
	"96.127.0.0/17", "103.4.8.0/22", "103.4.12.0/22", "103.8.172.0/22",
	"103.246.148.0/23", "103.246.150.0/23", "107.20.0.0/14", "122.248.192.0/18",
	"172.96.97.0/24", "172.96.98.0/24", "174.129.0.0/16", "175.41.128.0/18",
	"175.41.192.0/18", "176.32.64.0/19", "176.32.96.0/21", "176.32.104.0/21",
	"176.32.112.0/21", "176.32.120.0/22", "176.32.125.0/25", "176.34.0.0/19",
	"176.34.32.0/19", "176.34.64.0/18", "176.34.128.0/17", "177.71.128.0/17",
	"177.72.240.0/21", "178.236.0.0/20", "184.72.0.0/18", "184.72.64.0/18",
	"184.72.128.0/17", "184.73.0.0/16", "184.169.128.0/17", "185.48.120.0/22",
	"185.143.16.0/24", "203.83.220.0/22", "204.236.128.0/18", "204.236.192.0/18",
	"204.246.160.0/22", "204.246.164.0/22", "204.246.168.0/22", "204.246.174.0/23",
	"204.246.176.0/20", "205.251.192.0/19", "205.251.224.0/22", "205.251.228.0/22",
	"205.251.232.0/22", "205.251.236.0/22", "205.251.240.0/22", "205.251.244.0/23",
	"205.251.247.0/24", "205.251.248.0/24", "205.251.249.0/24", "205.251.250.0/23",
	"205.251.252.0/23", "205.251.254.0/24", "207.171.160.0/20", "207.171.176.0/20",
	"216.137.32.0/19", "216.182.224.0/20", "54.183.255.128/26", "54.228.16.0/26",
	"54.232.40.64/26", "54.241.32.64/26", "54.243.31.192/26", "54.244.52.192/26",
	"54.245.168.0/26", "54.248.220.0/26", "54.250.253.192/26", "54.251.31.128/26",
	"54.252.79.128/26", "54.252.254.192/26", "54.255.254.192/26", "107.23.255.0/26",
	"176.34.159.192/26", "177.71.207.128/26", "52.82.188.0/22", "52.92.0.0/20",
	"52.92.16.0/20", "52.92.32.0/22", "52.92.39.0/24", "52.92.40.0/21",
	"52.92.48.0/22", "52.92.52.0/22", "52.92.56.0/22", "52.92.60.0/22",
	"52.92.64.0/22", "52.92.68.0/22", "52.92.72.0/22", "52.92.76.0/22",
	"52.92.80.0/22", "52.92.84.0/22", "52.92.88.0/22", "52.92.248.0/22",
	"52.92.252.0/22", "52.95.128.0/21", "52.95.136.0/23", "52.95.138.0/24",
	"52.95.142.0/23", "52.95.144.0/24", "52.95.145.0/24", "52.95.146.0/23",
	"52.95.148.0/23", "52.95.150.0/24", "52.95.154.0/23", "52.95.156.0/24",
	"52.216.0.0/15", "52.218.0.0/17", "52.218.128.0/17", "52.219.0.0/20",
	"52.219.16.0/22", "52.219.20.0/22", "52.219.24.0/21", "52.219.32.0/21",
	"52.219.40.0/22", "52.219.44.0/22", "52.219.56.0/22", "52.219.60.0/23",
	"52.219.62.0/23", "52.219.64.0/22", "52.219.68.0/22", "52.219.72.0/22",
	"52.219.76.0/22", "52.219.80.0/20", "54.222.20.0/22", "54.222.48.0/22",
	"54.231.0.0/17", "54.231.128.0/19", "54.231.160.0/19", "54.231.192.0/20",
	"54.231.224.0/21", "54.231.232.0/21", "54.231.240.0/22", "54.231.248.0/22",
	"54.231.252.0/24", "54.231.253.0/24", "13.54.0.0/15", "13.56.0.0/16",
	"13.57.0.0/16", "13.58.0.0/15", "13.112.0.0/14", "13.124.0.0/16",
	"13.125.0.0/16", "13.126.0.0/15", "13.209.0.0/16", "13.210.0.0/15",
	"13.228.0.0/15", "13.230.0.0/15", "13.232.0.0/14", "13.236.0.0/14",
	"13.250.0.0/15", "18.194.0.0/15", "18.196.0.0/15", "18.216.0.0/14",
	"18.220.0.0/14", "18.231.0.0/16", "23.20.0.0/14", "34.192.0.0/12",
	"34.208.0.0/12", "34.224.0.0/12", "34.240.0.0/13", "34.248.0.0/13",
	"35.154.0.0/16", "35.155.0.0/16", "35.156.0.0/14", "35.160.0.0/13",
	"35.168.0.0/13", "35.176.0.0/15", "35.178.0.0/15", "35.180.0.0/15",
	"35.182.0.0/15", "46.51.128.0/18", "46.51.192.0/20", "46.51.216.0/21",
	"46.51.224.0/19", "46.137.0.0/17", "46.137.128.0/18", "46.137.192.0/19",
	"46.137.224.0/19", "50.16.0.0/15", "50.18.0.0/16", "50.19.0.0/16",
	"50.112.0.0/16", "52.0.0.0/15", "52.2.0.0/15", "52.4.0.0/14",
	"52.8.0.0/16", "52.9.0.0/16", "52.10.0.0/15", "52.12.0.0/15",
	"52.14.0.0/16", "52.15.0.0/16", "52.16.0.0/15", "52.18.0.0/15",
	"52.20.0.0/14", "52.24.0.0/14", "52.28.0.0/16", "52.29.0.0/16",
	"52.30.0.0/15", "52.32.0.0/14", "52.36.0.0/14", "52.40.0.0/14",
	"52.44.0.0/15", "52.47.0.0/16", "52.48.0.0/14", "52.52.0.0/15",
	"52.54.0.0/15", "52.56.0.0/16", "52.57.0.0/16", "52.58.0.0/15",
	"52.60.0.0/16", "52.61.0.0/16", "52.62.0.0/15", "52.64.0.0/17",
	"52.64.128.0/17", "52.65.0.0/16", "52.66.0.0/16", "52.67.0.0/16",
	"52.68.0.0/15", "52.70.0.0/15", "52.72.0.0/15", "52.74.0.0/16",
	"52.76.0.0/17", "52.76.128.0/17", "52.77.0.0/16", "52.78.0.0/16",
	"52.79.0.0/16", "52.80.0.0/16", "52.83.0.0/16", "52.86.0.0/15",
	"52.88.0.0/15", "52.90.0.0/15", "52.95.240.0/24", "52.95.241.0/24",
	"52.95.242.0/24", "52.95.243.0/24", "52.95.244.0/24", "52.95.245.0/24",
	"52.95.246.0/24", "52.95.247.0/24", "52.95.248.0/24", "52.95.249.0/24",
	"52.95.250.0/24", "52.95.251.0/24", "52.95.252.0/24", "52.95.253.0/24",
	"52.95.254.0/24", "52.95.255.0/28", "52.95.255.16/28", "52.95.255.32/28",
	"52.95.255.48/28", "52.95.255.64/28", "52.95.255.80/28", "52.95.255.96/28",
	"52.95.255.112/28", "52.95.255.128/28", "52.95.255.144/28", "52.192.0.0/15",
	"52.196.0.0/14", "52.200.0.0/13", "52.208.0.0/13", "52.220.0.0/15",
	"52.222.0.0/17", "54.64.0.0/15", "54.66.0.0/16", "54.67.0.0/16",
	"54.68.0.0/14", "54.72.0.0/15", "54.74.0.0/15", "54.76.0.0/15",
	"54.78.0.0/16", "54.79.0.0/16", "54.80.0.0/13", "54.88.0.0/14",
	"54.92.0.0/17", "54.92.128.0/17", "54.93.0.0/16", "54.94.0.0/16",
	"54.95.0.0/16", "54.144.0.0/14", "54.148.0.0/15", "54.150.0.0/16",
	"54.151.0.0/17", "54.151.128.0/17", "54.152.0.0/16", "54.153.0.0/17",
	"54.153.128.0/17", "54.154.0.0/16", "54.155.0.0/16", "54.156.0.0/14",
	"54.160.0.0/13", "54.168.0.0/16", "54.169.0.0/16", "54.170.0.0/15",
	"54.172.0.0/15", "54.174.0.0/15", "54.176.0.0/15", "54.178.0.0/16",
	"54.179.0.0/16", "54.183.0.0/16", "54.184.0.0/13", "54.193.0.0/16",
	"54.194.0.0/15", "54.196.0.0/15", "54.198.0.0/16", "54.199.0.0/16",
	"54.200.0.0/15", "54.202.0.0/15", "54.204.0.0/15", "54.206.0.0/16",
	"54.207.0.0/16", "54.208.0.0/15", "54.210.0.0/15", "54.212.0.0/15",
	"54.214.0.0/16", "54.215.0.0/16", "54.216.0.0/15", "54.218.0.0/16",
	"54.219.0.0/16", "54.220.0.0/16", "54.221.0.0/16", "54.222.128.0/17",
	"54.223.0.0/16", "54.224.0.0/15", "54.226.0.0/15", "54.228.0.0/16",
	"54.229.0.0/16", "54.232.0.0/16", "54.233.0.0/18", "54.233.64.0/18",
	"54.233.128.0/17", "54.234.0.0/15", "54.236.0.0/15", "54.238.0.0/16",
	"54.241.0.0/16", "54.242.0.0/15", "54.244.0.0/16", "54.245.0.0/16",
	"54.246.0.0/16", "54.247.0.0/16", "54.248.0.0/15", "54.250.0.0/16",
	"54.251.0.0/16", "54.252.0.0/16", "54.253.0.0/16", "54.254.0.0/16",
	"54.255.0.0/16", "67.202.0.0/18", "72.44.32.0/19", "75.101.128.0/17",
	"79.125.0.0/17", "96.127.0.0/17", "103.4.8.0/22", "103.4.12.0/22",
	"107.20.0.0/14", "122.248.192.0/18", "174.129.0.0/16", "175.41.128.0/18",
	"175.41.192.0/18", "176.32.64.0/19", "176.34.0.0/19", "176.34.32.0/19",
	"176.34.64.0/18", "176.34.128.0/17", "177.71.128.0/17", "184.72.0.0/18",
	"184.72.64.0/18", "184.72.128.0/17", "184.73.0.0/16", "184.169.128.0/17",
	"185.48.120.0/22", "204.236.128.0/18", "204.236.192.0/18", "216.182.224.0/20",
	"52.95.110.0/24", "205.251.192.0/21", "13.32.0.0/15", "13.54.63.128/26",
	"13.59.250.0/26", "13.113.203.0/24", "13.228.69.0/24", "34.195.252.0/24",
	"34.226.14.0/24", "35.158.136.0/24", "35.162.63.192/26", "35.167.191.128/26",
	"52.15.127.128/26", "52.46.0.0/18", "52.52.191.128/26", "52.56.127.0/25",
	"52.57.254.0/24", "52.66.194.128/26", "52.78.247.128/26", "52.84.0.0/15",
	"52.199.127.192/26", "52.212.248.0/26", "52.220.191.0/26", "52.222.128.0/17",
	"54.182.0.0/16", "54.192.0.0/16", "54.230.0.0/16", "54.233.255.128/26",
	"54.239.128.0/18", "54.239.192.0/19", "54.240.128.0/18", "204.246.164.0/22",
	"204.246.168.0/22", "204.246.174.0/23", "204.246.176.0/20", "205.251.192.0/19",
	"205.251.249.0/24", "205.251.250.0/23", "205.251.252.0/23", "205.251.254.0/24",
	"216.137.32.0/19",
}
var ipv6Prefs = []string{
	"2400:6500:0:7000::/56", "2400:6500:0:7100::/56", "2400:6500:0:7200::/56", "2400:6500:0:7400::/56",
	"2400:6500:0:7500::/56", "2400:6500:100:7100::/56", "2400:6500:100:7200::/56", "2400:6500:ff00::/64",
	"2400:6700:ff00::/64", "2403:b300:ff00::/64", "2406:da00:2000::/40", "2406:da00:4000::/40",
	"2406:da00:8000::/40", "2406:da00:a000::/40", "2406:da00:c000::/40", "2406:da00:ff00::/64",
	"2406:da12::/36", "2406:da14::/36", "2406:da18::/36", "2406:da1a::/36",
	"2406:da1c::/36", "2406:daa0:2000::/40", "2406:daa0:4000::/40", "2406:daa0:8000::/40",
	"2406:daa0:a000::/40", "2406:daa0:c000::/40", "2406:daf8:2000::/40", "2406:daf8:4000::/40",
	"2406:daf8:8000::/40", "2406:daf8:a000::/40", "2406:daf8:c000::/40", "2406:daf9:2000::/40",
	"2406:daf9:4000::/40", "2406:daf9:8000::/40", "2406:daf9:a000::/40", "2406:daf9:c000::/40",
	"2406:dafa:2000::/40", "2406:dafa:4000::/40", "2406:dafa:8000::/40", "2406:dafa:a000::/40",
	"2406:dafa:c000::/40", "2406:dafc:2000::/40", "2406:dafc:4000::/40", "2406:dafc:8000::/40",
	"2406:dafc:a000::/40", "2406:dafc:c000::/40", "2406:dafe:2000::/40", "2406:dafe:4000::/40",
	"2406:dafe:8000::/40", "2406:dafe:a000::/40", "2406:dafe:c000::/40", "2406:daff:2000::/40",
	"2406:daff:4000::/40", "2406:daff:8000::/40", "2406:daff:a000::/40", "2406:daff:c000::/40",
	"240f:8000:4000::/40", "240f:8000:8000::/40", "240f:8014::/36", "240f:8018::/36",
	"240f:80a0:4000::/40", "240f:80a0:8000::/40", "240f:80f8:4000::/40", "240f:80f8:8000::/40",
	"240f:80f9:4000::/40", "240f:80f9:8000::/40", "240f:80fa:4000::/40", "240f:80fa:8000::/40",
	"240f:80fc:4000::/40", "240f:80fc:8000::/40", "240f:80fe:4000::/40", "240f:80fe:8000::/40",
	"240f:80ff:4000::/40", "240f:80ff:8000::/40", "2600:1f00:1000::/40", "2600:1f00:2000::/40",
	"2600:1f00:4000::/40", "2600:1f00:6000::/40", "2600:1f00:8000::/40", "2600:1f00:c000::/40",
	"2600:1f00:e000::/40", "2600:1f11::/36", "2600:1f12::/36", "2600:1f14::/35",
	"2600:1f16::/36", "2600:1f18::/33", "2600:1f1c::/36", "2600:1f1e::/36",
	"2600:1fa0:1000::/40", "2600:1fa0:2000::/40", "2600:1fa0:4000::/40", "2600:1fa0:6000::/40",
	"2600:1fa0:8000::/40", "2600:1fa0:c000::/40", "2600:1fa0:e000::/40", "2600:1ff8:1000::/40",
	"2600:1ff8:2000::/40", "2600:1ff8:4000::/40", "2600:1ff8:6000::/40", "2600:1ff8:8000::/40",
	"2600:1ff8:c000::/40", "2600:1ff8:e000::/40", "2600:1ff9:1000::/40", "2600:1ff9:2000::/40",
	"2600:1ff9:4000::/40", "2600:1ff9:6000::/40", "2600:1ff9:8000::/40", "2600:1ff9:c000::/40",
	"2600:1ff9:e000::/40", "2600:1ffa:1000::/40", "2600:1ffa:2000::/40", "2600:1ffa:4000::/40",
	"2600:1ffa:6000::/40", "2600:1ffa:8000::/40", "2600:1ffa:c000::/40", "2600:1ffa:e000::/40",
	"2600:1ffc:1000::/40", "2600:1ffc:2000::/40", "2600:1ffc:4000::/40", "2600:1ffc:6000::/40",
	"2600:1ffc:8000::/40", "2600:1ffc:c000::/40", "2600:1ffc:e000::/40", "2600:1ffe:1000::/40",
	"2600:1ffe:2000::/40", "2600:1ffe:4000::/40", "2600:1ffe:6000::/40", "2600:1ffe:8000::/40",
	"2600:1ffe:c000::/40", "2600:1ffe:e000::/40", "2600:1fff:1000::/40", "2600:1fff:2000::/40",
	"2600:1fff:4000::/40", "2600:1fff:6000::/40", "2600:1fff:8000::/40", "2600:1fff:c000::/40",
	"2600:1fff:e000::/40", "2600:9000::/28", "2620:107:300f::/64", "2620:107:4000:5::/64",
	"2620:107:4000:7000::/56", "2620:107:4000:7100::/56", "2620:107:4000:7200::/56", "2620:107:4000:7400::/56",
	"2620:107:4000:7700::/56", "2620:107:4000:7800::/56", "2620:108:7000::/44", "2620:108:d000::/44",
	"2804:800:0:7000::/56", "2804:800:ff00::/64", "2a01:578:0:7000::/56", "2a01:578:0:7100::/56",
	"2a01:578:0:7200::/56", "2a01:578:3::/64", "2a01:578:13::/64", "2a05:d000:2000::/40",
	"2a05:d000:4000::/40", "2a05:d000:8000::/40", "2a05:d000:c000::/40", "2a05:d012::/36",
	"2a05:d014::/36", "2a05:d018::/36", "2a05:d01c::/36", "2a05:d050:2000::/40",
	"2a05:d050:4000::/40", "2a05:d050:8000::/40", "2a05:d050:c000::/40", "2a05:d078:2000::/40",
	"2a05:d078:4000::/40", "2a05:d078:8000::/40", "2a05:d078:c000::/40", "2a05:d079:2000::/40",
	"2a05:d079:4000::/40", "2a05:d079:8000::/40", "2a05:d079:c000::/40", "2a05:d07a:2000::/40",
	"2a05:d07a:4000::/40", "2a05:d07a:8000::/40", "2a05:d07a:c000::/40", "2a05:d07c:2000::/40",
	"2a05:d07c:4000::/40", "2a05:d07c:8000::/40", "2a05:d07c:c000::/40", "2a05:d07e:2000::/40",
	"2a05:d07e:4000::/40", "2a05:d07e:8000::/40", "2a05:d07e:c000::/40", "2a05:d07f:2000::/40",
	"2a05:d07f:4000::/40", "2a05:d07f:8000::/40", "2a05:d07f:c000::/40", "2400:6500:ff00::36fb:1f80/122",
	"2400:6500:ff00::36ff:fec0/122", "2400:6700:ff00::36f8:dc00/122", "2400:6700:ff00::36fa:fdc0/122", "2403:b300:ff00::36fc:4f80/122",
	"2403:b300:ff00::36fc:fec0/122", "2406:da00:ff00::36f3:1fc0/122", "2406:da00:ff00::6b17:ff00/122", "2406:da14:7ff:f800::/53",
	"2406:da14:fff:f800::/53", "2406:da18:7ff:f800::/53", "2406:da18:fff:f800::/53", "2406:da1c:7ff:f800::/53",
	"2406:da1c:fff:f800::/53", "2600:1f14:7ff:f800::/53", "2600:1f14:fff:f800::/53", "2600:1f18:3fff:f800::/53",
	"2600:1f18:7fff:f800::/53", "2600:1f1c:7ff:f800::/53", "2600:1f1c:fff:f800::/53", "2600:1f1e:7ff:f800::/53",
	"2600:1f1e:fff:f800::/53", "2620:107:300f::36b7:ff80/122", "2620:107:300f::36f1:2040/122", "2620:108:700f::36f4:34c0/122",
	"2620:108:700f::36f5:a800/122", "2804:800:ff00::36e8:2840/122", "2804:800:ff00::b147:cf80/122", "2a01:578:3::36e4:1000/122",
	"2a01:578:3::b022:9fc0/122", "2a05:d018:7ff:f800::/53", "2a05:d018:fff:f800::/53", "2406:daa0:2000::/40",
	"2406:daa0:4000::/40", "2406:daa0:8000::/40", "2406:daa0:a000::/40", "2406:daa0:c000::/40",
	"2406:daf8:2000::/40", "2406:daf8:4000::/40", "2406:daf8:8000::/40", "2406:daf8:a000::/40",
	"2406:daf8:c000::/40", "2406:daf9:2000::/40", "2406:daf9:4000::/40", "2406:daf9:8000::/40",
	"2406:daf9:a000::/40", "2406:daf9:c000::/40", "2406:dafa:2000::/40", "2406:dafa:4000::/40",
	"2406:dafa:8000::/40", "2406:dafa:a000::/40", "2406:dafa:c000::/40", "240f:80a0:4000::/40",
	"240f:80a0:8000::/40", "240f:80f8:4000::/40", "240f:80f8:8000::/40", "240f:80f9:4000::/40",
	"240f:80f9:8000::/40", "240f:80fa:4000::/40", "240f:80fa:8000::/40", "2600:1fa0:1000::/40",
	"2600:1fa0:2000::/40", "2600:1fa0:4000::/40", "2600:1fa0:6000::/40", "2600:1fa0:8000::/40",
	"2600:1fa0:c000::/40", "2600:1fa0:e000::/40", "2600:1ff8:1000::/40", "2600:1ff8:2000::/40",
	"2600:1ff8:4000::/40", "2600:1ff8:6000::/40", "2600:1ff8:8000::/40", "2600:1ff8:c000::/40",
	"2600:1ff8:e000::/40", "2600:1ff9:1000::/40", "2600:1ff9:2000::/40", "2600:1ff9:4000::/40",
	"2600:1ff9:6000::/40", "2600:1ff9:8000::/40", "2600:1ff9:c000::/40", "2600:1ff9:e000::/40",
	"2600:1ffa:1000::/40", "2600:1ffa:2000::/40", "2600:1ffa:4000::/40", "2600:1ffa:6000::/40",
	"2600:1ffa:8000::/40", "2600:1ffa:c000::/40", "2600:1ffa:e000::/40", "2a05:d050:2000::/40",
	"2a05:d050:4000::/40", "2a05:d050:8000::/40", "2a05:d050:c000::/40", "2a05:d078:2000::/40",
	"2a05:d078:4000::/40", "2a05:d078:8000::/40", "2a05:d078:c000::/40", "2a05:d079:2000::/40",
	"2a05:d079:4000::/40", "2a05:d079:8000::/40", "2a05:d079:c000::/40", "2a05:d07a:2000::/40",
	"2a05:d07a:4000::/40", "2a05:d07a:8000::/40", "2a05:d07a:c000::/40", "2400:6500:ff00::/64",
	"2400:6700:ff00::/64", "2403:b300:ff00::/64", "2406:da00:2000::/40", "2406:da00:4000::/40",
	"2406:da00:8000::/40", "2406:da00:a000::/40", "2406:da00:c000::/40", "2406:da00:ff00::/64",
	"2406:da12::/36", "2406:da14::/36", "2406:da18::/36", "2406:da1a::/36",
	"2406:da1c::/36", "2406:daff:2000::/40", "2406:daff:4000::/40", "2406:daff:8000::/40",
	"2406:daff:a000::/40", "2406:daff:c000::/40", "240f:8000:4000::/40", "240f:8000:8000::/40",
	"240f:8014::/36", "240f:8018::/36", "240f:80ff:4000::/40", "240f:80ff:8000::/40",
	"2600:1f00:1000::/40", "2600:1f00:2000::/40", "2600:1f00:4000::/40", "2600:1f00:6000::/40",
	"2600:1f00:8000::/40", "2600:1f00:c000::/40", "2600:1f00:e000::/40", "2600:1f11::/36",
	"2600:1f12::/36", "2600:1f14::/35", "2600:1f16::/36", "2600:1f18::/33",
	"2600:1f1c::/36", "2600:1f1e::/36", "2600:1fff:1000::/40", "2600:1fff:2000::/40",
	"2600:1fff:4000::/40", "2600:1fff:6000::/40", "2600:1fff:8000::/40", "2600:1fff:c000::/40",
	"2600:1fff:e000::/40", "2620:107:300f::/64", "2620:108:700f::/64", "2620:108:d00f::/64",
	"2804:800:ff00::/64", "2a01:578:3::/64", "2a01:578:13::/64", "2a05:d000:2000::/40",
	"2a05:d000:4000::/40", "2a05:d000:8000::/40", "2a05:d000:c000::/40", "2a05:d012::/36",
	"2a05:d014::/36", "2a05:d018::/36", "2a05:d01c::/36", "2a05:d07f:2000::/40",
	"2a05:d07f:4000::/40", "2a05:d07f:8000::/40", "2a05:d07f:c000::/40", "2600:9000::/28",
}
