"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.WORKFLOW_STEP_TYPE = exports.WORKFLOW_STEP_TO_RESOURCE_TYPE_MAP = exports.WORKFLOW_STATE = exports.WORKFLOW_RESOURCE_TYPE = exports.MODEL_STATE = exports.MODEL_ALGORITHM = void 0;
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
/**
 ********** WORKFLOW TYPES/INTERFACES **********
 */
/**
 ********** MODAL SUB-FORM TYPES/INTERFACES **********
 We persist sub-forms in the form modals s.t. data is only
 saved back to the parent form if the user explicitly saves.

 We define the structure of the forms here.
 */
// Ingest docs modal
// Search request modal
// Configure template modal
// Configure expression modal
// Configure multi-expression modal
/**
 ********** WORKSPACE TYPES/INTERFACES **********
 */
/**
 * An input field for a component. Specifies enough configuration for the
 * UI node to render it properly (help text, links, etc.)
 */
/**
 * The base interface the components will implement.
 */
/**
 * We need to include some extra instance-specific data to the ReactFlow component
 * to perform extra functionality, such as deleting the node from the ReactFlowInstance.
 */
/**
 ********** USE CASE TEMPLATE TYPES/INTERFACES **********
 */
// A stateless template of a workflow
// An instance of a workflow based on a workflow template
/**
 ********** ML PLUGIN TYPES/INTERFACES **********
 */
// Based off of https://github.com/opensearch-project/ml-commons/blob/main/common/src/main/java/org/opensearch/ml/common/model/MLModelState.java
let MODEL_STATE = exports.MODEL_STATE = /*#__PURE__*/function (MODEL_STATE) {
  MODEL_STATE["REGISTERED"] = "Registered";
  MODEL_STATE["REGISTERING"] = "Registering";
  MODEL_STATE["DEPLOYING"] = "Deploying";
  MODEL_STATE["DEPLOYED"] = "Deployed";
  MODEL_STATE["PARTIALLY_DEPLOYED"] = "Partially deployed";
  MODEL_STATE["UNDEPLOYED"] = "Undeployed";
  MODEL_STATE["DEPLOY_FAILED"] = "Deploy failed";
  return MODEL_STATE;
}({}); // Based off of https://github.com/opensearch-project/ml-commons/blob/main/common/src/main/java/org/opensearch/ml/common/FunctionName.java
let MODEL_ALGORITHM = exports.MODEL_ALGORITHM = /*#__PURE__*/function (MODEL_ALGORITHM) {
  MODEL_ALGORITHM["LINEAR_REGRESSION"] = "Linear regression";
  MODEL_ALGORITHM["KMEANS"] = "K-means";
  MODEL_ALGORITHM["AD_LIBSVM"] = "AD LIBSVM";
  MODEL_ALGORITHM["SAMPLE_ALGO"] = "Sample algorithm";
  MODEL_ALGORITHM["LOCAL_SAMPLE_CALCULATOR"] = "Local sample calculator";
  MODEL_ALGORITHM["FIT_RCF"] = "Fit RCF";
  MODEL_ALGORITHM["BATCH_RCF"] = "Batch RCF";
  MODEL_ALGORITHM["ANOMALY_LOCALIZATION"] = "Anomaly localization";
  MODEL_ALGORITHM["RCF_SUMMARIZE"] = "RCF summarize";
  MODEL_ALGORITHM["LOGISTIC_REGRESSION"] = "Logistic regression";
  MODEL_ALGORITHM["TEXT_EMBEDDING"] = "Text embedding";
  MODEL_ALGORITHM["METRICS_CORRELATION"] = "Metrics correlation";
  MODEL_ALGORITHM["REMOTE"] = "Remote";
  MODEL_ALGORITHM["SPARSE_ENCODING"] = "Sparse encoding";
  MODEL_ALGORITHM["SPARSE_TOKENIZE"] = "Sparse tokenize";
  MODEL_ALGORITHM["TEXT_SIMILARITY"] = "Text similarity";
  MODEL_ALGORITHM["QUESTION_ANSWERING"] = "Question answering";
  MODEL_ALGORITHM["AGENT"] = "Agent";
  return MODEL_ALGORITHM;
}({}); // Based off of JSONSchema. For more info, see https://json-schema.org/understanding-json-schema/reference/type
// For rendering options, we extract the name (the key in the input/output obj) and combine into a single obj.
// Also persist an optional field to dynamically run / not run validation on the form later on
/**
 ********** MISC TYPES/INTERFACES ************
 */
// Based off of https://github.com/opensearch-project/flow-framework/blob/main/src/main/java/org/opensearch/flowframework/model/State.java
let WORKFLOW_STATE = exports.WORKFLOW_STATE = /*#__PURE__*/function (WORKFLOW_STATE) {
  WORKFLOW_STATE["NOT_STARTED"] = "Not started";
  WORKFLOW_STATE["PROVISIONING"] = "Provisioning";
  WORKFLOW_STATE["FAILED"] = "Failed";
  WORKFLOW_STATE["COMPLETED"] = "Completed";
  return WORKFLOW_STATE;
}({});
// Based off of https://github.com/opensearch-project/flow-framework/blob/main/src/main/java/org/opensearch/flowframework/common/WorkflowResources.java
let WORKFLOW_RESOURCE_TYPE = exports.WORKFLOW_RESOURCE_TYPE = /*#__PURE__*/function (WORKFLOW_RESOURCE_TYPE) {
  WORKFLOW_RESOURCE_TYPE["PIPELINE_ID"] = "Ingest pipeline";
  WORKFLOW_RESOURCE_TYPE["INDEX_NAME"] = "Index";
  WORKFLOW_RESOURCE_TYPE["MODEL_ID"] = "Model";
  WORKFLOW_RESOURCE_TYPE["MODEL_GROUP_ID"] = "Model group";
  WORKFLOW_RESOURCE_TYPE["CONNECTOR_ID"] = "Connector";
  return WORKFLOW_RESOURCE_TYPE;
}({});
let WORKFLOW_STEP_TYPE = exports.WORKFLOW_STEP_TYPE = /*#__PURE__*/function (WORKFLOW_STEP_TYPE) {
  WORKFLOW_STEP_TYPE["CREATE_INGEST_PIPELINE_STEP_TYPE"] = "create_ingest_pipeline";
  WORKFLOW_STEP_TYPE["CREATE_SEARCH_PIPELINE_STEP_TYPE"] = "create_search_pipeline";
  WORKFLOW_STEP_TYPE["CREATE_INDEX_STEP_TYPE"] = "create_index";
  return WORKFLOW_STEP_TYPE;
}({}); // We cannot disambiguate ingest vs. search pipelines based on workflow resource type. To work around
// that, we maintain this map from workflow step type -> formatted type
let WORKFLOW_STEP_TO_RESOURCE_TYPE_MAP = exports.WORKFLOW_STEP_TO_RESOURCE_TYPE_MAP = /*#__PURE__*/function (WORKFLOW_STEP_TO_RESOURCE_TYPE_MAP) {
  WORKFLOW_STEP_TO_RESOURCE_TYPE_MAP["create_ingest_pipeline"] = "Ingest pipeline";
  WORKFLOW_STEP_TO_RESOURCE_TYPE_MAP["create_search_pipeline"] = "Search pipeline";
  WORKFLOW_STEP_TO_RESOURCE_TYPE_MAP["create_index"] = "Index";
  return WORKFLOW_STEP_TO_RESOURCE_TYPE_MAP;
}({});
/**
 ********** OPENSEARCH TYPES/INTERFACES ************
 */
// from https://opensearch.org/docs/latest/ingest-pipelines/simulate-ingest/#example-specify-a-pipeline-in-the-path
// from https://opensearch.org/docs/latest/ingest-pipelines/simulate-ingest/#example-specify-a-pipeline-in-the-path
// from https://opensearch.org/docs/latest/ingest-pipelines/simulate-ingest/#example-specify-a-pipeline-in-the-path
// verbose mode
// from https://opensearch.org/docs/latest/ingest-pipelines/simulate-ingest/#query-parameters
// verbose mode
// from https://opensearch.org/docs/latest/ingest-pipelines/simulate-ingest/#query-parameters
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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