// Copyright 2025 Google LLC.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

// Code generated file. DO NOT EDIT.

// Package biglake provides access to the BigLake API.
//
// For product documentation, see: https://cloud.google.com/bigquery/
//
// # Library status
//
// These client libraries are officially supported by Google. However, this
// library is considered complete and is in maintenance mode. This means
// that we will address critical bugs and security issues but will not add
// any new features.
//
// When possible, we recommend using our newer
// [Cloud Client Libraries for Go](https://pkg.go.dev/cloud.google.com/go)
// that are still actively being worked and iterated on.
//
// # Creating a client
//
// Usage example:
//
//	import "google.golang.org/api/biglake/v1"
//	...
//	ctx := context.Background()
//	biglakeService, err := biglake.NewService(ctx)
//
// In this example, Google Application Default Credentials are used for
// authentication. For information on how to create and obtain Application
// Default Credentials, see https://developers.google.com/identity/protocols/application-default-credentials.
//
// # Other authentication options
//
// By default, all available scopes (see "Constants") are used to authenticate.
// To restrict scopes, use [google.golang.org/api/option.WithScopes]:
//
//	biglakeService, err := biglake.NewService(ctx, option.WithScopes(biglake.CloudPlatformScope))
//
// To use an API key for authentication (note: some APIs do not support API
// keys), use [google.golang.org/api/option.WithAPIKey]:
//
//	biglakeService, err := biglake.NewService(ctx, option.WithAPIKey("AIza..."))
//
// To use an OAuth token (e.g., a user token obtained via a three-legged OAuth
// flow, use [google.golang.org/api/option.WithTokenSource]:
//
//	config := &oauth2.Config{...}
//	// ...
//	token, err := config.Exchange(ctx, ...)
//	biglakeService, err := biglake.NewService(ctx, option.WithTokenSource(config.TokenSource(ctx, token)))
//
// See [google.golang.org/api/option.ClientOption] for details on options.
package biglake // import "google.golang.org/api/biglake/v1"

import (
	"bytes"
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"log/slog"
	"net/http"
	"net/url"
	"strconv"
	"strings"

	"github.com/googleapis/gax-go/v2/internallog"
	googleapi "google.golang.org/api/googleapi"
	internal "google.golang.org/api/internal"
	gensupport "google.golang.org/api/internal/gensupport"
	option "google.golang.org/api/option"
	internaloption "google.golang.org/api/option/internaloption"
	htransport "google.golang.org/api/transport/http"
)

// Always reference these packages, just in case the auto-generated code
// below doesn't.
var _ = bytes.NewBuffer
var _ = strconv.Itoa
var _ = fmt.Sprintf
var _ = json.NewDecoder
var _ = io.Copy
var _ = url.Parse
var _ = gensupport.MarshalJSON
var _ = googleapi.Version
var _ = errors.New
var _ = strings.Replace
var _ = context.Canceled
var _ = internaloption.WithDefaultEndpoint
var _ = internal.Version
var _ = internallog.New

const apiId = "biglake:v1"
const apiName = "biglake"
const apiVersion = "v1"
const basePath = "https://biglake.googleapis.com/"
const basePathTemplate = "https://biglake.UNIVERSE_DOMAIN/"
const mtlsBasePath = "https://biglake.mtls.googleapis.com/"

// OAuth2 scopes used by this API.
const (
	// View and manage your data in Google BigQuery and see the email address for
	// your Google Account
	BigqueryScope = "https://www.googleapis.com/auth/bigquery"

	// See, edit, configure, and delete your Google Cloud data and see the email
	// address for your Google Account.
	CloudPlatformScope = "https://www.googleapis.com/auth/cloud-platform"
)

// NewService creates a new Service.
func NewService(ctx context.Context, opts ...option.ClientOption) (*Service, error) {
	scopesOption := internaloption.WithDefaultScopes(
		"https://www.googleapis.com/auth/bigquery",
		"https://www.googleapis.com/auth/cloud-platform",
	)
	// NOTE: prepend, so we don't override user-specified scopes.
	opts = append([]option.ClientOption{scopesOption}, opts...)
	opts = append(opts, internaloption.WithDefaultEndpoint(basePath))
	opts = append(opts, internaloption.WithDefaultEndpointTemplate(basePathTemplate))
	opts = append(opts, internaloption.WithDefaultMTLSEndpoint(mtlsBasePath))
	opts = append(opts, internaloption.EnableNewAuthLibrary())
	client, endpoint, err := htransport.NewClient(ctx, opts...)
	if err != nil {
		return nil, err
	}
	s := &Service{client: client, BasePath: basePath, logger: internaloption.GetLogger(opts)}
	s.Projects = NewProjectsService(s)
	if err != nil {
		return nil, err
	}
	if endpoint != "" {
		s.BasePath = endpoint
	}
	return s, nil
}

// New creates a new Service. It uses the provided http.Client for requests.
//
// Deprecated: please use NewService instead.
// To provide a custom HTTP client, use option.WithHTTPClient.
// If you are using google.golang.org/api/googleapis/transport.APIKey, use option.WithAPIKey with NewService instead.
func New(client *http.Client) (*Service, error) {
	if client == nil {
		return nil, errors.New("client is nil")
	}
	return NewService(context.Background(), option.WithHTTPClient(client))
}

type Service struct {
	client    *http.Client
	logger    *slog.Logger
	BasePath  string // API endpoint base URL
	UserAgent string // optional additional User-Agent fragment

	Projects *ProjectsService
}

func (s *Service) userAgent() string {
	if s.UserAgent == "" {
		return googleapi.UserAgent
	}
	return googleapi.UserAgent + " " + s.UserAgent
}

func NewProjectsService(s *Service) *ProjectsService {
	rs := &ProjectsService{s: s}
	rs.Locations = NewProjectsLocationsService(s)
	return rs
}

type ProjectsService struct {
	s *Service

	Locations *ProjectsLocationsService
}

func NewProjectsLocationsService(s *Service) *ProjectsLocationsService {
	rs := &ProjectsLocationsService{s: s}
	rs.Catalogs = NewProjectsLocationsCatalogsService(s)
	return rs
}

type ProjectsLocationsService struct {
	s *Service

	Catalogs *ProjectsLocationsCatalogsService
}

func NewProjectsLocationsCatalogsService(s *Service) *ProjectsLocationsCatalogsService {
	rs := &ProjectsLocationsCatalogsService{s: s}
	rs.Databases = NewProjectsLocationsCatalogsDatabasesService(s)
	return rs
}

type ProjectsLocationsCatalogsService struct {
	s *Service

	Databases *ProjectsLocationsCatalogsDatabasesService
}

func NewProjectsLocationsCatalogsDatabasesService(s *Service) *ProjectsLocationsCatalogsDatabasesService {
	rs := &ProjectsLocationsCatalogsDatabasesService{s: s}
	rs.Tables = NewProjectsLocationsCatalogsDatabasesTablesService(s)
	return rs
}

type ProjectsLocationsCatalogsDatabasesService struct {
	s *Service

	Tables *ProjectsLocationsCatalogsDatabasesTablesService
}

func NewProjectsLocationsCatalogsDatabasesTablesService(s *Service) *ProjectsLocationsCatalogsDatabasesTablesService {
	rs := &ProjectsLocationsCatalogsDatabasesTablesService{s: s}
	return rs
}

type ProjectsLocationsCatalogsDatabasesTablesService struct {
	s *Service
}

// Catalog: Catalog is the container of databases.
type Catalog struct {
	// CreateTime: Output only. The creation time of the catalog.
	CreateTime string `json:"createTime,omitempty"`
	// DeleteTime: Output only. The deletion time of the catalog. Only set after
	// the catalog is deleted.
	DeleteTime string `json:"deleteTime,omitempty"`
	// ExpireTime: Output only. The time when this catalog is considered expired.
	// Only set after the catalog is deleted.
	ExpireTime string `json:"expireTime,omitempty"`
	// Name: Output only. The resource name. Format:
	// projects/{project_id_or_number}/locations/{location_id}/catalogs/{catalog_id}
	Name string `json:"name,omitempty"`
	// UpdateTime: Output only. The last modification time of the catalog.
	UpdateTime string `json:"updateTime,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "CreateTime") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CreateTime") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Catalog) MarshalJSON() ([]byte, error) {
	type NoMethod Catalog
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Database: Database is the container of tables.
type Database struct {
	// CreateTime: Output only. The creation time of the database.
	CreateTime string `json:"createTime,omitempty"`
	// DeleteTime: Output only. The deletion time of the database. Only set after
	// the database is deleted.
	DeleteTime string `json:"deleteTime,omitempty"`
	// ExpireTime: Output only. The time when this database is considered expired.
	// Only set after the database is deleted.
	ExpireTime string `json:"expireTime,omitempty"`
	// HiveOptions: Options of a Hive database.
	HiveOptions *HiveDatabaseOptions `json:"hiveOptions,omitempty"`
	// Name: Output only. The resource name. Format:
	// projects/{project_id_or_number}/locations/{location_id}/catalogs/{catalog_id}
	// /databases/{database_id}
	Name string `json:"name,omitempty"`
	// Type: The database type.
	//
	// Possible values:
	//   "TYPE_UNSPECIFIED" - The type is not specified.
	//   "HIVE" - Represents a database storing tables compatible with Hive
	// Metastore tables.
	Type string `json:"type,omitempty"`
	// UpdateTime: Output only. The last modification time of the database.
	UpdateTime string `json:"updateTime,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "CreateTime") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CreateTime") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Database) MarshalJSON() ([]byte, error) {
	type NoMethod Database
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// HiveDatabaseOptions: Options of a Hive database.
type HiveDatabaseOptions struct {
	// LocationUri: Cloud Storage folder URI where the database data is stored,
	// starting with "gs://".
	LocationUri string `json:"locationUri,omitempty"`
	// Parameters: Stores user supplied Hive database parameters.
	Parameters map[string]string `json:"parameters,omitempty"`
	// ForceSendFields is a list of field names (e.g. "LocationUri") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "LocationUri") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s HiveDatabaseOptions) MarshalJSON() ([]byte, error) {
	type NoMethod HiveDatabaseOptions
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// HiveTableOptions: Options of a Hive table.
type HiveTableOptions struct {
	// Parameters: Stores user supplied Hive table parameters.
	Parameters map[string]string `json:"parameters,omitempty"`
	// StorageDescriptor: Stores physical storage information of the data.
	StorageDescriptor *StorageDescriptor `json:"storageDescriptor,omitempty"`
	// TableType: Hive table type. For example, MANAGED_TABLE, EXTERNAL_TABLE.
	TableType string `json:"tableType,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Parameters") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Parameters") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s HiveTableOptions) MarshalJSON() ([]byte, error) {
	type NoMethod HiveTableOptions
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListCatalogsResponse: Response message for the ListCatalogs method.
type ListCatalogsResponse struct {
	// Catalogs: The catalogs from the specified project.
	Catalogs []*Catalog `json:"catalogs,omitempty"`
	// NextPageToken: A token, which can be sent as `page_token` to retrieve the
	// next page. If this field is omitted, there are no subsequent pages.
	NextPageToken string `json:"nextPageToken,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Catalogs") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Catalogs") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListCatalogsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListCatalogsResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListDatabasesResponse: Response message for the ListDatabases method.
type ListDatabasesResponse struct {
	// Databases: The databases from the specified catalog.
	Databases []*Database `json:"databases,omitempty"`
	// NextPageToken: A token, which can be sent as `page_token` to retrieve the
	// next page. If this field is omitted, there are no subsequent pages.
	NextPageToken string `json:"nextPageToken,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Databases") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Databases") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListDatabasesResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListDatabasesResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListTablesResponse: Response message for the ListTables method.
type ListTablesResponse struct {
	// NextPageToken: A token, which can be sent as `page_token` to retrieve the
	// next page. If this field is omitted, there are no subsequent pages.
	NextPageToken string `json:"nextPageToken,omitempty"`
	// Tables: The tables from the specified database.
	Tables []*Table `json:"tables,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "NextPageToken") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "NextPageToken") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListTablesResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListTablesResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// RenameTableRequest: Request message for the RenameTable method in
// MetastoreService
type RenameTableRequest struct {
	// NewName: Required. The new `name` for the specified table, must be in the
	// same database. Format:
	// projects/{project_id_or_number}/locations/{location_id}/catalogs/{catalog_id}
	// /databases/{database_id}/tables/{table_id}
	NewName string `json:"newName,omitempty"`
	// ForceSendFields is a list of field names (e.g. "NewName") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "NewName") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s RenameTableRequest) MarshalJSON() ([]byte, error) {
	type NoMethod RenameTableRequest
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// SerDeInfo: Serializer and deserializer information.
type SerDeInfo struct {
	// SerializationLib: The fully qualified Java class name of the serialization
	// library.
	SerializationLib string `json:"serializationLib,omitempty"`
	// ForceSendFields is a list of field names (e.g. "SerializationLib") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "SerializationLib") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s SerDeInfo) MarshalJSON() ([]byte, error) {
	type NoMethod SerDeInfo
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// StorageDescriptor: Stores physical storage information of the data.
type StorageDescriptor struct {
	// InputFormat: The fully qualified Java class name of the input format.
	InputFormat string `json:"inputFormat,omitempty"`
	// LocationUri: Cloud Storage folder URI where the table data is stored,
	// starting with "gs://".
	LocationUri string `json:"locationUri,omitempty"`
	// OutputFormat: The fully qualified Java class name of the output format.
	OutputFormat string `json:"outputFormat,omitempty"`
	// SerdeInfo: Serializer and deserializer information.
	SerdeInfo *SerDeInfo `json:"serdeInfo,omitempty"`
	// ForceSendFields is a list of field names (e.g. "InputFormat") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "InputFormat") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s StorageDescriptor) MarshalJSON() ([]byte, error) {
	type NoMethod StorageDescriptor
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Table: Represents a table.
type Table struct {
	// CreateTime: Output only. The creation time of the table.
	CreateTime string `json:"createTime,omitempty"`
	// DeleteTime: Output only. The deletion time of the table. Only set after the
	// table is deleted.
	DeleteTime string `json:"deleteTime,omitempty"`
	// Etag: The checksum of a table object computed by the server based on the
	// value of other fields. It may be sent on update requests to ensure the
	// client has an up-to-date value before proceeding. It is only checked for
	// update table operations.
	Etag string `json:"etag,omitempty"`
	// ExpireTime: Output only. The time when this table is considered expired.
	// Only set after the table is deleted.
	ExpireTime string `json:"expireTime,omitempty"`
	// HiveOptions: Options of a Hive table.
	HiveOptions *HiveTableOptions `json:"hiveOptions,omitempty"`
	// Name: Output only. The resource name. Format:
	// projects/{project_id_or_number}/locations/{location_id}/catalogs/{catalog_id}
	// /databases/{database_id}/tables/{table_id}
	Name string `json:"name,omitempty"`
	// Type: The table type.
	//
	// Possible values:
	//   "TYPE_UNSPECIFIED" - The type is not specified.
	//   "HIVE" - Represents a table compatible with Hive Metastore tables.
	Type string `json:"type,omitempty"`
	// UpdateTime: Output only. The last modification time of the table.
	UpdateTime string `json:"updateTime,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "CreateTime") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CreateTime") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Table) MarshalJSON() ([]byte, error) {
	type NoMethod Table
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type ProjectsLocationsCatalogsCreateCall struct {
	s          *Service
	parent     string
	catalog    *Catalog
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Create: Creates a new catalog.
//
//   - parent: The parent resource where this catalog will be created. Format:
//     projects/{project_id_or_number}/locations/{location_id}.
func (r *ProjectsLocationsCatalogsService) Create(parent string, catalog *Catalog) *ProjectsLocationsCatalogsCreateCall {
	c := &ProjectsLocationsCatalogsCreateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	c.catalog = catalog
	return c
}

// CatalogId sets the optional parameter "catalogId": Required. The ID to use
// for the catalog, which will become the final component of the catalog's
// resource name.
func (c *ProjectsLocationsCatalogsCreateCall) CatalogId(catalogId string) *ProjectsLocationsCatalogsCreateCall {
	c.urlParams_.Set("catalogId", catalogId)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ProjectsLocationsCatalogsCreateCall) Fields(s ...googleapi.Field) *ProjectsLocationsCatalogsCreateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ProjectsLocationsCatalogsCreateCall) Context(ctx context.Context) *ProjectsLocationsCatalogsCreateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ProjectsLocationsCatalogsCreateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ProjectsLocationsCatalogsCreateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.catalog)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+parent}/catalogs")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "biglake.projects.locations.catalogs.create", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "biglake.projects.locations.catalogs.create" call.
// Any non-2xx status code is an error. Response headers are in either
// *Catalog.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *ProjectsLocationsCatalogsCreateCall) Do(opts ...googleapi.CallOption) (*Catalog, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Catalog{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "biglake.projects.locations.catalogs.create", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ProjectsLocationsCatalogsDeleteCall struct {
	s          *Service
	name       string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Delete: Deletes an existing catalog specified by the catalog ID.
//
//   - name: The name of the catalog to delete. Format:
//     projects/{project_id_or_number}/locations/{location_id}/catalogs/{catalog_i
//     d}.
func (r *ProjectsLocationsCatalogsService) Delete(name string) *ProjectsLocationsCatalogsDeleteCall {
	c := &ProjectsLocationsCatalogsDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ProjectsLocationsCatalogsDeleteCall) Fields(s ...googleapi.Field) *ProjectsLocationsCatalogsDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ProjectsLocationsCatalogsDeleteCall) Context(ctx context.Context) *ProjectsLocationsCatalogsDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ProjectsLocationsCatalogsDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ProjectsLocationsCatalogsDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "biglake.projects.locations.catalogs.delete", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "biglake.projects.locations.catalogs.delete" call.
// Any non-2xx status code is an error. Response headers are in either
// *Catalog.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *ProjectsLocationsCatalogsDeleteCall) Do(opts ...googleapi.CallOption) (*Catalog, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Catalog{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "biglake.projects.locations.catalogs.delete", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ProjectsLocationsCatalogsGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets the catalog specified by the resource name.
//
//   - name: The name of the catalog to retrieve. Format:
//     projects/{project_id_or_number}/locations/{location_id}/catalogs/{catalog_i
//     d}.
func (r *ProjectsLocationsCatalogsService) Get(name string) *ProjectsLocationsCatalogsGetCall {
	c := &ProjectsLocationsCatalogsGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ProjectsLocationsCatalogsGetCall) Fields(s ...googleapi.Field) *ProjectsLocationsCatalogsGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *ProjectsLocationsCatalogsGetCall) IfNoneMatch(entityTag string) *ProjectsLocationsCatalogsGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ProjectsLocationsCatalogsGetCall) Context(ctx context.Context) *ProjectsLocationsCatalogsGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ProjectsLocationsCatalogsGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ProjectsLocationsCatalogsGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "biglake.projects.locations.catalogs.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "biglake.projects.locations.catalogs.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *Catalog.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *ProjectsLocationsCatalogsGetCall) Do(opts ...googleapi.CallOption) (*Catalog, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Catalog{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "biglake.projects.locations.catalogs.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ProjectsLocationsCatalogsListCall struct {
	s            *Service
	parent       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: List all catalogs in a specified project.
//
//   - parent: The parent, which owns this collection of catalogs. Format:
//     projects/{project_id_or_number}/locations/{location_id}.
func (r *ProjectsLocationsCatalogsService) List(parent string) *ProjectsLocationsCatalogsListCall {
	c := &ProjectsLocationsCatalogsListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	return c
}

// PageSize sets the optional parameter "pageSize": The maximum number of
// catalogs to return. The service may return fewer than this value. If
// unspecified, at most 50 catalogs will be returned. The maximum value is
// 1000; values above 1000 will be coerced to 1000.
func (c *ProjectsLocationsCatalogsListCall) PageSize(pageSize int64) *ProjectsLocationsCatalogsListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": A page token, received
// from a previous `ListCatalogs` call. Provide this to retrieve the subsequent
// page. When paginating, all other parameters provided to `ListCatalogs` must
// match the call that provided the page token.
func (c *ProjectsLocationsCatalogsListCall) PageToken(pageToken string) *ProjectsLocationsCatalogsListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ProjectsLocationsCatalogsListCall) Fields(s ...googleapi.Field) *ProjectsLocationsCatalogsListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *ProjectsLocationsCatalogsListCall) IfNoneMatch(entityTag string) *ProjectsLocationsCatalogsListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ProjectsLocationsCatalogsListCall) Context(ctx context.Context) *ProjectsLocationsCatalogsListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ProjectsLocationsCatalogsListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ProjectsLocationsCatalogsListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+parent}/catalogs")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "biglake.projects.locations.catalogs.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "biglake.projects.locations.catalogs.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListCatalogsResponse.ServerResponse.Header or (if a response was returned
// at all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *ProjectsLocationsCatalogsListCall) Do(opts ...googleapi.CallOption) (*ListCatalogsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListCatalogsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "biglake.projects.locations.catalogs.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *ProjectsLocationsCatalogsListCall) Pages(ctx context.Context, f func(*ListCatalogsResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type ProjectsLocationsCatalogsDatabasesCreateCall struct {
	s          *Service
	parent     string
	database   *Database
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Create: Creates a new database.
//
//   - parent: The parent resource where this database will be created. Format:
//     projects/{project_id_or_number}/locations/{location_id}/catalogs/{catalog_i
//     d}.
func (r *ProjectsLocationsCatalogsDatabasesService) Create(parent string, database *Database) *ProjectsLocationsCatalogsDatabasesCreateCall {
	c := &ProjectsLocationsCatalogsDatabasesCreateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	c.database = database
	return c
}

// DatabaseId sets the optional parameter "databaseId": Required. The ID to use
// for the database, which will become the final component of the database's
// resource name.
func (c *ProjectsLocationsCatalogsDatabasesCreateCall) DatabaseId(databaseId string) *ProjectsLocationsCatalogsDatabasesCreateCall {
	c.urlParams_.Set("databaseId", databaseId)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ProjectsLocationsCatalogsDatabasesCreateCall) Fields(s ...googleapi.Field) *ProjectsLocationsCatalogsDatabasesCreateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ProjectsLocationsCatalogsDatabasesCreateCall) Context(ctx context.Context) *ProjectsLocationsCatalogsDatabasesCreateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ProjectsLocationsCatalogsDatabasesCreateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ProjectsLocationsCatalogsDatabasesCreateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.database)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+parent}/databases")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "biglake.projects.locations.catalogs.databases.create", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "biglake.projects.locations.catalogs.databases.create" call.
// Any non-2xx status code is an error. Response headers are in either
// *Database.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *ProjectsLocationsCatalogsDatabasesCreateCall) Do(opts ...googleapi.CallOption) (*Database, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Database{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "biglake.projects.locations.catalogs.databases.create", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ProjectsLocationsCatalogsDatabasesDeleteCall struct {
	s          *Service
	name       string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Delete: Deletes an existing database specified by the database ID.
//
//   - name: The name of the database to delete. Format:
//     projects/{project_id_or_number}/locations/{location_id}/catalogs/{catalog_i
//     d}/databases/{database_id}.
func (r *ProjectsLocationsCatalogsDatabasesService) Delete(name string) *ProjectsLocationsCatalogsDatabasesDeleteCall {
	c := &ProjectsLocationsCatalogsDatabasesDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ProjectsLocationsCatalogsDatabasesDeleteCall) Fields(s ...googleapi.Field) *ProjectsLocationsCatalogsDatabasesDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ProjectsLocationsCatalogsDatabasesDeleteCall) Context(ctx context.Context) *ProjectsLocationsCatalogsDatabasesDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ProjectsLocationsCatalogsDatabasesDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ProjectsLocationsCatalogsDatabasesDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "biglake.projects.locations.catalogs.databases.delete", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "biglake.projects.locations.catalogs.databases.delete" call.
// Any non-2xx status code is an error. Response headers are in either
// *Database.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *ProjectsLocationsCatalogsDatabasesDeleteCall) Do(opts ...googleapi.CallOption) (*Database, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Database{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "biglake.projects.locations.catalogs.databases.delete", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ProjectsLocationsCatalogsDatabasesGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets the database specified by the resource name.
//
//   - name: The name of the database to retrieve. Format:
//     projects/{project_id_or_number}/locations/{location_id}/catalogs/{catalog_i
//     d}/databases/{database_id}.
func (r *ProjectsLocationsCatalogsDatabasesService) Get(name string) *ProjectsLocationsCatalogsDatabasesGetCall {
	c := &ProjectsLocationsCatalogsDatabasesGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ProjectsLocationsCatalogsDatabasesGetCall) Fields(s ...googleapi.Field) *ProjectsLocationsCatalogsDatabasesGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *ProjectsLocationsCatalogsDatabasesGetCall) IfNoneMatch(entityTag string) *ProjectsLocationsCatalogsDatabasesGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ProjectsLocationsCatalogsDatabasesGetCall) Context(ctx context.Context) *ProjectsLocationsCatalogsDatabasesGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ProjectsLocationsCatalogsDatabasesGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ProjectsLocationsCatalogsDatabasesGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "biglake.projects.locations.catalogs.databases.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "biglake.projects.locations.catalogs.databases.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *Database.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *ProjectsLocationsCatalogsDatabasesGetCall) Do(opts ...googleapi.CallOption) (*Database, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Database{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "biglake.projects.locations.catalogs.databases.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ProjectsLocationsCatalogsDatabasesListCall struct {
	s            *Service
	parent       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: List all databases in a specified catalog.
//
//   - parent: The parent, which owns this collection of databases. Format:
//     projects/{project_id_or_number}/locations/{location_id}/catalogs/{catalog_i
//     d}.
func (r *ProjectsLocationsCatalogsDatabasesService) List(parent string) *ProjectsLocationsCatalogsDatabasesListCall {
	c := &ProjectsLocationsCatalogsDatabasesListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	return c
}

// PageSize sets the optional parameter "pageSize": The maximum number of
// databases to return. The service may return fewer than this value. If
// unspecified, at most 50 databases will be returned. The maximum value is
// 1000; values above 1000 will be coerced to 1000.
func (c *ProjectsLocationsCatalogsDatabasesListCall) PageSize(pageSize int64) *ProjectsLocationsCatalogsDatabasesListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": A page token, received
// from a previous `ListDatabases` call. Provide this to retrieve the
// subsequent page. When paginating, all other parameters provided to
// `ListDatabases` must match the call that provided the page token.
func (c *ProjectsLocationsCatalogsDatabasesListCall) PageToken(pageToken string) *ProjectsLocationsCatalogsDatabasesListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ProjectsLocationsCatalogsDatabasesListCall) Fields(s ...googleapi.Field) *ProjectsLocationsCatalogsDatabasesListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *ProjectsLocationsCatalogsDatabasesListCall) IfNoneMatch(entityTag string) *ProjectsLocationsCatalogsDatabasesListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ProjectsLocationsCatalogsDatabasesListCall) Context(ctx context.Context) *ProjectsLocationsCatalogsDatabasesListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ProjectsLocationsCatalogsDatabasesListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ProjectsLocationsCatalogsDatabasesListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+parent}/databases")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "biglake.projects.locations.catalogs.databases.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "biglake.projects.locations.catalogs.databases.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListDatabasesResponse.ServerResponse.Header or (if a response was returned
// at all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *ProjectsLocationsCatalogsDatabasesListCall) Do(opts ...googleapi.CallOption) (*ListDatabasesResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListDatabasesResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "biglake.projects.locations.catalogs.databases.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *ProjectsLocationsCatalogsDatabasesListCall) Pages(ctx context.Context, f func(*ListDatabasesResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type ProjectsLocationsCatalogsDatabasesPatchCall struct {
	s          *Service
	name       string
	database   *Database
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Patch: Updates an existing database specified by the database ID.
//
//   - name: Output only. The resource name. Format:
//     projects/{project_id_or_number}/locations/{location_id}/catalogs/{catalog_i
//     d}/databases/{database_id}.
func (r *ProjectsLocationsCatalogsDatabasesService) Patch(name string, database *Database) *ProjectsLocationsCatalogsDatabasesPatchCall {
	c := &ProjectsLocationsCatalogsDatabasesPatchCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.database = database
	return c
}

// UpdateMask sets the optional parameter "updateMask": The list of fields to
// update. For the `FieldMask` definition, see
// https://developers.google.com/protocol-buffers/docs/reference/google.protobuf#fieldmask
// If not set, defaults to all of the fields that are allowed to update.
func (c *ProjectsLocationsCatalogsDatabasesPatchCall) UpdateMask(updateMask string) *ProjectsLocationsCatalogsDatabasesPatchCall {
	c.urlParams_.Set("updateMask", updateMask)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ProjectsLocationsCatalogsDatabasesPatchCall) Fields(s ...googleapi.Field) *ProjectsLocationsCatalogsDatabasesPatchCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ProjectsLocationsCatalogsDatabasesPatchCall) Context(ctx context.Context) *ProjectsLocationsCatalogsDatabasesPatchCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ProjectsLocationsCatalogsDatabasesPatchCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ProjectsLocationsCatalogsDatabasesPatchCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.database)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PATCH", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "biglake.projects.locations.catalogs.databases.patch", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "biglake.projects.locations.catalogs.databases.patch" call.
// Any non-2xx status code is an error. Response headers are in either
// *Database.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *ProjectsLocationsCatalogsDatabasesPatchCall) Do(opts ...googleapi.CallOption) (*Database, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Database{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "biglake.projects.locations.catalogs.databases.patch", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ProjectsLocationsCatalogsDatabasesTablesCreateCall struct {
	s          *Service
	parent     string
	table      *Table
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Create: Creates a new table.
//
//   - parent: The parent resource where this table will be created. Format:
//     projects/{project_id_or_number}/locations/{location_id}/catalogs/{catalog_i
//     d}/databases/{database_id}.
func (r *ProjectsLocationsCatalogsDatabasesTablesService) Create(parent string, table *Table) *ProjectsLocationsCatalogsDatabasesTablesCreateCall {
	c := &ProjectsLocationsCatalogsDatabasesTablesCreateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	c.table = table
	return c
}

// TableId sets the optional parameter "tableId": Required. The ID to use for
// the table, which will become the final component of the table's resource
// name.
func (c *ProjectsLocationsCatalogsDatabasesTablesCreateCall) TableId(tableId string) *ProjectsLocationsCatalogsDatabasesTablesCreateCall {
	c.urlParams_.Set("tableId", tableId)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ProjectsLocationsCatalogsDatabasesTablesCreateCall) Fields(s ...googleapi.Field) *ProjectsLocationsCatalogsDatabasesTablesCreateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ProjectsLocationsCatalogsDatabasesTablesCreateCall) Context(ctx context.Context) *ProjectsLocationsCatalogsDatabasesTablesCreateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ProjectsLocationsCatalogsDatabasesTablesCreateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ProjectsLocationsCatalogsDatabasesTablesCreateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.table)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+parent}/tables")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "biglake.projects.locations.catalogs.databases.tables.create", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "biglake.projects.locations.catalogs.databases.tables.create" call.
// Any non-2xx status code is an error. Response headers are in either
// *Table.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *ProjectsLocationsCatalogsDatabasesTablesCreateCall) Do(opts ...googleapi.CallOption) (*Table, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Table{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "biglake.projects.locations.catalogs.databases.tables.create", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ProjectsLocationsCatalogsDatabasesTablesDeleteCall struct {
	s          *Service
	name       string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Delete: Deletes an existing table specified by the table ID.
//
//   - name: The name of the table to delete. Format:
//     projects/{project_id_or_number}/locations/{location_id}/catalogs/{catalog_i
//     d}/databases/{database_id}/tables/{table_id}.
func (r *ProjectsLocationsCatalogsDatabasesTablesService) Delete(name string) *ProjectsLocationsCatalogsDatabasesTablesDeleteCall {
	c := &ProjectsLocationsCatalogsDatabasesTablesDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ProjectsLocationsCatalogsDatabasesTablesDeleteCall) Fields(s ...googleapi.Field) *ProjectsLocationsCatalogsDatabasesTablesDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ProjectsLocationsCatalogsDatabasesTablesDeleteCall) Context(ctx context.Context) *ProjectsLocationsCatalogsDatabasesTablesDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ProjectsLocationsCatalogsDatabasesTablesDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ProjectsLocationsCatalogsDatabasesTablesDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "biglake.projects.locations.catalogs.databases.tables.delete", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "biglake.projects.locations.catalogs.databases.tables.delete" call.
// Any non-2xx status code is an error. Response headers are in either
// *Table.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *ProjectsLocationsCatalogsDatabasesTablesDeleteCall) Do(opts ...googleapi.CallOption) (*Table, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Table{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "biglake.projects.locations.catalogs.databases.tables.delete", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ProjectsLocationsCatalogsDatabasesTablesGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets the table specified by the resource name.
//
//   - name: The name of the table to retrieve. Format:
//     projects/{project_id_or_number}/locations/{location_id}/catalogs/{catalog_i
//     d}/databases/{database_id}/tables/{table_id}.
func (r *ProjectsLocationsCatalogsDatabasesTablesService) Get(name string) *ProjectsLocationsCatalogsDatabasesTablesGetCall {
	c := &ProjectsLocationsCatalogsDatabasesTablesGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ProjectsLocationsCatalogsDatabasesTablesGetCall) Fields(s ...googleapi.Field) *ProjectsLocationsCatalogsDatabasesTablesGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *ProjectsLocationsCatalogsDatabasesTablesGetCall) IfNoneMatch(entityTag string) *ProjectsLocationsCatalogsDatabasesTablesGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ProjectsLocationsCatalogsDatabasesTablesGetCall) Context(ctx context.Context) *ProjectsLocationsCatalogsDatabasesTablesGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ProjectsLocationsCatalogsDatabasesTablesGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ProjectsLocationsCatalogsDatabasesTablesGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "biglake.projects.locations.catalogs.databases.tables.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "biglake.projects.locations.catalogs.databases.tables.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *Table.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *ProjectsLocationsCatalogsDatabasesTablesGetCall) Do(opts ...googleapi.CallOption) (*Table, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Table{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "biglake.projects.locations.catalogs.databases.tables.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ProjectsLocationsCatalogsDatabasesTablesListCall struct {
	s            *Service
	parent       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: List all tables in a specified database.
//
//   - parent: The parent, which owns this collection of tables. Format:
//     projects/{project_id_or_number}/locations/{location_id}/catalogs/{catalog_i
//     d}/databases/{database_id}.
func (r *ProjectsLocationsCatalogsDatabasesTablesService) List(parent string) *ProjectsLocationsCatalogsDatabasesTablesListCall {
	c := &ProjectsLocationsCatalogsDatabasesTablesListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	return c
}

// PageSize sets the optional parameter "pageSize": The maximum number of
// tables to return. The service may return fewer than this value. If
// unspecified, at most 50 tables will be returned. The maximum value is 1000;
// values above 1000 will be coerced to 1000.
func (c *ProjectsLocationsCatalogsDatabasesTablesListCall) PageSize(pageSize int64) *ProjectsLocationsCatalogsDatabasesTablesListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": A page token, received
// from a previous `ListTables` call. Provide this to retrieve the subsequent
// page. When paginating, all other parameters provided to `ListTables` must
// match the call that provided the page token.
func (c *ProjectsLocationsCatalogsDatabasesTablesListCall) PageToken(pageToken string) *ProjectsLocationsCatalogsDatabasesTablesListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// View sets the optional parameter "view": The view for the returned tables.
//
// Possible values:
//
//	"TABLE_VIEW_UNSPECIFIED" - Default value. The API will default to the
//
// BASIC view.
//
//	"BASIC" - Include only table names. This is the default value.
//	"FULL" - Include everything.
func (c *ProjectsLocationsCatalogsDatabasesTablesListCall) View(view string) *ProjectsLocationsCatalogsDatabasesTablesListCall {
	c.urlParams_.Set("view", view)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ProjectsLocationsCatalogsDatabasesTablesListCall) Fields(s ...googleapi.Field) *ProjectsLocationsCatalogsDatabasesTablesListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *ProjectsLocationsCatalogsDatabasesTablesListCall) IfNoneMatch(entityTag string) *ProjectsLocationsCatalogsDatabasesTablesListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ProjectsLocationsCatalogsDatabasesTablesListCall) Context(ctx context.Context) *ProjectsLocationsCatalogsDatabasesTablesListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ProjectsLocationsCatalogsDatabasesTablesListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ProjectsLocationsCatalogsDatabasesTablesListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+parent}/tables")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "biglake.projects.locations.catalogs.databases.tables.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "biglake.projects.locations.catalogs.databases.tables.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListTablesResponse.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *ProjectsLocationsCatalogsDatabasesTablesListCall) Do(opts ...googleapi.CallOption) (*ListTablesResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListTablesResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "biglake.projects.locations.catalogs.databases.tables.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *ProjectsLocationsCatalogsDatabasesTablesListCall) Pages(ctx context.Context, f func(*ListTablesResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type ProjectsLocationsCatalogsDatabasesTablesPatchCall struct {
	s          *Service
	name       string
	table      *Table
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Patch: Updates an existing table specified by the table ID.
//
//   - name: Output only. The resource name. Format:
//     projects/{project_id_or_number}/locations/{location_id}/catalogs/{catalog_i
//     d}/databases/{database_id}/tables/{table_id}.
func (r *ProjectsLocationsCatalogsDatabasesTablesService) Patch(name string, table *Table) *ProjectsLocationsCatalogsDatabasesTablesPatchCall {
	c := &ProjectsLocationsCatalogsDatabasesTablesPatchCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.table = table
	return c
}

// UpdateMask sets the optional parameter "updateMask": The list of fields to
// update. For the `FieldMask` definition, see
// https://developers.google.com/protocol-buffers/docs/reference/google.protobuf#fieldmask
// If not set, defaults to all of the fields that are allowed to update.
func (c *ProjectsLocationsCatalogsDatabasesTablesPatchCall) UpdateMask(updateMask string) *ProjectsLocationsCatalogsDatabasesTablesPatchCall {
	c.urlParams_.Set("updateMask", updateMask)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ProjectsLocationsCatalogsDatabasesTablesPatchCall) Fields(s ...googleapi.Field) *ProjectsLocationsCatalogsDatabasesTablesPatchCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ProjectsLocationsCatalogsDatabasesTablesPatchCall) Context(ctx context.Context) *ProjectsLocationsCatalogsDatabasesTablesPatchCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ProjectsLocationsCatalogsDatabasesTablesPatchCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ProjectsLocationsCatalogsDatabasesTablesPatchCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.table)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PATCH", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "biglake.projects.locations.catalogs.databases.tables.patch", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "biglake.projects.locations.catalogs.databases.tables.patch" call.
// Any non-2xx status code is an error. Response headers are in either
// *Table.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *ProjectsLocationsCatalogsDatabasesTablesPatchCall) Do(opts ...googleapi.CallOption) (*Table, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Table{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "biglake.projects.locations.catalogs.databases.tables.patch", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ProjectsLocationsCatalogsDatabasesTablesRenameCall struct {
	s                  *Service
	name               string
	renametablerequest *RenameTableRequest
	urlParams_         gensupport.URLParams
	ctx_               context.Context
	header_            http.Header
}

// Rename: Renames an existing table specified by the table ID.
//
//   - name: The table's `name` field is used to identify the table to rename.
//     Format:
//     projects/{project_id_or_number}/locations/{location_id}/catalogs/{catalog_i
//     d}/databases/{database_id}/tables/{table_id}.
func (r *ProjectsLocationsCatalogsDatabasesTablesService) Rename(name string, renametablerequest *RenameTableRequest) *ProjectsLocationsCatalogsDatabasesTablesRenameCall {
	c := &ProjectsLocationsCatalogsDatabasesTablesRenameCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.renametablerequest = renametablerequest
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ProjectsLocationsCatalogsDatabasesTablesRenameCall) Fields(s ...googleapi.Field) *ProjectsLocationsCatalogsDatabasesTablesRenameCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ProjectsLocationsCatalogsDatabasesTablesRenameCall) Context(ctx context.Context) *ProjectsLocationsCatalogsDatabasesTablesRenameCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ProjectsLocationsCatalogsDatabasesTablesRenameCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ProjectsLocationsCatalogsDatabasesTablesRenameCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.renametablerequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}:rename")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "biglake.projects.locations.catalogs.databases.tables.rename", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "biglake.projects.locations.catalogs.databases.tables.rename" call.
// Any non-2xx status code is an error. Response headers are in either
// *Table.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *ProjectsLocationsCatalogsDatabasesTablesRenameCall) Do(opts ...googleapi.CallOption) (*Table, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Table{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "biglake.projects.locations.catalogs.databases.tables.rename", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}
