//go:build ignore

package main

import (
	"encoding/csv"
	"io"
	"log"
	"os"
	"sort"
	"strings"
	"text/template"
	"unicode"
	"unicode/utf8"
)

const codeTemplate = `
// Code generated by gen.go; DO NOT EDIT.

package multicodec

const (
{{- range .Entries }}
	// {{ if .IsDeprecated }}Deprecated: {{ end }}{{ .ConstName }} is a {{ .Status }} code tagged "{{ .Tag }}"{{ if .Description }} and described by: {{ .Description }}{{ end }}.
	{{ .ConstName }} Code = {{ .Code }} // {{ .Name }}
{{ end -}}
)

var knownCodes = []Code{
{{- range .Entries }}
	{{ .ConstName }},
{{- end }}
}

func (c Code) Tag() string {
	switch c {
{{- range $i, $tag := .Tags }}
	case {{ $first := true }}
	{{- range $.Entries }}
		{{- /* we don't include the only deprecated code, as it's a duplicate */ -}}
		{{- if not .IsDeprecated }}{{ if eq .Tag $tag }}
			{{- if not $first }},
{{ end }}{{ $first = false -}}
			{{ .ConstName -}}
		{{ end }}{{ end -}}
	{{ end -}}
	:
		return {{ printf "%q" $tag }}
{{ end -}}
	default:
		return "<unknown>"
	}
}
`

type tableEntry struct {
	Name        string
	Tag         string
	Code        string
	Status      string
	Description string
}

func (c tableEntry) IsDeprecated() bool {
	return strings.Contains(c.Description, "deprecated")
}

func (c tableEntry) ConstName() string {
	var b strings.Builder
	var last rune
	for _, part := range strings.Split(c.Name, "-") {
		first, firstSize := utf8.DecodeRuneInString(part)
		if unicode.IsNumber(last) && unicode.IsNumber(first) {
			// 123-456 should result in 123_456 for readability.
			b.WriteByte('_')
		}
		b.WriteRune(unicode.ToUpper(first))
		b.WriteString(part[firstSize:])
		last, _ = utf8.DecodeLastRuneInString(part)
	}
	return b.String()
}

func main() {
	f, err := os.Open("multicodec/table.csv")
	if err != nil {
		log.Fatal(err)
	}
	defer f.Close()

	var tmplData struct {
		Entries []tableEntry
		Tags    []string
	}
	tags := make(map[string]bool)
	csvReader := csv.NewReader(f)
	csvReader.Read() // skip the header line
	for {
		record, err := csvReader.Read()
		if err == io.EOF {
			break
		}
		if err != nil {
			log.Fatal(err)
		}
		entry := tableEntry{
			Name:        strings.TrimSpace(record[0]),
			Tag:         strings.TrimSpace(record[1]),
			Code:        strings.TrimSpace(record[2]),
			Status:      strings.TrimSpace(record[3]),
			Description: strings.TrimSpace(record[4]),
		}
		tmplData.Entries = append(tmplData.Entries, entry)
		tags[entry.Tag] = true // use a map to deduplicate
	}
	for tag := range tags {
		tmplData.Tags = append(tmplData.Tags, tag)
	}
	sort.Strings(tmplData.Tags)

	tmpl, err := template.New("").
		Funcs(template.FuncMap{"ToTitle": strings.Title}).
		Parse(codeTemplate)
	if err != nil {
		log.Fatal(err)
	}

	out, err := os.Create("code_table.go")
	if err != nil {
		log.Fatal(err)
	}
	defer out.Close()

	if err := tmpl.Execute(out, tmplData); err != nil {
		log.Fatal(err)
	}
}
