// Copyright 2025 Google LLC
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package slhdsa

import (
	"slices"
	"testing"
)

func TestForsSkGenKat(t *testing.T) {
	par := SLH_DSA_SHA2_128s
	skSeed := []byte{0x37, 0xfc, 0x9f, 0xf0, 0x9a, 0x60, 0xd1, 0xe2, 0xdd, 0xcd, 0xb7, 0x36, 0xb, 0xd6, 0xe5, 0x31}
	pkSeed := []byte{0x4e, 0xf6, 0x27, 0x9b, 0x12, 0x84, 0x60, 0xa0, 0x77, 0x0, 0x23, 0xe0, 0x84, 0xf1, 0x68, 0x29}
	adrs := &address{0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0xb, 0x34, 0xf3, 0xfa, 0xdc, 0x2a, 0x8e, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x88, 0x0, 0x0, 0x0, 0x2, 0x0, 0x0, 0x36, 0x0}
	idx := uint32(0xd804)
	want := []byte{0x34, 0xdb, 0x6b, 0x7e, 0xd4, 0xa, 0xbe, 0xec, 0x74, 0x9d, 0x27, 0x8b, 0x14, 0x32, 0x58, 0x40}
	if got := par.forsSkGen(skSeed, pkSeed, adrs, idx); !slices.Equal(got, want) {
		t.Errorf("par.forsSkGen() = %v, want %v", got, want)
	}
}

func TestForsNodeKat(t *testing.T) {
	par := SLH_DSA_SHA2_128s
	skSeed := []byte{0x6a, 0x1e, 0x4f, 0xcc, 0x1c, 0x9, 0x75, 0xc, 0x5f, 0xb6, 0xbb, 0x14, 0x25, 0xfa, 0x41, 0x13}
	i := uint32(0x69d)
	z := uint32(0x5)
	pkSeed := []byte{0xc1, 0x98, 0x2c, 0x25, 0x6f, 0xcd, 0xef, 0xcb, 0x26, 0xc3, 0xb7, 0x50, 0x7c, 0x18, 0x5f, 0x7e}
	adrs := &address{0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x7, 0x98, 0xfb, 0xae, 0x9e, 0x90, 0xd3, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x0, 0x98, 0x0, 0x0, 0x0, 0x5, 0x0, 0x0, 0x6, 0x9c}
	want := []byte{0x1d, 0xb9, 0x24, 0xec, 0xf6, 0x9f, 0x9e, 0xae, 0xbb, 0xfe, 0x1c, 0xb, 0xef, 0x5b, 0x8e, 0x36}
	if got := par.forsNode(skSeed, i, z, pkSeed, adrs); !slices.Equal(got, want) {
		t.Errorf("par.forsNode() = %v, want %v", got, want)
	}
}

func TestForsSignkat(t *testing.T) {
	par := SLH_DSA_SHA2_128s
	md := []byte{0x70, 0x37, 0xf1, 0x79, 0xf8, 0x74, 0x7b, 0x7b, 0xf7, 0xbf, 0xce, 0x47, 0x67, 0xfb, 0xce, 0x28, 0x97, 0xac, 0xb2, 0x75, 0x49}
	skSeed := []byte{0x7a, 0x21, 0x8b, 0x4a, 0x12, 0x13, 0x1d, 0x82, 0xfb, 0x12, 0x9, 0xa6, 0xeb, 0xb4, 0x46, 0xa1}
	pkSeed := []byte{0x5d, 0xce, 0xdb, 0x6d, 0xba, 0xc0, 0xa2, 0x49, 0x6a, 0x42, 0x0, 0xeb, 0x91, 0x9a, 0x3f, 0xd0}
	adrs := &address{0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x2e, 0x7c, 0xd8, 0x6e, 0x34, 0xa3, 0xd9, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x1, 0xad, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0}
	want := []byte{0x62, 0x88, 0x6a, 0x8d, 0x36, 0xd4, 0xb0, 0xfd, 0xa, 0xc1, 0xfa, 0x12, 0x51, 0xeb, 0x2b, 0x9f, 0x41, 0xa9, 0xed, 0x1f, 0x9c, 0x13, 0x79, 0xaf, 0x6f, 0x2e, 0x26, 0x8, 0x0, 0x48, 0xf2, 0x1d, 0xf9, 0x44, 0xa9, 0x15, 0xde, 0x49, 0xa1, 0xb9, 0x60, 0x78, 0xba, 0x47, 0xc6, 0x3e, 0x28, 0x9, 0xd, 0xe1, 0xbb, 0xd3, 0xb7, 0xee, 0x2e, 0xf2, 0x37, 0x87, 0x9c, 0x18, 0x31, 0x35, 0x3, 0x34, 0x8c, 0xc7, 0x5b, 0x6b, 0xcf, 0xab, 0x13, 0x2f, 0x48, 0xae, 0xa2, 0x15, 0x3a, 0x19, 0x36, 0x65, 0x78, 0xb2, 0xc8, 0x5f, 0x1, 0xbf, 0x7d, 0x47, 0xd9, 0xa9, 0x6b, 0x1b, 0xa6, 0x53, 0x7f, 0xba, 0x8c, 0xda, 0x6e, 0x47, 0xbe, 0x6a, 0x53, 0x75, 0xcf, 0x8e, 0x63, 0xfd, 0x36, 0x71, 0xfb, 0x3b, 0x5c, 0x38, 0x79, 0xed, 0xc6, 0x52, 0x8, 0x70, 0xea, 0x26, 0xc6, 0x94, 0xbf, 0x29, 0xa6, 0xde, 0x85, 0x23, 0x35, 0xb7, 0x48, 0x40, 0xd4, 0x3d, 0xe8, 0xfe, 0xe4, 0x49, 0xdc, 0x4e, 0xe9, 0x52, 0x1e, 0x41, 0x36, 0x8b, 0x81, 0xda, 0x39, 0xae, 0x2, 0x3b, 0xc5, 0xe5, 0xd7, 0x75, 0xc, 0xb0, 0x3d, 0x86, 0x4c, 0x55, 0x4a, 0x42, 0x7, 0xf9, 0xd6, 0x0, 0x34, 0x9a, 0x6e, 0x84, 0x5b, 0xb7, 0xc4, 0xbd, 0x52, 0x5d, 0x8b, 0x5c, 0xdb, 0x26, 0x8a, 0xed, 0x39, 0x5c, 0xb0, 0x52, 0x74, 0xb0, 0x4a, 0x27, 0xf8, 0xfe, 0xed, 0x7c, 0xe9, 0xec, 0x1d, 0x1e, 0xf, 0x6e, 0x3d, 0x5d, 0x2c, 0xb, 0x9c, 0x2, 0x64, 0xc8, 0x95, 0x8a, 0x71, 0x4d, 0x10, 0x33, 0x96, 0x1, 0xe2, 0xca, 0x87, 0x65, 0x91, 0xce, 0xf9, 0x4d, 0xd, 0x7c, 0xf9, 0x1b, 0x9f, 0xf5, 0x3b, 0xc, 0xec, 0xe1, 0xc8, 0xee, 0x41, 0x1f, 0x4d, 0xc8, 0x1e, 0x13, 0xe3, 0xe6, 0x38, 0xdc, 0x86, 0x7e, 0xa0, 0x86, 0xc1, 0x23, 0x14, 0x1d, 0x2e, 0x16, 0x57, 0xf8, 0x35, 0xec, 0x97, 0xd, 0x6f, 0x6c, 0x1, 0x28, 0x41, 0xd3, 0x43, 0x72, 0x77, 0xbf, 0x2b, 0x70, 0x5d, 0x3e, 0x8, 0x7a, 0x5a, 0xad, 0x4a, 0xbf, 0x90, 0x50, 0x2d, 0xb6, 0x89, 0x2, 0x21, 0x35, 0x43, 0xa3, 0x6f, 0x9c, 0xfe, 0xae, 0x5e, 0x26, 0x62, 0x42, 0xc0, 0xde, 0x8c, 0x34, 0xb5, 0x56, 0x54, 0xda, 0xf7, 0x5a, 0xcf, 0xe8, 0xf5, 0xe3, 0xa5, 0xbc, 0x4, 0xf2, 0xed, 0x92, 0xfa, 0x22, 0xcd, 0xe9, 0xd9, 0xe8, 0xfc, 0xc3, 0x82, 0xa9, 0x5d, 0x81, 0xd9, 0x9a, 0x24, 0xbe, 0x2d, 0xc8, 0xe6, 0x1a, 0xdb, 0xb3, 0xd9, 0x46, 0xf7, 0x7d, 0x39, 0x99, 0x56, 0x4a, 0x17, 0x7c, 0xfb, 0x9c, 0x30, 0x39, 0x44, 0xcb, 0x1b, 0x88, 0xd, 0x88, 0x7, 0x4c, 0x27, 0x69, 0x67, 0x78, 0xc0, 0x1c, 0xa8, 0x69, 0xbd, 0xa2, 0x1e, 0x8f, 0xcc, 0x94, 0x3c, 0xa6, 0x24, 0xa4, 0x10, 0x76, 0x2f, 0x6a, 0x11, 0x7, 0x14, 0xf3, 0x82, 0xa4, 0x99, 0xe9, 0xc0, 0xbb, 0x89, 0xae, 0x3, 0x9a, 0xd, 0xff, 0xd5, 0xac, 0x5, 0xf4, 0x6b, 0x14, 0x85, 0x36, 0x85, 0x1d, 0x62, 0x15, 0x86, 0xae, 0xd3, 0xfa, 0xa, 0xf8, 0xf9, 0x6d, 0x81, 0x9b, 0x83, 0xbe, 0xdd, 0x7, 0x31, 0xfa, 0xd0, 0xd3, 0x53, 0xc0, 0x0, 0x26, 0x40, 0xd7, 0x67, 0x16, 0x68, 0xc9, 0xf0, 0x18, 0x3c, 0x6a, 0x6a, 0x62, 0xef, 0x6f, 0x7, 0xee, 0x45, 0x69, 0x2b, 0xa3, 0x69, 0x8b, 0x48, 0x13, 0xbc, 0xff, 0xc2, 0x9c, 0xd4, 0xba, 0x51, 0xe6, 0x5c, 0x57, 0xe5, 0xb9, 0xcb, 0x77, 0x67, 0x7, 0xf6, 0x9c, 0x6a, 0x77, 0x43, 0x82, 0xb1, 0x9, 0x8b, 0x2, 0x95, 0xa0, 0xb7, 0x6e, 0xa6, 0x70, 0x49, 0x6d, 0x36, 0x4a, 0x1c, 0x3a, 0x7f, 0x20, 0xef, 0x32, 0xa, 0xd4, 0xbd, 0xb7, 0xd7, 0xc1, 0xd1, 0x4f, 0x4b, 0x14, 0x47, 0x85, 0x53, 0xc0, 0x3f, 0x25, 0x76, 0x97, 0x65, 0xf1, 0xce, 0x4e, 0xca, 0xa3, 0x9b, 0xd3, 0x70, 0x47, 0x23, 0x58, 0x69, 0xc5, 0x96, 0xe2, 0x7e, 0xd1, 0xf3, 0x54, 0xb9, 0xcd, 0xed, 0xf6, 0xb5, 0x8c, 0x25, 0xf2, 0x45, 0x9c, 0xfa, 0x9d, 0x5c, 0xcb, 0x40, 0xfe, 0x64, 0x27, 0x70, 0x9d, 0x75, 0xc6, 0x1d, 0x1, 0xbe, 0x42, 0x27, 0x53, 0xeb, 0xbc, 0xd1, 0xac, 0xdb, 0xab, 0x21, 0x27, 0x73, 0x48, 0x5a, 0x29, 0x30, 0x67, 0xb8, 0x67, 0x53, 0x11, 0xa7, 0xee, 0xe8, 0x67, 0xf8, 0xed, 0xb4, 0xed, 0x9c, 0x3b, 0x3, 0x2c, 0xed, 0x14, 0xdc, 0x1d, 0x6d, 0x26, 0xde, 0x3d, 0xe3, 0x88, 0xe5, 0x50, 0x96, 0xf1, 0x7b, 0x54, 0x81, 0xb0, 0x5d, 0x45, 0x9f, 0xc7, 0x23, 0x9b, 0x5d, 0x10, 0xdb, 0x34, 0x90, 0x0, 0xa8, 0xaa, 0x41, 0x13, 0x8e, 0xc1, 0x28, 0x28, 0x59, 0x92, 0x99, 0x10, 0xf6, 0xe0, 0x22, 0x2d, 0x93, 0x7c, 0x69, 0xb4, 0x48, 0xb4, 0xb2, 0xa4, 0x46, 0xf5, 0x73, 0x90, 0x3e, 0xda, 0xbd, 0xa0, 0xd6, 0x35, 0x47, 0xc0, 0x83, 0xa7, 0xb7, 0xba, 0xc8, 0x2c, 0x33, 0x5a, 0x29, 0x85, 0x7e, 0x4d, 0x38, 0xbc, 0xb5, 0x60, 0xa1, 0xc7, 0x59, 0xaf, 0xd1, 0xe0, 0x81, 0x91, 0x11, 0x67, 0xbe, 0x79, 0xc1, 0x26, 0x93, 0xc6, 0x52, 0x1b, 0x22, 0x4c, 0xde, 0x58, 0x90, 0x20, 0xae, 0xc0, 0xa0, 0xea, 0xa7, 0x35, 0x7f, 0xb6, 0x5d, 0x32, 0xc9, 0x44, 0x30, 0x4b, 0x8e, 0x95, 0x43, 0x31, 0x44, 0x6a, 0xde, 0x72, 0xcd, 0x38, 0x6f, 0x20, 0xb8, 0x2c, 0x65, 0x6c, 0x4c, 0x76, 0xe1, 0x46, 0x9f, 0x2a, 0x3b, 0x3e, 0x6c, 0x31, 0x33, 0x69, 0x39, 0x70, 0x82, 0xe4, 0x31, 0xe0, 0x79, 0xda, 0x89, 0x7c, 0x80, 0x13, 0x3f, 0xff, 0x40, 0xff, 0x6, 0x8d, 0xe5, 0xac, 0x68, 0x17, 0x13, 0x21, 0x99, 0xa1, 0xe9, 0xd, 0x6a, 0x3f, 0xca, 0xac, 0x58, 0xcb, 0xea, 0xba, 0x69, 0x86, 0x4, 0xaa, 0x7c, 0xb3, 0x60, 0x44, 0x46, 0x85, 0x3e, 0x15, 0xb3, 0xf8, 0x1a, 0x1a, 0x63, 0xaf, 0x96, 0x67, 0x48, 0x83, 0xc6, 0x3c, 0xdf, 0x82, 0xce, 0xec, 0x17, 0xbb, 0x50, 0xd, 0x26, 0xab, 0x1e, 0xce, 0x26, 0xa6, 0x2, 0x66, 0xc, 0xed, 0x9e, 0xcf, 0xa0, 0x0, 0x94, 0x65, 0x1a, 0x69, 0xdf, 0xb6, 0x13, 0x3e, 0x8c, 0xdb, 0xd5, 0x1c, 0x67, 0x7e, 0x2b, 0x1c, 0xcc, 0xb3, 0xe2, 0x33, 0x1b, 0xf6, 0x3a, 0x49, 0xef, 0x43, 0x5c, 0xb6, 0xfe, 0x95, 0xf7, 0x7e, 0xf5, 0x63, 0xd7, 0xa, 0x5f, 0x25, 0x6a, 0xea, 0x84, 0xe, 0x5d, 0xac, 0x36, 0x6d, 0x50, 0x75, 0x59, 0x92, 0x6c, 0xb7, 0xe2, 0x94, 0x3c, 0xdf, 0x6, 0x4e, 0x16, 0x34, 0x2b, 0xee, 0xa6, 0xd1, 0x75, 0xa8, 0x53, 0x68, 0x86, 0x9f, 0xbe, 0x3a, 0xf8, 0x6a, 0xe2, 0x71, 0xb6, 0x80, 0x92, 0x1b, 0xdf, 0xca, 0x67, 0x43, 0x4a, 0x83, 0x3, 0x64, 0x5d, 0x89, 0x40, 0xe5, 0xe2, 0x38, 0x93, 0x66, 0xd9, 0x7c, 0x3d, 0xa6, 0xb5, 0x38, 0x4f, 0xbe, 0x5b, 0x8d, 0xbe, 0x77, 0x8a, 0x36, 0x48, 0xc0, 0xdc, 0xb0, 0xf8, 0xaa, 0x84, 0x87, 0x7, 0xd8, 0x3, 0xa7, 0x96, 0xfc, 0x1b, 0xea, 0x86, 0x2e, 0x8, 0xe1, 0x88, 0x33, 0x31, 0xd8, 0x26, 0x40, 0x1d, 0xc2, 0x8, 0x83, 0xf, 0xc6, 0x26, 0x26, 0xd8, 0xdc, 0xc2, 0x93, 0x2b, 0x4, 0x73, 0x7f, 0x3, 0xfc, 0xa9, 0x8a, 0xff, 0x47, 0xaa, 0x8f, 0xa2, 0xb0, 0xe2, 0x56, 0xc0, 0xd0, 0x51, 0xb5, 0xe7, 0xd3, 0x6b, 0xbf, 0x4c, 0x3e, 0x6c, 0x5b, 0x19, 0x61, 0x4f, 0x7e, 0x3c, 0x36, 0xcd, 0x9a, 0xd2, 0x57, 0x3b, 0x3e, 0xa5, 0x2b, 0xaa, 0x6f, 0x38, 0x70, 0x2a, 0xb4, 0x14, 0x6b, 0x17, 0xc7, 0x31, 0x1a, 0x2b, 0xa9, 0xa1, 0x13, 0xf3, 0xaf, 0xf7, 0x54, 0x98, 0x69, 0xc9, 0x1a, 0x85, 0x14, 0xb6, 0x90, 0x67, 0x13, 0xc8, 0x9a, 0x48, 0x51, 0xaa, 0x6d, 0xf2, 0x19, 0x5d, 0x68, 0x59, 0x36, 0xe1, 0x4a, 0x1d, 0xb8, 0xe6, 0x8b, 0x1f, 0xc, 0x4c, 0xc8, 0x65, 0xdf, 0x33, 0x4f, 0xca, 0x2, 0xe4, 0x33, 0x25, 0x99, 0xee, 0xad, 0x6c, 0x2a, 0x47, 0x70, 0x4f, 0xfd, 0xaf, 0x6c, 0x9c, 0x5, 0x62, 0xce, 0x4f, 0x69, 0xa9, 0x4, 0x42, 0x2a, 0xef, 0xe6, 0x81, 0x7d, 0x39, 0x9e, 0x54, 0xff, 0x27, 0x2b, 0x9f, 0x6f, 0xdb, 0xc7, 0x8e, 0xed, 0x24, 0x77, 0xcf, 0xda, 0xb7, 0x53, 0x42, 0xff, 0x34, 0x8c, 0x99, 0xe, 0x56, 0x9d, 0xee, 0xbb, 0xf2, 0xa8, 0xfb, 0xd0, 0x1d, 0xb6, 0xd1, 0x16, 0x80, 0x6f, 0x57, 0x58, 0x6c, 0x6b, 0x6d, 0xf2, 0x79, 0x1c, 0xf6, 0x6, 0x3d, 0xaf, 0x53, 0x64, 0x79, 0x13, 0xcf, 0xe2, 0x3e, 0x13, 0x65, 0xf0, 0xd7, 0xc1, 0x84, 0x15, 0x66, 0x72, 0xe6, 0x8a, 0x9c, 0x66, 0x7f, 0x15, 0xb6, 0x1f, 0x33, 0x89, 0x11, 0x86, 0xb9, 0x7e, 0x5b, 0x8f, 0x4f, 0x5a, 0xdf, 0x34, 0xed, 0x81, 0xea, 0xb6, 0x7a, 0xa0, 0x85, 0x5, 0x86, 0xff, 0x10, 0x65, 0xe4, 0x2f, 0x1, 0x6, 0x14, 0xaa, 0x38, 0x61, 0xd9, 0x9a, 0x9c, 0xa3, 0x34, 0x6a, 0x90, 0xd, 0x89, 0x7b, 0xf8, 0x4c, 0x3b, 0xf6, 0x9a, 0xe2, 0xb4, 0xd8, 0xc2, 0x8b, 0x3a, 0x5f, 0x48, 0x9d, 0x2, 0xb5, 0x18, 0xa7, 0xa9, 0x3d, 0xf, 0xde, 0xb6, 0xc9, 0x32, 0x14, 0xe3, 0x69, 0x29, 0x7e, 0x5f, 0x7e, 0x33, 0xbb, 0xfd, 0xcf, 0xc8, 0xe6, 0xba, 0x46, 0xdc, 0xbc, 0x1, 0x91, 0xa, 0xc, 0x94, 0xad, 0x5a, 0xc5, 0x45, 0x68, 0x86, 0x6b, 0x82, 0x38, 0xf2, 0x9d, 0x2a, 0x6e, 0xd0, 0xd8, 0x23, 0x21, 0x56, 0xb0, 0x3e, 0xf4, 0x30, 0x6f, 0xab, 0xf6, 0x93, 0xfd, 0x8a, 0x50, 0x45, 0x17, 0xa4, 0xa2, 0xe1, 0x59, 0x9c, 0xce, 0xa6, 0x62, 0x40, 0x53, 0x1b, 0x76, 0xea, 0x56, 0x68, 0x2f, 0x9f, 0x76, 0xa2, 0xda, 0xbd, 0x4, 0xdd, 0x3d, 0x3d, 0x39, 0xc3, 0x8, 0xf3, 0x76, 0x44, 0x79, 0xb5, 0x1, 0x7, 0x75, 0x47, 0x78, 0xdc, 0xdc, 0x2b, 0x1, 0xbc, 0x67, 0xec, 0xf8, 0xb3, 0x15, 0x1f, 0x72, 0x1, 0x0, 0x90, 0x49, 0xe7, 0x48, 0xcc, 0x41, 0xe5, 0xdf, 0x9e, 0xc2, 0x7d, 0x4d, 0x57, 0x3b, 0x83, 0xf6, 0x65, 0x8b, 0x87, 0x9, 0x43, 0x88, 0x8, 0x22, 0xda, 0x98, 0xec, 0x2b, 0xe6, 0xd9, 0x31, 0x17, 0xd4, 0x56, 0xf9, 0xb7, 0x14, 0xc0, 0x8b, 0x6a, 0xb8, 0xa7, 0x74, 0xbf, 0xde, 0x4d, 0x60, 0xba, 0xdf, 0x8a, 0x1a, 0xea, 0x95, 0xc7, 0x4a, 0x52, 0xdc, 0xe8, 0x20, 0xc8, 0xfd, 0x60, 0x43, 0xf4, 0x9e, 0x54, 0x1f, 0x66, 0x19, 0x7e, 0x3c, 0x7c, 0xe2, 0x20, 0xbc, 0x7, 0x8, 0xa1, 0x43, 0x4c, 0x37, 0x2, 0x94, 0x5a, 0xfc, 0xc4, 0x73, 0x19, 0xd1, 0x15, 0xc4, 0x10, 0x89, 0x92, 0x9a, 0x16, 0x5b, 0xb, 0x31, 0xfb, 0xce, 0x20, 0xfc, 0x18, 0x3f, 0x58, 0xdc, 0x3c, 0x21, 0x1d, 0xb4, 0xf7, 0x93, 0xf8, 0xb2, 0xbe, 0xb5, 0xbe, 0x8c, 0x17, 0xbc, 0x49, 0x22, 0x5, 0x40, 0x99, 0x59, 0x48, 0x39, 0xf5, 0x36, 0x70, 0x9f, 0x70, 0x65, 0xf6, 0xed, 0x39, 0xb2, 0x22, 0xef, 0x62, 0x5c, 0xae, 0xdc, 0x2b, 0x77, 0x83, 0x15, 0xf4, 0x9b, 0x77, 0xbb, 0xfb, 0x32, 0x0, 0x42, 0xb7, 0x54, 0xf0, 0x97, 0x66, 0x60, 0xfd, 0x9a, 0xa6, 0x43, 0xb5, 0x74, 0xf6, 0x12, 0x58, 0x95, 0x8f, 0xdb, 0x78, 0xe5, 0xd8, 0x49, 0x3a, 0xb8, 0x7c, 0x9, 0x70, 0x39, 0x42, 0xfb, 0xfa, 0x98, 0x82, 0x95, 0x25, 0x39, 0x2e, 0x97, 0x7a, 0x87, 0xb2, 0x7c, 0xfd, 0xee, 0xac, 0x10, 0x84, 0xea, 0xc, 0xc1, 0xb9, 0x4e, 0x89, 0x2e, 0x67, 0x9e, 0x5e, 0xfa, 0xe4, 0xc1, 0xd2, 0xc2, 0x3d, 0x70, 0x9c, 0xa, 0xc6, 0x3a, 0x98, 0x92, 0xfc, 0xa9, 0xbd, 0x83, 0x37, 0xa9, 0xb1, 0xe9, 0x29, 0xc7, 0xec, 0x91, 0x5, 0x7a, 0x55, 0x21, 0xd5, 0xb5, 0x93, 0x24, 0xf, 0xf1, 0xc, 0xa9, 0x1f, 0x50, 0x62, 0x6a, 0x2e, 0xb0, 0x47, 0x78, 0x40, 0x99, 0xb0, 0x81, 0x7f, 0x65, 0x97, 0xc9, 0x6d, 0xbf, 0xed, 0xff, 0x22, 0x52, 0x6e, 0xf1, 0xf9, 0x84, 0xe2, 0xfa, 0xff, 0x81, 0xe2, 0x8f, 0xfd, 0x40, 0xa3, 0xc3, 0x2f, 0x4f, 0x8b, 0x3b, 0x83, 0xd1, 0x89, 0xce, 0x52, 0x2a, 0x28, 0xe, 0x8a, 0xbf, 0x5b, 0x63, 0xc9, 0xf1, 0x46, 0x24, 0x5f, 0xc3, 0xfd, 0xfe, 0x1b, 0x1a, 0xe7, 0x39, 0x26, 0xd5, 0x96, 0xb2, 0x3e, 0xf4, 0x19, 0x93, 0xec, 0x3, 0x7, 0xc5, 0xfc, 0x5c, 0x44, 0x7a, 0x17, 0xa, 0x84, 0xa6, 0xba, 0x21, 0x6d, 0xf2, 0x2c, 0x37, 0x52, 0xed, 0x5b, 0xd2, 0x93, 0x5a, 0xe1, 0x98, 0xb2, 0xa1, 0x5e, 0x8d, 0x92, 0x4a, 0x43, 0x69, 0x66, 0xf7, 0x2b, 0x39, 0x95, 0xe5, 0x74, 0x2e, 0x26, 0x1f, 0x30, 0x32, 0x7f, 0x80, 0x36, 0x4c, 0xc0, 0xe6, 0xd8, 0x18, 0x1b, 0x49, 0x66, 0xd0, 0xa6, 0x6a, 0xb3, 0xa5, 0xbb, 0x84, 0xcc, 0x8f, 0x57, 0x86, 0xfe, 0x75, 0x20, 0xcc, 0xf2, 0x91, 0xe1, 0xa5, 0xae, 0x4f, 0x77, 0x43, 0xbd, 0xfe, 0x94, 0x28, 0x8b, 0xfc, 0xe8, 0x16, 0xd, 0x9e, 0x4f, 0x93, 0xe2, 0x56, 0x75, 0xdb, 0xcc, 0x9a, 0x4b, 0xa0, 0xb4, 0xb, 0xf5, 0x87, 0xf6, 0xcc, 0x8b, 0x3f, 0x55, 0x15, 0x1d, 0x27, 0x48, 0xa7, 0x84, 0x6f, 0x68, 0xca, 0x51, 0x4f, 0xc, 0xa7, 0x1b, 0xb0, 0x8a, 0x34, 0x6c, 0x56, 0x1c, 0x72, 0x76, 0xe6, 0x10, 0xb0, 0xb5, 0xf9, 0x67, 0x1, 0x61, 0x0, 0xe3, 0x9c, 0xb8, 0x51, 0x7a, 0x90, 0xaa, 0xa2, 0x23, 0xb0, 0xdf, 0x99, 0x9f, 0x10, 0xc3, 0xa3, 0xc, 0xba, 0xad, 0xd9, 0x60, 0xc9, 0x8e, 0xce, 0x46, 0x88, 0x37, 0x80, 0xaf, 0x85, 0xeb, 0xbc, 0x37, 0xaf, 0x89, 0x85, 0xa2, 0xbf, 0xfd, 0x89, 0x76, 0x40, 0x37, 0x8c, 0x61, 0x86, 0x2e, 0x2a, 0x97, 0x2e, 0xe2, 0xa8, 0x28, 0xe9, 0x80, 0x10, 0x56, 0x24, 0x83, 0xb4, 0x2d, 0x5c, 0x55, 0x80, 0xa7, 0x6b, 0x16, 0xde, 0x76, 0x7f, 0x92, 0x32, 0x9c, 0x16, 0x1d, 0x91, 0x7, 0xe6, 0xde, 0x1b, 0x90, 0xb5, 0x10, 0x9c, 0xf, 0x13, 0x85, 0x14, 0x76, 0x2f, 0x29, 0x2, 0x9a, 0x12, 0xa8, 0x40, 0xf5, 0x8e, 0xa9, 0x1e, 0x8d, 0x5a, 0x76, 0x42, 0x28, 0x67, 0x1a, 0x85, 0x55, 0xde, 0xb4, 0x1c, 0x3c, 0x6, 0x91, 0x66, 0xed, 0x2e, 0x30, 0xd6, 0x9c, 0x3a, 0x9f, 0x97, 0x1a, 0x42, 0x31, 0x2c, 0x61, 0x5, 0x3d, 0xf9, 0x68, 0x77, 0x5d, 0x61, 0x2c, 0x42, 0xab, 0x45, 0x47, 0x2b, 0xee, 0xb4, 0xc5, 0x37, 0x13, 0xbf, 0xaf, 0xb1, 0x6f, 0x76, 0x59, 0x37, 0x2, 0x7a, 0xc7, 0x82, 0x8c, 0x90, 0x3a, 0x3c, 0xd8, 0x91, 0xe2, 0x96, 0xf8, 0x71, 0xea, 0xeb, 0x4e, 0x36, 0xf8, 0xd9, 0xf7, 0x1, 0x19, 0xde, 0x13, 0xda, 0x9a, 0x75, 0xa2, 0x3b, 0x3, 0x2c, 0x12, 0x47, 0x48, 0xc9, 0xb9, 0x30, 0x80, 0xd8, 0x17, 0xf8, 0x54, 0x20, 0xc5, 0x10, 0xdc, 0xf3, 0x25, 0x92, 0x24, 0x96, 0x22, 0xf, 0xe9, 0x99, 0x6f, 0x12, 0x10, 0x82, 0x2, 0x82, 0x7c, 0x7d, 0xf0, 0x14, 0x27, 0x18, 0xb9, 0xc2, 0x98, 0x93, 0x78, 0x45, 0x93, 0x52, 0xef, 0x1d, 0x30, 0xcd, 0xc4, 0x36, 0x8d, 0x7d, 0x9c, 0xdc, 0x6a, 0x17, 0xe7, 0x7a, 0xc8, 0xf4, 0xc3, 0x5f, 0x5d, 0xbc, 0x7d, 0x53, 0x6e, 0x68, 0xd2, 0x3d, 0xeb, 0xf9, 0x4e, 0x53, 0x1, 0xf8, 0xd2, 0xe0, 0xa0, 0x3b, 0xcd, 0x85, 0x33, 0x1, 0xaf, 0x9b, 0x60, 0x35, 0x2d, 0x92, 0xb4, 0xf8, 0x23, 0x19, 0x5c, 0xbf, 0xff, 0xe3, 0x12, 0x39, 0x76, 0x9d, 0x5, 0x30, 0x0, 0x80, 0xea, 0x79, 0x9f, 0xe6, 0x57, 0xae, 0x46, 0x55, 0x46, 0xd3, 0x25, 0x35, 0x57, 0x39, 0x27, 0x63, 0xa0, 0x41, 0xd2, 0xb9, 0xc5, 0x76, 0x14, 0xa7, 0xd, 0x6a, 0xf7, 0x7d, 0xf0, 0x46, 0xd9, 0xf7, 0xde, 0x24, 0xff, 0x13, 0x1c, 0x2a, 0xca, 0x90, 0xa5, 0xe5, 0x34, 0xbd, 0x5e, 0xa5, 0xf, 0xe6, 0x59, 0xd4, 0x99, 0x20, 0x62, 0xc0, 0x6f, 0xf5, 0xd1, 0xc7, 0x62, 0x7d, 0xea, 0x98, 0xcf, 0x2f, 0xe9, 0xaa, 0x8f, 0xb5, 0xed, 0x9d, 0xb, 0x98, 0x2f, 0x2f, 0x2e, 0xf0, 0x7a, 0x23, 0x11, 0x9c, 0x8a, 0xd2, 0xcb, 0x84, 0xa1, 0x7, 0xc3, 0x9c, 0x6f, 0x7e, 0x11, 0x4c, 0x38, 0xa2, 0x5e, 0xe5, 0x5c, 0x48, 0x2a, 0x68, 0xd9, 0x11, 0x52, 0x95, 0xd2, 0x80, 0x3b, 0x9c, 0x52, 0x22, 0x47, 0x3e, 0x5c, 0xd3, 0xcb, 0x8e, 0x53, 0xa2, 0x7b, 0xd8, 0x41, 0xf2, 0x9, 0x3e, 0x64, 0x42, 0x4f, 0x72, 0x85, 0x53, 0x12, 0xc5, 0xfb, 0x4f, 0x79, 0x53, 0xbf, 0xd0, 0xb2, 0x8e, 0x3, 0xe0, 0x24, 0xc3, 0x9e, 0x6a, 0xf9, 0x48, 0xef, 0x3a, 0x36, 0x6f, 0xee, 0x9e, 0x64, 0xaf, 0x1c, 0xb3, 0x0, 0x4d, 0x97, 0xf9, 0x8b, 0xb1, 0xad, 0x1b, 0xef, 0xb, 0xcf, 0x69, 0xa8, 0x3c, 0x98, 0x96, 0x1d, 0xbd, 0xeb, 0x6c, 0x5f, 0xe5, 0x66, 0xf5, 0xb0, 0x8b, 0x15, 0x9a, 0x3b, 0x58, 0xc6, 0xee, 0x62, 0x51, 0xe3, 0xf8, 0x33, 0x1, 0x92, 0xb2, 0x8f, 0x30, 0x22, 0x8a, 0x3e, 0x18, 0x9b, 0x8f, 0x4c, 0x3, 0x4e, 0x91, 0xdc, 0x4d, 0x11, 0x4b, 0x2f, 0xcb, 0x84, 0xa5, 0x7f, 0x69, 0x3f, 0x30, 0x47, 0x29, 0x1b, 0x69, 0xc0, 0x96, 0x8f, 0x12, 0xf9, 0x1a, 0x19, 0x91, 0x0, 0xcd, 0x83, 0xe8, 0xd0, 0xb1, 0x12, 0x8, 0xc3, 0x59, 0x89, 0x9f, 0xc, 0xfe, 0xb2, 0x55, 0x8f, 0x81, 0x77, 0x61, 0x10, 0x43, 0x4b, 0x61, 0xa0, 0x19, 0xdd, 0x52, 0xc7, 0x2e, 0xa9, 0x39, 0xc2, 0x5, 0x69, 0x68, 0x88, 0xaa, 0x60, 0x6a, 0x8a, 0xa6, 0x22, 0xb0, 0x4b, 0xd4, 0x14, 0xe2, 0x16, 0x2d, 0xab, 0x4, 0xfd, 0x39, 0xd1, 0xf2, 0x93, 0xae, 0x2d, 0xba, 0x8, 0x4a, 0x79, 0x9b, 0x67, 0x50, 0x10, 0x88, 0xbb, 0x75, 0xd8, 0x5a, 0x7a, 0xf5, 0x30, 0xf3, 0x3f, 0xc6, 0x4, 0xb9, 0xde, 0xfd, 0xa, 0x0, 0x8f, 0x37, 0x10, 0xcf, 0x87, 0x59, 0x72, 0xbb, 0x9d, 0x4d, 0xed, 0xb3, 0xf, 0xfa, 0x79, 0x26, 0x68, 0xf2, 0x9e, 0xd2, 0x2b, 0xb, 0xf6, 0x34, 0x11, 0x5c, 0x8e, 0x4d, 0x7b, 0x81, 0x3e, 0xe8, 0xcc, 0x2f, 0xcc, 0xae, 0xe5, 0x17, 0x24, 0x55, 0x79, 0x45, 0xe8, 0x2e, 0x66, 0x9b, 0x48, 0x67, 0xee, 0x83, 0xc6, 0xef, 0x7a, 0x1e, 0x94, 0x7, 0x73, 0x3c, 0x79, 0xf7, 0x59, 0xb, 0xbc, 0xc6, 0x2b, 0x66, 0xfd, 0xe, 0x9d, 0xcf, 0xc0, 0x9f, 0xc8, 0x75, 0xfd, 0xa9, 0x36, 0x2b, 0xa3, 0x5, 0x7b, 0xc4, 0x48, 0x33, 0x9b, 0xe9, 0x32, 0xa0, 0x24, 0x7f, 0xe4, 0xdf, 0xe4, 0x41, 0xea, 0xa2, 0x98, 0xbb, 0x41, 0x84, 0x17, 0x25, 0x5d, 0x4d, 0xd6, 0xe4, 0x3, 0xb9, 0xb1, 0x2, 0x1f, 0x7d, 0x54, 0x3a, 0x76, 0x1e, 0xb2, 0x0, 0x62, 0xe5, 0x7c, 0x5c, 0xf5, 0x82, 0x86, 0x85, 0xaf, 0x70, 0x66, 0x91, 0x62, 0xde, 0xb8, 0x5d, 0x95, 0x7f, 0x72, 0xd1, 0xe2, 0xb2, 0x1d, 0x9b, 0x4b, 0x16, 0x74, 0x95, 0xbb, 0x97, 0x8f, 0x34, 0xf0, 0x87, 0x55, 0xd1, 0x65, 0xe7, 0x2f, 0xb3, 0x30, 0x5f, 0x77, 0x74, 0x6b, 0xa9, 0x99, 0x43, 0x13, 0x56, 0x5b, 0xd2, 0x6d, 0x49, 0x80, 0x16, 0x1f, 0xa5, 0x84, 0xba, 0x71, 0xea, 0x7f, 0x1b, 0x62, 0x70, 0x33, 0xa, 0xd8, 0xd0, 0x21, 0xd, 0xbd, 0x7c, 0x27, 0xfa, 0xa4, 0x79, 0x4f, 0xae, 0x96, 0x84, 0x8d, 0x28, 0x13, 0x41, 0xa8, 0x1b, 0xf7, 0x32, 0xc7, 0xe1, 0xb, 0xfe, 0x68, 0xec, 0x47, 0x72, 0xc7, 0xa0, 0x64, 0xd0, 0x53, 0xf5, 0xf, 0x19, 0x4b, 0x44, 0xd4, 0xcb, 0xfc, 0x60, 0x94, 0x5d, 0x44, 0x13, 0x89, 0xb, 0x80, 0x89, 0xf9, 0xb1, 0x1, 0xa5, 0xec, 0x28, 0x70, 0xa1, 0x52, 0x35, 0xe, 0x44, 0x41, 0xfa, 0xca, 0xb2, 0xb3, 0x2e, 0x4d, 0xf5, 0xf5, 0xf5, 0xca, 0xe5, 0x1e, 0x81, 0xce, 0x45, 0xbb, 0xc1, 0x2e, 0x46, 0x2a, 0xee, 0xf, 0xa6, 0xf2, 0x1f, 0x75, 0xd1, 0x3d, 0xb0, 0x5f, 0x81, 0x4a, 0xbb, 0x27, 0xae, 0xe9, 0xda, 0x69, 0xa1, 0x7a, 0xe1, 0x89, 0x30, 0xd, 0x54, 0xd4, 0x4d, 0xb, 0xc7, 0xcf, 0x48, 0xf5, 0xed, 0xe8, 0x72, 0xd9, 0xa5, 0x81, 0x58, 0x13, 0x95, 0xf2, 0x1b, 0x66, 0x51, 0x9, 0x57, 0xc, 0x8e, 0xb0, 0x20, 0x8a, 0x1f, 0x36, 0x75, 0xa5, 0x6b, 0x4, 0xe9, 0xe, 0xb2, 0xf6, 0x70, 0x21, 0xe1, 0x4, 0xea, 0xd3, 0x93, 0x3b, 0xde, 0x33, 0xa5, 0xb4, 0xcd, 0x70, 0xcc, 0xf9, 0x24, 0x5e, 0xa4, 0x72, 0x4d, 0x79, 0x1d, 0xc0, 0x60, 0x99, 0x13, 0xc2, 0x8f, 0xd1, 0x47, 0xef, 0xac, 0xe2, 0xa4, 0xb7, 0xcb, 0xf, 0x12, 0x91, 0x44, 0xa0, 0x83, 0x92, 0x55, 0x8f, 0x72, 0x68, 0xeb, 0x4b, 0xa, 0x43, 0x30, 0x76, 0x40, 0x5d, 0xb5, 0x86, 0x4a, 0x0, 0x28, 0x36, 0xe8, 0x1d, 0xda, 0xe1, 0x69, 0x5e, 0xf9, 0x15, 0xbe, 0x6a, 0xe5, 0x48, 0xef, 0x7e, 0x67, 0xa, 0x4c, 0x9a, 0x22, 0x40, 0xb0, 0xff, 0x9d, 0xca, 0xa6, 0x32, 0x89, 0xa7, 0xe9, 0xbf, 0xb2, 0xa1, 0xe6, 0x82, 0x86, 0x2f, 0xb3, 0x94, 0x9f, 0xb5, 0xb9, 0x79, 0xd9, 0xce, 0xdc, 0x92, 0xd4, 0x7b, 0x8d, 0xf2, 0x36, 0xf5, 0x66, 0xfd, 0x22, 0x53, 0x14, 0xb1, 0x23, 0x32, 0x72, 0xf7, 0xb1, 0xfa, 0x81, 0x91, 0x98, 0xe6, 0xc3, 0x4e, 0x42, 0x12, 0x6c, 0xee, 0x58, 0xa8, 0xa4, 0xd, 0x9e, 0xde, 0x28, 0x6e, 0xca, 0x43, 0x15, 0xee, 0x1d, 0xbc, 0xf1, 0x2c}
	if got := par.forsSign(md, skSeed, pkSeed, adrs); !slices.Equal(got, want) {
		t.Errorf("par.forsSign() = %v, want %v", got, want)
	}
}

func TestForsPkFromSigKat(t *testing.T) {
	par := SLH_DSA_SHA2_128s
	sig := []byte{0x62, 0x88, 0x6a, 0x8d, 0x36, 0xd4, 0xb0, 0xfd, 0xa, 0xc1, 0xfa, 0x12, 0x51, 0xeb, 0x2b, 0x9f, 0x41, 0xa9, 0xed, 0x1f, 0x9c, 0x13, 0x79, 0xaf, 0x6f, 0x2e, 0x26, 0x8, 0x0, 0x48, 0xf2, 0x1d, 0xf9, 0x44, 0xa9, 0x15, 0xde, 0x49, 0xa1, 0xb9, 0x60, 0x78, 0xba, 0x47, 0xc6, 0x3e, 0x28, 0x9, 0xd, 0xe1, 0xbb, 0xd3, 0xb7, 0xee, 0x2e, 0xf2, 0x37, 0x87, 0x9c, 0x18, 0x31, 0x35, 0x3, 0x34, 0x8c, 0xc7, 0x5b, 0x6b, 0xcf, 0xab, 0x13, 0x2f, 0x48, 0xae, 0xa2, 0x15, 0x3a, 0x19, 0x36, 0x65, 0x78, 0xb2, 0xc8, 0x5f, 0x1, 0xbf, 0x7d, 0x47, 0xd9, 0xa9, 0x6b, 0x1b, 0xa6, 0x53, 0x7f, 0xba, 0x8c, 0xda, 0x6e, 0x47, 0xbe, 0x6a, 0x53, 0x75, 0xcf, 0x8e, 0x63, 0xfd, 0x36, 0x71, 0xfb, 0x3b, 0x5c, 0x38, 0x79, 0xed, 0xc6, 0x52, 0x8, 0x70, 0xea, 0x26, 0xc6, 0x94, 0xbf, 0x29, 0xa6, 0xde, 0x85, 0x23, 0x35, 0xb7, 0x48, 0x40, 0xd4, 0x3d, 0xe8, 0xfe, 0xe4, 0x49, 0xdc, 0x4e, 0xe9, 0x52, 0x1e, 0x41, 0x36, 0x8b, 0x81, 0xda, 0x39, 0xae, 0x2, 0x3b, 0xc5, 0xe5, 0xd7, 0x75, 0xc, 0xb0, 0x3d, 0x86, 0x4c, 0x55, 0x4a, 0x42, 0x7, 0xf9, 0xd6, 0x0, 0x34, 0x9a, 0x6e, 0x84, 0x5b, 0xb7, 0xc4, 0xbd, 0x52, 0x5d, 0x8b, 0x5c, 0xdb, 0x26, 0x8a, 0xed, 0x39, 0x5c, 0xb0, 0x52, 0x74, 0xb0, 0x4a, 0x27, 0xf8, 0xfe, 0xed, 0x7c, 0xe9, 0xec, 0x1d, 0x1e, 0xf, 0x6e, 0x3d, 0x5d, 0x2c, 0xb, 0x9c, 0x2, 0x64, 0xc8, 0x95, 0x8a, 0x71, 0x4d, 0x10, 0x33, 0x96, 0x1, 0xe2, 0xca, 0x87, 0x65, 0x91, 0xce, 0xf9, 0x4d, 0xd, 0x7c, 0xf9, 0x1b, 0x9f, 0xf5, 0x3b, 0xc, 0xec, 0xe1, 0xc8, 0xee, 0x41, 0x1f, 0x4d, 0xc8, 0x1e, 0x13, 0xe3, 0xe6, 0x38, 0xdc, 0x86, 0x7e, 0xa0, 0x86, 0xc1, 0x23, 0x14, 0x1d, 0x2e, 0x16, 0x57, 0xf8, 0x35, 0xec, 0x97, 0xd, 0x6f, 0x6c, 0x1, 0x28, 0x41, 0xd3, 0x43, 0x72, 0x77, 0xbf, 0x2b, 0x70, 0x5d, 0x3e, 0x8, 0x7a, 0x5a, 0xad, 0x4a, 0xbf, 0x90, 0x50, 0x2d, 0xb6, 0x89, 0x2, 0x21, 0x35, 0x43, 0xa3, 0x6f, 0x9c, 0xfe, 0xae, 0x5e, 0x26, 0x62, 0x42, 0xc0, 0xde, 0x8c, 0x34, 0xb5, 0x56, 0x54, 0xda, 0xf7, 0x5a, 0xcf, 0xe8, 0xf5, 0xe3, 0xa5, 0xbc, 0x4, 0xf2, 0xed, 0x92, 0xfa, 0x22, 0xcd, 0xe9, 0xd9, 0xe8, 0xfc, 0xc3, 0x82, 0xa9, 0x5d, 0x81, 0xd9, 0x9a, 0x24, 0xbe, 0x2d, 0xc8, 0xe6, 0x1a, 0xdb, 0xb3, 0xd9, 0x46, 0xf7, 0x7d, 0x39, 0x99, 0x56, 0x4a, 0x17, 0x7c, 0xfb, 0x9c, 0x30, 0x39, 0x44, 0xcb, 0x1b, 0x88, 0xd, 0x88, 0x7, 0x4c, 0x27, 0x69, 0x67, 0x78, 0xc0, 0x1c, 0xa8, 0x69, 0xbd, 0xa2, 0x1e, 0x8f, 0xcc, 0x94, 0x3c, 0xa6, 0x24, 0xa4, 0x10, 0x76, 0x2f, 0x6a, 0x11, 0x7, 0x14, 0xf3, 0x82, 0xa4, 0x99, 0xe9, 0xc0, 0xbb, 0x89, 0xae, 0x3, 0x9a, 0xd, 0xff, 0xd5, 0xac, 0x5, 0xf4, 0x6b, 0x14, 0x85, 0x36, 0x85, 0x1d, 0x62, 0x15, 0x86, 0xae, 0xd3, 0xfa, 0xa, 0xf8, 0xf9, 0x6d, 0x81, 0x9b, 0x83, 0xbe, 0xdd, 0x7, 0x31, 0xfa, 0xd0, 0xd3, 0x53, 0xc0, 0x0, 0x26, 0x40, 0xd7, 0x67, 0x16, 0x68, 0xc9, 0xf0, 0x18, 0x3c, 0x6a, 0x6a, 0x62, 0xef, 0x6f, 0x7, 0xee, 0x45, 0x69, 0x2b, 0xa3, 0x69, 0x8b, 0x48, 0x13, 0xbc, 0xff, 0xc2, 0x9c, 0xd4, 0xba, 0x51, 0xe6, 0x5c, 0x57, 0xe5, 0xb9, 0xcb, 0x77, 0x67, 0x7, 0xf6, 0x9c, 0x6a, 0x77, 0x43, 0x82, 0xb1, 0x9, 0x8b, 0x2, 0x95, 0xa0, 0xb7, 0x6e, 0xa6, 0x70, 0x49, 0x6d, 0x36, 0x4a, 0x1c, 0x3a, 0x7f, 0x20, 0xef, 0x32, 0xa, 0xd4, 0xbd, 0xb7, 0xd7, 0xc1, 0xd1, 0x4f, 0x4b, 0x14, 0x47, 0x85, 0x53, 0xc0, 0x3f, 0x25, 0x76, 0x97, 0x65, 0xf1, 0xce, 0x4e, 0xca, 0xa3, 0x9b, 0xd3, 0x70, 0x47, 0x23, 0x58, 0x69, 0xc5, 0x96, 0xe2, 0x7e, 0xd1, 0xf3, 0x54, 0xb9, 0xcd, 0xed, 0xf6, 0xb5, 0x8c, 0x25, 0xf2, 0x45, 0x9c, 0xfa, 0x9d, 0x5c, 0xcb, 0x40, 0xfe, 0x64, 0x27, 0x70, 0x9d, 0x75, 0xc6, 0x1d, 0x1, 0xbe, 0x42, 0x27, 0x53, 0xeb, 0xbc, 0xd1, 0xac, 0xdb, 0xab, 0x21, 0x27, 0x73, 0x48, 0x5a, 0x29, 0x30, 0x67, 0xb8, 0x67, 0x53, 0x11, 0xa7, 0xee, 0xe8, 0x67, 0xf8, 0xed, 0xb4, 0xed, 0x9c, 0x3b, 0x3, 0x2c, 0xed, 0x14, 0xdc, 0x1d, 0x6d, 0x26, 0xde, 0x3d, 0xe3, 0x88, 0xe5, 0x50, 0x96, 0xf1, 0x7b, 0x54, 0x81, 0xb0, 0x5d, 0x45, 0x9f, 0xc7, 0x23, 0x9b, 0x5d, 0x10, 0xdb, 0x34, 0x90, 0x0, 0xa8, 0xaa, 0x41, 0x13, 0x8e, 0xc1, 0x28, 0x28, 0x59, 0x92, 0x99, 0x10, 0xf6, 0xe0, 0x22, 0x2d, 0x93, 0x7c, 0x69, 0xb4, 0x48, 0xb4, 0xb2, 0xa4, 0x46, 0xf5, 0x73, 0x90, 0x3e, 0xda, 0xbd, 0xa0, 0xd6, 0x35, 0x47, 0xc0, 0x83, 0xa7, 0xb7, 0xba, 0xc8, 0x2c, 0x33, 0x5a, 0x29, 0x85, 0x7e, 0x4d, 0x38, 0xbc, 0xb5, 0x60, 0xa1, 0xc7, 0x59, 0xaf, 0xd1, 0xe0, 0x81, 0x91, 0x11, 0x67, 0xbe, 0x79, 0xc1, 0x26, 0x93, 0xc6, 0x52, 0x1b, 0x22, 0x4c, 0xde, 0x58, 0x90, 0x20, 0xae, 0xc0, 0xa0, 0xea, 0xa7, 0x35, 0x7f, 0xb6, 0x5d, 0x32, 0xc9, 0x44, 0x30, 0x4b, 0x8e, 0x95, 0x43, 0x31, 0x44, 0x6a, 0xde, 0x72, 0xcd, 0x38, 0x6f, 0x20, 0xb8, 0x2c, 0x65, 0x6c, 0x4c, 0x76, 0xe1, 0x46, 0x9f, 0x2a, 0x3b, 0x3e, 0x6c, 0x31, 0x33, 0x69, 0x39, 0x70, 0x82, 0xe4, 0x31, 0xe0, 0x79, 0xda, 0x89, 0x7c, 0x80, 0x13, 0x3f, 0xff, 0x40, 0xff, 0x6, 0x8d, 0xe5, 0xac, 0x68, 0x17, 0x13, 0x21, 0x99, 0xa1, 0xe9, 0xd, 0x6a, 0x3f, 0xca, 0xac, 0x58, 0xcb, 0xea, 0xba, 0x69, 0x86, 0x4, 0xaa, 0x7c, 0xb3, 0x60, 0x44, 0x46, 0x85, 0x3e, 0x15, 0xb3, 0xf8, 0x1a, 0x1a, 0x63, 0xaf, 0x96, 0x67, 0x48, 0x83, 0xc6, 0x3c, 0xdf, 0x82, 0xce, 0xec, 0x17, 0xbb, 0x50, 0xd, 0x26, 0xab, 0x1e, 0xce, 0x26, 0xa6, 0x2, 0x66, 0xc, 0xed, 0x9e, 0xcf, 0xa0, 0x0, 0x94, 0x65, 0x1a, 0x69, 0xdf, 0xb6, 0x13, 0x3e, 0x8c, 0xdb, 0xd5, 0x1c, 0x67, 0x7e, 0x2b, 0x1c, 0xcc, 0xb3, 0xe2, 0x33, 0x1b, 0xf6, 0x3a, 0x49, 0xef, 0x43, 0x5c, 0xb6, 0xfe, 0x95, 0xf7, 0x7e, 0xf5, 0x63, 0xd7, 0xa, 0x5f, 0x25, 0x6a, 0xea, 0x84, 0xe, 0x5d, 0xac, 0x36, 0x6d, 0x50, 0x75, 0x59, 0x92, 0x6c, 0xb7, 0xe2, 0x94, 0x3c, 0xdf, 0x6, 0x4e, 0x16, 0x34, 0x2b, 0xee, 0xa6, 0xd1, 0x75, 0xa8, 0x53, 0x68, 0x86, 0x9f, 0xbe, 0x3a, 0xf8, 0x6a, 0xe2, 0x71, 0xb6, 0x80, 0x92, 0x1b, 0xdf, 0xca, 0x67, 0x43, 0x4a, 0x83, 0x3, 0x64, 0x5d, 0x89, 0x40, 0xe5, 0xe2, 0x38, 0x93, 0x66, 0xd9, 0x7c, 0x3d, 0xa6, 0xb5, 0x38, 0x4f, 0xbe, 0x5b, 0x8d, 0xbe, 0x77, 0x8a, 0x36, 0x48, 0xc0, 0xdc, 0xb0, 0xf8, 0xaa, 0x84, 0x87, 0x7, 0xd8, 0x3, 0xa7, 0x96, 0xfc, 0x1b, 0xea, 0x86, 0x2e, 0x8, 0xe1, 0x88, 0x33, 0x31, 0xd8, 0x26, 0x40, 0x1d, 0xc2, 0x8, 0x83, 0xf, 0xc6, 0x26, 0x26, 0xd8, 0xdc, 0xc2, 0x93, 0x2b, 0x4, 0x73, 0x7f, 0x3, 0xfc, 0xa9, 0x8a, 0xff, 0x47, 0xaa, 0x8f, 0xa2, 0xb0, 0xe2, 0x56, 0xc0, 0xd0, 0x51, 0xb5, 0xe7, 0xd3, 0x6b, 0xbf, 0x4c, 0x3e, 0x6c, 0x5b, 0x19, 0x61, 0x4f, 0x7e, 0x3c, 0x36, 0xcd, 0x9a, 0xd2, 0x57, 0x3b, 0x3e, 0xa5, 0x2b, 0xaa, 0x6f, 0x38, 0x70, 0x2a, 0xb4, 0x14, 0x6b, 0x17, 0xc7, 0x31, 0x1a, 0x2b, 0xa9, 0xa1, 0x13, 0xf3, 0xaf, 0xf7, 0x54, 0x98, 0x69, 0xc9, 0x1a, 0x85, 0x14, 0xb6, 0x90, 0x67, 0x13, 0xc8, 0x9a, 0x48, 0x51, 0xaa, 0x6d, 0xf2, 0x19, 0x5d, 0x68, 0x59, 0x36, 0xe1, 0x4a, 0x1d, 0xb8, 0xe6, 0x8b, 0x1f, 0xc, 0x4c, 0xc8, 0x65, 0xdf, 0x33, 0x4f, 0xca, 0x2, 0xe4, 0x33, 0x25, 0x99, 0xee, 0xad, 0x6c, 0x2a, 0x47, 0x70, 0x4f, 0xfd, 0xaf, 0x6c, 0x9c, 0x5, 0x62, 0xce, 0x4f, 0x69, 0xa9, 0x4, 0x42, 0x2a, 0xef, 0xe6, 0x81, 0x7d, 0x39, 0x9e, 0x54, 0xff, 0x27, 0x2b, 0x9f, 0x6f, 0xdb, 0xc7, 0x8e, 0xed, 0x24, 0x77, 0xcf, 0xda, 0xb7, 0x53, 0x42, 0xff, 0x34, 0x8c, 0x99, 0xe, 0x56, 0x9d, 0xee, 0xbb, 0xf2, 0xa8, 0xfb, 0xd0, 0x1d, 0xb6, 0xd1, 0x16, 0x80, 0x6f, 0x57, 0x58, 0x6c, 0x6b, 0x6d, 0xf2, 0x79, 0x1c, 0xf6, 0x6, 0x3d, 0xaf, 0x53, 0x64, 0x79, 0x13, 0xcf, 0xe2, 0x3e, 0x13, 0x65, 0xf0, 0xd7, 0xc1, 0x84, 0x15, 0x66, 0x72, 0xe6, 0x8a, 0x9c, 0x66, 0x7f, 0x15, 0xb6, 0x1f, 0x33, 0x89, 0x11, 0x86, 0xb9, 0x7e, 0x5b, 0x8f, 0x4f, 0x5a, 0xdf, 0x34, 0xed, 0x81, 0xea, 0xb6, 0x7a, 0xa0, 0x85, 0x5, 0x86, 0xff, 0x10, 0x65, 0xe4, 0x2f, 0x1, 0x6, 0x14, 0xaa, 0x38, 0x61, 0xd9, 0x9a, 0x9c, 0xa3, 0x34, 0x6a, 0x90, 0xd, 0x89, 0x7b, 0xf8, 0x4c, 0x3b, 0xf6, 0x9a, 0xe2, 0xb4, 0xd8, 0xc2, 0x8b, 0x3a, 0x5f, 0x48, 0x9d, 0x2, 0xb5, 0x18, 0xa7, 0xa9, 0x3d, 0xf, 0xde, 0xb6, 0xc9, 0x32, 0x14, 0xe3, 0x69, 0x29, 0x7e, 0x5f, 0x7e, 0x33, 0xbb, 0xfd, 0xcf, 0xc8, 0xe6, 0xba, 0x46, 0xdc, 0xbc, 0x1, 0x91, 0xa, 0xc, 0x94, 0xad, 0x5a, 0xc5, 0x45, 0x68, 0x86, 0x6b, 0x82, 0x38, 0xf2, 0x9d, 0x2a, 0x6e, 0xd0, 0xd8, 0x23, 0x21, 0x56, 0xb0, 0x3e, 0xf4, 0x30, 0x6f, 0xab, 0xf6, 0x93, 0xfd, 0x8a, 0x50, 0x45, 0x17, 0xa4, 0xa2, 0xe1, 0x59, 0x9c, 0xce, 0xa6, 0x62, 0x40, 0x53, 0x1b, 0x76, 0xea, 0x56, 0x68, 0x2f, 0x9f, 0x76, 0xa2, 0xda, 0xbd, 0x4, 0xdd, 0x3d, 0x3d, 0x39, 0xc3, 0x8, 0xf3, 0x76, 0x44, 0x79, 0xb5, 0x1, 0x7, 0x75, 0x47, 0x78, 0xdc, 0xdc, 0x2b, 0x1, 0xbc, 0x67, 0xec, 0xf8, 0xb3, 0x15, 0x1f, 0x72, 0x1, 0x0, 0x90, 0x49, 0xe7, 0x48, 0xcc, 0x41, 0xe5, 0xdf, 0x9e, 0xc2, 0x7d, 0x4d, 0x57, 0x3b, 0x83, 0xf6, 0x65, 0x8b, 0x87, 0x9, 0x43, 0x88, 0x8, 0x22, 0xda, 0x98, 0xec, 0x2b, 0xe6, 0xd9, 0x31, 0x17, 0xd4, 0x56, 0xf9, 0xb7, 0x14, 0xc0, 0x8b, 0x6a, 0xb8, 0xa7, 0x74, 0xbf, 0xde, 0x4d, 0x60, 0xba, 0xdf, 0x8a, 0x1a, 0xea, 0x95, 0xc7, 0x4a, 0x52, 0xdc, 0xe8, 0x20, 0xc8, 0xfd, 0x60, 0x43, 0xf4, 0x9e, 0x54, 0x1f, 0x66, 0x19, 0x7e, 0x3c, 0x7c, 0xe2, 0x20, 0xbc, 0x7, 0x8, 0xa1, 0x43, 0x4c, 0x37, 0x2, 0x94, 0x5a, 0xfc, 0xc4, 0x73, 0x19, 0xd1, 0x15, 0xc4, 0x10, 0x89, 0x92, 0x9a, 0x16, 0x5b, 0xb, 0x31, 0xfb, 0xce, 0x20, 0xfc, 0x18, 0x3f, 0x58, 0xdc, 0x3c, 0x21, 0x1d, 0xb4, 0xf7, 0x93, 0xf8, 0xb2, 0xbe, 0xb5, 0xbe, 0x8c, 0x17, 0xbc, 0x49, 0x22, 0x5, 0x40, 0x99, 0x59, 0x48, 0x39, 0xf5, 0x36, 0x70, 0x9f, 0x70, 0x65, 0xf6, 0xed, 0x39, 0xb2, 0x22, 0xef, 0x62, 0x5c, 0xae, 0xdc, 0x2b, 0x77, 0x83, 0x15, 0xf4, 0x9b, 0x77, 0xbb, 0xfb, 0x32, 0x0, 0x42, 0xb7, 0x54, 0xf0, 0x97, 0x66, 0x60, 0xfd, 0x9a, 0xa6, 0x43, 0xb5, 0x74, 0xf6, 0x12, 0x58, 0x95, 0x8f, 0xdb, 0x78, 0xe5, 0xd8, 0x49, 0x3a, 0xb8, 0x7c, 0x9, 0x70, 0x39, 0x42, 0xfb, 0xfa, 0x98, 0x82, 0x95, 0x25, 0x39, 0x2e, 0x97, 0x7a, 0x87, 0xb2, 0x7c, 0xfd, 0xee, 0xac, 0x10, 0x84, 0xea, 0xc, 0xc1, 0xb9, 0x4e, 0x89, 0x2e, 0x67, 0x9e, 0x5e, 0xfa, 0xe4, 0xc1, 0xd2, 0xc2, 0x3d, 0x70, 0x9c, 0xa, 0xc6, 0x3a, 0x98, 0x92, 0xfc, 0xa9, 0xbd, 0x83, 0x37, 0xa9, 0xb1, 0xe9, 0x29, 0xc7, 0xec, 0x91, 0x5, 0x7a, 0x55, 0x21, 0xd5, 0xb5, 0x93, 0x24, 0xf, 0xf1, 0xc, 0xa9, 0x1f, 0x50, 0x62, 0x6a, 0x2e, 0xb0, 0x47, 0x78, 0x40, 0x99, 0xb0, 0x81, 0x7f, 0x65, 0x97, 0xc9, 0x6d, 0xbf, 0xed, 0xff, 0x22, 0x52, 0x6e, 0xf1, 0xf9, 0x84, 0xe2, 0xfa, 0xff, 0x81, 0xe2, 0x8f, 0xfd, 0x40, 0xa3, 0xc3, 0x2f, 0x4f, 0x8b, 0x3b, 0x83, 0xd1, 0x89, 0xce, 0x52, 0x2a, 0x28, 0xe, 0x8a, 0xbf, 0x5b, 0x63, 0xc9, 0xf1, 0x46, 0x24, 0x5f, 0xc3, 0xfd, 0xfe, 0x1b, 0x1a, 0xe7, 0x39, 0x26, 0xd5, 0x96, 0xb2, 0x3e, 0xf4, 0x19, 0x93, 0xec, 0x3, 0x7, 0xc5, 0xfc, 0x5c, 0x44, 0x7a, 0x17, 0xa, 0x84, 0xa6, 0xba, 0x21, 0x6d, 0xf2, 0x2c, 0x37, 0x52, 0xed, 0x5b, 0xd2, 0x93, 0x5a, 0xe1, 0x98, 0xb2, 0xa1, 0x5e, 0x8d, 0x92, 0x4a, 0x43, 0x69, 0x66, 0xf7, 0x2b, 0x39, 0x95, 0xe5, 0x74, 0x2e, 0x26, 0x1f, 0x30, 0x32, 0x7f, 0x80, 0x36, 0x4c, 0xc0, 0xe6, 0xd8, 0x18, 0x1b, 0x49, 0x66, 0xd0, 0xa6, 0x6a, 0xb3, 0xa5, 0xbb, 0x84, 0xcc, 0x8f, 0x57, 0x86, 0xfe, 0x75, 0x20, 0xcc, 0xf2, 0x91, 0xe1, 0xa5, 0xae, 0x4f, 0x77, 0x43, 0xbd, 0xfe, 0x94, 0x28, 0x8b, 0xfc, 0xe8, 0x16, 0xd, 0x9e, 0x4f, 0x93, 0xe2, 0x56, 0x75, 0xdb, 0xcc, 0x9a, 0x4b, 0xa0, 0xb4, 0xb, 0xf5, 0x87, 0xf6, 0xcc, 0x8b, 0x3f, 0x55, 0x15, 0x1d, 0x27, 0x48, 0xa7, 0x84, 0x6f, 0x68, 0xca, 0x51, 0x4f, 0xc, 0xa7, 0x1b, 0xb0, 0x8a, 0x34, 0x6c, 0x56, 0x1c, 0x72, 0x76, 0xe6, 0x10, 0xb0, 0xb5, 0xf9, 0x67, 0x1, 0x61, 0x0, 0xe3, 0x9c, 0xb8, 0x51, 0x7a, 0x90, 0xaa, 0xa2, 0x23, 0xb0, 0xdf, 0x99, 0x9f, 0x10, 0xc3, 0xa3, 0xc, 0xba, 0xad, 0xd9, 0x60, 0xc9, 0x8e, 0xce, 0x46, 0x88, 0x37, 0x80, 0xaf, 0x85, 0xeb, 0xbc, 0x37, 0xaf, 0x89, 0x85, 0xa2, 0xbf, 0xfd, 0x89, 0x76, 0x40, 0x37, 0x8c, 0x61, 0x86, 0x2e, 0x2a, 0x97, 0x2e, 0xe2, 0xa8, 0x28, 0xe9, 0x80, 0x10, 0x56, 0x24, 0x83, 0xb4, 0x2d, 0x5c, 0x55, 0x80, 0xa7, 0x6b, 0x16, 0xde, 0x76, 0x7f, 0x92, 0x32, 0x9c, 0x16, 0x1d, 0x91, 0x7, 0xe6, 0xde, 0x1b, 0x90, 0xb5, 0x10, 0x9c, 0xf, 0x13, 0x85, 0x14, 0x76, 0x2f, 0x29, 0x2, 0x9a, 0x12, 0xa8, 0x40, 0xf5, 0x8e, 0xa9, 0x1e, 0x8d, 0x5a, 0x76, 0x42, 0x28, 0x67, 0x1a, 0x85, 0x55, 0xde, 0xb4, 0x1c, 0x3c, 0x6, 0x91, 0x66, 0xed, 0x2e, 0x30, 0xd6, 0x9c, 0x3a, 0x9f, 0x97, 0x1a, 0x42, 0x31, 0x2c, 0x61, 0x5, 0x3d, 0xf9, 0x68, 0x77, 0x5d, 0x61, 0x2c, 0x42, 0xab, 0x45, 0x47, 0x2b, 0xee, 0xb4, 0xc5, 0x37, 0x13, 0xbf, 0xaf, 0xb1, 0x6f, 0x76, 0x59, 0x37, 0x2, 0x7a, 0xc7, 0x82, 0x8c, 0x90, 0x3a, 0x3c, 0xd8, 0x91, 0xe2, 0x96, 0xf8, 0x71, 0xea, 0xeb, 0x4e, 0x36, 0xf8, 0xd9, 0xf7, 0x1, 0x19, 0xde, 0x13, 0xda, 0x9a, 0x75, 0xa2, 0x3b, 0x3, 0x2c, 0x12, 0x47, 0x48, 0xc9, 0xb9, 0x30, 0x80, 0xd8, 0x17, 0xf8, 0x54, 0x20, 0xc5, 0x10, 0xdc, 0xf3, 0x25, 0x92, 0x24, 0x96, 0x22, 0xf, 0xe9, 0x99, 0x6f, 0x12, 0x10, 0x82, 0x2, 0x82, 0x7c, 0x7d, 0xf0, 0x14, 0x27, 0x18, 0xb9, 0xc2, 0x98, 0x93, 0x78, 0x45, 0x93, 0x52, 0xef, 0x1d, 0x30, 0xcd, 0xc4, 0x36, 0x8d, 0x7d, 0x9c, 0xdc, 0x6a, 0x17, 0xe7, 0x7a, 0xc8, 0xf4, 0xc3, 0x5f, 0x5d, 0xbc, 0x7d, 0x53, 0x6e, 0x68, 0xd2, 0x3d, 0xeb, 0xf9, 0x4e, 0x53, 0x1, 0xf8, 0xd2, 0xe0, 0xa0, 0x3b, 0xcd, 0x85, 0x33, 0x1, 0xaf, 0x9b, 0x60, 0x35, 0x2d, 0x92, 0xb4, 0xf8, 0x23, 0x19, 0x5c, 0xbf, 0xff, 0xe3, 0x12, 0x39, 0x76, 0x9d, 0x5, 0x30, 0x0, 0x80, 0xea, 0x79, 0x9f, 0xe6, 0x57, 0xae, 0x46, 0x55, 0x46, 0xd3, 0x25, 0x35, 0x57, 0x39, 0x27, 0x63, 0xa0, 0x41, 0xd2, 0xb9, 0xc5, 0x76, 0x14, 0xa7, 0xd, 0x6a, 0xf7, 0x7d, 0xf0, 0x46, 0xd9, 0xf7, 0xde, 0x24, 0xff, 0x13, 0x1c, 0x2a, 0xca, 0x90, 0xa5, 0xe5, 0x34, 0xbd, 0x5e, 0xa5, 0xf, 0xe6, 0x59, 0xd4, 0x99, 0x20, 0x62, 0xc0, 0x6f, 0xf5, 0xd1, 0xc7, 0x62, 0x7d, 0xea, 0x98, 0xcf, 0x2f, 0xe9, 0xaa, 0x8f, 0xb5, 0xed, 0x9d, 0xb, 0x98, 0x2f, 0x2f, 0x2e, 0xf0, 0x7a, 0x23, 0x11, 0x9c, 0x8a, 0xd2, 0xcb, 0x84, 0xa1, 0x7, 0xc3, 0x9c, 0x6f, 0x7e, 0x11, 0x4c, 0x38, 0xa2, 0x5e, 0xe5, 0x5c, 0x48, 0x2a, 0x68, 0xd9, 0x11, 0x52, 0x95, 0xd2, 0x80, 0x3b, 0x9c, 0x52, 0x22, 0x47, 0x3e, 0x5c, 0xd3, 0xcb, 0x8e, 0x53, 0xa2, 0x7b, 0xd8, 0x41, 0xf2, 0x9, 0x3e, 0x64, 0x42, 0x4f, 0x72, 0x85, 0x53, 0x12, 0xc5, 0xfb, 0x4f, 0x79, 0x53, 0xbf, 0xd0, 0xb2, 0x8e, 0x3, 0xe0, 0x24, 0xc3, 0x9e, 0x6a, 0xf9, 0x48, 0xef, 0x3a, 0x36, 0x6f, 0xee, 0x9e, 0x64, 0xaf, 0x1c, 0xb3, 0x0, 0x4d, 0x97, 0xf9, 0x8b, 0xb1, 0xad, 0x1b, 0xef, 0xb, 0xcf, 0x69, 0xa8, 0x3c, 0x98, 0x96, 0x1d, 0xbd, 0xeb, 0x6c, 0x5f, 0xe5, 0x66, 0xf5, 0xb0, 0x8b, 0x15, 0x9a, 0x3b, 0x58, 0xc6, 0xee, 0x62, 0x51, 0xe3, 0xf8, 0x33, 0x1, 0x92, 0xb2, 0x8f, 0x30, 0x22, 0x8a, 0x3e, 0x18, 0x9b, 0x8f, 0x4c, 0x3, 0x4e, 0x91, 0xdc, 0x4d, 0x11, 0x4b, 0x2f, 0xcb, 0x84, 0xa5, 0x7f, 0x69, 0x3f, 0x30, 0x47, 0x29, 0x1b, 0x69, 0xc0, 0x96, 0x8f, 0x12, 0xf9, 0x1a, 0x19, 0x91, 0x0, 0xcd, 0x83, 0xe8, 0xd0, 0xb1, 0x12, 0x8, 0xc3, 0x59, 0x89, 0x9f, 0xc, 0xfe, 0xb2, 0x55, 0x8f, 0x81, 0x77, 0x61, 0x10, 0x43, 0x4b, 0x61, 0xa0, 0x19, 0xdd, 0x52, 0xc7, 0x2e, 0xa9, 0x39, 0xc2, 0x5, 0x69, 0x68, 0x88, 0xaa, 0x60, 0x6a, 0x8a, 0xa6, 0x22, 0xb0, 0x4b, 0xd4, 0x14, 0xe2, 0x16, 0x2d, 0xab, 0x4, 0xfd, 0x39, 0xd1, 0xf2, 0x93, 0xae, 0x2d, 0xba, 0x8, 0x4a, 0x79, 0x9b, 0x67, 0x50, 0x10, 0x88, 0xbb, 0x75, 0xd8, 0x5a, 0x7a, 0xf5, 0x30, 0xf3, 0x3f, 0xc6, 0x4, 0xb9, 0xde, 0xfd, 0xa, 0x0, 0x8f, 0x37, 0x10, 0xcf, 0x87, 0x59, 0x72, 0xbb, 0x9d, 0x4d, 0xed, 0xb3, 0xf, 0xfa, 0x79, 0x26, 0x68, 0xf2, 0x9e, 0xd2, 0x2b, 0xb, 0xf6, 0x34, 0x11, 0x5c, 0x8e, 0x4d, 0x7b, 0x81, 0x3e, 0xe8, 0xcc, 0x2f, 0xcc, 0xae, 0xe5, 0x17, 0x24, 0x55, 0x79, 0x45, 0xe8, 0x2e, 0x66, 0x9b, 0x48, 0x67, 0xee, 0x83, 0xc6, 0xef, 0x7a, 0x1e, 0x94, 0x7, 0x73, 0x3c, 0x79, 0xf7, 0x59, 0xb, 0xbc, 0xc6, 0x2b, 0x66, 0xfd, 0xe, 0x9d, 0xcf, 0xc0, 0x9f, 0xc8, 0x75, 0xfd, 0xa9, 0x36, 0x2b, 0xa3, 0x5, 0x7b, 0xc4, 0x48, 0x33, 0x9b, 0xe9, 0x32, 0xa0, 0x24, 0x7f, 0xe4, 0xdf, 0xe4, 0x41, 0xea, 0xa2, 0x98, 0xbb, 0x41, 0x84, 0x17, 0x25, 0x5d, 0x4d, 0xd6, 0xe4, 0x3, 0xb9, 0xb1, 0x2, 0x1f, 0x7d, 0x54, 0x3a, 0x76, 0x1e, 0xb2, 0x0, 0x62, 0xe5, 0x7c, 0x5c, 0xf5, 0x82, 0x86, 0x85, 0xaf, 0x70, 0x66, 0x91, 0x62, 0xde, 0xb8, 0x5d, 0x95, 0x7f, 0x72, 0xd1, 0xe2, 0xb2, 0x1d, 0x9b, 0x4b, 0x16, 0x74, 0x95, 0xbb, 0x97, 0x8f, 0x34, 0xf0, 0x87, 0x55, 0xd1, 0x65, 0xe7, 0x2f, 0xb3, 0x30, 0x5f, 0x77, 0x74, 0x6b, 0xa9, 0x99, 0x43, 0x13, 0x56, 0x5b, 0xd2, 0x6d, 0x49, 0x80, 0x16, 0x1f, 0xa5, 0x84, 0xba, 0x71, 0xea, 0x7f, 0x1b, 0x62, 0x70, 0x33, 0xa, 0xd8, 0xd0, 0x21, 0xd, 0xbd, 0x7c, 0x27, 0xfa, 0xa4, 0x79, 0x4f, 0xae, 0x96, 0x84, 0x8d, 0x28, 0x13, 0x41, 0xa8, 0x1b, 0xf7, 0x32, 0xc7, 0xe1, 0xb, 0xfe, 0x68, 0xec, 0x47, 0x72, 0xc7, 0xa0, 0x64, 0xd0, 0x53, 0xf5, 0xf, 0x19, 0x4b, 0x44, 0xd4, 0xcb, 0xfc, 0x60, 0x94, 0x5d, 0x44, 0x13, 0x89, 0xb, 0x80, 0x89, 0xf9, 0xb1, 0x1, 0xa5, 0xec, 0x28, 0x70, 0xa1, 0x52, 0x35, 0xe, 0x44, 0x41, 0xfa, 0xca, 0xb2, 0xb3, 0x2e, 0x4d, 0xf5, 0xf5, 0xf5, 0xca, 0xe5, 0x1e, 0x81, 0xce, 0x45, 0xbb, 0xc1, 0x2e, 0x46, 0x2a, 0xee, 0xf, 0xa6, 0xf2, 0x1f, 0x75, 0xd1, 0x3d, 0xb0, 0x5f, 0x81, 0x4a, 0xbb, 0x27, 0xae, 0xe9, 0xda, 0x69, 0xa1, 0x7a, 0xe1, 0x89, 0x30, 0xd, 0x54, 0xd4, 0x4d, 0xb, 0xc7, 0xcf, 0x48, 0xf5, 0xed, 0xe8, 0x72, 0xd9, 0xa5, 0x81, 0x58, 0x13, 0x95, 0xf2, 0x1b, 0x66, 0x51, 0x9, 0x57, 0xc, 0x8e, 0xb0, 0x20, 0x8a, 0x1f, 0x36, 0x75, 0xa5, 0x6b, 0x4, 0xe9, 0xe, 0xb2, 0xf6, 0x70, 0x21, 0xe1, 0x4, 0xea, 0xd3, 0x93, 0x3b, 0xde, 0x33, 0xa5, 0xb4, 0xcd, 0x70, 0xcc, 0xf9, 0x24, 0x5e, 0xa4, 0x72, 0x4d, 0x79, 0x1d, 0xc0, 0x60, 0x99, 0x13, 0xc2, 0x8f, 0xd1, 0x47, 0xef, 0xac, 0xe2, 0xa4, 0xb7, 0xcb, 0xf, 0x12, 0x91, 0x44, 0xa0, 0x83, 0x92, 0x55, 0x8f, 0x72, 0x68, 0xeb, 0x4b, 0xa, 0x43, 0x30, 0x76, 0x40, 0x5d, 0xb5, 0x86, 0x4a, 0x0, 0x28, 0x36, 0xe8, 0x1d, 0xda, 0xe1, 0x69, 0x5e, 0xf9, 0x15, 0xbe, 0x6a, 0xe5, 0x48, 0xef, 0x7e, 0x67, 0xa, 0x4c, 0x9a, 0x22, 0x40, 0xb0, 0xff, 0x9d, 0xca, 0xa6, 0x32, 0x89, 0xa7, 0xe9, 0xbf, 0xb2, 0xa1, 0xe6, 0x82, 0x86, 0x2f, 0xb3, 0x94, 0x9f, 0xb5, 0xb9, 0x79, 0xd9, 0xce, 0xdc, 0x92, 0xd4, 0x7b, 0x8d, 0xf2, 0x36, 0xf5, 0x66, 0xfd, 0x22, 0x53, 0x14, 0xb1, 0x23, 0x32, 0x72, 0xf7, 0xb1, 0xfa, 0x81, 0x91, 0x98, 0xe6, 0xc3, 0x4e, 0x42, 0x12, 0x6c, 0xee, 0x58, 0xa8, 0xa4, 0xd, 0x9e, 0xde, 0x28, 0x6e, 0xca, 0x43, 0x15, 0xee, 0x1d, 0xbc, 0xf1, 0x2c}
	md := []byte{0x70, 0x37, 0xf1, 0x79, 0xf8, 0x74, 0x7b, 0x7b, 0xf7, 0xbf, 0xce, 0x47, 0x67, 0xfb, 0xce, 0x28, 0x97, 0xac, 0xb2, 0x75, 0x49}
	pkSeed := []byte{0x5d, 0xce, 0xdb, 0x6d, 0xba, 0xc0, 0xa2, 0x49, 0x6a, 0x42, 0x0, 0xeb, 0x91, 0x9a, 0x3f, 0xd0}
	adrs := &address{0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x2e, 0x7c, 0xd8, 0x6e, 0x34, 0xa3, 0xd9, 0x0, 0x0, 0x0, 0x3, 0x0, 0x0, 0x1, 0xad, 0x0, 0x0, 0x0, 0xb, 0x0, 0x0, 0x0, 0x1b}
	want := []byte{0x54, 0xda, 0x9d, 0xd9, 0xcc, 0x2c, 0x3c, 0x28, 0x36, 0xec, 0xec, 0xab, 0x15, 0xd9, 0xb7, 0xb4}
	if got := par.forsPkFromSig(sig, md, pkSeed, adrs); !slices.Equal(got, want) {
		t.Errorf("par.forsPkFromSig() = %v, want %v", got, want)
	}
}
