<?php

declare(strict_types=1);

/*
 * Copyright (c) 2023-2024 François Kooman <fkooman@tuxed.net>
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

namespace fkooman\Radius\Tests;

use fkooman\Radius\RadiusPacket;
use PHPUnit\Framework\TestCase;
use RangeException;

/**
 * @covers \fkooman\Radius\RadiusPacket
 *
 * @uses \fkooman\Radius\Utils
 * @uses \fkooman\Radius\AttributeCollection
 * @uses \fkooman\Radius\Password
 */
class RadiusPacketTest extends TestCase
{
    public function testAccessRequest(): void
    {
        $radiusPacket = new RadiusPacket(RadiusPacket::ACCESS_REQUEST, 0, hex2bin('d5374dae15308373fe120fcbead802c9'));
        $this->assertSame('01000014d5374dae15308373fe120fcbead802c9', bin2hex($radiusPacket->toBytes()));
        $this->assertTrue($radiusPacket->isAccessRequest());
        $this->assertSame('Access-Request', $radiusPacket->packetType());
        $this->assertSame(0, $radiusPacket->packetId());
        $this->assertSame('RADIUS PACKET: TYPE=Access-Request ID=0 AUTHENTICATOR=0xd5374dae15308373fe120fcbead802c9', (string) $radiusPacket);
    }

    public function testFromData(): void
    {
        $radiusPacket = RadiusPacket::fromBytes(hex2bin('0200003d349eb4c917a67ce7277f7223199ea7ed121757656c636f6d65203e3e3e20626172203c3c3c202150129e6d4cfe8a01709f08e183a4e2d681b9'));
        $this->assertTrue($radiusPacket->isAccessAccept());
        $this->assertSame('Access-Accept', $radiusPacket->packetType());
    }

    public function testInvalidLength(): void
    {
        $this->expectException(RangeException::class);
        $this->expectExceptionMessage('string length MUST be exactly 22 octets');
        RadiusPacket::fromBytes(hex2bin('0100001600000000000000000000000000000000'));
    }
}
