<?php

declare(strict_types=1);

/*
 * Copyright (c) 2023-2024 François Kooman <fkooman@tuxed.net>
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

namespace fkooman\Radius\Tests;

use fkooman\Radius\RadiusPacket;
use fkooman\Radius\ResponseAuthenticator;
use PHPUnit\Framework\TestCase;

/**
 * @covers \fkooman\Radius\ResponseAuthenticator
 *
 * @uses \fkooman\Radius\RadiusPacket
 * @uses \fkooman\Radius\AttributeCollection
 * @uses \fkooman\Radius\Utils
 */
class ResponseAuthenticatorTest extends TestCase
{
    public function testCalculate(): void
    {
        //$ SERVER_URI=udp://[fd43::5]:1812 php example/client.php foo bar
        //(LOG) I Access-Request for User-Name "foo"
        //(LOG) D Access-Request for User-Password "bar"
        //(LOG) D Server: udp://[fd43::5]:1812
        //(LOG) D --> 010000488f8ec0e66c02f23a899e966e232c25c70105666f6f0212bac86248bf3fa63d190aaa8b7da6fbf9200b6d792d6e61732d69645012728d9688523fa19c2de3780d4aeb4c45
        //(LOG) D <-- 02000040970ba44e0889fec20692ced3a093b25d121a41757468656e7469636174696f6e207375636365656465645012ac959ffa995fd0e61f379ffe862a5caa
        //*** Access-Accept ***
        //Reply-Message:
        //	Authentication succeeded
        //Message-Authenticator:
        //	0xac959ffa995fd0e61f379ffe862a5caa

        // we recreate the response packet from scratch and then calculate
        // the "Response Authenticator"
        $responsePacket = new RadiusPacket(RadiusPacket::ACCESS_ACCEPT, 0);
        $responsePacket->attributeCollection()->set('Reply-Message', 'Authentication succeeded');
        $responsePacket->attributeCollection()->set('Message-Authenticator', hex2bin('ac959ffa995fd0e61f379ffe862a5caa'));
        $responsePacket->setPacketAuthenticator(ResponseAuthenticator::calculate($responsePacket, 's3cr3t', hex2bin('8f8ec0e66c02f23a899e966e232c25c7')));
        $this->assertSame(
            '02000040970ba44e0889fec20692ced3a093b25d121a41757468656e7469636174696f6e207375636365656465645012ac959ffa995fd0e61f379ffe862a5caa',
            bin2hex($responsePacket->toBytes())
        );
    }

    public function testVerify(): void
    {
        //(LOG) I Access-Request for User-Name "foo"
        //(LOG) D Access-Request for User-Password "bar"
        //(LOG) D Server: udp://10.43.43.5:1812
        //(LOG) D --> 01000048445ddd60b2a31e4e6b1b6d0d17b7b0c30105666f6f021257fb55787ea1780b3f1c46b3dd6cb1f6200b6d792d6e61732d69645012668183f84ab1f5b6e9f9f543e6fce0d3
        //(LOG) D <-- 02000040aace592a1f353c07774022ad02c6d83e121a41757468656e7469636174696f6e2073756363656564656450126c6bfa01eadd064c3076bbb7ce8850a1
        $responsePacket = RadiusPacket::fromBytes(hex2bin('02000040aace592a1f353c07774022ad02c6d83e121a41757468656e7469636174696f6e2073756363656564656450126c6bfa01eadd064c3076bbb7ce8850a1'));
        $this->assertTrue(ResponseAuthenticator::verify($responsePacket, 's3cr3t', hex2bin('445ddd60b2a31e4e6b1b6d0d17b7b0c3')));
    }
}
