<?php

declare(strict_types=1);

/*
 * Copyright (c) 2017-2022 François Kooman <fkooman@tuxed.net>
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

namespace fkooman\SeCookie\Tests;

use fkooman\SeCookie\Cookie;
use fkooman\SeCookie\CookieOptions;
use PHPUnit\Framework\TestCase;

/**
 * @internal
 *
 * @coversNothing
 */
final class CookieTest extends TestCase
{
    public function testNoCookie(): void
    {
        $cookieOptions = new CookieOptions();
        $testCookie = new TestCookie($cookieOptions, []);

        static::assertSame([], $testCookie->getHeadersSent());
    }

    public function testSimple(): void
    {
        $cookieOptions = new CookieOptions();
        $testCookie = new TestCookie($cookieOptions, []);
        $testCookie->set('foo', 'bar');
        static::assertSame(
            [
                'Set-Cookie: foo=bar; HttpOnly; SameSite=Lax; Secure',
            ],
            $testCookie->getHeadersSent()
        );
    }

    public function testDeleteCookie(): void
    {
        $cookieOptions = new CookieOptions();
        $testCookie = new TestCookie($cookieOptions, []);
        $testCookie->delete('foo');
        static::assertSame(
            [
                'Set-Cookie: foo=; HttpOnly; Max-Age=0; SameSite=Lax; Secure',
            ],
            $testCookie->getHeadersSent()
        );
    }

    public function testDeleteCookieWithMaxAge(): void
    {
        $cookieOptions = CookieOptions::init()->withMaxAge(12345);
        $testCookie = new TestCookie($cookieOptions, []);
        $testCookie->delete('foo');
        static::assertSame(
            [
                'Set-Cookie: foo=; HttpOnly; Max-Age=0; SameSite=Lax; Secure',
            ],
            $testCookie->getHeadersSent()
        );
    }

    public function testAttributeValues(): void
    {
        $cookieOptions = CookieOptions::init()->withPath('/foo/')->withMaxAge(12345);
        $testCookie = new TestCookie($cookieOptions, []);
        $testCookie->set('foo', 'bar');
        static::assertSame(
            [
                'Set-Cookie: foo=bar; HttpOnly; Max-Age=12345; Path=/foo/; SameSite=Lax; Secure',
            ],
            $testCookie->getHeadersSent()
        );
    }

    public function testGetCookie(): void
    {
        $cookieOptions = new CookieOptions();
        $testCookie = new TestCookie($cookieOptions, ['SID' => 'abcdef']);
        static::assertSame('abcdef', $testCookie->get('SID'));
    }

    public function testMissingCookie(): void
    {
        $cookieOptions = new CookieOptions();
        $testCookie = new TestCookie($cookieOptions, []);
        static::assertNull($testCookie->get('SID'));
    }

    public function testSetSameSiteNoneCookie(): void
    {
        $cookieOptions = CookieOptions::init()->withSameSiteNone();
        $testCookie = new TestCookie($cookieOptions, []);
        $testCookie->set('foo', 'bar');
        static::assertSame(
            [
                'Set-Cookie: foo=bar; HttpOnly; SameSite=None; Secure',
                'Set-Cookie: foo' . Cookie::NO_SAME_SITE_POSTFIX . '=bar; HttpOnly; Secure',
            ],
            $testCookie->getHeadersSent()
        );
    }

    public function testGetSameSiteNoneCookie(): void
    {
        $cookieOptions = CookieOptions::init()->withSameSiteNone();
        // "foo" cookie should take precedence
        $testCookie = new TestCookie($cookieOptions, ['foo' . Cookie::NO_SAME_SITE_POSTFIX => 'bar', 'foo' => 'baz']);
        static::assertSame('baz', $testCookie->get('foo'));
        $testCookie = new TestCookie($cookieOptions, ['foo' . Cookie::NO_SAME_SITE_POSTFIX => 'bar']);
        static::assertSame('bar', $testCookie->get('foo'));
    }
}
