<?php

declare(strict_types=1);

/*
 * Copyright (c) 2017-2022 François Kooman <fkooman@tuxed.net>
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

namespace fkooman\SeCookie\Tests;

use DateTimeImmutable;
use fkooman\SeCookie\ActiveSession;
use fkooman\SeCookie\PdoSessionStorage;
use PDO;
use PHPUnit\Framework\TestCase;

/**
 * @internal
 *
 * @coversNothing
 */
final class PdoSessionStorageTest extends TestCase
{
    private PdoSessionStorage $sessionStorage;

    protected function setUp(): void
    {
        $this->sessionStorage = new PdoSessionStorage(new PDO('sqlite::memory:'));
        $this->sessionStorage->init();
    }

    public function testStoreRetrieve(): void
    {
        $this->sessionStorage->store(
            new ActiveSession(
                'session_id',
                new DateTimeImmutable('2021-05-05T09:00:00+00:00'),
                [
                    'foo' => 'bar',
                ]
            )
        );
        $activeSession = $this->sessionStorage->retrieve('session_id');
        static::assertSame('session_id', $activeSession->sessionId());
        static::assertSame(
            [
                'foo' => 'bar',
            ],
            $activeSession->sessionData()
        );
        static::assertSame('2021-05-05T09:00:00+00:00', $activeSession->expiresAt()->format(DateTimeImmutable::ATOM));
    }

    public function setRetrieveMissing(): void
    {
        static::assertNull($this->sessionStorage->retrieve('foo'));
    }

    public function testDestroy(): void
    {
        $this->sessionStorage->store(
            new ActiveSession(
                'session_id',
                new DateTimeImmutable('2021-05-05T09:00:00+00:00'),
                [
                    'foo' => 'bar',
                ]
            )
        );
        static::assertNotNull($this->sessionStorage->retrieve('session_id'));
        $this->sessionStorage->destroy('session_id');
        static::assertNull($this->sessionStorage->retrieve('session_id'));
    }
}
