package poly1305

import (
	"bytes"
	"crypto/rand"
	"math/big"
	"testing"

	"golang.org/x/crypto/poly1305"
)

var testVectors = []struct {
	key [32]byte
	msg []byte
	mac [16]byte
}{
	{
		// from nacl-20110221/tests/onetimeauth2.c
		[32]byte{
			0xee, 0xa6, 0xa7, 0x25, 0x1c, 0x1e, 0x72, 0x91, 0x6d, 0x11, 0xc2,
			0xcb, 0x21, 0x4d, 0x3c, 0x25, 0x25, 0x39, 0x12, 0x1d, 0x8e, 0x23,
			0x4e, 0x65, 0x2d, 0x65, 0x1f, 0xa4, 0xc8, 0xcf, 0xf8, 0x80,
		},
		[]byte{
			0x8e, 0x99, 0x3b, 0x9f, 0x48, 0x68, 0x12, 0x73, 0xc2, 0x96, 0x50, 0xba,
			0x32, 0xfc, 0x76, 0xce, 0x48, 0x33, 0x2e, 0xa7, 0x16, 0x4d, 0x96, 0xa4,
			0x47, 0x6f, 0xb8, 0xc5, 0x31, 0xa1, 0x18, 0x6a, 0xc0, 0xdf, 0xc1, 0x7c,
			0x98, 0xdc, 0xe8, 0x7b, 0x4d, 0xa7, 0xf0, 0x11, 0xec, 0x48, 0xc9, 0x72,
			0x71, 0xd2, 0xc2, 0x0f, 0x9b, 0x92, 0x8f, 0xe2, 0x27, 0x0d, 0x6f, 0xb8,
			0x63, 0xd5, 0x17, 0x38, 0xb4, 0x8e, 0xee, 0xe3, 0x14, 0xa7, 0xcc, 0x8a,
			0xb9, 0x32, 0x16, 0x45, 0x48, 0xe5, 0x26, 0xae, 0x90, 0x22, 0x43, 0x68,
			0x51, 0x7a, 0xcf, 0xea, 0xbd, 0x6b, 0xb3, 0x73, 0x2b, 0xc0, 0xe9, 0xda,
			0x99, 0x83, 0x2b, 0x61, 0xca, 0x01, 0xb6, 0xde, 0x56, 0x24, 0x4a, 0x9e,
			0x88, 0xd5, 0xf9, 0xb3, 0x79, 0x73, 0xf6, 0x22, 0xa4, 0x3d, 0x14, 0xa6,
			0x59, 0x9b, 0x1f, 0x65, 0x4c, 0xb4, 0x5a, 0x74, 0xe3, 0x55, 0xa5,
		},
		[16]byte{
			0xf3, 0xff, 0xc7, 0x70, 0x3f, 0x94, 0x00, 0xe5,
			0x2a, 0x7d, 0xfb, 0x4b, 0x3d, 0x33, 0x05, 0xd9,
		},
	},
	// custom test vectors:
	{
		[32]byte{},
		nil,
		[16]byte{},
	},
	{
		[32]byte{},
		[]byte{},
		[16]byte{},
	},
	{
		[32]byte{1, 2, 3},
		[]byte{4, 5, 6},
		[16]byte{0x4, 0xd, 0x1c, 0x1c, 0x14, 0x3},
	},
	{
		[32]byte{},
		[]byte{
			0xee, 0xa6, 0xa7, 0x25, 0x1c, 0x1e, 0x72, 0x91, 0x6d, 0x11, 0xc2,
			0xcb, 0x21, 0x4d, 0x3c, 0x25, 0x25, 0x39, 0x12, 0x1d, 0x8e, 0x23,
			0x4e, 0x65, 0x2d, 0x65, 0x1f, 0xa4, 0xc8, 0xcf, 0xf8, 0x80,
		},
		[16]byte{},
	},
	{
		[32]byte{
			0xee, 0xa6, 0xa7, 0x25, 0x1c, 0x1e, 0x72, 0x91, 0x6d, 0x11, 0xc2,
			0xcb, 0x21, 0x4d, 0x3c, 0x25, 0x25, 0x39, 0x12, 0x1d, 0x8e, 0x23,
			0x4e, 0x65, 0x2d, 0x65, 0x1f, 0xa4, 0xc8, 0xcf, 0xf8, 0x80,
		},
		nil,
		[16]byte{
			0x25, 0x39, 0x12, 0x1d, 0x8e, 0x23, 0x4e, 0x65,
			0x2d, 0x65, 0x1f, 0xa4, 0xc8, 0xcf, 0xf8, 0x80,
		},
	},
	{
		[32]byte{
			0x74, 0x68, 0x69, 0x73, 0x20, 0x73, 0x65, 0x63,
			0x72, 0x65, 0x74, 0x20, 0x6b, 0x65, 0x79, 0x20,
			0x69, 0x73, 0x20, 0x33, 0x32, 0x20, 0x62, 0x79,
			0x74, 0x65, 0x73, 0x20, 0x6c, 0x6f, 0x6e, 0x67,
		},
		[]byte("a 17 byte message"),
		[16]byte{
			0xcb, 0x61, 0x33, 0x5a, 0xed, 0x29, 0xf4, 0xe2,
			0x4d, 0x5d, 0xc6, 0xc8, 0x8c, 0x6d, 0x03, 0x1f,
		},
	},
}

func TestVectors(t *testing.T) {
	for i, test := range testVectors {
		var s0, s1, s2 [16]byte
		Sum(&s0, test.msg, &test.key)
		if !bytes.Equal(s0[:], test.mac[:]) {
			t.Errorf("test %d.0:\n  expected=%x\n  actually=%x", i, test.mac, s0)
		}
		SumSporadic(&s1, test.msg, &test.key)
		if !bytes.Equal(s1[:], test.mac[:]) {
			t.Errorf("test %d.1:\n  expected=%x\n  actually=%x", i, test.mac, s1)
		}
		poly1305.Sum(&s2, test.msg, &test.key)
		if !bytes.Equal(s2[:], test.mac[:]) {
			t.Errorf("test %d.2:\n  expected=%x\n  actually=%x", i, test.mac, s2)
		}
	}
}

func TestRandom(t *testing.T) {
	var key [32]byte
	for i := 0; i < 10000; i++ {
		msg := make([]byte, i)
		rand.Read(msg)
		rand.Read(key[:])

		var s0, s1, s2 [16]byte
		poly1305.Sum(&s0, msg, &key)
		Sum(&s1, msg, &key)
		SumSporadic(&s2, msg, &key)

		if !bytes.Equal(s0[:], s1[:]) {
			t.Fatalf("key=%x msg=%x\n  expected=%x\n  actually=%x", key, msg, s0, s1)
		}
		if !bytes.Equal(s1[:], s2[:]) {
			t.Fatalf("key=%x msg=%x\n  expected=%x\n  actually=%x", key, msg, s1, s2)
		}
	}
}

func Sum(out *[16]byte, in []byte, key *[32]byte) {
	h := New(key)
	h.Write(in)
	sum := h.Sum(nil)
	copy(out[:], sum)
}

func SumSporadic(out *[16]byte, in []byte, key *[32]byte) {
	h := New(key)
	for len(in) > 0 {
		i := randInt(len(in))
		h.Write(in[:i])
		in = in[i:]
	}
	sum := h.Sum(nil)
	copy(out[:], sum)
}

func randInt(max int) int {
	m := big.NewInt(int64(max) + 1)
	n, err := rand.Int(rand.Reader, m)
	if err != nil {
		panic(err)
	}
	return int(n.Int64())
}
