package appsec

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"

	"github.com/akamai/AkamaiOPEN-edgegrid-golang/v11/pkg/session"
	validation "github.com/go-ozzo/ozzo-validation/v4"
)

type (
	// The MalwarePolicyAction interface supports retrieving and modifying the action associated with
	// a specified malware policy, or with all malware policies in a security policy.
	MalwarePolicyAction interface {
		// GetMalwarePolicyActions retrieves the actions for a specific malware protection policy.
		//
		// See: https://techdocs.akamai.com/application-security/reference/get-malware-policies-actions
		GetMalwarePolicyActions(ctx context.Context, params GetMalwarePolicyActionsRequest) (*GetMalwarePolicyActionsResponse, error)

		// UpdateMalwarePolicyAction modifies the actions for a specific malware protection policy.
		//
		// See: https://techdocs.akamai.com/application-security/reference/put-malware-policy-action
		UpdateMalwarePolicyAction(ctx context.Context, params UpdateMalwarePolicyActionRequest) (*UpdateMalwarePolicyActionResponse, error)

		// UpdateMalwarePolicyActions is for Akamai internal use only.
		UpdateMalwarePolicyActions(ctx context.Context, params UpdateMalwarePolicyActionsRequest) (*UpdateMalwarePolicyActionsResponse, error)
	}

	// GetMalwarePolicyActionsRequest is used to retrieve a configuration's malware policies and their associated actions.
	GetMalwarePolicyActionsRequest struct {
		ConfigID        int    `json:"configID"`
		Version         int    `json:"version"`
		PolicyID        string `json:"policyID"`
		MalwarePolicyID int    `json:"id"`
	}

	// MalwarePolicyActionBody defines the actions for a specific malware policy.
	MalwarePolicyActionBody struct {
		MalwarePolicyID int    `json:"id"`
		Action          string `json:"action"`
		UnscannedAction string `json:"unscannedAction"`
	}

	// GetMalwarePolicyActionsResponse is returned from a call to GetMalwarePolicyActions.
	GetMalwarePolicyActionsResponse struct {
		MalwarePolicyActions []MalwarePolicyActionBody `json:"malwarePolicyActions"`
	}

	// UpdateMalwarePolicyActionRequest is used to update the actions for a malware policy.
	UpdateMalwarePolicyActionRequest struct {
		ConfigID        int    `json:"configID"`
		Version         int    `json:"version"`
		PolicyID        string `json:"policyID"`
		MalwarePolicyID int    `json:"id"`
		Action          string `json:"action"`
		UnscannedAction string `json:"unscannedAction"`
	}

	// UpdateMalwarePolicyActionResponse is returned from a call to UpdateMalwarePolicy.
	UpdateMalwarePolicyActionResponse struct {
		Action          string `json:"action"`
		UnscannedAction string `json:"unscannedAction"`
	}

	// UpdateMalwarePolicyActionsRequest is used to update the actions for multiple malware policies.
	UpdateMalwarePolicyActionsRequest struct {
		ConfigID             int
		Version              int
		PolicyID             string
		MalwarePolicyActions json.RawMessage `json:"-"`
	}

	// UpdateMalwarePolicyActionsResponse is returned from a call to UpdateMalwarePolicyActions.
	UpdateMalwarePolicyActionsResponse GetMalwarePolicyActionsResponse
)

// Validate validates a GetMalwarePolicyActionsRequest.
func (v GetMalwarePolicyActionsRequest) Validate() error {
	return validation.Errors{
		"ConfigID": validation.Validate(v.ConfigID, validation.Required),
		"Version":  validation.Validate(v.Version, validation.Required),
		"PolicyID": validation.Validate(v.PolicyID, validation.Required),
	}.Filter()
}

// Validate validates an UpdateMalwarePolicyActionRequest.
func (v UpdateMalwarePolicyActionRequest) Validate() error {
	return validation.Errors{
		"ConfigID":        validation.Validate(v.ConfigID, validation.Required),
		"Version":         validation.Validate(v.Version, validation.Required),
		"PolicyID":        validation.Validate(v.PolicyID, validation.Required),
		"MalwarePolicyID": validation.Validate(v.MalwarePolicyID, validation.Required),
		"Action":          validation.Validate(v.Action, validation.Required),
		"UnscannedAction": validation.Validate(v.UnscannedAction, validation.Required),
	}.Filter()
}

// Validate validates an UpdateMalwarePolicyActionsRequest.
func (v UpdateMalwarePolicyActionsRequest) Validate() error {
	return validation.Errors{
		"ConfigID":             validation.Validate(v.ConfigID, validation.Required),
		"Version":              validation.Validate(v.Version, validation.Required),
		"PolicyID":             validation.Validate(v.PolicyID, validation.Required),
		"MalwarePolicyActions": validation.Validate(v.MalwarePolicyActions, validation.Required),
	}.Filter()
}

func (p *appsec) GetMalwarePolicyActions(ctx context.Context, params GetMalwarePolicyActionsRequest) (*GetMalwarePolicyActionsResponse, error) {
	logger := p.Log(ctx)
	logger.Debug("GetMalwarePolicyActions")

	if err := params.Validate(); err != nil {
		return nil, fmt.Errorf("%w: %s", ErrStructValidation, err.Error())
	}

	uri := fmt.Sprintf(
		"/appsec/v1/configs/%d/versions/%d/security-policies/%s/malware-policies",
		params.ConfigID,
		params.Version,
		params.PolicyID)

	req, err := http.NewRequestWithContext(ctx, http.MethodGet, uri, nil)
	if err != nil {
		return nil, fmt.Errorf("failed to create GetMalwarePolicyActions request: %w", err)
	}

	var result GetMalwarePolicyActionsResponse
	resp, err := p.Exec(req, &result)
	if err != nil {
		return nil, fmt.Errorf("get malware policy actions request failed: %w", err)
	}
	defer session.CloseResponseBody(resp)

	if resp.StatusCode != http.StatusOK {
		return nil, p.Error(resp)
	}

	if params.MalwarePolicyID != 0 {
		var filteredResult GetMalwarePolicyActionsResponse
		for _, val := range result.MalwarePolicyActions {
			if val.MalwarePolicyID == params.MalwarePolicyID {
				filteredResult.MalwarePolicyActions = append(filteredResult.MalwarePolicyActions, val)
			}
		}
		return &filteredResult, nil
	}

	return &result, nil
}

func (p *appsec) UpdateMalwarePolicyAction(ctx context.Context, params UpdateMalwarePolicyActionRequest) (*UpdateMalwarePolicyActionResponse, error) {
	logger := p.Log(ctx)
	logger.Debug("UpdateMalwarePolicyAction")

	if err := params.Validate(); err != nil {
		return nil, fmt.Errorf("%w: %s", ErrStructValidation, err.Error())
	}

	uri := fmt.Sprintf(
		"/appsec/v1/configs/%d/versions/%d/security-policies/%s/malware-policies/%d",
		params.ConfigID,
		params.Version,
		params.PolicyID,
		params.MalwarePolicyID,
	)

	req, err := http.NewRequestWithContext(ctx, http.MethodPut, uri, nil)
	if err != nil {
		return nil, fmt.Errorf("failed to create UpdateMalwarePolicyAction request: %w", err)
	}

	var result UpdateMalwarePolicyActionResponse
	resp, err := p.Exec(req, &result, params)
	if err != nil {
		return nil, fmt.Errorf("update malware policy action request failed: %w", err)
	}
	defer session.CloseResponseBody(resp)

	if resp.StatusCode != http.StatusOK {
		return nil, p.Error(resp)
	}

	return &result, nil
}

func (p *appsec) UpdateMalwarePolicyActions(ctx context.Context, params UpdateMalwarePolicyActionsRequest) (*UpdateMalwarePolicyActionsResponse, error) {
	logger := p.Log(ctx)
	logger.Debug("UpdateMalwarePolicyActions")

	if err := params.Validate(); err != nil {
		return nil, fmt.Errorf("%w: %s", ErrStructValidation, err.Error())
	}

	uri := fmt.Sprintf(
		"/appsec/v1/configs/%d/versions/%d/security-policies/%s/malware-policies",
		params.ConfigID,
		params.Version,
		params.PolicyID,
	)

	req, err := http.NewRequestWithContext(ctx, http.MethodPut, uri, nil)
	if err != nil {
		return nil, fmt.Errorf("failed to create UpdateMalwarePolicyActions request: %w", err)
	}

	var result UpdateMalwarePolicyActionsResponse
	resp, err := p.Exec(req, &result, params.MalwarePolicyActions)
	if err != nil {
		return nil, fmt.Errorf("update malware policy actions request failed: %w", err)
	}
	defer session.CloseResponseBody(resp)

	if resp.StatusCode != http.StatusOK {
		return nil, p.Error(resp)
	}

	return &result, nil
}
