/*
 * Copyright (c) 2022 Andrea Biscuola <a@abiscuola.com>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

package main

import (
	"bufio"
	"fmt"
	"io"
	"mime"
	"net/url"
	"regexp"
	"strings"
	"time"

	"github.com/makeworld-the-better-one/go-gemini"
)

// Fetch RSS/ATOM feeds over gemini or page subscriptions as described
// here:
//
//	gemini://gemini.circumlunar.space/docs/companion/subscription.gmi
func FetchGemfeed(urlstr string) (feeds []Feed, e error) {
	url, e := url.Parse(urlstr)
	if e != nil {
		return
	}

	resp, e := gemini.Fetch(urlstr)
	if e != nil {
		return
	}
	defer resp.Body.Close()

	if resp.Status != gemini.StatusSuccess {
		return nil, fmt.Errorf("%s", resp.Meta)
	}

	mediatype, _, e := mime.ParseMediaType(resp.Meta)
	if e != nil {
		return nil, fmt.Errorf("can't parse MIME type %s: %v", resp.Meta, e)
	}

	switch mediatype {
	case "application/rss+xml", "application/atom+xml", "text/xml":
		return RssFromReader(resp.Body)
	case "text/gemini":
		return parseGemfeed(resp.Body, url)
	default:
		return nil, fmt.Errorf("want text/gemini; got %s", mediatype)
	}
}

func parseGemfeed(r io.Reader, url *url.URL) (feeds []Feed, e error) {
	var author string

	author = TildeUser(url.Path)
	if author == "" {
		author = url.Hostname()
	}

	scanner := bufio.NewScanner(r)
	for scanner.Scan() {
		if !strings.HasPrefix(scanner.Text(), "=> ") {
			continue
		}
		regxp := regexp.MustCompile(`=>\s+.*\s+[0-9]+-[0-9]+-[0-9]+\s+.*`)
		if !regxp.MatchString(scanner.Text()) {
			continue
		}

		regxp = regexp.MustCompile(`\s+`)
		entry := regxp.ReplaceAllString(scanner.Text(), " ")

		fields := strings.Split(entry, " ")

		parsed, e := url.Parse(fields[1])
		if e != nil {
			// found an un-parsable link, skipping
			continue
		}
		resolved := url.ResolveReference(parsed)

		fdate, e := time.Parse("2006-01-02", fields[2])
		if e != nil {
			continue
		}

		title := strings.Join(fields[3:], " ")
		feeds = append(feeds, Feed{
			Title:           strings.TrimLeft(title, " -:"),
			PublishedParsed: fdate,
			Published:       fields[2],
			Link:            resolved.String(),
			Author:          author,
		})
	}

	if e := scanner.Err(); e != nil {
		return nil, e
	}

	return
}
