// Copyright 2017 Google Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package gogen

import (
	"bytes"
	"fmt"
	"io"
	"sort"
	"strings"
	"text/template"

	"github.com/openconfig/gnmi/errlist"
	gpb "github.com/openconfig/gnmi/proto/gnmi"
	"github.com/openconfig/goyang/pkg/yang"
	"github.com/openconfig/ygot/genutil"
	"github.com/openconfig/ygot/internal/igenutil"
	"github.com/openconfig/ygot/util"
	"github.com/openconfig/ygot/ygen"
	"github.com/openconfig/ygot/ygot"
)

const (
	// defaultSchemaVarName is the default variable name that should
	// be used for the output JSON schema when stored. It can be overridden
	// by specifying a name within the GoOptions field of the YANGCodeGenerator
	// instance used for code generation.
	defaultSchemaVarName string = "ySchema"
	// defaultPackageName specifies the default name that should be
	// used for the generated Go package.
	defaultPackageName string = "ocstructs"
	// DefaultAnnotationPrefix is the default string that is used to prefix the name
	// of metadata fields in the output Go structs.
	DefaultAnnotationPrefix string = "Λ"
	// annotationFieldType defines the type that should be used for the
	// annotation/metadata fields within each struct when they are generated.
	annotationFieldType string = "[]ygot.Annotation"
)

// The methods in this file take the structs that have been generated by
// processing the goyang output, and generate the Go code snippets that
// will be output to the caller of the library.
//
// These functions break down into two different sets:
//	1) Those that generate output for containers or lists, which are both
//	   Go structs.
//	2) Those that generate output for enumerated entities.
//
// For structs, some additional output is also generated. For example, if a struct
// storing the characteristics of a YANG container A is input, then the resulting
// GoStructCodeSnippet struct will contain the definition of the Go struct
// used to create an instance of A in the structDef string. The listKeys string
// contains any structs that are used as the key to a multi-key list. The methods
// string contains any functions that are generated with A as the receiver.
//
// As a worked example, if we have the following YANG input:
//
//	container A {
//		list B {
//			key "one two";
//			leaf one { type string; }
//			leaf two { type int8; }
//		}
//	}
//
// The resulting generated Go code consists of:
//
//	type A struct {
//		B map[A_B_Key]*A_B
//	}
//
// Where A_B is the struct that will be created for the list A/B, and A_B_Key is
// a struct created specifically to key the map for list B, based on it having
// multiple keys.
//
// The functions generated provide a method to create a new instance of B, or
// append an A_B struct to the map B whilst populating the relevant key fields
// of the list, or autocreating the key struct from the fields of the input
// structure. i.e., they are of the form:
//
// func (*A) NewB(string, int8) (*A_B, error) {
//	// Create a new instance of B, and its key, and append it to the map
//	// field of A.
// }
//
// func (*A) AppendB(*A_B) error {
//	// Append the instance of B that was handed to the function to the map
//	// field of A.
// }
//
// Each enumerated node within the YANG schema (an identity, enumeration leaf
// or typedef containin an enumeration) is output to a Go enumeration. This is
// formed by defining a new type, based on int64 - with constants defined for
// the values of the enumeration. Value 0 is always assigned to be UNSET, such
// that it is possible to determine that the enumerated value was not modified,
// with each subsequent value being assigned the next numeric value.
//
// If the name of an enumerated value is not Go safe - e.g., VALUE-ONE - it is
// converted to a Go-safe name using safeGoEnumeratedName, this function
// replaces characters that cannot be used within Go identifiers with
// underscores (where such characters are allowed in YANG identifiers).
//
// For example, a YANG identity defined in module "test-module" as:
//	identity BASE-IDENTITY;
//	identity VALUE-ONE { base "BASE-IDENTITY"; }
//	identity VALUE-TWO { base "BASE-IDENTITY"; }
//	identity VALUE-THREE { base "BASE-IDENTITY"; }
//
// is output to the following code:
//
//	type TestModule_BaseIdentity int64
//	const (
//		TestModule_BaseIdentity_UNSET = 0
//		TestModule_BaseIdentity_VALUE_ONE = 1
//		TestModule_BaseIdentity_VALUE_TWO = 2
//		TestModule_BaseIdentity_VALUE_THREE = 3
//	)
//
// Calling code can then set any identityref with a base of BASE-IDENTITY
// by setting a value to one of these constants.

// GoStructCodeSnippet is used to store the generated code snippets associated with
// a particular Go struct entity (generated from a container or list).
type GoStructCodeSnippet struct {
	// StructName is the name of the struct that is contained within the snippet.
	// It is stored such that callers can identify the struct to control where it
	// is output.
	StructName string
	// StructDef stores the code snippet that represents the struct that is
	// the input when code generation is performed.
	StructDef string
	// ListKeys stores code snippets that are associated with structs that are
	// generated to represent the keys of multi-key lists. In the case that the
	// Go struct for which the code is being generated does not contain a list
	// with multiple keys, this string is empty.
	ListKeys string
	// Methods contains code snippsets that represent functions that have the
	// input struct as a receiver, that help the user create new entries within
	// lists, without needing to populate the keys of the list.
	Methods string
	// Interfaces contains code snippets that represent interfaces that are
	// used within the generated struct. Used when there are interfaces that
	// represent multi-type unions generated.
	Interfaces string
}

// String returns the contents of the receiver GoStructCodeSnippet as a string.
func (g GoStructCodeSnippet) String() string {
	var b strings.Builder
	for _, s := range []string{g.StructDef, g.ListKeys, g.Methods, g.Interfaces} {
		genutil.WriteIfNotEmpty(&b, s)
	}
	return b.String()
}

// goStructField contains a definition of a field within a Go struct.
type goStructField struct {
	// YANGName is the YANG name of the key field.
	YANGName string
	// Name is the field's name.
	Name string
	// Type is the Go type of the field.
	Type string
	// IsScalarField represents whether the element is a leaf, rather than a
	// leaf-list or container. It is set to false explicitly where there are
	// scalar types that are not mapped to pointers (particularly, enumerated
	// types.
	IsScalarField bool
	// Tags specifies the tags that should be used to annotate the field.
	Tags string
	// IsYANGContainer stores whether the field is a YANG container. This value
	// is used in templates to determine whether GetOrCreate methods should be
	// created.
	IsYANGContainer bool
	// IsYANGList stores whether the field is a YANG list. This value is used
	// in templates to determine whether GetXXX methods should be created using
	// the base template.
	IsYANGList bool
}

// goUnionInterface contains a definition of an interface that should
// be generated for a multi-type union in YANG.
type goUnionInterface struct {
	Name                 string                 // Name is the name of the interface
	Types                map[string]string      // Types is a map keyed by the camelcase type name, with values of the Go types in the union.
	LeafPath             string                 // LeafPath stores the path of the leaf for which the multi-type union is being generated.
	ParentReceiver       string                 // ParentReceiver is the name of the struct that is a parent of this union field. It is used to allow methods to be created which simplify handling the union in the calling code.
	TypeNames            []string               // TypeNames is a list of Go type names within the union.
	ConversionSpecs      []*unionConversionSpec // ConversionSpecs contains information on how to convert primitive types to their own union-satisfying types.
	HasUnsupported       bool                   // HasUnsupported indicates that at least one of the union's subtypes is unsupported.
	SubtypeDocumentation string                 // SubtypeDocumentation gives a documentation-style string on the subtypes of the union.
}

// generatedGoStruct is used to repesent a Go structure to be handed to a template for output.
type generatedGoStruct struct {
	StructName      string           // StructName is the name of the struct being output.
	YANGPath        string           // YANGPath is the schema path of the struct being output.
	Fields          []*goStructField // Fields is the slice of fields of the struct, described as goStructField structs.
	BelongingModule string           // BelongingModule is the module in which namespace the GoStruct belongs.
}

// yangFieldMap maps a YANG identifier to its Go identifier.
type yangFieldMap struct {
	// YANGName is the field's name in the YANG schema.
	YANGName string
	// GoName is the field's name in the Go struct.
	GoName string
	// IsPtr indicates that the key field is a pointer.
	IsPtr bool
}

// generatedGoEnumeration is used to represent a Go enumerated value to be handed
// to a template for output.
type generatedGoEnumeration struct {
	// EnumerationPrefix is the prefix that should be used to any value for
	// the generated output. For example, if EnumerationPrefix is set to
	// OpenconfigBGP_AfiSafi then the enumerated "AFI-SAFIs" will be named
	// OpenconfigBGP_AfiSafi_IPV4, etc., where "IPV4" is the name of one of
	// the enumerated values. The generated type that is referred to is the
	// EnumerationPrefix with a further prefix of E_ such that it can be
	// distinguished from a value of the enumeration in documentation.
	EnumerationPrefix string
	// Values is a map of numeric index to string which represents the valus of the
	// enumerated type. The numeric value may be explicitly assigned by the schema,
	// or populated by goyang during the parsing of the module.
	Values map[int64]string
}

// generatedLeafGetter is used to represent the parameters required to generate a
// getter for a leaf within the generated Go code.
type generatedLeafGetter struct {
	// Name is the name of the field. It is used as a suffix to Get to generate
	// the getter.
	Name string
	// Type is the type of the field, returned by the generated method.
	Type string
	// Zero is the value that should be returned if the field is set to nil.
	Zero string
	// Default is the default value specified in the YANG schema for the type
	// or leaf. For leaf-lists, this would be a Go literal for a slice.
	Default *string
	// IsPtr stores whether the value is a pointer, such that it can be checked
	// against nil, or against the zero value.
	IsPtr bool
	// Receiver is the name of the receiver for the getter method.
	Receiver string
}

// generatedLeafSetter is used to represent the parameters required to generate a
// setter for a leaf within the generated Go code.
type generatedLeafSetter struct {
	// Name is the name of the field. It is used as a suffix to Set to generate
	// the setter.
	Name string
	// Type is the type of the field, required by the setter method.
	Type string
	// IsPtr stores whether the value is a pointer.
	IsPtr bool
	// Receiver is the name of the receiver for the setter method.
	Receiver string
}

// generatedDefaultMethod is used to represent parameters required to generate
// a PopulateDefaults method for a GoStruct that recursively populates default
// values within the subtree.
type generatedDefaultMethod struct {
	// Receiver is the name of the receiver for the default method.
	Receiver string
	// ChildContainerNames are the names of the container fields of the GoStruct.
	ChildContainerNames []string
	// ChildUnorderedListNames are the names of the unordered list fields of the GoStruct.
	ChildUnorderedListNames []string
	// ChildOrderedListNames are the names of the ordered list fields of the GoStruct.
	ChildOrderedListNames []string
	// Leaves represent the leaf fields of the GoStruct.
	Leaves []*generatedLeafGetter
}

// mustMakeTemplate generates a template.Template for a particular named source
// template; with a common set of helper functions.
func mustMakeTemplate(name, src string) *template.Template {
	return template.Must(template.New(name).Funcs(igenutil.TemplateHelperFunctions).Parse(src))
}

var (
	// goCommonHeaderTemplate is populated and output at the top of the generated code package
	goCommonHeaderTemplate = mustMakeTemplate("commonHeader", `
{{- /**/ -}}
/*
Package {{ .PackageName }} is a generated package which contains definitions
of structs which represent a YANG schema. The generated schema can be
compressed by a series of transformations (compression was {{ .CompressEnabled }}
in this case).

This package was generated by {{ .GeneratingBinary }}
using the following YANG input files:
{{- range $inputFile := .YANGFiles }}
	- {{ $inputFile }}
{{- end }}
Imported modules were sourced from:
{{- range $importPath := .IncludePaths }}
	- {{ $importPath }}
{{- end }}
*/
package {{ .PackageName }}

import (
	"encoding/json"
	"fmt"
	"reflect"

	"{{ .GoOptions.YgotImportPath }}"

{{- if .GenerateSchema }}
	"{{ .GoOptions.GoyangImportPath }}"
	"{{ .GoOptions.YtypesImportPath }}"
{{- end }}
{{- if .GoOptions.IncludeModelData }}
	gpb "{{ .GoOptions.GNMIProtoPath }}"
{{- end }}
)
`)

	// goOneOffHeaderTemplate defines the template for package code that should
	// be output in only one file.
	goOneOffHeaderTemplate = mustMakeTemplate("oneoffHeader", `
// {{ .BinaryTypeName }} is a type that is used for fields that have a YANG type of
// binary. It is used such that binary fields can be distinguished from
// leaf-lists of uint8s (which are mapped to []uint8, equivalent to
// []byte in reflection).
type {{ .BinaryTypeName }} []byte

// {{ .EmptyTypeName }} is a type that is used for fields that have a YANG type of
// empty. It is used such that empty fields can be distinguished from boolean fields
// in the generated code.
type {{ .EmptyTypeName }} bool

{{- if .GoOptions.GenerateSimpleUnions }}

// UnionInt8 is an int8 type assignable to unions of which it is a subtype.
type UnionInt8 int8

// UnionInt16 is an int16 type assignable to unions of which it is a subtype.
type UnionInt16 int16

// UnionInt32 is an int32 type assignable to unions of which it is a subtype.
type UnionInt32 int32

// UnionInt64 is an int64 type assignable to unions of which it is a subtype.
type UnionInt64 int64

// UnionUint8 is a uint8 type assignable to unions of which it is a subtype.
type UnionUint8 uint8

// UnionUint16 is a uint16 type assignable to unions of which it is a subtype.
type UnionUint16 uint16

// UnionUint32 is a uint32 type assignable to unions of which it is a subtype.
type UnionUint32 uint32

// UnionUint64 is a uint64 type assignable to unions of which it is a subtype.
type UnionUint64 uint64

// UnionFloat64 is a float64 type assignable to unions of which it is a subtype.
type UnionFloat64 float64

// UnionString is a string type assignable to unions of which it is a subtype.
type UnionString string

// UnionBool is a bool type assignable to unions of which it is a subtype.
type UnionBool bool

// UnionUnsupported is an interface{} wrapper type for unsupported types. It is
// assignable to unions of which it is a subtype.
type UnionUnsupported struct {
	Value interface{}
}

{{- end }}

{{- if .GenerateSchema }}

var (
	SchemaTree map[string]*yang.Entry
	ΛEnumTypes map[string][]reflect.Type
)

func init() {
	var err error
	initΛEnumTypes()
	if SchemaTree, err = UnzipSchema(); err != nil {
		panic("schema error: " +  err.Error())
	}
}

// Schema returns the details of the generated schema.
func Schema() (*ytypes.Schema, error) {
	uzp, err := UnzipSchema()
	if err != nil {
		return nil, fmt.Errorf("cannot unzip schema, %v", err)
	}

	return &ytypes.Schema{
		Root: {{ .FakeRootName }},
		SchemaTree: uzp,
		Unmarshal: Unmarshal,
	}, nil
}

// UnzipSchema unzips the zipped schema and returns a map of yang.Entry nodes,
// keyed by the name of the struct that the yang.Entry describes the schema for.
func UnzipSchema() (map[string]*yang.Entry, error) {
	var schemaTree map[string]*yang.Entry
	var err error
	if schemaTree, err = ygot.GzipToSchema(ySchema); err != nil {
		return nil, fmt.Errorf("could not unzip the schema; %v", err)
	}
	return schemaTree, nil
}

// Unmarshal unmarshals data, which must be RFC7951 JSON format, into
// destStruct, which must be non-nil and the correct GoStruct type. It returns
// an error if the destStruct is not found in the schema or the data cannot be
// unmarshaled. The supplied options (opts) are used to control the behaviour
// of the unmarshal function - for example, determining whether errors are
// thrown for unknown fields in the input JSON.
func Unmarshal(data []byte, destStruct ygot.GoStruct, opts ...ytypes.UnmarshalOpt) error {
	tn := reflect.TypeOf(destStruct).Elem().Name()
	schema, ok := SchemaTree[tn]
	if !ok {
		return fmt.Errorf("could not find schema for type %s", tn )
	}
	var jsonTree interface{}
	if err := json.Unmarshal([]byte(data), &jsonTree); err != nil {
		return err
	}
	return ytypes.Unmarshal(schema, destStruct, jsonTree, opts...)
}

{{- end }}

{{- if .GoOptions.IncludeModelData }}
// ΓModelData contains the catalogue information corresponding to the modules for
// which Go code was generated.
var ΓModelData = []*gpb.ModelData{
{{- range $idx, $model := .ModelData }}
	{
		Name: "{{ .Name }}",
		{{- with $model.Organization }}
		Organization: "{{ . }}",
		{{- end }}
		{{- with $model.Version }}
		Version: "{{ . }}",
		{{- end }}
	},
{{- end }}
}
{{- end }}
`)

	// goStructTemplate takes an input generatedGoStruct, which contains a definition of
	// a container or list YANG schema node, and generates the Go code from it. The
	// Fields slice in the generatedGoStruct contains the child schema nodes of the
	// YANG schema node, along with the mapped Go type that is to be used to represent
	// them. The logic populating the generatedGoStruct handles non-scalar child schema
	// nodes: leaf-lists are mapped into slices; lists are mapped into a map or slice of
	// structs; and containers are mapped into structs.
	goStructTemplate = mustMakeTemplate("struct", `
// {{ .StructName }} represents the {{ .YANGPath }} YANG schema element.
type {{ .StructName }} struct {
{{- range $idx, $field := .Fields }}
	{{- if $field.IsScalarField }}
	{{ $field.Name }}	*{{ $field.Type }}	`+"`"+`{{ $field.Tags }}`+"`"+`
	{{- else }}
	{{ $field.Name }}	{{ $field.Type }}	`+"`"+`{{ $field.Tags }}`+"`"+`
	{{- end }}
{{- end }}
}

// IsYANGGoStruct ensures that {{ .StructName }} implements the yang.GoStruct
// interface. This allows functions that need to handle this struct to
// identify it as being generated by ygen.
func (*{{ .StructName }}) IsYANGGoStruct() {}
`)

	// goStructValidatorTemplate takes an input generatedGoStruct, which contains
	// a definition of a YANG schema node, and generates the Go validation code
	// from it.
	goStructValidatorTemplate = mustMakeTemplate("structValidator", `
// Validate validates s against the YANG schema corresponding to its type.
func (t *{{ .StructName }}) ΛValidate(opts ...ygot.ValidationOption) error {
	if err := ytypes.Validate(SchemaTree["{{ .StructName }}"], t, opts...); err != nil {
		return err
	}
	return nil
}
`)

	// goStructValidatorProxyTemplate creates a proxy for the ΛValidate function with the
	// user definable name.
	goStructValidatorProxyTemplate = mustMakeTemplate("structValidatorProxy", `
// Validate validates s against the YANG schema corresponding to its type.
func (t *{{ .StructName }}) {{ .ValidateProxyFnName }}(opts ...ygot.ValidationOption) error {
	return t.ΛValidate(opts...)
}
`)

	// goContainerGetterTemplate defines a template that generates a getter function
	// for the field of a generated struct. It is generated only for YANG containers.
	goContainerGetterTemplate = mustMakeTemplate("getContainer", `
// Get{{ .Field.Name }} returns the value of the {{ .Field.Name }} struct pointer
// from {{ .StructName }}. If the receiver or the field {{ .Field.Name }} is nil, nil
// is returned such that the Get* methods can be safely chained.
func (t *{{ .StructName }}) Get{{ .Field.Name }}() {{ .Field.Type }} {
	if t != nil && t.{{ .Field.Name }} != nil {
		return t.{{ .Field.Name }}
	}
	return nil
}
`)

	// goGetOrCreateStructTemplate is a template that generates a getter
	// function for a struct field of the receiver struct. The function generated
	// creates the field if it does not exist.
	goGetOrCreateStructTemplate = mustMakeTemplate("getOrCreateStruct", `
// GetOrCreate{{ .Field.Name }} retrieves the value of the {{ .Field.Name }} field
// or returns the existing field if it already exists.
func (t *{{ .StructName }}) GetOrCreate{{ .Field.Name }}() {{ .Field.Type }} {
	if t.{{ .Field.Name }} != nil {
		return t.{{ .Field.Name }}
	}
	t.{{ .Field.Name }} = &{{ stripAsteriskPrefix .Field.Type }}{}
	return t.{{ .Field.Name }}
}
`)

	// goEnumDefinitionTemplate takes an input generatedGoEnumeration struct
	// and outputs the Go code that is associated with the enumerated type to be
	// generated.
	goEnumDefinitionTemplate = mustMakeTemplate("enumDefinition", `
// E_{{ .EnumerationPrefix }} is a derived int64 type which is used to represent
// the enumerated node {{ .EnumerationPrefix }}. An additional value named
// {{ .EnumerationPrefix }}_UNSET is added to the enumeration which is used as
// the nil value, indicating that the enumeration was not explicitly set by
// the program importing the generated structures.
type E_{{ .EnumerationPrefix }} int64

// IsYANGGoEnum ensures that {{ .EnumerationPrefix }} implements the yang.GoEnum
// interface. This ensures that {{ .EnumerationPrefix }} can be identified as a
// mapped type for a YANG enumeration.
func (E_{{ .EnumerationPrefix }}) IsYANGGoEnum() {}

// ΛMap returns the value lookup map associated with  {{ .EnumerationPrefix }}.
func (E_{{ .EnumerationPrefix }}) ΛMap() map[string]map[int64]ygot.EnumDefinition { return ΛEnum; }

// String returns a logging-friendly string for E_{{ .EnumerationPrefix }}.
func (e E_{{ .EnumerationPrefix }}) String() string {
	return ygot.EnumLogString(e, int64(e), "E_{{ .EnumerationPrefix }}")
}

{{ $enumName := .EnumerationPrefix -}}
const (
	{{- range $i, $val := .Values }}
	// {{ $enumName }}_{{ $val }} corresponds to the value {{ $val }} of {{ $enumName }}
	{{ $enumName }}_{{ $val }} E_{{ $enumName }} = {{ $i }}
	{{- end }}
)
`)

	// goLeafGetterTemplate defines a template for a function that, for a
	// particular leaf, generates a getter method.
	goLeafGetterTemplate = mustMakeTemplate("getLeaf", `
// Get{{ .Name }} retrieves the value of the leaf {{ .Name }} from the {{ .Receiver }}
// struct. If the field is unset but has a default value in the YANG schema,
// then the default value will be returned.
// Caution should be exercised whilst using this method since when without a
// default value, it will return the Go zero value if the field is explicitly
// unset. If the caller explicitly does not care if {{ .Name }} is set, it can
// safely use t.Get{{ .Name }}() to retrieve the value. In the case that the
// caller has different actions based on whether the leaf is set or unset, it
// should use 'if t.{{ .Name }} == nil' before retrieving the leaf's value.
func (t *{{ .Receiver }}) Get{{ .Name }}() {{ .Type }} {
	if t == nil || t.{{ .Name }} == {{ if .IsPtr -}} nil {{- else }} {{ .Zero }} {{- end }} {
		{{- if .Default }}
		return {{ .Default }}
		{{- else }}
		return {{ .Zero }}
		{{- end }}
	}
	return {{ if .IsPtr -}} * {{- end -}} t.{{ .Name }}
}
`)

	// goLeafSetterTemplate defines a template for a function that, for a
	// particular leaf, generates a setter method.
	goLeafSetterTemplate = mustMakeTemplate("setLeaf", `
// Set{{ .Name }} sets the value of the leaf {{ .Name }} in the {{ .Receiver }}
// struct.
func (t *{{ .Receiver }}) Set{{ .Name }}(v {{ .Type }}) {
	t.{{ .Name }} = {{ if .IsPtr -}} & {{- end -}} v
}
`)

	// goDefaultMethodTemplate is a template for generating a PopulateDefaults method
	// for a GoStruct that recursively populates default values within the subtree.
	goDefaultMethodTemplate = mustMakeTemplate("populateDefaults", `
// PopulateDefaults recursively populates unset leaf fields in the {{ .Receiver }}
// with default values as specified in the YANG schema, instantiating any nil
// container fields.
func (t *{{ .Receiver }}) PopulateDefaults() {
	if (t == nil) {
		return
	}
	ygot.BuildEmptyTree(t)

	{{- range $Leaf := .Leaves }}
	{{- if $Leaf.Default }}
	if t.{{ $Leaf.Name }} == {{ if $Leaf.IsPtr -}} nil {{- else }} {{ $Leaf.Zero }} {{- end }} {
		{{- if $Leaf.IsPtr }}
		var v {{ $Leaf.Type }} = {{ $Leaf.Default }}
		t.{{ $Leaf.Name }} = &v
		{{- else }}
		t.{{ $Leaf.Name }} = {{ $Leaf.Default }}
		{{- end }}
	}
	{{- end }}
	{{- end }}
	{{- range $containerName := .ChildContainerNames }}
	t.{{ $containerName }}.PopulateDefaults()
	{{- end }}
	{{- range $listName := .ChildUnorderedListNames }}
	for _, e := range t.{{ $listName }} {
		e.PopulateDefaults()
	}
	{{- end }}
	{{- range $listName := .ChildOrderedListNames }}
	for _, e := range t.{{ $listName }}.Values() {
		e.PopulateDefaults()
	}
	{{- end }}
}
`)

	// goEnumMapTemplate provides a template to output a constant map which
	// can be used to resolve the string value of any enumeration within the
	// schema.
	goEnumMapTemplate = mustMakeTemplate("enumMap", `
// ΛEnum is a map, keyed by the name of the type defined for each enum in the
// generated Go code, which provides a mapping between the constant int64 value
// of each value of the enumeration, and the string that is used to represent it
// in the YANG schema. The map is named ΛEnum in order to avoid clash with any
// valid YANG identifier.
var ΛEnum = map[string]map[int64]ygot.EnumDefinition{
	{{- range $enumName, $enumValues := . }}
	"E_{{ $enumName }}": {
		{{- range $value, $valDef := $enumValues }}
		{{ $value }}: {Name: "{{ $valDef.Name }}"
			{{- if ne $valDef.DefiningModule "" -}}
				, DefiningModule: "{{ $valDef.DefiningModule }}"
			{{- end -}}
		},
		{{- end }}
	},
	{{- end }}
}
`)

	// goEnumTypeMapTemplate provides a template to output a constant map which
	// can be used to resolve a schemapath to the set of enumerated types that
	// are valid for the leaf or leaf-list defined at the path specified.
	goEnumTypeMapTemplate = mustMakeTemplate("enumTypeMap", `
// ΛEnumTypes is a map, keyed by a YANG schema path, of the enumerated types that
// correspond with the leaf. The type is represented as a reflect.Type. The naming
// of the map ensures that there are no clashes with valid YANG identifiers.
func initΛEnumTypes(){
  ΛEnumTypes = map[string][]reflect.Type{
  {{- range $schemapath, $types := . }}
	"{{ $schemapath }}": []reflect.Type{
		{{- range $i, $t := $types }}
		reflect.TypeOf(({{ $t }})(0)),
		{{- end }}
	},
	{{- end }}
  }
}
`)

	// goEnumTypeMapAccessTemplate provides a template to output an accessor
	// function with a generated struct as receiver, it returns the enum type
	// map associated with the generated code.
	goEnumTypeMapAccessTemplate = mustMakeTemplate("enumTypeMapAccessor", `
// ΛEnumTypeMap returns a map, keyed by YANG schema path, of the enumerated types
// that are included in the generated code.
func (t *{{ .StructName }}) ΛEnumTypeMap() map[string][]reflect.Type { return ΛEnumTypes }
`)

	// goBelongingModuleTemplate provides a template to output a
	// function that has a generated struct as receiver, and returns the
	// name of the module in which namespace the generated struct belongs.
	goBelongingModuleTemplate = mustMakeTemplate("belongingModuleMethod", `
// ΛBelongingModule returns the name of the module that defines the namespace
// of {{ .StructName }}.
func (*{{ .StructName }}) ΛBelongingModule() string {
	return "{{ .BelongingModule }}"
}
`)

	// schemaVarTemplate provides a template to output a constant byte
	// slice which contains the serialised schema of the YANG modules for
	// which code generation was performed.
	schemaVarTemplate = mustMakeTemplate("schemaVar", `
var (
	// {{ .VarName }} is a byte slice contain a gzip compressed representation of the
	// YANG schema from which the Go code was generated. When uncompressed the
	// contents of the byte slice is a JSON document containing an object, keyed
	// on the name of the generated struct, and containing the JSON marshalled
	// contents of a goyang yang.Entry struct, which defines the schema for the
	// fields within the struct.
	{{ .VarName }} = []byte{
{{- range $i, $line := .Schema }}
		{{ $line }}
{{- end }}
	}
)
`)

	// unionTypeTemplate outputs the type that corresponds to a multi-type union
	// in the YANG schema.
	unionTypeTemplate = mustMakeTemplate("unionType", `
// {{ .Name }} is an interface that is implemented by valid types for the union
// for the leaf {{ .LeafPath }} within the YANG schema.
type {{ .Name }} interface {
	Is_{{ .Name }}()
}
{{ $intfName := .Name }}
{{- $path := .LeafPath }}
{{- range $typeName, $type := .Types }}
// {{ $intfName }}_{{ $typeName }} is used when {{ $path }}
// is to be set to a {{ $type }} value.
type {{ $intfName }}_{{ $typeName }} struct {
	{{ $typeName }}	{{ $type }}
}

// Is_{{ $intfName }} ensures that {{ $intfName }}_{{ $typeName }}
// implements the {{ $intfName }} interface.
func (*{{ $intfName }}_{{ $typeName }}) Is_{{ $intfName }}() {}
{{ end -}}
`)

	// unionHelperTemplate defines a template that defines a helper method
	// with a particular receiver type that allows an input type to be converted
	// to its corresponding type in the union type.
	unionHelperTemplate = mustMakeTemplate("unionHelper", `
{{- $intfName := .Name }}
{{- $path := .LeafPath }}
// To_{{ .Name }} takes an input interface{} and attempts to convert it to a struct
// which implements the {{ .Name }} union. It returns an error if the interface{} supplied
// cannot be converted to a type within the union.
func (t *{{ .ParentReceiver }}) To_{{ .Name }}(i interface{}) ({{ .Name }}, error) {
	switch v := i.(type) {
	{{ range $typeName, $type := .Types -}}
	case {{ $type }}:
		return &{{ $intfName }}_{{ $typeName }}{v}, nil
	{{ end -}}
	default:
		return nil, fmt.Errorf("cannot convert %v to {{ .Name }}, unknown union type, got: %T, want any of [
		{{- $length := len .TypeNames -}}
		{{- range $i, $type := .TypeNames -}}
			{{ $type }}
			{{- if ne (inc $i) $length -}}, {{ end -}}
		{{- end -}}
		]", i, i)
	}
}
`)

	// unionTypeSimpleTemplate outputs the type that corresponds to a multi-type union
	// in the YANG schema. It does so by enhancing generated typedefs.
	unionTypeSimpleTemplate = mustMakeTemplate("unionTypeSimple", `
// {{ .Name }} is an interface that is implemented by valid types for the union
// for the leaf {{ .LeafPath }} within the YANG schema.
// Union type can be one of [{{ .SubtypeDocumentation }}].
type {{ .Name }} interface {
	// Union type can be one of [{{ .SubtypeDocumentation }}]
	Documentation_for_{{ .Name }}()
}
{{ $intfName := .Name -}}
{{- $path := .LeafPath -}}
{{- range $typeName, $type := .Types }}
// Documentation_for_{{ $intfName }} ensures that {{ $typeName }}
// implements the {{ $intfName }} interface.
func ({{ $typeName }}) Documentation_for_{{ $intfName }}() {}
{{ end -}}
`)

	// unionHelperSimpleTemplate defines a template that defines a helper method
	// with a particular receiver type that allows an input type to be converted
	// to its corresponding type in the union type.
	unionHelperSimpleTemplate = mustMakeTemplate("unionHelperSimple", `
{{- $intfName := .Name }}
{{- $path := .LeafPath }}
// To_{{ .Name }} takes an input interface{} and attempts to convert it to a struct
// which implements the {{ .Name }} union. It returns an error if the interface{} supplied
// cannot be converted to a type within the union.
func (t *{{ .ParentReceiver }}) To_{{ .Name }}(i interface{}) ({{ .Name }}, error) {
	if v, ok := i.({{ .Name }}); ok {
		return v, nil
	}
	{{ $length := len .ConversionSpecs -}} {{ $hasLength := ne $length 0 -}} {{ if or $hasLength .HasUnsupported -}}
	switch v := i.(type) {
	{{ range $i, $conversionSpec := .ConversionSpecs -}}
	case {{ $conversionSpec.PrimitiveType }}:
		return {{ $conversionSpec.ConversionSnippet }}, nil
	{{ end -}}
	{{ if .HasUnsupported -}}
	case interface{}:
		return &UnionUnsupported{v}, nil
	{{ end -}}
	}
	{{ end -}}
	return nil, fmt.Errorf("cannot convert %v to {{ .Name }}, unknown union type, got: %T, want any of [
	{{- $length := len .TypeNames -}}
	{{- range $i, $type := .TypeNames -}}
		{{ $type }}
		{{- if ne (inc $i) $length -}}, {{ end -}}
	{{- end -}}
	]", i, i)
}
`)
)

// writeGoHeader outputs the package header, including the package name and
// comments that is to be included with the generated code. The input set of
// files (yangFiles) are output to indicate the modules for which code
// generation was targeted, along with the includePaths indicating where
// imported modules were sourced from. If the cfg.GoOptions.YgotImport path
// is not set, then it is set to the value of GoDefaultYgotImportPath. In a similar manner
// an unset cfg.GoOptions.GoyangImportPath results in the goyang path being set to
// GoDefaultYgotImportPath, and an unset cfg.GoOptions.YtypesImportPath results in the
// path for ytypes being set to GoDefaultYtypesImportPath. The supplied rootName is the
// name of the fake root struct, if it was produced - and is used to output a schema
// definition in the file header.
//
// The header returned is split into two strings, the common header is a header that
// should be used for all files within the output package. The one off header should
// be included in only one file of the package.
func writeGoHeader(yangFiles, includePaths []string, cfg *CodeGenerator, rootName string, modelData []*gpb.ModelData) (string, string, error) {
	// Determine the running binary's name.
	if cfg.Caller == "" {
		cfg.Caller = genutil.CallerName()
	}

	if cfg.GoOptions.PackageName == "" {
		cfg.GoOptions.PackageName = defaultPackageName
	}

	if cfg.GoOptions.YgotImportPath == "" {
		cfg.GoOptions.YgotImportPath = genutil.GoDefaultYgotImportPath
	}
	if cfg.GoOptions.GoyangImportPath == "" {
		cfg.GoOptions.GoyangImportPath = genutil.GoDefaultGoyangImportPath
	}
	if cfg.GoOptions.YtypesImportPath == "" {
		cfg.GoOptions.YtypesImportPath = genutil.GoDefaultYtypesImportPath
	}
	if cfg.GoOptions.GNMIProtoPath == "" {
		cfg.GoOptions.GNMIProtoPath = genutil.GoDefaultGNMIImportPath
	}

	// Build input to the header template which stores parameters which are included
	// in the header of generated code.
	s := struct {
		PackageName      string           // PackgeName is the name of the package to be generated.
		YANGFiles        []string         // YANGFiles contains the list of input YANG source files for code generation.
		IncludePaths     []string         // IncludePaths contains the list of paths that included modules were searched for in.
		CompressEnabled  bool             // CompressEnabled indicates whether compression is enabled.
		GeneratingBinary string           // GeneratingBinary is the name of the binary generating the code.
		GenerateSchema   bool             // GenerateSchema stores whether the generator requested that the schema was to be stored with the output code.
		GoOptions        GoOpts           // GoOptions stores additional Go-specific options for the output code, including package paths.
		BinaryTypeName   string           // BinaryTypeName is the name of the type used for YANG binary types.
		EmptyTypeName    string           // EmptyTypeName is the name of the type used for YANG empty types.
		FakeRootName     string           // FakeRootName is the name of the fake root struct in the YANG type
		ModelData        []*gpb.ModelData // ModelData contains the gNMI ModelData definition for the input types.
	}{
		PackageName:      cfg.GoOptions.PackageName,
		YANGFiles:        yangFiles,
		IncludePaths:     includePaths,
		CompressEnabled:  cfg.IROptions.TransformationOptions.CompressBehaviour.CompressEnabled(),
		GeneratingBinary: cfg.Caller,
		GenerateSchema:   cfg.GoOptions.GenerateJSONSchema,
		GoOptions:        cfg.GoOptions,
		BinaryTypeName:   ygot.BinaryTypeName,
		EmptyTypeName:    ygot.EmptyTypeName,
		ModelData:        modelData,
	}

	s.FakeRootName = "nil"
	if cfg.IROptions.TransformationOptions.GenerateFakeRoot && rootName != "" {
		s.FakeRootName = fmt.Sprintf("&%s{}", rootName)
	}

	var common bytes.Buffer
	if err := goCommonHeaderTemplate.Execute(&common, s); err != nil {
		return "", "", err
	}

	var oneoff bytes.Buffer
	if err := goOneOffHeaderTemplate.Execute(&oneoff, s); err != nil {
		return "", "", err
	}

	return common.String(), oneoff.String(), nil
}

// IsScalarField determines which fields should be converted to pointers when
// outputting structs; this is done to allow checks against nil.
func IsScalarField(field *ygen.NodeDetails) bool {
	switch {
	// A non-singleton-leaf always has a generated type for which nil is valid.
	case field.Type != ygen.LeafNode:
		return false
	// A union shouldn't be a pointer since its field type is an interface;
	case len(field.LangType.UnionTypes) >= 2:
		return false
	// an enumerated value shouldn't be a pointer either since its has an UNSET value;
	case field.LangType.IsEnumeratedValue:
		return false
	// an unmapped type (interface{}), byte slice, or a leaflist can also use nil already, so they should also not be pointers.
	case field.LangType.NativeType == ygot.BinaryTypeName, field.LangType.NativeType == ygot.EmptyTypeName, field.LangType.NativeType == "interface{}":
		return false
	}
	return true
}

// writeGoStruct generates code snippets for targetStruct. The parameter goStructElements
// contains other Directory structs for which code is being generated, that may be referenced
// during the generation of the code corresponding to targetStruct (e.g., to determine a
// child container's struct name).
//
// writeGoStruct takes the following additional arguments:
//   - targetStruct - the YANG directory (container/list) to be converted to generated code.
//   - goStructElements - All existing YANG directories (for looking up children).
//   - generatedUnions - Running map of generated unions to avoid generating the
//     same union twice.
//   - goOpts - Go specific code generation options as a GoOpts struct.
//
// writeGoStruct returns a GoStructCodeSnippet which contains
//  1. The generated struct for targetStruct (structDef)
//  2. Additional generated structs that are keys for any multi-key lists that are children
//     of targetStruct (listKeys).
//  3. Methods with the struct corresponding to targetStruct as a receiver, e.g., for each
//     list a NewListMember() method is generated.
func writeGoStruct(targetStruct *ygen.ParsedDirectory, goStructElements map[string]*ygen.ParsedDirectory, generatedUnions map[string]bool, goOpts GoOpts) (GoStructCodeSnippet, []error) {
	if targetStruct == nil {
		return GoStructCodeSnippet{}, []error{fmt.Errorf("cannot create code for nil targetStruct")}
	}

	var errs []error

	// structDef is used to store the attributes of the structure for which code is being
	// generated.
	structDef := generatedGoStruct{
		StructName:      targetStruct.Name,
		YANGPath:        targetStruct.Path,
		BelongingModule: targetStruct.BelongingModule,
	}

	// associatedListKeyStructs is a slice containing the key structures for any multi-keyed
	// lists that are fields of the struct.
	associatedListKeyStructs := []*generatedGoMultiKeyListStruct{}

	var associatedOrderedMapStructs []*generatedOrderedMapStruct

	// associatedListMethods is a slice of pointers to generatedGoListMethod structs
	// which describe methods that use the target struct as a receiver. These structs
	// represent the methods that are used as helpers such as those methods that allow
	// a new member to be created within the list (populating the keys), and an
	// existing list member to be appended to the list.
	var associatedListMethods []*generatedGoListMethod

	// associatedLeafGetters is a slice of structs which define the set of leaf getters
	// to generated for the struct.
	var associatedLeafGetters []*generatedLeafGetter

	// associatedLeafSetters is a slice of structs which define the set of leaf setters
	// to generated for the struct.
	var associatedLeafSetters []*generatedLeafSetter

	associatedDefaultMethod := generatedDefaultMethod{
		Receiver: targetStruct.Name,
	}

	// definedNameMap defines a map, keyed by YANG identifier to the Go struct field name.
	definedNameMap := map[string]*yangFieldMap{}

	// genUnions stores the set of multi-type YANG unions that must have
	// code generated for them.
	genUnions := []goUnionInterface{}
	// genUnionSet stores a set of union type names such that we can process
	// each appearance of a union type within the struct once and only once.
	genUnionSet := map[string]bool{}

	annotationPrefix := goOpts.AnnotationPrefix
	// Set the default annotation prefix if it is unset.
	if goOpts.AnnotationPrefix == "" {
		annotationPrefix = DefaultAnnotationPrefix
	}

	if goOpts.AddAnnotationFields {
		// Add the top-level struct metadata field.
		structDef.Fields = append(structDef.Fields, &goStructField{
			Name: fmt.Sprintf("%sMetadata", annotationPrefix),
			Type: annotationFieldType,
			Tags: `path:"@" ygotAnnotation:"true"`,
		})
	}

	goFieldNameMap := ygen.GoFieldNameMap(targetStruct)
	// Alphabetically order fields to produce deterministic output.
	for _, fName := range targetStruct.OrderedFieldNames() {
		// Iterate through the fields of the struct that we are generating code for.
		// For each field, calculate the name of the field (ensuring that it is unique), and
		// the corresponding type. fieldDef is used to store the definition of the field (name
		// and type) that are calculated.
		var fieldDef *goStructField

		field := targetStruct.Fields[fName]
		fieldName := goFieldNameMap[fName]
		definedNameMap[fName] = &yangFieldMap{YANGName: fName, GoName: fieldName}

		switch field.Type {
		case ygen.ListNode:
			// If the field within the struct is a list, then generate code for this list. This
			// includes extracting any new types that are required to represent the key of a
			// list that has multiple keys.
			fieldType, multiKeyListKey, listMethods, orderedMapSpec, listErr := yangListFieldToGoType(field, fieldName, targetStruct, goStructElements, !goOpts.GenerateOrderedListsAsUnorderedMaps)
			if listErr != nil {
				errs = append(errs, listErr)
			}

			fieldDef = &goStructField{
				Name:       fieldName,
				Type:       fieldType,
				IsYANGList: true,
			}

			if listMethods != nil {
				associatedListMethods = append(associatedListMethods, listMethods)
			}

			if orderedMapSpec != nil {
				associatedOrderedMapStructs = append(associatedOrderedMapStructs, orderedMapSpec)
				associatedDefaultMethod.ChildOrderedListNames = append(associatedDefaultMethod.ChildOrderedListNames, fieldName)
			} else {
				associatedDefaultMethod.ChildUnorderedListNames = append(associatedDefaultMethod.ChildUnorderedListNames, fieldName)
			}

			if multiKeyListKey != nil {
				// If the list had multiple keys, add the struct that represented the list
				// type to the slice of those that should have code generated for them.
				associatedListKeyStructs = append(associatedListKeyStructs, multiKeyListKey)
			}

		case ygen.ContainerNode:
			// This is a YANG container, so it is represented in code using a pointer to the struct type that
			// is defined for the entity. findMappableEntities has already determined which fields are to
			// be output, so no filtering of the set of fields is required here.
			dir, ok := goStructElements[field.YANGDetails.Path]
			if !ok {
				errs = append(errs, fmt.Errorf("could not resolve %s into a defined struct, %v", field.YANGDetails.Path, goStructElements))
				continue
			}

			fieldDef = &goStructField{
				Name:            fieldName,
				Type:            fmt.Sprintf("*%s", dir.Name),
				IsYANGContainer: true,
			}
			associatedDefaultMethod.ChildContainerNames = append(associatedDefaultMethod.ChildContainerNames, fieldName)
		case ygen.LeafNode, ygen.LeafListNode:
			// Only if this union has more than one subtype do we generate the union;
			// otherwise, we use that subtype directly.
			// Also, make sure to process a union type once and only once within the struct.
			// Even if the union has already been processed in another struct, we still need
			// to generate the union helper with this struct as the receiver and do other
			// processing as well. On the other hand, if the union type is used by multiple
			// fields, we assume that it's due to a leafref and use the same union name
			// without doing further code generation. XXX(wenbli): It's possible that it's a
			// name collision instead, but because of its low chance and the extra complexity
			// required to resolve it (e.g. storing the union entry so we make sure the name
			// is used for the right union entry), we ignore it and allow wrong code to be
			// generated.
			if len(field.LangType.UnionTypes) > 1 && !genUnionSet[field.LangType.NativeType] {
				genUnionSet[field.LangType.NativeType] = true

				intf := goUnionInterface{
					Name:           field.LangType.NativeType,
					Types:          map[string]string{},
					LeafPath:       field.YANGDetails.Path,
					ParentReceiver: targetStruct.Name,
				}

				var genTypes []string
				for t := range field.LangType.UnionTypes {
					tn := yang.CamelCase(t)
					// Ensure that we sanitise the type name to be used in the
					// output struct.
					if t == "interface{}" {
						tn = "Interface"
					}
					if goOpts.GenerateSimpleUnions {
						if simpleName, ok := ygot.SimpleUnionBuiltinGoTypes[t]; ok {
							tn = simpleName
						}
					}
					intf.Types[tn] = t
					genTypes = append(genTypes, tn)
					intf.TypeNames = append(intf.TypeNames, t)
				}
				// Sort the names of the types into deterministic order.
				sort.Strings(intf.TypeNames)
				sort.Strings(genTypes)
				// Populate the union type conversion snippets.
				for _, t := range intf.TypeNames {
					if cs, ok := unionConversionSnippets[t]; ok {
						switch t {
						case "interface{}":
							intf.HasUnsupported = true
						default:
							intf.ConversionSpecs = append(intf.ConversionSpecs, cs)
						}
					}
				}
				// Create the subtype documentation string.
				intf.SubtypeDocumentation = strings.Join(genTypes, ", ")
				genUnions = append(genUnions, intf)
			}

			fType := field.LangType.NativeType
			zeroValue := field.LangType.ZeroValue

			if field.Type == ygen.LeafListNode {
				// We represent a leaf-list in the output
				// code using a slice of the type that the element was mapped to.
				fType = fmt.Sprintf("[]%s", fType)
				// Slices have a nil zero value rather than the value of their
				// underlying type.
				zeroValue = "nil"
			}

			scalarField := IsScalarField(field)

			definedNameMap[fName].IsPtr = scalarField

			// If we are generating leaf getters, then append the relevant information
			// to the associatedLeafGetters slice to be generated along with other
			// associated methods.
			associatedLeafGetters = append(associatedLeafGetters, &generatedLeafGetter{
				Name:     fieldName,
				Type:     fType,
				Zero:     zeroValue,
				IsPtr:    scalarField,
				Receiver: targetStruct.Name,
				Default:  field.LangType.DefaultValue,
			})

			// If we are generating leaf setters, then append the relevant information
			// to the associatedLeafSetters slice to be generated along with other
			// associated methods.
			associatedLeafSetters = append(associatedLeafSetters, &generatedLeafSetter{
				Name:     fieldName,
				Type:     fType,
				IsPtr:    scalarField,
				Receiver: targetStruct.Name,
			})

			fieldDef = &goStructField{
				Name:          fieldName,
				Type:          fType,
				IsScalarField: scalarField,
			}
		default:
			errs = append(errs, fmt.Errorf("unknown entity type for mapping to Go: %s, Kind: %v", field.YANGDetails.Path, field.Type))
			continue
		}

		var tagBuf, metadataTagBuf bytes.Buffer
		// addSchemaPathsToBuffers adds the slice of paths to the tag
		// and metadata tag buffers.
		addSchemaPathsToBuffers := func(schemaPaths [][]string, addToMetadata bool) {
			for i, path := range schemaPaths {
				// Copy the path elements to avoid modifying the IR.
				p := append([]string{}, path...)

				tagBuf.WriteString(util.SlicePathToString(p))

				// Prepend "@" to the last element in the schema path.
				p[len(p)-1] = fmt.Sprintf("@%s", p[len(p)-1])
				if addToMetadata {
					metadataTagBuf.WriteString(util.SlicePathToString(p))
				}

				if i != len(schemaPaths)-1 {
					tagBuf.WriteRune('|')
					if addToMetadata {
						metadataTagBuf.WriteRune('|')
					}
				}
			}
			tagBuf.WriteByte('"')
			if addToMetadata {
				metadataTagBuf.WriteByte('"')
			}
		}

		tagBuf.WriteString(`path:"`)
		metadataTagBuf.WriteString(`path:"`)
		addSchemaPathsToBuffers(field.MappedPaths, true)

		// Append a tag indicating the module that instantiates this field.
		tagBuf.WriteString(` module:"`)
		addSchemaPathsToBuffers(field.MappedPathModules, false)

		if goOpts.IgnoreShadowSchemaPaths {
			if len(field.ShadowMappedPaths) > 0 {
				tagBuf.WriteString(` shadow-path:"`)
				addSchemaPathsToBuffers(field.ShadowMappedPaths, false)
			}
			if len(field.ShadowMappedPathModules) > 0 {
				// Append a tag indicating the module that instantiates this field.
				tagBuf.WriteString(` shadow-module:"`)
				addSchemaPathsToBuffers(field.ShadowMappedPathModules, false)
			}
		}

		metadataTagBuf.WriteString(` ygotAnnotation:"true"`)

		if goOpts.AddYangPresence {
			if field.Type == ygen.ContainerNode && field.YANGDetails.PresenceStatement != nil {
				tagBuf.WriteString(` yangPresence:"true"`)
			}
		}

		fieldDef.Tags = tagBuf.String()

		// Append the generated field definition to the set of fields of the struct.
		structDef.Fields = append(structDef.Fields, fieldDef)

		if goOpts.AddAnnotationFields {
			// Append the definition of the field annotation to the set of fields in the
			// struct.
			structDef.Fields = append(structDef.Fields, &goStructField{
				Name: fmt.Sprintf("%s%s", annotationPrefix, fieldDef.Name),
				Type: annotationFieldType,
				Tags: metadataTagBuf.String(),
			})
		}
	}

	// structBuf is used to store the code associated with the struct defined for
	// the target YANG entity.
	var structBuf bytes.Buffer
	if err := goStructTemplate.Execute(&structBuf, structDef); err != nil {
		errs = append(errs, err)
	}

	// listkeyBuf is a buffer which stores the code associated with structs that
	// are associated with the structs generated to act as list keys.
	var listkeyBuf bytes.Buffer
	for _, listKey := range associatedListKeyStructs {
		if err := goListKeyTemplate.Execute(&listkeyBuf, listKey); err != nil {
			errs = append(errs, err)
		}
	}

	// methodBuf is used to store the code generated for methods that have the
	// target entity's generated struct as a receiver.
	var methodBuf bytes.Buffer
	for _, method := range associatedListMethods {
		if err := goNewListMemberTemplate.Execute(&methodBuf, method); err != nil {
			errs = append(errs, err)
		}

		if goOpts.GenerateRenameMethod {
			if err := goListMemberRenameTemplate.Execute(&methodBuf, method); err != nil {
				errs = append(errs, err)
			}
		}

		if goOpts.GenerateGetters {
			if err := generateGetOrCreateList(&methodBuf, method); err != nil {
				errs = append(errs, err)
			}
			if err := generateListGetter(&methodBuf, method); err != nil {
				errs = append(errs, err)
			}
		}

		if goOpts.GenerateDeleteMethod {
			if err := generateListDelete(&methodBuf, method); err != nil {
				errs = append(errs, err)
			}
		}

		if goOpts.GenerateAppendMethod {
			if err := generateListAppend(&methodBuf, method); err != nil {
				errs = append(errs, err)
			}
		}
	}

	if goOpts.GenerateGetters {
		if err := generateGetOrCreateStruct(&methodBuf, structDef); err != nil {
			errs = append(errs, err)
		}
		if err := generateContainerGetters(&methodBuf, structDef); err != nil {
			errs = append(errs, err)
		}
	}

	if goOpts.GenerateLeafGetters {
		if err := generateLeafGetters(&methodBuf, associatedLeafGetters); err != nil {
			errs = append(errs, err)
		}
	}

	if goOpts.GenerateLeafSetters {
		if err := generateLeafSetters(&methodBuf, associatedLeafSetters); err != nil {
			errs = append(errs, err)
		}
	}

	for _, s := range associatedOrderedMapStructs {
		if err := generateOrderedMapParentMethods(&methodBuf, s); err != nil {
			errs = append(errs, err)
		}
		if err := generateOrderedMapStruct(&methodBuf, s); err != nil {
			errs = append(errs, err)
		}
	}

	if goOpts.GeneratePopulateDefault {
		associatedDefaultMethod.Leaves = associatedLeafGetters
		if err := goDefaultMethodTemplate.Execute(&methodBuf, associatedDefaultMethod); err != nil {
			errs = append(errs, err)
		}
	}

	if err := generateGetListKey(&methodBuf, targetStruct, definedNameMap); err != nil {
		errs = append(errs, err)
	}

	// interfaceBuf is used to store the code generated for interfaces that
	// are used for multi-type unions within the struct.
	var interfaceBuf bytes.Buffer
	for _, intf := range genUnions {
		if goOpts.GenerateSimpleUnions {
			if _, ok := generatedUnions[intf.Name]; !ok {
				if err := unionTypeSimpleTemplate.Execute(&interfaceBuf, intf); err != nil {
					errs = append(errs, err)
				}
				generatedUnions[intf.Name] = true
			}
			if err := unionHelperSimpleTemplate.Execute(&interfaceBuf, intf); err != nil {
				errs = append(errs, err)
			}
		} else {
			if _, ok := generatedUnions[intf.Name]; !ok {
				if err := unionTypeTemplate.Execute(&interfaceBuf, intf); err != nil {
					errs = append(errs, err)
				}
				generatedUnions[intf.Name] = true
			}
			if err := unionHelperTemplate.Execute(&interfaceBuf, intf); err != nil {
				errs = append(errs, err)
			}
		}
	}

	if goOpts.GenerateJSONSchema {
		if err := generateValidator(&methodBuf, structDef, goOpts.ValidateFunctionName); err != nil {
			errs = append(errs, err)
		}

		if err := generateEnumTypeMapAccessor(&methodBuf, structDef); err != nil {
			errs = append(errs, err)
		}
	}

	if err := generateBelongingModuleFunction(&methodBuf, structDef); err != nil {
		errs = append(errs, err)
	}

	return GoStructCodeSnippet{
		StructName: structDef.StructName,
		StructDef:  structBuf.String(),
		Methods:    methodBuf.String(),
		ListKeys:   listkeyBuf.String(),
		Interfaces: interfaceBuf.String(),
	}, errs
}

// mappedPathTag returns a generated Go Struct tag containing the stringified
// input paths separated by '|'. If prefix is supplied, it is prepended to the
// last element in each path.
func mappedPathTag(paths [][]string, prefix string) string {
	var buf bytes.Buffer
	buf.WriteString(`path:"`)
	for i, p := range paths {
		// Copy the slice so that we don't modify it for future callers.
		np := append(p[:0:0], p...)
		np[len(np)-1] = fmt.Sprintf("%s%s", prefix, np[len(np)-1])
		buf.WriteString(util.SlicePathToString(np))

		if i != len(paths)-1 {
			buf.WriteRune('|')
		}
	}
	buf.WriteString(`"`)
	return buf.String()
}

// generateValidator generates a validation function string for structDef and
// appends it to the supplied buffer.
// Assuming structDef represents the following struct:
//
//	type MyStruct struct {
//	  field1 *string
//	}
//
// the validation function generated for the struct will be:
//
//	func (t *MyStruct) ΛValidate(value interface{}) error {
//	  if err := ytypes.Validate(schemaMap["MyStruct"], value); err != nil {
//	    return err
//	  }
//	  return nil
//	}
func generateValidator(buf *bytes.Buffer, structDef generatedGoStruct, validateProxyFunctionName string) error {
	var err error
	if err = goStructValidatorTemplate.Execute(buf, structDef); err != nil {
		return err
	}
	if validateProxyFunctionName != "" {
		parameters := &struct {
			ValidateProxyFnName string
			StructName          string
		}{
			ValidateProxyFnName: validateProxyFunctionName,
			StructName:          structDef.StructName,
		}
		err = goStructValidatorProxyTemplate.Execute(buf, parameters)
	}
	return err
}

// goTmplFieldDetails stores a goStructField along with additional details
// corresponding to it. It is used withAin templates that handle individual
// fields.
type goTmplFieldDetails struct {
	Field      *goStructField // Field stores the definition of the field with which other details are associated.
	StructName string         // StructName is the name of the struct that the field is a member of.
}

// generateGetOrCreateStruct generates a getter method for the YANG container
// (Go struct ptr) fields of structDef, and appends it to the supplied buffer.
// Assuming that structDef represents the following struct:
//
//	 type MyStruct struct {
//			Container *MyStruct_Container
//	 }
//
// the getter function generated for the struct will be:
//
//	func (s *MyStruct) GetOrCreateContainer() *MyStruct_Container {
//	  if s.Container != nil {
//	    return s.Container
//	  }
//	  s.Container = &MyStruct_Container{}
//	  return s.Container
//	}
func generateGetOrCreateStruct(buf *bytes.Buffer, structDef generatedGoStruct) error {
	for _, f := range structDef.Fields {
		if f.IsYANGContainer {
			tmpStruct := goTmplFieldDetails{
				StructName: structDef.StructName,
				Field:      f,
			}
			if err := goGetOrCreateStructTemplate.Execute(buf, tmpStruct); err != nil {
				return err
			}
		}
	}
	return nil
}

// generateContainerGetters generates GetXXX methods for container fields of the
// supplied struct described by structDef.
//
// The method is defined by the goContainerGetterTemplate. This template returns
// the value if the field is set, otherwise returns nil. The generated getters
// are safe to call with a nil receiver.
func generateContainerGetters(buf *bytes.Buffer, structDef generatedGoStruct) error {
	for _, f := range structDef.Fields {
		// Only YANG containers have getters generated for them.
		if !f.IsYANGContainer {
			continue
		}
		tmpStruct := goTmplFieldDetails{
			StructName: structDef.StructName,
			Field:      f,
		}
		if err := goContainerGetterTemplate.Execute(buf, tmpStruct); err != nil {
			return err
		}
	}
	return nil
}

// generateLeafGetters generates GetXXX methods for the leaf fields described by
// the supplied slice of generatedLeafGetter structs.
func generateLeafGetters(buf *bytes.Buffer, leaves []*generatedLeafGetter) error {
	var errs errlist.List
	for _, l := range leaves {
		if err := goLeafGetterTemplate.Execute(buf, l); err != nil {
			errs.Add(err)
		}
	}
	return errs.Err()
}

// generateLeafSetters generates SetXXX methods for the leaf fields described by
// the supplied slice of generatedLeafSetter structs.
func generateLeafSetters(buf *bytes.Buffer, leaves []*generatedLeafSetter) error {
	var errs errlist.List
	for _, l := range leaves {
		if err := goLeafSetterTemplate.Execute(buf, l); err != nil {
			errs.Add(err)
		}
	}
	return errs.Err()
}

// generateEnumTypeMapAccessor generates a function which returns the defined
// enumTypeMap for a struct.
func generateEnumTypeMapAccessor(b *bytes.Buffer, s generatedGoStruct) error {
	return goEnumTypeMapAccessTemplate.Execute(b, s)
}

// generateBelongingModuleFunction generates a function which returns the
// belonging module as a string.
func generateBelongingModuleFunction(b io.Writer, s generatedGoStruct) error {
	return goBelongingModuleTemplate.Execute(b, s)
}
