/* Copyright (C) 2025 The cairomm Development Team
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, see <https://www.gnu.org/licenses/>.
 */

#include <cairomm/mesh_pattern.h>
#include <cairomm/private.h>

namespace Cairo
{
MeshPattern::MeshPattern(cairo_pattern_t* cobject, bool has_reference) :
    Pattern(cobject, has_reference)
{}

MeshPattern::~MeshPattern()
{
  // pattern is destroyed in base class
}

void MeshPattern::begin_patch()
{
  cairo_mesh_pattern_begin_patch(m_cobject);
  check_object_status_and_throw_exception(*this);
}

void MeshPattern::end_patch()
{
  cairo_mesh_pattern_end_patch(m_cobject);
  check_object_status_and_throw_exception(*this);
}


void MeshPattern::move_to(double x, double y)
{
  cairo_mesh_pattern_move_to(m_cobject, x, y);
  check_object_status_and_throw_exception(*this);
}

void MeshPattern::line_to(double x, double y)
{
  cairo_mesh_pattern_line_to(m_cobject, x, y);
  check_object_status_and_throw_exception(*this);
}

void MeshPattern::curve_to(double x1, double y1, double x2, double y2,
                           double x3, double y3)
{
  cairo_mesh_pattern_curve_to(m_cobject, x1, y1, x2, y2, x3, y3);
  check_object_status_and_throw_exception(*this);
}

void MeshPattern::set_control_point(unsigned int point_num, double x, double y)
{
  cairo_mesh_pattern_set_control_point(m_cobject, point_num, x, y);
  check_object_status_and_throw_exception(*this);
}


void MeshPattern::set_corner_color_rgb(unsigned int corner_num, double red,
                                       double green, double blue)
{
  cairo_mesh_pattern_set_corner_color_rgb(m_cobject, corner_num, red, green,
                                          blue);
  check_object_status_and_throw_exception(*this);
}

void MeshPattern::set_corner_color_rgba(unsigned int corner_num, double red,
                                        double green, double blue, double alpha)
{
  cairo_mesh_pattern_set_corner_color_rgba(m_cobject, corner_num, red, green,
                                           blue, alpha);
  check_object_status_and_throw_exception(*this);
}

unsigned int MeshPattern::get_patch_count() const
{
  unsigned int res;
  cairo_mesh_pattern_get_patch_count(m_cobject, &res);
  return res;
}

RefPtr<Path> MeshPattern::get_path(unsigned int patch_num) const
{
  cairo_path_t* cpath = cairo_mesh_pattern_get_path(m_cobject, patch_num);
  auto cpp_path = make_refptr_for_instance<Path>(new Path(cpath, true /* take ownership */));
  // If an exception is thrown, cpp_path's destructor will call ~Path(),
  // which will destroy cpath.
  check_object_status_and_throw_exception(*cpp_path);
  return cpp_path;
}

void MeshPattern::get_control_point(unsigned int patch_num, unsigned int point_num,
                                    double& x, double& y)
{
  cairo_mesh_pattern_get_control_point(m_cobject, patch_num, point_num, &x, &y);
  check_object_status_and_throw_exception(*this);
}

void MeshPattern::get_corner_color_rgba(unsigned int patch_num, unsigned int corner_num,
                                        double& red, double& green, double& blue,
                                        double& alpha)
{
  cairo_mesh_pattern_get_corner_color_rgba(m_cobject, patch_num, corner_num,
                                           &red, &green, &blue, &alpha);
  check_object_status_and_throw_exception(*this);
}

RefPtr<MeshPattern> MeshPattern::create()
{
  cairo_pattern_t* cobject = cairo_pattern_create_mesh();
  auto cpp_object = make_refptr_for_instance<MeshPattern>(new MeshPattern(cobject, true /* has reference */));
  // If an exception is thrown, cpp_object's destructor will call ~MeshPattern(),
  // which will destroy cobject.
  check_object_status_and_throw_exception(*cpp_object);
  return cpp_object;
}

} //namespace Cairo
