/* Copyright (C) 2025 The cairomm Development Team
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, see <https://www.gnu.org/licenses/>.
 */

#include <string>
#include <iostream>
#include <cairomm/context.h>
#include <cairomm/surface.h>
#include <cmath>

int main()
{
#ifdef CAIRO_HAS_PDF_SURFACE

  const std::string filename = "mapped-image.pdf";
  const int width = 600;
  const int height = 400;

  auto pdf_surface = Cairo::PdfSurface::create(filename, width, height);
  {
    auto mapped_image_surface = pdf_surface->map_to_image();  
    auto cr = Cairo::Context::create(mapped_image_surface);

    cr->save(); // save the state of the context
    cr->set_source_rgb(0.86, 0.85, 0.47);
    cr->paint();    // fill image with the color
    cr->restore();  // color is back to black now

    cr->save();
    // draw a border around the image
    cr->set_line_width(20.0);    // make the line wider
    cr->rectangle(0.0, 0.0, width, height);
    cr->stroke();

    cr->set_source_rgba(0.0, 0.0, 0.0, 0.7);
    // draw a circle in the center of the image
    cr->arc(width / 2.0, height / 2.0, 
            height / 4.0, 0.0, 2.0 * M_PI);
    cr->stroke();

    // draw a diagonal line
    cr->move_to(width / 4.0, height / 4.0);
    cr->line_to(width * 3.0 / 4.0, height * 3.0 / 4.0);
    cr->stroke();
    cr->restore();
  }
  // First the Context, then the MappedImageSurface are deleted.
  // When mapped_image_surface is deleted, its content will be uploaded to
  // its target surface, i.e. pdf_surface.

  pdf_surface->show_page();

  std::cout << "Wrote PDF file \"" << filename << "\"" << std::endl;
  return 0;
#else
  std::cout << "You must compile cairo with PDF support for this example to work."
      << std::endl;
  return 1;
#endif
}
