// Copyright 2015 Matthew Holt and The Caddy Authors
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package caddypki

import (
	"crypto/x509"
	"fmt"
	"log"
	"runtime/debug"
	"time"

	"go.uber.org/zap"
)

func (p *PKI) maintenanceForCA(ca *CA) {
	defer func() {
		if err := recover(); err != nil {
			log.Printf("[PANIC] PKI maintenance for CA %s: %v\n%s", ca.ID, err, debug.Stack())
		}
	}()

	interval := time.Duration(ca.MaintenanceInterval)
	if interval <= 0 {
		interval = defaultMaintenanceInterval
	}
	ticker := time.NewTicker(interval)
	defer ticker.Stop()

	for {
		select {
		case <-ticker.C:
			_ = p.renewCertsForCA(ca)
		case <-p.ctx.Done():
			return
		}
	}
}

func (p *PKI) renewCerts() {
	for _, ca := range p.CAs {
		err := p.renewCertsForCA(ca)
		if err != nil {
			p.log.Error("renewing intermediate certificates",
				zap.Error(err),
				zap.String("ca", ca.ID))
		}
	}
}

func (p *PKI) renewCertsForCA(ca *CA) error {
	ca.mu.Lock()
	defer ca.mu.Unlock()

	log := p.log.With(zap.String("ca", ca.ID))

	// only maintain the root if it's not manually provided in the config
	if ca.Root == nil {
		if ca.needsRenewal(ca.root) {
			// TODO: implement root renewal (use same key)
			log.Warn("root certificate expiring soon (FIXME: ROOT RENEWAL NOT YET IMPLEMENTED)",
				zap.Duration("time_remaining", time.Until(ca.interChain[0].NotAfter)),
			)
		}
	}

	// only maintain the intermediate if it's not manually provided in the config
	if ca.Intermediate == nil {
		if ca.needsRenewal(ca.interChain[0]) {
			log.Info("intermediate expires soon; renewing",
				zap.Duration("time_remaining", time.Until(ca.interChain[0].NotAfter)),
			)

			rootCert, rootKey, err := ca.loadOrGenRoot()
			if err != nil {
				return fmt.Errorf("loading root key: %v", err)
			}
			interCert, interKey, err := ca.genIntermediate(rootCert, rootKey)
			if err != nil {
				return fmt.Errorf("generating new certificate: %v", err)
			}
			ca.interChain, ca.interKey = []*x509.Certificate{interCert}, interKey

			log.Info("renewed intermediate",
				zap.Time("new_expiration", ca.interChain[0].NotAfter),
			)
		}
	}

	return nil
}

// needsRenewal reports whether the certificate is within its renewal window
// (i.e. the fraction of lifetime remaining is less than or equal to RenewalWindowRatio).
func (ca *CA) needsRenewal(cert *x509.Certificate) bool {
	ratio := ca.RenewalWindowRatio
	if ratio <= 0 {
		ratio = defaultRenewalWindowRatio
	}
	lifetime := cert.NotAfter.Sub(cert.NotBefore)
	renewalWindow := time.Duration(float64(lifetime) * ratio)
	renewalWindowStart := cert.NotAfter.Add(-renewalWindow)
	return time.Now().After(renewalWindowStart)
}
