// Package fastcdc implements the [FastCDC] algorithm for content-defined
// chunking.
//
// [FastCDC]: https://www.usenix.org/conference/atc16/technical-sessions/presentation/xia
package fastcdc

import (
	"errors"
	"io"
	"math/bits"
)

const (
	// theoretical upper bound for the average chunk size
	maxAvgSize = 1 << 61

	// parameters from the original FastCDC paper
	refMinSize = 2048
	refMaxSize = 65536
	refAvgSize = 8192
	refMaskS   = 0x0003590703530000
	refMaskL   = 0x0000d90003530000
)

// ErrParam is returned by NewChunker when one or more of the size parameters
// are invalid.
var ErrParam = errors.New("invalid parameter")

// Chunker wraps an io.Reader and returns data in chunks of bytes.
// Successive calls to Next will split the input data into variable sized
// chunks according to the FastCDC algorithm. The Chunk method returns the
// current chunk. The process stops at EOF or the first I/O error. Use the Err
// method to see if a non-EOF error occurred.
type Chunker struct {
	// Wrapped reader
	r io.Reader

	// Sticky error
	err error

	// Buffer used to read from r
	buf []byte

	// start and end of the last chunk found by Next()
	cbegin, cend int

	// number of valid bytes in buf
	valid int

	// FastCDC parameters
	avgSize, minSize int
	maskS, maskL     uint64
}

// NewRefChunker returns a new chunker that wraps r and uses the reference
// parameters for FastCDC. The minimal, maximal and average sizes of chunks
// returned by this chunker are 2KB, 64KB and 8KB respectively.
func NewRefChunker(r io.Reader) *Chunker {
	var c Chunker

	c.r = r
	c.buf = make([]byte, refMaxSize)
	c.minSize = refMinSize
	c.avgSize = refAvgSize
	c.maskS = refMaskS
	c.maskL = refMaskL

	return &c
}

// NewChunker returns a new chunker that wraps r and uses the given parameters
// for FastCDC. minSize is the minimum size, in bytes, of chunks, that the
// Chunker should return. Likewise, avgSize and maxSize set the average
// (expected) and maximum size of chunks in bytes. avgSize must be a power
// of two and the parameters must be ordered so that
// 1 <= minSize <= avgSize <= maxSize.
func NewChunker(r io.Reader, minSize, avgSize, maxSize int) (*Chunker, error) {
	if minSize < 1 || minSize > avgSize || avgSize > maxSize || avgSize > maxAvgSize {
		return nil, ErrParam
	}

	// avgSize must be a power of 2
	if (avgSize & (avgSize - 1)) != 0 {
		return nil, ErrParam
	}

	var c Chunker
	c.r = r
	c.buf = make([]byte, maxSize)
	c.minSize = minSize
	c.avgSize = avgSize

	var avgbits, sbits, lbits uint64

	avgbits = 64 - uint64(bits.LeadingZeros(uint(avgSize-1)))
	sbits = avgbits + 2
	lbits = avgbits - 2

	c.maskS = ((1 << sbits) - 1) << (64 - sbits)
	c.maskL = ((1 << lbits) - 1) << (64 - lbits)

	return &c, nil
}

// Err returns the first non-EOF error that was encountered by the Chunker.
func (c *Chunker) Err() error {
	if c.err == io.EOF {
		return nil
	}

	return c.err
}

// Chunk returns the most recent chunk of data generated by a call to Next.
// The underlying array points to data that might be overwritten by subsequent
// calls to Next.
func (c *Chunker) Chunk() []byte {
	return c.buf[c.cbegin:c.cend]
}

// Next retrieves the next chunk of data, which will then be available through
// the Chunk method. It returns false when the chunking process stops, either
// by reaching the end of the input stream or an error. After Next returns
// false, the Err method will return any error that occurred during scanning,
// except that if it was io.EOF, Err will return nil.
func (c *Chunker) Next() bool {
	// If there is unprocessed data left, try to find a boundary therein
	if c.cend < c.valid {
		n := c.findBoundary(c.buf[c.cend:c.valid])

		if n > 0 {
			c.cbegin = c.cend
			c.cend += n
			return true
		}
	}

	// More data is needed. Copy the leftovers to the beginning
	copy(c.buf, c.buf[c.cend:c.valid])
	c.valid -= c.cend
	c.cbegin = 0
	c.cend = 0

	// Fill as much of the buffer as possible, unless c.err != nil
	c.fillBuffer()

	// Chunking is done when there is no more data to process
	if c.valid == c.cend {
		return false
	}

	// Skip chunking if there is less than the minimum size of data left
	if c.valid <= c.minSize {
		c.cend = c.valid
		return true
	}

	// Use FastCDC to find the boundary of the next chunk
	n := c.findBoundary(c.buf[:c.valid])

	// Return the complete buffer if no boundary was found
	if n == 0 {
		n = c.valid
	}

	c.cend = n
	return true
}

func (c *Chunker) fillBuffer() {
	for c.err == nil && c.valid < len(c.buf) {
		n, err := c.r.Read(c.buf[c.valid:])
		c.valid += n
		c.err = err
	}
}

func (c *Chunker) findBoundary(data []byte) int {
	avgSize := c.avgSize

	if len(data) < c.avgSize {
		avgSize = len(data)
	}

	var fp uint64

	for i := c.minSize; i < avgSize; i++ {
		fp = (fp << 1) + gear[data[i]]

		if (fp & c.maskS) == 0 {
			return i
		}
	}

	for i := avgSize; i < len(data); i++ {
		fp = (fp << 1) + gear[data[i]]

		if (fp & c.maskL) == 0 {
			return i
		}
	}

	return 0
}
