package zabbix_test

import (
	"encoding/binary"
	"encoding/json"
	"errors"
	"fmt"
	"net"
	"testing"
	"time"

	zabbix "github.com/datadope-io/go-zabbix/v2"
)

type ZabbixRequestData struct {
	Host  string `json:"host"`
	Key   string `json:"key"`
	Value string `json:"value"`
	Clock int64  `json:"clock"`
}

type ZabbixRequest struct {
	Request      string              `json:"request"`
	Data         []ZabbixRequestData `json:"data"`
	Clock        int                 `json:"clock"`
	Host         string              `json:"host"`
	HostMetadata string              `json:"host_metadata"`
}

func TestSendActiveMetric(t *testing.T) {
	zabbixAddress := "127.0.0.1:10051"

	// Simulate a Zabbix server to get the data sent
	listener, lerr := net.Listen("tcp", zabbixAddress)
	if lerr != nil {
		t.Fatal(lerr)
	}
	defer listener.Close()

	errs := make(chan error, 1)

	go func(chan error) {
		conn, err := listener.Accept()
		if err != nil {
			errs <- err
		}

		// Obtain request from the mock zabbix server
		// Read protocol header and version
		header := make([]byte, 5)
		_, err = conn.Read(header)
		if err != nil {
			errs <- err
		}

		// Read data length
		dataLengthRaw := make([]byte, 8)
		_, err = conn.Read(dataLengthRaw)
		if err != nil {
			errs <- err
		}

		dataLength := binary.LittleEndian.Uint64(dataLengthRaw)

		// Read data content
		content := make([]byte, dataLength)
		_, err = conn.Read(content)
		if err != nil {
			errs <- err
		}

		// The zabbix output checks that there are not errors
		// Zabbix header length not used, set to 1
		resp := []byte("ZBXD\x01\x00\x00\x00\x00\x00\x00\x00\x00{\"response\":\"success\",\"info\":\"processed: 1; failed: 0; total: 1; seconds spent: 0.000030\"}")
		_, err = conn.Write(resp)
		if err != nil {
			errs <- err
		}

		// Close connection after reading the client data
		conn.Close()

		// Strip zabbix header and get JSON request
		var request ZabbixRequest
		err = json.Unmarshal(content, &request)
		if err != nil {
			errs <- err
		}

		expectedRequest := "agent data"
		if expectedRequest != request.Request {
			errs <- fmt.Errorf("Incorrect request field received, expected '%s'", expectedRequest)
		}

		// End zabbix fake backend
		errs <- nil
	}(errs)

	m := zabbix.NewMetric("zabbixAgent1", "ping", "13", true)

	s := zabbix.NewSender(zabbixAddress)
	resActive, resTrapper, err := s.SendMetrics([]*zabbix.Metric{m})
	if errors.Is(err, zabbix.ErrActive) {
		t.Fatalf("error sending active metric: %v", err)
	}
	if errors.Is(err, zabbix.ErrTrapper) {
		t.Fatalf("trapper error should be nil, we are not sending trapper metrics: %v", err)
	}

	raInfo, err := resActive.GetInfo()
	if err != nil {
		t.Fatalf("error in response Trapper: %v", err)
	}

	if raInfo.Failed != 0 {
		t.Errorf("Failed error expected 0 got %d", raInfo.Failed)
	}
	if raInfo.Processed != 1 {
		t.Errorf("Processed error expected 1 got %d", raInfo.Processed)
	}
	if raInfo.Total != 1 {
		t.Errorf("Total error expected 1 got %d", raInfo.Total)
	}

	_, err = resTrapper.GetInfo()
	if err == nil {
		t.Fatalf("No response trapper expected: %v", err)
	}

	// Wait for zabbix server emulator to finish
	err = <-errs
	if err != nil {
		t.Fatalf("Fake zabbix backend should not produce any errors: %v", err)
	}
}

func TestSendTrapperMetric(t *testing.T) {
	zabbixAddress := "127.0.0.1:10051"

	// Simulate a Zabbix server to get the data sent
	listener, lerr := net.Listen("tcp", zabbixAddress)
	if lerr != nil {
		t.Fatal(lerr)
	}
	defer listener.Close()

	errs := make(chan error, 1)

	go func(chan error) {
		conn, err := listener.Accept()
		if err != nil {
			errs <- err
		}

		// Obtain request from the mock zabbix server
		// Read protocol header and version
		header := make([]byte, 5)
		_, err = conn.Read(header)
		if err != nil {
			errs <- err
		}

		// Read data length
		dataLengthRaw := make([]byte, 8)
		_, err = conn.Read(dataLengthRaw)
		if err != nil {
			errs <- err
		}

		dataLength := binary.LittleEndian.Uint64(dataLengthRaw)

		// Read data content
		content := make([]byte, dataLength)
		_, err = conn.Read(content)
		if err != nil {
			errs <- err
		}

		// The zabbix output checks that there are not errors
		resp := []byte("ZBXD\x01\x00\x00\x00\x00\x00\x00\x00\x00{\"response\":\"success\",\"info\":\"processed: 1; failed: 0; total: 1; seconds spent: 0.000030\"}")
		_, err = conn.Write(resp)
		if err != nil {
			errs <- err
		}

		// Close connection after reading the client data
		conn.Close()

		// Strip zabbix header and get JSON request
		var request ZabbixRequest
		err = json.Unmarshal(content, &request)
		if err != nil {
			errs <- err
		}

		expectedRequest := "sender data"
		if expectedRequest != request.Request {
			errs <- fmt.Errorf("Incorrect request field received, expected '%s'", expectedRequest)
		}

		// End zabbix fake backend
		errs <- nil
	}(errs)

	m := zabbix.NewMetric("zabbixAgent1", "ping", "13", false)

	s := zabbix.NewSender(zabbixAddress)
	resActive, resTrapper, err := s.SendMetrics([]*zabbix.Metric{m})
	if errors.Is(err, zabbix.ErrTrapper) {
		t.Fatalf("error sending trapper metric: %v", err)
	}
	if errors.Is(err, zabbix.ErrActive) {
		t.Fatalf("active error should be nil, we are not sending zabbix agent metrics: %v", err)
	}

	rtInfo, err := resTrapper.GetInfo()
	if err != nil {
		t.Fatalf("error in response Trapper: %v", err)
	}

	if rtInfo.Failed != 0 {
		t.Errorf("Failed error expected 0 got %d", rtInfo.Failed)
	}
	if rtInfo.Processed != 1 {
		t.Errorf("Processed error expected 1 got %d", rtInfo.Processed)
	}
	if rtInfo.Total != 1 {
		t.Errorf("Total error expected 1 got %d", rtInfo.Total)
	}

	_, err = resActive.GetInfo()
	if err == nil {
		t.Fatalf("No response active expected: %v", err)
	}

	// Wait for zabbix server emulator to finish
	err = <-errs
	if err != nil {
		t.Fatalf("Fake zabbix backend should not produce any errors: %v", err)
	}
}

func TestSendActiveAndTrapperMetric(t *testing.T) {
	zabbixAddress := "127.0.0.1:10051"

	// Simulate a Zabbix server to get the data sent
	listener, lerr := net.Listen("tcp", zabbixAddress)
	if lerr != nil {
		t.Fatal(lerr)
	}
	defer listener.Close()

	errs := make(chan error, 1)

	go func(chan error) {
		for i := 0; i < 2; i++ {
			conn, err := listener.Accept()
			if err != nil {
				errs <- err
			}

			// Obtain request from the mock zabbix server
			// Read protocol header and version
			header := make([]byte, 5)
			_, err = conn.Read(header)
			if err != nil {
				errs <- err
			}

			// Read data length
			dataLengthRaw := make([]byte, 8)
			_, err = conn.Read(dataLengthRaw)
			if err != nil {
				errs <- err
			}

			dataLength := binary.LittleEndian.Uint64(dataLengthRaw)

			// Read data content
			content := make([]byte, dataLength)
			_, err = conn.Read(content)
			if err != nil {
				errs <- err
			}

			// Strip zabbix header and get JSON request
			var request ZabbixRequest
			err = json.Unmarshal(content, &request)
			if err != nil {
				errs <- err
			}

			resp := []byte("")

			if request.Request == "sender data" {
				resp = []byte("ZBXD\x01\x00\x00\x00\x00\x00\x00\x00\x00{\"response\":\"success\",\"info\":\"processed: 1; failed: 0; total: 1; seconds spent: 0.000030\"}")
			} else if request.Request == "agent data" {
				resp = []byte("ZBXD\x01\x00\x00\x00\x00\x00\x00\x00\x00{\"response\":\"success\",\"info\":\"processed: 1; failed: 0; total: 1; seconds spent: 0.111111\"}")
			}

			// The zabbix output checks that there are not errors
			_, err = conn.Write(resp)
			if err != nil {
				errs <- err
			}

			// Close connection after reading the client data
			conn.Close()
		}

		// End zabbix fake backend
		errs <- nil
	}(errs)

	m := zabbix.NewMetric("zabbixAgent1", "ping", "13", true)
	m2 := zabbix.NewMetric("zabbixTrapper1", "pong", "13", false)

	s := zabbix.NewSender(zabbixAddress)
	resActive, resTrapper, err := s.SendMetrics([]*zabbix.Metric{m, m2})

	if errors.Is(err, zabbix.ErrActive) {
		t.Fatalf("error sending active metric: %v", err)
	}
	if errors.Is(err, zabbix.ErrTrapper) {
		t.Fatalf("error sending trapper metric: %v", err)
	}

	raInfo, err := resActive.GetInfo()
	if err != nil {
		t.Fatalf("error in response Trapper: %v", err)
	}

	if raInfo.Failed != 0 {
		t.Errorf("Failed error expected 0 got %d", raInfo.Failed)
	}
	if raInfo.Processed != 1 {
		t.Errorf("Processed error expected 1 got %d", raInfo.Processed)
	}
	if raInfo.Total != 1 {
		t.Errorf("Total error expected 1 got %d", raInfo.Total)
	}

	rtInfo, err := resTrapper.GetInfo()
	if err != nil {
		t.Fatalf("error in response Trapper: %v", err)
	}

	if rtInfo.Failed != 0 {
		t.Errorf("Failed error expected 0 got %d", rtInfo.Failed)
	}
	if rtInfo.Processed != 1 {
		t.Errorf("Processed error expected 1 got %d", rtInfo.Processed)
	}
	if rtInfo.Total != 1 {
		t.Errorf("Total error expected 1 got %d", rtInfo.Total)
	}

	// Wait for zabbix server emulator to finish
	err = <-errs
	if err != nil {
		t.Fatalf("Fake zabbix backend should not produce any errors: %v", err)
	}
}

func TestSendActiveAndTrapperMetricErrors(t *testing.T) {
	zabbixAddress := "127.0.0.1:10051"

	m := zabbix.NewMetric("zabbixAgent1", "ping", "13", true)
	m2 := zabbix.NewMetric("zabbixTrapper1", "pong", "13", false)

	s := zabbix.NewSender(zabbixAddress)
	_, _, err := s.SendMetrics([]*zabbix.Metric{m, m2})
	if !errors.Is(err, zabbix.ErrActive) {
		t.Fatalf("An active error should be present: %v", err)
	}
	if !errors.Is(err, zabbix.ErrTrapper) {
		t.Fatalf("A trapper error should be present: %v", err)
	}
}

func TestRegisterHostOK(t *testing.T) {
	zabbixAddress := "127.0.0.1:10051"

	// Simulate a Zabbix server to get the data sent
	listener, lerr := net.Listen("tcp", zabbixAddress)
	if lerr != nil {
		t.Fatal(lerr)
	}
	defer listener.Close()

	errs := make(chan error, 1)

	go func(chan error) {
		for i := 0; i < 2; i++ {
			conn, err := listener.Accept()
			if err != nil {
				errs <- err
			}

			// Obtain request from the mock zabbix server
			// Read protocol header and version
			header := make([]byte, 5)
			_, err = conn.Read(header)
			if err != nil {
				errs <- err
			}

			// Read data length
			dataLengthRaw := make([]byte, 8)
			_, err = conn.Read(dataLengthRaw)
			if err != nil {
				errs <- err
			}

			dataLength := binary.LittleEndian.Uint64(dataLengthRaw)

			// Read data content
			content := make([]byte, dataLength)
			_, err = conn.Read(content)
			if err != nil {
				errs <- err
			}

			// Strip zabbix header and get JSON request
			var request ZabbixRequest
			err = json.Unmarshal(content, &request)
			if err != nil {
				errs <- err
			}

			expectedRequest := "active checks"
			if expectedRequest != request.Request {
				errs <- fmt.Errorf("Incorrect request field received, expected '%s'", expectedRequest)
			}

			// If the host does not exist, the first response will be an error
			resp := []byte("ZBXD\x01\x00\x00\x00\x00\x00\x00\x00\x00{\"response\":\"failed\",\"info\": \"host [prueba] not found\"}")

			// Next response is the valid one
			if i == 1 {
				resp = []byte("ZBXD\x01\x00\x00\x00\x00\x00\x00\x00\x00{\"response\":\"success\",\"data\": [{\"key\":\"net.if.in[eth0]\",\"delay\":60,\"lastlogsize\":0,\"mtime\":0}]}")
			}

			_, err = conn.Write(resp)
			if err != nil {
				errs <- err
			}

			// Close connection after reading the client data
			conn.Close()
		}

		// End zabbix fake backend
		errs <- nil
	}(errs)

	s := zabbix.NewSender(zabbixAddress)
	err := s.RegisterHost("prueba", "prueba")
	if err != nil {
		t.Fatalf("register host error: %v", err)
	}

	// Wait for zabbix server emulator to finish
	err = <-errs
	if err != nil {
		t.Fatalf("Fake zabbix backend should not produce any errors: %v", err)
	}
}

func TestRegisterHostError(t *testing.T) {
	zabbixAddress := "127.0.0.1:10051"

	// Simulate a Zabbix server to get the data sent
	listener, lerr := net.Listen("tcp", zabbixAddress)
	if lerr != nil {
		t.Fatal(lerr)
	}
	defer listener.Close()

	errs := make(chan error, 1)

	go func(chan error) {
		for i := 0; i < 2; i++ {
			conn, err := listener.Accept()
			if err != nil {
				errs <- err
			}

			// Obtain request from the mock zabbix server
			// Read protocol header and version
			header := make([]byte, 5)
			_, err = conn.Read(header)
			if err != nil {
				errs <- err
			}

			// Read data length
			dataLengthRaw := make([]byte, 8)
			_, err = conn.Read(dataLengthRaw)
			if err != nil {
				errs <- err
			}

			dataLength := binary.LittleEndian.Uint64(dataLengthRaw)

			// Read data content
			content := make([]byte, dataLength)
			_, err = conn.Read(content)
			if err != nil {
				errs <- err
			}

			// Strip zabbix header and get JSON request
			var request ZabbixRequest
			err = json.Unmarshal(content, &request)
			if err != nil {
				errs <- err
			}

			expectedRequest := "active checks"
			if expectedRequest != request.Request {
				errs <- fmt.Errorf("Incorrect request field received, expected '%s'", expectedRequest)
			}

			// Simulate error always
			resp := []byte("ZBXD\x01\x00\x00\x00\x00\x00\x00\x00\x00{\"response\":\"failed\",\"info\": \"host [prueba] not found\"}")
			_, err = conn.Write(resp)
			if err != nil {
				errs <- err
			}

			// Close connection after reading the client data
			conn.Close()
		}

		// End zabbix fake backend
		errs <- nil
	}(errs)

	s := zabbix.NewSender(zabbixAddress)
	err := s.RegisterHost("prueba", "prueba")
	if err == nil {
		t.Fatalf("should return an error: %v", err)
	}

	// Wait for zabbix server emulator to finish
	err = <-errs
	if err != nil {
		t.Fatalf("Fake zabbix backend should not produce any errors: %v", err)
	}
}

func TestInvalidResponseHeader(t *testing.T) {
	zabbixAddress := "127.0.0.1:10051"

	// Simulate a Zabbix server to get the data sent
	listener, lerr := net.Listen("tcp", zabbixAddress)
	if lerr != nil {
		t.Fatal(lerr)
	}
	defer listener.Close()

	errs := make(chan error, 1)

	go func(chan error) {
		conn, err := listener.Accept()
		if err != nil {
			errs <- err
		}

		// Obtain request from the mock zabbix server
		// Read protocol header and version
		header := make([]byte, 5)
		_, err = conn.Read(header)
		if err != nil {
			errs <- err
		}

		// Read data length
		dataLengthRaw := make([]byte, 8)
		_, err = conn.Read(dataLengthRaw)
		if err != nil {
			errs <- err
		}

		dataLength := binary.LittleEndian.Uint64(dataLengthRaw)

		// Read data content
		content := make([]byte, dataLength)
		_, err = conn.Read(content)
		if err != nil {
			errs <- err
		}

		// The zabbix output checks that there are not errors
		// Zabbix header length not used, set to 1
		resp := []byte("BXD\x01\x00\x00\x00\x00\x00\x00\x00\x00{\"response\":\"success\",\"info\":\"processed: 1; failed: 0; total: 1; seconds spent: 0.000030\"}")
		_, err = conn.Write(resp)
		if err != nil {
			errs <- err
		}

		// Close connection after reading the client data
		conn.Close()

		// Strip zabbix header and get JSON request
		var request ZabbixRequest
		err = json.Unmarshal(content, &request)
		if err != nil {
			errs <- err
		}

		expectedRequest := "agent data"
		if expectedRequest != request.Request {
			errs <- fmt.Errorf("Incorrect request field received, expected '%s'", expectedRequest)
		}

		// End zabbix fake backend
		errs <- nil
	}(errs)

	m := zabbix.NewMetric("zabbixAgent1", "ping", "13", true)

	s := zabbix.NewSender(zabbixAddress)
	_, _, err := s.SendMetrics([]*zabbix.Metric{m})
	if err == nil {
		t.Fatal("Expected an error because an incorrect Zabbix protocol header")
	} else {
		t.Logf("Expected error: %v", err)
	}

	// Wait for zabbix server emulator to finish
	err = <-errs
	if err != nil {
		t.Fatalf("Fake zabbix backend should not produce any errors: %v", err)
	}
}

func TestGetInfoValidData(t *testing.T) {
	r := zabbix.Response{
		Response: "success",
		Info:     "processed: 1; failed: 0; total: 1; seconds spent: 0.0030",
	}

	ri, err := r.GetInfo()
	if err != nil {
		t.Fatalf("Error getting info: %v", err)
	}

	if ri.Processed != 1 {
		t.Fatalf("Incorrect processed value: %d", ri.Processed)
	}

	if ri.Failed != 0 {
		t.Fatalf("Incorrect failed value: %d", ri.Failed)
	}

	if ri.Total != 1 {
		t.Fatalf("Incorrect total value: %d", ri.Total)
	}

	if ri.Spent != time.Duration(3*time.Millisecond) {
		t.Fatalf("Incorrect seconds spent value: %v", ri.Spent)
	}
}

func TestGetInfoFailed(t *testing.T) {
	r := zabbix.Response{
		Response: "failed",
		Info:     "host [prueba] not found",
	}

	_, err := r.GetInfo()
	if err == nil {
		t.Fatalf("Should return an error as response is failed")
	}
}

func TestGetInfoInvalidData(t *testing.T) {
	r := zabbix.Response{
		Response: "success",
		Info:     "processed: 1; failed: 0; total: 1",
	}

	_, err := r.GetInfo()
	if err == nil {
		t.Fatalf("Should return an error as info is not valid")
	}
}

func TestGetInfoInvalidDataProcessedValue(t *testing.T) {
	r := zabbix.Response{
		Response: "success",
		Info:     "processed: a; failed: 0; total: 1; seconds spent: 0.0030",
	}

	_, err := r.GetInfo()
	if err == nil {
		t.Fatalf("Should return an error as processed value in info is not valid")
	}
}

func TestGetInfoInvalidDataFailedValue(t *testing.T) {
	r := zabbix.Response{
		Response: "success",
		Info:     "processed: 1; failed: a; total: 1; seconds spent: 0.0030",
	}

	_, err := r.GetInfo()
	if err == nil {
		t.Fatalf("Should return an error as failed value in info is not valid")
	}
}

func TestGetInfoInvalidDataTotalValue(t *testing.T) {
	r := zabbix.Response{
		Response: "success",
		Info:     "processed: 1; failed: 0; total: a; seconds spent: 0.0030",
	}

	_, err := r.GetInfo()
	if err == nil {
		t.Fatalf("Should return an error as total value in info is not valid")
	}
}

func TestGetInfoInvalidDataSpentValue(t *testing.T) {
	r := zabbix.Response{
		Response: "success",
		Info:     "processed: 1; failed: 0; total: 1; seconds spent: a",
	}

	_, err := r.GetInfo()
	if err == nil {
		t.Fatalf("Should return an error as seconds spent value in info is not valid")
	}
}
