package gokeepasslib

import (
	"bytes"
	"testing"
)

func TestNewBlockHMACBuilder(t *testing.T) {
	cases := []struct {
		title           string
		masterSeed      [32]byte
		transformedKey  [32]byte
		expectedBaseKey []byte
	}{
		{
			title:          "empty init data",
			masterSeed:     [32]byte{},
			transformedKey: [32]byte{},
			expectedBaseKey: []byte{
				0xbf, 0xb1, 0xcd, 0x67, 0x77, 0xff, 0x2b, 0x35,
				0x65, 0x62, 0x33, 0xa7, 0x0f, 0xe8, 0x67, 0x04,
				0x6a, 0xae, 0x70, 0x7d, 0x4f, 0x90, 0x12, 0x1a,
				0x5e, 0x6c, 0xda, 0x3c, 0x2e, 0xa4, 0x50, 0x83,
				0xc0, 0x5b, 0xfb, 0x38, 0x7e, 0xb1, 0xc4, 0x95,
				0xdf, 0xf1, 0x5b, 0xff, 0xac, 0x38, 0x1a, 0xd7,
				0x87, 0x77, 0x1e, 0xf8, 0x40, 0x2c, 0x36, 0x38,
				0xbf, 0x5f, 0x4c, 0x9d, 0xd4, 0x33, 0xa5, 0x7b,
			},
		},
		{
			title: "example data",
			masterSeed: [32]byte{
				0x19, 0x5e, 0x97, 0xa2, 0x36, 0xe8, 0x04, 0x1d,
				0x45, 0x17, 0xee, 0x81, 0xdb, 0x99, 0x9b, 0xf8,
				0xd7, 0x48, 0xbc, 0x03, 0x35, 0x66, 0x3a, 0xb3,
				0x89, 0x28, 0x86, 0xea, 0xb0, 0x08, 0x0a, 0x3d,
			},
			transformedKey: [32]byte{
				0x5f, 0xbd, 0x2e, 0xd3, 0x5b, 0x90, 0xde, 0x89,
				0x73, 0xdd, 0xfc, 0xde, 0xbe, 0x2c, 0xef, 0x9d,
				0x52, 0x95, 0x61, 0xc1, 0x92, 0xb4, 0x00, 0xe4,
				0xcb, 0x07, 0x75, 0x18, 0x76, 0x79, 0xfc, 0x3b,
			},
			expectedBaseKey: []byte{
				0x2f, 0x7f, 0xe3, 0x82, 0x3d, 0x2b, 0x3b, 0xc0,
				0xc8, 0xf5, 0xf9, 0x09, 0xeb, 0xdc, 0xa2, 0x6a,
				0x9c, 0x1c, 0x57, 0xb6, 0xa2, 0x76, 0x4c, 0xc6,
				0xab, 0x16, 0x79, 0xbd, 0xfa, 0xd1, 0x7b, 0x6f,
				0x86, 0x43, 0x0b, 0x11, 0x8b, 0x7b, 0x7b, 0x9c,
				0xa3, 0x36, 0xd8, 0xe0, 0xf5, 0x73, 0x58, 0xc2,
				0x2d, 0x10, 0x2a, 0xdc, 0x9e, 0xd8, 0xde, 0xae,
				0xcf, 0xd9, 0x96, 0xbc, 0xe3, 0xd0, 0xaa, 0xc9,
			},
		},
	}

	for _, c := range cases {
		t.Run(c.title, func(t *testing.T) {
			builder := NewBlockHMACBuilder(c.masterSeed[:], c.transformedKey[:])

			if !bytes.Equal(builder.baseKey, c.expectedBaseKey) {
				t.Errorf(
					"Received HMAC '% x', expected '% x'",
					builder.baseKey,
					c.expectedBaseKey,
				)
			}
		})
	}
}

// Note: the seed, key and data here are just examplary and mainly serve the purpose of
// making sure that any future refactorings do not break anything
func TestBlockHMACBuilderBuildHMAC(t *testing.T) {
	masterSeed := [32]byte{
		0x19, 0x5e, 0x97, 0xa2, 0x36, 0xe8, 0x04, 0x1d,
		0x45, 0x17, 0xee, 0x81, 0xdb, 0x99, 0x9b, 0xf8,
		0xd7, 0x48, 0xbc, 0x03, 0x35, 0x66, 0x3a, 0xb3,
		0x89, 0x28, 0x86, 0xea, 0xb0, 0x08, 0x0a, 0x3d,
	}

	transformedKey := [32]byte{
		0x5f, 0xbd, 0x2e, 0xd3, 0x5b, 0x90, 0xde, 0x89,
		0x73, 0xdd, 0xfc, 0xde, 0xbe, 0x2c, 0xef, 0x9d,
		0x52, 0x95, 0x61, 0xc1, 0x92, 0xb4, 0x00, 0xe4,
		0xcb, 0x07, 0x75, 0x18, 0x76, 0x79, 0xfc, 0x3b,
	}

	cases := []struct {
		title          string
		masterSeed     [32]byte
		transformedKey [32]byte
		index          uint64
		length         uint32
		data           []byte
		expectedHMAC   []byte
	}{
		{
			title:          "build hmac for index 1 with length 100",
			masterSeed:     masterSeed,
			transformedKey: transformedKey,
			index:          1,
			length:         100,
			data: []byte{
				0x94, 0xee, 0xb5, 0xa5, 0x29, 0x14, 0x16, 0x28, 0x9e, 0x17,
				0x5a, 0x46, 0x53, 0x53, 0x8d, 0x5a, 0xc9, 0x73, 0x0d, 0xdc,
				0x0d, 0xc6, 0x5c, 0xe5, 0xd6, 0x10, 0xb4, 0x28, 0x1f, 0xb9,
				0x1d, 0x77, 0x00, 0x2f, 0x1d, 0x98, 0xa0, 0x22, 0x30, 0xba,
				0xcc, 0x36, 0x9f, 0x60, 0x57, 0x4e, 0xa0, 0xb4, 0x8d, 0x79,
				0xf5, 0x2c, 0x23, 0x98, 0x4f, 0xfa, 0xcb, 0x55, 0x01, 0x2c,
				0xc3, 0x50, 0x87, 0xe3, 0x83, 0x70, 0xa5, 0x57, 0x55, 0x41,
				0x5c, 0x7d, 0x77, 0x52, 0x17, 0x08, 0x21, 0xb3, 0x52, 0x79,
				0x50, 0xe0, 0xfa, 0xc7, 0xca, 0xee, 0x4b, 0x03, 0x09, 0x95,
				0x3b, 0x13, 0xe6, 0xd7, 0xbf, 0xba, 0x3b, 0xbe, 0xdc, 0x6a,
			},
			expectedHMAC: []byte{
				0xcf, 0x35, 0xfc, 0xb0, 0x5f, 0x02, 0xc6, 0xaf,
				0x13, 0x1b, 0x5c, 0x80, 0xfe, 0xf6, 0x8f, 0xee,
				0x54, 0xe1, 0x68, 0xba, 0xb1, 0xd4, 0x3a, 0xa8,
				0x8c, 0x09, 0x96, 0xa8, 0x98, 0xad, 0xab, 0x01,
			},
		},
		{
			title:          "build hmac for index 2 with length 100",
			masterSeed:     masterSeed,
			transformedKey: transformedKey,
			index:          2,
			length:         100,
			data: []byte{
				0x94, 0xee, 0xb5, 0xa5, 0x29, 0x14, 0x16, 0x28, 0x9e, 0x17,
				0x5a, 0x46, 0x53, 0x53, 0x8d, 0x5a, 0xc9, 0x73, 0x0d, 0xdc,
				0x0d, 0xc6, 0x5c, 0xe5, 0xd6, 0x10, 0xb4, 0x28, 0x1f, 0xb9,
				0x1d, 0x77, 0x00, 0x2f, 0x1d, 0x98, 0xa0, 0x22, 0x30, 0xba,
				0xcc, 0x36, 0x9f, 0x60, 0x57, 0x4e, 0xa0, 0xb4, 0x8d, 0x79,
				0xf5, 0x2c, 0x23, 0x98, 0x4f, 0xfa, 0xcb, 0x55, 0x01, 0x2c,
				0xc3, 0x50, 0x87, 0xe3, 0x83, 0x70, 0xa5, 0x57, 0x55, 0x41,
				0x5c, 0x7d, 0x77, 0x52, 0x17, 0x08, 0x21, 0xb3, 0x52, 0x79,
				0x50, 0xe0, 0xfa, 0xc7, 0xca, 0xee, 0x4b, 0x03, 0x09, 0x95,
				0x3b, 0x13, 0xe6, 0xd7, 0xbf, 0xba, 0x3b, 0xbe, 0xdc, 0x6a,
			},
			expectedHMAC: []byte{
				0xa5, 0x79, 0x78, 0x66, 0xd5, 0xd0, 0x5d, 0x2f,
				0xfe, 0x89, 0xf3, 0xce, 0xcc, 0x33, 0x07, 0x0b,
				0xab, 0xad, 0xf8, 0x4a, 0xaf, 0xa2, 0x6a, 0x14,
				0x52, 0x14, 0xff, 0x65, 0x22, 0xd8, 0x56, 0x89,
			},
		},
		{
			title:          "build hmac for index 1 with length 50",
			masterSeed:     masterSeed,
			transformedKey: transformedKey,
			index:          2,
			length:         50,
			data: []byte{
				0x94, 0xee, 0xb5, 0xa5, 0x29, 0x14, 0x16, 0x28, 0x9e, 0x17,
				0x5a, 0x46, 0x53, 0x53, 0x8d, 0x5a, 0xc9, 0x73, 0x0d, 0xdc,
				0x0d, 0xc6, 0x5c, 0xe5, 0xd6, 0x10, 0xb4, 0x28, 0x1f, 0xb9,
				0x1d, 0x77, 0x00, 0x2f, 0x1d, 0x98, 0xa0, 0x22, 0x30, 0xba,
				0xcc, 0x36, 0x9f, 0x60, 0x57, 0x4e, 0xa0, 0xb4, 0x8d, 0x79,
			},
			expectedHMAC: []byte{
				0x6a, 0x85, 0xf4, 0x5f, 0x66, 0x9d, 0x9e, 0xa5,
				0xed, 0x77, 0x4f, 0x74, 0x08, 0x99, 0x2e, 0x69,
				0xac, 0x86, 0x9b, 0x56, 0x71, 0x91, 0x16, 0x23,
				0x08, 0xec, 0x08, 0x99, 0x09, 0x33, 0x80, 0x17,
			},
		},
	}

	for _, c := range cases {
		t.Run(c.title, func(t *testing.T) {
			builder := NewBlockHMACBuilder(c.masterSeed[:], c.transformedKey[:])

			builtHMAC := builder.BuildHMAC(c.index, c.length, c.data)

			if !bytes.Equal(builtHMAC, c.expectedHMAC) {
				t.Errorf(
					"Received HMAC '% x', expected '% x'",
					builtHMAC,
					c.expectedHMAC,
				)
			}
		})
	}
}
